/*-
 * Copyright (c) 1996-1998,2000 Shunsuke Akiyama <akiyama@FreeBSD.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: odcontrol.c,v 1.12 2000/10/19 15:35:55 akiyama Exp $
 */

/*
 * odcontrol - SCSI optical disk drive control utility version 1.3
 */

/*
 * TODO :
 *
 * 1. Implement verbose mode.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdarg.h>

#include "odcontrol.h"

/*
 * cache mode control argument
 */

char *cache_args[] = { "off", "write", "read", "on", NULL };

#define C_CACHE_OFF	0
#define C_CACHE_WR	1
#define C_CACHE_RD	2
#define C_CACHE_ON	3

/*
 * command table
 */

int exec_cache(), exec_eject(), exec_format(), exec_verify();

struct cmd_tab commands[] = {
	{
		"cache",
		"[on|read|write|off] (drive cache control)",
		exec_cache,
		"/dev/r%s.ctl"
	},
	{
		"eject",
		"(eject a media)",
		exec_eject,
		"/dev/r%sc"
	},
	{
		"format",
		"(physical format a media)",
		exec_format,
		"/dev/r%s.ctl"
	},
	{
		"verify",
		"(verify a media)",
		exec_verify,
		"/dev/r%s.ctl"
	},
	{
		NULL,
		NULL,
		NULL,
		NULL
	}
};

/*
 * global variables
 */

char *program = "odcontrol";

int timeouts;
int verbose;
int have_message;

char *fdevice = DEFAULT_DEVICE;
int bus_no;
int target_no;
int lun_no;

/*
 *  xprintf() - printf() wrapper for quiet mode.
 */

void
xprintf(char *fmt, ...)
{
	va_list ap;

	if (verbose < 0) {
		return;
	}

	va_start(ap, fmt);
	vprintf(fmt, ap);
	va_end(ap);
}

/*
 *  xfprintf() - fprintf() wrapper for quiet mode.
 */

void
xfprintf(FILE *fp, char *fmt, ...)
{
	va_list ap;

	if (verbose < 0) {
		return;
	}

	va_start(ap, fmt);
	vfprintf(fp, fmt, ap);
	va_end(ap);
}

/*
 *  get_timeout() - get timeout value if specified in command line.
 */

int
get_timeout(defval)
     int defval;
{
	int retval;

	if (timeouts > 0) {
		retval = timeouts;
	} else {
		retval = defval;
	}

	return retval;
}

/*
 * exec_cache() - view or set current drive cache mode.
 */

int
exec_cache(argc, argv)
	int argc;
	char **argv;
{
	u_char data[255], page8[255];
	struct scsi_cache_ctrl *ctl;
	struct scsi_mode_parm_hdr pheader, *mph;
	int curval, mask, newval;
	int sts;
	char *arg, **p;
	int n, p8_len;

	sts = open_device();
	if (sts != ERR_NONE) {
		return sts;
	}

	/* get current setting */
	sts = get_cache_setting(data, 255);
	if (sts != ERR_NONE) {
		close_device();
		return sts;
	}

	mph = (struct scsi_mode_parm_hdr *) data;
	bcopy(mph, &pheader, sizeof (struct scsi_mode_parm_hdr));
	ctl = (struct scsi_cache_ctrl *)
	  (data + sizeof (struct scsi_mode_parm_hdr) + mph->blk_desc_len);
	p8_len = ctl->page_len + 2;
	bcopy(ctl, page8, p8_len);
	curval = ctl->cache_mode;

	if (argc != 0) {	/* set cache mode */
		/* find argument */
		arg = *argv;
		for (n = 0, p = cache_args; *p != NULL; ++p) {
			if (strcasecmp(arg, *p) == 0) {
				break;
			}
			n++;
		}
		if (*p == NULL) {
			xfprintf(stderr, "%s: unknown argument: %s\n",
				 program, arg);
			close_device();
			return ERR_COMMAND;
		}

		/* get new setting value */
		newval = 0;
		switch (n) {
		      case C_CACHE_OFF:	/* off */
			newval = SCSI_MODE_PAGE8_RCD;
			newval &= ~SCSI_MODE_PAGE8_WCE;
			break;
		      case C_CACHE_WR:	/* write enable */
			newval = SCSI_MODE_PAGE8_WCE | SCSI_MODE_PAGE8_RCD;
			break;
		      case C_CACHE_RD:	/* read enable */
			newval &= ~SCSI_MODE_PAGE8_RCD;
			newval &= ~SCSI_MODE_PAGE8_WCE;
			break;
		      case C_CACHE_ON:	/* on */
			newval &= ~SCSI_MODE_PAGE8_RCD;
			newval |= SCSI_MODE_PAGE8_WCE;
			break;
		}

		/* get changable mask data */
		sts = get_cache_mask(data, 255);
		if (sts != ERR_NONE) {
			close_device();
			return sts;
		}
		mph = (struct scsi_mode_parm_hdr *) data;
		ctl = (struct scsi_cache_ctrl *)
		  (data + sizeof (struct scsi_mode_parm_hdr) +
		   mph->blk_desc_len);
		mask = ctl->cache_mode;

		/* change only changable data */
		if (!(mask & SCSI_MODE_PAGE8_RCD)) {
			newval &= ~SCSI_MODE_PAGE8_RCD;
			newval |= (curval & SCSI_MODE_PAGE8_RCD);
		}
		if (!(mask & SCSI_MODE_PAGE8_WCE)) {
			newval &= ~SCSI_MODE_PAGE8_WCE;
			newval |= (curval & SCSI_MODE_PAGE8_WCE);
		}

		curval &= ~(SCSI_MODE_PAGE8_WCE | SCSI_MODE_PAGE8_RCD);
		newval |= curval;

		/* set mode parameter header */
		mph = (struct scsi_mode_parm_hdr *) data;
		bcopy(&pheader, data, sizeof (struct scsi_mode_parm_hdr));
		mph->mode_parm_len = 0;
		mph->blk_desc_len = 0;

		/* set cache control parameter (page 8) */
		ctl = (struct scsi_cache_ctrl *)
		  (data + sizeof (struct scsi_mode_parm_hdr));
		bcopy(page8, ctl, p8_len);
		ctl->page_len = p8_len - 2;
		ctl->cache_mode = newval;
		ctl->page &= 0x7f;		/* clear PS bit */

		/* set cache control parameter (page 8) */
		n = sizeof (struct scsi_mode_parm_hdr) + p8_len;
		sts = set_cache_setting(data, n);
		if (sts != ERR_NONE) {
			close_device();
			return sts;
		}

		/* re-read cache control parameter (page 8) */
		sts = get_cache_setting(data, 255);
		if (sts != ERR_NONE) {
			close_device();
			return sts;
		}
		mph = (struct scsi_mode_parm_hdr *) data;
		ctl = (struct scsi_cache_ctrl *)
		  (data + sizeof (struct scsi_mode_parm_hdr) +
		   mph->blk_desc_len);
		curval = ctl->cache_mode;
	}
	
	/* print current setting */
	xprintf("read cache: %s\n",
		(curval & SCSI_MODE_PAGE8_RCD) ? "off" : "on");
	xprintf("write cache: %s\n",
		(curval & SCSI_MODE_PAGE8_WCE) ? "on" : "off");

	close_device();

	return sts;
}

/*
 * exec_eject() - eject loaded media.
 */

int
exec_eject(argc, argv)
	int argc;
	char **argv;
{
	int sts;

	sts = eject_media();

	return sts;
}

/*
 * exec_format() - format a media.
 */

int
exec_format(argc, argv)
	int argc;
	char **argv;
{
	int sts;

	sts = open_device();
	if (sts != ERR_NONE) {
		return sts;
	}

	xprintf("formatting a media (scbus%d:target%d:lun%d) ... ",
		bus_no, target_no, lun_no);
	fflush(stdout);
	have_message = 1;

	sts = format_unit();
	if (sts == 0) {
		xprintf("done\n");
	}

	close_device();

	return sts;
}

/*
 * exec_verify() - verify a media.
 */

int
exec_verify(argc, argv)
	int argc;
	char **argv;
{
	int sts;
	u_int max, addr, len, bad;
	u_int reassigned = 0;
	char *mesg = "\b\b\b\b\b\b\b\b\b%3u%% done";

	sts = open_device();
	if (sts != ERR_NONE) {
		return sts;
	}

	sts = read_capacity(&max);
	if (sts != ERR_NONE) {
		close_device();
		return sts;
	}

	xprintf("verifying a media (scbus%d:target%d:lun%d) ... %3u%% done",
		bus_no, target_no, lun_no, 0);
	fflush(stdout);

	for (addr = 0; addr < max; addr += len) {
		have_message = 3;
		len = max - addr;
		if (len > MAX_VERIFY_LENGTH) {
			len = MAX_VERIFY_LENGTH;
		}
		sts = verify_media(addr, len, &bad);
		if (sts < 0) {
			have_message = 2;
			sts = reassign_block(bad);
			if (sts != ERR_NONE) {
				break;
			}
			len = bad - addr;
			reassigned++;
		} else if (sts != ERR_NONE) {
			break;
		}
		xprintf(mesg, (addr * 100) / max);
		fflush(stdout);
	}

	if (sts == 0) {
		xprintf(mesg, (addr * 100) / max);
		xprintf("\n");
		if (reassigned) {
			xprintf("%u blocks reassigned\n", reassigned);
		}
		sleep(1);
	}
	close_device();

	return sts;
}

/*
 * usage() - print usage.
 */

void
usage()
{
	struct cmd_tab *p;

	xfprintf(stderr,
		 "usage: %s [-f device] [-t timeout] command [arg]\n",
		 program);
	xfprintf(stderr, "\tcommands:\n");
	for (p = commands; p->c_name != NULL; ++p) {
		xfprintf(stderr, "\t  %s %s\n", p->c_name, p->help);
	}
	exit(ERR_COMMAND);
}

/*
 *  main() - odcontrol main procedure.
 */

int
main(argc, argv)
	int argc;
	char **argv;
{
	int ch;
	struct cmd_tab *cmd;
	int sts;

	have_message = verbose = 0;
	timeouts = 0;

	while ((ch = getopt(argc, argv, "f:t:vq?")) != EOF) {
		switch (ch) {
		case 'f':
			fdevice = optarg;
			break;
		case 't':
			timeouts = atoi(optarg);
			if (timeouts < 1) {
				usage();
			}
			timeouts *= 1000;
			break;
		case 'v':
			verbose = 1;
			break;
		case 'q' :
			verbose = -1;
			break;
		case '?':
		default:
			usage();
		}
	}
	argc -= optind;
	argv += optind;

	if (argc == 0) {
		usage();
	}
	if (strlen(fdevice) > 128) {
		xfprintf(stderr, "%s: device name too long\n", program);
		exit(ERR_COMMAND);
	}

	for (cmd = commands; cmd->c_name != NULL; ++cmd) {
		if (strcasecmp(cmd->c_name, *argv) == 0) {
			argc--;
			argv++;
			sts = get_device(fdevice, cmd);
			if (sts < 0) {
				perror(program);
				exit(ERR_COMMAND);
			}
			sts = (cmd->func)(argc, argv);
			break;
		}
	}
	if (cmd->c_name == NULL) {
		xfprintf(stderr, "%s: unknown command: %s\n", program, *argv);
		usage();
	}

	exit(sts);
}
