/*-
 * Copyright (c) 1996-1998,2000 Shunsuke Akiyama <akiyama@FreeBSD.org>.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *	$Id: scsiio.c,v 1.3 2000/09/24 13:07:20 akiyama Exp $
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <string.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <errno.h>
#include <sys/cdio.h>

#include "odcontrol.h"

int fd;
char device[256];
struct scsi_addr unit_addr;

/*
 * SCSI utility routines
 *  ONLY for little endian machines.
 *  MSB at the lowest address, LSB at the highest.
 */

void
scsi_uto2b(val, bytes)
	u_int val;
	u_char *bytes;
{
	*bytes++ = (val & 0xff00) >> 8;
	*bytes = val & 0xff;
}

u_int
scsi_2btou(bytes)
	u_char *bytes;
{
	u_int rc;

        rc = (*bytes++ << 8);
        rc += *bytes;

        return rc;
}

void
scsi_uto4b(val, bytes)
	u_int val;
	u_char *bytes;
{
	*bytes++ = (val & 0xff000000) >> 24;
	*bytes++ = (val & 0xff0000) >> 16;
	*bytes++ = (val & 0xff00) >> 8;
	*bytes =    val & 0xff;
}

u_int
scsi_4btou(bytes)
	u_char *bytes;
{
	u_int rc;

	rc = (*bytes++ << 24);
	rc += (*bytes++ << 16);
	rc += (*bytes++ << 8);
	rc +=  *bytes;

	return rc;
}

/*
 *  scsi_identify() - get tartget unit address.
 */

int
scsi_identify()
{
	int sts;

	sts = ioctl(fd, SCIOCIDENTIFY, &unit_addr);

	return sts;
}

/*
 *  scsi_error() - print SCSI error.
 */

void
scsi_error(req)
	scsireq_t *req;
{
	struct scsi_sense_data *sd;
	int skey, asc, ascq;
	static char *retstat[] = {
		"Timeout", "Busy", "Sense Key", "Unknown"
	};
	static char *sense_keys[] = {
		"No Sense", "Recoverd Error", "Not Ready",
		"Medium Error", "Hardware Error", "Illegal Request",
		"Unit Attention", "Data Protect", "Blank Check",
		"Vendor Unique", "Copy Aborted", "Aborted Command",
		"Equal", "Volume Overflow", "Reserved"
	};

	extern char *scsi_sense_desc();

	/* message termination */
	switch (have_message) {
	case 2:
	case 3:
		xfprintf(stderr, ", ");
		/* FALLTHROUGH */
	case 1:
		xfprintf(stderr, "aborted\n");
		break;
	default:
		break;
	}
	xfprintf(stderr, "%s: SCSI error: %s",
		 program, retstat[req->retsts - 1]);
	if (req->retsts == SCCMD_SENSE) {
		sd = (struct scsi_sense_data *) req->sense;
		skey = sd->sense_key & SCSI_SENSE_KEY;
		xfprintf(stderr, " - %s\n", sense_keys[skey]);
		if (sd->add_sense_len >= 5) {
			asc = sd->asc;
		} else {
			asc = 0;
		}
		if (sd->add_sense_len >= 6) {
			ascq = sd->ascq;
		} else {
			ascq = 0;
		}
		xfprintf(stderr, "%s: [%02X-%02X] %s\n",
			 program, asc, ascq, scsi_sense_desc(asc, ascq));
	} else {
		xfprintf(stderr, "\n");
	}
}

/*
 * test_unit_ready() - send test unit ready to target.
 */

int
test_unit_ready()
{
	int sts;
	scsireq_t scsireq;
	struct scsi_test_unit_ready *tur;
	struct scsi_sense_data *sd;
	int skey;

	bzero(&scsireq, sizeof (scsireq_t));
	scsireq.timeout = get_timeout(DEFAULT_TIMEOUT);	/* 10 sec */

	tur = (struct scsi_test_unit_ready *) scsireq.cmd;
	tur->op_code = SCSI_TEST_UNIT_READY;
	tur->lun = (unit_addr.lun << 5);

	scsireq.cmdlen = sizeof (struct scsi_test_unit_ready);
	scsireq.senselen = SENSEBUFLEN;

	sts = ioctl(fd, SCIOCCOMMAND, &scsireq);
	if (sts < 0) {
		perror(program);
		return ERR_SYSCALL;
	}
	if (scsireq.retsts != SCCMD_OK) {
		sd = (struct scsi_sense_data *) scsireq.sense;
		skey = sd->sense_key & SCSI_SENSE_KEY;
		if (scsireq.retsts != SCCMD_SENSE) {
			scsi_error(&scsireq);
			sts = ERR_SCSI;
		} else if (skey == SCSI_NO_SENSE || skey == SCSI_NOT_READY
			   || skey == SCSI_UNIT_ATTENTION) {
			/*
			 * "No Sense", "Not Ready" and "Unit Attention"
			 * are not really errors here.
			 */
			sts = ERR_NONE;
		} else {
			scsi_error(&scsireq);
			sts = ERR_SCSI;
		}
	}

	return sts;
}	

/*
 * get current cache control setting
 */

int
get_cache_setting(curval, len)
	u_char *curval;
	int len;
{
	int sts;
	scsireq_t scsireq;
	struct scsi_mode_sense *ms;

	bzero(&scsireq, sizeof (scsireq));
	bzero(curval, len);
	scsireq.flags = SCCMD_READ;
	scsireq.timeout = get_timeout(DEFAULT_TIMEOUT);	/* 10 sec */

	ms = (struct scsi_mode_sense *) scsireq.cmd;
	ms->op_code = SCSI_MODE_SENSE;
	ms->lun_flag = (unit_addr.lun << 5);
	ms->page = 8;
	ms->alloc_len = len;

	scsireq.cmdlen = sizeof (struct scsi_mode_sense);
	scsireq.databuf = curval;
	scsireq.datalen = len;
	scsireq.senselen = SENSEBUFLEN;

	sts = ioctl(fd, SCIOCCOMMAND, &scsireq);
	if (sts < 0) {
		perror(program);
		return ERR_SYSCALL;
	}
	if (scsireq.retsts != SCCMD_OK) {
		scsi_error(&scsireq);
		sts = ERR_SCSI;
	}

	return sts;
}

/*
 * get changable cache control parameter mask
 */

int
get_cache_mask(maskval, len)
	u_char *maskval;
	int len;
{
	int sts;
	scsireq_t scsireq;
	struct scsi_mode_sense *ms;

	bzero(&scsireq, sizeof (scsireq));
	bzero(maskval, len);
	scsireq.flags = SCCMD_READ;
	scsireq.timeout = get_timeout(DEFAULT_TIMEOUT);	/* 10 sec */

	ms = (struct scsi_mode_sense *) scsireq.cmd;
	ms->op_code = SCSI_MODE_SENSE;
	ms->lun_flag = (unit_addr.lun << 5);
	ms->page = 0x40 | 8;
	ms->alloc_len = len;

	scsireq.cmdlen = sizeof (struct scsi_mode_sense);
	scsireq.databuf = maskval;
	scsireq.datalen = len;
	scsireq.senselen = SENSEBUFLEN;

	sts = ioctl(fd, SCIOCCOMMAND, &scsireq);
	if (sts < 0) {
		perror(program);
		return ERR_SYSCALL;
	}
	if (scsireq.retsts != SCCMD_OK) {
		scsi_error(&scsireq);
		sts = ERR_SCSI;
	}

	return sts;
}

/*
 * set cache control parameter
 */

int
set_cache_setting(newval, len)
	u_char *newval;
	int len;
{
	int sts;
	scsireq_t scsireq;
	struct scsi_mode_select *ms;

	bzero(&scsireq, sizeof (scsireq));
	scsireq.flags = SCCMD_WRITE;
	scsireq.timeout = get_timeout(DEFAULT_TIMEOUT);	/* 10 sec */

	ms = (struct scsi_mode_select *) scsireq.cmd;
	ms->op_code = SCSI_MODE_SELECT;
	ms->lun_flag = (unit_addr.lun << 5) | 0x10;
	ms->plist_len= len;

	scsireq.cmdlen = sizeof (struct scsi_mode_select);
	scsireq.databuf = newval;
	scsireq.datalen = len;
	scsireq.senselen = SENSEBUFLEN;

	sts = ioctl(fd, SCIOCCOMMAND, &scsireq);
	if (sts < 0) {
		perror(program);
		return ERR_SYSCALL;
	}
	if (scsireq.retsts != SCCMD_OK) {
		scsi_error(&scsireq);
		sts = ERR_SCSI;
	}

	return sts;
}

/*
 * initiate format unit command
 */

int
format_unit()
{
	int sts;
	scsireq_t scsireq;
	struct scsi_format_unit *fu;

	bzero(&scsireq, sizeof (scsireq_t));
	scsireq.timeout = get_timeout(DEFAULT_TIMEOUT_LONG);	/* 3600 sec */

	fu = (struct scsi_format_unit *) scsireq.cmd;
	fu->opcode = SCSI_FORMAT_UNIT;
	fu->byte2 = (unit_addr.lun << 5);
	scsi_uto2b(0, fu->interleave);		/* default interleave */

	scsireq.cmdlen = sizeof (struct scsi_format_unit);
	scsireq.databuf = NULL;
	scsireq.datalen = 0;
	scsireq.senselen = SENSEBUFLEN;

	sts = ioctl(fd, SCIOCCOMMAND, &scsireq);
	if (sts < 0) {
		perror(program);
		return ERR_SYSCALL;
	}
	if (scsireq.retsts != SCCMD_OK) {
		scsi_error(&scsireq);
		sts = ERR_SCSI;
	}

	return sts;
}

/*
 * get unit capacity
 */

int
read_capacity(max)
	u_int *max;
{
	int sts;
	scsireq_t scsireq;
	struct scsi_read_capacity *rc;
	struct scsi_read_capacity_data rdcap_data;

	bzero(&scsireq, sizeof (scsireq));
	bzero(&rdcap_data, sizeof (rdcap_data));
	scsireq.flags = SCCMD_READ;
	scsireq.timeout = get_timeout(DEFAULT_TIMEOUT);

	rc = (struct scsi_read_capacity *) &scsireq.cmd;
	rc->op_code = SCSI_READ_CAPACITY;
	rc->lun_flag = (unit_addr.lun << 5);

	scsireq.cmdlen = sizeof (struct scsi_read_capacity);
	scsireq.databuf = (u_char *) &rdcap_data;
	scsireq.datalen = sizeof (rdcap_data);
	scsireq.senselen = SENSEBUFLEN;

	sts = ioctl(fd, SCIOCCOMMAND, &scsireq);
	if (sts < 0) {
		perror(program);
		return ERR_SYSCALL;
	}
	if (scsireq.retsts != SCCMD_OK) {
		scsi_error(&scsireq);
		return ERR_SCSI;
	}

	*max = scsi_4btou(rdcap_data.addr) + 1;

	return ERR_NONE;
}

/*
 * verify media blocks
 */

int
verify_media(addr, len, bad)
	u_int addr;
	u_int len;
	u_int *bad;
{
	int sts;
	scsireq_t scsireq;
	struct scsi_verify_blocks *vc;
	struct scsi_sense_data *sense;

	bzero(&scsireq, sizeof (scsireq));
	scsireq.timeout = get_timeout(DEFAULT_TIMEOUT);	/* 10 sec */

	vc = (struct scsi_verify_blocks *) &scsireq.cmd;
	vc->opcode = SCSI_VERIFY;
	vc->byte2 = (unit_addr.lun << 5);
	scsi_uto4b(addr, vc->lb_addr);
	scsi_uto2b(len, vc->lb_count);

	scsireq.cmdlen = sizeof (struct scsi_verify_blocks);
	scsireq.databuf = NULL;
	scsireq.datalen = 0;
	scsireq.senselen = SENSEBUFLEN;

	sts = ioctl(fd, SCIOCCOMMAND, &scsireq);
	if (sts < 0) {
		perror(program);
		return ERR_SYSCALL;
	}
	if (scsireq.retsts != SCCMD_OK) {
		sense = (struct scsi_sense_data *) scsireq.sense;
		if (scsireq.retsts == SCCMD_SENSE
		    && (sense->sense_key & SCSI_SENSE_KEY)
		       == SCSI_MEDIUM_ERROR) {
			*bad = scsi_4btou(sense->info);
			sts = ERR_BADBLOCK;
		} else {
			scsi_error(&scsireq);
			sts = ERR_SCSI;
		}
	}

	return sts;
}

/*
 * reassign bad block
 */

int
reassign_block(bad)
	u_int bad;
{
	int sts;
	scsireq_t scsireq;
	struct scsi_reassign_blocks *rb;
	struct scsi_defect_list defect;

	bzero(&scsireq, sizeof (scsireq));
	scsireq.flags = SCCMD_WRITE;
	scsireq.timeout = get_timeout(DEFAULT_TIMEOUT);	/* 10 sec */

	rb = (struct scsi_reassign_blocks *) scsireq.cmd;
	rb->opcode = SCSI_REASSIGN_BLOCKS;
	rb->byte2 = (unit_addr.lun << 5);

	bzero(&defect, sizeof (defect));
	scsi_uto2b(4, defect.list_len);
	scsi_uto4b(bad, defect.lb_addr);

	scsireq.cmdlen = sizeof (struct scsi_reassign_blocks);
	scsireq.databuf = (caddr_t) &defect;
	scsireq.datalen = sizeof (defect);
	scsireq.senselen = SENSEBUFLEN;

	sts = ioctl(fd, SCIOCCOMMAND, &scsireq);
	if (sts < 0) {
		perror(program);
		return ERR_SYSCALL;
	}
	if (scsireq.retsts != SCCMD_OK) {
		scsi_error(&scsireq);
		sts = ERR_SCSI;
	}

	return sts;
}

/*
 * eject media
 */

int
eject_media(void)
{
	int sts;

	fd = open(device, O_RDONLY);
	if (fd < 0) {
		if (errno == ENXIO) {
			/* no media */
			return ERR_NONE;
		}
		return ERR_SYSCALL;
	}
	sts = ioctl(fd, CDIOCALLOW);
	if (sts >= 0) {
		ioctl(fd, CDIOCEJECT);
		sts = ERR_NONE;
	} else {
		sts = ERR_SYSCALL;
	}
	close(fd);

	return sts;
}

/*
 * get device name and check it
 */

int
get_device(dev, cmd)
	char *dev;
	struct cmd_tab *cmd;
{
	int sts;
	struct stat sb;

	sts = stat(dev, &sb);		/* check full path name */
	if (sts < 0) {
		/* device name abbreviation */
		sprintf(device, cmd->dev_proto, dev);
		sts = stat(device, &sb);
	} else {
		/* full path name */
		strcpy(device, dev);
	}

	return sts;
}

/*
 * close the device
 */

void
close_device()
{
	close(fd);
}

/*
 * open specified device
 */

int
open_device()
{
	int sts;

	fd = open(device, O_RDWR);
	if (fd < 0) {
		perror(program);
		return ERR_SYSCALL;
	}

	/* get tartget address */
	sts = scsi_identify();
	if (sts < 0) {
		perror(program);
		close(fd);
		return ERR_SYSCALL;
	}

	bus_no = unit_addr.scbus;
	target_no = unit_addr.target;
	lun_no = unit_addr.lun;

	/* clear "Unit Attention" */
	sts = test_unit_ready();
	if (sts != ERR_NONE) {
		close_device();
		return sts;
	}

	return ERR_NONE;
}
