/*
 *	pccardd UNIX-domain socket interface
 *	Copyright (C) 1996 by Tatsumi Hosokawa <hosokawa@mt.cs.keio.ac.jp>
 */
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <signal.h>
#include <setjmp.h>
#include <syslog.h>

#include "cardd.h"

static void
cardnum(char *buf)
{
	int	i = 0;
	struct slot *sp;

	for (sp = slots; sp; sp = sp->next) {
		i++;
	}
	if (i > MAXSLOT) {
		i = MAXSLOT;
	}
	sprintf(buf, "%2d", i);
}

static void 
cardname(char *buf, int slot) 
{
	struct slot	*sp;

	for (sp = slots; sp; sp = sp->next) {
		if (sp->slot == slot) {
			char *manuf, *vers, *drv, *stat;
			if (sp->cis) {
				manuf = sp->cis->manuf;
				vers = sp->cis->vers;
				if (sp->config && sp->config->driver && 
					sp->config->driver->name) {
					drv = sp->config->driver->name;
				}
				else {
					drv = "";
				}
			}
			else {
				manuf = vers = drv = "";
			}
			switch (sp->state) {
			case empty:
				stat = "0";
				break;
			case filled:
				stat = "1";
				break;
			case inactive:
				stat = "2";
				break;
			default:
				stat = "9";
			}
			sprintf(buf, "%d~%s~%s~%s~%s", slot, manuf, vers, drv, stat);
			return;
		}
	}
}

static void
cardpwr(int slot, int pwon)
{
	struct slot	*sp;

	for (sp = slots; sp; sp = sp->next) {
		if (sp->slot == slot) {
			if (ioctl(sp->fd, PIOCSVIR, &pwon) < 0) {
				logerr("invaild arguments for cardpwr");
			}
			break;
		}
	}
}

static int sock = 0;
static int slen = 0;
static struct sockaddr_un sun;

void
set_socket(int s)
{
	sock = s;
}

void
stat_changed(struct slot *sp)
{
	int	len;
	char	buf[512];

	if (!slen) {
		return;
	}
	cardname(buf, sp->slot);
	len = strlen(buf);
	if (sendto(sock, buf, len, 0, (struct sockaddr *)&sun, slen) != len) {
		syslog(LOG_ERR, "sendto failed");
		slen = 0;
	}
}

void 
process_client(void)
{
	char	buf[512], obuf[512];
	int	len;
	int	snum;

	if (!sock) {
		return;
	}
	slen = sizeof(sun);
	len = recvfrom(sock, buf, sizeof(buf), 0, 
			(struct sockaddr *)&sun, &slen);
	if (len < 0) {
		syslog(LOG_ERR, "recvfrom failed");
	}
	obuf[0] = 0;
	switch (buf[0]) {
	case 'S':
		cardnum(obuf);
		break;
	case 'N':
		sscanf(buf + 1, "%d", &snum);
		if (snum >= 0 && snum <= MAXSLOT) {
			cardname(obuf, snum);
		}
		else {
			syslog(LOG_ERR, "Illegal slot requests for N command");
		}
		break;
	case 'P':
		sscanf(buf + 1, "%d", &snum);
		if (snum >= 0 && snum <= MAXSLOT) {
			syslog(LOG_NOTICE, "slot %d: spring has come", snum);
			cardpwr(snum, 1);
		}
		else {
			syslog(LOG_ERR, "Illegal slot requests for P command");
		}
		break;
	case 'Q':
		sscanf(buf + 1, "%d", &snum);
		if (snum >= 0 && snum <= MAXSLOT) {
			syslog(LOG_NOTICE, "slot %d: hybernation", snum);
			cardpwr(snum, 0);
		}
		else {
			syslog(LOG_ERR, "Illegal slot requests for Q command");
		}
		break;
	default:
		syslog(LOG_ERR, "Unknown control message from socket");
		break;
	}
	len = strlen(obuf);
	if (len) {
		if (sendto(sock, obuf, len, 0, (struct sockaddr *)&sun, slen)
			!= len) {
			syslog(LOG_ERR, "sendto failed");
			slen = 0;
		}
	}
	else {
		if (sendto(sock, 0, 0, 0, (struct sockaddr *)&sun, slen)
			!= len) {
			syslog(LOG_ERR, "sendto failed");
			slen = 0;
		}
	}
}
