/*
 * Copyright (c) 1995 Andrew McRae.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * $Id: cardd.h,v 1.12 1998/03/09 05:18:55 hosokawa Exp $
 *
 *	Common include file for PCMCIA daemon
 */

/*
 * Code cleanup, bug-fix and extension
 * by Tatsumi Hosokawa <hosokawa@mt.cs.keio.ac.jp>
 */

#include <bitstring.h>

#include <pccard/cardinfo.h>
#include <pccard/cis.h>
#include <pccard/i82365reg.h>

#ifdef WAVELAN
#include <machine/wavelan.h>
#endif /* WAVELAN */

#include "readcis.h"

#ifndef EXTERN
#define EXTERN extern
#endif

struct cmd {
	struct cmd *next;
	char   *line;		/* Command line */
	int     macro;		/* Contains macros */
};

struct card_io {
	struct card_io *next;
	int     addr;
	int     size;
};

struct card_mem {
	struct card_mem *next;
	int     addr;
	int     cardaddr;
	int     size;
	int	flags;
};

struct card_config {
	struct card_config *next;
	unsigned char index;
	struct driver *driver;
	struct card_io *card_io;
	struct card_mem *card_mem;
	int     irq;
	int     flags;
	char    inuse;
};

/* ethernet MAC address check */
struct ether_mac {
	struct ether_mac *next;
	u_char  addr[3];
};

struct ether {
	struct ether *next;
	int     type;
	int     attr;
	struct ether_mac *ether_mac;
};
#define	ETHTYPE_GENERIC		0
#define	ETHTYPE_ATTR2HEX	1
#define	ETHTYPE_WAVELAN		2
 
struct card {
	struct card *next;
	char   *manuf;
	char   *version;
	char   *add_info1;
	char   *add_info2;
	u_char  func_id;
	int     deftype;
	int     reset_time;		/* Reset time */
	struct ether *ether;		/* For net cards, ether at offset */
	struct card_config *config;	/* List of configs */
	struct cmd *insert;		/* Insert commands */
	struct cmd *remove;		/* Remove commands */
};

struct driver {
	struct driver *next;
	char   *name;
	char   *kernel;			/* Kernel driver base name */
	int     unit;			/* Unit of driver */
	/*
	 * The rest of the structure is allocated dynamically.
	 * Once allocated, it stays allocated.
	 */
	struct card *card;		/* Current card, if any */
	struct card_config *config;	/* Config back ptr */
	unsigned int mem;		/* Allocated host address (if any) */
	int     inuse;
};

/*
 *	Defines one allocation block i.e a starting address
 *	and size. Used for either memory or I/O ports
 */
struct allocblk {
	struct allocblk *next;
	int     addr;			/* Address */
	int     size;			/* Size */
	int     flags;			/* Flags for block */
	int     cardaddr;		/* Card address */
};

/*
 *	Slot structure - data held for each slot.
 */
struct slot {
	struct slot *next;
	int     fd;
	int     mask;
	int     slot;
	char   *name;
	enum cardstate state;
	struct cis *cis;
	struct card *card;		/* Current card */
	struct card_config *config;	/* Current configuration */
	struct cis_config *card_config;
	char    devname[16];
	unsigned char eaddr[6];		/* If any */
	struct allocblk io;		/* I/O block spec */
	struct allocblk mem;		/* Memory block spec */
#ifdef WAVELAN
	struct wavelan_conf wl;
#endif
	int     irq;			/* Irq value */
	int     flags;			/* Resource assignment flags */
};

/*
 * Slot resource assignment/configuration flags
 */
#define	IO_ASSIGNED	0x1
#define	MEM_ASSIGNED	0x2
#define	IRQ_ASSIGNED	0x4

#define	EADDR_CONFIGED	0x8
#define	WL_CONFIGED	0x10

#define AFLAGS	(IO_ASSIGNED | MEM_ASSIGNED | IRQ_ASSIGNED)
#define	CFLAGS	(EADDR_CONFIGED | WL_CONFIGED)

EXTERN struct slot *slots, *current_slot;

#ifdef SLOT_ALLOC_IRQ
/* quick and qirty hack for slot patch by nakagawa */
struct slot_pool {
	int flag;
	struct allocblk *pool_ioblks;		/* I/O blocks in the pool */
	struct allocblk *pool_mem;		/* Memory in the pool */
	int     pool_irq[16];			/* IRQ allocations */
};

EXTERN struct slot_pool pool_slots[MAXSLOT];
#endif /* SLOT_ALLOC_IRQ */

EXTERN struct allocblk *pool_ioblks;		/* I/O blocks in the pool */
EXTERN struct allocblk *pool_mem;		/* Memory in the pool */
EXTERN int     pool_irq[16];			/* IRQ allocations */
EXTERN struct driver *drivers;			/* List of drivers */
EXTERN struct card *cards;
EXTERN struct card *last_card;
EXTERN bitstr_t *mem_avail;
EXTERN bitstr_t *io_avail;
EXTERN bitstr_t *mem_kern;
EXTERN bitstr_t *io_kern;

EXTERN int verbose;
EXTERN int nosyslog;
EXTERN int irqflag;

char   *newstr(char *);
void		 die(char *);
void   *xmalloc(int);
void		 log_setup(void);
void    logmsg(const char *, ...);
void		 logerr(char *);
void		 reset_slot(struct slot *);
void    execute(struct cmd *);
void    readslots(void);
void    slot_change(struct slot *);
void    set_socket(int);
void    stat_changed(struct slot *);
void    process_client(void);
void    readfile(char *, u_int);
int     bit_fns(bitstr_t *, int, int, int);
unsigned long alloc_memory(int);

#define	IOPORTS	0x10000
#define	MEMUNIT	0x1000
#define	MEMSTART 0xA0000
#define	MEMEND	0x100000
#define	MEMBLKS	((MEMEND-MEMSTART)/MEMUNIT)
#define	MEM2BIT(x) (((x)-MEMSTART)/MEMUNIT)
#define	BIT2MEM(x) (((x)*MEMUNIT)+MEMSTART)

#define	DT_VERS	0
#define	DT_FUNC	1
