/* Copyright (C) 1989-1992 Aladdin Enterprises.  All rights reserved.

This file is part of Ghostscript.

Ghostscript is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY.  No author or distributor accepts responsibility
to anyone for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.  Refer
to the Ghostscript General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
Ghostscript, but only under the conditions described in the Ghostscript
General Public License.  A copy of this license is supposed to have been
given to you along with Ghostscript so you can know your rights and
responsibilities.  It should be in a file named COPYING.  Among other
things, the copyright notice and this notice must be preserved on all
copies.  */

/* gdevlips4.c */
/*
 * GhostScript printer driver for Canon LBP and BJ printers(LIPS-III/IV/IVc)
 *
 *  v1.0.0 95/12/22 yasumasa@i1nws1.fukui-u.ac.jp
 *                Initial release
 *  v1.0.1 95/12/29 yasumasa@i1nws1.fukui-u.ac.jp
 *                Cleaned to be able to compile other compiler than GCC
 *  v1.1.0 96/11/25 msatoh@mrit.mei.co.jp
 *                Add LBP-2030 support
 *  v1.2.0 97/04/01 sawada@i1nws1.fukui-u.ac.jp
 *                gs4.03 support and bug fix
 *
 * If any comment, write to dvipr@i1nws1.fuis.fukui-u.ac.jp
 */
#include "gdevprn.h"

#define	LIPS_ESC	"\x1b"
#define	LIPS_FF		"\x0c"
#define	LIPS_CSI	LIPS_ESC "["
#define	LIPS_DCS	LIPS_ESC "P"
#define	LIPS_ST		LIPS_ESC "\\"

#define	MMETER_PER_INCH	25.4

#define	_FALSE	(1 == 0)
#define	_TRUE	(1 == 1)

#undef TRUE
#undef FALSE

typedef enum tagBOOL {TRUE = _TRUE, FALSE = _FALSE} BOOL;
typedef enum tagPDLMODE {
	PDL_UNKNOWN,
	PDL_LIPS3,
	PDL_LIPS3FIG,
	PDL_LIPS4,
	PDL_LIPS4FIG,
	PDL_LIPS4C,
	PDL_LIPS4CFIG,
	PDL_LBP2030
} PDLMODE;

private int LipsGetBoundingBox(gx_device_printer *pDev, PDLMODE pdl, int *ulx, int *uly, int *lrx, int *lry);
private PDLMODE LipsGetPdlMode(gx_device_printer *pDev);
private void LipsJobStart(PDLMODE pdl, FILE *fp);
private void LipsJobEnd(PDLMODE pdl, FILE *fp);
private dev_proc_map_rgb_color(MapRgbToColor);
private dev_proc_map_color_rgb(MapColorToRgb);
private dev_proc_print_page(LipsPrintPage);
private int LipsOpen(gx_device *pDev);
private int LipsLineOrder(byte *pBuff, byte *rBuff, byte *gBuff, byte *bBuff, int Length);
private int LipsRunLengthEncode(byte *inBuff, byte *outBuff, int Length);

gx_device_procs PrnLipsProcs =
	prn_procs(LipsOpen,	gdev_prn_output_page, gdev_prn_close);
gx_device_procs PrnLipsColorProcs =
	prn_color_procs(LipsOpen, gdev_prn_output_page, gdev_prn_close,	MapRgbToColor, MapColorToRgb);

gx_device_printer gs_lips3_device =
	prn_device(
		PrnLipsProcs, "lips3",
		DEFAULT_WIDTH_10THS_A4, DEFAULT_HEIGHT_10THS_A4,
		300, 300,
		0., 0., 0., 0.,
		1, LipsPrintPage
	);
gx_device_printer gs_lips3fig_device =
	prn_device(
		PrnLipsProcs, "lips3fig",
		DEFAULT_WIDTH_10THS_A4, DEFAULT_HEIGHT_10THS_A4,
		300, 300,
		0., 0., 0., 0.,
		1, LipsPrintPage
	);
gx_device_printer gs_lips4_device =
	prn_device(
		PrnLipsProcs, "lips4",
		DEFAULT_WIDTH_10THS_A4, DEFAULT_HEIGHT_10THS_A4,
		600, 600,
		0., 0., 0., 0.,
		1, LipsPrintPage
	);
gx_device_printer gs_lips4fig_device =
	prn_device(
		PrnLipsProcs, "lips4fig",
		DEFAULT_WIDTH_10THS_A4, DEFAULT_HEIGHT_10THS_A4,
		600, 600,
		0., 0., 0., 0.,
		1, LipsPrintPage
	);
gx_device_printer gs_lips4c_device =
	prn_device(
		PrnLipsColorProcs, "lips4c",
		DEFAULT_WIDTH_10THS_A4, DEFAULT_HEIGHT_10THS_A4,
		360, 360,
		0., 0., 0., 0.,
		24, LipsPrintPage
	);
gx_device_printer gs_lips4cfig_device =
	prn_device(
		PrnLipsColorProcs, "lips4cfig",
		DEFAULT_WIDTH_10THS_A4, DEFAULT_HEIGHT_10THS_A4,
		360, 360,
		0., 0., 0., 0.,
		24, LipsPrintPage
	);
gx_device_printer gs_lbp2030_device =
	prn_device (
		PrnLipsColorProcs, "lbp2030",
		DEFAULT_WIDTH_10THS_A4, DEFAULT_HEIGHT_10THS_A4,
		600, 600,
		0., 0., 0., 0.,
		24, LipsPrintPage
	);

private int LipsPrintPage(gx_device_printer *pDev, FILE *fp)
{
	int		x, y, ulx, uly, lrx, lry, nBytesPerLine;
	int		rLen, gLen, bLen;
	byte	*pBuff;
	byte	*rBuff, *rComBuff, *gBuff, *gComBuff, *bBuff, *bComBuff;
	PDLMODE	pdl;
	
	nBytesPerLine = gdev_prn_raster(pDev);
	if ((pdl = LipsGetPdlMode(pDev)) == PDL_UNKNOWN)
		return 0;
	LipsGetBoundingBox(pDev, pdl, &ulx, &uly, &lrx, &lry);
	if (!(pBuff = (byte *)gs_malloc(1, nBytesPerLine, "LipsLineBuffer")))
		return_error(gs_error_VMerror);
	if (pdl == PDL_LBP2030){
		if (!(rBuff = (byte *)gs_malloc(1, (lrx - ulx + 1), "LipsLineBuffer")))
			return_error(gs_error_VMerror);
		if (!(gBuff = (byte *)gs_malloc(1, (lrx - ulx + 1), "LipsLineBuffer")))
			return_error(gs_error_VMerror);
		if (!(bBuff = (byte *)gs_malloc(1, (lrx - ulx + 1), "LipsLineBuffer")))
			return_error(gs_error_VMerror);
		if (!(rComBuff = (byte *)gs_malloc(1, (lrx - ulx + 1) * 2, "LipsLineBuffer")))
			return_error(gs_error_VMerror);
		if (!(gComBuff = (byte *)gs_malloc(1, (lrx - ulx + 1) * 2, "LipsLineBuffer")))
			return_error(gs_error_VMerror);
		if (!(bComBuff = (byte *)gs_malloc(1, (lrx - ulx + 1) * 2, "LipsLineBuffer")))
			return_error(gs_error_VMerror);
	}
	LipsJobStart(pdl, fp);
	fprintf(fp, "%s2yGS:%s job%s", LIPS_DCS, pDev->dname, LIPS_ST);
	if (uly / pDev->y_pixels_per_inch < 63. / 300.)
		uly = 63. * pDev->y_pixels_per_inch / 300.;
	fprintf(fp, "%s%d;%df", LIPS_CSI,
		(int)(((double)uly / pDev->y_pixels_per_inch - 63. / 300.) * 720.),
		(int)(((double)ulx / pDev->x_pixels_per_inch) * 720.));

	if (pdl == PDL_LBP2030) {
		fprintf (fp, "%s%d;%d;%d&u", LIPS_CSI, (int) ((lrx - ulx + 1 + 7) / 8), lry - uly + 1, (int) pDev->x_pixels_per_inch);
		for (y = uly; y <= lry; y++) {
			gdev_prn_copy_scan_lines (pDev, y, pBuff, nBytesPerLine);
			LipsLineOrder(pBuff + ulx * 3, rBuff, gBuff, bBuff, lrx - ulx + 1);

			rLen = LipsRunLengthEncode(rBuff, rComBuff, lrx - ulx + 1);
			gLen = LipsRunLengthEncode(gBuff, gComBuff, lrx - ulx + 1);
			bLen = LipsRunLengthEncode(bBuff, bComBuff, lrx - ulx + 1);

			fprintf(fp, "%s%d;%d;%d;10;1;%d;8;11;;1.r", 
				LIPS_CSI, rLen + gLen + bLen, lrx - ulx + 1, 
				(int)pDev->x_pixels_per_inch, lrx - ulx + 1);

			fwrite (rComBuff, rLen, 1, fp);
			fwrite (gComBuff, gLen, 1, fp);
			fwrite (bComBuff, bLen, 1, fp);
		}

	}else if (pdl == PDL_LIPS4C || pdl == PDL_LIPS4CFIG) {
		fprintf(fp, "%s%d;%d;%d;0;;;8;10;;0.r", LIPS_CSI, ((lrx - ulx) + 1) * 3 * (lry - uly + 1), (lrx - ulx) + 1, (int)pDev->x_pixels_per_inch);
		for (y = uly; y <= lry; y++) {
			gdev_prn_copy_scan_lines(pDev, y, pBuff, nBytesPerLine);
			fwrite(&pBuff[ulx * 3], ((lrx - ulx) + 1) * 3, 1, fp);
		}
	} else {
		if (pdl == PDL_LIPS4)
			fprintf(fp, "%s%d;%d;%d;0;;;;;;3.r", LIPS_CSI, ((lrx - ulx) / 8 + 1) * (lry - uly + 1), (lrx - ulx) / 8 + 1, (int)pDev->x_pixels_per_inch);
		else
			fprintf(fp, "%s%d;%d;%d;0;.r", LIPS_CSI, ((lrx - ulx) / 8 + 1) * (lry - uly + 1), (lrx - ulx) / 8 + 1, (int)pDev->x_pixels_per_inch);
		for (y = uly; y <= lry; y++) {
			gdev_prn_copy_scan_lines(pDev, y, pBuff, nBytesPerLine);
			fwrite(&pBuff[ulx / 8], (lrx - ulx) / 8 + 1, 1, fp);
		}
	}
	LipsJobEnd(pdl, fp);
	gs_free(pBuff, 1, nBytesPerLine, "LipsLineBuffer");
	if (pdl == PDL_LBP2030){
		gs_free (rBuff, 1, lrx - ulx + 1, "LipsLineBuffer");
		gs_free (gBuff, 1, lrx - ulx + 1, "LipsLineBuffer");
		gs_free (bBuff, 1, lrx - ulx + 1, "LipsLineBuffer");
		gs_free (rComBuff, 1, (lrx - ulx + 1) * 2, "LipsLineBuffer");
		gs_free (gComBuff, 1, (lrx - ulx + 1) * 2, "LipsLineBuffer");
		gs_free (bComBuff, 1, (lrx - ulx + 1) * 2, "LipsLineBuffer");
	}

	return 0;
}

private int LipsOpen(gx_device *pDev)
{
	PDLMODE	pdl;
	private const float margins[4] = {
		5. / MMETER_PER_INCH,
		5. / MMETER_PER_INCH,
		5. / MMETER_PER_INCH,
		5. / MMETER_PER_INCH,
	};
	private const float c_margins[4] = {
		5. / MMETER_PER_INCH,
		8. / MMETER_PER_INCH,
		5. / MMETER_PER_INCH,
		2. / MMETER_PER_INCH,
	};

	pdl = LipsGetPdlMode((gx_device_printer*)pDev);
	if (pdl == PDL_LIPS3 || pdl == PDL_LIPS3FIG || pdl == PDL_LIPS4 || pdl == PDL_LIPS4FIG) {
		gx_device_set_margins(pDev, margins, true);
	} else if (pdl == PDL_LIPS4C || pdl == PDL_LIPS4CFIG) {
		gx_device_set_margins(pDev, c_margins, true);
	}

	return gdev_prn_open(pDev);
}

private int LipsGetBoundingBox(gx_device_printer *pDev, PDLMODE pdl, int *ulx, int *uly, int *lrx, int *lry)
{
	int		x, y, nBytesPerLine;
	byte	*pBuff;
	BOOL	bData;

	nBytesPerLine = gdev_prn_raster(pDev);
	*ulx = *uly = *lrx = *lry = -1;
	if (!(pBuff = (byte *)gs_malloc(1, nBytesPerLine, "LipsLineBuffer")))
		return_error(gs_error_VMerror);
	for (y = 0; y < pDev->height; y++) {
		bData = FALSE;
		gdev_prn_copy_scan_lines(pDev, y, pBuff, nBytesPerLine);
		if (pdl == PDL_LIPS4C || pdl == PDL_LIPS4CFIG || pdl == PDL_LBP2030) {
			for (x = 0; x < nBytesPerLine; x += 3)
				if (pBuff[x] != 0xff || pBuff[x + 1] != 0xff || pBuff[x + 2] != 0xff) {
					bData = TRUE;
					if (*ulx == -1 || x / 3 < *ulx)
						*ulx = x / 3;
					if (*lrx == -1 || x / 3 > *lrx)
						*lrx = x / 3;
				}
		} else {
			for (x = 0; x < nBytesPerLine; x++)
				if (pBuff[x]) {
					bData = TRUE;
					if (*ulx == -1 || x * 8 < *ulx)
						*ulx = x * 8;
					if (*lrx == -1 || x * 8 > *lrx)
						*lrx = x * 8;
				}
		}
		if (bData) {
			*lry = y;
			if (*uly == -1)
				*uly = y;
		}
	}
	gs_free(pBuff, 1, nBytesPerLine, "LipsLineBuffer");
	return 0;
}

private PDLMODE LipsGetPdlMode(gx_device_printer *pDev)
{
	if (!strcmp(pDev->dname, "lips3fig"))
		return PDL_LIPS3FIG;
	if (!strcmp(pDev->dname, "lips3"))
		return PDL_LIPS3;
	if (!strcmp(pDev->dname, "lips4fig"))
		return PDL_LIPS4FIG;
	if (!strcmp(pDev->dname, "lips4c"))
		return PDL_LIPS4C;
	if (!strcmp(pDev->dname, "lips4cfig"))
		return PDL_LIPS4CFIG;
	if (!strcmp(pDev->dname, "lips4"))
		return PDL_LIPS4;
	if (!strcmp(pDev->dname, "lbp2030"))
		return PDL_LBP2030;
	return PDL_UNKNOWN;		
}

private void LipsJobStart(PDLMODE pdl, FILE *fp)
{
	if (pdl == PDL_LIPS3FIG || pdl == PDL_LIPS4FIG || pdl == PDL_LIPS4CFIG)
		return;
	fprintf(fp, "%s%%@", LIPS_ESC);
	if (pdl == PDL_LIPS3)
		fprintf(fp, "%s31;300;0JGS:lips3%s", LIPS_DCS, LIPS_ST);
	else if (pdl == PDL_LIPS4)
		fprintf(fp, "%s41;600;0JGS:lips4%s", LIPS_DCS, LIPS_ST);
	else if (pdl == PDL_LIPS4C)
		fprintf(fp, "%s41;360;0JGS:lips4c%s", LIPS_DCS, LIPS_ST);
	else if (pdl == PDL_LBP2030)
		fprintf(fp, "%s41;600;0JGS:lbp2030%s", LIPS_DCS, LIPS_ST);
	if (pdl == PDL_LIPS4C || pdl == PDL_LBP2030)
		fprintf(fp, "%s1\"p", LIPS_CSI);		
	fprintf(fp, "%s<", LIPS_ESC);
	if (pdl == PDL_LIPS4C)
		fprintf(fp, "%s2&z", LIPS_CSI);
	fprintf(fp, "%s?1l", LIPS_CSI);	/* Disable Auto NF */
	fprintf(fp, "%s?2h", LIPS_CSI);	/* Disable Auto FF */
	fprintf(fp, "%s?4l", LIPS_CSI);	/* Disable Auto LF at CR */
	fprintf(fp, "%s?5l", LIPS_CSI);	/* Disable Auto CR at LF */
	fprintf(fp, "%s?6l", LIPS_CSI);	/* Disable Auto CR at FF */
	fprintf(fp, "%s?7l", LIPS_CSI);	/* Disable Auto CR at VT */
	fprintf(fp, "%s?3h", LIPS_CSI);	/* Disable Auto CAP Movement */
	fprintf(fp, "%s11h", LIPS_CSI); /* Size Unit Mode */
	fprintf(fp, "%s2 I", LIPS_CSI); /* SelectSizeUnit */
}

private void LipsJobEnd(PDLMODE pdl, FILE *fp)
{
	if (pdl == PDL_LIPS3FIG || pdl == PDL_LIPS4FIG || pdl == PDL_LIPS4CFIG)
		return;
	fprintf(fp, "%s", LIPS_FF);
	fprintf(fp, "%s0J%s", LIPS_DCS, LIPS_ST);
}
private gx_color_index MapRgbToColor(gx_device *pDev, gx_color_value r, gx_color_value g, gx_color_value b)
{
	return gx_color_value_to_byte(b) + ((uint)gx_color_value_to_byte(g) << 8) + ((ulong)gx_color_value_to_byte(r) << 16);
}

private int MapColorToRgb(gx_device *pDev, gx_color_index color, gx_color_value pRgb[3])
{
	pRgb[0] = gx_color_value_from_byte(color >> 16);
	pRgb[1] = gx_color_value_from_byte((color >> 8) & 0xff);
	pRgb[2] = gx_color_value_from_byte(color & 0xff);
	return 0;
}

private int
LipsLineOrder(byte *pBuff, byte *rBuff, byte *gBuff, byte *bBuff, int Length){
	byte *ptr = pBuff;
	while(ptr < pBuff + Length * 3){
		*rBuff++ = *ptr++; 
		*gBuff++ = *ptr++; 
		*bBuff++ = *ptr++; 
	}
	return 0;
}

#define RLECOUNTMAX 0xff /* 256 times */
private int
LipsRunLengthEncode(byte *inBuff, byte *outBuff, int Length){
	int		i = 0;
	byte	value;
	int		count = 0;
	byte	*ptr = inBuff;

	value = *ptr;
	ptr++;

	while( ptr < inBuff + Length){
		if (*ptr == value){
			count++;
			if (count > RLECOUNTMAX){
				*outBuff++ = RLECOUNTMAX;
				*outBuff++ = value;
				i += 2;
				count = 0;
			}
				
		}else{
			*outBuff++ = count;
			*outBuff++ = value;
			i += 2;
			count = 0;
			value = *ptr;
		}
		ptr++;
	}		
	*outBuff++ = count;
	*outBuff++ = value;
	i += 2;
			
	return i;
}
