// Copyright 2018-2026 the Deno authors. MIT license.
/* Copyright Joyent, Inc. and other Node contributors. All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

// This module ports:
// - https://github.com/libuv/libuv/blob/master/src/win/error.c
//
// Only the constants referenced in op_node_sys_to_uv_error are included.
// The full set of Windows error constants was stripped to reduce compile times.
// Original source: https://raw.githubusercontent.com/mic101/windows/master/WRK-v1.2/public/sdk/inc/winerror.h

use deno_core::op2;

#[op2]
#[string]
pub fn op_node_sys_to_uv_error(err: i32) -> String {
  let uv_err = match err {
    ERROR_NOACCESS => "EACCES",
    WSAEACCES => "EACCES",
    ERROR_CANT_ACCESS_FILE => "EACCES",
    ERROR_ADDRESS_ALREADY_ASSOCIATED => "EADDRINUSE",
    WSAEADDRINUSE => "EADDRINUSE",
    WSAEADDRNOTAVAIL => "EADDRNOTAVAIL",
    WSAEAFNOSUPPORT => "EAFNOSUPPORT",
    WSAEWOULDBLOCK => "EAGAIN",
    WSAEALREADY => "EALREADY",
    ERROR_INVALID_FLAGS => "EBADF",
    ERROR_INVALID_HANDLE => "EBADF",
    ERROR_LOCK_VIOLATION => "EBUSY",
    ERROR_PIPE_BUSY => "EBUSY",
    ERROR_SHARING_VIOLATION => "EBUSY",
    ERROR_OPERATION_ABORTED => "ECANCELED",
    WSAEINTR => "ECANCELED",
    ERROR_NO_UNICODE_TRANSLATION => "ECHARSET",
    ERROR_CONNECTION_ABORTED => "ECONNABORTED",
    WSAECONNABORTED => "ECONNABORTED",
    ERROR_CONNECTION_REFUSED => "ECONNREFUSED",
    WSAECONNREFUSED => "ECONNREFUSED",
    ERROR_NETNAME_DELETED => "ECONNRESET",
    WSAECONNRESET => "ECONNRESET",
    ERROR_ALREADY_EXISTS => "EEXIST",
    ERROR_FILE_EXISTS => "EEXIST",
    ERROR_BUFFER_OVERFLOW => "EFAULT",
    WSAEFAULT => "EFAULT",
    ERROR_HOST_UNREACHABLE => "EHOSTUNREACH",
    WSAEHOSTUNREACH => "EHOSTUNREACH",
    ERROR_INSUFFICIENT_BUFFER => "EINVAL",
    ERROR_INVALID_DATA => "EINVAL",
    ERROR_INVALID_NAME => "ENOENT",
    ERROR_INVALID_PARAMETER => "EINVAL",
    WSAEINVAL => "EINVAL",
    WSAEPFNOSUPPORT => "EINVAL",
    ERROR_NOT_A_REPARSE_POINT => "EINVAL",
    ERROR_BEGINNING_OF_MEDIA => "EIO",
    ERROR_BUS_RESET => "EIO",
    ERROR_CRC => "EIO",
    ERROR_DEVICE_DOOR_OPEN => "EIO",
    ERROR_DEVICE_REQUIRES_CLEANING => "EIO",
    ERROR_DISK_CORRUPT => "EIO",
    ERROR_EOM_OVERFLOW => "EIO",
    ERROR_FILEMARK_DETECTED => "EIO",
    ERROR_GEN_FAILURE => "EIO",
    ERROR_INVALID_BLOCK_LENGTH => "EIO",
    ERROR_IO_DEVICE => "EIO",
    ERROR_NO_DATA_DETECTED => "EIO",
    ERROR_NO_SIGNAL_SENT => "EIO",
    ERROR_OPEN_FAILED => "EIO",
    ERROR_SETMARK_DETECTED => "EIO",
    ERROR_SIGNAL_REFUSED => "EIO",
    WSAEISCONN => "EISCONN",
    ERROR_CANT_RESOLVE_FILENAME => "ELOOP",
    ERROR_TOO_MANY_OPEN_FILES => "EMFILE",
    WSAEMFILE => "EMFILE",
    WSAEMSGSIZE => "EMSGSIZE",
    ERROR_FILENAME_EXCED_RANGE => "ENAMETOOLONG",
    ERROR_NETWORK_UNREACHABLE => "ENETUNREACH",
    WSAENETUNREACH => "ENETUNREACH",
    WSAENOBUFS => "ENOBUFS",
    ERROR_BAD_PATHNAME => "ENOENT",
    ERROR_DIRECTORY => "ENOTDIR",
    ERROR_ENVVAR_NOT_FOUND => "ENOENT",
    ERROR_FILE_NOT_FOUND => "ENOENT",
    ERROR_INVALID_DRIVE => "ENOENT",
    ERROR_INVALID_REPARSE_DATA => "ENOENT",
    ERROR_MOD_NOT_FOUND => "ENOENT",
    ERROR_PATH_NOT_FOUND => "ENOENT",
    WSAHOST_NOT_FOUND => "ENOENT",
    WSANO_DATA => "ENOENT",
    ERROR_NOT_ENOUGH_MEMORY => "ENOMEM",
    ERROR_OUTOFMEMORY => "ENOMEM",
    ERROR_CANNOT_MAKE => "ENOSPC",
    ERROR_DISK_FULL => "ENOSPC",
    ERROR_EA_TABLE_FULL => "ENOSPC",
    ERROR_END_OF_MEDIA => "ENOSPC",
    ERROR_HANDLE_DISK_FULL => "ENOSPC",
    ERROR_NOT_CONNECTED => "ENOTCONN",
    WSAENOTCONN => "ENOTCONN",
    ERROR_DIR_NOT_EMPTY => "ENOTEMPTY",
    WSAENOTSOCK => "ENOTSOCK",
    ERROR_NOT_SUPPORTED => "ENOTSUP",
    ERROR_BROKEN_PIPE => "EOF",
    ERROR_ACCESS_DENIED => "EPERM",
    ERROR_PRIVILEGE_NOT_HELD => "EPERM",
    ERROR_BAD_PIPE => "EPIPE",
    ERROR_NO_DATA => "EPIPE",
    ERROR_PIPE_NOT_CONNECTED => "EPIPE",
    WSAESHUTDOWN => "EPIPE",
    WSAEPROTONOSUPPORT => "EPROTONOSUPPORT",
    ERROR_WRITE_PROTECT => "EROFS",
    ERROR_SEM_TIMEOUT => "ETIMEDOUT",
    WSAETIMEDOUT => "ETIMEDOUT",
    ERROR_NOT_SAME_DEVICE => "EXDEV",
    ERROR_INVALID_FUNCTION => "EISDIR",
    ERROR_META_EXPANSION_TOO_LONG => "E2BIG",
    WSAESOCKTNOSUPPORT => "ESOCKTNOSUPPORT",
    _ => "UNKNOWN",
  };
  uv_err.to_string()
}

// Windows system error codes
const ERROR_INVALID_FUNCTION: i32 = 1;
const ERROR_FILE_NOT_FOUND: i32 = 2;
const ERROR_PATH_NOT_FOUND: i32 = 3;
const ERROR_TOO_MANY_OPEN_FILES: i32 = 4;
const ERROR_ACCESS_DENIED: i32 = 5;
const ERROR_INVALID_HANDLE: i32 = 6;
const ERROR_NOT_ENOUGH_MEMORY: i32 = 8;
const ERROR_INVALID_DATA: i32 = 13;
const ERROR_OUTOFMEMORY: i32 = 14;
const ERROR_INVALID_DRIVE: i32 = 15;
const ERROR_NOT_SAME_DEVICE: i32 = 17;
const ERROR_WRITE_PROTECT: i32 = 19;
const ERROR_CRC: i32 = 23;
const ERROR_GEN_FAILURE: i32 = 31;
const ERROR_SHARING_VIOLATION: i32 = 32;
const ERROR_LOCK_VIOLATION: i32 = 33;
const ERROR_HANDLE_DISK_FULL: i32 = 39;
const ERROR_NOT_SUPPORTED: i32 = 50;
const ERROR_NETNAME_DELETED: i32 = 64;
const ERROR_FILE_EXISTS: i32 = 80;
const ERROR_CANNOT_MAKE: i32 = 82;
const ERROR_INVALID_PARAMETER: i32 = 87;
const ERROR_BROKEN_PIPE: i32 = 109;
const ERROR_OPEN_FAILED: i32 = 110;
const ERROR_BUFFER_OVERFLOW: i32 = 111;
const ERROR_DISK_FULL: i32 = 112;
const ERROR_SEM_TIMEOUT: i32 = 121;
const ERROR_INSUFFICIENT_BUFFER: i32 = 122;
const ERROR_INVALID_NAME: i32 = 123;
const ERROR_MOD_NOT_FOUND: i32 = 126;
const ERROR_DIR_NOT_EMPTY: i32 = 145;
const ERROR_SIGNAL_REFUSED: i32 = 156;
const ERROR_BAD_PATHNAME: i32 = 161;
const ERROR_ALREADY_EXISTS: i32 = 183;
const ERROR_ENVVAR_NOT_FOUND: i32 = 203;
const ERROR_NO_SIGNAL_SENT: i32 = 205;
const ERROR_FILENAME_EXCED_RANGE: i32 = 206;
const ERROR_META_EXPANSION_TOO_LONG: i32 = 208;
const ERROR_BAD_PIPE: i32 = 230;
const ERROR_PIPE_BUSY: i32 = 231;
const ERROR_NO_DATA: i32 = 232;
const ERROR_PIPE_NOT_CONNECTED: i32 = 233;
const ERROR_DIRECTORY: i32 = 267;
const ERROR_EA_TABLE_FULL: i32 = 277;
const ERROR_OPERATION_ABORTED: i32 = 995;
const ERROR_NOACCESS: i32 = 998;
const ERROR_INVALID_FLAGS: i32 = 1004;
const ERROR_END_OF_MEDIA: i32 = 1100;
const ERROR_FILEMARK_DETECTED: i32 = 1101;
const ERROR_BEGINNING_OF_MEDIA: i32 = 1102;
const ERROR_SETMARK_DETECTED: i32 = 1103;
const ERROR_NO_DATA_DETECTED: i32 = 1104;
const ERROR_INVALID_BLOCK_LENGTH: i32 = 1106;
const ERROR_BUS_RESET: i32 = 1111;
const ERROR_NO_UNICODE_TRANSLATION: i32 = 1113;
const ERROR_IO_DEVICE: i32 = 1117;
const ERROR_EOM_OVERFLOW: i32 = 1129;
const ERROR_DEVICE_REQUIRES_CLEANING: i32 = 1165;
const ERROR_DEVICE_DOOR_OPEN: i32 = 1166;
const ERROR_CONNECTION_REFUSED: i32 = 1225;
const ERROR_ADDRESS_ALREADY_ASSOCIATED: i32 = 1227;
const ERROR_NETWORK_UNREACHABLE: i32 = 1231;
const ERROR_HOST_UNREACHABLE: i32 = 1232;
const ERROR_CONNECTION_ABORTED: i32 = 1236;
const ERROR_PRIVILEGE_NOT_HELD: i32 = 1314;
const ERROR_DISK_CORRUPT: i32 = 1393;
const ERROR_CANT_ACCESS_FILE: i32 = 1920;
const ERROR_CANT_RESOLVE_FILENAME: i32 = 1921;
const ERROR_NOT_CONNECTED: i32 = 2250;
const ERROR_NOT_A_REPARSE_POINT: i32 = 4390;
const ERROR_INVALID_REPARSE_DATA: i32 = 4392;

// Winsock error codes
const WSAEINTR: i32 = 10004;
const WSAEACCES: i32 = 10013;
const WSAEFAULT: i32 = 10014;
const WSAEINVAL: i32 = 10022;
const WSAEMFILE: i32 = 10024;
const WSAEWOULDBLOCK: i32 = 10035;
const WSAEALREADY: i32 = 10037;
const WSAENOTSOCK: i32 = 10038;
const WSAEMSGSIZE: i32 = 10040;
const WSAEPROTONOSUPPORT: i32 = 10043;
const WSAESOCKTNOSUPPORT: i32 = 10044;
const WSAEPFNOSUPPORT: i32 = 10046;
const WSAEAFNOSUPPORT: i32 = 10047;
const WSAEADDRINUSE: i32 = 10048;
const WSAEADDRNOTAVAIL: i32 = 10049;
const WSAENETUNREACH: i32 = 10051;
const WSAECONNABORTED: i32 = 10053;
const WSAECONNRESET: i32 = 10054;
const WSAENOBUFS: i32 = 10055;
const WSAEISCONN: i32 = 10056;
const WSAENOTCONN: i32 = 10057;
const WSAESHUTDOWN: i32 = 10058;
const WSAETIMEDOUT: i32 = 10060;
const WSAECONNREFUSED: i32 = 10061;
const WSAEHOSTUNREACH: i32 = 10065;
const WSAHOST_NOT_FOUND: i32 = 11001;
const WSANO_DATA: i32 = 11004;
