// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package core

import (
	"context"
	"fmt"
	"net/url"
)

type (
	// Netrc contains login and initialization information used by
	// an automated login process.
	Netrc struct {
		Machine  string `json:"machine"`
		Login    string `json:"login"`
		Password string `json:"password"`
	}

	// NetrcService returns a valid netrc file that can be used
	// to authenticate and clone a private repository. If
	// authentication is not required or enabled, a nil Netrc
	// file and nil error are returned.
	NetrcService interface {
		Create(context.Context, *User, *Repository) (*Netrc, error)
	}
)

// SetMachine sets the netrc machine from a URL value.
func (n *Netrc) SetMachine(address string) error {
	url, err := url.Parse(address)
	if err != nil {
		return err
	}
	n.Machine = url.Hostname()
	return nil
}

// String returns the string representation of a netrc file.
func (n *Netrc) String() string {
	return fmt.Sprintf("machine %s login %s password %s",
		n.Machine,
		n.Login,
		n.Password,
	)
}
