{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Interface for single-line text editing widgets.
-- 
-- Typical examples of editable widgets are t'GI.Gtk.Objects.Entry.Entry' and
-- t'GI.Gtk.Objects.SpinButton.SpinButton'. It contains functions for generically manipulating
-- an editable widget, a large number of action signals used for key bindings,
-- and several signals that an application can connect to modify the behavior
-- of a widget.
-- 
-- As an example of the latter usage, by connecting the following handler to
-- [Editable::insertText]("GI.Gtk.Interfaces.Editable#g:signal:insertText"), an application can convert all entry
-- into a widget into uppercase.
-- 
-- == Forcing entry to uppercase.
-- 
-- 
-- === /c code/
-- >#include <ctype.h>
-- >
-- >void
-- >insert_text_handler (GtkEditable *editable,
-- >                     const char  *text,
-- >                     int          length,
-- >                     int         *position,
-- >                     gpointer     data)
-- >{
-- >  char *result = g_utf8_strup (text, length);
-- >
-- >  g_signal_handlers_block_by_func (editable,
-- >                               (gpointer) insert_text_handler, data);
-- >  gtk_editable_insert_text (editable, result, length, position);
-- >  g_signal_handlers_unblock_by_func (editable,
-- >                                     (gpointer) insert_text_handler, data);
-- >
-- >  g_signal_stop_emission_by_name (editable, "insert_text");
-- >
-- >  g_free (result);
-- >}
-- 
-- 
-- == Implementing GtkEditable
-- 
-- The most likely scenario for implementing @GtkEditable@ on your own widget
-- is that you will embed a @GtkText@ inside a complex widget, and want to
-- delegate the editable functionality to that text widget. @GtkEditable@
-- provides some utility functions to make this easy.
-- 
-- In your class_init function, call 'GI.Gtk.Interfaces.Editable.editableInstallProperties',
-- passing the first available property ID:
-- 
-- 
-- === /c code/
-- >static void
-- >my_class_init (MyClass *class)
-- >{
-- >  ...
-- >  g_object_class_install_properties (object_class, NUM_PROPERTIES, props);
-- >  gtk_editable_install_properties (object_clas, NUM_PROPERTIES);
-- >  ...
-- >}
-- 
-- 
-- In your interface_init function for the @GtkEditable@ interface, provide
-- an implementation for the get_delegate vfunc that returns your text widget:
-- 
-- 
-- === /c code/
-- >GtkEditable *
-- >get_editable_delegate (GtkEditable *editable)
-- >{
-- >  return GTK_EDITABLE (MY_WIDGET (editable)->text_widget);
-- >}
-- >
-- >static void
-- >my_editable_init (GtkEditableInterface *iface)
-- >{
-- >  iface->get_delegate = get_editable_delegate;
-- >}
-- 
-- 
-- You don\'t need to provide any other vfuncs. The default implementations
-- work by forwarding to the delegate that the GtkEditableInterface.@/get_delegate()/@
-- vfunc returns.
-- 
-- In your instance_init function, create your text widget, and then call
-- 'GI.Gtk.Interfaces.Editable.editableInitDelegate':
-- 
-- 
-- === /c code/
-- >static void
-- >my_widget_init (MyWidget *self)
-- >{
-- >  ...
-- >  self->text_widget = gtk_text_new ();
-- >  gtk_editable_init_delegate (GTK_EDITABLE (self));
-- >  ...
-- >}
-- 
-- 
-- In your dispose function, call 'GI.Gtk.Interfaces.Editable.editableFinishDelegate' before
-- destroying your text widget:
-- 
-- 
-- === /c code/
-- >static void
-- >my_widget_dispose (GObject *object)
-- >{
-- >  ...
-- >  gtk_editable_finish_delegate (GTK_EDITABLE (self));
-- >  g_clear_pointer (&self->text_widget, gtk_widget_unparent);
-- >  ...
-- >}
-- 
-- 
-- Finally, use 'GI.Gtk.Interfaces.Editable.editableDelegateSetProperty' in your @set_property@
-- function (and similar for @get_property@), to set the editable properties:
-- 
-- 
-- === /c code/
-- >  ...
-- >  if (gtk_editable_delegate_set_property (object, prop_id, value, pspec))
-- >    return;
-- >
-- >  switch (prop_id)
-- >  ...
-- 
-- 
-- It is important to note that if you create a @GtkEditable@ that uses
-- a delegate, the low level [Editable::insertText]("GI.Gtk.Interfaces.Editable#g:signal:insertText") and
-- [Editable::deleteText]("GI.Gtk.Interfaces.Editable#g:signal:deleteText") signals will be propagated from the
-- \"wrapper\" editable to the delegate, but they will not be propagated from
-- the delegate to the \"wrapper\" editable, as they would cause an infinite
-- recursion. If you wish to connect to the [Editable::insertText]("GI.Gtk.Interfaces.Editable#g:signal:insertText")
-- and [Editable::deleteText]("GI.Gtk.Interfaces.Editable#g:signal:deleteText") signals, you will need to connect
-- to them on the delegate obtained via 'GI.Gtk.Interfaces.Editable.editableGetDelegate'.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Interfaces.Editable
    ( 

-- * Exported types
    Editable(..)                            ,
    IsEditable                              ,
    toEditable                              ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [actionSetEnabled]("GI.Gtk.Objects.Widget#g:method:actionSetEnabled"), [activate]("GI.Gtk.Objects.Widget#g:method:activate"), [activateAction]("GI.Gtk.Objects.Widget#g:method:activateAction"), [activateDefault]("GI.Gtk.Objects.Widget#g:method:activateDefault"), [addController]("GI.Gtk.Objects.Widget#g:method:addController"), [addCssClass]("GI.Gtk.Objects.Widget#g:method:addCssClass"), [addMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:addMnemonicLabel"), [addTickCallback]("GI.Gtk.Objects.Widget#g:method:addTickCallback"), [allocate]("GI.Gtk.Objects.Widget#g:method:allocate"), [announce]("GI.Gtk.Interfaces.Accessible#g:method:announce"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [childFocus]("GI.Gtk.Objects.Widget#g:method:childFocus"), [computeBounds]("GI.Gtk.Objects.Widget#g:method:computeBounds"), [computeExpand]("GI.Gtk.Objects.Widget#g:method:computeExpand"), [computePoint]("GI.Gtk.Objects.Widget#g:method:computePoint"), [computeTransform]("GI.Gtk.Objects.Widget#g:method:computeTransform"), [contains]("GI.Gtk.Objects.Widget#g:method:contains"), [createPangoContext]("GI.Gtk.Objects.Widget#g:method:createPangoContext"), [createPangoLayout]("GI.Gtk.Objects.Widget#g:method:createPangoLayout"), [delegateGetAccessiblePlatformState]("GI.Gtk.Interfaces.Editable#g:method:delegateGetAccessiblePlatformState"), [deleteSelection]("GI.Gtk.Interfaces.Editable#g:method:deleteSelection"), [deleteText]("GI.Gtk.Interfaces.Editable#g:method:deleteText"), [disposeTemplate]("GI.Gtk.Objects.Widget#g:method:disposeTemplate"), [dragCheckThreshold]("GI.Gtk.Objects.Widget#g:method:dragCheckThreshold"), [errorBell]("GI.Gtk.Objects.Widget#g:method:errorBell"), [finishDelegate]("GI.Gtk.Interfaces.Editable#g:method:finishDelegate"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [grabFocus]("GI.Gtk.Objects.Widget#g:method:grabFocus"), [hasCssClass]("GI.Gtk.Objects.Widget#g:method:hasCssClass"), [hasDefault]("GI.Gtk.Objects.Widget#g:method:hasDefault"), [hasFocus]("GI.Gtk.Objects.Widget#g:method:hasFocus"), [hasVisibleFocus]("GI.Gtk.Objects.Widget#g:method:hasVisibleFocus"), [hide]("GI.Gtk.Objects.Widget#g:method:hide"), [inDestruction]("GI.Gtk.Objects.Widget#g:method:inDestruction"), [initDelegate]("GI.Gtk.Interfaces.Editable#g:method:initDelegate"), [initTemplate]("GI.Gtk.Objects.Widget#g:method:initTemplate"), [insertActionGroup]("GI.Gtk.Objects.Widget#g:method:insertActionGroup"), [insertAfter]("GI.Gtk.Objects.Widget#g:method:insertAfter"), [insertBefore]("GI.Gtk.Objects.Widget#g:method:insertBefore"), [insertText]("GI.Gtk.Interfaces.Editable#g:method:insertText"), [isAncestor]("GI.Gtk.Objects.Widget#g:method:isAncestor"), [isDrawable]("GI.Gtk.Objects.Widget#g:method:isDrawable"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isFocus]("GI.Gtk.Objects.Widget#g:method:isFocus"), [isSensitive]("GI.Gtk.Objects.Widget#g:method:isSensitive"), [isVisible]("GI.Gtk.Objects.Widget#g:method:isVisible"), [keynavFailed]("GI.Gtk.Objects.Widget#g:method:keynavFailed"), [listMnemonicLabels]("GI.Gtk.Objects.Widget#g:method:listMnemonicLabels"), [map]("GI.Gtk.Objects.Widget#g:method:map"), [measure]("GI.Gtk.Objects.Widget#g:method:measure"), [mnemonicActivate]("GI.Gtk.Objects.Widget#g:method:mnemonicActivate"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [observeChildren]("GI.Gtk.Objects.Widget#g:method:observeChildren"), [observeControllers]("GI.Gtk.Objects.Widget#g:method:observeControllers"), [pick]("GI.Gtk.Objects.Widget#g:method:pick"), [queueAllocate]("GI.Gtk.Objects.Widget#g:method:queueAllocate"), [queueDraw]("GI.Gtk.Objects.Widget#g:method:queueDraw"), [queueResize]("GI.Gtk.Objects.Widget#g:method:queueResize"), [realize]("GI.Gtk.Objects.Widget#g:method:realize"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeController]("GI.Gtk.Objects.Widget#g:method:removeController"), [removeCssClass]("GI.Gtk.Objects.Widget#g:method:removeCssClass"), [removeMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:removeMnemonicLabel"), [removeTickCallback]("GI.Gtk.Objects.Widget#g:method:removeTickCallback"), [resetProperty]("GI.Gtk.Interfaces.Accessible#g:method:resetProperty"), [resetRelation]("GI.Gtk.Interfaces.Accessible#g:method:resetRelation"), [resetState]("GI.Gtk.Interfaces.Accessible#g:method:resetState"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [selectRegion]("GI.Gtk.Interfaces.Editable#g:method:selectRegion"), [shouldLayout]("GI.Gtk.Objects.Widget#g:method:shouldLayout"), [show]("GI.Gtk.Objects.Widget#g:method:show"), [sizeAllocate]("GI.Gtk.Objects.Widget#g:method:sizeAllocate"), [snapshotChild]("GI.Gtk.Objects.Widget#g:method:snapshotChild"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [translateCoordinates]("GI.Gtk.Objects.Widget#g:method:translateCoordinates"), [triggerTooltipQuery]("GI.Gtk.Objects.Widget#g:method:triggerTooltipQuery"), [unmap]("GI.Gtk.Objects.Widget#g:method:unmap"), [unparent]("GI.Gtk.Objects.Widget#g:method:unparent"), [unrealize]("GI.Gtk.Objects.Widget#g:method:unrealize"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [unsetStateFlags]("GI.Gtk.Objects.Widget#g:method:unsetStateFlags"), [updateNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:updateNextAccessibleSibling"), [updatePlatformState]("GI.Gtk.Interfaces.Accessible#g:method:updatePlatformState"), [updateProperty]("GI.Gtk.Interfaces.Accessible#g:method:updateProperty"), [updateRelation]("GI.Gtk.Interfaces.Accessible#g:method:updateRelation"), [updateState]("GI.Gtk.Interfaces.Accessible#g:method:updateState"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleParent"), [getAccessibleRole]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleRole"), [getAlignment]("GI.Gtk.Interfaces.Editable#g:method:getAlignment"), [getAllocatedBaseline]("GI.Gtk.Objects.Widget#g:method:getAllocatedBaseline"), [getAllocatedHeight]("GI.Gtk.Objects.Widget#g:method:getAllocatedHeight"), [getAllocatedWidth]("GI.Gtk.Objects.Widget#g:method:getAllocatedWidth"), [getAllocation]("GI.Gtk.Objects.Widget#g:method:getAllocation"), [getAncestor]("GI.Gtk.Objects.Widget#g:method:getAncestor"), [getAtContext]("GI.Gtk.Interfaces.Accessible#g:method:getAtContext"), [getBaseline]("GI.Gtk.Objects.Widget#g:method:getBaseline"), [getBounds]("GI.Gtk.Interfaces.Accessible#g:method:getBounds"), [getBuildableId]("GI.Gtk.Interfaces.Buildable#g:method:getBuildableId"), [getCanFocus]("GI.Gtk.Objects.Widget#g:method:getCanFocus"), [getCanTarget]("GI.Gtk.Objects.Widget#g:method:getCanTarget"), [getChars]("GI.Gtk.Interfaces.Editable#g:method:getChars"), [getChildVisible]("GI.Gtk.Objects.Widget#g:method:getChildVisible"), [getClipboard]("GI.Gtk.Objects.Widget#g:method:getClipboard"), [getColor]("GI.Gtk.Objects.Widget#g:method:getColor"), [getCssClasses]("GI.Gtk.Objects.Widget#g:method:getCssClasses"), [getCssName]("GI.Gtk.Objects.Widget#g:method:getCssName"), [getCursor]("GI.Gtk.Objects.Widget#g:method:getCursor"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDelegate]("GI.Gtk.Interfaces.Editable#g:method:getDelegate"), [getDirection]("GI.Gtk.Objects.Widget#g:method:getDirection"), [getDisplay]("GI.Gtk.Objects.Widget#g:method:getDisplay"), [getEditable]("GI.Gtk.Interfaces.Editable#g:method:getEditable"), [getEnableUndo]("GI.Gtk.Interfaces.Editable#g:method:getEnableUndo"), [getFirstAccessibleChild]("GI.Gtk.Interfaces.Accessible#g:method:getFirstAccessibleChild"), [getFirstChild]("GI.Gtk.Objects.Widget#g:method:getFirstChild"), [getFocusChild]("GI.Gtk.Objects.Widget#g:method:getFocusChild"), [getFocusOnClick]("GI.Gtk.Objects.Widget#g:method:getFocusOnClick"), [getFocusable]("GI.Gtk.Objects.Widget#g:method:getFocusable"), [getFontMap]("GI.Gtk.Objects.Widget#g:method:getFontMap"), [getFontOptions]("GI.Gtk.Objects.Widget#g:method:getFontOptions"), [getFrameClock]("GI.Gtk.Objects.Widget#g:method:getFrameClock"), [getHalign]("GI.Gtk.Objects.Widget#g:method:getHalign"), [getHasTooltip]("GI.Gtk.Objects.Widget#g:method:getHasTooltip"), [getHeight]("GI.Gtk.Objects.Widget#g:method:getHeight"), [getHexpand]("GI.Gtk.Objects.Widget#g:method:getHexpand"), [getHexpandSet]("GI.Gtk.Objects.Widget#g:method:getHexpandSet"), [getLastChild]("GI.Gtk.Objects.Widget#g:method:getLastChild"), [getLayoutManager]("GI.Gtk.Objects.Widget#g:method:getLayoutManager"), [getLimitEvents]("GI.Gtk.Objects.Widget#g:method:getLimitEvents"), [getMapped]("GI.Gtk.Objects.Widget#g:method:getMapped"), [getMarginBottom]("GI.Gtk.Objects.Widget#g:method:getMarginBottom"), [getMarginEnd]("GI.Gtk.Objects.Widget#g:method:getMarginEnd"), [getMarginStart]("GI.Gtk.Objects.Widget#g:method:getMarginStart"), [getMarginTop]("GI.Gtk.Objects.Widget#g:method:getMarginTop"), [getMaxWidthChars]("GI.Gtk.Interfaces.Editable#g:method:getMaxWidthChars"), [getName]("GI.Gtk.Objects.Widget#g:method:getName"), [getNative]("GI.Gtk.Objects.Widget#g:method:getNative"), [getNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:getNextAccessibleSibling"), [getNextSibling]("GI.Gtk.Objects.Widget#g:method:getNextSibling"), [getOpacity]("GI.Gtk.Objects.Widget#g:method:getOpacity"), [getOverflow]("GI.Gtk.Objects.Widget#g:method:getOverflow"), [getPangoContext]("GI.Gtk.Objects.Widget#g:method:getPangoContext"), [getParent]("GI.Gtk.Objects.Widget#g:method:getParent"), [getPlatformState]("GI.Gtk.Interfaces.Accessible#g:method:getPlatformState"), [getPosition]("GI.Gtk.Interfaces.Editable#g:method:getPosition"), [getPreferredSize]("GI.Gtk.Objects.Widget#g:method:getPreferredSize"), [getPrevSibling]("GI.Gtk.Objects.Widget#g:method:getPrevSibling"), [getPrimaryClipboard]("GI.Gtk.Objects.Widget#g:method:getPrimaryClipboard"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRealized]("GI.Gtk.Objects.Widget#g:method:getRealized"), [getReceivesDefault]("GI.Gtk.Objects.Widget#g:method:getReceivesDefault"), [getRequestMode]("GI.Gtk.Objects.Widget#g:method:getRequestMode"), [getRoot]("GI.Gtk.Objects.Widget#g:method:getRoot"), [getScaleFactor]("GI.Gtk.Objects.Widget#g:method:getScaleFactor"), [getSelectionBounds]("GI.Gtk.Interfaces.Editable#g:method:getSelectionBounds"), [getSensitive]("GI.Gtk.Objects.Widget#g:method:getSensitive"), [getSettings]("GI.Gtk.Objects.Widget#g:method:getSettings"), [getSize]("GI.Gtk.Objects.Widget#g:method:getSize"), [getSizeRequest]("GI.Gtk.Objects.Widget#g:method:getSizeRequest"), [getStateFlags]("GI.Gtk.Objects.Widget#g:method:getStateFlags"), [getStyleContext]("GI.Gtk.Objects.Widget#g:method:getStyleContext"), [getTemplateChild]("GI.Gtk.Objects.Widget#g:method:getTemplateChild"), [getText]("GI.Gtk.Interfaces.Editable#g:method:getText"), [getTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:getTooltipMarkup"), [getTooltipText]("GI.Gtk.Objects.Widget#g:method:getTooltipText"), [getValign]("GI.Gtk.Objects.Widget#g:method:getValign"), [getVexpand]("GI.Gtk.Objects.Widget#g:method:getVexpand"), [getVexpandSet]("GI.Gtk.Objects.Widget#g:method:getVexpandSet"), [getVisible]("GI.Gtk.Objects.Widget#g:method:getVisible"), [getWidth]("GI.Gtk.Objects.Widget#g:method:getWidth"), [getWidthChars]("GI.Gtk.Interfaces.Editable#g:method:getWidthChars").
-- 
-- ==== Setters
-- [setAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:setAccessibleParent"), [setAlignment]("GI.Gtk.Interfaces.Editable#g:method:setAlignment"), [setCanFocus]("GI.Gtk.Objects.Widget#g:method:setCanFocus"), [setCanTarget]("GI.Gtk.Objects.Widget#g:method:setCanTarget"), [setChildVisible]("GI.Gtk.Objects.Widget#g:method:setChildVisible"), [setCssClasses]("GI.Gtk.Objects.Widget#g:method:setCssClasses"), [setCursor]("GI.Gtk.Objects.Widget#g:method:setCursor"), [setCursorFromName]("GI.Gtk.Objects.Widget#g:method:setCursorFromName"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDirection]("GI.Gtk.Objects.Widget#g:method:setDirection"), [setEditable]("GI.Gtk.Interfaces.Editable#g:method:setEditable"), [setEnableUndo]("GI.Gtk.Interfaces.Editable#g:method:setEnableUndo"), [setFocusChild]("GI.Gtk.Objects.Widget#g:method:setFocusChild"), [setFocusOnClick]("GI.Gtk.Objects.Widget#g:method:setFocusOnClick"), [setFocusable]("GI.Gtk.Objects.Widget#g:method:setFocusable"), [setFontMap]("GI.Gtk.Objects.Widget#g:method:setFontMap"), [setFontOptions]("GI.Gtk.Objects.Widget#g:method:setFontOptions"), [setHalign]("GI.Gtk.Objects.Widget#g:method:setHalign"), [setHasTooltip]("GI.Gtk.Objects.Widget#g:method:setHasTooltip"), [setHexpand]("GI.Gtk.Objects.Widget#g:method:setHexpand"), [setHexpandSet]("GI.Gtk.Objects.Widget#g:method:setHexpandSet"), [setLayoutManager]("GI.Gtk.Objects.Widget#g:method:setLayoutManager"), [setLimitEvents]("GI.Gtk.Objects.Widget#g:method:setLimitEvents"), [setMarginBottom]("GI.Gtk.Objects.Widget#g:method:setMarginBottom"), [setMarginEnd]("GI.Gtk.Objects.Widget#g:method:setMarginEnd"), [setMarginStart]("GI.Gtk.Objects.Widget#g:method:setMarginStart"), [setMarginTop]("GI.Gtk.Objects.Widget#g:method:setMarginTop"), [setMaxWidthChars]("GI.Gtk.Interfaces.Editable#g:method:setMaxWidthChars"), [setName]("GI.Gtk.Objects.Widget#g:method:setName"), [setOpacity]("GI.Gtk.Objects.Widget#g:method:setOpacity"), [setOverflow]("GI.Gtk.Objects.Widget#g:method:setOverflow"), [setParent]("GI.Gtk.Objects.Widget#g:method:setParent"), [setPosition]("GI.Gtk.Interfaces.Editable#g:method:setPosition"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setReceivesDefault]("GI.Gtk.Objects.Widget#g:method:setReceivesDefault"), [setSensitive]("GI.Gtk.Objects.Widget#g:method:setSensitive"), [setSizeRequest]("GI.Gtk.Objects.Widget#g:method:setSizeRequest"), [setStateFlags]("GI.Gtk.Objects.Widget#g:method:setStateFlags"), [setText]("GI.Gtk.Interfaces.Editable#g:method:setText"), [setTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:setTooltipMarkup"), [setTooltipText]("GI.Gtk.Objects.Widget#g:method:setTooltipText"), [setValign]("GI.Gtk.Objects.Widget#g:method:setValign"), [setVexpand]("GI.Gtk.Objects.Widget#g:method:setVexpand"), [setVexpandSet]("GI.Gtk.Objects.Widget#g:method:setVexpandSet"), [setVisible]("GI.Gtk.Objects.Widget#g:method:setVisible"), [setWidthChars]("GI.Gtk.Interfaces.Editable#g:method:setWidthChars").

#if defined(ENABLE_OVERLOADING)
    ResolveEditableMethod                   ,
#endif

-- ** delegateGetAccessiblePlatformState #method:delegateGetAccessiblePlatformState#

#if defined(ENABLE_OVERLOADING)
    EditableDelegateGetAccessiblePlatformStateMethodInfo,
#endif
    editableDelegateGetAccessiblePlatformState,


-- ** delegateGetProperty #method:delegateGetProperty#

    editableDelegateGetProperty             ,


-- ** delegateSetProperty #method:delegateSetProperty#

    editableDelegateSetProperty             ,


-- ** deleteSelection #method:deleteSelection#

#if defined(ENABLE_OVERLOADING)
    EditableDeleteSelectionMethodInfo       ,
#endif
    editableDeleteSelection                 ,


-- ** deleteText #method:deleteText#

#if defined(ENABLE_OVERLOADING)
    EditableDeleteTextMethodInfo            ,
#endif
    editableDeleteText                      ,


-- ** finishDelegate #method:finishDelegate#

#if defined(ENABLE_OVERLOADING)
    EditableFinishDelegateMethodInfo        ,
#endif
    editableFinishDelegate                  ,


-- ** getAlignment #method:getAlignment#

#if defined(ENABLE_OVERLOADING)
    EditableGetAlignmentMethodInfo          ,
#endif
    editableGetAlignment                    ,


-- ** getChars #method:getChars#

#if defined(ENABLE_OVERLOADING)
    EditableGetCharsMethodInfo              ,
#endif
    editableGetChars                        ,


-- ** getDelegate #method:getDelegate#

#if defined(ENABLE_OVERLOADING)
    EditableGetDelegateMethodInfo           ,
#endif
    editableGetDelegate                     ,


-- ** getEditable #method:getEditable#

#if defined(ENABLE_OVERLOADING)
    EditableGetEditableMethodInfo           ,
#endif
    editableGetEditable                     ,


-- ** getEnableUndo #method:getEnableUndo#

#if defined(ENABLE_OVERLOADING)
    EditableGetEnableUndoMethodInfo         ,
#endif
    editableGetEnableUndo                   ,


-- ** getMaxWidthChars #method:getMaxWidthChars#

#if defined(ENABLE_OVERLOADING)
    EditableGetMaxWidthCharsMethodInfo      ,
#endif
    editableGetMaxWidthChars                ,


-- ** getPosition #method:getPosition#

#if defined(ENABLE_OVERLOADING)
    EditableGetPositionMethodInfo           ,
#endif
    editableGetPosition                     ,


-- ** getSelectionBounds #method:getSelectionBounds#

#if defined(ENABLE_OVERLOADING)
    EditableGetSelectionBoundsMethodInfo    ,
#endif
    editableGetSelectionBounds              ,


-- ** getText #method:getText#

#if defined(ENABLE_OVERLOADING)
    EditableGetTextMethodInfo               ,
#endif
    editableGetText                         ,


-- ** getWidthChars #method:getWidthChars#

#if defined(ENABLE_OVERLOADING)
    EditableGetWidthCharsMethodInfo         ,
#endif
    editableGetWidthChars                   ,


-- ** initDelegate #method:initDelegate#

#if defined(ENABLE_OVERLOADING)
    EditableInitDelegateMethodInfo          ,
#endif
    editableInitDelegate                    ,


-- ** insertText #method:insertText#

#if defined(ENABLE_OVERLOADING)
    EditableInsertTextMethodInfo            ,
#endif
    editableInsertText                      ,


-- ** installProperties #method:installProperties#

    editableInstallProperties               ,


-- ** selectRegion #method:selectRegion#

#if defined(ENABLE_OVERLOADING)
    EditableSelectRegionMethodInfo          ,
#endif
    editableSelectRegion                    ,


-- ** setAlignment #method:setAlignment#

#if defined(ENABLE_OVERLOADING)
    EditableSetAlignmentMethodInfo          ,
#endif
    editableSetAlignment                    ,


-- ** setEditable #method:setEditable#

#if defined(ENABLE_OVERLOADING)
    EditableSetEditableMethodInfo           ,
#endif
    editableSetEditable                     ,


-- ** setEnableUndo #method:setEnableUndo#

#if defined(ENABLE_OVERLOADING)
    EditableSetEnableUndoMethodInfo         ,
#endif
    editableSetEnableUndo                   ,


-- ** setMaxWidthChars #method:setMaxWidthChars#

#if defined(ENABLE_OVERLOADING)
    EditableSetMaxWidthCharsMethodInfo      ,
#endif
    editableSetMaxWidthChars                ,


-- ** setPosition #method:setPosition#

#if defined(ENABLE_OVERLOADING)
    EditableSetPositionMethodInfo           ,
#endif
    editableSetPosition                     ,


-- ** setText #method:setText#

#if defined(ENABLE_OVERLOADING)
    EditableSetTextMethodInfo               ,
#endif
    editableSetText                         ,


-- ** setWidthChars #method:setWidthChars#

#if defined(ENABLE_OVERLOADING)
    EditableSetWidthCharsMethodInfo         ,
#endif
    editableSetWidthChars                   ,




 -- * Properties


-- ** cursorPosition #attr:cursorPosition#
-- | The current position of the insertion cursor in chars.

#if defined(ENABLE_OVERLOADING)
    EditableCursorPositionPropertyInfo      ,
#endif
#if defined(ENABLE_OVERLOADING)
    editableCursorPosition                  ,
#endif
    getEditableCursorPosition               ,


-- ** editable #attr:editable#
-- | Whether the entry contents can be edited.

#if defined(ENABLE_OVERLOADING)
    EditableEditablePropertyInfo            ,
#endif
    constructEditableEditable               ,
#if defined(ENABLE_OVERLOADING)
    editableEditable                        ,
#endif
    getEditableEditable                     ,
    setEditableEditable                     ,


-- ** enableUndo #attr:enableUndo#
-- | If undo\/redo should be enabled for the editable.

#if defined(ENABLE_OVERLOADING)
    EditableEnableUndoPropertyInfo          ,
#endif
    constructEditableEnableUndo             ,
#if defined(ENABLE_OVERLOADING)
    editableEnableUndo                      ,
#endif
    getEditableEnableUndo                   ,
    setEditableEnableUndo                   ,


-- ** maxWidthChars #attr:maxWidthChars#
-- | The desired maximum width of the entry, in characters.

#if defined(ENABLE_OVERLOADING)
    EditableMaxWidthCharsPropertyInfo       ,
#endif
    constructEditableMaxWidthChars          ,
#if defined(ENABLE_OVERLOADING)
    editableMaxWidthChars                   ,
#endif
    getEditableMaxWidthChars                ,
    setEditableMaxWidthChars                ,


-- ** selectionBound #attr:selectionBound#
-- | The position of the opposite end of the selection from the cursor in chars.

#if defined(ENABLE_OVERLOADING)
    EditableSelectionBoundPropertyInfo      ,
#endif
#if defined(ENABLE_OVERLOADING)
    editableSelectionBound                  ,
#endif
    getEditableSelectionBound               ,


-- ** text #attr:text#
-- | The contents of the entry.

#if defined(ENABLE_OVERLOADING)
    EditableTextPropertyInfo                ,
#endif
    constructEditableText                   ,
#if defined(ENABLE_OVERLOADING)
    editableText                            ,
#endif
    getEditableText                         ,
    setEditableText                         ,


-- ** widthChars #attr:widthChars#
-- | Number of characters to leave space for in the entry.

#if defined(ENABLE_OVERLOADING)
    EditableWidthCharsPropertyInfo          ,
#endif
    constructEditableWidthChars             ,
#if defined(ENABLE_OVERLOADING)
    editableWidthChars                      ,
#endif
    getEditableWidthChars                   ,
    setEditableWidthChars                   ,


-- ** xalign #attr:xalign#
-- | The horizontal alignment, from 0 (left) to 1 (right).
-- 
-- Reversed for RTL layouts.

#if defined(ENABLE_OVERLOADING)
    EditableXalignPropertyInfo              ,
#endif
    constructEditableXalign                 ,
#if defined(ENABLE_OVERLOADING)
    editableXalign                          ,
#endif
    getEditableXalign                       ,
    setEditableXalign                       ,




 -- * Signals


-- ** changed #signal:changed#

    EditableChangedCallback                 ,
#if defined(ENABLE_OVERLOADING)
    EditableChangedSignalInfo               ,
#endif
    afterEditableChanged                    ,
    onEditableChanged                       ,


-- ** deleteText #signal:deleteText#

    EditableDeleteTextCallback              ,
#if defined(ENABLE_OVERLOADING)
    EditableDeleteTextSignalInfo            ,
#endif
    afterEditableDeleteText                 ,
    onEditableDeleteText                    ,


-- ** insertText #signal:insertText#

    EditableInsertTextCallback              ,
#if defined(ENABLE_OVERLOADING)
    EditableInsertTextSignalInfo            ,
#endif
    afterEditableInsertText                 ,
    onEditableInsertText                    ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.ObjectClass as GObject.ObjectClass
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.ComponentTransfer as Gsk.ComponentTransfer
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.GObject.Structs.ObjectClass as GObject.ObjectClass
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget

#endif

-- interface Editable 
-- | Memory-managed wrapper type.
newtype Editable = Editable (SP.ManagedPtr Editable)
    deriving (Eq)

instance SP.ManagedPtrNewtype Editable where
    toManagedPtr (Editable p) = p

foreign import ccall "gtk_editable_get_type"
    c_gtk_editable_get_type :: IO B.Types.GType

instance B.Types.TypedObject Editable where
    glibType = c_gtk_editable_get_type

instance B.Types.GObject Editable

-- | Type class for types which can be safely cast to t'Editable', for instance with `toEditable`.
class (SP.GObject o, O.IsDescendantOf Editable o) => IsEditable o
instance (SP.GObject o, O.IsDescendantOf Editable o) => IsEditable o

instance O.HasParentTypes Editable
type instance O.ParentTypes Editable = '[GObject.Object.Object, Gtk.Widget.Widget]

-- | Cast to t'Editable', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toEditable :: (MIO.MonadIO m, IsEditable o) => o -> m Editable
toEditable = MIO.liftIO . B.ManagedPtr.unsafeCastTo Editable

-- | Convert t'Editable' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Editable) where
    gvalueGType_ = c_gtk_editable_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Editable)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Editable)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Editable ptr
        else return P.Nothing
        
    

-- VVV Prop "cursor-position"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@cursor-position@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' editable #cursorPosition
-- @
getEditableCursorPosition :: (MonadIO m, IsEditable o) => o -> m Int32
getEditableCursorPosition obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "cursor-position"

#if defined(ENABLE_OVERLOADING)
data EditableCursorPositionPropertyInfo
instance AttrInfo EditableCursorPositionPropertyInfo where
    type AttrAllowedOps EditableCursorPositionPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint EditableCursorPositionPropertyInfo = IsEditable
    type AttrSetTypeConstraint EditableCursorPositionPropertyInfo = (~) ()
    type AttrTransferTypeConstraint EditableCursorPositionPropertyInfo = (~) ()
    type AttrTransferType EditableCursorPositionPropertyInfo = ()
    type AttrGetType EditableCursorPositionPropertyInfo = Int32
    type AttrLabel EditableCursorPositionPropertyInfo = "cursor-position"
    type AttrOrigin EditableCursorPositionPropertyInfo = Editable
    attrGet = getEditableCursorPosition
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.cursorPosition"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:attr:cursorPosition"
        })
#endif

-- VVV Prop "editable"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@editable@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' editable #editable
-- @
getEditableEditable :: (MonadIO m, IsEditable o) => o -> m Bool
getEditableEditable obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "editable"

-- | Set the value of the “@editable@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' editable [ #editable 'Data.GI.Base.Attributes.:=' value ]
-- @
setEditableEditable :: (MonadIO m, IsEditable o) => o -> Bool -> m ()
setEditableEditable obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "editable" val

-- | Construct a t'GValueConstruct' with valid value for the “@editable@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructEditableEditable :: (IsEditable o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructEditableEditable val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "editable" val

#if defined(ENABLE_OVERLOADING)
data EditableEditablePropertyInfo
instance AttrInfo EditableEditablePropertyInfo where
    type AttrAllowedOps EditableEditablePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint EditableEditablePropertyInfo = IsEditable
    type AttrSetTypeConstraint EditableEditablePropertyInfo = (~) Bool
    type AttrTransferTypeConstraint EditableEditablePropertyInfo = (~) Bool
    type AttrTransferType EditableEditablePropertyInfo = Bool
    type AttrGetType EditableEditablePropertyInfo = Bool
    type AttrLabel EditableEditablePropertyInfo = "editable"
    type AttrOrigin EditableEditablePropertyInfo = Editable
    attrGet = getEditableEditable
    attrSet = setEditableEditable
    attrPut = setEditableEditable
    attrTransfer _ v = do
        return v
    attrConstruct = constructEditableEditable
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editable"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:attr:editable"
        })
#endif

-- VVV Prop "enable-undo"
   -- Type: TBasicType TBoolean
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@enable-undo@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' editable #enableUndo
-- @
getEditableEnableUndo :: (MonadIO m, IsEditable o) => o -> m Bool
getEditableEnableUndo obj = MIO.liftIO $ B.Properties.getObjectPropertyBool obj "enable-undo"

-- | Set the value of the “@enable-undo@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' editable [ #enableUndo 'Data.GI.Base.Attributes.:=' value ]
-- @
setEditableEnableUndo :: (MonadIO m, IsEditable o) => o -> Bool -> m ()
setEditableEnableUndo obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyBool obj "enable-undo" val

-- | Construct a t'GValueConstruct' with valid value for the “@enable-undo@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructEditableEnableUndo :: (IsEditable o, MIO.MonadIO m) => Bool -> m (GValueConstruct o)
constructEditableEnableUndo val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBool "enable-undo" val

#if defined(ENABLE_OVERLOADING)
data EditableEnableUndoPropertyInfo
instance AttrInfo EditableEnableUndoPropertyInfo where
    type AttrAllowedOps EditableEnableUndoPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint EditableEnableUndoPropertyInfo = IsEditable
    type AttrSetTypeConstraint EditableEnableUndoPropertyInfo = (~) Bool
    type AttrTransferTypeConstraint EditableEnableUndoPropertyInfo = (~) Bool
    type AttrTransferType EditableEnableUndoPropertyInfo = Bool
    type AttrGetType EditableEnableUndoPropertyInfo = Bool
    type AttrLabel EditableEnableUndoPropertyInfo = "enable-undo"
    type AttrOrigin EditableEnableUndoPropertyInfo = Editable
    attrGet = getEditableEnableUndo
    attrSet = setEditableEnableUndo
    attrPut = setEditableEnableUndo
    attrTransfer _ v = do
        return v
    attrConstruct = constructEditableEnableUndo
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.enableUndo"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:attr:enableUndo"
        })
#endif

-- VVV Prop "max-width-chars"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@max-width-chars@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' editable #maxWidthChars
-- @
getEditableMaxWidthChars :: (MonadIO m, IsEditable o) => o -> m Int32
getEditableMaxWidthChars obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "max-width-chars"

-- | Set the value of the “@max-width-chars@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' editable [ #maxWidthChars 'Data.GI.Base.Attributes.:=' value ]
-- @
setEditableMaxWidthChars :: (MonadIO m, IsEditable o) => o -> Int32 -> m ()
setEditableMaxWidthChars obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "max-width-chars" val

-- | Construct a t'GValueConstruct' with valid value for the “@max-width-chars@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructEditableMaxWidthChars :: (IsEditable o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructEditableMaxWidthChars val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "max-width-chars" val

#if defined(ENABLE_OVERLOADING)
data EditableMaxWidthCharsPropertyInfo
instance AttrInfo EditableMaxWidthCharsPropertyInfo where
    type AttrAllowedOps EditableMaxWidthCharsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint EditableMaxWidthCharsPropertyInfo = IsEditable
    type AttrSetTypeConstraint EditableMaxWidthCharsPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint EditableMaxWidthCharsPropertyInfo = (~) Int32
    type AttrTransferType EditableMaxWidthCharsPropertyInfo = Int32
    type AttrGetType EditableMaxWidthCharsPropertyInfo = Int32
    type AttrLabel EditableMaxWidthCharsPropertyInfo = "max-width-chars"
    type AttrOrigin EditableMaxWidthCharsPropertyInfo = Editable
    attrGet = getEditableMaxWidthChars
    attrSet = setEditableMaxWidthChars
    attrPut = setEditableMaxWidthChars
    attrTransfer _ v = do
        return v
    attrConstruct = constructEditableMaxWidthChars
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.maxWidthChars"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:attr:maxWidthChars"
        })
#endif

-- VVV Prop "selection-bound"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@selection-bound@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' editable #selectionBound
-- @
getEditableSelectionBound :: (MonadIO m, IsEditable o) => o -> m Int32
getEditableSelectionBound obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "selection-bound"

#if defined(ENABLE_OVERLOADING)
data EditableSelectionBoundPropertyInfo
instance AttrInfo EditableSelectionBoundPropertyInfo where
    type AttrAllowedOps EditableSelectionBoundPropertyInfo = '[ 'AttrGet]
    type AttrBaseTypeConstraint EditableSelectionBoundPropertyInfo = IsEditable
    type AttrSetTypeConstraint EditableSelectionBoundPropertyInfo = (~) ()
    type AttrTransferTypeConstraint EditableSelectionBoundPropertyInfo = (~) ()
    type AttrTransferType EditableSelectionBoundPropertyInfo = ()
    type AttrGetType EditableSelectionBoundPropertyInfo = Int32
    type AttrLabel EditableSelectionBoundPropertyInfo = "selection-bound"
    type AttrOrigin EditableSelectionBoundPropertyInfo = Editable
    attrGet = getEditableSelectionBound
    attrSet = undefined
    attrPut = undefined
    attrTransfer _ = undefined
    attrConstruct = undefined
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.selectionBound"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:attr:selectionBound"
        })
#endif

-- VVV Prop "text"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@text@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' editable #text
-- @
getEditableText :: (MonadIO m, IsEditable o) => o -> m T.Text
getEditableText obj = MIO.liftIO $ checkUnexpectedNothing "getEditableText" $ B.Properties.getObjectPropertyString obj "text"

-- | Set the value of the “@text@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' editable [ #text 'Data.GI.Base.Attributes.:=' value ]
-- @
setEditableText :: (MonadIO m, IsEditable o) => o -> T.Text -> m ()
setEditableText obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyString obj "text" (Just val)

-- | Construct a t'GValueConstruct' with valid value for the “@text@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructEditableText :: (IsEditable o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructEditableText val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "text" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data EditableTextPropertyInfo
instance AttrInfo EditableTextPropertyInfo where
    type AttrAllowedOps EditableTextPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint EditableTextPropertyInfo = IsEditable
    type AttrSetTypeConstraint EditableTextPropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint EditableTextPropertyInfo = (~) T.Text
    type AttrTransferType EditableTextPropertyInfo = T.Text
    type AttrGetType EditableTextPropertyInfo = T.Text
    type AttrLabel EditableTextPropertyInfo = "text"
    type AttrOrigin EditableTextPropertyInfo = Editable
    attrGet = getEditableText
    attrSet = setEditableText
    attrPut = setEditableText
    attrTransfer _ v = do
        return v
    attrConstruct = constructEditableText
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.text"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:attr:text"
        })
#endif

-- VVV Prop "width-chars"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@width-chars@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' editable #widthChars
-- @
getEditableWidthChars :: (MonadIO m, IsEditable o) => o -> m Int32
getEditableWidthChars obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "width-chars"

-- | Set the value of the “@width-chars@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' editable [ #widthChars 'Data.GI.Base.Attributes.:=' value ]
-- @
setEditableWidthChars :: (MonadIO m, IsEditable o) => o -> Int32 -> m ()
setEditableWidthChars obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyInt32 obj "width-chars" val

-- | Construct a t'GValueConstruct' with valid value for the “@width-chars@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructEditableWidthChars :: (IsEditable o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructEditableWidthChars val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "width-chars" val

#if defined(ENABLE_OVERLOADING)
data EditableWidthCharsPropertyInfo
instance AttrInfo EditableWidthCharsPropertyInfo where
    type AttrAllowedOps EditableWidthCharsPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint EditableWidthCharsPropertyInfo = IsEditable
    type AttrSetTypeConstraint EditableWidthCharsPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint EditableWidthCharsPropertyInfo = (~) Int32
    type AttrTransferType EditableWidthCharsPropertyInfo = Int32
    type AttrGetType EditableWidthCharsPropertyInfo = Int32
    type AttrLabel EditableWidthCharsPropertyInfo = "width-chars"
    type AttrOrigin EditableWidthCharsPropertyInfo = Editable
    attrGet = getEditableWidthChars
    attrSet = setEditableWidthChars
    attrPut = setEditableWidthChars
    attrTransfer _ v = do
        return v
    attrConstruct = constructEditableWidthChars
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.widthChars"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:attr:widthChars"
        })
#endif

-- VVV Prop "xalign"
   -- Type: TBasicType TFloat
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Just False)

-- | Get the value of the “@xalign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' editable #xalign
-- @
getEditableXalign :: (MonadIO m, IsEditable o) => o -> m Float
getEditableXalign obj = MIO.liftIO $ B.Properties.getObjectPropertyFloat obj "xalign"

-- | Set the value of the “@xalign@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' editable [ #xalign 'Data.GI.Base.Attributes.:=' value ]
-- @
setEditableXalign :: (MonadIO m, IsEditable o) => o -> Float -> m ()
setEditableXalign obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyFloat obj "xalign" val

-- | Construct a t'GValueConstruct' with valid value for the “@xalign@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructEditableXalign :: (IsEditable o, MIO.MonadIO m) => Float -> m (GValueConstruct o)
constructEditableXalign val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyFloat "xalign" val

#if defined(ENABLE_OVERLOADING)
data EditableXalignPropertyInfo
instance AttrInfo EditableXalignPropertyInfo where
    type AttrAllowedOps EditableXalignPropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet, 'AttrPut]
    type AttrBaseTypeConstraint EditableXalignPropertyInfo = IsEditable
    type AttrSetTypeConstraint EditableXalignPropertyInfo = (~) Float
    type AttrTransferTypeConstraint EditableXalignPropertyInfo = (~) Float
    type AttrTransferType EditableXalignPropertyInfo = Float
    type AttrGetType EditableXalignPropertyInfo = Float
    type AttrLabel EditableXalignPropertyInfo = "xalign"
    type AttrOrigin EditableXalignPropertyInfo = Editable
    attrGet = getEditableXalign
    attrSet = setEditableXalign
    attrPut = setEditableXalign
    attrTransfer _ v = do
        return v
    attrConstruct = constructEditableXalign
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.xalign"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:attr:xalign"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Editable
type instance O.AttributeList Editable = EditableAttributeList
type EditableAttributeList = ('[ '("accessibleRole", Gtk.Accessible.AccessibleAccessibleRolePropertyInfo), '("canFocus", Gtk.Widget.WidgetCanFocusPropertyInfo), '("canTarget", Gtk.Widget.WidgetCanTargetPropertyInfo), '("cssClasses", Gtk.Widget.WidgetCssClassesPropertyInfo), '("cssName", Gtk.Widget.WidgetCssNamePropertyInfo), '("cursor", Gtk.Widget.WidgetCursorPropertyInfo), '("cursorPosition", EditableCursorPositionPropertyInfo), '("editable", EditableEditablePropertyInfo), '("enableUndo", EditableEnableUndoPropertyInfo), '("focusOnClick", Gtk.Widget.WidgetFocusOnClickPropertyInfo), '("focusable", Gtk.Widget.WidgetFocusablePropertyInfo), '("halign", Gtk.Widget.WidgetHalignPropertyInfo), '("hasDefault", Gtk.Widget.WidgetHasDefaultPropertyInfo), '("hasFocus", Gtk.Widget.WidgetHasFocusPropertyInfo), '("hasTooltip", Gtk.Widget.WidgetHasTooltipPropertyInfo), '("heightRequest", Gtk.Widget.WidgetHeightRequestPropertyInfo), '("hexpand", Gtk.Widget.WidgetHexpandPropertyInfo), '("hexpandSet", Gtk.Widget.WidgetHexpandSetPropertyInfo), '("layoutManager", Gtk.Widget.WidgetLayoutManagerPropertyInfo), '("limitEvents", Gtk.Widget.WidgetLimitEventsPropertyInfo), '("marginBottom", Gtk.Widget.WidgetMarginBottomPropertyInfo), '("marginEnd", Gtk.Widget.WidgetMarginEndPropertyInfo), '("marginStart", Gtk.Widget.WidgetMarginStartPropertyInfo), '("marginTop", Gtk.Widget.WidgetMarginTopPropertyInfo), '("maxWidthChars", EditableMaxWidthCharsPropertyInfo), '("name", Gtk.Widget.WidgetNamePropertyInfo), '("opacity", Gtk.Widget.WidgetOpacityPropertyInfo), '("overflow", Gtk.Widget.WidgetOverflowPropertyInfo), '("parent", Gtk.Widget.WidgetParentPropertyInfo), '("receivesDefault", Gtk.Widget.WidgetReceivesDefaultPropertyInfo), '("root", Gtk.Widget.WidgetRootPropertyInfo), '("scaleFactor", Gtk.Widget.WidgetScaleFactorPropertyInfo), '("selectionBound", EditableSelectionBoundPropertyInfo), '("sensitive", Gtk.Widget.WidgetSensitivePropertyInfo), '("text", EditableTextPropertyInfo), '("tooltipMarkup", Gtk.Widget.WidgetTooltipMarkupPropertyInfo), '("tooltipText", Gtk.Widget.WidgetTooltipTextPropertyInfo), '("valign", Gtk.Widget.WidgetValignPropertyInfo), '("vexpand", Gtk.Widget.WidgetVexpandPropertyInfo), '("vexpandSet", Gtk.Widget.WidgetVexpandSetPropertyInfo), '("visible", Gtk.Widget.WidgetVisiblePropertyInfo), '("widthChars", EditableWidthCharsPropertyInfo), '("widthRequest", Gtk.Widget.WidgetWidthRequestPropertyInfo), '("xalign", EditableXalignPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
editableCursorPosition :: AttrLabelProxy "cursorPosition"
editableCursorPosition = AttrLabelProxy

editableEditable :: AttrLabelProxy "editable"
editableEditable = AttrLabelProxy

editableEnableUndo :: AttrLabelProxy "enableUndo"
editableEnableUndo = AttrLabelProxy

editableMaxWidthChars :: AttrLabelProxy "maxWidthChars"
editableMaxWidthChars = AttrLabelProxy

editableSelectionBound :: AttrLabelProxy "selectionBound"
editableSelectionBound = AttrLabelProxy

editableText :: AttrLabelProxy "text"
editableText = AttrLabelProxy

editableWidthChars :: AttrLabelProxy "widthChars"
editableWidthChars = AttrLabelProxy

editableXalign :: AttrLabelProxy "xalign"
editableXalign = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveEditableMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveEditableMethod "actionSetEnabled" o = Gtk.Widget.WidgetActionSetEnabledMethodInfo
    ResolveEditableMethod "activate" o = Gtk.Widget.WidgetActivateMethodInfo
    ResolveEditableMethod "activateAction" o = Gtk.Widget.WidgetActivateActionMethodInfo
    ResolveEditableMethod "activateDefault" o = Gtk.Widget.WidgetActivateDefaultMethodInfo
    ResolveEditableMethod "addController" o = Gtk.Widget.WidgetAddControllerMethodInfo
    ResolveEditableMethod "addCssClass" o = Gtk.Widget.WidgetAddCssClassMethodInfo
    ResolveEditableMethod "addMnemonicLabel" o = Gtk.Widget.WidgetAddMnemonicLabelMethodInfo
    ResolveEditableMethod "addTickCallback" o = Gtk.Widget.WidgetAddTickCallbackMethodInfo
    ResolveEditableMethod "allocate" o = Gtk.Widget.WidgetAllocateMethodInfo
    ResolveEditableMethod "announce" o = Gtk.Accessible.AccessibleAnnounceMethodInfo
    ResolveEditableMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveEditableMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveEditableMethod "childFocus" o = Gtk.Widget.WidgetChildFocusMethodInfo
    ResolveEditableMethod "computeBounds" o = Gtk.Widget.WidgetComputeBoundsMethodInfo
    ResolveEditableMethod "computeExpand" o = Gtk.Widget.WidgetComputeExpandMethodInfo
    ResolveEditableMethod "computePoint" o = Gtk.Widget.WidgetComputePointMethodInfo
    ResolveEditableMethod "computeTransform" o = Gtk.Widget.WidgetComputeTransformMethodInfo
    ResolveEditableMethod "contains" o = Gtk.Widget.WidgetContainsMethodInfo
    ResolveEditableMethod "createPangoContext" o = Gtk.Widget.WidgetCreatePangoContextMethodInfo
    ResolveEditableMethod "createPangoLayout" o = Gtk.Widget.WidgetCreatePangoLayoutMethodInfo
    ResolveEditableMethod "delegateGetAccessiblePlatformState" o = EditableDelegateGetAccessiblePlatformStateMethodInfo
    ResolveEditableMethod "deleteSelection" o = EditableDeleteSelectionMethodInfo
    ResolveEditableMethod "deleteText" o = EditableDeleteTextMethodInfo
    ResolveEditableMethod "disposeTemplate" o = Gtk.Widget.WidgetDisposeTemplateMethodInfo
    ResolveEditableMethod "dragCheckThreshold" o = Gtk.Widget.WidgetDragCheckThresholdMethodInfo
    ResolveEditableMethod "errorBell" o = Gtk.Widget.WidgetErrorBellMethodInfo
    ResolveEditableMethod "finishDelegate" o = EditableFinishDelegateMethodInfo
    ResolveEditableMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveEditableMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveEditableMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveEditableMethod "grabFocus" o = Gtk.Widget.WidgetGrabFocusMethodInfo
    ResolveEditableMethod "hasCssClass" o = Gtk.Widget.WidgetHasCssClassMethodInfo
    ResolveEditableMethod "hasDefault" o = Gtk.Widget.WidgetHasDefaultMethodInfo
    ResolveEditableMethod "hasFocus" o = Gtk.Widget.WidgetHasFocusMethodInfo
    ResolveEditableMethod "hasVisibleFocus" o = Gtk.Widget.WidgetHasVisibleFocusMethodInfo
    ResolveEditableMethod "hide" o = Gtk.Widget.WidgetHideMethodInfo
    ResolveEditableMethod "inDestruction" o = Gtk.Widget.WidgetInDestructionMethodInfo
    ResolveEditableMethod "initDelegate" o = EditableInitDelegateMethodInfo
    ResolveEditableMethod "initTemplate" o = Gtk.Widget.WidgetInitTemplateMethodInfo
    ResolveEditableMethod "insertActionGroup" o = Gtk.Widget.WidgetInsertActionGroupMethodInfo
    ResolveEditableMethod "insertAfter" o = Gtk.Widget.WidgetInsertAfterMethodInfo
    ResolveEditableMethod "insertBefore" o = Gtk.Widget.WidgetInsertBeforeMethodInfo
    ResolveEditableMethod "insertText" o = EditableInsertTextMethodInfo
    ResolveEditableMethod "isAncestor" o = Gtk.Widget.WidgetIsAncestorMethodInfo
    ResolveEditableMethod "isDrawable" o = Gtk.Widget.WidgetIsDrawableMethodInfo
    ResolveEditableMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveEditableMethod "isFocus" o = Gtk.Widget.WidgetIsFocusMethodInfo
    ResolveEditableMethod "isSensitive" o = Gtk.Widget.WidgetIsSensitiveMethodInfo
    ResolveEditableMethod "isVisible" o = Gtk.Widget.WidgetIsVisibleMethodInfo
    ResolveEditableMethod "keynavFailed" o = Gtk.Widget.WidgetKeynavFailedMethodInfo
    ResolveEditableMethod "listMnemonicLabels" o = Gtk.Widget.WidgetListMnemonicLabelsMethodInfo
    ResolveEditableMethod "map" o = Gtk.Widget.WidgetMapMethodInfo
    ResolveEditableMethod "measure" o = Gtk.Widget.WidgetMeasureMethodInfo
    ResolveEditableMethod "mnemonicActivate" o = Gtk.Widget.WidgetMnemonicActivateMethodInfo
    ResolveEditableMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveEditableMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveEditableMethod "observeChildren" o = Gtk.Widget.WidgetObserveChildrenMethodInfo
    ResolveEditableMethod "observeControllers" o = Gtk.Widget.WidgetObserveControllersMethodInfo
    ResolveEditableMethod "pick" o = Gtk.Widget.WidgetPickMethodInfo
    ResolveEditableMethod "queueAllocate" o = Gtk.Widget.WidgetQueueAllocateMethodInfo
    ResolveEditableMethod "queueDraw" o = Gtk.Widget.WidgetQueueDrawMethodInfo
    ResolveEditableMethod "queueResize" o = Gtk.Widget.WidgetQueueResizeMethodInfo
    ResolveEditableMethod "realize" o = Gtk.Widget.WidgetRealizeMethodInfo
    ResolveEditableMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveEditableMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveEditableMethod "removeController" o = Gtk.Widget.WidgetRemoveControllerMethodInfo
    ResolveEditableMethod "removeCssClass" o = Gtk.Widget.WidgetRemoveCssClassMethodInfo
    ResolveEditableMethod "removeMnemonicLabel" o = Gtk.Widget.WidgetRemoveMnemonicLabelMethodInfo
    ResolveEditableMethod "removeTickCallback" o = Gtk.Widget.WidgetRemoveTickCallbackMethodInfo
    ResolveEditableMethod "resetProperty" o = Gtk.Accessible.AccessibleResetPropertyMethodInfo
    ResolveEditableMethod "resetRelation" o = Gtk.Accessible.AccessibleResetRelationMethodInfo
    ResolveEditableMethod "resetState" o = Gtk.Accessible.AccessibleResetStateMethodInfo
    ResolveEditableMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveEditableMethod "selectRegion" o = EditableSelectRegionMethodInfo
    ResolveEditableMethod "shouldLayout" o = Gtk.Widget.WidgetShouldLayoutMethodInfo
    ResolveEditableMethod "show" o = Gtk.Widget.WidgetShowMethodInfo
    ResolveEditableMethod "sizeAllocate" o = Gtk.Widget.WidgetSizeAllocateMethodInfo
    ResolveEditableMethod "snapshotChild" o = Gtk.Widget.WidgetSnapshotChildMethodInfo
    ResolveEditableMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveEditableMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveEditableMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveEditableMethod "translateCoordinates" o = Gtk.Widget.WidgetTranslateCoordinatesMethodInfo
    ResolveEditableMethod "triggerTooltipQuery" o = Gtk.Widget.WidgetTriggerTooltipQueryMethodInfo
    ResolveEditableMethod "unmap" o = Gtk.Widget.WidgetUnmapMethodInfo
    ResolveEditableMethod "unparent" o = Gtk.Widget.WidgetUnparentMethodInfo
    ResolveEditableMethod "unrealize" o = Gtk.Widget.WidgetUnrealizeMethodInfo
    ResolveEditableMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveEditableMethod "unsetStateFlags" o = Gtk.Widget.WidgetUnsetStateFlagsMethodInfo
    ResolveEditableMethod "updateNextAccessibleSibling" o = Gtk.Accessible.AccessibleUpdateNextAccessibleSiblingMethodInfo
    ResolveEditableMethod "updatePlatformState" o = Gtk.Accessible.AccessibleUpdatePlatformStateMethodInfo
    ResolveEditableMethod "updateProperty" o = Gtk.Accessible.AccessibleUpdatePropertyMethodInfo
    ResolveEditableMethod "updateRelation" o = Gtk.Accessible.AccessibleUpdateRelationMethodInfo
    ResolveEditableMethod "updateState" o = Gtk.Accessible.AccessibleUpdateStateMethodInfo
    ResolveEditableMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveEditableMethod "getAccessibleParent" o = Gtk.Accessible.AccessibleGetAccessibleParentMethodInfo
    ResolveEditableMethod "getAccessibleRole" o = Gtk.Accessible.AccessibleGetAccessibleRoleMethodInfo
    ResolveEditableMethod "getAlignment" o = EditableGetAlignmentMethodInfo
    ResolveEditableMethod "getAllocatedBaseline" o = Gtk.Widget.WidgetGetAllocatedBaselineMethodInfo
    ResolveEditableMethod "getAllocatedHeight" o = Gtk.Widget.WidgetGetAllocatedHeightMethodInfo
    ResolveEditableMethod "getAllocatedWidth" o = Gtk.Widget.WidgetGetAllocatedWidthMethodInfo
    ResolveEditableMethod "getAllocation" o = Gtk.Widget.WidgetGetAllocationMethodInfo
    ResolveEditableMethod "getAncestor" o = Gtk.Widget.WidgetGetAncestorMethodInfo
    ResolveEditableMethod "getAtContext" o = Gtk.Accessible.AccessibleGetAtContextMethodInfo
    ResolveEditableMethod "getBaseline" o = Gtk.Widget.WidgetGetBaselineMethodInfo
    ResolveEditableMethod "getBounds" o = Gtk.Accessible.AccessibleGetBoundsMethodInfo
    ResolveEditableMethod "getBuildableId" o = Gtk.Buildable.BuildableGetBuildableIdMethodInfo
    ResolveEditableMethod "getCanFocus" o = Gtk.Widget.WidgetGetCanFocusMethodInfo
    ResolveEditableMethod "getCanTarget" o = Gtk.Widget.WidgetGetCanTargetMethodInfo
    ResolveEditableMethod "getChars" o = EditableGetCharsMethodInfo
    ResolveEditableMethod "getChildVisible" o = Gtk.Widget.WidgetGetChildVisibleMethodInfo
    ResolveEditableMethod "getClipboard" o = Gtk.Widget.WidgetGetClipboardMethodInfo
    ResolveEditableMethod "getColor" o = Gtk.Widget.WidgetGetColorMethodInfo
    ResolveEditableMethod "getCssClasses" o = Gtk.Widget.WidgetGetCssClassesMethodInfo
    ResolveEditableMethod "getCssName" o = Gtk.Widget.WidgetGetCssNameMethodInfo
    ResolveEditableMethod "getCursor" o = Gtk.Widget.WidgetGetCursorMethodInfo
    ResolveEditableMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveEditableMethod "getDelegate" o = EditableGetDelegateMethodInfo
    ResolveEditableMethod "getDirection" o = Gtk.Widget.WidgetGetDirectionMethodInfo
    ResolveEditableMethod "getDisplay" o = Gtk.Widget.WidgetGetDisplayMethodInfo
    ResolveEditableMethod "getEditable" o = EditableGetEditableMethodInfo
    ResolveEditableMethod "getEnableUndo" o = EditableGetEnableUndoMethodInfo
    ResolveEditableMethod "getFirstAccessibleChild" o = Gtk.Accessible.AccessibleGetFirstAccessibleChildMethodInfo
    ResolveEditableMethod "getFirstChild" o = Gtk.Widget.WidgetGetFirstChildMethodInfo
    ResolveEditableMethod "getFocusChild" o = Gtk.Widget.WidgetGetFocusChildMethodInfo
    ResolveEditableMethod "getFocusOnClick" o = Gtk.Widget.WidgetGetFocusOnClickMethodInfo
    ResolveEditableMethod "getFocusable" o = Gtk.Widget.WidgetGetFocusableMethodInfo
    ResolveEditableMethod "getFontMap" o = Gtk.Widget.WidgetGetFontMapMethodInfo
    ResolveEditableMethod "getFontOptions" o = Gtk.Widget.WidgetGetFontOptionsMethodInfo
    ResolveEditableMethod "getFrameClock" o = Gtk.Widget.WidgetGetFrameClockMethodInfo
    ResolveEditableMethod "getHalign" o = Gtk.Widget.WidgetGetHalignMethodInfo
    ResolveEditableMethod "getHasTooltip" o = Gtk.Widget.WidgetGetHasTooltipMethodInfo
    ResolveEditableMethod "getHeight" o = Gtk.Widget.WidgetGetHeightMethodInfo
    ResolveEditableMethod "getHexpand" o = Gtk.Widget.WidgetGetHexpandMethodInfo
    ResolveEditableMethod "getHexpandSet" o = Gtk.Widget.WidgetGetHexpandSetMethodInfo
    ResolveEditableMethod "getLastChild" o = Gtk.Widget.WidgetGetLastChildMethodInfo
    ResolveEditableMethod "getLayoutManager" o = Gtk.Widget.WidgetGetLayoutManagerMethodInfo
    ResolveEditableMethod "getLimitEvents" o = Gtk.Widget.WidgetGetLimitEventsMethodInfo
    ResolveEditableMethod "getMapped" o = Gtk.Widget.WidgetGetMappedMethodInfo
    ResolveEditableMethod "getMarginBottom" o = Gtk.Widget.WidgetGetMarginBottomMethodInfo
    ResolveEditableMethod "getMarginEnd" o = Gtk.Widget.WidgetGetMarginEndMethodInfo
    ResolveEditableMethod "getMarginStart" o = Gtk.Widget.WidgetGetMarginStartMethodInfo
    ResolveEditableMethod "getMarginTop" o = Gtk.Widget.WidgetGetMarginTopMethodInfo
    ResolveEditableMethod "getMaxWidthChars" o = EditableGetMaxWidthCharsMethodInfo
    ResolveEditableMethod "getName" o = Gtk.Widget.WidgetGetNameMethodInfo
    ResolveEditableMethod "getNative" o = Gtk.Widget.WidgetGetNativeMethodInfo
    ResolveEditableMethod "getNextAccessibleSibling" o = Gtk.Accessible.AccessibleGetNextAccessibleSiblingMethodInfo
    ResolveEditableMethod "getNextSibling" o = Gtk.Widget.WidgetGetNextSiblingMethodInfo
    ResolveEditableMethod "getOpacity" o = Gtk.Widget.WidgetGetOpacityMethodInfo
    ResolveEditableMethod "getOverflow" o = Gtk.Widget.WidgetGetOverflowMethodInfo
    ResolveEditableMethod "getPangoContext" o = Gtk.Widget.WidgetGetPangoContextMethodInfo
    ResolveEditableMethod "getParent" o = Gtk.Widget.WidgetGetParentMethodInfo
    ResolveEditableMethod "getPlatformState" o = Gtk.Accessible.AccessibleGetPlatformStateMethodInfo
    ResolveEditableMethod "getPosition" o = EditableGetPositionMethodInfo
    ResolveEditableMethod "getPreferredSize" o = Gtk.Widget.WidgetGetPreferredSizeMethodInfo
    ResolveEditableMethod "getPrevSibling" o = Gtk.Widget.WidgetGetPrevSiblingMethodInfo
    ResolveEditableMethod "getPrimaryClipboard" o = Gtk.Widget.WidgetGetPrimaryClipboardMethodInfo
    ResolveEditableMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveEditableMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveEditableMethod "getRealized" o = Gtk.Widget.WidgetGetRealizedMethodInfo
    ResolveEditableMethod "getReceivesDefault" o = Gtk.Widget.WidgetGetReceivesDefaultMethodInfo
    ResolveEditableMethod "getRequestMode" o = Gtk.Widget.WidgetGetRequestModeMethodInfo
    ResolveEditableMethod "getRoot" o = Gtk.Widget.WidgetGetRootMethodInfo
    ResolveEditableMethod "getScaleFactor" o = Gtk.Widget.WidgetGetScaleFactorMethodInfo
    ResolveEditableMethod "getSelectionBounds" o = EditableGetSelectionBoundsMethodInfo
    ResolveEditableMethod "getSensitive" o = Gtk.Widget.WidgetGetSensitiveMethodInfo
    ResolveEditableMethod "getSettings" o = Gtk.Widget.WidgetGetSettingsMethodInfo
    ResolveEditableMethod "getSize" o = Gtk.Widget.WidgetGetSizeMethodInfo
    ResolveEditableMethod "getSizeRequest" o = Gtk.Widget.WidgetGetSizeRequestMethodInfo
    ResolveEditableMethod "getStateFlags" o = Gtk.Widget.WidgetGetStateFlagsMethodInfo
    ResolveEditableMethod "getStyleContext" o = Gtk.Widget.WidgetGetStyleContextMethodInfo
    ResolveEditableMethod "getTemplateChild" o = Gtk.Widget.WidgetGetTemplateChildMethodInfo
    ResolveEditableMethod "getText" o = EditableGetTextMethodInfo
    ResolveEditableMethod "getTooltipMarkup" o = Gtk.Widget.WidgetGetTooltipMarkupMethodInfo
    ResolveEditableMethod "getTooltipText" o = Gtk.Widget.WidgetGetTooltipTextMethodInfo
    ResolveEditableMethod "getValign" o = Gtk.Widget.WidgetGetValignMethodInfo
    ResolveEditableMethod "getVexpand" o = Gtk.Widget.WidgetGetVexpandMethodInfo
    ResolveEditableMethod "getVexpandSet" o = Gtk.Widget.WidgetGetVexpandSetMethodInfo
    ResolveEditableMethod "getVisible" o = Gtk.Widget.WidgetGetVisibleMethodInfo
    ResolveEditableMethod "getWidth" o = Gtk.Widget.WidgetGetWidthMethodInfo
    ResolveEditableMethod "getWidthChars" o = EditableGetWidthCharsMethodInfo
    ResolveEditableMethod "setAccessibleParent" o = Gtk.Accessible.AccessibleSetAccessibleParentMethodInfo
    ResolveEditableMethod "setAlignment" o = EditableSetAlignmentMethodInfo
    ResolveEditableMethod "setCanFocus" o = Gtk.Widget.WidgetSetCanFocusMethodInfo
    ResolveEditableMethod "setCanTarget" o = Gtk.Widget.WidgetSetCanTargetMethodInfo
    ResolveEditableMethod "setChildVisible" o = Gtk.Widget.WidgetSetChildVisibleMethodInfo
    ResolveEditableMethod "setCssClasses" o = Gtk.Widget.WidgetSetCssClassesMethodInfo
    ResolveEditableMethod "setCursor" o = Gtk.Widget.WidgetSetCursorMethodInfo
    ResolveEditableMethod "setCursorFromName" o = Gtk.Widget.WidgetSetCursorFromNameMethodInfo
    ResolveEditableMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveEditableMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveEditableMethod "setDirection" o = Gtk.Widget.WidgetSetDirectionMethodInfo
    ResolveEditableMethod "setEditable" o = EditableSetEditableMethodInfo
    ResolveEditableMethod "setEnableUndo" o = EditableSetEnableUndoMethodInfo
    ResolveEditableMethod "setFocusChild" o = Gtk.Widget.WidgetSetFocusChildMethodInfo
    ResolveEditableMethod "setFocusOnClick" o = Gtk.Widget.WidgetSetFocusOnClickMethodInfo
    ResolveEditableMethod "setFocusable" o = Gtk.Widget.WidgetSetFocusableMethodInfo
    ResolveEditableMethod "setFontMap" o = Gtk.Widget.WidgetSetFontMapMethodInfo
    ResolveEditableMethod "setFontOptions" o = Gtk.Widget.WidgetSetFontOptionsMethodInfo
    ResolveEditableMethod "setHalign" o = Gtk.Widget.WidgetSetHalignMethodInfo
    ResolveEditableMethod "setHasTooltip" o = Gtk.Widget.WidgetSetHasTooltipMethodInfo
    ResolveEditableMethod "setHexpand" o = Gtk.Widget.WidgetSetHexpandMethodInfo
    ResolveEditableMethod "setHexpandSet" o = Gtk.Widget.WidgetSetHexpandSetMethodInfo
    ResolveEditableMethod "setLayoutManager" o = Gtk.Widget.WidgetSetLayoutManagerMethodInfo
    ResolveEditableMethod "setLimitEvents" o = Gtk.Widget.WidgetSetLimitEventsMethodInfo
    ResolveEditableMethod "setMarginBottom" o = Gtk.Widget.WidgetSetMarginBottomMethodInfo
    ResolveEditableMethod "setMarginEnd" o = Gtk.Widget.WidgetSetMarginEndMethodInfo
    ResolveEditableMethod "setMarginStart" o = Gtk.Widget.WidgetSetMarginStartMethodInfo
    ResolveEditableMethod "setMarginTop" o = Gtk.Widget.WidgetSetMarginTopMethodInfo
    ResolveEditableMethod "setMaxWidthChars" o = EditableSetMaxWidthCharsMethodInfo
    ResolveEditableMethod "setName" o = Gtk.Widget.WidgetSetNameMethodInfo
    ResolveEditableMethod "setOpacity" o = Gtk.Widget.WidgetSetOpacityMethodInfo
    ResolveEditableMethod "setOverflow" o = Gtk.Widget.WidgetSetOverflowMethodInfo
    ResolveEditableMethod "setParent" o = Gtk.Widget.WidgetSetParentMethodInfo
    ResolveEditableMethod "setPosition" o = EditableSetPositionMethodInfo
    ResolveEditableMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveEditableMethod "setReceivesDefault" o = Gtk.Widget.WidgetSetReceivesDefaultMethodInfo
    ResolveEditableMethod "setSensitive" o = Gtk.Widget.WidgetSetSensitiveMethodInfo
    ResolveEditableMethod "setSizeRequest" o = Gtk.Widget.WidgetSetSizeRequestMethodInfo
    ResolveEditableMethod "setStateFlags" o = Gtk.Widget.WidgetSetStateFlagsMethodInfo
    ResolveEditableMethod "setText" o = EditableSetTextMethodInfo
    ResolveEditableMethod "setTooltipMarkup" o = Gtk.Widget.WidgetSetTooltipMarkupMethodInfo
    ResolveEditableMethod "setTooltipText" o = Gtk.Widget.WidgetSetTooltipTextMethodInfo
    ResolveEditableMethod "setValign" o = Gtk.Widget.WidgetSetValignMethodInfo
    ResolveEditableMethod "setVexpand" o = Gtk.Widget.WidgetSetVexpandMethodInfo
    ResolveEditableMethod "setVexpandSet" o = Gtk.Widget.WidgetSetVexpandSetMethodInfo
    ResolveEditableMethod "setVisible" o = Gtk.Widget.WidgetSetVisibleMethodInfo
    ResolveEditableMethod "setWidthChars" o = EditableSetWidthCharsMethodInfo
    ResolveEditableMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveEditableMethod t Editable, O.OverloadedMethod info Editable p) => OL.IsLabel t (Editable -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveEditableMethod t Editable, O.OverloadedMethod info Editable p, R.HasField t Editable p) => R.HasField t Editable p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveEditableMethod t Editable, O.OverloadedMethodInfo info Editable) => OL.IsLabel t (O.MethodProxy info Editable) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method Editable::delegate_get_accessible_platform_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable` implementation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "state"
--           , argType =
--               TInterface
--                 Name { namespace = "Gtk" , name = "AccessiblePlatformState" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "what kind of accessible state to retrieve"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_delegate_get_accessible_platform_state" gtk_editable_delegate_get_accessible_platform_state :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    CUInt ->                                -- state : TInterface (Name {namespace = "Gtk", name = "AccessiblePlatformState"})
    IO CInt

-- | Retrieves the accessible platform state from the editable delegate.
-- 
-- This is an helper function to retrieve the accessible state for
-- @GtkEditable@ interface implementations using a delegate pattern.
-- 
-- You should call this function in your editable widget implementation
-- of the t'GI.Gtk.Interfaces.Accessible.Accessible'.@/get_platform_state/@() virtual function, for
-- instance:
-- 
-- 
-- === /c code/
-- >static void
-- >accessible_interface_init (GtkAccessibleInterface *iface)
-- >{
-- >  iface->get_platform_state = your_editable_get_accessible_platform_state;
-- >}
-- >
-- >static gboolean
-- >your_editable_get_accessible_platform_state (GtkAccessible *accessible,
-- >                                             GtkAccessiblePlatformState state)
-- >{
-- >  return gtk_editable_delegate_get_accessible_platform_state (GTK_EDITABLE (accessible), state);
-- >}
-- 
-- 
-- Note that the widget which is the delegate *must* be a direct child of
-- this widget, otherwise your implementation of t'GI.Gtk.Interfaces.Accessible.Accessible'.@/get_platform_state/@()
-- might not even be called, as the platform change will originate from
-- the parent of the delegate, and, as a result, will not work properly.
-- 
-- So, if you can\'t ensure the direct child condition, you should give the
-- delegate the 'GI.Gtk.Enums.AccessibleRoleTextBox' role, or you can
-- change your tree to allow this function to work.
-- 
-- /Since: 4.10/
editableDelegateGetAccessiblePlatformState ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@ implementation
    -> Gtk.Enums.AccessiblePlatformState
    -- ^ /@state@/: what kind of accessible state to retrieve
    -> m Bool
    -- ^ __Returns:__ the accessible platform state of the delegate
editableDelegateGetAccessiblePlatformState editable state = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    let state' = (fromIntegral . fromEnum) state
    result <- gtk_editable_delegate_get_accessible_platform_state editable' state'
    let result' = (/= 0) result
    touchManagedPtr editable
    return result'

#if defined(ENABLE_OVERLOADING)
data EditableDelegateGetAccessiblePlatformStateMethodInfo
instance (signature ~ (Gtk.Enums.AccessiblePlatformState -> m Bool), MonadIO m, IsEditable a) => O.OverloadedMethod EditableDelegateGetAccessiblePlatformStateMethodInfo a signature where
    overloadedMethod = editableDelegateGetAccessiblePlatformState

instance O.OverloadedMethodInfo EditableDelegateGetAccessiblePlatformStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableDelegateGetAccessiblePlatformState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableDelegateGetAccessiblePlatformState"
        })


#endif

-- method Editable::delete_selection
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_delete_selection" gtk_editable_delete_selection :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO ()

-- | Deletes the currently selected text of the editable.
-- 
-- This call doesn’t do anything if there is no selected text.
editableDeleteSelection ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m ()
editableDeleteSelection editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    gtk_editable_delete_selection editable'
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableDeleteSelectionMethodInfo
instance (signature ~ (m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableDeleteSelectionMethodInfo a signature where
    overloadedMethod = editableDeleteSelection

instance O.OverloadedMethodInfo EditableDeleteSelectionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableDeleteSelection",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableDeleteSelection"
        })


#endif

-- method Editable::delete_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_pos"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "start position" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_pos"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "end position" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_delete_text" gtk_editable_delete_text :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    Int32 ->                                -- start_pos : TBasicType TInt
    Int32 ->                                -- end_pos : TBasicType TInt
    IO ()

-- | Deletes a sequence of characters.
-- 
-- The characters that are deleted are those characters at positions
-- from /@startPos@/ up to, but not including /@endPos@/. If /@endPos@/ is
-- negative, then the characters deleted are those from /@startPos@/ to
-- the end of the text.
-- 
-- Note that the positions are specified in characters, not bytes.
editableDeleteText ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> Int32
    -- ^ /@startPos@/: start position
    -> Int32
    -- ^ /@endPos@/: end position
    -> m ()
editableDeleteText editable startPos endPos = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    gtk_editable_delete_text editable' startPos endPos
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableDeleteTextMethodInfo
instance (signature ~ (Int32 -> Int32 -> m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableDeleteTextMethodInfo a signature where
    overloadedMethod = editableDeleteText

instance O.OverloadedMethodInfo EditableDeleteTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableDeleteText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableDeleteText"
        })


#endif

-- method Editable::finish_delegate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_finish_delegate" gtk_editable_finish_delegate :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO ()

-- | Undoes the setup done by 'GI.Gtk.Interfaces.Editable.editableInitDelegate'.
-- 
-- This is a helper function that should be called from dispose,
-- before removing the delegate object.
editableFinishDelegate ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m ()
editableFinishDelegate editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    gtk_editable_finish_delegate editable'
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableFinishDelegateMethodInfo
instance (signature ~ (m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableFinishDelegateMethodInfo a signature where
    overloadedMethod = editableFinishDelegate

instance O.OverloadedMethodInfo EditableFinishDelegateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableFinishDelegate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableFinishDelegate"
        })


#endif

-- method Editable::get_alignment
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFloat)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_alignment" gtk_editable_get_alignment :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO CFloat

-- | Gets the alignment of the editable.
editableGetAlignment ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m Float
    -- ^ __Returns:__ the alignment
editableGetAlignment editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    result <- gtk_editable_get_alignment editable'
    let result' = realToFrac result
    touchManagedPtr editable
    return result'

#if defined(ENABLE_OVERLOADING)
data EditableGetAlignmentMethodInfo
instance (signature ~ (m Float), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetAlignmentMethodInfo a signature where
    overloadedMethod = editableGetAlignment

instance O.OverloadedMethodInfo EditableGetAlignmentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetAlignment",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetAlignment"
        })


#endif

-- method Editable::get_chars
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_pos"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "start of text" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_pos"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "end of text" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_chars" gtk_editable_get_chars :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    Int32 ->                                -- start_pos : TBasicType TInt
    Int32 ->                                -- end_pos : TBasicType TInt
    IO CString

-- | Retrieves a sequence of characters.
-- 
-- The characters that are retrieved are those characters at positions
-- from /@startPos@/ up to, but not including /@endPos@/. If /@endPos@/ is negative,
-- then the characters retrieved are those characters from /@startPos@/ to
-- the end of the text.
-- 
-- Note that positions are specified in characters, not bytes.
editableGetChars ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> Int32
    -- ^ /@startPos@/: start of text
    -> Int32
    -- ^ /@endPos@/: end of text
    -> m T.Text
    -- ^ __Returns:__ a pointer to the contents of the widget as a
    --   string. This string is allocated by the @GtkEditable@ implementation
    --   and should be freed by the caller.
editableGetChars editable startPos endPos = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    result <- gtk_editable_get_chars editable' startPos endPos
    checkUnexpectedReturnNULL "editableGetChars" result
    result' <- cstringToText result
    freeMem result
    touchManagedPtr editable
    return result'

#if defined(ENABLE_OVERLOADING)
data EditableGetCharsMethodInfo
instance (signature ~ (Int32 -> Int32 -> m T.Text), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetCharsMethodInfo a signature where
    overloadedMethod = editableGetChars

instance O.OverloadedMethodInfo EditableGetCharsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetChars",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetChars"
        })


#endif

-- method Editable::get_delegate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Editable" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_delegate" gtk_editable_get_delegate :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO (Ptr Editable)

-- | Gets the @GtkEditable@ that /@editable@/ is delegating its
-- implementation to.
-- 
-- Typically, the delegate is a t'GI.Gtk.Objects.Text.Text' widget.
editableGetDelegate ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m (Maybe Editable)
    -- ^ __Returns:__ the delegate @GtkEditable@
editableGetDelegate editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    result <- gtk_editable_get_delegate editable'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Editable) result'
        return result''
    touchManagedPtr editable
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data EditableGetDelegateMethodInfo
instance (signature ~ (m (Maybe Editable)), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetDelegateMethodInfo a signature where
    overloadedMethod = editableGetDelegate

instance O.OverloadedMethodInfo EditableGetDelegateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetDelegate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetDelegate"
        })


#endif

-- method Editable::get_editable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_editable" gtk_editable_get_editable :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO CInt

-- | Retrieves whether /@editable@/ is editable.
editableGetEditable ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@editable@/ is editable.
editableGetEditable editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    result <- gtk_editable_get_editable editable'
    let result' = (/= 0) result
    touchManagedPtr editable
    return result'

#if defined(ENABLE_OVERLOADING)
data EditableGetEditableMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetEditableMethodInfo a signature where
    overloadedMethod = editableGetEditable

instance O.OverloadedMethodInfo EditableGetEditableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetEditable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetEditable"
        })


#endif

-- method Editable::get_enable_undo
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_enable_undo" gtk_editable_get_enable_undo :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO CInt

-- | Gets if undo\/redo actions are enabled for /@editable@/
editableGetEnableUndo ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m Bool
    -- ^ __Returns:__ 'P.True' if undo is enabled
editableGetEnableUndo editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    result <- gtk_editable_get_enable_undo editable'
    let result' = (/= 0) result
    touchManagedPtr editable
    return result'

#if defined(ENABLE_OVERLOADING)
data EditableGetEnableUndoMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetEnableUndoMethodInfo a signature where
    overloadedMethod = editableGetEnableUndo

instance O.OverloadedMethodInfo EditableGetEnableUndoMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetEnableUndo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetEnableUndo"
        })


#endif

-- method Editable::get_max_width_chars
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_max_width_chars" gtk_editable_get_max_width_chars :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO Int32

-- | Retrieves the desired maximum width of /@editable@/, in characters.
editableGetMaxWidthChars ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m Int32
    -- ^ __Returns:__ the maximum width of the entry, in characters
editableGetMaxWidthChars editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    result <- gtk_editable_get_max_width_chars editable'
    touchManagedPtr editable
    return result

#if defined(ENABLE_OVERLOADING)
data EditableGetMaxWidthCharsMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetMaxWidthCharsMethodInfo a signature where
    overloadedMethod = editableGetMaxWidthChars

instance O.OverloadedMethodInfo EditableGetMaxWidthCharsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetMaxWidthChars",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetMaxWidthChars"
        })


#endif

-- method Editable::get_position
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_position" gtk_editable_get_position :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO Int32

-- | Retrieves the current position of the cursor relative
-- to the start of the content of the editable.
-- 
-- Note that this position is in characters, not in bytes.
editableGetPosition ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m Int32
    -- ^ __Returns:__ the cursor position
editableGetPosition editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    result <- gtk_editable_get_position editable'
    touchManagedPtr editable
    return result

#if defined(ENABLE_OVERLOADING)
data EditableGetPositionMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetPositionMethodInfo a signature where
    overloadedMethod = editableGetPosition

instance O.OverloadedMethodInfo EditableGetPositionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetPosition",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetPosition"
        })


#endif

-- method Editable::get_selection_bounds
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_pos"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the starting position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "end_pos"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the end position"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_selection_bounds" gtk_editable_get_selection_bounds :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    Ptr Int32 ->                            -- start_pos : TBasicType TInt
    Ptr Int32 ->                            -- end_pos : TBasicType TInt
    IO CInt

-- | Retrieves the selection bound of the editable.
-- 
-- /@startPos@/ will be filled with the start of the selection and
-- /@endPos@/ with end. If no text was selected both will be identical
-- and 'P.False' will be returned.
-- 
-- Note that positions are specified in characters, not bytes.
editableGetSelectionBounds ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m ((Bool, Int32, Int32))
    -- ^ __Returns:__ 'P.True' if there is a non-empty selection, 'P.False' otherwise
editableGetSelectionBounds editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    startPos <- allocMem :: IO (Ptr Int32)
    endPos <- allocMem :: IO (Ptr Int32)
    result <- gtk_editable_get_selection_bounds editable' startPos endPos
    let result' = (/= 0) result
    startPos' <- peek startPos
    endPos' <- peek endPos
    touchManagedPtr editable
    freeMem startPos
    freeMem endPos
    return (result', startPos', endPos')

#if defined(ENABLE_OVERLOADING)
data EditableGetSelectionBoundsMethodInfo
instance (signature ~ (m ((Bool, Int32, Int32))), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetSelectionBoundsMethodInfo a signature where
    overloadedMethod = editableGetSelectionBounds

instance O.OverloadedMethodInfo EditableGetSelectionBoundsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetSelectionBounds",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetSelectionBounds"
        })


#endif

-- method Editable::get_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_text" gtk_editable_get_text :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO CString

-- | Retrieves the contents of /@editable@/.
-- 
-- The returned string is owned by GTK and must not be modified or freed.
editableGetText ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m T.Text
    -- ^ __Returns:__ a pointer to the contents of the editable
editableGetText editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    result <- gtk_editable_get_text editable'
    checkUnexpectedReturnNULL "editableGetText" result
    result' <- cstringToText result
    touchManagedPtr editable
    return result'

#if defined(ENABLE_OVERLOADING)
data EditableGetTextMethodInfo
instance (signature ~ (m T.Text), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetTextMethodInfo a signature where
    overloadedMethod = editableGetText

instance O.OverloadedMethodInfo EditableGetTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetText"
        })


#endif

-- method Editable::get_width_chars
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_get_width_chars" gtk_editable_get_width_chars :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO Int32

-- | Gets the number of characters of space reserved
-- for the contents of the editable.
editableGetWidthChars ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m Int32
    -- ^ __Returns:__ number of chars to request space for, or negative if unset
editableGetWidthChars editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    result <- gtk_editable_get_width_chars editable'
    touchManagedPtr editable
    return result

#if defined(ENABLE_OVERLOADING)
data EditableGetWidthCharsMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsEditable a) => O.OverloadedMethod EditableGetWidthCharsMethodInfo a signature where
    overloadedMethod = editableGetWidthChars

instance O.OverloadedMethodInfo EditableGetWidthCharsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableGetWidthChars",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableGetWidthChars"
        })


#endif

-- method Editable::init_delegate
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_init_delegate" gtk_editable_init_delegate :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    IO ()

-- | Sets up a delegate for @GtkEditable@.
-- 
-- This is assuming that the get_delegate vfunc in the @GtkEditable@
-- interface has been set up for the /@editable@/\'s type.
-- 
-- This is a helper function that should be called in instance init,
-- after creating the delegate object.
editableInitDelegate ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> m ()
editableInitDelegate editable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    gtk_editable_init_delegate editable'
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableInitDelegateMethodInfo
instance (signature ~ (m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableInitDelegateMethodInfo a signature where
    overloadedMethod = editableInitDelegate

instance O.OverloadedMethodInfo EditableInitDelegateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableInitDelegate",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableInitDelegate"
        })


#endif

-- method Editable::insert_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text to insert" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "length"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of the text in bytes, or -1"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "position"
--           , argType = TBasicType TInt
--           , direction = DirectionInout
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "location of the position text will be inserted at"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_insert_text" gtk_editable_insert_text :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    CString ->                              -- text : TBasicType TUTF8
    Int32 ->                                -- length : TBasicType TInt
    Ptr Int32 ->                            -- position : TBasicType TInt
    IO ()

-- | Inserts /@length@/ bytes of /@text@/ into the contents of the
-- widget, at position /@position@/.
-- 
-- Note that the position is in characters, not in bytes.
-- The function updates /@position@/ to point after the newly
-- inserted text.
editableInsertText ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> T.Text
    -- ^ /@text@/: the text to insert
    -> Int32
    -- ^ /@length@/: the length of the text in bytes, or -1
    -> Int32
    -- ^ /@position@/: location of the position text will be inserted at
    -> m (Int32)
editableInsertText editable text length_ position = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    text' <- textToCString text
    position' <- allocMem :: IO (Ptr Int32)
    poke position' position
    gtk_editable_insert_text editable' text' length_ position'
    position'' <- peek position'
    touchManagedPtr editable
    freeMem text'
    freeMem position'
    return position''

#if defined(ENABLE_OVERLOADING)
data EditableInsertTextMethodInfo
instance (signature ~ (T.Text -> Int32 -> Int32 -> m (Int32)), MonadIO m, IsEditable a) => O.OverloadedMethod EditableInsertTextMethodInfo a signature where
    overloadedMethod = editableInsertText

instance O.OverloadedMethodInfo EditableInsertTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableInsertText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableInsertText"
        })


#endif

-- method Editable::select_region
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "start_pos"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "start of region" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "end_pos"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "end of region" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_select_region" gtk_editable_select_region :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    Int32 ->                                -- start_pos : TBasicType TInt
    Int32 ->                                -- end_pos : TBasicType TInt
    IO ()

-- | Selects a region of text.
-- 
-- The characters that are selected are those characters at positions
-- from /@startPos@/ up to, but not including /@endPos@/. If /@endPos@/ is
-- negative, then the characters selected are those characters from
-- /@startPos@/ to  the end of the text.
-- 
-- Note that positions are specified in characters, not bytes.
editableSelectRegion ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> Int32
    -- ^ /@startPos@/: start of region
    -> Int32
    -- ^ /@endPos@/: end of region
    -> m ()
editableSelectRegion editable startPos endPos = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    gtk_editable_select_region editable' startPos endPos
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableSelectRegionMethodInfo
instance (signature ~ (Int32 -> Int32 -> m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableSelectRegionMethodInfo a signature where
    overloadedMethod = editableSelectRegion

instance O.OverloadedMethodInfo EditableSelectRegionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableSelectRegion",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableSelectRegion"
        })


#endif

-- method Editable::set_alignment
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "xalign"
--           , argType = TBasicType TFloat
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "The horizontal alignment, from 0 (left) to 1 (right).\n  Reversed for RTL layouts"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_set_alignment" gtk_editable_set_alignment :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    CFloat ->                               -- xalign : TBasicType TFloat
    IO ()

-- | Sets the alignment for the contents of the editable.
-- 
-- This controls the horizontal positioning of the contents when
-- the displayed text is shorter than the width of the editable.
editableSetAlignment ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> Float
    -- ^ /@xalign@/: The horizontal alignment, from 0 (left) to 1 (right).
    --   Reversed for RTL layouts
    -> m ()
editableSetAlignment editable xalign = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    let xalign' = realToFrac xalign
    gtk_editable_set_alignment editable' xalign'
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableSetAlignmentMethodInfo
instance (signature ~ (Float -> m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableSetAlignmentMethodInfo a signature where
    overloadedMethod = editableSetAlignment

instance O.OverloadedMethodInfo EditableSetAlignmentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableSetAlignment",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableSetAlignment"
        })


#endif

-- method Editable::set_editable
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "is_editable"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "%TRUE if the user is allowed to edit the text\n  in the widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_set_editable" gtk_editable_set_editable :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    CInt ->                                 -- is_editable : TBasicType TBoolean
    IO ()

-- | Determines if the user can edit the text in the editable widget.
editableSetEditable ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> Bool
    -- ^ /@isEditable@/: 'P.True' if the user is allowed to edit the text
    --   in the widget
    -> m ()
editableSetEditable editable isEditable = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    let isEditable' = (P.fromIntegral . P.fromEnum) isEditable
    gtk_editable_set_editable editable' isEditable'
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableSetEditableMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableSetEditableMethodInfo a signature where
    overloadedMethod = editableSetEditable

instance O.OverloadedMethodInfo EditableSetEditableMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableSetEditable",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableSetEditable"
        })


#endif

-- method Editable::set_enable_undo
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "enable_undo"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "if undo/redo should be enabled"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_set_enable_undo" gtk_editable_set_enable_undo :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    CInt ->                                 -- enable_undo : TBasicType TBoolean
    IO ()

-- | If enabled, changes to /@editable@/ will be saved for undo\/redo
-- actions.
-- 
-- This results in an additional copy of text changes and are not
-- stored in secure memory. As such, undo is forcefully disabled
-- when [Text:visibility]("GI.Gtk.Objects.Text#g:attr:visibility") is set to 'P.False'.
editableSetEnableUndo ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> Bool
    -- ^ /@enableUndo@/: if undo\/redo should be enabled
    -> m ()
editableSetEnableUndo editable enableUndo = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    let enableUndo' = (P.fromIntegral . P.fromEnum) enableUndo
    gtk_editable_set_enable_undo editable' enableUndo'
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableSetEnableUndoMethodInfo
instance (signature ~ (Bool -> m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableSetEnableUndoMethodInfo a signature where
    overloadedMethod = editableSetEnableUndo

instance O.OverloadedMethodInfo EditableSetEnableUndoMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableSetEnableUndo",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableSetEnableUndo"
        })


#endif

-- method Editable::set_max_width_chars
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_chars"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new desired maximum width, in characters"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_set_max_width_chars" gtk_editable_set_max_width_chars :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    Int32 ->                                -- n_chars : TBasicType TInt
    IO ()

-- | Sets the desired maximum width in characters of /@editable@/.
editableSetMaxWidthChars ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> Int32
    -- ^ /@nChars@/: the new desired maximum width, in characters
    -> m ()
editableSetMaxWidthChars editable nChars = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    gtk_editable_set_max_width_chars editable' nChars
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableSetMaxWidthCharsMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableSetMaxWidthCharsMethodInfo a signature where
    overloadedMethod = editableSetMaxWidthChars

instance O.OverloadedMethodInfo EditableSetMaxWidthCharsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableSetMaxWidthChars",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableSetMaxWidthChars"
        })


#endif

-- method Editable::set_position
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "position"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the position of the cursor"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_set_position" gtk_editable_set_position :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    Int32 ->                                -- position : TBasicType TInt
    IO ()

-- | Sets the cursor position in the editable to the given value.
-- 
-- The cursor is displayed before the character with the given (base 0)
-- index in the contents of the editable. The value must be less than
-- or equal to the number of characters in the editable. A value of -1
-- indicates that the position should be set after the last character
-- of the editable. Note that /@position@/ is in characters, not in bytes.
editableSetPosition ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> Int32
    -- ^ /@position@/: the position of the cursor
    -> m ()
editableSetPosition editable position = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    gtk_editable_set_position editable' position
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableSetPositionMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableSetPositionMethodInfo a signature where
    overloadedMethod = editableSetPosition

instance O.OverloadedMethodInfo EditableSetPositionMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableSetPosition",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableSetPosition"
        })


#endif

-- method Editable::set_text
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the text to set" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_set_text" gtk_editable_set_text :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    CString ->                              -- text : TBasicType TUTF8
    IO ()

-- | Sets the text in the editable to the given value.
-- 
-- This is replacing the current contents.
editableSetText ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> T.Text
    -- ^ /@text@/: the text to set
    -> m ()
editableSetText editable text = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    text' <- textToCString text
    gtk_editable_set_text editable' text'
    touchManagedPtr editable
    freeMem text'
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableSetTextMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableSetTextMethodInfo a signature where
    overloadedMethod = editableSetText

instance O.OverloadedMethodInfo EditableSetTextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableSetText",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableSetText"
        })


#endif

-- method Editable::set_width_chars
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "editable"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Editable" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkEditable`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_chars"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "width in chars" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_set_width_chars" gtk_editable_set_width_chars :: 
    Ptr Editable ->                         -- editable : TInterface (Name {namespace = "Gtk", name = "Editable"})
    Int32 ->                                -- n_chars : TBasicType TInt
    IO ()

-- | Changes the size request of the editable to be about the
-- right size for /@nChars@/ characters.
-- 
-- Note that it changes the size request, the size can still
-- be affected by how you pack the widget into containers.
-- If /@nChars@/ is -1, the size reverts to the default size.
editableSetWidthChars ::
    (B.CallStack.HasCallStack, MonadIO m, IsEditable a) =>
    a
    -- ^ /@editable@/: a @GtkEditable@
    -> Int32
    -- ^ /@nChars@/: width in chars
    -> m ()
editableSetWidthChars editable nChars = liftIO $ do
    editable' <- unsafeManagedPtrCastPtr editable
    gtk_editable_set_width_chars editable' nChars
    touchManagedPtr editable
    return ()

#if defined(ENABLE_OVERLOADING)
data EditableSetWidthCharsMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsEditable a) => O.OverloadedMethod EditableSetWidthCharsMethodInfo a signature where
    overloadedMethod = editableSetWidthChars

instance O.OverloadedMethodInfo EditableSetWidthCharsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable.editableSetWidthChars",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#v:editableSetWidthChars"
        })


#endif

-- method Editable::delegate_get_property
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GObject`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "prop_id"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a property ID" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TGValue
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "value to set" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pspec"
--           , argType = TParamSpec
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GParamSpec` for the property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_delegate_get_property" gtk_editable_delegate_get_property :: 
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    Word32 ->                               -- prop_id : TBasicType TUInt
    Ptr GValue ->                           -- value : TGValue
    Ptr GParamSpec ->                       -- pspec : TParamSpec
    IO CInt

-- | Gets a property of the @GtkEditable@ delegate for /@object@/.
-- 
-- This is helper function that should be called in the @get_property@
-- function of your @GtkEditable@ implementation, before handling your
-- own properties.
editableDelegateGetProperty ::
    (B.CallStack.HasCallStack, MonadIO m, GObject.Object.IsObject a) =>
    a
    -- ^ /@object@/: a @GObject@
    -> Word32
    -- ^ /@propId@/: a property ID
    -> GValue
    -- ^ /@value@/: value to set
    -> GParamSpec
    -- ^ /@pspec@/: the @GParamSpec@ for the property
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the property was found
editableDelegateGetProperty object propId value pspec = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    value' <- unsafeManagedPtrGetPtr value
    pspec' <- unsafeManagedPtrGetPtr pspec
    result <- gtk_editable_delegate_get_property object' propId value' pspec'
    let result' = (/= 0) result
    touchManagedPtr object
    touchManagedPtr value
    touchManagedPtr pspec
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Editable::delegate_set_property
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GObject`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "prop_id"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a property ID" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TGValue
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "value to set" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pspec"
--           , argType = TParamSpec
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the `GParamSpec` for the property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_delegate_set_property" gtk_editable_delegate_set_property :: 
    Ptr GObject.Object.Object ->            -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    Word32 ->                               -- prop_id : TBasicType TUInt
    Ptr GValue ->                           -- value : TGValue
    Ptr GParamSpec ->                       -- pspec : TParamSpec
    IO CInt

-- | Sets a property on the @GtkEditable@ delegate for /@object@/.
-- 
-- This is a helper function that should be called in the @set_property@
-- function of your @GtkEditable@ implementation, before handling your
-- own properties.
editableDelegateSetProperty ::
    (B.CallStack.HasCallStack, MonadIO m, GObject.Object.IsObject a) =>
    a
    -- ^ /@object@/: a @GObject@
    -> Word32
    -- ^ /@propId@/: a property ID
    -> GValue
    -- ^ /@value@/: value to set
    -> GParamSpec
    -- ^ /@pspec@/: the @GParamSpec@ for the property
    -> m Bool
    -- ^ __Returns:__ 'P.True' if the property was found
editableDelegateSetProperty object propId value pspec = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    value' <- unsafeManagedPtrGetPtr value
    pspec' <- unsafeManagedPtrGetPtr pspec
    result <- gtk_editable_delegate_set_property object' propId value' pspec'
    let result' = (/= 0) result
    touchManagedPtr object
    touchManagedPtr value
    touchManagedPtr pspec
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Editable::install_properties
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "object_class"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "ObjectClass" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GObjectClass`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "first_prop"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "property ID to use for the first property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_editable_install_properties" gtk_editable_install_properties :: 
    Ptr GObject.ObjectClass.ObjectClass ->  -- object_class : TInterface (Name {namespace = "GObject", name = "ObjectClass"})
    Word32 ->                               -- first_prop : TBasicType TUInt
    IO Word32

-- | Overrides the @GtkEditable@ properties for /@class@/.
-- 
-- This is a helper function that should be called in class_init,
-- after installing your own properties.
-- 
-- Note that your class must have \"text\", \"cursor-position\",
-- \"selection-bound\", \"editable\", \"width-chars\", \"max-width-chars\",
-- \"xalign\" and \"enable-undo\" properties for this function to work.
-- 
-- To handle the properties in your set_property and get_property
-- functions, you can either use 'GI.Gtk.Interfaces.Editable.editableDelegateSetProperty'
-- and 'GI.Gtk.Interfaces.Editable.editableDelegateGetProperty' (if you are using
-- a delegate), or remember the /@firstProp@/ offset and add it to the
-- values in the t'GI.Gtk.Enums.EditableProperties' enumeration to get the
-- property IDs for these properties.
editableInstallProperties ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GObject.ObjectClass.ObjectClass
    -- ^ /@objectClass@/: a @GObjectClass@
    -> Word32
    -- ^ /@firstProp@/: property ID to use for the first property
    -> m Word32
    -- ^ __Returns:__ the number of properties that were installed
editableInstallProperties objectClass firstProp = liftIO $ do
    objectClass' <- unsafeManagedPtrGetPtr objectClass
    result <- gtk_editable_install_properties objectClass' firstProp
    touchManagedPtr objectClass
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- signal Editable::changed
-- | Emitted at the end of a single user-visible operation on the
-- contents.
-- 
-- E.g., a paste operation that replaces the contents of the
-- selection will cause only one signal emission (even though it
-- is implemented by first deleting the selection, then inserting
-- the new content, and may cause multiple [notify](#g:signal:notify)[text](#g:signal:text) signals
-- to be emitted).
type EditableChangedCallback =
    IO ()

type C_EditableChangedCallback =
    Ptr Editable ->                         -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_EditableChangedCallback`.
foreign import ccall "wrapper"
    mk_EditableChangedCallback :: C_EditableChangedCallback -> IO (FunPtr C_EditableChangedCallback)

wrap_EditableChangedCallback :: 
    GObject a => (a -> EditableChangedCallback) ->
    C_EditableChangedCallback
wrap_EditableChangedCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [changed](#signal:changed) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' editable #changed callback
-- @
-- 
-- 
onEditableChanged :: (IsEditable a, MonadIO m) => a -> ((?self :: a) => EditableChangedCallback) -> m SignalHandlerId
onEditableChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_EditableChangedCallback wrapped
    wrapped'' <- mk_EditableChangedCallback wrapped'
    connectSignalFunPtr obj "changed" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [changed](#signal:changed) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' editable #changed callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterEditableChanged :: (IsEditable a, MonadIO m) => a -> ((?self :: a) => EditableChangedCallback) -> m SignalHandlerId
afterEditableChanged obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_EditableChangedCallback wrapped
    wrapped'' <- mk_EditableChangedCallback wrapped'
    connectSignalFunPtr obj "changed" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data EditableChangedSignalInfo
instance SignalInfo EditableChangedSignalInfo where
    type HaskellCallbackType EditableChangedSignalInfo = EditableChangedCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_EditableChangedCallback cb
        cb'' <- mk_EditableChangedCallback cb'
        connectSignalFunPtr obj "changed" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable::changed"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:signal:changed"})

#endif

-- signal Editable::delete-text
-- | Emitted when text is deleted from the widget by the user.
-- 
-- The default handler for this signal will normally be responsible for
-- deleting the text, so by connecting to this signal and then stopping
-- the signal with 'GI.GObject.Functions.signalStopEmission', it is possible to modify the
-- range of deleted text, or prevent it from being deleted entirely.
-- 
-- The /@startPos@/ and /@endPos@/ parameters are interpreted as for
-- 'GI.Gtk.Interfaces.Editable.editableDeleteText'.
type EditableDeleteTextCallback =
    Int32
    -- ^ /@startPos@/: the starting position
    -> Int32
    -- ^ /@endPos@/: the end position
    -> IO ()

type C_EditableDeleteTextCallback =
    Ptr Editable ->                         -- object
    Int32 ->
    Int32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_EditableDeleteTextCallback`.
foreign import ccall "wrapper"
    mk_EditableDeleteTextCallback :: C_EditableDeleteTextCallback -> IO (FunPtr C_EditableDeleteTextCallback)

wrap_EditableDeleteTextCallback :: 
    GObject a => (a -> EditableDeleteTextCallback) ->
    C_EditableDeleteTextCallback
wrap_EditableDeleteTextCallback gi'cb gi'selfPtr startPos endPos _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  startPos endPos


-- | Connect a signal handler for the [deleteText](#signal:deleteText) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' editable #deleteText callback
-- @
-- 
-- 
onEditableDeleteText :: (IsEditable a, MonadIO m) => a -> ((?self :: a) => EditableDeleteTextCallback) -> m SignalHandlerId
onEditableDeleteText obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_EditableDeleteTextCallback wrapped
    wrapped'' <- mk_EditableDeleteTextCallback wrapped'
    connectSignalFunPtr obj "delete-text" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [deleteText](#signal:deleteText) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' editable #deleteText callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterEditableDeleteText :: (IsEditable a, MonadIO m) => a -> ((?self :: a) => EditableDeleteTextCallback) -> m SignalHandlerId
afterEditableDeleteText obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_EditableDeleteTextCallback wrapped
    wrapped'' <- mk_EditableDeleteTextCallback wrapped'
    connectSignalFunPtr obj "delete-text" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data EditableDeleteTextSignalInfo
instance SignalInfo EditableDeleteTextSignalInfo where
    type HaskellCallbackType EditableDeleteTextSignalInfo = EditableDeleteTextCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_EditableDeleteTextCallback cb
        cb'' <- mk_EditableDeleteTextCallback cb'
        connectSignalFunPtr obj "delete-text" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable::delete-text"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:signal:deleteText"})

#endif

-- signal Editable::insert-text
-- | Emitted when text is inserted into the widget by the user.
-- 
-- The default handler for this signal will normally be responsible
-- for inserting the text, so by connecting to this signal and then
-- stopping the signal with 'GI.GObject.Functions.signalStopEmission', it is possible
-- to modify the inserted text, or prevent it from being inserted entirely.
type EditableInsertTextCallback =
    T.Text
    -- ^ /@text@/: the new text to insert
    -> Int32
    -- ^ /@length@/: the length of the new text, in bytes,
    --     or -1 if new_text is nul-terminated
    -> Int32
    -- ^ /@position@/: the position, in characters,
    --     at which to insert the new text. this is an in-out
    --     parameter.  After the signal emission is finished, it
    --     should point after the newly inserted text.
    -> IO (Int32)

type C_EditableInsertTextCallback =
    Ptr Editable ->                         -- object
    CString ->
    Int32 ->
    Ptr Int32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_EditableInsertTextCallback`.
foreign import ccall "wrapper"
    mk_EditableInsertTextCallback :: C_EditableInsertTextCallback -> IO (FunPtr C_EditableInsertTextCallback)

wrap_EditableInsertTextCallback :: 
    GObject a => (a -> EditableInsertTextCallback) ->
    C_EditableInsertTextCallback
wrap_EditableInsertTextCallback gi'cb gi'selfPtr text length_ position _ = do
    text' <- cstringToText text
    position' <- peek position
    outposition <- B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  text' length_ position'
    poke position outposition


-- | Connect a signal handler for the [insertText](#signal:insertText) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' editable #insertText callback
-- @
-- 
-- 
onEditableInsertText :: (IsEditable a, MonadIO m) => a -> ((?self :: a) => EditableInsertTextCallback) -> m SignalHandlerId
onEditableInsertText obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_EditableInsertTextCallback wrapped
    wrapped'' <- mk_EditableInsertTextCallback wrapped'
    connectSignalFunPtr obj "insert-text" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [insertText](#signal:insertText) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' editable #insertText callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterEditableInsertText :: (IsEditable a, MonadIO m) => a -> ((?self :: a) => EditableInsertTextCallback) -> m SignalHandlerId
afterEditableInsertText obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_EditableInsertTextCallback wrapped
    wrapped'' <- mk_EditableInsertTextCallback wrapped'
    connectSignalFunPtr obj "insert-text" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data EditableInsertTextSignalInfo
instance SignalInfo EditableInsertTextSignalInfo where
    type HaskellCallbackType EditableInsertTextSignalInfo = EditableInsertTextCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_EditableInsertTextCallback cb
        cb'' <- mk_EditableInsertTextCallback cb'
        connectSignalFunPtr obj "insert-text" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Editable::insert-text"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Editable.html#g:signal:insertText"})

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Editable = EditableSignalList
type EditableSignalList = ('[ '("changed", EditableChangedSignalInfo), '("deleteText", EditableDeleteTextSignalInfo), '("destroy", Gtk.Widget.WidgetDestroySignalInfo), '("directionChanged", Gtk.Widget.WidgetDirectionChangedSignalInfo), '("hide", Gtk.Widget.WidgetHideSignalInfo), '("insertText", EditableInsertTextSignalInfo), '("keynavFailed", Gtk.Widget.WidgetKeynavFailedSignalInfo), '("map", Gtk.Widget.WidgetMapSignalInfo), '("mnemonicActivate", Gtk.Widget.WidgetMnemonicActivateSignalInfo), '("moveFocus", Gtk.Widget.WidgetMoveFocusSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("queryTooltip", Gtk.Widget.WidgetQueryTooltipSignalInfo), '("realize", Gtk.Widget.WidgetRealizeSignalInfo), '("show", Gtk.Widget.WidgetShowSignalInfo), '("stateFlagsChanged", Gtk.Widget.WidgetStateFlagsChangedSignalInfo), '("unmap", Gtk.Widget.WidgetUnmapSignalInfo), '("unrealize", Gtk.Widget.WidgetUnrealizeSignalInfo)] :: [(Symbol, DK.Type)])

#endif


