// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A9__F190

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 8, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 8, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 8, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 8, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 8, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 8, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x)xyzw" };
#else
{ R"xyzw(amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_512_3_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1201)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0x7abe2d68u, 0x07aee943u, 0, 126 }, // 7abe2d6807aee943 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__16_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x8b07eeb0u, 0x8f33ee1cu, 0, 86 }, // 8b07eeb08f33ee1c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__16_16__CO__wave3_warp4_stg1--Arch_gfx1201
    { 0x45a1e73cu, 0xc3f773fcu, 0, 206 }, // 45a1e73cc3f773fc = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__16_16__CO__wave3_warp8_stg1--Arch_gfx1201
    { 0xd2a33480u, 0x50f092e3u, 0, 143 }, // d2a3348050f092e3 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1201
    { 0x35a74f07u, 0x17ecfef9u, 0, 223 }, // 35a74f0717ecfef9 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__16_16__CO__wave4_warp8_stg1--Arch_gfx1201
    { 0xb0d989aeu, 0x7d28f37cu, 23, 126 }, // b0d989ae7d28f37c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__32_16__CO__wave2_warp4_stg1--Arch_gfx1201
    { 0x055abc11u, 0x35c78fadu, 23, 86 }, // 055abc1135c78fad = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__32_16__CO__wave3_warp4_stg1--Arch_gfx1201
    { 0xaa43e65eu, 0xe0b732d1u, 23, 143 }, // aa43e65ee0b732d1 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1201
    { 0xaedf4965u, 0x4c63853cu, 103, 172 }, // aedf49654c63853c = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__64_16__CO__wave1_warp8_stg1--Arch_gfx1201
    { 0xcbe3e064u, 0x77bfcac5u, 103, 189 }, // cbe3e06477bfcac5 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__64_16__CO__wave2_warp8_stg1--Arch_gfx1201
    { 0x2d7da136u, 0x0d75fe64u, 103, 206 }, // 2d7da1360d75fe64 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__64_16__CO__wave3_warp8_stg1--Arch_gfx1201
    { 0x5ce02bbfu, 0xd2d29ae9u, 103, 223 }, // 5ce02bbfd2d29ae9 = b2sum -l 64 <<< amd-gfx120x/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_512_3_T_T_0___gfx120x__P__64_16__CO__wave4_warp8_stg1--Arch_gfx1201
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1201_mod0
{{ 0, 4, 4, 4, 0, 3, 3, 3, 1, 3},
 { 3, 4, 4, 4, 1, 3, 3, 1, 1, 4},
 { 3, 4, 3, 3, 3, 3, 3, 3, 4, 4},
 { 3,11, 3, 3, 1, 3, 3, 4, 4, 2},
 { 6, 8, 6, 6, 7, 3,11, 4, 4, 2},
 { 7, 9,11, 9, 9,10, 9,11, 9, 7},
 { 6,11,11, 9, 8,10, 9,10, 9, 8},
 { 6, 8, 8, 8,10,11, 8,11,11,11},
 { 7, 8,10, 8,11,11, 8,10,11,10},
 { 5, 9, 8,11,11, 8,10,10,10,10}}
// End of GPU gfx1201_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A9__F190 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 512
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx1201_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 512 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

