// Copyright © 2023-2025 Advanced Micro Devices, Inc.
// SPDX-License-Identifier: MIT

#include "../shim.bwd_kernel_dk_dv.h"
#include <aotriton/_internal/triton_kernel.h>
#include <aotriton/_internal/kernel_cluster.h>
#include <aotriton/cpp_tune.h>
#include <string_view>
#ifndef NDEBUG
#include <iostream>
#endif

#define CURRENT_ENTRY_PUBLIC Autotune_bwd_kernel_dk_dv__A7__F78

#define ARRAY_SIZE(array)  (sizeof(array) / sizeof(array[0]))

namespace { // Anonymous namespace

using namespace std::literals::string_view_literals;

#if AOTRITON_BUILD_FOR_TUNING
// PSels and Copts in JSON String
static const char* kernel_psels[] = {
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 16, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 32, "BLOCK_N": 16})xyzw",
  R"xyzw({"BLOCK_M": 64, "BLOCK_N": 16})xyzw"
};
static const char* kernel_copts[] = {
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 1, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 2, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 3, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 2, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw",
  R"xyzw({"waves_per_eu": 4, "num_warps": 4, "num_stages": 1})xyzw"
};
#endif

struct PerfFields {
  int16_t BLOCK_M;
    int16_t BLOCK_N;
};

static PerfFields image_perf_list [] = {
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 16, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 32, .BLOCK_N = 16 },
    { .BLOCK_M = 64, .BLOCK_N = 16 }
};

// u8R generates char8_t which is poorly supported almost everywhere.
constexpr pstring_view PACKAGE_PATH
#if defined(_WIN32)
{ LR"xyzw(amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx)xyzw" };
#else
{ R"xyzw(amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx)xyzw" };
#endif
constexpr std::string_view FUNC_NAME { R"xyzw(＊fp16@16_80_3_T_T_0)xyzw" };
constexpr std::string_view ARCH_NAME { R"xyzw(gfx1152)xyzw" };

// Checksum can be confirmed with `echo -n '<string>' | b2sum -l 64`
// For example:
//   $ echo -n 'amd-gfx110x/flash/attn_fwd/FONLY__^bf16@16,128,False,False,0,0___gfx1100__P__32_32_0_2_False__CO__wave3_warp2_stg1-Gpu-gfx1100' | b2sum -l 64
//   c4b51ee645d79580  -
static AOTRITON_NS::TritonKernelCompactMeta meta_list[] = {
    { 0xff2eb45eu, 0xf00b3474u, 0, 29 }, // ff2eb45ef00b3474 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__16_16__CO__wave1_warp2_stg1--Arch_gfx1152
    { 0xabd6c2b9u, 0xe7788b07u, 0, 109 }, // abd6c2b9e7788b07 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__16_16__CO__wave1_warp4_stg1--Arch_gfx1152
    { 0x3c432e52u, 0x338b8331u, 0, 126 }, // 3c432e52338b8331 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__16_16__CO__wave2_warp4_stg1--Arch_gfx1152
    { 0xbb85287au, 0x291d8826u, 0, 63 }, // bb85287a291d8826 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__16_16__CO__wave4_warp2_stg1--Arch_gfx1152
    { 0xdea4ab63u, 0x2542255bu, 0, 143 }, // dea4ab632542255b = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__16_16__CO__wave4_warp4_stg1--Arch_gfx1152
    { 0x356ecbf7u, 0x4b7d4d5du, 23, 29 }, // 356ecbf74b7d4d5d = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__32_16__CO__wave1_warp2_stg1--Arch_gfx1152
    { 0xe1638297u, 0xdaf217ffu, 23, 109 }, // e1638297daf217ff = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__32_16__CO__wave1_warp4_stg1--Arch_gfx1152
    { 0x81ab9695u, 0x877c7e62u, 23, 6 }, // 81ab9695877c7e62 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__32_16__CO__wave2_warp2_stg1--Arch_gfx1152
    { 0xdaeabf0eu, 0x595f3ab5u, 23, 46 }, // daeabf0e595f3ab5 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__32_16__CO__wave3_warp2_stg1--Arch_gfx1152
    { 0x844fad12u, 0x3b0f3a0du, 23, 63 }, // 844fad123b0f3a0d = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__32_16__CO__wave4_warp2_stg1--Arch_gfx1152
    { 0x546aaa99u, 0x405e6c8du, 23, 143 }, // 546aaa99405e6c8d = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__32_16__CO__wave4_warp4_stg1--Arch_gfx1152
    { 0xcdf95a42u, 0x701f68a9u, 103, 143 }, // cdf95a42701f68a9 = b2sum -l 64 <<< amd-gfx11xx/flash/bwd_kernel_dk_dv/FONLY__＊fp16@16_80_3_T_T_0___gfx11xx__P__64_16__CO__wave4_warp4_stg1--Arch_gfx1152
};

static constexpr int kTotalNumKernels = ARRAY_SIZE(meta_list);

using AOTRITON_NS::v3::flash::autotune::bwd_kernel_dk_dv_packed_string;

static AOTRITON_NS::TritonKernelCluster<kTotalNumKernels>
kernel_cluster(meta_list,
               bwd_kernel_dk_dv_packed_string);

static int8_t lut[1][10][10] =
{
// GPU gfx1152_mod0
{{ 4, 4, 4, 4, 0, 0, 0, 0, 0, 0},
 { 4, 4, 2, 4, 3, 3, 0, 0, 0, 0},
 { 4, 3, 0, 3, 0, 0, 3, 0, 0, 0},
 { 4, 0, 3, 0, 3, 0, 0, 0, 0, 0},
 { 4, 0, 3, 3, 3, 0, 0, 0, 0, 0},
 { 1, 0, 3, 7, 3, 3, 3, 3, 0, 1},
 { 6, 8, 7, 9, 3, 3, 3, 3,11, 1},
 {11, 5, 5, 5, 7, 3, 3, 8, 7, 7},
 {11, 8, 9, 5, 3, 3, 5, 3, 3, 3},
 {11, 5, 7, 7, 5,10, 7,10, 3, 3}}
// End of GPU gfx1152_mod0
}
;

}; // End of anonymous namespace

namespace AOTRITON_NS::v3::flash::autotune {

// using AOTRITON_NS::v2::flash::BwdKernelDkDvContext;

void CURRENT_ENTRY_PUBLIC(BwdKernelDkDvContext& context, int mod_number) {
#if AOTRITON_BUILD_FOR_TUNING
    int preferred_index = context._has_preferred_kernel;
    context._total_number_of_kernels = kTotalNumKernels;
#ifndef NDEBUG
    std::cerr << "Autotune_bwd_kernel_dk_dv__A7__F78 "
              << "kTotalNumKernels = " << kTotalNumKernels << " "
              << "_has_preferred_kernel = " << preferred_index << " "
              << std::endl;
#endif
    if (preferred_index != -1) {
        if (preferred_index >= kTotalNumKernels)
            return ;
        context.kernel_on_device = kernel_cluster.get(preferred_index);
        context.pp_args_index = 5;
        context.package_path = PACKAGE_PATH;
        context.func_name = FUNC_NAME;
        context.arch_name = ARCH_NAME;
        context._preferred_kernel_psels = kernel_psels[preferred_index];
        context._preferred_kernel_copts = kernel_copts[preferred_index];
        const auto& perf = image_perf_list[preferred_index];
        context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
        return ;
    }
#endif
    auto kernel_index = bwd_kernel_dk_dv__lut_lambda__0(*context.params, mod_number, lut);
    if (kernel_index < 0) {
      return ;
    }
    context.kernel_on_device = kernel_cluster.get(kernel_index);
    context.pp_args_index = 5;
    context.package_path = PACKAGE_PATH;
    context.func_name = FUNC_NAME;
    context.arch_name = ARCH_NAME;
#ifndef NDEBUG
    std::cerr << __FILE__ << " kernel_index = " << int(kernel_index) << std::endl;
#endif
    const auto& perf = image_perf_list[kernel_index];
    context.BLOCK_M = perf.BLOCK_M;
    context.BLOCK_N = perf.BLOCK_N;
}

#undef CURRENT_ENTRY_PUBLIC
}

// Human-readable Signature 
// Q = "*fp16:16"
// sm_scale = "fp32"
// L = "*fp32:16"
// D = "*fp32:16"
// stride_bz = 0
// num_head_q = "i32"
// cu_seqlens_q = "*i32:16"
// num_seqlens = "i32"
// dropout_p = "fp32"
// philox_seed_ptr = "*u64"
// philox_offset1 = "*u64"
// philox_offset2 = "u64"
// Window_left = "i32"
// BLOCK_DMODEL = 80
// CAUSAL_TYPE = 3
// ENABLE_DROPOUT = True
// PADDED_HEAD = True
// BIAS_TYPE = 0

// SELECT * FROM FLASH$bwd_kernel_dk_dv WHERE gpu IN ('gfx1100_mod0') AND inputs$Q_dtype = 'torch.float16' AND inputs$BLOCK_DMODEL = 80 AND inputs$CAUSAL_TYPE = 3 AND inputs$ENABLE_DROPOUT = True AND inputs$PADDED_HEAD = False AND inputs$BIAS_TYPE = 0

// vim: set fileencoding=utf-8

