/* bz-window.c
 *
 * Copyright 2025 Adam Masciola
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <https://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: GPL-3.0-or-later
 */

// This file is an utter mess
#include "config.h"

#include <glib/gi18n.h>

#include "bz-comet-overlay.h"
#include "bz-curated-view.h"
#include "bz-entry-group.h"
#include "bz-entry-inspector.h"
#include "bz-env.h"
#include "bz-error.h"
#include "bz-favorites-page.h"
#include "bz-flathub-page.h"
#include "bz-full-view.h"
#include "bz-global-progress.h"
#include "bz-installed-page.h"
#include "bz-io.h"
#include "bz-progress-bar.h"
#include "bz-search-widget.h"
#include "bz-transaction-manager.h"
#include "bz-update-dialog.h"
#include "bz-user-data-page.h"
#include "bz-util.h"
#include "bz-window.h"

struct _BzWindow
{
  AdwApplicationWindow parent_instance;

  BzStateInfo *state;

  GtkEventController *key_controller;

  gboolean breakpoint_applied;

  /* Template widgets */
  BzCometOverlay      *comet_overlay;
  AdwOverlaySplitView *split_view;
  AdwViewStack        *transactions_stack;
  AdwNavigationView   *navigation_view;
  BzFullView          *full_view;
  GtkToggleButton     *toggle_transactions;
  GtkToggleButton     *toggle_transactions_sidebar;
  BzSearchWidget      *search_widget;
  GtkButton           *update_button;
  GtkToggleButton     *transactions_pause;
  GtkButton           *transactions_stop;
  GtkButton           *transactions_clear;
  AdwToastOverlay     *toasts;
  AdwViewStack        *main_view_stack;
  GtkStack            *main_stack;
  GtkLabel            *debug_id_label;
};

G_DEFINE_FINAL_TYPE (BzWindow, bz_window, ADW_TYPE_APPLICATION_WINDOW)

enum
{
  PROP_0,

  PROP_STATE,

  LAST_PROP
};
static GParamSpec *props[LAST_PROP] = { 0 };

BZ_DEFINE_DATA (
    transact,
    Transact,
    {
      GWeakRef     *self;
      BzEntry      *entry;
      BzEntryGroup *group;
      gboolean      remove;
      gboolean      auto_confirm;
      GtkWidget    *source;
    },
    BZ_RELEASE_DATA (self, bz_weak_release);
    BZ_RELEASE_DATA (entry, g_object_unref);
    BZ_RELEASE_DATA (group, g_object_unref);
    BZ_RELEASE_DATA (source, g_object_unref))
static DexFuture *
transact_fiber (TransactData *data);

static void
update_dialog_response (BzUpdateDialog *dialog,
                        const char     *response,
                        BzWindow       *self);

static void
configure_install_dialog (AdwAlertDialog *alert,
                          const char     *title,
                          const char     *id);

static void
configure_remove_dialog (AdwAlertDialog *alert,
                         const char     *title,
                         const char     *id);

static GPtrArray *
create_entry_radio_buttons (AdwAlertDialog *alert,
                            GListStore     *store,
                            gboolean        remove);

static GtkWidget *
create_entry_radio_button (BzEntry    *entry,
                           GtkWidget **out_radio);

static DexFuture *
transact (BzWindow  *self,
          BzEntry   *entry,
          gboolean   remove,
          GtkWidget *source);

static void
try_transact (BzWindow     *self,
              BzEntry      *entry,
              BzEntryGroup *group,
              gboolean      remove,
              gboolean      auto_confirm,
              GtkWidget    *source);

static void
update (BzWindow *self,
        BzEntry **updates,
        guint     n_updates);

static void
search (BzWindow   *self,
        const char *text);

static void
check_transactions (BzWindow *self);

static void
set_page (BzWindow *self);

static void
bz_window_dispose (GObject *object)
{
  BzWindow *self = BZ_WINDOW (object);

  g_clear_object (&self->state);

  G_OBJECT_CLASS (bz_window_parent_class)->dispose (object);
}

static void
bz_window_get_property (GObject    *object,
                        guint       prop_id,
                        GValue     *value,
                        GParamSpec *pspec)
{
  BzWindow *self = BZ_WINDOW (object);

  switch (prop_id)
    {
    case PROP_STATE:
      g_value_set_object (value, self->state);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
bz_window_set_property (GObject      *object,
                        guint         prop_id,
                        const GValue *value,
                        GParamSpec   *pspec)
{
  // BzWindow *self = BZ_WINDOW (object);

  switch (prop_id)
    {
    case PROP_STATE:
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static gboolean
invert_boolean (gpointer object,
                gboolean value)
{
  return !value;
}

static gboolean
is_double_zero (gpointer object,
                double   value)
{
  return value == 0.0;
}

static gboolean
is_null (gpointer object,
         GObject *value)
{
  return value == NULL;
}

static gboolean
logical_and (gpointer object,
             gboolean value1,
             gboolean value2)
{
  return value1 && value2;
}

static gboolean
logical_or (gpointer object,
            gboolean value1,
            gboolean value2)
{
  return value1 || value2;
}

static char *
list_length (gpointer    object,
             GListModel *model)
{
  if (model == NULL)
    return g_strdup (0);

  return g_strdup_printf ("%u", g_list_model_get_n_items (model));
}

static void
browser_group_selected_cb (BzWindow     *self,
                           BzEntryGroup *group,
                           gpointer      browser)
{
  bz_window_show_group (self, group);
}

static void
search_widget_select_cb (BzWindow       *self,
                         BzEntryGroup   *group,
                         gboolean        should_install,
                         BzSearchWidget *search)
{
  if (should_install)
    {
      int      installable = 0;
      int      removable   = 0;
      gboolean remove      = FALSE;

      g_object_get (
          group,
          "installable", &installable,
          "removable", &removable,
          NULL);

      remove = removable > 0;
      try_transact (self, NULL, group, remove, FALSE, NULL);
    }
  else
    {
      bz_window_show_group (self, group);
    }
}

static void
full_view_install_cb (BzWindow   *self,
                      GtkWidget  *source,
                      BzFullView *view)
{
  try_transact (self, NULL, bz_full_view_get_entry_group (view), FALSE, TRUE, source);
}

static void
full_view_remove_cb (BzWindow   *self,
                     GtkWidget  *source,
                     BzFullView *view)
{
  try_transact (self, NULL, bz_full_view_get_entry_group (view), TRUE, FALSE, source);
}

static void
install_addon_cb (BzWindow   *self,
                  BzEntry    *entry,
                  BzFullView *view)
{
  try_transact (self, entry, NULL, FALSE, TRUE, NULL);
}

static void
remove_addon_cb (BzWindow   *self,
                 BzEntry    *entry,
                 BzFullView *view)
{
  try_transact (self, entry, NULL, TRUE, TRUE, NULL);
}

static void
install_entry_cb (BzWindow   *self,
                  BzEntry    *entry,
                  BzFullView *view)
{
  try_transact (self, entry, NULL, FALSE, FALSE, NULL);
}

static void
remove_installed_cb (BzWindow   *self,
                     BzEntry    *entry,
                     BzFullView *view)
{
  try_transact (self, entry, NULL, TRUE, FALSE, NULL);
}

static void
installed_page_show_cb (BzWindow   *self,
                        BzEntry    *entry,
                        BzFullView *view)
{
  g_autoptr (BzEntryGroup) group = NULL;

  group = bz_application_map_factory_convert_one (
      bz_state_info_get_application_factory (self->state),
      gtk_string_object_new (bz_entry_get_id (entry)));

  if (group != NULL)
    bz_window_show_group (self, group);
}

void
bz_window_show_app_id (BzWindow   *self,
                       const char *app_id)
{
  g_autoptr (BzEntryGroup) group = NULL;

  g_return_if_fail (BZ_IS_WINDOW (self));
  g_return_if_fail (app_id != NULL);

  group = bz_application_map_factory_convert_one (
      bz_state_info_get_application_factory (self->state),
      gtk_string_object_new (app_id));

  if (group != NULL)
    bz_window_show_group (self, group);
}

static void
page_toggled_cb (BzWindow       *self,
                 GParamSpec     *pspec,
                 AdwToggleGroup *toggles)
{
  set_page (self);
}

static void
browse_flathub_cb (BzWindow      *self,
                   BzCuratedView *widget)
{
  adw_view_stack_set_visible_child_name (self->main_view_stack, "flathub");
}

static void
open_search_cb (BzWindow       *self,
                BzSearchWidget *widget)
{
  adw_view_stack_set_visible_child_name (self->main_view_stack, "search");
}

static void
breakpoint_apply_cb (BzWindow      *self,
                     AdwBreakpoint *breakpoint)
{
  self->breakpoint_applied = TRUE;

  gtk_widget_add_css_class (GTK_WIDGET (self), "narrow");
}

static void
breakpoint_unapply_cb (BzWindow      *self,
                       AdwBreakpoint *breakpoint)
{
  self->breakpoint_applied = FALSE;

  gtk_widget_remove_css_class (GTK_WIDGET (self), "narrow");
}

static void
pause_transactions_cb (BzWindow        *self,
                       GtkToggleButton *toggle)
{
  gboolean paused = FALSE;

  paused = gtk_toggle_button_get_active (toggle);
  bz_transaction_manager_set_paused (
      bz_state_info_get_transaction_manager (self->state), paused);
  check_transactions (self);
}

static void
stop_transactions_cb (BzWindow  *self,
                      GtkButton *button)
{
  bz_transaction_manager_set_paused (bz_state_info_get_transaction_manager (self->state), TRUE);
  bz_transaction_manager_cancel_current (bz_state_info_get_transaction_manager (self->state));
}

static void
sync_cb (BzWindow  *self,
         GtkButton *button)
{
  g_action_group_activate_action (
      G_ACTION_GROUP (g_application_get_default ()),
      "sync-remotes", NULL);
}

static void
update_cb (BzWindow  *self,
           GtkButton *button)
{
  /* if the button is clickable, there have to be updates */
  bz_window_push_update_dialog (self);
}

static char *
update_amount_tooltip (gpointer    object,
                       GListModel *model)
{
  guint count;

  if (model == NULL)
    count = 0;
  else
    count = g_list_model_get_n_items (model);

  return g_strdup_printf (ngettext ("%d Update Available",
                                    "%d Updates Available",
                                    count),
                          count);
}

static void
transactions_clear_cb (BzWindow  *self,
                       GtkButton *button)
{
  bz_transaction_manager_clear_finished (
      bz_state_info_get_transaction_manager (self->state));
}

static void
action_escape (GtkWidget  *widget,
               const char *action_name,
               GVariant   *parameter)
{
  BzWindow   *self    = BZ_WINDOW (widget);
  GListModel *stack   = NULL;
  guint       n_pages = 0;

  stack   = adw_navigation_view_get_navigation_stack (self->navigation_view);
  n_pages = g_list_model_get_n_items (stack);

  adw_navigation_view_pop (self->navigation_view);
  if (n_pages <= 2)
    {
      gtk_toggle_button_set_active (self->toggle_transactions, FALSE);
      set_page (self);
    }
}

static char *
format_progress (gpointer object,
                 double   value)
{
  return g_strdup_printf ("%.0f%%", 100.0 * value);
}

static void
action_user_data (GtkWidget  *widget,
                  const char *action_name,
                  GVariant   *parameter)
{
  BzWindow          *self           = BZ_WINDOW (widget);
  AdwNavigationPage *user_data_page = NULL;

  user_data_page = ADW_NAVIGATION_PAGE (bz_user_data_page_new (self->state));
  adw_navigation_view_push (self->navigation_view, user_data_page);
}

static void
debug_id_inspect_cb (BzWindow  *self,
                     GtkButton *button)
{
  BzEntryGroup    *group             = NULL;
  g_autofree char *unique_id         = NULL;
  g_autoptr (GtkStringObject) string = NULL;
  g_autoptr (BzResult) result        = NULL;

  group = bz_full_view_get_entry_group (self->full_view);
  if (group == NULL)
    return;
  unique_id = bz_entry_group_dup_ui_entry_id (group);

  result = bz_application_map_factory_convert_one (
      bz_state_info_get_entry_factory (self->state),
      gtk_string_object_new (unique_id));
  if (result != NULL)
    {
      BzEntryInspector *inspector = NULL;

      inspector = bz_entry_inspector_new ();
      bz_entry_inspector_set_result (inspector, result);

      gtk_window_present (GTK_WINDOW (inspector));
    }
}

static void
bz_window_class_init (BzWindowClass *klass)
{
  GObjectClass   *object_class = G_OBJECT_CLASS (klass);
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (klass);

  object_class->dispose      = bz_window_dispose;
  object_class->get_property = bz_window_get_property;
  object_class->set_property = bz_window_set_property;

  props[PROP_STATE] =
      g_param_spec_object (
          "state",
          NULL, NULL,
          BZ_TYPE_STATE_INFO,
          G_PARAM_READABLE);

  g_object_class_install_properties (object_class, LAST_PROP, props);

  g_type_ensure (BZ_TYPE_COMET_OVERLAY);
  g_type_ensure (BZ_TYPE_SEARCH_WIDGET);
  g_type_ensure (BZ_TYPE_GLOBAL_PROGRESS);
  g_type_ensure (BZ_TYPE_PROGRESS_BAR);
  g_type_ensure (BZ_TYPE_CURATED_VIEW);
  g_type_ensure (BZ_TYPE_FULL_VIEW);
  g_type_ensure (BZ_TYPE_INSTALLED_PAGE);
  g_type_ensure (BZ_TYPE_FLATHUB_PAGE);

  gtk_widget_class_set_template_from_resource (widget_class, "/io/github/kolunmi/Bazaar/bz-window.ui");
  gtk_widget_class_bind_template_child (widget_class, BzWindow, comet_overlay);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, split_view);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, transactions_stack);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, navigation_view);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, full_view);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, toasts);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, toggle_transactions);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, toggle_transactions_sidebar);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, search_widget);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, update_button);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, transactions_pause);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, transactions_stop);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, transactions_clear);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, main_view_stack);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, main_stack);
  gtk_widget_class_bind_template_child (widget_class, BzWindow, debug_id_label);
  gtk_widget_class_bind_template_callback (widget_class, invert_boolean);
  gtk_widget_class_bind_template_callback (widget_class, is_double_zero);
  gtk_widget_class_bind_template_callback (widget_class, is_null);
  gtk_widget_class_bind_template_callback (widget_class, logical_and);
  gtk_widget_class_bind_template_callback (widget_class, logical_or);
  gtk_widget_class_bind_template_callback (widget_class, list_length);
  gtk_widget_class_bind_template_callback (widget_class, browser_group_selected_cb);
  gtk_widget_class_bind_template_callback (widget_class, search_widget_select_cb);
  gtk_widget_class_bind_template_callback (widget_class, full_view_install_cb);
  gtk_widget_class_bind_template_callback (widget_class, full_view_remove_cb);
  gtk_widget_class_bind_template_callback (widget_class, install_addon_cb);
  gtk_widget_class_bind_template_callback (widget_class, remove_addon_cb);
  gtk_widget_class_bind_template_callback (widget_class, remove_installed_cb);
  gtk_widget_class_bind_template_callback (widget_class, installed_page_show_cb);
  gtk_widget_class_bind_template_callback (widget_class, page_toggled_cb);
  gtk_widget_class_bind_template_callback (widget_class, breakpoint_apply_cb);
  gtk_widget_class_bind_template_callback (widget_class, breakpoint_unapply_cb);
  gtk_widget_class_bind_template_callback (widget_class, pause_transactions_cb);
  gtk_widget_class_bind_template_callback (widget_class, stop_transactions_cb);
  gtk_widget_class_bind_template_callback (widget_class, sync_cb);
  gtk_widget_class_bind_template_callback (widget_class, update_cb);
  gtk_widget_class_bind_template_callback (widget_class, update_amount_tooltip);
  gtk_widget_class_bind_template_callback (widget_class, transactions_clear_cb);
  gtk_widget_class_bind_template_callback (widget_class, browse_flathub_cb);
  gtk_widget_class_bind_template_callback (widget_class, open_search_cb);
  gtk_widget_class_bind_template_callback (widget_class, format_progress);
  gtk_widget_class_bind_template_callback (widget_class, debug_id_inspect_cb);

  gtk_widget_class_install_action (widget_class, "escape", NULL, action_escape);
  gtk_widget_class_install_action (widget_class, "window.user-data", NULL, action_user_data);
}

static gboolean
key_pressed (BzWindow              *self,
             guint                  keyval,
             guint                  keycode,
             GdkModifierType        state,
             GtkEventControllerKey *controller)
{
  gunichar unichar = 0;
  char     buf[32] = { 0 };

  /* Ignore if this is a modifier-shortcut of some sort */
  if (state & ~(GDK_NO_MODIFIER_MASK | GDK_SHIFT_MASK))
    return FALSE;

  unichar = gdk_keyval_to_unicode (keyval);
  if (unichar == 0 || !g_unichar_isgraph (unichar))
    return FALSE;
  g_unichar_to_utf8 (unichar, buf);

  adw_view_stack_set_visible_child_name (self->main_view_stack, "search");
  return bz_search_widget_ensure_active (self->search_widget, buf);
}

static void
bz_window_init (BzWindow *self)
{
  gtk_widget_init_template (GTK_WIDGET (self));

#ifdef DEVELOPMENT_BUILD
  gtk_widget_add_css_class (GTK_WIDGET (self), "devel");
#endif

  adw_view_stack_set_visible_child_name (self->main_view_stack, "flathub");

  self->key_controller = gtk_event_controller_key_new ();
  g_signal_connect_swapped (self->key_controller,
                            "key-pressed",
                            G_CALLBACK (key_pressed),
                            self);
  gtk_widget_add_controller (GTK_WIDGET (self), self->key_controller);
}

static void
app_busy_changed (BzWindow    *self,
                  GParamSpec  *pspec,
                  BzStateInfo *info)
{
  bz_search_widget_refresh (self->search_widget);
  set_page (self);
}

static void
transactions_active_changed (BzWindow             *self,
                             GParamSpec           *pspec,
                             BzTransactionManager *manager)
{
  check_transactions (self);
}

static void
has_transactions_changed (BzWindow             *self,
                          GParamSpec           *pspec,
                          BzTransactionManager *manager)
{
  check_transactions (self);
}

static void
has_inputs_changed (BzWindow          *self,
                    GParamSpec        *pspec,
                    BzContentProvider *provider)
{
  if (!bz_content_provider_get_has_inputs (provider))
    adw_view_stack_set_visible_child_name (self->main_view_stack, "flathub");
}

static void
checking_for_updates_changed (BzWindow    *self,
                              GParamSpec  *pspec,
                              BzStateInfo *info)
{
  gboolean busy                 = FALSE;
  gboolean checking_for_updates = FALSE;
  gboolean has_updates          = FALSE;

  busy                 = bz_state_info_get_busy (info);
  checking_for_updates = bz_state_info_get_checking_for_updates (info);
  has_updates          = bz_state_info_get_available_updates (info) != NULL;

  if (!busy && !checking_for_updates)
    {
      if (has_updates)
        {
          bz_comet_overlay_set_pulse_color (self->comet_overlay, NULL);
          bz_comet_overlay_pulse_child (
              self->comet_overlay,
              GTK_WIDGET (self->update_button));
        }
      /* TODO: this can be intrusive when idling checking for updates */
      // else
      //   adw_toast_overlay_add_toast (
      //       self->toasts,
      //       adw_toast_new_format (_ ("Up to date!")));
    }
}

static DexFuture *
transact_fiber (TransactData *data)
{
  g_autoptr (BzWindow) self             = NULL;
  BzEntry      *entry                   = data->entry;
  BzEntryGroup *group                   = data->group;
  gboolean      remove                  = data->remove;
  gboolean      auto_confirm            = data->auto_confirm;
  GtkWidget    *source                  = data->source;
  g_autoptr (GError) local_error        = NULL;
  g_autoptr (GListStore) store          = NULL;
  const char      *title                = NULL;
  const char      *id                   = NULL;
  g_autofree char *id_dup               = NULL;
  g_autoptr (AdwDialog) alert           = NULL;
  gboolean delete_user_data             = FALSE;
  g_autoptr (GPtrArray) radios          = NULL;
  g_autofree char *dialog_response      = NULL;
  gboolean         should_install       = FALSE;
  gboolean         should_remove        = FALSE;
  g_autoptr (DexFuture) transact_future = NULL;

  bz_weak_get_or_return_reject (self, data->self);

  if (group != NULL)
    {
      store = dex_await_object (bz_entry_group_dup_all_into_store (group), &local_error);
      if (store == NULL)
        {
          bz_show_error_for_widget (GTK_WIDGET (self), local_error->message);
          return dex_future_new_for_error (g_steal_pointer (&local_error));
        }
      title = bz_entry_group_get_title (group);
      id    = bz_entry_group_get_id (group);
    }
  else
    {
      title = bz_entry_get_title (entry);
      id    = bz_entry_get_id (entry);
    }
  /* id may become invalid after awaiting */
  id_dup = g_strdup (id);

  alert = g_object_ref_sink (adw_alert_dialog_new (NULL, NULL));
  if (remove)
    configure_remove_dialog (ADW_ALERT_DIALOG (alert), title, id);
  else
    configure_install_dialog (ADW_ALERT_DIALOG (alert), title, id);
  id    = NULL;
  title = NULL;

  radios = create_entry_radio_buttons (ADW_ALERT_DIALOG (alert), store, remove);
  if (!remove && auto_confirm && radios->len <= 1)
    {
      dialog_response = g_strdup (remove ? "remove" : "install");
      g_ptr_array_set_size (radios, 0);
      g_clear_object (&alert);
    }
  else
    {
      adw_dialog_present (alert, GTK_WIDGET (self));
      dialog_response = dex_await_string (
          bz_make_alert_dialog_future (ADW_ALERT_DIALOG (alert)),
          &local_error);
      if (dialog_response == NULL)
        return dex_future_new_for_error (g_steal_pointer (&local_error));
      if (remove && radios->len >= 2)
        {
          GtkCheckButton *delete_radio = g_ptr_array_index (radios, radios->len - 1);
          delete_user_data             = gtk_check_button_get_active (delete_radio);
        }
    }

  should_install = g_strcmp0 (dialog_response, "install") == 0;
  should_remove  = g_strcmp0 (dialog_response, "remove") == 0;
  if (!should_install && !should_remove)
    return dex_future_new_false ();

  if (group != NULL)
    {
      guint n_entries                    = 0;
      g_autoptr (BzEntry) selected_entry = NULL;

      n_entries = g_list_model_get_n_items (G_LIST_MODEL (store));
      for (guint i = 0; i < MIN (n_entries, radios->len); i++)
        {
          GtkCheckButton *check = NULL;

          check = g_ptr_array_index (radios, i);
          if (gtk_check_button_get_active (check))
            {
              selected_entry = g_list_model_get_item (G_LIST_MODEL (store), i);
              break;
            }
        }
      if (selected_entry == NULL)
        selected_entry = g_list_model_get_item (G_LIST_MODEL (store), 0);

      transact_future = transact (self, selected_entry, should_remove, source);
    }
  else
    transact_future = transact (self, entry, should_remove, source);
  g_clear_pointer (&radios, g_ptr_array_unref);

  if (!dex_await (g_steal_pointer (&transact_future), &local_error))
    return dex_future_new_for_error (g_steal_pointer (&local_error));

  if (delete_user_data)
    {
      if (group != NULL)
        bz_entry_group_reap_user_data (group);
      else
        dex_future_disown (bz_reap_user_data_dex (id_dup));
    }

  return dex_future_new_true ();
}

static void
update_dialog_response (BzUpdateDialog *dialog,
                        const char     *response,
                        BzWindow       *self)
{
  g_autoptr (GListModel) accepted_model = NULL;

  accepted_model = bz_update_dialog_was_accepted (dialog);
  if (accepted_model != NULL)
    {
      GListModel          *updates     = NULL;
      guint                n_updates   = 0;
      g_autofree BzEntry **updates_buf = NULL;

      updates     = bz_state_info_get_available_updates (self->state);
      n_updates   = g_list_model_get_n_items (updates);
      updates_buf = g_malloc_n (n_updates, sizeof (*updates_buf));

      for (guint i = 0; i < n_updates; i++)
        updates_buf[i] = g_list_model_get_item (updates, i);
      update (self, updates_buf, n_updates);

      for (guint i = 0; i < n_updates; i++)
        g_object_unref (updates_buf[i]);
      bz_state_info_set_available_updates (
          self->state, NULL);
    }
}

BzWindow *
bz_window_new (BzStateInfo *state)
{
  BzWindow *window = NULL;

  g_return_val_if_fail (BZ_IS_STATE_INFO (state), NULL);

  window        = g_object_new (BZ_TYPE_WINDOW, NULL);
  window->state = g_object_ref (state);

  g_signal_connect_object (state,
                           "notify::busy",
                           G_CALLBACK (app_busy_changed),
                           window, G_CONNECT_SWAPPED);
  g_signal_connect_object (state,
                           "notify::checking-for-updates",
                           G_CALLBACK (checking_for_updates_changed),
                           window, G_CONNECT_SWAPPED);

  /* these seem unsafe but BzApplication never
   * changes the objects we are connecting to
   */
  g_signal_connect_object (bz_state_info_get_transaction_manager (state),
                           "notify::active",
                           G_CALLBACK (transactions_active_changed),
                           window, G_CONNECT_SWAPPED);
  g_signal_connect_object (bz_state_info_get_transaction_manager (state),
                           "notify::has-transactions",
                           G_CALLBACK (has_transactions_changed),
                           window, G_CONNECT_SWAPPED);
  g_signal_connect_object (bz_state_info_get_curated_provider (state),
                           "notify::has-inputs",
                           G_CALLBACK (has_inputs_changed),
                           window, G_CONNECT_SWAPPED);

  g_object_notify_by_pspec (G_OBJECT (window), props[PROP_STATE]);

  set_page (window);
  check_transactions (window);
  return window;
}

void
bz_window_search (BzWindow   *self,
                  const char *text)
{
  g_return_if_fail (BZ_IS_WINDOW (self));
  search (self, text);
}

void
bz_window_toggle_transactions (BzWindow *self)
{
  g_return_if_fail (BZ_IS_WINDOW (self));
  gtk_toggle_button_set_active (
      self->toggle_transactions,
      !gtk_toggle_button_get_active (
          self->toggle_transactions));
}

void
bz_window_push_update_dialog (BzWindow *self)
{
  GListModel *available_updates = NULL;
  AdwDialog  *update_dialog     = NULL;

  g_return_if_fail (BZ_IS_WINDOW (self));

  available_updates = bz_state_info_get_available_updates (self->state);
  g_return_if_fail (available_updates != NULL);

  update_dialog = bz_update_dialog_new (available_updates);
  adw_dialog_set_content_width (update_dialog, 750);
  g_signal_connect (update_dialog, "response", G_CALLBACK (update_dialog_response), self);

  adw_dialog_present (update_dialog, GTK_WIDGET (self));
}

void
bz_window_show_entry (BzWindow *self,
                      BzEntry  *entry)
{
  /* TODO: IMPLEMENT ME! */
  bz_show_error_for_widget (
      GTK_WIDGET (self),
      _ ("The ability to inspect and install local .flatpak bundle files is coming soon! "
         "In the meantime, try running\n\n"
         "flatpak install --bundle your-bundle.flatpak\n\n"
         "on the command line."));
}

void
bz_window_show_group (BzWindow     *self,
                      BzEntryGroup *group)
{
  AdwNavigationPage *visible_page = NULL;

  g_return_if_fail (BZ_IS_WINDOW (self));
  g_return_if_fail (BZ_IS_ENTRY_GROUP (group));

  bz_full_view_set_entry_group (self->full_view, group);

  visible_page = adw_navigation_view_get_visible_page (self->navigation_view);
  if (visible_page != adw_navigation_view_find_page (self->navigation_view, "view"))
    adw_navigation_view_push_by_tag (self->navigation_view, "view");
}

void
bz_window_add_toast (BzWindow *self,
                     AdwToast *toast)
{
  g_return_if_fail (BZ_IS_WINDOW (self));
  g_return_if_fail (ADW_IS_TOAST (toast));

  adw_toast_overlay_add_toast (self->toasts, toast);
}

void
bz_window_push_page (BzWindow *self, AdwNavigationPage *page)
{
  g_return_if_fail (BZ_IS_WINDOW (self));
  g_return_if_fail (ADW_IS_NAVIGATION_PAGE (page));

  if (BZ_IS_FAVORITES_PAGE (page))
    {
      g_signal_connect_swapped (page, "install", G_CALLBACK (install_entry_cb), self);
      g_signal_connect_swapped (page, "remove", G_CALLBACK (remove_installed_cb), self);
      g_signal_connect_swapped (page, "show-entry", G_CALLBACK (installed_page_show_cb), self);

      g_object_bind_property (self->split_view, "show-sidebar",
                              page, "show-sidebar",
                              G_BINDING_BIDIRECTIONAL | G_BINDING_SYNC_CREATE);
    }

  adw_navigation_view_push (self->navigation_view, page);
}

BzStateInfo *
bz_window_get_state_info (BzWindow *self)
{
  g_return_val_if_fail (BZ_IS_WINDOW (self), NULL);
  return self->state;
}

static DexFuture *
transact (BzWindow  *self,
          BzEntry   *entry,
          gboolean   remove,
          GtkWidget *source)
{
  g_autoptr (BzTransaction) transaction = NULL;
  GdkPaintable *icon                    = NULL;
  GtkWidget    *transaction_target      = NULL;

  if (remove)
    transaction = bz_transaction_new_full (
        NULL, 0,
        NULL, 0,
        &entry, 1);
  else
    transaction = bz_transaction_new_full (
        &entry, 1,
        NULL, 0,
        NULL, 0);

  if (source == NULL)
    source = GTK_WIDGET (self->navigation_view);

  if (adw_overlay_split_view_get_show_sidebar (self->split_view))
    transaction_target = GTK_WIDGET (self->toggle_transactions_sidebar);
  else
    transaction_target = GTK_WIDGET (self->toggle_transactions);

  icon = bz_entry_get_icon_paintable (entry);
  if (icon != NULL)
    {
      g_autoptr (BzComet) comet = NULL;

      if (remove)
        {
          AdwStyleManager *style_manager = adw_style_manager_get_default ();
          gboolean         is_dark       = adw_style_manager_get_dark (style_manager);
          GdkRGBA          destructive_color;

          if (is_dark)
            destructive_color = (GdkRGBA) { 0.3, 0.2, 0.21, 0.6 };
          else
            destructive_color = (GdkRGBA) { 0.95, 0.84, 0.84, 0.6 };

          bz_comet_overlay_set_pulse_color (self->comet_overlay, &destructive_color);
        }
      else
        bz_comet_overlay_set_pulse_color (self->comet_overlay, NULL);

      comet = g_object_new (
          BZ_TYPE_COMET,
          "from", remove ? transaction_target : source,
          "to", remove ? source : transaction_target,
          "paintable", icon,
          NULL);
      bz_comet_overlay_spawn (self->comet_overlay, comet);
    }

  return bz_transaction_manager_add (
      bz_state_info_get_transaction_manager (self->state),
      transaction);
}

static void
try_transact (BzWindow     *self,
              BzEntry      *entry,
              BzEntryGroup *group,
              gboolean      remove,
              gboolean      auto_confirm,
              GtkWidget    *source)
{
  g_autoptr (TransactData) data = NULL;

  g_return_if_fail (entry != NULL || group != NULL);
  if (bz_state_info_get_busy (self->state))
    {
      adw_toast_overlay_add_toast (
          self->toasts,
          adw_toast_new_format (_ ("Can't do that right now!")));
      return;
    }

  data               = transact_data_new ();
  data->self         = bz_track_weak (self);
  data->entry        = bz_object_maybe_ref (entry);
  data->group        = bz_object_maybe_ref (group);
  data->remove       = remove;
  data->auto_confirm = auto_confirm;
  data->source       = bz_object_maybe_ref (source);

  dex_future_disown (dex_scheduler_spawn (
      dex_scheduler_get_default (),
      bz_get_dex_stack_size (),
      (DexFiberFunc) transact_fiber,
      transact_data_ref (data), transact_data_unref));
}

static gboolean
should_skip_entry (BzEntry *entry,
                   gboolean remove)
{
  gboolean is_installed;

  if (bz_entry_is_holding (entry))
    return TRUE;

  is_installed = bz_entry_is_installed (entry);

  return (!remove && is_installed) || (remove && !is_installed);
}

static GtkWidget *
create_entry_radio_button (BzEntry    *entry,
                           GtkWidget **out_radio)
{
  GtkWidget       *row;
  GtkWidget       *radio;
  g_autofree char *label;

  label = g_strdup (bz_entry_get_unique_id (entry));

  row = adw_action_row_new ();
  adw_preferences_row_set_title (ADW_PREFERENCES_ROW (row), label);

  radio = gtk_check_button_new ();
  gtk_widget_set_valign (radio, GTK_ALIGN_CENTER);
  adw_action_row_add_prefix (ADW_ACTION_ROW (row), radio);
  adw_action_row_set_activatable_widget (ADW_ACTION_ROW (row), radio);

  if (out_radio != NULL)
    *out_radio = radio;

  return row;
}

static GPtrArray *
create_entry_radio_buttons (AdwAlertDialog *alert,
                            GListStore     *store,
                            gboolean        remove)
{
  g_autoptr (GPtrArray) radios = NULL;
  GtkWidget *container         = NULL;

  container = gtk_box_new (GTK_ORIENTATION_VERTICAL, 12);

  radios = g_ptr_array_new ();
  if (store != NULL)
    {
      guint n_valid_entries = 0;

      for (guint i = 0; i < g_list_model_get_n_items (G_LIST_MODEL (store));)
        {
          g_autoptr (BzEntry) entry = NULL;

          entry = g_list_model_get_item (G_LIST_MODEL (store), i);
          if (should_skip_entry (entry, remove))
            {
              g_list_store_remove (store, i);
              continue;
            }
          n_valid_entries++;
          i++;
        }
      if (n_valid_entries > 1)
        {
          GtkWidget      *listbox           = NULL;
          GtkCheckButton *first_valid_radio = NULL;

          listbox = gtk_list_box_new ();
          gtk_list_box_set_selection_mode (GTK_LIST_BOX (listbox), GTK_SELECTION_NONE);
          gtk_widget_add_css_class (listbox, "boxed-list");

          for (guint i = 0; i < n_valid_entries; i++)
            {
              g_autoptr (BzEntry) entry = NULL;
              GtkWidget *row            = NULL;
              GtkWidget *radio          = NULL;

              entry = g_list_model_get_item (G_LIST_MODEL (store), i);
              row   = create_entry_radio_button (entry, &radio);
              g_ptr_array_add (radios, radio);

              if (first_valid_radio != NULL)
                gtk_check_button_set_group (GTK_CHECK_BUTTON (radio), first_valid_radio);
              else
                {
                  gtk_check_button_set_active (GTK_CHECK_BUTTON (radio), TRUE);
                  first_valid_radio = (GtkCheckButton *) radio;
                }

              gtk_list_box_append (GTK_LIST_BOX (listbox), row);
            }

          gtk_box_append (GTK_BOX (container), listbox);
        }
    }

  if (remove)
    {
      GtkWidget *listbox         = NULL;
      GtkWidget *keep_data_row   = NULL;
      GtkWidget *delete_data_row = NULL;
      GtkWidget *keep_radio      = NULL;
      GtkWidget *delete_radio    = NULL;

      listbox = gtk_list_box_new ();
      gtk_list_box_set_selection_mode (GTK_LIST_BOX (listbox), GTK_SELECTION_NONE);
      gtk_widget_add_css_class (listbox, "boxed-list");

      keep_data_row = adw_action_row_new ();
      adw_preferences_row_set_title (ADW_PREFERENCES_ROW (keep_data_row), _ ("Keep Data"));
      adw_action_row_set_subtitle (ADW_ACTION_ROW (keep_data_row), _ ("Allow restoring settings and content"));
      keep_radio = gtk_check_button_new ();
      gtk_widget_set_valign (keep_radio, GTK_ALIGN_CENTER);
      gtk_check_button_set_active (GTK_CHECK_BUTTON (keep_radio), TRUE);
      adw_action_row_add_prefix (ADW_ACTION_ROW (keep_data_row), keep_radio);
      adw_action_row_set_activatable_widget (ADW_ACTION_ROW (keep_data_row), keep_radio);
      gtk_list_box_append (GTK_LIST_BOX (listbox), keep_data_row);

      delete_data_row = adw_action_row_new ();
      adw_preferences_row_set_title (ADW_PREFERENCES_ROW (delete_data_row), _ ("Delete Data"));
      adw_action_row_set_subtitle (ADW_ACTION_ROW (delete_data_row), _ ("Permanently remove app data to save space"));
      delete_radio = gtk_check_button_new ();
      gtk_widget_set_valign (delete_radio, GTK_ALIGN_CENTER);
      gtk_check_button_set_group (GTK_CHECK_BUTTON (delete_radio), GTK_CHECK_BUTTON (keep_radio));
      adw_action_row_add_prefix (ADW_ACTION_ROW (delete_data_row), delete_radio);
      adw_action_row_set_activatable_widget (ADW_ACTION_ROW (delete_data_row), delete_radio);
      gtk_list_box_append (GTK_LIST_BOX (listbox), delete_data_row);

      g_ptr_array_add (radios, keep_radio);
      g_ptr_array_add (radios, delete_radio);
      gtk_box_append (GTK_BOX (container), listbox);
    }

  adw_alert_dialog_set_extra_child (alert, container);
  return g_steal_pointer (&radios);
}

static void
configure_install_dialog (AdwAlertDialog *alert,
                          const char     *title,
                          const char     *id)
{
  g_autofree char *heading = NULL;

  heading = g_strdup_printf (_ ("Install %s?"), title);

  adw_alert_dialog_set_heading (alert, heading);
  adw_alert_dialog_set_body (alert, _ ("May install additional shared components"));

  adw_alert_dialog_add_responses (alert,
                                  "cancel", _ ("Cancel"),
                                  "install", _ ("Install"),
                                  NULL);

  adw_alert_dialog_set_response_appearance (alert, "install", ADW_RESPONSE_SUGGESTED);
  adw_alert_dialog_set_default_response (alert, "install");
  adw_alert_dialog_set_close_response (alert, "cancel");
}

static void
configure_remove_dialog (AdwAlertDialog *alert,
                         const char     *title,
                         const char     *id)
{
  g_autofree char *heading = NULL;

  heading = g_strdup_printf (_ ("Remove %s?"), title);

  adw_alert_dialog_set_heading (alert, heading);
  adw_alert_dialog_set_body (
      alert, g_strdup_printf (_ ("It will not be possible to use %s after it is uninstalled."), title));

  adw_alert_dialog_add_responses (alert,
                                  "cancel", _ ("Cancel"),
                                  "remove", _ ("Remove"),
                                  NULL);

  adw_alert_dialog_set_response_appearance (alert, "remove", ADW_RESPONSE_DESTRUCTIVE);
  adw_alert_dialog_set_default_response (alert, "remove");
  adw_alert_dialog_set_close_response (alert, "cancel");
}

static void
update (BzWindow *self,
        BzEntry **updates,
        guint     n_updates)
{
  g_autoptr (BzTransaction) transaction = NULL;

  transaction = bz_transaction_new_full (
      NULL, 0,
      updates, n_updates,
      NULL, 0);
  dex_future_disown (bz_transaction_manager_add (
      bz_state_info_get_transaction_manager (self->state),
      transaction));
}

static void
search (BzWindow   *self,
        const char *initial)
{
  if (initial != NULL && *initial != '\0')
    {
      bz_search_widget_set_text (self->search_widget, initial);
    }

  adw_view_stack_set_visible_child_name (self->main_view_stack, "search");
  adw_navigation_view_pop_to_tag (self->navigation_view, "main");
  gtk_widget_grab_focus (GTK_WIDGET (self->search_widget));
}

static void
check_transactions (BzWindow *self)
{
  gboolean has_transactions = FALSE;
  gboolean paused           = FALSE;
  gboolean active           = FALSE;

  has_transactions = bz_transaction_manager_get_has_transactions (
      bz_state_info_get_transaction_manager (self->state));
  adw_view_stack_set_visible_child_name (
      self->transactions_stack,
      has_transactions
          ? "content"
          : "empty");

  paused = gtk_toggle_button_get_active (self->transactions_pause);
  active = bz_transaction_manager_get_active (
      bz_state_info_get_transaction_manager (self->state));
  if (paused)
    {
      gtk_button_set_icon_name (GTK_BUTTON (self->transactions_pause), "media-playback-start-symbolic");
      gtk_widget_set_tooltip_text (GTK_WIDGET (self->transactions_pause), _ ("Resume Current Tasks"));
      gtk_widget_add_css_class (GTK_WIDGET (self->transactions_pause), "suggested-action");
    }
  else
    {
      gtk_button_set_icon_name (GTK_BUTTON (self->transactions_pause), "media-playback-pause-symbolic");
      gtk_widget_set_tooltip_text (GTK_WIDGET (self->transactions_pause), _ ("Pause Current Tasks"));
      gtk_widget_remove_css_class (GTK_WIDGET (self->transactions_pause), "suggested-action");
    }

  if (active)
    gtk_widget_add_css_class (GTK_WIDGET (self->transactions_stop), "destructive-action");
  else
    gtk_widget_remove_css_class (GTK_WIDGET (self->transactions_stop), "destructive-action");
}

static void
set_page (BzWindow *self)
{
  const char *selected_navigation_page_name = NULL;

  if (self->state == NULL)
    return;

  if (bz_state_info_get_busy (self->state))
    {
      gtk_stack_set_visible_child_name (self->main_stack, "loading");
      adw_navigation_view_pop_to_tag (self->navigation_view, "main");
    }
  else
    {
      gtk_stack_set_visible_child_name (self->main_stack, "main");
    }

  selected_navigation_page_name = adw_navigation_view_get_visible_page_tag (self->navigation_view);

  if (g_strcmp0 (selected_navigation_page_name, "view") != 0)
    bz_full_view_set_entry_group (self->full_view, NULL);
}
