/*
 * Copyright (C) 2014-2026 CZ.NIC
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * In addition, as a special exception, the copyright holders give
 * permission to link the code of portions of this program with the
 * OpenSSL library under certain conditions as described in each
 * individual source file, and distribute linked combinations including
 * the two.
 */

#pragma once

#include <QAbstractTableModel>
#include <QList>

#include "src/datovka_shared/io/draft_db.h" /* DraftEntry */

namespace Json {
	class Int64StringList; /* Forward declaration. */
}

/*!
 * @brief Draft table model.
 */
class DraftTableModel : public QAbstractTableModel {
	Q_OBJECT

public:
	/*!
	 * @brief Additional roles.
	 */
	enum UserRoles {
		ROLE_PROXYSORT = (Qt::UserRole + 1) /*!< Allows fine-grained sorting of boolean values which may be displayed as icons or check boxes. */
	};
#if (QT_VERSION >= QT_VERSION_CHECK(5, 5, 0))
	Q_ENUM(UserRoles)
#else /* < Qt-5.5 */
	Q_ENUMS(UserRoles)
#endif /* >= Qt-5.5 */

	/*!
	 * @brief Column which this model holds.
	 */
	enum Columns {
		COL_DRAFT_ID = 0, /*!< Draft identifier. */
		COL_ACNT_ID, /*!< Account identifier. */
		COL_CREAT_TIME, /*!< Creation time. */
		COL_MOD_TIME, /*!< Modification time. */
		COL_ANNOTATION, /*!< Draft annotation. */
		COL_RECIPIENTS, /*!< Recipients. */

		MAX_COLNUM /* Maximal number of columns (convenience value). */
	};
#if (QT_VERSION >= QT_VERSION_CHECK(5, 5, 0))
	Q_ENUM(Columns)
#else /* < Qt-5.5 */
	Q_ENUMS(Columns)
#endif /* >= Qt-5.5 */

	/*!
	 * @brief Constructor.
	 *
	 * @param[in] parent Parent object.
	 */
	explicit DraftTableModel(QObject *parent = Q_NULLPTR);

	/*!
	 * @brief Returns number of rows under given parent.
	 *
	 * @param[in] parent Parent index.
	 * @return Number of rows.
	 */
	virtual
	int rowCount(const QModelIndex &parent = QModelIndex()) const
	    Q_DECL_OVERRIDE;

	/*!
	 * @brief Returns number of columns (for the children of given parent).
	 *
	 * @param[in] parent Parent index.
	 * @return Number of columns.
	 */
	virtual
	int columnCount(const QModelIndex &parent = QModelIndex()) const
	    Q_DECL_OVERRIDE;

	/*!
	 * @brief Returns the data stored under the given role.
	 *
	 * @param[in] index Position.
	 * @param[in] role  Role if the position.
	 * @return Data or invalid QVariant if no matching data found.
	 */
	virtual
	QVariant data(const QModelIndex &index,
	    int role = Qt::DisplayRole) const Q_DECL_OVERRIDE;

	/*!
	 * @brief Used for checkable elements.
	 *
	 * @param[in] index Index which to obtain flags for.
	 */
	virtual
	Qt::ItemFlags flags(const QModelIndex &index) const Q_DECL_OVERRIDE;

	/*!
	 * @brief Obtains header data.
	 *
	 * @param[in] section     Position.
	 * @param[in] orientation Orientation of the header.
	 * @param[in] role        Role of the data.
	 * @return Data or invalid QVariant in no matching data found.
	 */
	virtual
	QVariant headerData(int section, Qt::Orientation orientation,
	    int role = Qt::DisplayRole) const Q_DECL_OVERRIDE;

	/*!
	 * @brief Append data to the model.
	 *
	 * @param[in] entryList List of entries to append to the model.
	 */
	void appendData(const QList<DraftEntry> &entryList);
#ifdef Q_COMPILER_RVALUE_REFS
	void appendData(QList<DraftEntry> &&entryList);
#endif /* Q_COMPILER_RVALUE_REFS */

	/*!
	 * @brief Update existing or append data to the model.
	 *
	 * @param[in] entryList List of entries to update or append.
	 */
	void updateOrAppendData(const QList<DraftEntry> &entryList);

	/*!
	 * @brief Model content at row.
	 *
	 * @param[in] row Selected row.
	 * @return Held data at given \a row.
	 */
	DraftEntry entryAt(int row) const;

	/*!
	 * @brief Model content for identifier.
	 *
	 * @param[in] draftId Draft ID.
	 * @param[out] entry Held data for given \a draftId.
	 * @return True if data found, false else or on any error.
	 */
	bool entryForId(qint64 draftId, DraftEntry &entry) const;

	/*!
	 * @brief Clears the model content.
	 */
	void removeAllRows(void);

	/*!
	 * @brief Remove first entries matching the supplied identifiers.
	 *
	 * @param[in] ids Draft identifiers.
	 */
	void remove(const Json::Int64StringList &ids);

private:
	/*!
	 * @brief Find first row according to supplied entries.
	 *
	 * @param[in] draftId Draft ID.
	 * @return Non-negative row number if found, -1 else.
	 */
	int findRow(qint64 draftId) const;

	QList<DraftEntry> m_entries; /*!< List of model entries. */
};
