{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Implements a fragment shader using GLSL.
-- 
-- A fragment shader gets the coordinates being rendered as input and
-- produces the pixel values for that particular pixel. Additionally,
-- the shader can declare a set of other input arguments, called
-- uniforms (as they are uniform over all the calls to your shader in
-- each instance of use). A shader can also receive up to 4
-- textures that it can use as input when producing the pixel data.
-- 
-- @GskGLShader@ is usually used with @/gtk_snapshot_push_gl_shader()/@
-- to produce a t'GI.Gsk.Objects.GLShaderNode.GLShaderNode' in the rendering hierarchy,
-- and then its input textures are constructed by rendering the child
-- nodes to textures before rendering the shader node itself. (You can
-- pass texture nodes as children if you want to directly use a texture
-- as input).
-- 
-- The actual shader code is GLSL code that gets combined with
-- some other code into the fragment shader. Since the exact
-- capabilities of the GPU driver differs between different OpenGL
-- drivers and hardware, GTK adds some defines that you can use
-- to ensure your GLSL code runs on as many drivers as it can.
-- 
-- If the OpenGL driver is GLES, then the shader language version
-- is set to 100, and GSK_GLES will be defined in the shader.
-- 
-- Otherwise, if the OpenGL driver does not support the 3.2 core profile,
-- then the shader will run with language version 110 for GL2 and 130 for GL3,
-- and GSK_LEGACY will be defined in the shader.
-- 
-- If the OpenGL driver supports the 3.2 code profile, it will be used,
-- the shader language version is set to 150, and GSK_GL3 will be defined
-- in the shader.
-- 
-- The main function the shader must implement is:
-- 
-- 
-- === /glsl code/
-- > void mainImage(out vec4 fragColor,
-- >                in vec2 fragCoord,
-- >                in vec2 resolution,
-- >                in vec2 uv)
-- 
-- 
-- Where the input /@fragCoord@/ is the coordinate of the pixel we\'re
-- currently rendering, relative to the boundary rectangle that was
-- specified in the @GskGLShaderNode@, and /@resolution@/ is the width and
-- height of that rectangle. This is in the typical GTK coordinate
-- system with the origin in the top left. /@uv@/ contains the u and v
-- coordinates that can be used to index a texture at the
-- corresponding point. These coordinates are in the [0..1]x[0..1]
-- region, with 0, 0 being in the lower left corder (which is typical
-- for OpenGL).
-- 
-- The output /@fragColor@/ should be a RGBA color (with
-- premultiplied alpha) that will be used as the output for the
-- specified pixel location. Note that this output will be
-- automatically clipped to the clip region of the glshader node.
-- 
-- In addition to the function arguments the shader can define
-- up to 4 uniforms for textures which must be called u_textureN
-- (i.e. u_texture1 to u_texture4) as well as any custom uniforms
-- you want of types int, uint, bool, float, vec2, vec3 or vec4.
-- 
-- All textures sources contain premultiplied alpha colors, but if some
-- there are outer sources of colors there is a @/gsk_premultiply()/@ helper
-- to compute premultiplication when needed.
-- 
-- Note that GTK parses the uniform declarations, so each uniform has to
-- be on a line by itself with no other code, like so:
-- 
-- 
-- === /glsl code/
-- >uniform float u_time;
-- >uniform vec3 u_color;
-- >uniform sampler2D u_texture1;
-- >uniform sampler2D u_texture2;
-- 
-- 
-- GTK uses the \"gsk\" namespace in the symbols it uses in the
-- shader, so your code should not use any symbols with the prefix gsk
-- or GSK. There are some helper functions declared that you can use:
-- 
-- 
-- === /glsl code/
-- >vec4 GskTexture(sampler2D sampler, vec2 texCoords);
-- 
-- 
-- This samples a texture (e.g. u_texture1) at the specified
-- coordinates, and contains some helper ifdefs to ensure that
-- it works on all OpenGL versions.
-- 
-- You can compile the shader yourself using 'GI.Gsk.Objects.GLShader.gLShaderCompile',
-- otherwise the GSK renderer will do it when it handling the glshader
-- node. If errors occurs, the returned /@error@/ will include the glsl
-- sources, so you can see what GSK was passing to the compiler. You
-- can also set GSK_DEBUG=shaders in the environment to see the sources
-- and other relevant information about all shaders that GSK is handling.
-- 
-- = An example shader
-- 
-- 
-- === /glsl code/
-- >uniform float position;
-- >uniform sampler2D u_texture1;
-- >uniform sampler2D u_texture2;
-- >
-- >void mainImage(out vec4 fragColor,
-- >               in vec2 fragCoord,
-- >               in vec2 resolution,
-- >               in vec2 uv) {
-- >  vec4 source1 = GskTexture(u_texture1, uv);
-- >  vec4 source2 = GskTexture(u_texture2, uv);
-- >
-- >  fragColor = position * source1 + (1.0 - position) * source2;
-- >}
-- 

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gsk.Objects.GLShader
    ( 

-- * Exported types
    GLShader(..)                            ,
    IsGLShader                              ,
    toGLShader                              ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [compile]("GI.Gsk.Objects.GLShader#g:method:compile"), [findUniformByName]("GI.Gsk.Objects.GLShader#g:method:findUniformByName"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getArgBool]("GI.Gsk.Objects.GLShader#g:method:getArgBool"), [getArgFloat]("GI.Gsk.Objects.GLShader#g:method:getArgFloat"), [getArgInt]("GI.Gsk.Objects.GLShader#g:method:getArgInt"), [getArgUint]("GI.Gsk.Objects.GLShader#g:method:getArgUint"), [getArgVec2]("GI.Gsk.Objects.GLShader#g:method:getArgVec2"), [getArgVec3]("GI.Gsk.Objects.GLShader#g:method:getArgVec3"), [getArgVec4]("GI.Gsk.Objects.GLShader#g:method:getArgVec4"), [getArgsSize]("GI.Gsk.Objects.GLShader#g:method:getArgsSize"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getNTextures]("GI.Gsk.Objects.GLShader#g:method:getNTextures"), [getNUniforms]("GI.Gsk.Objects.GLShader#g:method:getNUniforms"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getResource]("GI.Gsk.Objects.GLShader#g:method:getResource"), [getSource]("GI.Gsk.Objects.GLShader#g:method:getSource"), [getUniformName]("GI.Gsk.Objects.GLShader#g:method:getUniformName"), [getUniformOffset]("GI.Gsk.Objects.GLShader#g:method:getUniformOffset"), [getUniformType]("GI.Gsk.Objects.GLShader#g:method:getUniformType").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveGLShaderMethod                   ,
#endif

-- ** compile #method:compile#

#if defined(ENABLE_OVERLOADING)
    GLShaderCompileMethodInfo               ,
#endif
    gLShaderCompile                         ,


-- ** findUniformByName #method:findUniformByName#

#if defined(ENABLE_OVERLOADING)
    GLShaderFindUniformByNameMethodInfo     ,
#endif
    gLShaderFindUniformByName               ,


-- ** getArgBool #method:getArgBool#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetArgBoolMethodInfo            ,
#endif
    gLShaderGetArgBool                      ,


-- ** getArgFloat #method:getArgFloat#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetArgFloatMethodInfo           ,
#endif
    gLShaderGetArgFloat                     ,


-- ** getArgInt #method:getArgInt#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetArgIntMethodInfo             ,
#endif
    gLShaderGetArgInt                       ,


-- ** getArgUint #method:getArgUint#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetArgUintMethodInfo            ,
#endif
    gLShaderGetArgUint                      ,


-- ** getArgVec2 #method:getArgVec2#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetArgVec2MethodInfo            ,
#endif
    gLShaderGetArgVec2                      ,


-- ** getArgVec3 #method:getArgVec3#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetArgVec3MethodInfo            ,
#endif
    gLShaderGetArgVec3                      ,


-- ** getArgVec4 #method:getArgVec4#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetArgVec4MethodInfo            ,
#endif
    gLShaderGetArgVec4                      ,


-- ** getArgsSize #method:getArgsSize#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetArgsSizeMethodInfo           ,
#endif
    gLShaderGetArgsSize                     ,


-- ** getNTextures #method:getNTextures#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetNTexturesMethodInfo          ,
#endif
    gLShaderGetNTextures                    ,


-- ** getNUniforms #method:getNUniforms#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetNUniformsMethodInfo          ,
#endif
    gLShaderGetNUniforms                    ,


-- ** getResource #method:getResource#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetResourceMethodInfo           ,
#endif
    gLShaderGetResource                     ,


-- ** getSource #method:getSource#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetSourceMethodInfo             ,
#endif
    gLShaderGetSource                       ,


-- ** getUniformName #method:getUniformName#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetUniformNameMethodInfo        ,
#endif
    gLShaderGetUniformName                  ,


-- ** getUniformOffset #method:getUniformOffset#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetUniformOffsetMethodInfo      ,
#endif
    gLShaderGetUniformOffset                ,


-- ** getUniformType #method:getUniformType#

#if defined(ENABLE_OVERLOADING)
    GLShaderGetUniformTypeMethodInfo        ,
#endif
    gLShaderGetUniformType                  ,


-- ** newFromBytes #method:newFromBytes#

    gLShaderNewFromBytes                    ,


-- ** newFromResource #method:newFromResource#

    gLShaderNewFromResource                 ,




 -- * Properties


-- ** resource #attr:resource#
-- | Resource containing the source code for the shader.
-- 
-- If the shader source is not coming from a resource, this
-- will be 'P.Nothing'.

#if defined(ENABLE_OVERLOADING)
    GLShaderResourcePropertyInfo            ,
#endif
    constructGLShaderResource               ,
#if defined(ENABLE_OVERLOADING)
    gLShaderResource                        ,
#endif
    getGLShaderResource                     ,


-- ** source #attr:source#
-- | The source code for the shader, as a @GBytes@.

#if defined(ENABLE_OVERLOADING)
    GLShaderSourcePropertyInfo              ,
#endif
    constructGLShaderSource                 ,
#if defined(ENABLE_OVERLOADING)
    gLShaderSource                          ,
#endif
    getGLShaderSource                       ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.Region as Cairo.Region
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Vec2 as Graphene.Vec2
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Callbacks as Gsk.Callbacks
import {-# SOURCE #-} qualified GI.Gsk.Enums as Gsk.Enums
import {-# SOURCE #-} qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import {-# SOURCE #-} qualified GI.Gsk.Objects.Renderer as Gsk.Renderer

#else
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Graphene.Structs.Vec2 as Graphene.Vec2
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import {-# SOURCE #-} qualified GI.Gsk.Enums as Gsk.Enums
import {-# SOURCE #-} qualified GI.Gsk.Objects.Renderer as Gsk.Renderer

#endif

-- | Memory-managed wrapper type.
newtype GLShader = GLShader (SP.ManagedPtr GLShader)
    deriving (Eq)

instance SP.ManagedPtrNewtype GLShader where
    toManagedPtr (GLShader p) = p

foreign import ccall "gsk_gl_shader_get_type"
    c_gsk_gl_shader_get_type :: IO B.Types.GType

instance B.Types.TypedObject GLShader where
    glibType = c_gsk_gl_shader_get_type

instance B.Types.GObject GLShader

-- | Type class for types which can be safely cast to t'GLShader', for instance with `toGLShader`.
class (SP.GObject o, O.IsDescendantOf GLShader o) => IsGLShader o
instance (SP.GObject o, O.IsDescendantOf GLShader o) => IsGLShader o

instance O.HasParentTypes GLShader
type instance O.ParentTypes GLShader = '[GObject.Object.Object]

-- | Cast to t'GLShader', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toGLShader :: (MIO.MonadIO m, IsGLShader o) => o -> m GLShader
toGLShader = MIO.liftIO . B.ManagedPtr.unsafeCastTo GLShader

-- | Convert t'GLShader' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe GLShader) where
    gvalueGType_ = c_gsk_gl_shader_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr GLShader)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr GLShader)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject GLShader ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveGLShaderMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveGLShaderMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveGLShaderMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveGLShaderMethod "compile" o = GLShaderCompileMethodInfo
    ResolveGLShaderMethod "findUniformByName" o = GLShaderFindUniformByNameMethodInfo
    ResolveGLShaderMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveGLShaderMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveGLShaderMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveGLShaderMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveGLShaderMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveGLShaderMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveGLShaderMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveGLShaderMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveGLShaderMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveGLShaderMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveGLShaderMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveGLShaderMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveGLShaderMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveGLShaderMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveGLShaderMethod "getArgBool" o = GLShaderGetArgBoolMethodInfo
    ResolveGLShaderMethod "getArgFloat" o = GLShaderGetArgFloatMethodInfo
    ResolveGLShaderMethod "getArgInt" o = GLShaderGetArgIntMethodInfo
    ResolveGLShaderMethod "getArgUint" o = GLShaderGetArgUintMethodInfo
    ResolveGLShaderMethod "getArgVec2" o = GLShaderGetArgVec2MethodInfo
    ResolveGLShaderMethod "getArgVec3" o = GLShaderGetArgVec3MethodInfo
    ResolveGLShaderMethod "getArgVec4" o = GLShaderGetArgVec4MethodInfo
    ResolveGLShaderMethod "getArgsSize" o = GLShaderGetArgsSizeMethodInfo
    ResolveGLShaderMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveGLShaderMethod "getNTextures" o = GLShaderGetNTexturesMethodInfo
    ResolveGLShaderMethod "getNUniforms" o = GLShaderGetNUniformsMethodInfo
    ResolveGLShaderMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveGLShaderMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveGLShaderMethod "getResource" o = GLShaderGetResourceMethodInfo
    ResolveGLShaderMethod "getSource" o = GLShaderGetSourceMethodInfo
    ResolveGLShaderMethod "getUniformName" o = GLShaderGetUniformNameMethodInfo
    ResolveGLShaderMethod "getUniformOffset" o = GLShaderGetUniformOffsetMethodInfo
    ResolveGLShaderMethod "getUniformType" o = GLShaderGetUniformTypeMethodInfo
    ResolveGLShaderMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveGLShaderMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveGLShaderMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveGLShaderMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveGLShaderMethod t GLShader, O.OverloadedMethod info GLShader p) => OL.IsLabel t (GLShader -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveGLShaderMethod t GLShader, O.OverloadedMethod info GLShader p, R.HasField t GLShader p) => R.HasField t GLShader p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveGLShaderMethod t GLShader, O.OverloadedMethodInfo info GLShader) => OL.IsLabel t (O.MethodProxy info GLShader) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- VVV Prop "resource"
   -- Type: TBasicType TUTF8
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Just True,Nothing)

-- | Get the value of the “@resource@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gLShader #resource
-- @
getGLShaderResource :: (MonadIO m, IsGLShader o) => o -> m (Maybe T.Text)
getGLShaderResource obj = MIO.liftIO $ B.Properties.getObjectPropertyString obj "resource"

-- | Construct a t'GValueConstruct' with valid value for the “@resource@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructGLShaderResource :: (IsGLShader o, MIO.MonadIO m) => T.Text -> m (GValueConstruct o)
constructGLShaderResource val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyString "resource" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data GLShaderResourcePropertyInfo
instance AttrInfo GLShaderResourcePropertyInfo where
    type AttrAllowedOps GLShaderResourcePropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint GLShaderResourcePropertyInfo = IsGLShader
    type AttrSetTypeConstraint GLShaderResourcePropertyInfo = (~) T.Text
    type AttrTransferTypeConstraint GLShaderResourcePropertyInfo = (~) T.Text
    type AttrTransferType GLShaderResourcePropertyInfo = T.Text
    type AttrGetType GLShaderResourcePropertyInfo = (Maybe T.Text)
    type AttrLabel GLShaderResourcePropertyInfo = "resource"
    type AttrOrigin GLShaderResourcePropertyInfo = GLShader
    attrGet = getGLShaderResource
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructGLShaderResource
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.resource"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#g:attr:resource"
        })
#endif

-- VVV Prop "source"
   -- Type: TInterface (Name {namespace = "GLib", name = "Bytes"})
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@source@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' gLShader #source
-- @
getGLShaderSource :: (MonadIO m, IsGLShader o) => o -> m GLib.Bytes.Bytes
getGLShaderSource obj = MIO.liftIO $ checkUnexpectedNothing "getGLShaderSource" $ B.Properties.getObjectPropertyBoxed obj "source" GLib.Bytes.Bytes

-- | Construct a t'GValueConstruct' with valid value for the “@source@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructGLShaderSource :: (IsGLShader o, MIO.MonadIO m) => GLib.Bytes.Bytes -> m (GValueConstruct o)
constructGLShaderSource val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyBoxed "source" (P.Just val)

#if defined(ENABLE_OVERLOADING)
data GLShaderSourcePropertyInfo
instance AttrInfo GLShaderSourcePropertyInfo where
    type AttrAllowedOps GLShaderSourcePropertyInfo = '[ 'AttrConstruct, 'AttrGet, 'AttrClear]
    type AttrBaseTypeConstraint GLShaderSourcePropertyInfo = IsGLShader
    type AttrSetTypeConstraint GLShaderSourcePropertyInfo = (~) GLib.Bytes.Bytes
    type AttrTransferTypeConstraint GLShaderSourcePropertyInfo = (~) GLib.Bytes.Bytes
    type AttrTransferType GLShaderSourcePropertyInfo = GLib.Bytes.Bytes
    type AttrGetType GLShaderSourcePropertyInfo = GLib.Bytes.Bytes
    type AttrLabel GLShaderSourcePropertyInfo = "source"
    type AttrOrigin GLShaderSourcePropertyInfo = GLShader
    attrGet = getGLShaderSource
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructGLShaderSource
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.source"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#g:attr:source"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList GLShader
type instance O.AttributeList GLShader = GLShaderAttributeList
type GLShaderAttributeList = ('[ '("resource", GLShaderResourcePropertyInfo), '("source", GLShaderSourcePropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
gLShaderResource :: AttrLabelProxy "resource"
gLShaderResource = AttrLabelProxy

gLShaderSource :: AttrLabelProxy "source"
gLShaderSource = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList GLShader = GLShaderSignalList
type GLShaderSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method GLShader::new_from_bytes
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "sourcecode"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "GLSL sourcecode for the shader, as a `GBytes`"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "GLShader" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_new_from_bytes" gsk_gl_shader_new_from_bytes :: 
    Ptr GLib.Bytes.Bytes ->                 -- sourcecode : TInterface (Name {namespace = "GLib", name = "Bytes"})
    IO (Ptr GLShader)

{-# DEPRECATED gLShaderNewFromBytes ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Creates a @GskGLShader@ that will render pixels using the specified code.
gLShaderNewFromBytes ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GLib.Bytes.Bytes
    -- ^ /@sourcecode@/: GLSL sourcecode for the shader, as a @GBytes@
    -> m GLShader
    -- ^ __Returns:__ A new @GskGLShader@
gLShaderNewFromBytes sourcecode = liftIO $ do
    sourcecode' <- unsafeManagedPtrGetPtr sourcecode
    result <- gsk_gl_shader_new_from_bytes sourcecode'
    checkUnexpectedReturnNULL "gLShaderNewFromBytes" result
    result' <- (wrapObject GLShader) result
    touchManagedPtr sourcecode
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method GLShader::new_from_resource
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "resource_path"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "path to a resource that contains the GLSL sourcecode for\n    the shader"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gsk" , name = "GLShader" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_new_from_resource" gsk_gl_shader_new_from_resource :: 
    CString ->                              -- resource_path : TBasicType TUTF8
    IO (Ptr GLShader)

{-# DEPRECATED gLShaderNewFromResource ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Creates a @GskGLShader@ that will render pixels using the specified code.
gLShaderNewFromResource ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    T.Text
    -- ^ /@resourcePath@/: path to a resource that contains the GLSL sourcecode for
    --     the shader
    -> m GLShader
    -- ^ __Returns:__ A new @GskGLShader@
gLShaderNewFromResource resourcePath = liftIO $ do
    resourcePath' <- textToCString resourcePath
    result <- gsk_gl_shader_new_from_resource resourcePath'
    checkUnexpectedReturnNULL "gLShaderNewFromResource" result
    result' <- (wrapObject GLShader) result
    freeMem resourcePath'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method GLShader::compile
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "renderer"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "Renderer" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskRenderer`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : True
-- Skip return : False

foreign import ccall "gsk_gl_shader_compile" gsk_gl_shader_compile :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Ptr Gsk.Renderer.Renderer ->            -- renderer : TInterface (Name {namespace = "Gsk", name = "Renderer"})
    Ptr (Ptr GError) ->                     -- error
    IO CInt

{-# DEPRECATED gLShaderCompile ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Tries to compile the /@shader@/ for the given /@renderer@/.
-- 
-- If there is a problem, this function returns 'P.False' and reports
-- an error. You should use this function before relying on the shader
-- for rendering and use a fallback with a simpler shader or without
-- shaders if it fails.
-- 
-- Note that this will modify the rendering state (for example
-- change the current GL context) and requires the renderer to be
-- set up. This means that the widget has to be realized. Commonly you
-- want to call this from the realize signal of a widget, or during
-- widget snapshot.
gLShaderCompile ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a, Gsk.Renderer.IsRenderer b) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> b
    -- ^ /@renderer@/: a @GskRenderer@
    -> m ()
    -- ^ /(Can throw 'Data.GI.Base.GError.GError')/
gLShaderCompile shader renderer = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    renderer' <- unsafeManagedPtrCastPtr renderer
    onException (do
        _ <- propagateGError $ gsk_gl_shader_compile shader' renderer'
        touchManagedPtr shader
        touchManagedPtr renderer
        return ()
     ) (do
        return ()
     )

#if defined(ENABLE_OVERLOADING)
data GLShaderCompileMethodInfo
instance (signature ~ (b -> m ()), MonadIO m, IsGLShader a, Gsk.Renderer.IsRenderer b) => O.OverloadedMethod GLShaderCompileMethodInfo a signature where
    overloadedMethod = gLShaderCompile

instance O.OverloadedMethodInfo GLShaderCompileMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderCompile",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderCompile"
        })


#endif

-- method GLShader::find_uniform_by_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "uniform name" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_find_uniform_by_name" gsk_gl_shader_find_uniform_by_name :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    CString ->                              -- name : TBasicType TUTF8
    IO Int32

{-# DEPRECATED gLShaderFindUniformByName ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Looks for a uniform by the name /@name@/, and returns the index
-- of the uniform, or -1 if it was not found.
gLShaderFindUniformByName ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> T.Text
    -- ^ /@name@/: uniform name
    -> m Int32
    -- ^ __Returns:__ The index of the uniform, or -1
gLShaderFindUniformByName shader name = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    name' <- textToCString name
    result <- gsk_gl_shader_find_uniform_by_name shader' name'
    touchManagedPtr shader
    freeMem name'
    return result

#if defined(ENABLE_OVERLOADING)
data GLShaderFindUniformByNameMethodInfo
instance (signature ~ (T.Text -> m Int32), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderFindUniformByNameMethodInfo a signature where
    overloadedMethod = gLShaderFindUniformByName

instance O.OverloadedMethodInfo GLShaderFindUniformByNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderFindUniformByName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderFindUniformByName"
        })


#endif

-- method GLShader::get_arg_bool
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "args"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "uniform arguments" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_arg_bool" gsk_gl_shader_get_arg_bool :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Ptr GLib.Bytes.Bytes ->                 -- args : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Int32 ->                                -- idx : TBasicType TInt
    IO CInt

{-# DEPRECATED gLShaderGetArgBool ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Gets the value of the uniform /@idx@/ in the /@args@/ block.
-- 
-- The uniform must be of bool type.
gLShaderGetArgBool ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> GLib.Bytes.Bytes
    -- ^ /@args@/: uniform arguments
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> m Bool
    -- ^ __Returns:__ The value
gLShaderGetArgBool shader args idx = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    args' <- unsafeManagedPtrGetPtr args
    result <- gsk_gl_shader_get_arg_bool shader' args' idx
    let result' = (/= 0) result
    touchManagedPtr shader
    touchManagedPtr args
    return result'

#if defined(ENABLE_OVERLOADING)
data GLShaderGetArgBoolMethodInfo
instance (signature ~ (GLib.Bytes.Bytes -> Int32 -> m Bool), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetArgBoolMethodInfo a signature where
    overloadedMethod = gLShaderGetArgBool

instance O.OverloadedMethodInfo GLShaderGetArgBoolMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetArgBool",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetArgBool"
        })


#endif

-- method GLShader::get_arg_float
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "args"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "uniform arguments" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TFloat)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_arg_float" gsk_gl_shader_get_arg_float :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Ptr GLib.Bytes.Bytes ->                 -- args : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Int32 ->                                -- idx : TBasicType TInt
    IO CFloat

{-# DEPRECATED gLShaderGetArgFloat ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Gets the value of the uniform /@idx@/ in the /@args@/ block.
-- 
-- The uniform must be of float type.
gLShaderGetArgFloat ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> GLib.Bytes.Bytes
    -- ^ /@args@/: uniform arguments
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> m Float
    -- ^ __Returns:__ The value
gLShaderGetArgFloat shader args idx = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    args' <- unsafeManagedPtrGetPtr args
    result <- gsk_gl_shader_get_arg_float shader' args' idx
    let result' = realToFrac result
    touchManagedPtr shader
    touchManagedPtr args
    return result'

#if defined(ENABLE_OVERLOADING)
data GLShaderGetArgFloatMethodInfo
instance (signature ~ (GLib.Bytes.Bytes -> Int32 -> m Float), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetArgFloatMethodInfo a signature where
    overloadedMethod = gLShaderGetArgFloat

instance O.OverloadedMethodInfo GLShaderGetArgFloatMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetArgFloat",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetArgFloat"
        })


#endif

-- method GLShader::get_arg_int
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "args"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "uniform arguments" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt32)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_arg_int" gsk_gl_shader_get_arg_int :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Ptr GLib.Bytes.Bytes ->                 -- args : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Int32 ->                                -- idx : TBasicType TInt
    IO Int32

{-# DEPRECATED gLShaderGetArgInt ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Gets the value of the uniform /@idx@/ in the /@args@/ block.
-- 
-- The uniform must be of int type.
gLShaderGetArgInt ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> GLib.Bytes.Bytes
    -- ^ /@args@/: uniform arguments
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> m Int32
    -- ^ __Returns:__ The value
gLShaderGetArgInt shader args idx = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    args' <- unsafeManagedPtrGetPtr args
    result <- gsk_gl_shader_get_arg_int shader' args' idx
    touchManagedPtr shader
    touchManagedPtr args
    return result

#if defined(ENABLE_OVERLOADING)
data GLShaderGetArgIntMethodInfo
instance (signature ~ (GLib.Bytes.Bytes -> Int32 -> m Int32), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetArgIntMethodInfo a signature where
    overloadedMethod = gLShaderGetArgInt

instance O.OverloadedMethodInfo GLShaderGetArgIntMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetArgInt",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetArgInt"
        })


#endif

-- method GLShader::get_arg_uint
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "args"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "uniform arguments" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUInt32)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_arg_uint" gsk_gl_shader_get_arg_uint :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Ptr GLib.Bytes.Bytes ->                 -- args : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Int32 ->                                -- idx : TBasicType TInt
    IO Word32

{-# DEPRECATED gLShaderGetArgUint ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Gets the value of the uniform /@idx@/ in the /@args@/ block.
-- 
-- The uniform must be of uint type.
gLShaderGetArgUint ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> GLib.Bytes.Bytes
    -- ^ /@args@/: uniform arguments
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> m Word32
    -- ^ __Returns:__ The value
gLShaderGetArgUint shader args idx = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    args' <- unsafeManagedPtrGetPtr args
    result <- gsk_gl_shader_get_arg_uint shader' args' idx
    touchManagedPtr shader
    touchManagedPtr args
    return result

#if defined(ENABLE_OVERLOADING)
data GLShaderGetArgUintMethodInfo
instance (signature ~ (GLib.Bytes.Bytes -> Int32 -> m Word32), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetArgUintMethodInfo a signature where
    overloadedMethod = gLShaderGetArgUint

instance O.OverloadedMethodInfo GLShaderGetArgUintMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetArgUint",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetArgUint"
        })


#endif

-- method GLShader::get_arg_vec2
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "args"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "uniform arguments" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_value"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Vec2" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the uniform value in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_arg_vec2" gsk_gl_shader_get_arg_vec2 :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Ptr GLib.Bytes.Bytes ->                 -- args : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Int32 ->                                -- idx : TBasicType TInt
    Ptr Graphene.Vec2.Vec2 ->               -- out_value : TInterface (Name {namespace = "Graphene", name = "Vec2"})
    IO ()

{-# DEPRECATED gLShaderGetArgVec2 ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Gets the value of the uniform /@idx@/ in the /@args@/ block.
-- 
-- The uniform must be of vec2 type.
gLShaderGetArgVec2 ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> GLib.Bytes.Bytes
    -- ^ /@args@/: uniform arguments
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> Graphene.Vec2.Vec2
    -- ^ /@outValue@/: location to store the uniform value in
    -> m ()
gLShaderGetArgVec2 shader args idx outValue = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    args' <- unsafeManagedPtrGetPtr args
    outValue' <- unsafeManagedPtrGetPtr outValue
    gsk_gl_shader_get_arg_vec2 shader' args' idx outValue'
    touchManagedPtr shader
    touchManagedPtr args
    touchManagedPtr outValue
    return ()

#if defined(ENABLE_OVERLOADING)
data GLShaderGetArgVec2MethodInfo
instance (signature ~ (GLib.Bytes.Bytes -> Int32 -> Graphene.Vec2.Vec2 -> m ()), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetArgVec2MethodInfo a signature where
    overloadedMethod = gLShaderGetArgVec2

instance O.OverloadedMethodInfo GLShaderGetArgVec2MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetArgVec2",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetArgVec2"
        })


#endif

-- method GLShader::get_arg_vec3
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "args"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "uniform arguments" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_value"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Vec3" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store the uniform value in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_arg_vec3" gsk_gl_shader_get_arg_vec3 :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Ptr GLib.Bytes.Bytes ->                 -- args : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Int32 ->                                -- idx : TBasicType TInt
    Ptr Graphene.Vec3.Vec3 ->               -- out_value : TInterface (Name {namespace = "Graphene", name = "Vec3"})
    IO ()

{-# DEPRECATED gLShaderGetArgVec3 ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Gets the value of the uniform /@idx@/ in the /@args@/ block.
-- 
-- The uniform must be of vec3 type.
gLShaderGetArgVec3 ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> GLib.Bytes.Bytes
    -- ^ /@args@/: uniform arguments
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> Graphene.Vec3.Vec3
    -- ^ /@outValue@/: location to store the uniform value in
    -> m ()
gLShaderGetArgVec3 shader args idx outValue = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    args' <- unsafeManagedPtrGetPtr args
    outValue' <- unsafeManagedPtrGetPtr outValue
    gsk_gl_shader_get_arg_vec3 shader' args' idx outValue'
    touchManagedPtr shader
    touchManagedPtr args
    touchManagedPtr outValue
    return ()

#if defined(ENABLE_OVERLOADING)
data GLShaderGetArgVec3MethodInfo
instance (signature ~ (GLib.Bytes.Bytes -> Int32 -> Graphene.Vec3.Vec3 -> m ()), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetArgVec3MethodInfo a signature where
    overloadedMethod = gLShaderGetArgVec3

instance O.OverloadedMethodInfo GLShaderGetArgVec3MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetArgVec3",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetArgVec3"
        })


#endif

-- method GLShader::get_arg_vec4
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "args"
--           , argType = TInterface Name { namespace = "GLib" , name = "Bytes" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "uniform arguments" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "out_value"
--           , argType =
--               TInterface Name { namespace = "Graphene" , name = "Vec4" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "location to store set the uniform value in"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_arg_vec4" gsk_gl_shader_get_arg_vec4 :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Ptr GLib.Bytes.Bytes ->                 -- args : TInterface (Name {namespace = "GLib", name = "Bytes"})
    Int32 ->                                -- idx : TBasicType TInt
    Ptr Graphene.Vec4.Vec4 ->               -- out_value : TInterface (Name {namespace = "Graphene", name = "Vec4"})
    IO ()

{-# DEPRECATED gLShaderGetArgVec4 ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Gets the value of the uniform /@idx@/ in the /@args@/ block.
-- 
-- The uniform must be of vec4 type.
gLShaderGetArgVec4 ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> GLib.Bytes.Bytes
    -- ^ /@args@/: uniform arguments
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> Graphene.Vec4.Vec4
    -- ^ /@outValue@/: location to store set the uniform value in
    -> m ()
gLShaderGetArgVec4 shader args idx outValue = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    args' <- unsafeManagedPtrGetPtr args
    outValue' <- unsafeManagedPtrGetPtr outValue
    gsk_gl_shader_get_arg_vec4 shader' args' idx outValue'
    touchManagedPtr shader
    touchManagedPtr args
    touchManagedPtr outValue
    return ()

#if defined(ENABLE_OVERLOADING)
data GLShaderGetArgVec4MethodInfo
instance (signature ~ (GLib.Bytes.Bytes -> Int32 -> Graphene.Vec4.Vec4 -> m ()), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetArgVec4MethodInfo a signature where
    overloadedMethod = gLShaderGetArgVec4

instance O.OverloadedMethodInfo GLShaderGetArgVec4MethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetArgVec4",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetArgVec4"
        })


#endif

-- method GLShader::get_args_size
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TSize)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_args_size" gsk_gl_shader_get_args_size :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    IO FCT.CSize

{-# DEPRECATED gLShaderGetArgsSize ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Get the size of the data block used to specify arguments for this shader.
gLShaderGetArgsSize ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> m FCT.CSize
    -- ^ __Returns:__ The size of the data block
gLShaderGetArgsSize shader = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    result <- gsk_gl_shader_get_args_size shader'
    touchManagedPtr shader
    return result

#if defined(ENABLE_OVERLOADING)
data GLShaderGetArgsSizeMethodInfo
instance (signature ~ (m FCT.CSize), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetArgsSizeMethodInfo a signature where
    overloadedMethod = gLShaderGetArgsSize

instance O.OverloadedMethodInfo GLShaderGetArgsSizeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetArgsSize",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetArgsSize"
        })


#endif

-- method GLShader::get_n_textures
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_n_textures" gsk_gl_shader_get_n_textures :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    IO Int32

{-# DEPRECATED gLShaderGetNTextures ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Returns the number of textures that the shader requires.
-- 
-- This can be used to check that the a passed shader works
-- in your usecase. It is determined by looking at the highest
-- u_textureN value that the shader defines.
gLShaderGetNTextures ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> m Int32
    -- ^ __Returns:__ The number of texture inputs required by /@shader@/
gLShaderGetNTextures shader = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    result <- gsk_gl_shader_get_n_textures shader'
    touchManagedPtr shader
    return result

#if defined(ENABLE_OVERLOADING)
data GLShaderGetNTexturesMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetNTexturesMethodInfo a signature where
    overloadedMethod = gLShaderGetNTextures

instance O.OverloadedMethodInfo GLShaderGetNTexturesMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetNTextures",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetNTextures"
        })


#endif

-- method GLShader::get_n_uniforms
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_n_uniforms" gsk_gl_shader_get_n_uniforms :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    IO Int32

{-# DEPRECATED gLShaderGetNUniforms ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Get the number of declared uniforms for this shader.
gLShaderGetNUniforms ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> m Int32
    -- ^ __Returns:__ The number of declared uniforms
gLShaderGetNUniforms shader = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    result <- gsk_gl_shader_get_n_uniforms shader'
    touchManagedPtr shader
    return result

#if defined(ENABLE_OVERLOADING)
data GLShaderGetNUniformsMethodInfo
instance (signature ~ (m Int32), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetNUniformsMethodInfo a signature where
    overloadedMethod = gLShaderGetNUniforms

instance O.OverloadedMethodInfo GLShaderGetNUniformsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetNUniforms",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetNUniforms"
        })


#endif

-- method GLShader::get_resource
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_resource" gsk_gl_shader_get_resource :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    IO CString

{-# DEPRECATED gLShaderGetResource ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Gets the resource path for the GLSL sourcecode being used
-- to render this shader.
gLShaderGetResource ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> m (Maybe T.Text)
    -- ^ __Returns:__ The resource path for the shader
gLShaderGetResource shader = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    result <- gsk_gl_shader_get_resource shader'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- cstringToText result'
        return result''
    touchManagedPtr shader
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data GLShaderGetResourceMethodInfo
instance (signature ~ (m (Maybe T.Text)), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetResourceMethodInfo a signature where
    overloadedMethod = gLShaderGetResource

instance O.OverloadedMethodInfo GLShaderGetResourceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetResource",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetResource"
        })


#endif

-- method GLShader::get_source
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GLib" , name = "Bytes" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_source" gsk_gl_shader_get_source :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    IO (Ptr GLib.Bytes.Bytes)

{-# DEPRECATED gLShaderGetSource ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Gets the GLSL sourcecode being used to render this shader.
gLShaderGetSource ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> m GLib.Bytes.Bytes
    -- ^ __Returns:__ The source code for the shader
gLShaderGetSource shader = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    result <- gsk_gl_shader_get_source shader'
    checkUnexpectedReturnNULL "gLShaderGetSource" result
    result' <- (newBoxed GLib.Bytes.Bytes) result
    touchManagedPtr shader
    return result'

#if defined(ENABLE_OVERLOADING)
data GLShaderGetSourceMethodInfo
instance (signature ~ (m GLib.Bytes.Bytes), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetSourceMethodInfo a signature where
    overloadedMethod = gLShaderGetSource

instance O.OverloadedMethodInfo GLShaderGetSourceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetSource",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetSource"
        })


#endif

-- method GLShader::get_uniform_name
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TUTF8)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_uniform_name" gsk_gl_shader_get_uniform_name :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Int32 ->                                -- idx : TBasicType TInt
    IO CString

{-# DEPRECATED gLShaderGetUniformName ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Get the name of the declared uniform for this shader at index /@idx@/.
gLShaderGetUniformName ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> m T.Text
    -- ^ __Returns:__ The name of the declared uniform
gLShaderGetUniformName shader idx = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    result <- gsk_gl_shader_get_uniform_name shader' idx
    checkUnexpectedReturnNULL "gLShaderGetUniformName" result
    result' <- cstringToText result
    touchManagedPtr shader
    return result'

#if defined(ENABLE_OVERLOADING)
data GLShaderGetUniformNameMethodInfo
instance (signature ~ (Int32 -> m T.Text), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetUniformNameMethodInfo a signature where
    overloadedMethod = gLShaderGetUniformName

instance O.OverloadedMethodInfo GLShaderGetUniformNameMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetUniformName",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetUniformName"
        })


#endif

-- method GLShader::get_uniform_offset
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_uniform_offset" gsk_gl_shader_get_uniform_offset :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Int32 ->                                -- idx : TBasicType TInt
    IO Int32

{-# DEPRECATED gLShaderGetUniformOffset ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Get the offset into the data block where data for this uniforms is stored.
gLShaderGetUniformOffset ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> m Int32
    -- ^ __Returns:__ The data offset
gLShaderGetUniformOffset shader idx = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    result <- gsk_gl_shader_get_uniform_offset shader' idx
    touchManagedPtr shader
    return result

#if defined(ENABLE_OVERLOADING)
data GLShaderGetUniformOffsetMethodInfo
instance (signature ~ (Int32 -> m Int32), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetUniformOffsetMethodInfo a signature where
    overloadedMethod = gLShaderGetUniformOffset

instance O.OverloadedMethodInfo GLShaderGetUniformOffsetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetUniformOffset",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetUniformOffset"
        })


#endif

-- method GLShader::get_uniform_type
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "shader"
--           , argType =
--               TInterface Name { namespace = "Gsk" , name = "GLShader" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GskGLShader`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "idx"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "index of the uniform"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gsk" , name = "GLUniformType" })
-- throws : False
-- Skip return : False

foreign import ccall "gsk_gl_shader_get_uniform_type" gsk_gl_shader_get_uniform_type :: 
    Ptr GLShader ->                         -- shader : TInterface (Name {namespace = "Gsk", name = "GLShader"})
    Int32 ->                                -- idx : TBasicType TInt
    IO CUInt

{-# DEPRECATED gLShaderGetUniformType ["(Since version 4.16)","GTK\\'s new Vulkan-focused rendering","  does not support this feature. Use <http://developer.gnome.org/gsk/stable/../gtk4/class.GLArea.html GtkGLArea>","  for OpenGL rendering."] #-}
-- | Get the type of the declared uniform for this shader at index /@idx@/.
gLShaderGetUniformType ::
    (B.CallStack.HasCallStack, MonadIO m, IsGLShader a) =>
    a
    -- ^ /@shader@/: a @GskGLShader@
    -> Int32
    -- ^ /@idx@/: index of the uniform
    -> m Gsk.Enums.GLUniformType
    -- ^ __Returns:__ The type of the declared uniform
gLShaderGetUniformType shader idx = liftIO $ do
    shader' <- unsafeManagedPtrCastPtr shader
    result <- gsk_gl_shader_get_uniform_type shader' idx
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr shader
    return result'

#if defined(ENABLE_OVERLOADING)
data GLShaderGetUniformTypeMethodInfo
instance (signature ~ (Int32 -> m Gsk.Enums.GLUniformType), MonadIO m, IsGLShader a) => O.OverloadedMethod GLShaderGetUniformTypeMethodInfo a signature where
    overloadedMethod = gLShaderGetUniformType

instance O.OverloadedMethodInfo GLShaderGetUniformTypeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gsk.Objects.GLShader.gLShaderGetUniformType",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gsk-4.0.8/docs/GI-Gsk-Objects-GLShader.html#v:gLShaderGetUniformType"
        })


#endif


