{-# LANGUAGE TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- An interface for describing UI elements for Assistive Technologies.
-- 
-- Every accessible implementation has:
-- 
-- * a “role”, represented by a value of the t'GI.Gtk.Enums.AccessibleRole' enumeration
-- * “attributes”, represented by a set of t'GI.Gtk.Enums.AccessibleState',
--   t'GI.Gtk.Enums.AccessibleProperty' and t'GI.Gtk.Enums.AccessibleRelation' values
-- 
-- 
-- The role cannot be changed after instantiating a @GtkAccessible@
-- implementation.
-- 
-- The attributes are updated every time a UI element\'s state changes in
-- a way that should be reflected by assistive technologies. For instance,
-- if a @GtkWidget@ visibility changes, the 'GI.Gtk.Enums.AccessibleStateHidden'
-- state will also change to reflect the [Widget:visible]("GI.Gtk.Objects.Widget#g:attr:visible") property.
-- 
-- Every accessible implementation is part of a tree of accessible objects.
-- Normally, this tree corresponds to the widget tree, but can be customized
-- by reimplementing the t'GI.Gtk.Interfaces.Accessible.Accessible'.@/get_accessible_parent/@(),
-- t'GI.Gtk.Interfaces.Accessible.Accessible'.@/get_first_accessible_child/@() and
-- t'GI.Gtk.Interfaces.Accessible.Accessible'.@/get_next_accessible_sibling/@() virtual functions.
-- 
-- Note that you can not create a top-level accessible object as of now,
-- which means that you must always have a parent accessible object.
-- 
-- Also note that when an accessible object does not correspond to a widget,
-- and it has children, whose implementation you don\'t control,
-- it is necessary to ensure the correct shape of the a11y tree
-- by calling 'GI.Gtk.Interfaces.Accessible.accessibleSetAccessibleParent' and
-- updating the sibling by 'GI.Gtk.Interfaces.Accessible.accessibleUpdateNextAccessibleSibling'.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Interfaces.Accessible
    ( 

-- * Exported types
    Accessible(..)                          ,
    IsAccessible                            ,
    toAccessible                            ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [announce]("GI.Gtk.Interfaces.Accessible#g:method:announce"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [resetProperty]("GI.Gtk.Interfaces.Accessible#g:method:resetProperty"), [resetRelation]("GI.Gtk.Interfaces.Accessible#g:method:resetRelation"), [resetState]("GI.Gtk.Interfaces.Accessible#g:method:resetState"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [updateNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:updateNextAccessibleSibling"), [updatePlatformState]("GI.Gtk.Interfaces.Accessible#g:method:updatePlatformState"), [updateProperty]("GI.Gtk.Interfaces.Accessible#g:method:updateProperty"), [updateRelation]("GI.Gtk.Interfaces.Accessible#g:method:updateRelation"), [updateState]("GI.Gtk.Interfaces.Accessible#g:method:updateState"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleParent"), [getAccessibleRole]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleRole"), [getAtContext]("GI.Gtk.Interfaces.Accessible#g:method:getAtContext"), [getBounds]("GI.Gtk.Interfaces.Accessible#g:method:getBounds"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getFirstAccessibleChild]("GI.Gtk.Interfaces.Accessible#g:method:getFirstAccessibleChild"), [getNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:getNextAccessibleSibling"), [getPlatformState]("GI.Gtk.Interfaces.Accessible#g:method:getPlatformState"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:setAccessibleParent"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveAccessibleMethod                 ,
#endif

-- ** announce #method:announce#

#if defined(ENABLE_OVERLOADING)
    AccessibleAnnounceMethodInfo            ,
#endif
    accessibleAnnounce                      ,


-- ** getAccessibleParent #method:getAccessibleParent#

#if defined(ENABLE_OVERLOADING)
    AccessibleGetAccessibleParentMethodInfo ,
#endif
    accessibleGetAccessibleParent           ,


-- ** getAccessibleRole #method:getAccessibleRole#

#if defined(ENABLE_OVERLOADING)
    AccessibleGetAccessibleRoleMethodInfo   ,
#endif
    accessibleGetAccessibleRole             ,


-- ** getAtContext #method:getAtContext#

#if defined(ENABLE_OVERLOADING)
    AccessibleGetAtContextMethodInfo        ,
#endif
    accessibleGetAtContext                  ,


-- ** getBounds #method:getBounds#

#if defined(ENABLE_OVERLOADING)
    AccessibleGetBoundsMethodInfo           ,
#endif
    accessibleGetBounds                     ,


-- ** getFirstAccessibleChild #method:getFirstAccessibleChild#

#if defined(ENABLE_OVERLOADING)
    AccessibleGetFirstAccessibleChildMethodInfo,
#endif
    accessibleGetFirstAccessibleChild       ,


-- ** getNextAccessibleSibling #method:getNextAccessibleSibling#

#if defined(ENABLE_OVERLOADING)
    AccessibleGetNextAccessibleSiblingMethodInfo,
#endif
    accessibleGetNextAccessibleSibling      ,


-- ** getPlatformState #method:getPlatformState#

#if defined(ENABLE_OVERLOADING)
    AccessibleGetPlatformStateMethodInfo    ,
#endif
    accessibleGetPlatformState              ,


-- ** resetProperty #method:resetProperty#

#if defined(ENABLE_OVERLOADING)
    AccessibleResetPropertyMethodInfo       ,
#endif
    accessibleResetProperty                 ,


-- ** resetRelation #method:resetRelation#

#if defined(ENABLE_OVERLOADING)
    AccessibleResetRelationMethodInfo       ,
#endif
    accessibleResetRelation                 ,


-- ** resetState #method:resetState#

#if defined(ENABLE_OVERLOADING)
    AccessibleResetStateMethodInfo          ,
#endif
    accessibleResetState                    ,


-- ** setAccessibleParent #method:setAccessibleParent#

#if defined(ENABLE_OVERLOADING)
    AccessibleSetAccessibleParentMethodInfo ,
#endif
    accessibleSetAccessibleParent           ,


-- ** updateNextAccessibleSibling #method:updateNextAccessibleSibling#

#if defined(ENABLE_OVERLOADING)
    AccessibleUpdateNextAccessibleSiblingMethodInfo,
#endif
    accessibleUpdateNextAccessibleSibling   ,


-- ** updatePlatformState #method:updatePlatformState#

#if defined(ENABLE_OVERLOADING)
    AccessibleUpdatePlatformStateMethodInfo ,
#endif
    accessibleUpdatePlatformState           ,


-- ** updateProperty #method:updateProperty#

#if defined(ENABLE_OVERLOADING)
    AccessibleUpdatePropertyMethodInfo      ,
#endif
    accessibleUpdateProperty                ,


-- ** updateRelation #method:updateRelation#

#if defined(ENABLE_OVERLOADING)
    AccessibleUpdateRelationMethodInfo      ,
#endif
    accessibleUpdateRelation                ,


-- ** updateState #method:updateState#

#if defined(ENABLE_OVERLOADING)
    AccessibleUpdateStateMethodInfo         ,
#endif
    accessibleUpdateState                   ,




 -- * Properties


-- ** accessibleRole #attr:accessibleRole#
-- | The accessible role of the given @GtkAccessible@ implementation.
-- 
-- The accessible role cannot be changed once set.

#if defined(ENABLE_OVERLOADING)
    AccessibleAccessibleRolePropertyInfo    ,
#endif
#if defined(ENABLE_OVERLOADING)
    accessibleAccessibleRole                ,
#endif
    constructAccessibleAccessibleRole       ,
    getAccessibleAccessibleRole             ,
    setAccessibleAccessibleRole             ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Objects.Display as Gdk.Display
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext

#endif

-- interface Accessible 
-- | Memory-managed wrapper type.
newtype Accessible = Accessible (SP.ManagedPtr Accessible)
    deriving (Eq)

instance SP.ManagedPtrNewtype Accessible where
    toManagedPtr (Accessible p) = p

foreign import ccall "gtk_accessible_get_type"
    c_gtk_accessible_get_type :: IO B.Types.GType

instance B.Types.TypedObject Accessible where
    glibType = c_gtk_accessible_get_type

instance B.Types.GObject Accessible

-- | Type class for types which can be safely cast to t'Accessible', for instance with `toAccessible`.
class (SP.GObject o, O.IsDescendantOf Accessible o) => IsAccessible o
instance (SP.GObject o, O.IsDescendantOf Accessible o) => IsAccessible o

instance O.HasParentTypes Accessible
type instance O.ParentTypes Accessible = '[GObject.Object.Object]

-- | Cast to t'Accessible', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toAccessible :: (MIO.MonadIO m, IsAccessible o) => o -> m Accessible
toAccessible = MIO.liftIO . B.ManagedPtr.unsafeCastTo Accessible

-- | Convert t'Accessible' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Accessible) where
    gvalueGType_ = c_gtk_accessible_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Accessible)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Accessible)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Accessible ptr
        else return P.Nothing
        
    

-- VVV Prop "accessible-role"
   -- Type: TInterface (Name {namespace = "Gtk", name = "AccessibleRole"})
   -- Flags: [PropertyReadable,PropertyWritable]
   -- Nullable: (Just False,Nothing)

-- | Get the value of the “@accessible-role@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' accessible #accessibleRole
-- @
getAccessibleAccessibleRole :: (MonadIO m, IsAccessible o) => o -> m Gtk.Enums.AccessibleRole
getAccessibleAccessibleRole obj = MIO.liftIO $ B.Properties.getObjectPropertyEnum obj "accessible-role"

-- | Set the value of the “@accessible-role@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.set' accessible [ #accessibleRole 'Data.GI.Base.Attributes.:=' value ]
-- @
setAccessibleAccessibleRole :: (MonadIO m, IsAccessible o) => o -> Gtk.Enums.AccessibleRole -> m ()
setAccessibleAccessibleRole obj val = MIO.liftIO $ do
    B.Properties.setObjectPropertyEnum obj "accessible-role" val

-- | Construct a t'GValueConstruct' with valid value for the “@accessible-role@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructAccessibleAccessibleRole :: (IsAccessible o, MIO.MonadIO m) => Gtk.Enums.AccessibleRole -> m (GValueConstruct o)
constructAccessibleAccessibleRole val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyEnum "accessible-role" val

#if defined(ENABLE_OVERLOADING)
data AccessibleAccessibleRolePropertyInfo
instance AttrInfo AccessibleAccessibleRolePropertyInfo where
    type AttrAllowedOps AccessibleAccessibleRolePropertyInfo = '[ 'AttrSet, 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint AccessibleAccessibleRolePropertyInfo = IsAccessible
    type AttrSetTypeConstraint AccessibleAccessibleRolePropertyInfo = (~) Gtk.Enums.AccessibleRole
    type AttrTransferTypeConstraint AccessibleAccessibleRolePropertyInfo = (~) Gtk.Enums.AccessibleRole
    type AttrTransferType AccessibleAccessibleRolePropertyInfo = Gtk.Enums.AccessibleRole
    type AttrGetType AccessibleAccessibleRolePropertyInfo = Gtk.Enums.AccessibleRole
    type AttrLabel AccessibleAccessibleRolePropertyInfo = "accessible-role"
    type AttrOrigin AccessibleAccessibleRolePropertyInfo = Accessible
    attrGet = getAccessibleAccessibleRole
    attrSet = setAccessibleAccessibleRole
    attrTransfer _ v = do
        return v
    attrConstruct = constructAccessibleAccessibleRole
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleRole"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#g:attr:accessibleRole"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Accessible
type instance O.AttributeList Accessible = AccessibleAttributeList
type AccessibleAttributeList = ('[ '("accessibleRole", AccessibleAccessibleRolePropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
accessibleAccessibleRole :: AttrLabelProxy "accessibleRole"
accessibleAccessibleRole = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type family ResolveAccessibleMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveAccessibleMethod "announce" o = AccessibleAnnounceMethodInfo
    ResolveAccessibleMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveAccessibleMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveAccessibleMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveAccessibleMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveAccessibleMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveAccessibleMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveAccessibleMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveAccessibleMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveAccessibleMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveAccessibleMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveAccessibleMethod "resetProperty" o = AccessibleResetPropertyMethodInfo
    ResolveAccessibleMethod "resetRelation" o = AccessibleResetRelationMethodInfo
    ResolveAccessibleMethod "resetState" o = AccessibleResetStateMethodInfo
    ResolveAccessibleMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveAccessibleMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveAccessibleMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveAccessibleMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveAccessibleMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveAccessibleMethod "updateNextAccessibleSibling" o = AccessibleUpdateNextAccessibleSiblingMethodInfo
    ResolveAccessibleMethod "updatePlatformState" o = AccessibleUpdatePlatformStateMethodInfo
    ResolveAccessibleMethod "updateProperty" o = AccessibleUpdatePropertyMethodInfo
    ResolveAccessibleMethod "updateRelation" o = AccessibleUpdateRelationMethodInfo
    ResolveAccessibleMethod "updateState" o = AccessibleUpdateStateMethodInfo
    ResolveAccessibleMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveAccessibleMethod "getAccessibleParent" o = AccessibleGetAccessibleParentMethodInfo
    ResolveAccessibleMethod "getAccessibleRole" o = AccessibleGetAccessibleRoleMethodInfo
    ResolveAccessibleMethod "getAtContext" o = AccessibleGetAtContextMethodInfo
    ResolveAccessibleMethod "getBounds" o = AccessibleGetBoundsMethodInfo
    ResolveAccessibleMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveAccessibleMethod "getFirstAccessibleChild" o = AccessibleGetFirstAccessibleChildMethodInfo
    ResolveAccessibleMethod "getNextAccessibleSibling" o = AccessibleGetNextAccessibleSiblingMethodInfo
    ResolveAccessibleMethod "getPlatformState" o = AccessibleGetPlatformStateMethodInfo
    ResolveAccessibleMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveAccessibleMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveAccessibleMethod "setAccessibleParent" o = AccessibleSetAccessibleParentMethodInfo
    ResolveAccessibleMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveAccessibleMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveAccessibleMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveAccessibleMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveAccessibleMethod t Accessible, O.OverloadedMethod info Accessible p) => OL.IsLabel t (Accessible -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveAccessibleMethod t Accessible, O.OverloadedMethod info Accessible p, R.HasField t Accessible p) => R.HasField t Accessible p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveAccessibleMethod t Accessible, O.OverloadedMethodInfo info Accessible) => OL.IsLabel t (O.MethodProxy info Accessible) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- method Accessible::announce
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "message"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the string to announce"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "priority"
--           , argType =
--               TInterface
--                 Name
--                   { namespace = "Gtk" , name = "AccessibleAnnouncementPriority" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the priority of the announcement"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_announce" gtk_accessible_announce :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    CString ->                              -- message : TBasicType TUTF8
    CUInt ->                                -- priority : TInterface (Name {namespace = "Gtk", name = "AccessibleAnnouncementPriority"})
    IO ()

-- | Requests the user\'s screen reader to announce the given message.
-- 
-- This kind of notification is useful for messages that
-- either have only a visual representation or that are not
-- exposed visually at all, e.g. a notification about a
-- successful operation.
-- 
-- Also, by using this API, you can ensure that the message
-- does not interrupts the user\'s current screen reader output.
-- 
-- /Since: 4.14/
accessibleAnnounce ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> T.Text
    -- ^ /@message@/: the string to announce
    -> Gtk.Enums.AccessibleAnnouncementPriority
    -- ^ /@priority@/: the priority of the announcement
    -> m ()
accessibleAnnounce self message priority = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    message' <- textToCString message
    let priority' = (fromIntegral . fromEnum) priority
    gtk_accessible_announce self' message' priority'
    touchManagedPtr self
    freeMem message'
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleAnnounceMethodInfo
instance (signature ~ (T.Text -> Gtk.Enums.AccessibleAnnouncementPriority -> m ()), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleAnnounceMethodInfo a signature where
    overloadedMethod = accessibleAnnounce

instance O.OverloadedMethodInfo AccessibleAnnounceMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleAnnounce",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleAnnounce"
        })


#endif

-- method Accessible::get_accessible_parent
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Accessible" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_get_accessible_parent" gtk_accessible_get_accessible_parent :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    IO (Ptr Accessible)

-- | Retrieves the accessible parent for an accessible object.
-- 
-- This function returns @NULL@ for top level widgets.
-- 
-- /Since: 4.10/
accessibleGetAccessibleParent ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> m (Maybe Accessible)
    -- ^ __Returns:__ the accessible parent
accessibleGetAccessibleParent self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_accessible_get_accessible_parent self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject Accessible) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data AccessibleGetAccessibleParentMethodInfo
instance (signature ~ (m (Maybe Accessible)), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleGetAccessibleParentMethodInfo a signature where
    overloadedMethod = accessibleGetAccessibleParent

instance O.OverloadedMethodInfo AccessibleGetAccessibleParentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleGetAccessibleParent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleGetAccessibleParent"
        })


#endif

-- method Accessible::get_accessible_role
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just
--               (TInterface Name { namespace = "Gtk" , name = "AccessibleRole" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_get_accessible_role" gtk_accessible_get_accessible_role :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    IO CUInt

-- | Retrieves the accessible role of an accessible object.
accessibleGetAccessibleRole ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> m Gtk.Enums.AccessibleRole
    -- ^ __Returns:__ the accessible role
accessibleGetAccessibleRole self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_accessible_get_accessible_role self'
    let result' = (toEnum . fromIntegral) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data AccessibleGetAccessibleRoleMethodInfo
instance (signature ~ (m Gtk.Enums.AccessibleRole), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleGetAccessibleRoleMethodInfo a signature where
    overloadedMethod = accessibleGetAccessibleRole

instance O.OverloadedMethodInfo AccessibleGetAccessibleRoleMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleGetAccessibleRole",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleGetAccessibleRole"
        })


#endif

-- method Accessible::get_at_context
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "ATContext" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_get_at_context" gtk_accessible_get_at_context :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    IO (Ptr Gtk.ATContext.ATContext)

-- | Retrieves the implementation for the given accessible object.
-- 
-- /Since: 4.10/
accessibleGetAtContext ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> m Gtk.ATContext.ATContext
    -- ^ __Returns:__ the accessible implementation object
accessibleGetAtContext self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_accessible_get_at_context self'
    checkUnexpectedReturnNULL "accessibleGetAtContext" result
    result' <- (wrapObject Gtk.ATContext.ATContext) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data AccessibleGetAtContextMethodInfo
instance (signature ~ (m Gtk.ATContext.ATContext), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleGetAtContextMethodInfo a signature where
    overloadedMethod = accessibleGetAtContext

instance O.OverloadedMethodInfo AccessibleGetAtContextMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleGetAtContext",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleGetAtContext"
        })


#endif

-- method Accessible::get_bounds
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "x"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the x coordinate of the top left corner of the accessible"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "y"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the y coordinate of the top left corner of the widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "width"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the width of the accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       , Arg
--           { argCName = "height"
--           , argType = TBasicType TInt
--           , direction = DirectionOut
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the height of the accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_get_bounds" gtk_accessible_get_bounds :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    Ptr Int32 ->                            -- x : TBasicType TInt
    Ptr Int32 ->                            -- y : TBasicType TInt
    Ptr Int32 ->                            -- width : TBasicType TInt
    Ptr Int32 ->                            -- height : TBasicType TInt
    IO CInt

-- | Queries the coordinates and dimensions of this accessible
-- 
-- This functionality can be overridden by @GtkAccessible@
-- implementations, e.g. to get the bounds from an ignored
-- child widget.
-- 
-- /Since: 4.10/
accessibleGetBounds ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> m ((Bool, Int32, Int32, Int32, Int32))
    -- ^ __Returns:__ true if the bounds are valid, and false otherwise
accessibleGetBounds self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    x <- allocMem :: IO (Ptr Int32)
    y <- allocMem :: IO (Ptr Int32)
    width <- allocMem :: IO (Ptr Int32)
    height <- allocMem :: IO (Ptr Int32)
    result <- gtk_accessible_get_bounds self' x y width height
    let result' = (/= 0) result
    x' <- peek x
    y' <- peek y
    width' <- peek width
    height' <- peek height
    touchManagedPtr self
    freeMem x
    freeMem y
    freeMem width
    freeMem height
    return (result', x', y', width', height')

#if defined(ENABLE_OVERLOADING)
data AccessibleGetBoundsMethodInfo
instance (signature ~ (m ((Bool, Int32, Int32, Int32, Int32))), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleGetBoundsMethodInfo a signature where
    overloadedMethod = accessibleGetBounds

instance O.OverloadedMethodInfo AccessibleGetBoundsMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleGetBounds",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleGetBounds"
        })


#endif

-- method Accessible::get_first_accessible_child
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Accessible" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_get_first_accessible_child" gtk_accessible_get_first_accessible_child :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    IO (Ptr Accessible)

-- | Retrieves the first accessible child of an accessible object.
-- 
-- /Since: 4.10/
accessibleGetFirstAccessibleChild ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> m (Maybe Accessible)
    -- ^ __Returns:__ the first accessible child
accessibleGetFirstAccessibleChild self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_accessible_get_first_accessible_child self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject Accessible) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data AccessibleGetFirstAccessibleChildMethodInfo
instance (signature ~ (m (Maybe Accessible)), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleGetFirstAccessibleChildMethodInfo a signature where
    overloadedMethod = accessibleGetFirstAccessibleChild

instance O.OverloadedMethodInfo AccessibleGetFirstAccessibleChildMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleGetFirstAccessibleChild",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleGetFirstAccessibleChild"
        })


#endif

-- method Accessible::get_next_accessible_sibling
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Accessible" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_get_next_accessible_sibling" gtk_accessible_get_next_accessible_sibling :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    IO (Ptr Accessible)

-- | Retrieves the next accessible sibling of an accessible object
-- 
-- /Since: 4.10/
accessibleGetNextAccessibleSibling ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> m (Maybe Accessible)
    -- ^ __Returns:__ the next accessible sibling
accessibleGetNextAccessibleSibling self = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    result <- gtk_accessible_get_next_accessible_sibling self'
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (wrapObject Accessible) result'
        return result''
    touchManagedPtr self
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data AccessibleGetNextAccessibleSiblingMethodInfo
instance (signature ~ (m (Maybe Accessible)), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleGetNextAccessibleSiblingMethodInfo a signature where
    overloadedMethod = accessibleGetNextAccessibleSibling

instance O.OverloadedMethodInfo AccessibleGetNextAccessibleSiblingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleGetNextAccessibleSibling",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleGetNextAccessibleSibling"
        })


#endif

-- method Accessible::get_platform_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "state"
--           , argType =
--               TInterface
--                 Name { namespace = "Gtk" , name = "AccessiblePlatformState" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "platform state to query"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_get_platform_state" gtk_accessible_get_platform_state :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    CUInt ->                                -- state : TInterface (Name {namespace = "Gtk", name = "AccessiblePlatformState"})
    IO CInt

-- | Queries a platform state, such as focus.
-- 
-- This functionality can be overridden by @GtkAccessible@
-- implementations, e.g. to get platform state from an ignored
-- child widget, as is the case for @GtkText@ wrappers.
-- 
-- /Since: 4.10/
accessibleGetPlatformState ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> Gtk.Enums.AccessiblePlatformState
    -- ^ /@state@/: platform state to query
    -> m Bool
    -- ^ __Returns:__ the value of state for the accessible
accessibleGetPlatformState self state = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let state' = (fromIntegral . fromEnum) state
    result <- gtk_accessible_get_platform_state self' state'
    let result' = (/= 0) result
    touchManagedPtr self
    return result'

#if defined(ENABLE_OVERLOADING)
data AccessibleGetPlatformStateMethodInfo
instance (signature ~ (Gtk.Enums.AccessiblePlatformState -> m Bool), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleGetPlatformStateMethodInfo a signature where
    overloadedMethod = accessibleGetPlatformState

instance O.OverloadedMethodInfo AccessibleGetPlatformStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleGetPlatformState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleGetPlatformState"
        })


#endif

-- method Accessible::reset_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "AccessibleProperty" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the accessible property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_reset_property" gtk_accessible_reset_property :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    CUInt ->                                -- property : TInterface (Name {namespace = "Gtk", name = "AccessibleProperty"})
    IO ()

-- | Resets the accessible property to its default value.
accessibleResetProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> Gtk.Enums.AccessibleProperty
    -- ^ /@property@/: the accessible property
    -> m ()
accessibleResetProperty self property = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let property' = (fromIntegral . fromEnum) property
    gtk_accessible_reset_property self' property'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleResetPropertyMethodInfo
instance (signature ~ (Gtk.Enums.AccessibleProperty -> m ()), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleResetPropertyMethodInfo a signature where
    overloadedMethod = accessibleResetProperty

instance O.OverloadedMethodInfo AccessibleResetPropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleResetProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleResetProperty"
        })


#endif

-- method Accessible::reset_relation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "relation"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "AccessibleRelation" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the accessible relation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_reset_relation" gtk_accessible_reset_relation :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    CUInt ->                                -- relation : TInterface (Name {namespace = "Gtk", name = "AccessibleRelation"})
    IO ()

-- | Resets the accessible relation to its default value.
accessibleResetRelation ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> Gtk.Enums.AccessibleRelation
    -- ^ /@relation@/: the accessible relation
    -> m ()
accessibleResetRelation self relation = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let relation' = (fromIntegral . fromEnum) relation
    gtk_accessible_reset_relation self' relation'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleResetRelationMethodInfo
instance (signature ~ (Gtk.Enums.AccessibleRelation -> m ()), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleResetRelationMethodInfo a signature where
    overloadedMethod = accessibleResetRelation

instance O.OverloadedMethodInfo AccessibleResetRelationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleResetRelation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleResetRelation"
        })


#endif

-- method Accessible::reset_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "state"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "AccessibleState" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the accessible state"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_reset_state" gtk_accessible_reset_state :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    CUInt ->                                -- state : TInterface (Name {namespace = "Gtk", name = "AccessibleState"})
    IO ()

-- | Resets the accessible state to its default value.
accessibleResetState ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> Gtk.Enums.AccessibleState
    -- ^ /@state@/: the accessible state
    -> m ()
accessibleResetState self state = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let state' = (fromIntegral . fromEnum) state
    gtk_accessible_reset_state self' state'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleResetStateMethodInfo
instance (signature ~ (Gtk.Enums.AccessibleState -> m ()), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleResetStateMethodInfo a signature where
    overloadedMethod = accessibleResetState

instance O.OverloadedMethodInfo AccessibleResetStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleResetState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleResetState"
        })


#endif

-- method Accessible::set_accessible_parent
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parent"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the parent accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "next_sibling"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the sibling accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_set_accessible_parent" gtk_accessible_set_accessible_parent :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    Ptr Accessible ->                       -- parent : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    Ptr Accessible ->                       -- next_sibling : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    IO ()

-- | Sets the parent and sibling of an accessible object.
-- 
-- This function is meant to be used by accessible implementations that are
-- not part of the widget hierarchy, and but act as a logical bridge between
-- widgets. For instance, if a widget creates an object that holds metadata
-- for each child, and you want that object to implement the @GtkAccessible@
-- interface, you will use this function to ensure that the parent of each
-- child widget is the metadata object, and the parent of each metadata
-- object is the container widget.
-- 
-- /Since: 4.10/
accessibleSetAccessibleParent ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a, IsAccessible b, IsAccessible c) =>
    a
    -- ^ /@self@/: an accessible object
    -> Maybe (b)
    -- ^ /@parent@/: the parent accessible object
    -> Maybe (c)
    -- ^ /@nextSibling@/: the sibling accessible object
    -> m ()
accessibleSetAccessibleParent self parent nextSibling = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    maybeParent <- case parent of
        Nothing -> return FP.nullPtr
        Just jParent -> do
            jParent' <- unsafeManagedPtrCastPtr jParent
            return jParent'
    maybeNextSibling <- case nextSibling of
        Nothing -> return FP.nullPtr
        Just jNextSibling -> do
            jNextSibling' <- unsafeManagedPtrCastPtr jNextSibling
            return jNextSibling'
    gtk_accessible_set_accessible_parent self' maybeParent maybeNextSibling
    touchManagedPtr self
    whenJust parent touchManagedPtr
    whenJust nextSibling touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleSetAccessibleParentMethodInfo
instance (signature ~ (Maybe (b) -> Maybe (c) -> m ()), MonadIO m, IsAccessible a, IsAccessible b, IsAccessible c) => O.OverloadedMethod AccessibleSetAccessibleParentMethodInfo a signature where
    overloadedMethod = accessibleSetAccessibleParent

instance O.OverloadedMethodInfo AccessibleSetAccessibleParentMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleSetAccessibleParent",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleSetAccessibleParent"
        })


#endif

-- method Accessible::update_next_accessible_sibling
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "new_sibling"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the new next accessible sibling to set"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_update_next_accessible_sibling" gtk_accessible_update_next_accessible_sibling :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    Ptr Accessible ->                       -- new_sibling : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    IO ()

-- | Updates the next accessible sibling.
-- 
-- That might be useful when a new child of a custom accessible
-- is created, and it needs to be linked to a previous child.
-- 
-- /Since: 4.10/
accessibleUpdateNextAccessibleSibling ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a, IsAccessible b) =>
    a
    -- ^ /@self@/: an accessible object
    -> Maybe (b)
    -- ^ /@newSibling@/: the new next accessible sibling to set
    -> m ()
accessibleUpdateNextAccessibleSibling self newSibling = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    maybeNewSibling <- case newSibling of
        Nothing -> return FP.nullPtr
        Just jNewSibling -> do
            jNewSibling' <- unsafeManagedPtrCastPtr jNewSibling
            return jNewSibling'
    gtk_accessible_update_next_accessible_sibling self' maybeNewSibling
    touchManagedPtr self
    whenJust newSibling touchManagedPtr
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleUpdateNextAccessibleSiblingMethodInfo
instance (signature ~ (Maybe (b) -> m ()), MonadIO m, IsAccessible a, IsAccessible b) => O.OverloadedMethod AccessibleUpdateNextAccessibleSiblingMethodInfo a signature where
    overloadedMethod = accessibleUpdateNextAccessibleSibling

instance O.OverloadedMethodInfo AccessibleUpdateNextAccessibleSiblingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleUpdateNextAccessibleSibling",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleUpdateNextAccessibleSibling"
        })


#endif

-- method Accessible::update_platform_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "state"
--           , argType =
--               TInterface
--                 Name { namespace = "Gtk" , name = "AccessiblePlatformState" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the platform state to update"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_update_platform_state" gtk_accessible_update_platform_state :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    CUInt ->                                -- state : TInterface (Name {namespace = "Gtk", name = "AccessiblePlatformState"})
    IO ()

-- | Informs ATs that the platform state has changed.
-- 
-- This function should be used by @GtkAccessible@ implementations that
-- have a platform state but are not widgets. Widgets handle platform
-- states automatically.
-- 
-- /Since: 4.18/
accessibleUpdatePlatformState ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> Gtk.Enums.AccessiblePlatformState
    -- ^ /@state@/: the platform state to update
    -> m ()
accessibleUpdatePlatformState self state = liftIO $ do
    self' <- unsafeManagedPtrCastPtr self
    let state' = (fromIntegral . fromEnum) state
    gtk_accessible_update_platform_state self' state'
    touchManagedPtr self
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleUpdatePlatformStateMethodInfo
instance (signature ~ (Gtk.Enums.AccessiblePlatformState -> m ()), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleUpdatePlatformStateMethodInfo a signature where
    overloadedMethod = accessibleUpdatePlatformState

instance O.OverloadedMethodInfo AccessibleUpdatePlatformStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleUpdatePlatformState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleUpdatePlatformState"
        })


#endif

-- method Accessible::update_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_properties"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of accessible properties to set"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "properties"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 1
--                 (TInterface
--                    Name { namespace = "Gtk" , name = "AccessibleProperty" })
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of accessible properties"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "values"
--           , argType = TCArray False (-1) 1 TGValue
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of `GValues`, one for each property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_properties"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of accessible properties to set"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          , Arg
--              { argCName = "n_properties"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of accessible properties to set"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_update_property_value" gtk_accessible_update_property_value :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    Int32 ->                                -- n_properties : TBasicType TInt
    Ptr CUInt ->                            -- properties : TCArray False (-1) 1 (TInterface (Name {namespace = "Gtk", name = "AccessibleProperty"}))
    Ptr B.GValue.GValue ->                  -- values : TCArray False (-1) 1 TGValue
    IO ()

-- | Updates an array of accessible properties.
-- 
-- This function should be called by @GtkWidget@ types whenever an accessible
-- property change must be communicated to assistive technologies.
-- 
-- This function is meant to be used by language bindings.
accessibleUpdateProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> [Gtk.Enums.AccessibleProperty]
    -- ^ /@properties@/: an array of accessible properties
    -> [GValue]
    -- ^ /@values@/: an array of @GValues@, one for each property
    -> m ()
accessibleUpdateProperty self properties values = liftIO $ do
    let nProperties = fromIntegral $ P.length values
    let properties_expected_length_ = fromIntegral $ P.length properties
    when (properties_expected_length_ /= nProperties) $
        error "Gtk.accessibleUpdateProperty : length of 'properties' does not agree with that of 'values'."
    self' <- unsafeManagedPtrCastPtr self
    let properties' = map (fromIntegral . fromEnum) properties
    properties'' <- packStorableArray properties'
    values' <- B.GValue.packGValueArray values
    gtk_accessible_update_property_value self' nProperties properties'' values'
    touchManagedPtr self
    mapM_ touchManagedPtr values
    freeMem properties''
    freeMem values'
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleUpdatePropertyMethodInfo
instance (signature ~ ([Gtk.Enums.AccessibleProperty] -> [GValue] -> m ()), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleUpdatePropertyMethodInfo a signature where
    overloadedMethod = accessibleUpdateProperty

instance O.OverloadedMethodInfo AccessibleUpdatePropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleUpdateProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleUpdateProperty"
        })


#endif

-- method Accessible::update_relation
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible object"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_relations"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of accessible relations to set"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "relations"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 1
--                 (TInterface
--                    Name { namespace = "Gtk" , name = "AccessibleRelation" })
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of accessible relations"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "values"
--           , argType = TCArray False (-1) 1 TGValue
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of `GValues`, one for each relation"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_relations"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of accessible relations to set"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          , Arg
--              { argCName = "n_relations"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of accessible relations to set"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_update_relation_value" gtk_accessible_update_relation_value :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    Int32 ->                                -- n_relations : TBasicType TInt
    Ptr CUInt ->                            -- relations : TCArray False (-1) 1 (TInterface (Name {namespace = "Gtk", name = "AccessibleRelation"}))
    Ptr B.GValue.GValue ->                  -- values : TCArray False (-1) 1 TGValue
    IO ()

-- | Updates an array of accessible relations.
-- 
-- This function should be called by @GtkWidget@ types whenever an accessible
-- relation change must be communicated to assistive technologies.
-- 
-- This function is meant to be used by language bindings.
accessibleUpdateRelation ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible object
    -> [Gtk.Enums.AccessibleRelation]
    -- ^ /@relations@/: an array of accessible relations
    -> [GValue]
    -- ^ /@values@/: an array of @GValues@, one for each relation
    -> m ()
accessibleUpdateRelation self relations values = liftIO $ do
    let nRelations = fromIntegral $ P.length values
    let relations_expected_length_ = fromIntegral $ P.length relations
    when (relations_expected_length_ /= nRelations) $
        error "Gtk.accessibleUpdateRelation : length of 'relations' does not agree with that of 'values'."
    self' <- unsafeManagedPtrCastPtr self
    let relations' = map (fromIntegral . fromEnum) relations
    relations'' <- packStorableArray relations'
    values' <- B.GValue.packGValueArray values
    gtk_accessible_update_relation_value self' nRelations relations'' values'
    touchManagedPtr self
    mapM_ touchManagedPtr values
    freeMem relations''
    freeMem values'
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleUpdateRelationMethodInfo
instance (signature ~ ([Gtk.Enums.AccessibleRelation] -> [GValue] -> m ()), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleUpdateRelationMethodInfo a signature where
    overloadedMethod = accessibleUpdateRelation

instance O.OverloadedMethodInfo AccessibleUpdateRelationMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleUpdateRelation",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleUpdateRelation"
        })


#endif

-- method Accessible::update_state
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "self"
--           , argType =
--               TInterface Name { namespace = "Gtk" , name = "Accessible" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an accessible objedct"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_states"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of accessible states to set"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "states"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 1
--                 (TInterface Name { namespace = "Gtk" , name = "AccessibleState" })
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of accessible states"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "values"
--           , argType = TCArray False (-1) 1 TGValue
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of `GValues`, one for each state"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_states"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of accessible states to set"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          , Arg
--              { argCName = "n_states"
--              , argType = TBasicType TInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of accessible states to set"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_accessible_update_state_value" gtk_accessible_update_state_value :: 
    Ptr Accessible ->                       -- self : TInterface (Name {namespace = "Gtk", name = "Accessible"})
    Int32 ->                                -- n_states : TBasicType TInt
    Ptr CUInt ->                            -- states : TCArray False (-1) 1 (TInterface (Name {namespace = "Gtk", name = "AccessibleState"}))
    Ptr B.GValue.GValue ->                  -- values : TCArray False (-1) 1 TGValue
    IO ()

-- | Updates an array of accessible states.
-- 
-- This function should be called by @GtkWidget@ types whenever an accessible
-- state change must be communicated to assistive technologies.
-- 
-- This function is meant to be used by language bindings.
accessibleUpdateState ::
    (B.CallStack.HasCallStack, MonadIO m, IsAccessible a) =>
    a
    -- ^ /@self@/: an accessible objedct
    -> [Gtk.Enums.AccessibleState]
    -- ^ /@states@/: an array of accessible states
    -> [GValue]
    -- ^ /@values@/: an array of @GValues@, one for each state
    -> m ()
accessibleUpdateState self states values = liftIO $ do
    let nStates = fromIntegral $ P.length values
    let states_expected_length_ = fromIntegral $ P.length states
    when (states_expected_length_ /= nStates) $
        error "Gtk.accessibleUpdateState : length of 'states' does not agree with that of 'values'."
    self' <- unsafeManagedPtrCastPtr self
    let states' = map (fromIntegral . fromEnum) states
    states'' <- packStorableArray states'
    values' <- B.GValue.packGValueArray values
    gtk_accessible_update_state_value self' nStates states'' values'
    touchManagedPtr self
    mapM_ touchManagedPtr values
    freeMem states''
    freeMem values'
    return ()

#if defined(ENABLE_OVERLOADING)
data AccessibleUpdateStateMethodInfo
instance (signature ~ ([Gtk.Enums.AccessibleState] -> [GValue] -> m ()), MonadIO m, IsAccessible a) => O.OverloadedMethod AccessibleUpdateStateMethodInfo a signature where
    overloadedMethod = accessibleUpdateState

instance O.OverloadedMethodInfo AccessibleUpdateStateMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Interfaces.Accessible.accessibleUpdateState",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Interfaces-Accessible.html#v:accessibleUpdateState"
        })


#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Accessible = AccessibleSignalList
type AccessibleSignalList = ('[ '("notify", GObject.Object.ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif


