// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef EXTENSIONS_RENDERER_CHROME_SETTING_H_
#define EXTENSIONS_RENDERER_CHROME_SETTING_H_

#include <string>

#include "base/memory/raw_ptr.h"
#include "base/values.h"
#include "extensions/renderer/bindings/argument_spec.h"
#include "gin/public/wrappable_pointer_tags.h"
#include "gin/wrappable.h"
#include "v8/include/v8-forward.h"

namespace gin {
class Arguments;
}

namespace extensions {
class APIEventHandler;
class APIRequestHandler;
class BindingAccessChecker;

// The custom implementation of the ChromeSetting type exposed to APIs.
class ChromeSetting final : public gin::Wrappable<ChromeSetting> {
 public:
  static constexpr gin::WrapperInfo kWrapperInfo = {{gin::kEmbedderNativeGin},
                                                    gin::kChromeSetting};

  ChromeSetting(const ChromeSetting&) = delete;

  ChromeSetting& operator=(const ChromeSetting&) = delete;

  ChromeSetting(APIRequestHandler* request_handler,
                APIEventHandler* event_handler,
                const APITypeReferenceMap* type_refs,
                const BindingAccessChecker* access_checker,
                const std::string& pref_name,
                const base::Value::Dict& argument_spec);

  ~ChromeSetting() override;

  // Creates a ChromeSetting object for the given property.
  static v8::Local<v8::Object> Create(
      v8::Isolate* isolate,
      const std::string& property_name,
      const base::Value::List* property_values,
      APIRequestHandler* request_handler,
      APIEventHandler* event_handler,
      APITypeReferenceMap* type_refs,
      const BindingAccessChecker* access_checker);

 private:
  // gin::Wrappable:
  gin::ObjectTemplateBuilder GetObjectTemplateBuilder(
      v8::Isolate* isolate) final;

  const char* GetHumanReadableName() const override;

  const gin::WrapperInfo* wrapper_info() const override;

  // JS function handlers:
  void Get(gin::Arguments* arguments);
  void Set(gin::Arguments* arguments);
  void Clear(gin::Arguments* arguments);

  // Returns the onChange event associated with the ChromeSetting.
  v8::Local<v8::Value> GetOnChangeEvent(gin::Arguments* arguments);

  // Common function handling endpoint.
  void HandleFunction(const std::string& function_name,
                      gin::Arguments* arguments);

  raw_ptr<APIRequestHandler, DanglingUntriaged> request_handler_;

  raw_ptr<APIEventHandler, DanglingUntriaged> event_handler_;

  raw_ptr<const APITypeReferenceMap, DanglingUntriaged> type_refs_;

  const raw_ptr<const BindingAccessChecker, DanglingUntriaged> access_checker_;

  // The name of the preference this ChromeSetting is managing.
  std::string pref_name_;

  // The type of argument that calling set() on the ChromeSetting expects (since
  // different settings can take a different type of argument depending on the
  // preference it manages).
  ArgumentSpec argument_spec_;
};

}  // namespace extensions

#endif  // EXTENSIONS_RENDERER_CHROME_SETTING_H_
