/** utf8 helper functions
 * (C) 2018-.. Bram Matthys and the UnrealIRCd team
 * License: GPLv2 or newer
 */

/* This contains functions that came from antimixedutf8
 * and from confusables.txt
 */

#include "unrealircd.h"

ModuleHeader MOD_HEADER
= {
	"utf8functions",
	"1.0.1",
	"UTF8 helper functions",
	"UnrealIRCd Team",
	"unrealircd-6",
};

typedef struct UnicodeBlocks {
	uint32_t start;
	uint32_t end;
	const char *name;
	int score;
} UnicodeBlocks;

typedef struct ConfusablesConversionTable {
	uint32_t from;
	uint32_t to;
} ConfusablesConversionTable;

/* This is the list of all the unicode blocks.
 * Currently based on Unicode 16.0.0 from 2024-02-02.
 * To create this I used the following magic:
 * wget https://unicode.org/Public/UNIDATA/Blocks.txt
 * And then:
 * cat Blocks.txt|\
 * egrep -v '^\#'|\
 * grep -F ..|\
 * sed 's/\.\./;/g'|\
 * sed 's/; /;/g'|\
 * awk -F ';' '{ print "\t{0x" $1 ", 0x" $2 ", \"" $3 "\", 1}," }'|\
 * sed 's/"Emoticons", 1/"Emoticons", 0/g'|\
 * sed 's/"Mathematical Alphanumeric Symbols", 1/"Mathematical Alphanumeric Symbols", 3/g'
 *
 * If blocks are added then you will need to change UNICODE_BLOCK_COUNT
 * which is in include/struct.h
 *
 * Important: don't change this list and then REHASH.
 * Such hot-reloading will cause pages to be off (=WRONG).
 * Only do this with restarts, eg. with a release upgrade.
 *
 * NOTE IF YOU TWEAK ANY OF THE SCORES BELOW:
 * Then also update the sed command from a few lines up :)
 */
UnicodeBlocks unicode_blocks[UNICODE_BLOCK_COUNT] =
{
	{0x0000, 0x007F, "Basic Latin", 1},
	{0x0080, 0x00FF, "Latin-1 Supplement", 1},
	{0x0100, 0x017F, "Latin Extended-A", 1},
	{0x0180, 0x024F, "Latin Extended-B", 1},
	{0x0250, 0x02AF, "IPA Extensions", 1},
	{0x02B0, 0x02FF, "Spacing Modifier Letters", 1},
	{0x0300, 0x036F, "Combining Diacritical Marks", 1},
	{0x0370, 0x03FF, "Greek and Coptic", 1},
	{0x0400, 0x04FF, "Cyrillic", 1},
	{0x0500, 0x052F, "Cyrillic Supplement", 1},
	{0x0530, 0x058F, "Armenian", 1},
	{0x0590, 0x05FF, "Hebrew", 1},
	{0x0600, 0x06FF, "Arabic", 1},
	{0x0700, 0x074F, "Syriac", 1},
	{0x0750, 0x077F, "Arabic Supplement", 1},
	{0x0780, 0x07BF, "Thaana", 1},
	{0x07C0, 0x07FF, "NKo", 1},
	{0x0800, 0x083F, "Samaritan", 1},
	{0x0840, 0x085F, "Mandaic", 1},
	{0x0860, 0x086F, "Syriac Supplement", 1},
	{0x0870, 0x089F, "Arabic Extended-B", 1},
	{0x08A0, 0x08FF, "Arabic Extended-A", 1},
	{0x0900, 0x097F, "Devanagari", 1},
	{0x0980, 0x09FF, "Bengali", 1},
	{0x0A00, 0x0A7F, "Gurmukhi", 1},
	{0x0A80, 0x0AFF, "Gujarati", 1},
	{0x0B00, 0x0B7F, "Oriya", 1},
	{0x0B80, 0x0BFF, "Tamil", 1},
	{0x0C00, 0x0C7F, "Telugu", 1},
	{0x0C80, 0x0CFF, "Kannada", 1},
	{0x0D00, 0x0D7F, "Malayalam", 1},
	{0x0D80, 0x0DFF, "Sinhala", 1},
	{0x0E00, 0x0E7F, "Thai", 1},
	{0x0E80, 0x0EFF, "Lao", 1},
	{0x0F00, 0x0FFF, "Tibetan", 1},
	{0x1000, 0x109F, "Myanmar", 1},
	{0x10A0, 0x10FF, "Georgian", 1},
	{0x1100, 0x11FF, "Hangul Jamo", 1},
	{0x1200, 0x137F, "Ethiopic", 1},
	{0x1380, 0x139F, "Ethiopic Supplement", 1},
	{0x13A0, 0x13FF, "Cherokee", 1},
	{0x1400, 0x167F, "Unified Canadian Aboriginal Syllabics", 1},
	{0x1680, 0x169F, "Ogham", 1},
	{0x16A0, 0x16FF, "Runic", 1},
	{0x1700, 0x171F, "Tagalog", 1},
	{0x1720, 0x173F, "Hanunoo", 1},
	{0x1740, 0x175F, "Buhid", 1},
	{0x1760, 0x177F, "Tagbanwa", 1},
	{0x1780, 0x17FF, "Khmer", 1},
	{0x1800, 0x18AF, "Mongolian", 1},
	{0x18B0, 0x18FF, "Unified Canadian Aboriginal Syllabics Extended", 1},
	{0x1900, 0x194F, "Limbu", 1},
	{0x1950, 0x197F, "Tai Le", 1},
	{0x1980, 0x19DF, "New Tai Lue", 1},
	{0x19E0, 0x19FF, "Khmer Symbols", 1},
	{0x1A00, 0x1A1F, "Buginese", 1},
	{0x1A20, 0x1AAF, "Tai Tham", 1},
	{0x1AB0, 0x1AFF, "Combining Diacritical Marks Extended", 1},
	{0x1B00, 0x1B7F, "Balinese", 1},
	{0x1B80, 0x1BBF, "Sundanese", 1},
	{0x1BC0, 0x1BFF, "Batak", 1},
	{0x1C00, 0x1C4F, "Lepcha", 1},
	{0x1C50, 0x1C7F, "Ol Chiki", 1},
	{0x1C80, 0x1C8F, "Cyrillic Extended-C", 1},
	{0x1C90, 0x1CBF, "Georgian Extended", 1},
	{0x1CC0, 0x1CCF, "Sundanese Supplement", 1},
	{0x1CD0, 0x1CFF, "Vedic Extensions", 1},
	{0x1D00, 0x1D7F, "Phonetic Extensions", 1},
	{0x1D80, 0x1DBF, "Phonetic Extensions Supplement", 1},
	{0x1DC0, 0x1DFF, "Combining Diacritical Marks Supplement", 1},
	{0x1E00, 0x1EFF, "Latin Extended Additional", 1},
	{0x1F00, 0x1FFF, "Greek Extended", 1},
	{0x2000, 0x206F, "General Punctuation", 0},
	{0x2070, 0x209F, "Superscripts and Subscripts", 1},
	{0x20A0, 0x20CF, "Currency Symbols", 1},
	{0x20D0, 0x20FF, "Combining Diacritical Marks for Symbols", 1},
	{0x2100, 0x214F, "Letterlike Symbols", 1},
	{0x2150, 0x218F, "Number Forms", 1},
	{0x2190, 0x21FF, "Arrows", 1},
	{0x2200, 0x22FF, "Mathematical Operators", 1},
	{0x2300, 0x23FF, "Miscellaneous Technical", 1},
	{0x2400, 0x243F, "Control Pictures", 1},
	{0x2440, 0x245F, "Optical Character Recognition", 1},
	{0x2460, 0x24FF, "Enclosed Alphanumerics", 1},
	{0x2500, 0x257F, "Box Drawing", 1},
	{0x2580, 0x259F, "Block Elements", 1},
	{0x25A0, 0x25FF, "Geometric Shapes", 1},
	{0x2600, 0x26FF, "Miscellaneous Symbols", 1},
	{0x2700, 0x27BF, "Dingbats", 1},
	{0x27C0, 0x27EF, "Miscellaneous Mathematical Symbols-A", 1},
	{0x27F0, 0x27FF, "Supplemental Arrows-A", 1},
	{0x2800, 0x28FF, "Braille Patterns", 1},
	{0x2900, 0x297F, "Supplemental Arrows-B", 1},
	{0x2980, 0x29FF, "Miscellaneous Mathematical Symbols-B", 1},
	{0x2A00, 0x2AFF, "Supplemental Mathematical Operators", 1},
	{0x2B00, 0x2BFF, "Miscellaneous Symbols and Arrows", 1},
	{0x2C00, 0x2C5F, "Glagolitic", 1},
	{0x2C60, 0x2C7F, "Latin Extended-C", 1},
	{0x2C80, 0x2CFF, "Coptic", 1},
	{0x2D00, 0x2D2F, "Georgian Supplement", 1},
	{0x2D30, 0x2D7F, "Tifinagh", 1},
	{0x2D80, 0x2DDF, "Ethiopic Extended", 1},
	{0x2DE0, 0x2DFF, "Cyrillic Extended-A", 1},
	{0x2E00, 0x2E7F, "Supplemental Punctuation", 1},
	{0x2E80, 0x2EFF, "CJK Radicals Supplement", 1},
	{0x2F00, 0x2FDF, "Kangxi Radicals", 1},
	{0x2FF0, 0x2FFF, "Ideographic Description Characters", 1},
	{0x3000, 0x303F, "CJK Symbols and Punctuation", 1},
	{0x3040, 0x309F, "Hiragana", 1},
	{0x30A0, 0x30FF, "Katakana", 1},
	{0x3100, 0x312F, "Bopomofo", 1},
	{0x3130, 0x318F, "Hangul Compatibility Jamo", 1},
	{0x3190, 0x319F, "Kanbun", 1},
	{0x31A0, 0x31BF, "Bopomofo Extended", 1},
	{0x31C0, 0x31EF, "CJK Strokes", 1},
	{0x31F0, 0x31FF, "Katakana Phonetic Extensions", 1},
	{0x3200, 0x32FF, "Enclosed CJK Letters and Months", 1},
	{0x3300, 0x33FF, "CJK Compatibility", 1},
	{0x3400, 0x4DBF, "CJK Unified Ideographs Extension A", 1},
	{0x4DC0, 0x4DFF, "Yijing Hexagram Symbols", 1},
	{0x4E00, 0x9FFF, "CJK Unified Ideographs", 1},
	{0xA000, 0xA48F, "Yi Syllables", 1},
	{0xA490, 0xA4CF, "Yi Radicals", 1},
	{0xA4D0, 0xA4FF, "Lisu", 1},
	{0xA500, 0xA63F, "Vai", 1},
	{0xA640, 0xA69F, "Cyrillic Extended-B", 1},
	{0xA6A0, 0xA6FF, "Bamum", 1},
	{0xA700, 0xA71F, "Modifier Tone Letters", 1},
	{0xA720, 0xA7FF, "Latin Extended-D", 1},
	{0xA800, 0xA82F, "Syloti Nagri", 1},
	{0xA830, 0xA83F, "Common Indic Number Forms", 1},
	{0xA840, 0xA87F, "Phags-pa", 1},
	{0xA880, 0xA8DF, "Saurashtra", 1},
	{0xA8E0, 0xA8FF, "Devanagari Extended", 1},
	{0xA900, 0xA92F, "Kayah Li", 1},
	{0xA930, 0xA95F, "Rejang", 1},
	{0xA960, 0xA97F, "Hangul Jamo Extended-A", 1},
	{0xA980, 0xA9DF, "Javanese", 1},
	{0xA9E0, 0xA9FF, "Myanmar Extended-B", 1},
	{0xAA00, 0xAA5F, "Cham", 1},
	{0xAA60, 0xAA7F, "Myanmar Extended-A", 1},
	{0xAA80, 0xAADF, "Tai Viet", 1},
	{0xAAE0, 0xAAFF, "Meetei Mayek Extensions", 1},
	{0xAB00, 0xAB2F, "Ethiopic Extended-A", 1},
	{0xAB30, 0xAB6F, "Latin Extended-E", 1},
	{0xAB70, 0xABBF, "Cherokee Supplement", 1},
	{0xABC0, 0xABFF, "Meetei Mayek", 1},
	{0xAC00, 0xD7AF, "Hangul Syllables", 1},
	{0xD7B0, 0xD7FF, "Hangul Jamo Extended-B", 1},
	{0xD800, 0xDB7F, "High Surrogates", 1},
	{0xDB80, 0xDBFF, "High Private Use Surrogates", 1},
	{0xDC00, 0xDFFF, "Low Surrogates", 1},
	{0xE000, 0xF8FF, "Private Use Area", 1},
	{0xF900, 0xFAFF, "CJK Compatibility Ideographs", 1},
	{0xFB00, 0xFB4F, "Alphabetic Presentation Forms", 1},
	{0xFB50, 0xFDFF, "Arabic Presentation Forms-A", 1},
	{0xFE00, 0xFE0F, "Variation Selectors", 1},
	{0xFE10, 0xFE1F, "Vertical Forms", 1},
	{0xFE20, 0xFE2F, "Combining Half Marks", 1},
	{0xFE30, 0xFE4F, "CJK Compatibility Forms", 1},
	{0xFE50, 0xFE6F, "Small Form Variants", 1},
	{0xFE70, 0xFEFF, "Arabic Presentation Forms-B", 1},
	{0xFF00, 0xFFEF, "Halfwidth and Fullwidth Forms", 1},
	{0xFFF0, 0xFFFF, "Specials", 1},
	{0x10000, 0x1007F, "Linear B Syllabary", 1},
	{0x10080, 0x100FF, "Linear B Ideograms", 1},
	{0x10100, 0x1013F, "Aegean Numbers", 1},
	{0x10140, 0x1018F, "Ancient Greek Numbers", 1},
	{0x10190, 0x101CF, "Ancient Symbols", 1},
	{0x101D0, 0x101FF, "Phaistos Disc", 1},
	{0x10280, 0x1029F, "Lycian", 1},
	{0x102A0, 0x102DF, "Carian", 1},
	{0x102E0, 0x102FF, "Coptic Epact Numbers", 1},
	{0x10300, 0x1032F, "Old Italic", 1},
	{0x10330, 0x1034F, "Gothic", 1},
	{0x10350, 0x1037F, "Old Permic", 1},
	{0x10380, 0x1039F, "Ugaritic", 1},
	{0x103A0, 0x103DF, "Old Persian", 1},
	{0x10400, 0x1044F, "Deseret", 1},
	{0x10450, 0x1047F, "Shavian", 1},
	{0x10480, 0x104AF, "Osmanya", 1},
	{0x104B0, 0x104FF, "Osage", 1},
	{0x10500, 0x1052F, "Elbasan", 1},
	{0x10530, 0x1056F, "Caucasian Albanian", 1},
	{0x10570, 0x105BF, "Vithkuqi", 1},
	{0x105C0, 0x105FF, "Todhri", 1},
	{0x10600, 0x1077F, "Linear A", 1},
	{0x10780, 0x107BF, "Latin Extended-F", 1},
	{0x10800, 0x1083F, "Cypriot Syllabary", 1},
	{0x10840, 0x1085F, "Imperial Aramaic", 1},
	{0x10860, 0x1087F, "Palmyrene", 1},
	{0x10880, 0x108AF, "Nabataean", 1},
	{0x108E0, 0x108FF, "Hatran", 1},
	{0x10900, 0x1091F, "Phoenician", 1},
	{0x10920, 0x1093F, "Lydian", 1},
	{0x10980, 0x1099F, "Meroitic Hieroglyphs", 1},
	{0x109A0, 0x109FF, "Meroitic Cursive", 1},
	{0x10A00, 0x10A5F, "Kharoshthi", 1},
	{0x10A60, 0x10A7F, "Old South Arabian", 1},
	{0x10A80, 0x10A9F, "Old North Arabian", 1},
	{0x10AC0, 0x10AFF, "Manichaean", 1},
	{0x10B00, 0x10B3F, "Avestan", 1},
	{0x10B40, 0x10B5F, "Inscriptional Parthian", 1},
	{0x10B60, 0x10B7F, "Inscriptional Pahlavi", 1},
	{0x10B80, 0x10BAF, "Psalter Pahlavi", 1},
	{0x10C00, 0x10C4F, "Old Turkic", 1},
	{0x10C80, 0x10CFF, "Old Hungarian", 1},
	{0x10D00, 0x10D3F, "Hanifi Rohingya", 1},
	{0x10D40, 0x10D8F, "Garay", 1},
	{0x10E60, 0x10E7F, "Rumi Numeral Symbols", 1},
	{0x10E80, 0x10EBF, "Yezidi", 1},
	{0x10EC0, 0x10EFF, "Arabic Extended-C", 1},
	{0x10F00, 0x10F2F, "Old Sogdian", 1},
	{0x10F30, 0x10F6F, "Sogdian", 1},
	{0x10F70, 0x10FAF, "Old Uyghur", 1},
	{0x10FB0, 0x10FDF, "Chorasmian", 1},
	{0x10FE0, 0x10FFF, "Elymaic", 1},
	{0x11000, 0x1107F, "Brahmi", 1},
	{0x11080, 0x110CF, "Kaithi", 1},
	{0x110D0, 0x110FF, "Sora Sompeng", 1},
	{0x11100, 0x1114F, "Chakma", 1},
	{0x11150, 0x1117F, "Mahajani", 1},
	{0x11180, 0x111DF, "Sharada", 1},
	{0x111E0, 0x111FF, "Sinhala Archaic Numbers", 1},
	{0x11200, 0x1124F, "Khojki", 1},
	{0x11280, 0x112AF, "Multani", 1},
	{0x112B0, 0x112FF, "Khudawadi", 1},
	{0x11300, 0x1137F, "Grantha", 1},
	{0x11380, 0x113FF, "Tulu-Tigalari", 1},
	{0x11400, 0x1147F, "Newa", 1},
	{0x11480, 0x114DF, "Tirhuta", 1},
	{0x11580, 0x115FF, "Siddham", 1},
	{0x11600, 0x1165F, "Modi", 1},
	{0x11660, 0x1167F, "Mongolian Supplement", 1},
	{0x11680, 0x116CF, "Takri", 1},
	{0x116D0, 0x116FF, "Myanmar Extended-C", 1},
	{0x11700, 0x1174F, "Ahom", 1},
	{0x11800, 0x1184F, "Dogra", 1},
	{0x118A0, 0x118FF, "Warang Citi", 1},
	{0x11900, 0x1195F, "Dives Akuru", 1},
	{0x119A0, 0x119FF, "Nandinagari", 1},
	{0x11A00, 0x11A4F, "Zanabazar Square", 1},
	{0x11A50, 0x11AAF, "Soyombo", 1},
	{0x11AB0, 0x11ABF, "Unified Canadian Aboriginal Syllabics Extended-A", 1},
	{0x11AC0, 0x11AFF, "Pau Cin Hau", 1},
	{0x11B00, 0x11B5F, "Devanagari Extended-A", 1},
	{0x11BC0, 0x11BFF, "Sunuwar", 1},
	{0x11C00, 0x11C6F, "Bhaiksuki", 1},
	{0x11C70, 0x11CBF, "Marchen", 1},
	{0x11D00, 0x11D5F, "Masaram Gondi", 1},
	{0x11D60, 0x11DAF, "Gunjala Gondi", 1},
	{0x11EE0, 0x11EFF, "Makasar", 1},
	{0x11F00, 0x11F5F, "Kawi", 1},
	{0x11FB0, 0x11FBF, "Lisu Supplement", 1},
	{0x11FC0, 0x11FFF, "Tamil Supplement", 1},
	{0x12000, 0x123FF, "Cuneiform", 1},
	{0x12400, 0x1247F, "Cuneiform Numbers and Punctuation", 1},
	{0x12480, 0x1254F, "Early Dynastic Cuneiform", 1},
	{0x12F90, 0x12FFF, "Cypro-Minoan", 1},
	{0x13000, 0x1342F, "Egyptian Hieroglyphs", 1},
	{0x13430, 0x1345F, "Egyptian Hieroglyph Format Controls", 1},
	{0x13460, 0x143FF, "Egyptian Hieroglyphs Extended-A", 1},
	{0x14400, 0x1467F, "Anatolian Hieroglyphs", 1},
	{0x16100, 0x1613F, "Gurung Khema", 1},
	{0x16800, 0x16A3F, "Bamum Supplement", 1},
	{0x16A40, 0x16A6F, "Mro", 1},
	{0x16A70, 0x16ACF, "Tangsa", 1},
	{0x16AD0, 0x16AFF, "Bassa Vah", 1},
	{0x16B00, 0x16B8F, "Pahawh Hmong", 1},
	{0x16D40, 0x16D7F, "Kirat Rai", 1},
	{0x16E40, 0x16E9F, "Medefaidrin", 1},
	{0x16F00, 0x16F9F, "Miao", 1},
	{0x16FE0, 0x16FFF, "Ideographic Symbols and Punctuation", 1},
	{0x17000, 0x187FF, "Tangut", 1},
	{0x18800, 0x18AFF, "Tangut Components", 1},
	{0x18B00, 0x18CFF, "Khitan Small Script", 1},
	{0x18D00, 0x18D7F, "Tangut Supplement", 1},
	{0x1AFF0, 0x1AFFF, "Kana Extended-B", 1},
	{0x1B000, 0x1B0FF, "Kana Supplement", 1},
	{0x1B100, 0x1B12F, "Kana Extended-A", 1},
	{0x1B130, 0x1B16F, "Small Kana Extension", 1},
	{0x1B170, 0x1B2FF, "Nushu", 1},
	{0x1BC00, 0x1BC9F, "Duployan", 1},
	{0x1BCA0, 0x1BCAF, "Shorthand Format Controls", 1},
	{0x1CC00, 0x1CEBF, "Symbols for Legacy Computing Supplement", 1},
	{0x1CF00, 0x1CFCF, "Znamenny Musical Notation", 1},
	{0x1D000, 0x1D0FF, "Byzantine Musical Symbols", 1},
	{0x1D100, 0x1D1FF, "Musical Symbols", 1},
	{0x1D200, 0x1D24F, "Ancient Greek Musical Notation", 1},
	{0x1D2C0, 0x1D2DF, "Kaktovik Numerals", 1},
	{0x1D2E0, 0x1D2FF, "Mayan Numerals", 1},
	{0x1D300, 0x1D35F, "Tai Xuan Jing Symbols", 1},
	{0x1D360, 0x1D37F, "Counting Rod Numerals", 1},
	{0x1D400, 0x1D7FF, "Mathematical Alphanumeric Symbols", 3},
	{0x1D800, 0x1DAAF, "Sutton SignWriting", 1},
	{0x1DF00, 0x1DFFF, "Latin Extended-G", 1},
	{0x1E000, 0x1E02F, "Glagolitic Supplement", 1},
	{0x1E030, 0x1E08F, "Cyrillic Extended-D", 1},
	{0x1E100, 0x1E14F, "Nyiakeng Puachue Hmong", 1},
	{0x1E290, 0x1E2BF, "Toto", 1},
	{0x1E2C0, 0x1E2FF, "Wancho", 1},
	{0x1E4D0, 0x1E4FF, "Nag Mundari", 1},
	{0x1E5D0, 0x1E5FF, "Ol Onal", 1},
	{0x1E7E0, 0x1E7FF, "Ethiopic Extended-B", 1},
	{0x1E800, 0x1E8DF, "Mende Kikakui", 1},
	{0x1E900, 0x1E95F, "Adlam", 1},
	{0x1EC70, 0x1ECBF, "Indic Siyaq Numbers", 1},
	{0x1ED00, 0x1ED4F, "Ottoman Siyaq Numbers", 1},
	{0x1EE00, 0x1EEFF, "Arabic Mathematical Alphabetic Symbols", 1},
	{0x1F000, 0x1F02F, "Mahjong Tiles", 1},
	{0x1F030, 0x1F09F, "Domino Tiles", 1},
	{0x1F0A0, 0x1F0FF, "Playing Cards", 1},
	{0x1F100, 0x1F1FF, "Enclosed Alphanumeric Supplement", 1},
	{0x1F200, 0x1F2FF, "Enclosed Ideographic Supplement", 1},
	{0x1F300, 0x1F5FF, "Miscellaneous Symbols and Pictographs", 1},
	{0x1F600, 0x1F64F, "Emoticons", 0},
	{0x1F650, 0x1F67F, "Ornamental Dingbats", 1},
	{0x1F680, 0x1F6FF, "Transport and Map Symbols", 1},
	{0x1F700, 0x1F77F, "Alchemical Symbols", 1},
	{0x1F780, 0x1F7FF, "Geometric Shapes Extended", 1},
	{0x1F800, 0x1F8FF, "Supplemental Arrows-C", 1},
	{0x1F900, 0x1F9FF, "Supplemental Symbols and Pictographs", 1},
	{0x1FA00, 0x1FA6F, "Chess Symbols", 1},
	{0x1FA70, 0x1FAFF, "Symbols and Pictographs Extended-A", 1},
	{0x1FB00, 0x1FBFF, "Symbols for Legacy Computing", 1},
	{0x20000, 0x2A6DF, "CJK Unified Ideographs Extension B", 1},
	{0x2A700, 0x2B73F, "CJK Unified Ideographs Extension C", 1},
	{0x2B740, 0x2B81F, "CJK Unified Ideographs Extension D", 1},
	{0x2B820, 0x2CEAF, "CJK Unified Ideographs Extension E", 1},
	{0x2CEB0, 0x2EBEF, "CJK Unified Ideographs Extension F", 1},
	{0x2EBF0, 0x2EE5F, "CJK Unified Ideographs Extension I", 1},
	{0x2F800, 0x2FA1F, "CJK Compatibility Ideographs Supplement", 1},
	{0x30000, 0x3134F, "CJK Unified Ideographs Extension G", 1},
	{0x31350, 0x323AF, "CJK Unified Ideographs Extension H", 1},
	{0xE0000, 0xE007F, "Tags", 1},
	{0xE0100, 0xE01EF, "Variation Selectors Supplement", 1},
	{0xF0000, 0xFFFFF, "Supplementary Private Use Area-A", 1},
	{0x100000, 0x10FFFF, "Supplementary Private Use Area-B", 1},
};

/* Generated by Syzop */
ConfusablesConversionTable confusables_table[] =
{
	// starting with a0
	{0xa0, 0x20}, 	//   ->  
	{0xa1, 0x69}, 	// ¡ -> i
	{0xa2, 0x63}, 	// ¢ -> c
	{0xa5, 0x59}, 	// ¥ -> Y
	{0xa6, 0x7c}, 	// ¦ -> |
	{0xa8, 0x22}, 	// ¨ -> "
	{0xaa, 0x61}, 	// ª -> a
	{0xac, 0x21}, 	// ¬ -> !
	{0xaf, 0x2d}, 	// ¯ -> -
	{0xb2, 0x32}, 	// ² -> 2
	{0xb3, 0x33}, 	// ³ -> 3
	{0xb4, 0x27}, 	// ´ -> '
	{0xb5, 0x75}, 	// µ -> u
	{0xb6, 0x50}, 	// ¶ -> P
	{0xb7, 0x2a}, 	// · -> *
	{0xb8, 0x2c}, 	// ¸ -> ,
	{0xb9, 0x31}, 	// ¹ -> 1
	{0xba, 0x6f}, 	// º -> o
	{0xbf, 0x3f}, 	// ¿ -> ?
	{0xc0, 0x41}, 	// À -> A
	{0xc1, 0x41}, 	// Á -> A
	{0xc2, 0x41}, 	// Â -> A
	{0xc3, 0x41}, 	// Ã -> A
	{0xc4, 0x41}, 	// Ä -> A
	{0xc5, 0x41}, 	// Å -> A
	{0xc6, 0x41}, 	// Æ -> A
	{0xc7, 0x43}, 	// Ç -> C
	{0xc8, 0x45}, 	// È -> E
	{0xc9, 0x45}, 	// É -> E
	{0xca, 0x45}, 	// Ê -> E
	{0xcb, 0x45}, 	// Ë -> E
	{0xcc, 0x49}, 	// Ì -> I
	{0xcd, 0x49}, 	// Í -> I
	{0xce, 0x49}, 	// Î -> I
	{0xcf, 0x49}, 	// Ï -> I
	{0xd0, 0x44}, 	// Ð -> D
	{0xd1, 0x4e}, 	// Ñ -> N
	{0xd2, 0x4f}, 	// Ò -> O
	{0xd3, 0x4f}, 	// Ó -> O
	{0xd4, 0x4f}, 	// Ô -> O
	{0xd5, 0x4f}, 	// Õ -> O
	{0xd6, 0x4f}, 	// Ö -> O
	{0xd7, 0x78}, 	// × -> x
	{0xd8, 0x4f}, 	// Ø -> O
	{0xd9, 0x55}, 	// Ù -> U
	{0xda, 0x55}, 	// Ú -> U
	{0xdb, 0x55}, 	// Û -> U
	{0xdc, 0x55}, 	// Ü -> U
	{0xdd, 0x59}, 	// Ý -> Y
	{0xe0, 0x61}, 	// à -> a
	{0xe1, 0x61}, 	// á -> a
	{0xe2, 0x61}, 	// â -> a
	{0xe3, 0x61}, 	// ã -> a
	{0xe4, 0x61}, 	// ä -> a
	{0xe5, 0x61}, 	// å -> a
	{0xe6, 0x61}, 	// æ -> a
	{0xe7, 0x63}, 	// ç -> c
	{0xe8, 0x65}, 	// è -> e
	{0xe9, 0x65}, 	// é -> e
	{0xea, 0x65}, 	// ê -> e
	{0xeb, 0x65}, 	// ë -> e
	{0xec, 0x69}, 	// ì -> i
	{0xed, 0x69}, 	// í -> i
	{0xee, 0x69}, 	// î -> i
	{0xef, 0x69}, 	// ï -> i
	{0xf0, 0x64}, 	// ð -> d
	{0xf1, 0x6e}, 	// ñ -> n
	{0xf2, 0x6f}, 	// ò -> o
	{0xf3, 0x6f}, 	// ó -> o
	{0xf4, 0x6f}, 	// ô -> o
	{0xf5, 0x6f}, 	// õ -> o
	{0xf6, 0x6f}, 	// ö -> o
	{0xf7, 0x2f}, 	// ÷ -> /
	{0xf8, 0x6f}, 	// ø -> o
	{0xf9, 0x75}, 	// ù -> u
	{0xfa, 0x75}, 	// ú -> u
	{0xfb, 0x75}, 	// û -> u
	{0xfc, 0x75}, 	// ü -> u
	{0xfd, 0x79}, 	// ý -> y
	{0xff, 0x79}, 	// ÿ -> y
	{0x100, 0x41}, 	// Ā -> A
	{0x101, 0x61}, 	// ā -> a
	{0x102, 0x41}, 	// Ă -> A
	{0x103, 0x61}, 	// ă -> a
	{0x104, 0x41}, 	// Ą -> A
	{0x105, 0x61}, 	// ą -> a
	{0x106, 0x43}, 	// Ć -> C
	{0x107, 0x63}, 	// ć -> c
	{0x108, 0x43}, 	// Ĉ -> C
	{0x109, 0x63}, 	// ĉ -> c
	{0x10a, 0x43}, 	// Ċ -> C
	{0x10b, 0x63}, 	// ċ -> c
	{0x10c, 0x43}, 	// Č -> C
	{0x10d, 0x63}, 	// č -> c
	{0x10e, 0x44}, 	// Ď -> D
	{0x10f, 0x64}, 	// ď -> d
	{0x110, 0x44}, 	// Đ -> D
	{0x111, 0x64}, 	// đ -> d
	{0x112, 0x45}, 	// Ē -> E
	{0x113, 0x65}, 	// ē -> e
	{0x114, 0x45}, 	// Ĕ -> E
	{0x115, 0x65}, 	// ĕ -> e
	{0x116, 0x45}, 	// Ė -> E
	{0x117, 0x65}, 	// ė -> e
	{0x118, 0x45}, 	// Ę -> E
	{0x119, 0x65}, 	// ę -> e
	{0x11a, 0x45}, 	// Ě -> E
	{0x11b, 0x65}, 	// ě -> e
	{0x11c, 0x47}, 	// Ĝ -> G
	{0x11d, 0x67}, 	// ĝ -> g
	{0x11e, 0x47}, 	// Ğ -> G
	{0x11f, 0x67}, 	// ğ -> g
	{0x120, 0x47}, 	// Ġ -> G
	{0x121, 0x67}, 	// ġ -> g
	{0x122, 0x47}, 	// Ģ -> G
	{0x123, 0x67}, 	// ģ -> g
	{0x124, 0x48}, 	// Ĥ -> H
	{0x125, 0x68}, 	// ĥ -> h
	{0x126, 0x48}, 	// Ħ -> H
	{0x127, 0x68}, 	// ħ -> h
	{0x128, 0x49}, 	// Ĩ -> I
	{0x129, 0x69}, 	// ĩ -> i
	{0x12a, 0x49}, 	// Ī -> I
	{0x12b, 0x69}, 	// ī -> i
	{0x12c, 0x49}, 	// Ĭ -> I
	{0x12d, 0x69}, 	// ĭ -> i
	{0x12e, 0x49}, 	// Į -> I
	{0x12f, 0x69}, 	// į -> i
	{0x130, 0x49}, 	// İ -> I
	{0x131, 0x69}, 	// ı -> i
	{0x132, 0x6c}, 	// Ĳ -> l
	{0x133, 0x69}, 	// ĳ -> i
	{0x134, 0x4a}, 	// Ĵ -> J
	{0x135, 0x6a}, 	// ĵ -> j
	{0x136, 0x4b}, 	// Ķ -> K
	{0x137, 0x6b}, 	// ķ -> k
	{0x138, 0x6b}, 	// ĸ -> k
	{0x139, 0x4c}, 	// Ĺ -> L
	{0x13a, 0x6c}, 	// ĺ -> l
	{0x13b, 0x4c}, 	// Ļ -> L
	{0x13c, 0x6c}, 	// ļ -> l
	{0x13d, 0x4c}, 	// Ľ -> L
	{0x13e, 0x6c}, 	// ľ -> l
	{0x13f, 0x6c}, 	// Ŀ -> l
	{0x140, 0x6c}, 	// ŀ -> l
	{0x141, 0x4c}, 	// Ł -> L
	{0x142, 0x6c}, 	// ł -> l
	{0x143, 0x4e}, 	// Ń -> N
	{0x144, 0x6e}, 	// ń -> n
	{0x145, 0x4e}, 	// Ņ -> N
	{0x146, 0x6e}, 	// ņ -> n
	{0x147, 0x4e}, 	// Ň -> N
	{0x148, 0x6e}, 	// ň -> n
	{0x149, 0x6e}, 	// ŉ -> n
	{0x14c, 0x4f}, 	// Ō -> O
	{0x14d, 0x6f}, 	// ō -> o
	{0x14e, 0x4f}, 	// Ŏ -> O
	{0x14f, 0x6f}, 	// ŏ -> o
	{0x150, 0xd6}, 	// Ő -> Ö
	{0x151, 0x6f}, 	// ő -> o
	{0x152, 0x4f}, 	// Œ -> O
	{0x153, 0x6f}, 	// œ -> o
	{0x154, 0x52}, 	// Ŕ -> R
	{0x155, 0x72}, 	// ŕ -> r
	{0x156, 0x52}, 	// Ŗ -> R
	{0x157, 0x72}, 	// ŗ -> r
	{0x158, 0x52}, 	// Ř -> R
	{0x159, 0x72}, 	// ř -> r
	{0x15a, 0x53}, 	// Ś -> S
	{0x15b, 0x73}, 	// ś -> s
	{0x15c, 0x53}, 	// Ŝ -> S
	{0x15d, 0x73}, 	// ŝ -> s
	{0x15e, 0x53}, 	// Ş -> S
	{0x15f, 0x73}, 	// ş -> s
	{0x160, 0x53}, 	// Š -> S
	{0x161, 0x73}, 	// š -> s
	{0x162, 0x54}, 	// Ţ -> T
	{0x163, 0x74}, 	// ţ -> t
	{0x164, 0x54}, 	// Ť -> T
	{0x165, 0x74}, 	// ť -> t
	{0x166, 0x54}, 	// Ŧ -> T
	{0x167, 0x74}, 	// ŧ -> t
	{0x168, 0x55}, 	// Ũ -> U
	{0x169, 0x75}, 	// ũ -> u
	{0x16a, 0x55}, 	// Ū -> U
	{0x16b, 0x75}, 	// ū -> u
	{0x16c, 0x55}, 	// Ŭ -> U
	{0x16d, 0x75}, 	// ŭ -> u
	{0x16e, 0x55}, 	// Ů -> U
	{0x16f, 0x75}, 	// ů -> u
	{0x170, 0x55}, 	// Ű -> U
	{0x171, 0x75}, 	// ű -> u
	{0x172, 0x55}, 	// Ų -> U
	{0x173, 0x75}, 	// ų -> u
	{0x174, 0x57}, 	// Ŵ -> W
	{0x175, 0x77}, 	// ŵ -> w
	{0x176, 0x59}, 	// Ŷ -> Y
	{0x177, 0x79}, 	// ŷ -> y
	{0x178, 0x59}, 	// Ÿ -> Y
	{0x179, 0x5a}, 	// Ź -> Z
	{0x17a, 0x7a}, 	// ź -> z
	{0x17b, 0x5a}, 	// Ż -> Z
	{0x17c, 0x7a}, 	// ż -> z
	{0x17d, 0x5a}, 	// Ž -> Z
	{0x17e, 0x7a}, 	// ž -> z
	{0x17f, 0x66}, 	// ſ -> f
	{0x180, 0x62}, 	// ƀ -> b
	{0x181, 0x42}, 	// Ɓ -> B
	{0x182, 0x62}, 	// Ƃ -> b
	{0x183, 0x62}, 	// ƃ -> b
	{0x184, 0x62}, 	// Ƅ -> b
	{0x185, 0x62}, 	// ƅ -> b
	{0x186, 0x4f}, 	// Ɔ -> O
	{0x187, 0x43}, 	// Ƈ -> C
	{0x188, 0x63}, 	// ƈ -> c
	{0x189, 0x44}, 	// Ɖ -> D
	{0x18a, 0x44}, 	// Ɗ -> D
	{0x18b, 0x44}, 	// Ƌ -> D
	{0x18c, 0x64}, 	// ƌ -> d
	{0x18d, 0x67}, 	// ƍ -> g
	{0x18e, 0x33}, 	// Ǝ -> 3
	{0x18f, 0x40}, 	// Ə -> @
	{0x190, 0x45}, 	// Ɛ -> E
	{0x191, 0x46}, 	// Ƒ -> F
	{0x192, 0x66}, 	// ƒ -> f
	{0x193, 0x47}, 	// Ɠ -> G
	{0x194, 0x47}, 	// Ɣ -> G
	{0x196, 0x6c}, 	// Ɩ -> l
	{0x197, 0x6c}, 	// Ɨ -> l
	{0x198, 0x4b}, 	// Ƙ -> K
	{0x199, 0x6b}, 	// ƙ -> k
	{0x19a, 0x6c}, 	// ƚ -> l
	{0x19b, 0x6c}, 	// ƛ -> l
	{0x19c, 0x57}, 	// Ɯ -> W
	{0x19d, 0x4e}, 	// Ɲ -> N
	{0x19e, 0x6e}, 	// ƞ -> n
	{0x19f, 0x4f}, 	// Ɵ -> O
	{0x1a0, 0x4f}, 	// Ơ -> O
	{0x1a1, 0x6f}, 	// ơ -> o
	{0x1a4, 0x50}, 	// Ƥ -> P
	{0x1a5, 0x70}, 	// ƥ -> p
	{0x1a6, 0x52}, 	// Ʀ -> R
	{0x1a7, 0x32}, 	// Ƨ -> 2
	{0x1a8, 0x32}, 	// ƨ -> 2
	{0x1ab, 0x74}, 	// ƫ -> t
	{0x1ac, 0x54}, 	// Ƭ -> T
	{0x1ad, 0x74}, 	// ƭ -> t
	{0x1ae, 0x54}, 	// Ʈ -> T
	{0x1af, 0x55}, 	// Ư -> U
	{0x1b0, 0x75}, 	// ư -> u
	{0x1b1, 0x59}, 	// Ʊ -> Y
	{0x1b2, 0x56}, 	// Ʋ -> V
	{0x1b3, 0x59}, 	// Ƴ -> Y
	{0x1b4, 0x79}, 	// ƴ -> y
	{0x1b5, 0x5a}, 	// Ƶ -> Z
	{0x1b6, 0x7a}, 	// ƶ -> z
	{0x1bb, 0x32}, 	// ƻ -> 2
	{0x1bc, 0x35}, 	// Ƽ -> 5
	{0x1bd, 0x73}, 	// ƽ -> s
	{0x1bf, 0x77}, 	// ƿ -> w
	{0x1c0, 0x6c}, 	// ǀ -> l
	{0x1c1, 0x6c}, 	// ǁ -> l
	{0x1c3, 0x21}, 	// ǃ -> !
	{0x1c4, 0x44}, 	// Ǆ -> D
	{0x1c5, 0x44}, 	// ǅ -> D
	{0x1c6, 0x64}, 	// ǆ -> d
	{0x1c7, 0x4c}, 	// Ǉ -> L
	{0x1c8, 0x4c}, 	// ǈ -> L
	{0x1c9, 0x6c}, 	// ǉ -> l
	{0x1ca, 0x4e}, 	// Ǌ -> N
	{0x1cb, 0x4e}, 	// ǋ -> N
	{0x1cc, 0x6e}, 	// ǌ -> n
	{0x1cd, 0x41}, 	// Ǎ -> A
	{0x1ce, 0x61}, 	// ǎ -> a
	{0x1cf, 0x49}, 	// Ǐ -> I
	{0x1d0, 0x69}, 	// ǐ -> i
	{0x1d1, 0x4f}, 	// Ǒ -> O
	{0x1d2, 0x6f}, 	// ǒ -> o
	{0x1d3, 0x55}, 	// Ǔ -> U
	{0x1d4, 0x75}, 	// ǔ -> u
	{0x1d5, 0x55}, 	// Ǖ -> U
	{0x1d6, 0x75}, 	// ǖ -> u
	{0x1d7, 0x55}, 	// Ǘ -> U
	{0x1d8, 0x75}, 	// ǘ -> u
	{0x1d9, 0x55}, 	// Ǚ -> U
	{0x1da, 0x75}, 	// ǚ -> u
	{0x1db, 0x55}, 	// Ǜ -> U
	{0x1dc, 0x75}, 	// ǜ -> u
	{0x1dd, 0x40}, 	// ǝ -> @
	{0x1de, 0x41}, 	// Ǟ -> A
	{0x1df, 0x61}, 	// ǟ -> a
	{0x1e0, 0x41}, 	// Ǡ -> A
	{0x1e1, 0x61}, 	// ǡ -> a
	{0x1e4, 0x47}, 	// Ǥ -> G
	{0x1e5, 0x67}, 	// ǥ -> g
	{0x1e6, 0x47}, 	// Ǧ -> G
	{0x1e7, 0x67}, 	// ǧ -> g
	{0x1e8, 0x4b}, 	// Ǩ -> K
	{0x1e9, 0x6b}, 	// ǩ -> k
	{0x1ea, 0x4f}, 	// Ǫ -> O
	{0x1eb, 0x6f}, 	// ǫ -> o
	{0x1ec, 0x4f}, 	// Ǭ -> O
	{0x1ed, 0x6f}, 	// ǭ -> o
	{0x1f0, 0x6a}, 	// ǰ -> j
	{0x1f1, 0x44}, 	// Ǳ -> D
	{0x1f2, 0x44}, 	// ǲ -> D
	{0x1f3, 0x64}, 	// ǳ -> d
	{0x1f4, 0x47}, 	// Ǵ -> G
	{0x1f5, 0x67}, 	// ǵ -> g
	{0x1f7, 0x57}, 	// Ƿ -> W
	{0x1f8, 0x4e}, 	// Ǹ -> N
	{0x1f9, 0x6e}, 	// ǹ -> n
	{0x1fa, 0x41}, 	// Ǻ -> A
	{0x1fb, 0x61}, 	// ǻ -> a
	{0x1fe, 0x4f}, 	// Ǿ -> O
	{0x1ff, 0x6f}, 	// ǿ -> o
	{0x200, 0x41}, 	// Ȁ -> A
	{0x201, 0x61}, 	// ȁ -> a
	{0x202, 0x41}, 	// Ȃ -> A
	{0x203, 0x61}, 	// ȃ -> a
	{0x204, 0x45}, 	// Ȅ -> E
	{0x205, 0x65}, 	// ȅ -> e
	{0x206, 0x45}, 	// Ȇ -> E
	{0x207, 0x65}, 	// ȇ -> e
	{0x208, 0x49}, 	// Ȉ -> I
	{0x209, 0x69}, 	// ȉ -> i
	{0x20a, 0x49}, 	// Ȋ -> I
	{0x20b, 0x69}, 	// ȋ -> i
	{0x20c, 0x4f}, 	// Ȍ -> O
	{0x20d, 0x6f}, 	// ȍ -> o
	{0x20e, 0x4f}, 	// Ȏ -> O
	{0x20f, 0x6f}, 	// ȏ -> o
	{0x210, 0x52}, 	// Ȑ -> R
	{0x211, 0x72}, 	// ȑ -> r
	{0x212, 0x52}, 	// Ȓ -> R
	{0x213, 0x72}, 	// ȓ -> r
	{0x214, 0x55}, 	// Ȕ -> U
	{0x215, 0x75}, 	// ȕ -> u
	{0x216, 0x55}, 	// Ȗ -> U
	{0x217, 0x75}, 	// ȗ -> u
	{0x218, 0x53}, 	// Ș -> S
	{0x219, 0x73}, 	// ș -> s
	{0x21a, 0x54}, 	// Ț -> T
	{0x21b, 0x74}, 	// ț -> t
	{0x21c, 0x59}, 	// Ȝ -> Y
	{0x21d, 0x79}, 	// ȝ -> y
	{0x21e, 0x48}, 	// Ȟ -> H
	{0x21f, 0x68}, 	// ȟ -> h
	{0x220, 0x4e}, 	// Ƞ -> N
	{0x221, 0x64}, 	// ȡ -> d
	{0x224, 0x5a}, 	// Ȥ -> Z
	{0x225, 0x7a}, 	// ȥ -> z
	{0x226, 0xc5}, 	// Ȧ -> Å
	{0x227, 0x62}, 	// ȧ -> b
	{0x228, 0x45}, 	// Ȩ -> E
	{0x229, 0x65}, 	// ȩ -> e
	{0x22a, 0x4f}, 	// Ȫ -> O
	{0x22b, 0x6f}, 	// ȫ -> o
	{0x22c, 0x4f}, 	// Ȭ -> O
	{0x22d, 0x6f}, 	// ȭ -> o
	{0x22e, 0x4f}, 	// Ȯ -> O
	{0x22f, 0x6f}, 	// ȯ -> o
	{0x230, 0x4f}, 	// Ȱ -> O
	{0x231, 0x6f}, 	// ȱ -> o
	{0x232, 0x59}, 	// Ȳ -> Y
	{0x233, 0x79}, 	// ȳ -> y
	{0x234, 0x6c}, 	// ȴ -> l
	{0x235, 0x6e}, 	// ȵ -> n
	{0x236, 0x74}, 	// ȶ -> t
	{0x237, 0x6a}, 	// ȷ -> j
	{0x23a, 0x41}, 	// Ⱥ -> A
	{0x23b, 0x43}, 	// Ȼ -> C
	{0x23c, 0x63}, 	// ȼ -> c
	{0x23d, 0x4c}, 	// Ƚ -> L
	{0x23e, 0x54}, 	// Ⱦ -> T
	{0x23f, 0x73}, 	// ȿ -> s
	{0x240, 0x7a}, 	// ɀ -> z
	{0x243, 0x42}, 	// Ƀ -> B
	{0x244, 0x55}, 	// Ʉ -> U
	{0x245, 0x5e}, 	// Ʌ -> ^
	{0x246, 0x45}, 	// Ɇ -> E
	{0x247, 0x65}, 	// ɇ -> e
	{0x248, 0x4a}, 	// Ɉ -> J
	{0x249, 0x6a}, 	// ɉ -> j
	{0x24a, 0x71}, 	// Ɋ -> q
	{0x24b, 0x71}, 	// ɋ -> q
	{0x24c, 0x52}, 	// Ɍ -> R
	{0x24d, 0x72}, 	// ɍ -> r
	{0x24e, 0x59}, 	// Ɏ -> Y
	{0x24f, 0x79}, 	// ɏ -> y
	{0x250, 0x61}, 	// ɐ -> a
	{0x251, 0x61}, 	// ɑ -> a
	{0x252, 0x61}, 	// ɒ -> a
	{0x253, 0x62}, 	// ɓ -> b
	{0x254, 0x6f}, 	// ɔ -> o
	{0x255, 0x63}, 	// ɕ -> c
	{0x256, 0x64}, 	// ɖ -> d
	{0x257, 0x64}, 	// ɗ -> d
	{0x258, 0x65}, 	// ɘ -> e
	{0x259, 0x40}, 	// ə -> @
	{0x25a, 0x40}, 	// ɚ -> @
	{0x25b, 0x65}, 	// ɛ -> e
	{0x25c, 0x65}, 	// ɜ -> e
	{0x25d, 0x65}, 	// ɝ -> e
	{0x25e, 0x65}, 	// ɞ -> e
	{0x25f, 0x6a}, 	// ɟ -> j
	{0x260, 0x67}, 	// ɠ -> g
	{0x261, 0x67}, 	// ɡ -> g
	{0x262, 0x67}, 	// ɢ -> g
	{0x263, 0x79}, 	// ɣ -> y
	{0x264, 0x75}, 	// ɤ -> u
	{0x265, 0x59}, 	// ɥ -> Y
	{0x266, 0x68}, 	// ɦ -> h
	{0x267, 0x68}, 	// ɧ -> h
	{0x268, 0x69}, 	// ɨ -> i
	{0x269, 0x69}, 	// ɩ -> i
	{0x26a, 0x69}, 	// ɪ -> i
	{0x26b, 0x6c}, 	// ɫ -> l
	{0x26c, 0x6c}, 	// ɬ -> l
	{0x26d, 0x6c}, 	// ɭ -> l
	{0x26e, 0x6c}, 	// ɮ -> l
	{0x26f, 0x77}, 	// ɯ -> w
	{0x270, 0x57}, 	// ɰ -> W
	{0x271, 0x72}, 	// ɱ -> r
	{0x272, 0x6e}, 	// ɲ -> n
	{0x273, 0x6e}, 	// ɳ -> n
	{0x274, 0x6e}, 	// ɴ -> n
	{0x275, 0x6f}, 	// ɵ -> o
	{0x276, 0x6f}, 	// ɶ -> o
	{0x277, 0x4f}, 	// ɷ -> O
	{0x278, 0x46}, 	// ɸ -> F
	{0x279, 0x72}, 	// ɹ -> r
	{0x27a, 0x72}, 	// ɺ -> r
	{0x27b, 0x72}, 	// ɻ -> r
	{0x27c, 0x72}, 	// ɼ -> r
	{0x27d, 0x72}, 	// ɽ -> r
	{0x27e, 0x72}, 	// ɾ -> r
	{0x27f, 0x72}, 	// ɿ -> r
	{0x280, 0x52}, 	// ʀ -> R
	{0x281, 0x52}, 	// ʁ -> R
	{0x282, 0x73}, 	// ʂ -> s
	{0x283, 0x53}, 	// ʃ -> S
	{0x284, 0x6a}, 	// ʄ -> j
	{0x285, 0x53}, 	// ʅ -> S
	{0x286, 0x53}, 	// ʆ -> S
	{0x287, 0x74}, 	// ʇ -> t
	{0x288, 0x74}, 	// ʈ -> t
	{0x289, 0x75}, 	// ʉ -> u
	{0x28a, 0x55}, 	// ʊ -> U
	{0x28b, 0x75}, 	// ʋ -> u
	{0x28c, 0x5e}, 	// ʌ -> ^
	{0x28d, 0x77}, 	// ʍ -> w
	{0x28e, 0x79}, 	// ʎ -> y
	{0x28f, 0x79}, 	// ʏ -> y
	{0x290, 0x7a}, 	// ʐ -> z
	{0x291, 0x7a}, 	// ʑ -> z
	{0x292, 0x5a}, 	// ʒ -> Z
	{0x293, 0x5a}, 	// ʓ -> Z
	{0x294, 0x3f}, 	// ʔ -> ?
	{0x295, 0x3f}, 	// ʕ -> ?
	{0x296, 0x3f}, 	// ʖ -> ?
	{0x297, 0x43}, 	// ʗ -> C
	{0x298, 0x6f}, 	// ʘ -> o
	{0x299, 0x42}, 	// ʙ -> B
	{0x29a, 0x45}, 	// ʚ -> E
	{0x29b, 0x47}, 	// ʛ -> G
	{0x29c, 0x48}, 	// ʜ -> H
	{0x29d, 0x6a}, 	// ʝ -> j
	{0x29e, 0x6b}, 	// ʞ -> k
	{0x29f, 0x4c}, 	// ʟ -> L
	{0x2a0, 0x71}, 	// ʠ -> q
	{0x2a1, 0x3f}, 	// ʡ -> ?
	{0x2a2, 0x3f}, 	// ʢ -> ?
	{0x2a3, 0x64}, 	// ʣ -> d
	{0x2a4, 0x64}, 	// ʤ -> d
	{0x2a5, 0x64}, 	// ʥ -> d
	{0x2a6, 0x74}, 	// ʦ -> t
	{0x2a7, 0x74}, 	// ʧ -> t
	{0x2a8, 0x74}, 	// ʨ -> t
	{0x2a9, 0x66}, 	// ʩ -> f
	{0x2aa, 0x6c}, 	// ʪ -> l
	{0x2ab, 0x6c}, 	// ʫ -> l
	{0x2ae, 0x68}, 	// ʮ -> h
	{0x2af, 0x68}, 	// ʯ -> h
	{0x2b0, 0x68}, 	// ʰ -> h
	{0x2b1, 0x68}, 	// ʱ -> h
	{0x2b2, 0x6a}, 	// ʲ -> j
	{0x2b3, 0x72}, 	// ʳ -> r
	{0x2b4, 0x72}, 	// ʴ -> r
	{0x2b5, 0x72}, 	// ʵ -> r
	{0x2b6, 0x72}, 	// ʶ -> r
	{0x2b7, 0x77}, 	// ʷ -> w
	{0x2b8, 0x79}, 	// ʸ -> y
	{0x2b9, 0x27}, 	// ʹ -> '
	{0x2ba, 0x22}, 	// ʺ -> "
	{0x2bb, 0x60}, 	// ʻ -> `
	{0x2bc, 0x27}, 	// ʼ -> '
	{0x2bd, 0x60}, 	// ʽ -> `
	{0x2be, 0x60}, 	// ʾ -> `
	{0x2bf, 0x27}, 	// ʿ -> '
	{0x2c0, 0x3f}, 	// ˀ -> ?
	{0x2c1, 0x3f}, 	// ˁ -> ?
	{0x2c2, 0x3c}, 	// ˂ -> <
	{0x2c3, 0x3e}, 	// ˃ -> >
	{0x2c4, 0x5e}, 	// ˄ -> ^
	{0x2c5, 0x56}, 	// ˅ -> V
	{0x2c6, 0x5e}, 	// ˆ -> ^
	{0x2c7, 0x56}, 	// ˇ -> V
	{0x2c8, 0x27}, 	// ˈ -> '
	{0x2c9, 0x2d}, 	// ˉ -> -
	{0x2ca, 0x2f}, 	// ˊ -> /
	{0x2cb, 0x5c}, 	// ˋ -> backslash
	{0x2cc, 0x2c}, 	// ˌ -> ,
	{0x2cd, 0x5f}, 	// ˍ -> _
	{0x2ce, 0x5c}, 	// ˎ -> backslash
	{0x2cf, 0x2f}, 	// ˏ -> /
	{0x2d0, 0x3a}, 	// ː -> :
	{0x2d1, 0x2e}, 	// ˑ -> .
	{0x2d2, 0x60}, 	// ˒ -> `
	{0x2d3, 0x27}, 	// ˓ -> '
	{0x2d4, 0x5e}, 	// ˔ -> ^
	{0x2d5, 0x56}, 	// ˕ -> V
	{0x2d6, 0x2b}, 	// ˖ -> +
	{0x2d7, 0x2d}, 	// ˗ -> -
	{0x2d8, 0x56}, 	// ˘ -> V
	{0x2d9, 0x2e}, 	// ˙ -> .
	{0x2da, 0x40}, 	// ˚ -> @
	{0x2db, 0x69}, 	// ˛ -> i
	{0x2dc, 0x7e}, 	// ˜ -> ~
	{0x2dd, 0x22}, 	// ˝ -> "
	{0x2de, 0x52}, 	// ˞ -> R
	{0x2df, 0x58}, 	// ˟ -> X
	{0x2e0, 0x47}, 	// ˠ -> G
	{0x2e1, 0x6c}, 	// ˡ -> l
	{0x2e2, 0x73}, 	// ˢ -> s
	{0x2e3, 0x78}, 	// ˣ -> x
	{0x2e4, 0x3f}, 	// ˤ -> ?
	{0x2ec, 0x56}, 	// ˬ -> V
	{0x2ed, 0x3d}, 	// ˭ -> =
	{0x2ee, 0x22}, 	// ˮ -> "
	{0x363, 0x61}, 	// ͣ -> a
	{0x364, 0x65}, 	// ͤ -> e
	{0x365, 0x69}, 	// ͥ -> i
	{0x366, 0x6f}, 	// ͦ -> o
	{0x367, 0x75}, 	// ͧ -> u
	{0x368, 0x63}, 	// ͨ -> c
	{0x369, 0x64}, 	// ͩ -> d
	{0x36a, 0x68}, 	// ͪ -> h
	{0x36b, 0x6d}, 	// ͫ -> m
	{0x36c, 0x72}, 	// ͬ -> r
	{0x36d, 0x74}, 	// ͭ -> t
	{0x36e, 0x76}, 	// ͮ -> v
	{0x36f, 0x78}, 	// ͯ -> x
	{0x374, 0x27}, 	// ʹ -> '
	{0x375, 0x2c}, 	// ͵ -> ,
	{0x37a, 0x69}, 	// ͺ -> i
	{0x37e, 0x3f}, 	// ; -> ?
	{0x37f, 0x4a}, 	// Ϳ -> J
	{0x386, 0x41}, 	// Ά -> A
	{0x387, 0x3b}, 	// · -> ;
	{0x388, 0x45}, 	// Έ -> E
	{0x389, 0x45}, 	// Ή -> E
	{0x38a, 0x49}, 	// Ί -> I
	{0x38c, 0x4f}, 	// Ό -> O
	{0x38e, 0x55}, 	// Ύ -> U
	{0x38f, 0x4f}, 	// Ώ -> O
	{0x390, 0x49}, 	// ΐ -> I
	{0x391, 0x41}, 	// Α -> A
	{0x392, 0x42}, 	// Β -> B
	{0x393, 0x47}, 	// Γ -> G
	{0x394, 0x44}, 	// Δ -> D
	{0x395, 0x45}, 	// Ε -> E
	{0x396, 0x5a}, 	// Ζ -> Z
	{0x397, 0x48}, 	// Η -> H
	{0x398, 0x4f}, 	// Θ -> O
	{0x399, 0x6c}, 	// Ι -> l
	{0x39a, 0x4b}, 	// Κ -> K
	{0x39b, 0x4c}, 	// Λ -> L
	{0x39c, 0x4d}, 	// Μ -> M
	{0x39d, 0x4e}, 	// Ν -> N
	{0x39f, 0x4f}, 	// Ο -> O
	{0x3a0, 0x50}, 	// Π -> P
	{0x3a1, 0x50}, 	// Ρ -> P
	{0x3a3, 0x53}, 	// Σ -> S
	{0x3a4, 0x54}, 	// Τ -> T
	{0x3a5, 0x59}, 	// Υ -> Y
	{0x3a7, 0x58}, 	// Χ -> X
	{0x3a9, 0x4f}, 	// Ω -> O
	{0x3aa, 0x49}, 	// Ϊ -> I
	{0x3ab, 0x55}, 	// Ϋ -> U
	{0x3ac, 0x61}, 	// ά -> a
	{0x3ad, 0x65}, 	// έ -> e
	{0x3ae, 0x65}, 	// ή -> e
	{0x3af, 0x69}, 	// ί -> i
	{0x3b0, 0x75}, 	// ΰ -> u
	{0x3b1, 0x61}, 	// α -> a
	{0x3b2, 0x62}, 	// β -> b
	{0x3b3, 0x79}, 	// γ -> y
	{0x3b4, 0x64}, 	// δ -> d
	{0x3b5, 0x65}, 	// ε -> e
	{0x3b6, 0x7a}, 	// ζ -> z
	{0x3b7, 0x6e}, 	// η -> n
	{0x3b8, 0x4f}, 	// θ -> O
	{0x3b9, 0x69}, 	// ι -> i
	{0x3ba, 0x6b}, 	// κ -> k
	{0x3bb, 0x6c}, 	// λ -> l
	{0x3bc, 0x75}, 	// μ -> u
	{0x3bd, 0x76}, 	// ν -> v
	{0x3be, 0x78}, 	// ξ -> x
	{0x3bf, 0x6f}, 	// ο -> o
	{0x3c0, 0x6e}, 	// π -> n
	{0x3c1, 0x70}, 	// ρ -> p
	{0x3c2, 0x63}, 	// ς -> c
	{0x3c3, 0x6f}, 	// σ -> o
	{0x3c4, 0x74}, 	// τ -> t
	{0x3c5, 0x75}, 	// υ -> u
	{0x3c9, 0x6f}, 	// ω -> o
	{0x3ca, 0x69}, 	// ϊ -> i
	{0x3cb, 0x75}, 	// ϋ -> u
	{0x3cc, 0x6f}, 	// ό -> o
	{0x3cd, 0x75}, 	// ύ -> u
	{0x3ce, 0x6f}, 	// ώ -> o
	{0x3d0, 0x62}, 	// ϐ -> b
	{0x3d1, 0x4f}, 	// ϑ -> O
	{0x3d2, 0x59}, 	// ϒ -> Y
	{0x3d3, 0x55}, 	// ϓ -> U
	{0x3d4, 0x55}, 	// ϔ -> U
	{0x3d6, 0x70}, 	// ϖ -> p
	{0x3d7, 0x26}, 	// ϗ -> &
	{0x3db, 0x63}, 	// ϛ -> c
	{0x3dc, 0x46}, 	// Ϝ -> F
	{0x3dd, 0x66}, 	// ϝ -> f
	{0x3de, 0x51}, 	// Ϟ -> Q
	{0x3df, 0x71}, 	// ϟ -> q
	{0x3e4, 0x46}, 	// Ϥ -> F
	{0x3e5, 0x66}, 	// ϥ -> f
	{0x3e8, 0x48}, 	// Ϩ -> H
	{0x3e9, 0x68}, 	// ϩ -> h
	{0x3ea, 0x47}, 	// Ϫ -> G
	{0x3eb, 0x67}, 	// ϫ -> g
	{0x3f0, 0x78}, 	// ϰ -> x
	{0x3f1, 0x70}, 	// ϱ -> p
	{0x3f2, 0x63}, 	// ϲ -> c
	{0x3f3, 0x6a}, 	// ϳ -> j
	{0x3f4, 0x4f}, 	// ϴ -> O
	{0x3f8, 0x62}, 	// ϸ -> b
	{0x3f9, 0x43}, 	// Ϲ -> C
	{0x3fa, 0x4d}, 	// Ϻ -> M
	{0x405, 0x53}, 	// Ѕ -> S
	{0x406, 0x6c}, 	// І -> l
	{0x408, 0x4a}, 	// Ј -> J
	{0x40d, 0x49}, 	// Ѝ -> I
	{0x40e, 0x55}, 	// Ў -> U
	{0x410, 0x41}, 	// А -> A
	{0x411, 0x62}, 	// Б -> b
	{0x412, 0x42}, 	// В -> B
	{0x413, 0x47}, 	// Г -> G
	{0x414, 0x44}, 	// Д -> D
	{0x415, 0x45}, 	// Е -> E
	{0x417, 0x5a}, 	// З -> Z
	{0x418, 0x49}, 	// И -> I
	{0x419, 0x49}, 	// Й -> I
	{0x41a, 0x4b}, 	// К -> K
	{0x41b, 0x4c}, 	// Л -> L
	{0x41c, 0x4d}, 	// М -> M
	{0x41d, 0x48}, 	// Н -> H
	{0x41e, 0x4f}, 	// О -> O
	{0x41f, 0x50}, 	// П -> P
	{0x420, 0x50}, 	// Р -> P
	{0x421, 0x43}, 	// С -> C
	{0x422, 0x54}, 	// Т -> T
	{0x423, 0x59}, 	// У -> Y
	{0x424, 0x46}, 	// Ф -> F
	{0x425, 0x58}, 	// Х -> X
	{0x42a, 0x27}, 	// Ъ -> '
	{0x42b, 0x62}, 	// Ы -> b
	{0x42c, 0x62}, 	// Ь -> b
	{0x42d, 0x45}, 	// Э -> E
	{0x42e, 0x6c}, 	// Ю -> l
	{0x430, 0x61}, 	// а -> a
	{0x431, 0x62}, 	// б -> b
	{0x432, 0x76}, 	// в -> v
	{0x433, 0x72}, 	// г -> r
	{0x434, 0x64}, 	// д -> d
	{0x435, 0x65}, 	// е -> e
	{0x437, 0x7a}, 	// з -> z
	{0x438, 0x69}, 	// и -> i
	{0x439, 0x69}, 	// й -> i
	{0x43a, 0x6b}, 	// к -> k
	{0x43b, 0x6c}, 	// л -> l
	{0x43c, 0x6d}, 	// м -> m
	{0x43d, 0x6e}, 	// н -> n
	{0x43e, 0x6f}, 	// о -> o
	{0x43f, 0x6e}, 	// п -> n
	{0x440, 0x70}, 	// р -> p
	{0x441, 0x63}, 	// с -> c
	{0x442, 0x74}, 	// т -> t
	{0x443, 0x79}, 	// у -> y
	{0x444, 0x66}, 	// ф -> f
	{0x445, 0x78}, 	// х -> x
	{0x44a, 0x62}, 	// ъ -> b
	{0x44b, 0x79}, 	// ы -> y
	{0x44c, 0x27}, 	// ь -> '
	{0x44d, 0x65}, 	// э -> e
	{0x450, 0x65}, 	// ѐ -> e
	{0x451, 0x65}, 	// ё -> e
	{0x453, 0x72}, 	// ѓ -> r
	{0x455, 0x73}, 	// ѕ -> s
	{0x456, 0x69}, 	// і -> i
	{0x457, 0x69}, 	// ї -> i
	{0x458, 0x6a}, 	// ј -> j
	{0x45b, 0x68}, 	// ћ -> h
	{0x45d, 0x69}, 	// ѝ -> i
	{0x45e, 0x75}, 	// ў -> u
	{0x460, 0x4f}, 	// Ѡ -> O
	{0x461, 0x77}, 	// ѡ -> w
	{0x462, 0x62}, 	// Ѣ -> b
	{0x463, 0x62}, 	// ѣ -> b
	{0x466, 0x45}, 	// Ѧ -> E
	{0x467, 0x65}, 	// ѧ -> e
	{0x46a, 0x4f}, 	// Ѫ -> O
	{0x46b, 0x6f}, 	// ѫ -> o
	{0x472, 0x4f}, 	// Ѳ -> O
	{0x473, 0x6f}, 	// ѳ -> o
	{0x474, 0x56}, 	// Ѵ -> V
	{0x475, 0x76}, 	// ѵ -> v
	{0x476, 0x59}, 	// Ѷ -> Y
	{0x477, 0x79}, 	// ѷ -> y
	{0x478, 0x75}, 	// Ѹ -> u
	{0x479, 0x75}, 	// ѹ -> u
	{0x47a, 0x4f}, 	// Ѻ -> O
	{0x47b, 0x6f}, 	// ѻ -> o
	{0x47c, 0x4f}, 	// Ѽ -> O
	{0x47d, 0x77}, 	// ѽ -> w
	{0x480, 0x51}, 	// Ҁ -> Q
	{0x481, 0x71}, 	// ҁ -> q
	{0x48c, 0x62}, 	// Ҍ -> b
	{0x48d, 0x62}, 	// ҍ -> b
	{0x491, 0x72}, 	// ґ -> r
	{0x493, 0x72}, 	// ғ -> r
	{0x49a, 0x4b}, 	// Қ -> K
	{0x49b, 0x6b}, 	// қ -> k
	{0x49d, 0x6b}, 	// ҝ -> k
	{0x49e, 0x4b}, 	// Ҟ -> K
	{0x49f, 0x6b}, 	// ҟ -> k
	{0x4a2, 0x48}, 	// Ң -> H
	{0x4aa, 0x43}, 	// Ҫ -> C
	{0x4ab, 0x63}, 	// ҫ -> c
	{0x4ac, 0x54}, 	// Ҭ -> T
	{0x4ae, 0x59}, 	// Ү -> Y
	{0x4af, 0x79}, 	// ү -> y
	{0x4b0, 0x59}, 	// Ұ -> Y
	{0x4b1, 0x79}, 	// ұ -> y
	{0x4b2, 0x58}, 	// Ҳ -> X
	{0x4b3, 0x78}, 	// ҳ -> x
	{0x4ba, 0x48}, 	// Һ -> H
	{0x4bb, 0x68}, 	// һ -> h
	{0x4bd, 0x65}, 	// ҽ -> e
	{0x4bf, 0x65}, 	// ҿ -> e
	{0x4c0, 0x6c}, 	// Ӏ -> l
	{0x4c7, 0x48}, 	// Ӈ -> H
	{0x4c9, 0x48}, 	// Ӊ -> H
	{0x4cd, 0x4d}, 	// Ӎ -> M
	{0x4cf, 0x69}, 	// ӏ -> i
	{0x4d0, 0x61}, 	// Ӑ -> a
	{0x4d1, 0x61}, 	// ӑ -> a
	{0x4d2, 0x41}, 	// Ӓ -> A
	{0x4d3, 0x61}, 	// ӓ -> a
	{0x4d4, 0x41}, 	// Ӕ -> A
	{0x4d5, 0x61}, 	// ӕ -> a
	{0x4d7, 0x65}, 	// ӗ -> e
	{0x4d8, 0x40}, 	// Ә -> @
	{0x4d9, 0x40}, 	// ә -> @
	{0x4da, 0x40}, 	// Ӛ -> @
	{0x4db, 0x40}, 	// ӛ -> @
	{0x4de, 0x5a}, 	// Ӟ -> Z
	{0x4df, 0x7a}, 	// ӟ -> z
	{0x4e2, 0x49}, 	// Ӣ -> I
	{0x4e3, 0x69}, 	// ӣ -> i
	{0x4e4, 0x49}, 	// Ӥ -> I
	{0x4e5, 0x69}, 	// ӥ -> i
	{0x4e6, 0x4f}, 	// Ӧ -> O
	{0x4e7, 0x6f}, 	// ӧ -> o
	{0x4e8, 0x4f}, 	// Ө -> O
	{0x4e9, 0x6f}, 	// ө -> o
	{0x4ea, 0x4f}, 	// Ӫ -> O
	{0x4eb, 0x6f}, 	// ӫ -> o
	{0x4ec, 0x45}, 	// Ӭ -> E
	{0x4ed, 0x65}, 	// ӭ -> e
	{0x4ee, 0x55}, 	// Ӯ -> U
	{0x4ef, 0x79}, 	// ӯ -> y
	{0x4f0, 0x55}, 	// Ӱ -> U
	{0x4f1, 0x75}, 	// ӱ -> u
	{0x4f2, 0x55}, 	// Ӳ -> U
	{0x4f3, 0x75}, 	// ӳ -> u
	{0x4f8, 0x59}, 	// Ӹ -> Y
	{0x4f9, 0x79}, 	// ӹ -> y
	{0x501, 0x64}, 	// ԁ -> d
	{0x50c, 0x47}, 	// Ԍ -> G
	{0x51b, 0x71}, 	// ԛ -> q
	{0x51c, 0x57}, 	// Ԝ -> W
	{0x51d, 0x77}, 	// ԝ -> w
	{0x531, 0x41}, 	// Ա -> A
	{0x532, 0x42}, 	// Բ -> B
	{0x533, 0x47}, 	// Գ -> G
	{0x534, 0x44}, 	// Դ -> D
	{0x535, 0x45}, 	// Ե -> E
	{0x536, 0x5a}, 	// Զ -> Z
	{0x537, 0x74}, 	// Է -> t
	{0x538, 0x45}, 	// Ը -> E
	{0x53b, 0x49}, 	// Ի -> I
	{0x53c, 0x4c}, 	// Լ -> L
	{0x53f, 0x4b}, 	// Կ -> K
	{0x540, 0x48}, 	// Հ -> H
	{0x544, 0x4d}, 	// Մ -> M
	{0x545, 0x59}, 	// Յ -> Y
	{0x546, 0x4e}, 	// Ն -> N
	{0x548, 0x4f}, 	// Ո -> O
	{0x54a, 0x50}, 	// Պ -> P
	{0x54b, 0x4a}, 	// Ջ -> J
	{0x54d, 0x55}, 	// Ս -> U
	{0x54e, 0x56}, 	// Վ -> V
	{0x54f, 0x53}, 	// Տ -> S
	{0x550, 0x52}, 	// Ր -> R
	{0x552, 0x57}, 	// Ւ -> W
	{0x555, 0x4f}, 	// Օ -> O
	{0x556, 0x46}, 	// Ֆ -> F
	{0x559, 0x3c}, 	// ՙ -> <
	{0x55a, 0x27}, 	// ՚ -> '
	{0x55b, 0x2f}, 	// ՛ -> /
	{0x55c, 0x21}, 	// ՜ -> !
	{0x55d, 0x2c}, 	// ՝ -> ,
	{0x55e, 0x3f}, 	// ՞ -> ?
	{0x55f, 0x2e}, 	// ՟ -> .
	{0x561, 0x77}, 	// ա -> w
	{0x562, 0x62}, 	// բ -> b
	{0x563, 0x71}, 	// գ -> q
	{0x564, 0x64}, 	// դ -> d
	{0x565, 0x74}, 	// ե -> t
	{0x566, 0x71}, 	// զ -> q
	{0x567, 0x65}, 	// է -> e
	{0x568, 0x65}, 	// ը -> e
	{0x56b, 0x68}, 	// ի -> h
	{0x56c, 0x6c}, 	// լ -> l
	{0x56e, 0x6f}, 	// ծ -> o
	{0x56f, 0x6b}, 	// կ -> k
	{0x570, 0x68}, 	// հ -> h
	{0x574, 0x6d}, 	// մ -> m
	{0x575, 0x6a}, 	// յ -> j
	{0x576, 0x6e}, 	// ն -> n
	{0x578, 0x6e}, 	// ո -> n
	{0x57a, 0x70}, 	// պ -> p
	{0x57b, 0x6a}, 	// ջ -> j
	{0x57c, 0x6e}, 	// ռ -> n
	{0x57d, 0x75}, 	// ս -> u
	{0x57e, 0x76}, 	// վ -> v
	{0x57f, 0x74}, 	// տ -> t
	{0x580, 0x6e}, 	// ր -> n
	{0x581, 0x67}, 	// ց -> g
	{0x582, 0x77}, 	// ւ -> w
	{0x584, 0x66}, 	// ք -> f
	{0x585, 0x6f}, 	// օ -> o
	{0x586, 0x66}, 	// ֆ -> f
	{0x589, 0x3a}, 	// ։ -> :
	{0x58a, 0x2d}, 	// ֊ -> -
	{0x5b1, 0x65}, 	// ֱ -> e
	{0x5b2, 0x61}, 	// ֲ -> a
	{0x5b3, 0x6f}, 	// ֳ -> o
	{0x5b4, 0x69}, 	// ִ -> i
	{0x5b5, 0x65}, 	// ֵ -> e
	{0x5b6, 0x65}, 	// ֶ -> e
	{0x5b7, 0x61}, 	// ַ -> a
	{0x5b8, 0x61}, 	// ָ -> a
	{0x5b9, 0x6f}, 	// ֹ -> o
	{0x5ba, 0x6f}, 	// ֺ -> o
	{0x5bb, 0x75}, 	// ֻ -> u
	{0x5be, 0x2d}, 	// ־ -> -
	{0x5c0, 0x6c}, 	// ׀ -> l
	{0x5c3, 0x2e}, 	// ׃ -> .
	{0x5c6, 0x6e}, 	// ׆ -> n
	{0x5c7, 0x6f}, 	// ׇ -> o
	{0x5d0, 0x41}, 	// א -> A
	{0x5d1, 0x62}, 	// ב -> b
	{0x5d2, 0x67}, 	// ג -> g
	{0x5d3, 0x64}, 	// ד -> d
	{0x5d4, 0x68}, 	// ה -> h
	{0x5d5, 0x6c}, 	// ו -> l
	{0x5d6, 0x7a}, 	// ז -> z
	{0x5d7, 0x48}, 	// ח -> H
	{0x5d8, 0x76}, 	// ט -> v
	{0x5d9, 0x79}, 	// י -> y
	{0x5dc, 0x6c}, 	// ל -> l
	{0x5dd, 0x6d}, 	// ם -> m
	{0x5de, 0x6d}, 	// מ -> m
	{0x5df, 0x6c}, 	// ן -> l
	{0x5e0, 0x6e}, 	// נ -> n
	{0x5e1, 0x6f}, 	// ס -> o
	{0x5e2, 0x60}, 	// ע -> `
	{0x5e3, 0x70}, 	// ף -> p
	{0x5e4, 0x70}, 	// פ -> p
	{0x5e7, 0x6b}, 	// ק -> k
	{0x5e8, 0x72}, 	// ר -> r
	{0x5ea, 0x74}, 	// ת -> t
	{0x5f0, 0x6c}, 	// װ -> l
	{0x5f1, 0x6c}, 	// ױ -> l
	{0x5f3, 0x27}, 	// ׳ -> '
	{0x5f4, 0x22}, 	// ״ -> "
	{0x60c, 0x2c}, 	// ، -> ,
	{0x61b, 0x3b}, 	// ؛ -> ;
	{0x61f, 0x3f}, 	// ؟ -> ?
	{0x622, 0x61}, 	// آ -> a
	{0x623, 0x6c}, 	// أ -> l
	{0x625, 0x6c}, 	// إ -> l
	{0x627, 0x6c}, 	// ا -> l
	{0x628, 0x62}, 	// ب -> b
	{0x629, 0x40}, 	// ة -> @
	{0x62a, 0x74}, 	// ت -> t
	{0x62c, 0x6a}, 	// ج -> j
	{0x62d, 0x48}, 	// ح -> H
	{0x62f, 0x64}, 	// د -> d
	{0x631, 0x72}, 	// ر -> r
	{0x632, 0x7a}, 	// ز -> z
	{0x633, 0x73}, 	// س -> s
	{0x635, 0x53}, 	// ص -> S
	{0x636, 0x44}, 	// ض -> D
	{0x637, 0x54}, 	// ط -> T
	{0x638, 0x5a}, 	// ظ -> Z
	{0x639, 0x60}, 	// ع -> `
	{0x63a, 0x47}, 	// غ -> G
	{0x641, 0x66}, 	// ف -> f
	{0x642, 0x71}, 	// ق -> q
	{0x643, 0x6b}, 	// ك -> k
	{0x644, 0x6c}, 	// ل -> l
	{0x645, 0x6d}, 	// م -> m
	{0x646, 0x6e}, 	// ن -> n
	{0x647, 0x6f}, 	// ه -> o
	{0x648, 0x77}, 	// و -> w
	{0x649, 0x7e}, 	// ى -> ~
	{0x64a, 0x79}, 	// ي -> y
	{0x64e, 0x61}, 	// َ -> a
	{0x64f, 0x75}, 	// ُ -> u
	{0x650, 0x69}, 	// ِ -> i
	{0x651, 0x57}, 	// ّ -> W
	{0x654, 0x27}, 	// ٔ -> '
	{0x655, 0x27}, 	// ٕ -> '
	{0x660, 0x30}, 	// ٠ -> 0
	{0x661, 0x6c}, 	// ١ -> l
	{0x662, 0x32}, 	// ٢ -> 2
	{0x663, 0x33}, 	// ٣ -> 3
	{0x664, 0x34}, 	// ٤ -> 4
	{0x665, 0x6f}, 	// ٥ -> o
	{0x666, 0x36}, 	// ٦ -> 6
	{0x667, 0x56}, 	// ٧ -> V
	{0x668, 0x38}, 	// ٨ -> 8
	{0x669, 0x39}, 	// ٩ -> 9
	{0x66a, 0x25}, 	// ٪ -> %
	{0x66b, 0x2e}, 	// ٫ -> .
	{0x66c, 0x2c}, 	// ٬ -> ,
	{0x66d, 0x2a}, 	// ٭ -> *
	{0x671, 0x27}, 	// ٱ -> '
	{0x672, 0x6c}, 	// ٲ -> l
	{0x673, 0x6c}, 	// ٳ -> l
	{0x675, 0x6c}, 	// ٵ -> l
	{0x67b, 0x62}, 	// ٻ -> b
	{0x67c, 0x74}, 	// ټ -> t
	{0x67d, 0x54}, 	// ٽ -> T
	{0x67e, 0x70}, 	// پ -> p
	{0x682, 0x48}, 	// ڂ -> H
	{0x685, 0x48}, 	// څ -> H
	{0x689, 0x44}, 	// ډ -> D
	{0x68a, 0x44}, 	// ڊ -> D
	{0x68e, 0x64}, 	// ڎ -> d
	{0x68f, 0x44}, 	// ڏ -> D
	{0x690, 0x44}, 	// ڐ -> D
	{0x692, 0x52}, 	// ڒ -> R
	{0x693, 0x52}, 	// ړ -> R
	{0x694, 0x52}, 	// ڔ -> R
	{0x695, 0x52}, 	// ڕ -> R
	{0x696, 0x52}, 	// ږ -> R
	{0x697, 0x52}, 	// ڗ -> R
	{0x698, 0x6a}, 	// ژ -> j
	{0x699, 0x52}, 	// ڙ -> R
	{0x69a, 0x53}, 	// ښ -> S
	{0x69b, 0x53}, 	// ڛ -> S
	{0x69c, 0x53}, 	// ڜ -> S
	{0x69d, 0x53}, 	// ڝ -> S
	{0x69e, 0x53}, 	// ڞ -> S
	{0x69f, 0x54}, 	// ڟ -> T
	{0x6a1, 0x46}, 	// ڡ -> F
	{0x6a2, 0x46}, 	// ڢ -> F
	{0x6a3, 0x46}, 	// ڣ -> F
	{0x6a4, 0x76}, 	// ڤ -> v
	{0x6a5, 0x66}, 	// ڥ -> f
	{0x6a7, 0x51}, 	// ڧ -> Q
	{0x6a8, 0x51}, 	// ڨ -> Q
	{0x6aa, 0x6b}, 	// ڪ -> k
	{0x6ab, 0x4b}, 	// ګ -> K
	{0x6ac, 0x4b}, 	// ڬ -> K
	{0x6ae, 0x4b}, 	// ڮ -> K
	{0x6af, 0x67}, 	// گ -> g
	{0x6b0, 0x47}, 	// ڰ -> G
	{0x6b1, 0x4e}, 	// ڱ -> N
	{0x6b2, 0x47}, 	// ڲ -> G
	{0x6b3, 0x47}, 	// ڳ -> G
	{0x6b4, 0x47}, 	// ڴ -> G
	{0x6b5, 0x4c}, 	// ڵ -> L
	{0x6b6, 0x4c}, 	// ڶ -> L
	{0x6b7, 0x4c}, 	// ڷ -> L
	{0x6b8, 0x4c}, 	// ڸ -> L
	{0x6b9, 0x4e}, 	// ڹ -> N
	{0x6ba, 0x4e}, 	// ں -> N
	{0x6bb, 0x4e}, 	// ڻ -> N
	{0x6bc, 0x4e}, 	// ڼ -> N
	{0x6bd, 0x4e}, 	// ڽ -> N
	{0x6be, 0x6f}, 	// ھ -> o
	{0x6c1, 0x6f}, 	// ہ -> o
	{0x6c2, 0x48}, 	// ۂ -> H
	{0x6c3, 0x40}, 	// ۃ -> @
	{0x6c4, 0x57}, 	// ۄ -> W
	{0x6c7, 0x75}, 	// ۇ -> u
	{0x6ca, 0x57}, 	// ۊ -> W
	{0x6cb, 0x76}, 	// ۋ -> v
	{0x6cc, 0x79}, 	// ی -> y
	{0x6cd, 0x59}, 	// ۍ -> Y
	{0x6ce, 0x59}, 	// ێ -> Y
	{0x6cf, 0x57}, 	// ۏ -> W
	{0x6d2, 0x79}, 	// ے -> y
	{0x6d4, 0x2e}, 	// ۔ -> .
	{0x6d5, 0x6f}, 	// ە -> o
	{0x6dd, 0x40}, 	// ۝ -> @
	{0x6de, 0x23}, 	// ۞ -> #
	{0x6e9, 0x5e}, 	// ۩ -> ^
	{0x6f0, 0x30}, 	// ۰ -> 0
	{0x6f1, 0x6c}, 	// ۱ -> l
	{0x6f2, 0x32}, 	// ۲ -> 2
	{0x6f3, 0x33}, 	// ۳ -> 3
	{0x6f4, 0x34}, 	// ۴ -> 4
	{0x6f5, 0x6f}, 	// ۵ -> o
	{0x6f6, 0x36}, 	// ۶ -> 6
	{0x6f7, 0x56}, 	// ۷ -> V
	{0x6f8, 0x38}, 	// ۸ -> 8
	{0x6f9, 0x39}, 	// ۹ -> 9
	{0x6fb, 0x44}, 	// ۻ -> D
	{0x6fd, 0x26}, 	// ۽ -> &
	{0x6ff, 0x6f}, 	// ۿ -> o
	{0x701, 0x2f}, 	// ܁ -> /
	{0x702, 0x2c}, 	// ܂ -> ,
	{0x703, 0x21}, 	// ܃ -> !
	{0x704, 0x21}, 	// ܄ -> !
	{0x705, 0x2d}, 	// ܅ -> -
	{0x706, 0x2c}, 	// ܆ -> ,
	{0x707, 0x2c}, 	// ܇ -> ,
	{0x708, 0x3b}, 	// ܈ -> ;
	{0x709, 0x3f}, 	// ܉ -> ?
	{0x70a, 0x7e}, 	// ܊ -> ~
	{0x70b, 0x7b}, 	// ܋ -> {
	{0x70c, 0x7d}, 	// ܌ -> }
	{0x70d, 0x2a}, 	// ܍ -> *
	{0x710, 0x27}, 	// ܐ -> '
	{0x712, 0x62}, 	// ܒ -> b
	{0x713, 0x67}, 	// ܓ -> g
	{0x714, 0x67}, 	// ܔ -> g
	{0x715, 0x64}, 	// ܕ -> d
	{0x716, 0x64}, 	// ܖ -> d
	{0x717, 0x68}, 	// ܗ -> h
	{0x718, 0x77}, 	// ܘ -> w
	{0x719, 0x7a}, 	// ܙ -> z
	{0x71a, 0x48}, 	// ܚ -> H
	{0x71b, 0x74}, 	// ܛ -> t
	{0x71c, 0x74}, 	// ܜ -> t
	{0x71d, 0x79}, 	// ܝ -> y
	{0x71f, 0x6b}, 	// ܟ -> k
	{0x720, 0x6c}, 	// ܠ -> l
	{0x721, 0x6d}, 	// ܡ -> m
	{0x722, 0x6e}, 	// ܢ -> n
	{0x723, 0x73}, 	// ܣ -> s
	{0x724, 0x73}, 	// ܤ -> s
	{0x725, 0x60}, 	// ܥ -> `
	{0x726, 0x70}, 	// ܦ -> p
	{0x727, 0x70}, 	// ܧ -> p
	{0x728, 0x53}, 	// ܨ -> S
	{0x729, 0x71}, 	// ܩ -> q
	{0x72a, 0x72}, 	// ܪ -> r
	{0x72c, 0x74}, 	// ܬ -> t
	{0x730, 0x61}, 	// ܰ -> a
	{0x731, 0x61}, 	// ܱ -> a
	{0x732, 0x61}, 	// ܲ -> a
	{0x733, 0x41}, 	// ܳ -> A
	{0x734, 0x41}, 	// ܴ -> A
	{0x735, 0x41}, 	// ܵ -> A
	{0x736, 0x65}, 	// ܶ -> e
	{0x737, 0x65}, 	// ܷ -> e
	{0x738, 0x65}, 	// ܸ -> e
	{0x739, 0x45}, 	// ܹ -> E
	{0x73a, 0x69}, 	// ܺ -> i
	{0x73b, 0x69}, 	// ܻ -> i
	{0x73c, 0x75}, 	// ܼ -> u
	{0x73d, 0x75}, 	// ܽ -> u
	{0x73e, 0x75}, 	// ܾ -> u
	{0x73f, 0x6f}, 	// ܿ -> o
	{0x741, 0x60}, 	// ݁ -> `
	{0x742, 0x27}, 	// ݂ -> '
	{0x745, 0x58}, 	// ݅ -> X
	{0x746, 0x51}, 	// ݆ -> Q
	{0x747, 0x40}, 	// ݇ -> @
	{0x748, 0x40}, 	// ݈ -> @
	{0x749, 0x7c}, 	// ݉ -> |
	{0x74a, 0x2b}, 	// ݊ -> +
	{0x780, 0x68}, 	// ހ -> h
	{0x782, 0x6e}, 	// ނ -> n
	{0x783, 0x72}, 	// ރ -> r
	{0x784, 0x62}, 	// ބ -> b
	{0x785, 0x4c}, 	// ޅ -> L
	{0x786, 0x6b}, 	// ކ -> k
	{0x787, 0x27}, 	// އ -> '
	{0x788, 0x76}, 	// ވ -> v
	{0x789, 0x6d}, 	// މ -> m
	{0x78a, 0x66}, 	// ފ -> f
	{0x78d, 0x6c}, 	// ލ -> l
	{0x78e, 0x67}, 	// ގ -> g
	{0x790, 0x73}, 	// ސ -> s
	{0x791, 0x64}, 	// ޑ -> d
	{0x792, 0x7a}, 	// ޒ -> z
	{0x793, 0x74}, 	// ޓ -> t
	{0x794, 0x79}, 	// ޔ -> y
	{0x795, 0x70}, 	// ޕ -> p
	{0x796, 0x6a}, 	// ޖ -> j
	{0x79c, 0x7a}, 	// ޜ -> z
	{0x79e, 0x73}, 	// ޞ -> s
	{0x79f, 0x64}, 	// ޟ -> d
	{0x7a0, 0x74}, 	// ޠ -> t
	{0x7a1, 0x7a}, 	// ޡ -> z
	{0x7a2, 0x60}, 	// ޢ -> `
	{0x7a4, 0x71}, 	// ޤ -> q
	{0x7a5, 0x77}, 	// ޥ -> w
	{0x7a6, 0x61}, 	// ަ -> a
	{0x7a8, 0x69}, 	// ި -> i
	{0x7aa, 0x75}, 	// ު -> u
	{0x7ac, 0x65}, 	// ެ -> e
	{0x7ae, 0x6f}, 	// ޮ -> o
	{0x7c0, 0x4f}, 	// ߀ -> O
	{0x7ca, 0x6c}, 	// ߊ -> l
	{0x901, 0x4e}, 	// ँ -> N
	{0x902, 0x4e}, 	// ं -> N
	{0x903, 0x48}, 	// ः -> H
	{0x905, 0x61}, 	// अ -> a
	{0x907, 0x69}, 	// इ -> i
	{0x909, 0x75}, 	// उ -> u
	{0x90b, 0x52}, 	// ऋ -> R
	{0x90c, 0x4c}, 	// ऌ -> L
	{0x90e, 0x65}, 	// ऎ -> e
	{0x90f, 0x65}, 	// ए -> e
	{0x912, 0x6f}, 	// ऒ -> o
	{0x913, 0x6f}, 	// ओ -> o
	{0x915, 0x6b}, 	// क -> k
	{0x917, 0x67}, 	// ग -> g
	{0x91a, 0x63}, 	// च -> c
	{0x91c, 0x6a}, 	// ज -> j
	{0x924, 0x74}, 	// त -> t
	{0x926, 0x64}, 	// द -> d
	{0x928, 0x6e}, 	// न -> n
	{0x92a, 0x70}, 	// प -> p
	{0x92c, 0x62}, 	// ब -> b
	{0x92e, 0x6d}, 	// म -> m
	{0x92f, 0x79}, 	// य -> y
	{0x930, 0x72}, 	// र -> r
	{0x932, 0x6c}, 	// ल -> l
	{0x933, 0x6c}, 	// ळ -> l
	{0x935, 0x76}, 	// व -> v
	{0x938, 0x73}, 	// स -> s
	{0x939, 0x68}, 	// ह -> h
	{0x93c, 0x27}, 	// ़ -> '
	{0x93d, 0x27}, 	// ऽ -> '
	{0x93f, 0x69}, 	// ि -> i
	{0x941, 0x75}, 	// ु -> u
	{0x943, 0x52}, 	// ृ -> R
	{0x946, 0x65}, 	// ॆ -> e
	{0x947, 0x65}, 	// े -> e
	{0x94a, 0x6f}, 	// ॊ -> o
	{0x94b, 0x6f}, 	// ो -> o
	{0x951, 0x27}, 	// ॑ -> '
	{0x952, 0x27}, 	// ॒ -> '
	{0x953, 0x60}, 	// ॓ -> `
	{0x954, 0x27}, 	// ॔ -> '
	{0x958, 0x71}, 	// क़ -> q
	{0x95b, 0x7a}, 	// ज़ -> z
	{0x95e, 0x66}, 	// फ़ -> f
	{0x962, 0x4c}, 	// ॢ -> L
	{0x966, 0x6f}, 	// ० -> o
	{0x967, 0x31}, 	// १ -> 1
	{0x968, 0x32}, 	// २ -> 2
	{0x969, 0x33}, 	// ३ -> 3
	{0x96a, 0x34}, 	// ४ -> 4
	{0x96b, 0x35}, 	// ५ -> 5
	{0x96c, 0x36}, 	// ६ -> 6
	{0x96d, 0x37}, 	// ७ -> 7
	{0x96e, 0x38}, 	// ८ -> 8
	{0x96f, 0x39}, 	// ९ -> 9
	{0x970, 0x2e}, 	// ॰ -> .
	{0x981, 0x4e}, 	// ঁ -> N
	{0x982, 0x4e}, 	// ং -> N
	{0x983, 0x48}, 	// ঃ -> H
	{0x985, 0x61}, 	// অ -> a
	{0x987, 0x69}, 	// ই -> i
	{0x989, 0x75}, 	// উ -> u
	{0x98b, 0x52}, 	// ঋ -> R
	{0x98f, 0x65}, 	// এ -> e
	{0x993, 0x6f}, 	// ও -> o
	{0x995, 0x6b}, 	// ক -> k
	{0x997, 0x67}, 	// গ -> g
	{0x99a, 0x63}, 	// চ -> c
	{0x99c, 0x6a}, 	// জ -> j
	{0x9a4, 0x74}, 	// ত -> t
	{0x9a6, 0x64}, 	// দ -> d
	{0x9a8, 0x6e}, 	// ন -> n
	{0x9aa, 0x70}, 	// প -> p
	{0x9ac, 0x62}, 	// ব -> b
	{0x9ae, 0x6d}, 	// ম -> m
	{0x9af, 0x79}, 	// য -> y
	{0x9b0, 0x72}, 	// র -> r
	{0x9b2, 0x6c}, 	// ল -> l
	{0x9b8, 0x73}, 	// স -> s
	{0x9b9, 0x68}, 	// হ -> h
	{0x9bc, 0x27}, 	// ় -> '
	{0x9bf, 0x69}, 	// ি -> i
	{0x9c1, 0x75}, 	// ু -> u
	{0x9c3, 0x52}, 	// ৃ -> R
	{0x9c7, 0x65}, 	// ে -> e
	{0x9cb, 0x6f}, 	// ো -> o
	{0x9d7, 0x2b}, 	// ৗ -> +
	{0x9e2, 0x4c}, 	// ৢ -> L
	{0x9e6, 0x4f}, 	// ০ -> O
	{0x9e7, 0x31}, 	// ১ -> 1
	{0x9e8, 0x32}, 	// ২ -> 2
	{0x9e9, 0x33}, 	// ৩ -> 3
	{0x9ea, 0x34}, 	// ৪ -> 4
	{0x9eb, 0x35}, 	// ৫ -> 5
	{0x9ec, 0x36}, 	// ৬ -> 6
	{0x9ed, 0x37}, 	// ৭ -> 7
	{0x9ee, 0x38}, 	// ৮ -> 8
	{0x9ef, 0x39}, 	// ৯ -> 9
	{0xa02, 0x4e}, 	// ਂ -> N
	{0xa05, 0x61}, 	// ਅ -> a
	{0xa07, 0x69}, 	// ਇ -> i
	{0xa09, 0x75}, 	// ਉ -> u
	{0xa15, 0x6b}, 	// ਕ -> k
	{0xa17, 0x67}, 	// ਗ -> g
	{0xa1a, 0x63}, 	// ਚ -> c
	{0xa1c, 0x6a}, 	// ਜ -> j
	{0xa24, 0x74}, 	// ਤ -> t
	{0xa26, 0x64}, 	// ਦ -> d
	{0xa28, 0x6e}, 	// ਨ -> n
	{0xa2a, 0x70}, 	// ਪ -> p
	{0xa2c, 0x62}, 	// ਬ -> b
	{0xa2e, 0x6d}, 	// ਮ -> m
	{0xa2f, 0x79}, 	// ਯ -> y
	{0xa30, 0x72}, 	// ਰ -> r
	{0xa32, 0x6c}, 	// ਲ -> l
	{0xa35, 0x76}, 	// ਵ -> v
	{0xa38, 0x73}, 	// ਸ -> s
	{0xa39, 0x68}, 	// ਹ -> h
	{0xa3c, 0x27}, 	// ਼ -> '
	{0xa3f, 0x69}, 	// ਿ -> i
	{0xa41, 0x75}, 	// ੁ -> u
	{0xa5b, 0x7a}, 	// ਜ਼ -> z
	{0xa5e, 0x66}, 	// ਫ਼ -> f
	{0xa66, 0x6f}, 	// ੦ -> o
	{0xa67, 0x31}, 	// ੧ -> 1
	{0xa68, 0x32}, 	// ੨ -> 2
	{0xa69, 0x33}, 	// ੩ -> 3
	{0xa6a, 0x34}, 	// ੪ -> 4
	{0xa6b, 0x35}, 	// ੫ -> 5
	{0xa6c, 0x36}, 	// ੬ -> 6
	{0xa6d, 0x37}, 	// ੭ -> 7
	{0xa6e, 0x38}, 	// ੮ -> 8
	{0xa6f, 0x39}, 	// ੯ -> 9
	{0xa70, 0x4e}, 	// ੰ -> N
	{0xa71, 0x48}, 	// ੱ -> H
	{0xa81, 0x4e}, 	// ઁ -> N
	{0xa82, 0x4e}, 	// ં -> N
	{0xa83, 0x48}, 	// ઃ -> H
	{0xa85, 0x61}, 	// અ -> a
	{0xa87, 0x69}, 	// ઇ -> i
	{0xa89, 0x75}, 	// ઉ -> u
	{0xa8b, 0x52}, 	// ઋ -> R
	{0xa8f, 0x65}, 	// એ -> e
	{0xa93, 0x6f}, 	// ઓ -> o
	{0xa95, 0x6b}, 	// ક -> k
	{0xa97, 0x67}, 	// ગ -> g
	{0xa9a, 0x63}, 	// ચ -> c
	{0xa9c, 0x6a}, 	// જ -> j
	{0xaa4, 0x74}, 	// ત -> t
	{0xaa6, 0x64}, 	// દ -> d
	{0xaa8, 0x6e}, 	// ન -> n
	{0xaaa, 0x70}, 	// પ -> p
	{0xaac, 0x62}, 	// બ -> b
	{0xaae, 0x6d}, 	// મ -> m
	{0xab0, 0x72}, 	// ર -> r
	{0xab2, 0x6c}, 	// લ -> l
	{0xab5, 0x76}, 	// વ -> v
	{0xab8, 0x73}, 	// સ -> s
	{0xab9, 0x68}, 	// હ -> h
	{0xabc, 0x27}, 	// ઼ -> '
	{0xabd, 0x27}, 	// ઽ -> '
	{0xabf, 0x69}, 	// િ -> i
	{0xac1, 0x75}, 	// ુ -> u
	{0xac3, 0x52}, 	// ૃ -> R
	{0xac7, 0x65}, 	// ે -> e
	{0xacb, 0x6f}, 	// ો -> o
	{0xae6, 0x6f}, 	// ૦ -> o
	{0xae7, 0x31}, 	// ૧ -> 1
	{0xae8, 0x32}, 	// ૨ -> 2
	{0xae9, 0x33}, 	// ૩ -> 3
	{0xaea, 0x34}, 	// ૪ -> 4
	{0xaeb, 0x35}, 	// ૫ -> 5
	{0xaec, 0x36}, 	// ૬ -> 6
	{0xaed, 0x37}, 	// ૭ -> 7
	{0xaee, 0x38}, 	// ૮ -> 8
	{0xaef, 0x39}, 	// ૯ -> 9
	{0xb01, 0x4e}, 	// ଁ -> N
	{0xb02, 0x4e}, 	// ଂ -> N
	{0xb03, 0x48}, 	// ଃ -> H
	{0xb05, 0x61}, 	// ଅ -> a
	{0xb07, 0x69}, 	// ଇ -> i
	{0xb09, 0x75}, 	// ଉ -> u
	{0xb0b, 0x52}, 	// ଋ -> R
	{0xb0c, 0x4c}, 	// ଌ -> L
	{0xb0f, 0x65}, 	// ଏ -> e
	{0xb13, 0x6f}, 	// ଓ -> o
	{0xb15, 0x6b}, 	// କ -> k
	{0xb17, 0x67}, 	// ଗ -> g
	{0xb1a, 0x63}, 	// ଚ -> c
	{0xb1c, 0x6a}, 	// ଜ -> j
	{0xb20, 0x4f}, 	// ଠ -> O
	{0xb24, 0x74}, 	// ତ -> t
	{0xb26, 0x64}, 	// ଦ -> d
	{0xb28, 0x6e}, 	// ନ -> n
	{0xb2a, 0x70}, 	// ପ -> p
	{0xb2c, 0x62}, 	// ବ -> b
	{0xb2e, 0x6d}, 	// ମ -> m
	{0xb2f, 0x79}, 	// ଯ -> y
	{0xb30, 0x72}, 	// ର -> r
	{0xb32, 0x6c}, 	// ଲ -> l
	{0xb38, 0x73}, 	// ସ -> s
	{0xb39, 0x68}, 	// ହ -> h
	{0xb3c, 0x27}, 	// ଼ -> '
	{0xb3d, 0x27}, 	// ଽ -> '
	{0xb3f, 0x69}, 	// ି -> i
	{0xb41, 0x75}, 	// ୁ -> u
	{0xb43, 0x52}, 	// ୃ -> R
	{0xb47, 0x65}, 	// େ -> e
	{0xb4b, 0x6f}, 	// ୋ -> o
	{0xb56, 0x2b}, 	// ୖ -> +
	{0xb57, 0x2b}, 	// ୗ -> +
	{0xb66, 0x4f}, 	// ୦ -> O
	{0xb67, 0x31}, 	// ୧ -> 1
	{0xb68, 0x32}, 	// ୨ -> 2
	{0xb69, 0x33}, 	// ୩ -> 3
	{0xb6a, 0x34}, 	// ୪ -> 4
	{0xb6b, 0x35}, 	// ୫ -> 5
	{0xb6c, 0x36}, 	// ୬ -> 6
	{0xb6d, 0x37}, 	// ୭ -> 7
	{0xb6e, 0x38}, 	// ୮ -> 8
	{0xb6f, 0x39}, 	// ୯ -> 9
	{0xb82, 0x4e}, 	// ஂ -> N
	{0xb83, 0x48}, 	// ஃ -> H
	{0xb85, 0x61}, 	// அ -> a
	{0xb87, 0x69}, 	// இ -> i
	{0xb89, 0x75}, 	// உ -> u
	{0xb8e, 0x65}, 	// எ -> e
	{0xb92, 0x6f}, 	// ஒ -> o
	{0xb95, 0x6b}, 	// க -> k
	{0xb9a, 0x63}, 	// ச -> c
	{0xb9c, 0x6a}, 	// ஜ -> j
	{0xba4, 0x74}, 	// த -> t
	{0xba8, 0x6e}, 	// ந -> n
	{0xbaa, 0x70}, 	// ப -> p
	{0xbae, 0x6d}, 	// ம -> m
	{0xbaf, 0x79}, 	// ய -> y
	{0xbb0, 0x72}, 	// ர -> r
	{0xbb2, 0x6c}, 	// ல -> l
	{0xbb5, 0x76}, 	// வ -> v
	{0xbb8, 0x73}, 	// ஸ -> s
	{0xbb9, 0x68}, 	// ஹ -> h
	{0xbbf, 0x69}, 	// ி -> i
	{0xbc1, 0x75}, 	// ு -> u
	{0xbc6, 0x65}, 	// ெ -> e
	{0xbca, 0x6f}, 	// ொ -> o
	{0xbd7, 0x2b}, 	// ௗ -> +
	{0xbe6, 0x6f}, 	// ௦ -> o
	{0xbe7, 0x31}, 	// ௧ -> 1
	{0xbe8, 0x32}, 	// ௨ -> 2
	{0xbe9, 0x33}, 	// ௩ -> 3
	{0xbea, 0x34}, 	// ௪ -> 4
	{0xbeb, 0x35}, 	// ௫ -> 5
	{0xbec, 0x36}, 	// ௬ -> 6
	{0xbed, 0x37}, 	// ௭ -> 7
	{0xbee, 0x38}, 	// ௮ -> 8
	{0xbef, 0x39}, 	// ௯ -> 9
	{0xc01, 0x4e}, 	// ఁ -> N
	{0xc02, 0x6f}, 	// ం -> o
	{0xc03, 0x48}, 	// ః -> H
	{0xc05, 0x61}, 	// అ -> a
	{0xc07, 0x69}, 	// ఇ -> i
	{0xc09, 0x75}, 	// ఉ -> u
	{0xc0b, 0x52}, 	// ఋ -> R
	{0xc0c, 0x4c}, 	// ఌ -> L
	{0xc0e, 0x65}, 	// ఎ -> e
	{0xc12, 0x6f}, 	// ఒ -> o
	{0xc15, 0x6b}, 	// క -> k
	{0xc17, 0x67}, 	// గ -> g
	{0xc1a, 0x63}, 	// చ -> c
	{0xc1c, 0x6a}, 	// జ -> j
	{0xc24, 0x74}, 	// త -> t
	{0xc26, 0x64}, 	// ద -> d
	{0xc28, 0x6e}, 	// న -> n
	{0xc2a, 0x70}, 	// ప -> p
	{0xc2c, 0x62}, 	// బ -> b
	{0xc2e, 0x6d}, 	// మ -> m
	{0xc2f, 0x79}, 	// య -> y
	{0xc30, 0x72}, 	// ర -> r
	{0xc32, 0x6c}, 	// ల -> l
	{0xc35, 0x76}, 	// వ -> v
	{0xc38, 0x73}, 	// స -> s
	{0xc39, 0x68}, 	// హ -> h
	{0xc3f, 0x69}, 	// ి -> i
	{0xc41, 0x75}, 	// ు -> u
	{0xc43, 0x52}, 	// ృ -> R
	{0xc46, 0x65}, 	// ె -> e
	{0xc4a, 0x6f}, 	// ొ -> o
	{0xc55, 0x2b}, 	// ౕ -> +
	{0xc56, 0x2b}, 	// ౖ -> +
	{0xc66, 0x6f}, 	// ౦ -> o
	{0xc67, 0x31}, 	// ౧ -> 1
	{0xc68, 0x32}, 	// ౨ -> 2
	{0xc69, 0x33}, 	// ౩ -> 3
	{0xc6a, 0x34}, 	// ౪ -> 4
	{0xc6b, 0x35}, 	// ౫ -> 5
	{0xc6c, 0x36}, 	// ౬ -> 6
	{0xc6d, 0x37}, 	// ౭ -> 7
	{0xc6e, 0x38}, 	// ౮ -> 8
	{0xc6f, 0x39}, 	// ౯ -> 9
	{0xc82, 0x6f}, 	// ಂ -> o
	{0xc83, 0x48}, 	// ಃ -> H
	{0xc85, 0x61}, 	// ಅ -> a
	{0xc87, 0x69}, 	// ಇ -> i
	{0xc89, 0x75}, 	// ಉ -> u
	{0xc8b, 0x52}, 	// ಋ -> R
	{0xc8c, 0x4c}, 	// ಌ -> L
	{0xc8e, 0x65}, 	// ಎ -> e
	{0xc92, 0x6f}, 	// ಒ -> o
	{0xc95, 0x6b}, 	// ಕ -> k
	{0xc97, 0x67}, 	// ಗ -> g
	{0xc9a, 0x63}, 	// ಚ -> c
	{0xc9c, 0x6a}, 	// ಜ -> j
	{0xca4, 0x74}, 	// ತ -> t
	{0xca6, 0x64}, 	// ದ -> d
	{0xca8, 0x6e}, 	// ನ -> n
	{0xcaa, 0x70}, 	// ಪ -> p
	{0xcac, 0x62}, 	// ಬ -> b
	{0xcae, 0x6d}, 	// ಮ -> m
	{0xcaf, 0x79}, 	// ಯ -> y
	{0xcb0, 0x72}, 	// ರ -> r
	{0xcb2, 0x6c}, 	// ಲ -> l
	{0xcb5, 0x76}, 	// ವ -> v
	{0xcb8, 0x73}, 	// ಸ -> s
	{0xcb9, 0x68}, 	// ಹ -> h
	{0xcbf, 0x69}, 	// ಿ -> i
	{0xcc1, 0x75}, 	// ು -> u
	{0xcc3, 0x52}, 	// ೃ -> R
	{0xcc6, 0x65}, 	// ೆ -> e
	{0xcca, 0x6f}, 	// ೊ -> o
	{0xcd5, 0x2b}, 	// ೕ -> +
	{0xcd6, 0x2b}, 	// ೖ -> +
	{0xce6, 0x6f}, 	// ೦ -> o
	{0xce7, 0x31}, 	// ೧ -> 1
	{0xce8, 0x32}, 	// ೨ -> 2
	{0xce9, 0x33}, 	// ೩ -> 3
	{0xcea, 0x34}, 	// ೪ -> 4
	{0xceb, 0x35}, 	// ೫ -> 5
	{0xcec, 0x36}, 	// ೬ -> 6
	{0xced, 0x37}, 	// ೭ -> 7
	{0xcee, 0x38}, 	// ೮ -> 8
	{0xcef, 0x39}, 	// ೯ -> 9
	{0xd02, 0x6f}, 	// ം -> o
	{0xd03, 0x48}, 	// ഃ -> H
	{0xd05, 0x61}, 	// അ -> a
	{0xd07, 0x69}, 	// ഇ -> i
	{0xd09, 0x75}, 	// ഉ -> u
	{0xd0b, 0x52}, 	// ഋ -> R
	{0xd0c, 0x4c}, 	// ഌ -> L
	{0xd0e, 0x65}, 	// എ -> e
	{0xd12, 0x6f}, 	// ഒ -> o
	{0xd15, 0x6b}, 	// ക -> k
	{0xd17, 0x67}, 	// ഗ -> g
	{0xd1a, 0x63}, 	// ച -> c
	{0xd1c, 0x6a}, 	// ജ -> j
	{0xd20, 0x6f}, 	// ഠ -> o
	{0xd24, 0x74}, 	// ത -> t
	{0xd26, 0x64}, 	// ദ -> d
	{0xd28, 0x6e}, 	// ന -> n
	{0xd2a, 0x70}, 	// പ -> p
	{0xd2c, 0x62}, 	// ബ -> b
	{0xd2e, 0x6d}, 	// മ -> m
	{0xd2f, 0x79}, 	// യ -> y
	{0xd30, 0x72}, 	// ര -> r
	{0xd32, 0x6c}, 	// ല -> l
	{0xd35, 0x76}, 	// വ -> v
	{0xd38, 0x73}, 	// സ -> s
	{0xd39, 0x68}, 	// ഹ -> h
	{0xd3f, 0x69}, 	// ി -> i
	{0xd41, 0x75}, 	// ു -> u
	{0xd43, 0x52}, 	// ൃ -> R
	{0xd46, 0x65}, 	// െ -> e
	{0xd4a, 0x6f}, 	// ൊ -> o
	{0xd57, 0x2b}, 	// ൗ -> +
	{0xd5f, 0x6f}, 	// ൟ -> o
	{0xd66, 0x6f}, 	// ൦ -> o
	{0xd67, 0x31}, 	// ൧ -> 1
	{0xd68, 0x32}, 	// ൨ -> 2
	{0xd69, 0x33}, 	// ൩ -> 3
	{0xd6a, 0x34}, 	// ൪ -> 4
	{0xd6b, 0x35}, 	// ൫ -> 5
	{0xd6c, 0x36}, 	// ൬ -> 6
	{0xd6d, 0x37}, 	// ൭ -> 7
	{0xd6e, 0x38}, 	// ൮ -> 8
	{0xd6f, 0x39}, 	// ൯ -> 9
	{0xd82, 0x6f}, 	// ං -> o
	{0xd83, 0x48}, 	// ඃ -> H
	{0xd85, 0x61}, 	// අ -> a
	{0xd89, 0x69}, 	// ඉ -> i
	{0xd8b, 0x75}, 	// උ -> u
	{0xd8d, 0x52}, 	// ඍ -> R
	{0xd8f, 0x4c}, 	// ඏ -> L
	{0xd91, 0x65}, 	// එ -> e
	{0xd94, 0x6f}, 	// ඔ -> o
	{0xd9a, 0x6b}, 	// ක -> k
	{0xd9c, 0x67}, 	// ග -> g
	{0xda0, 0x63}, 	// ච -> c
	{0xda2, 0x6a}, 	// ජ -> j
	{0xdad, 0x74}, 	// ත -> t
	{0xdaf, 0x64}, 	// ද -> d
	{0xdb1, 0x6e}, 	// න -> n
	{0xdb4, 0x70}, 	// ප -> p
	{0xdb6, 0x62}, 	// බ -> b
	{0xdb8, 0x6d}, 	// ම -> m
	{0xdba, 0x79}, 	// ය -> y
	{0xdbb, 0x72}, 	// ර -> r
	{0xdbd, 0x6c}, 	// ල -> l
	{0xdc0, 0x76}, 	// ව -> v
	{0xdc3, 0x73}, 	// ස -> s
	{0xdc4, 0x68}, 	// හ -> h
	{0xdc6, 0x66}, 	// ෆ -> f
	{0xdd2, 0x69}, 	// ි -> i
	{0xdd4, 0x75}, 	// ු -> u
	{0xdd8, 0x52}, 	// ෘ -> R
	{0xdd9, 0x65}, 	// ෙ -> e
	{0xddc, 0x6f}, 	// ො -> o
	{0xddf, 0x4c}, 	// ෟ -> L
	{0xe01, 0x6b}, 	// ก -> k
	{0xe0d, 0x79}, 	// ญ -> y
	{0xe0e, 0x64}, 	// ฎ -> d
	{0xe0f, 0x74}, 	// ฏ -> t
	{0xe13, 0x6e}, 	// ณ -> n
	{0xe14, 0x64}, 	// ด -> d
	{0xe15, 0x74}, 	// ต -> t
	{0xe19, 0x6e}, 	// น -> n
	{0xe1a, 0x62}, 	// บ -> b
	{0xe1b, 0x70}, 	// ป -> p
	{0xe1d, 0x66}, 	// ฝ -> f
	{0xe1f, 0x66}, 	// ฟ -> f
	{0xe21, 0x6d}, 	// ม -> m
	{0xe22, 0x79}, 	// ย -> y
	{0xe23, 0x72}, 	// ร -> r
	{0xe24, 0x52}, 	// ฤ -> R
	{0xe25, 0x6c}, 	// ล -> l
	{0xe26, 0x4c}, 	// ฦ -> L
	{0xe27, 0x77}, 	// ว -> w
	{0xe28, 0x73}, 	// ศ -> s
	{0xe29, 0x73}, 	// ษ -> s
	{0xe2a, 0x73}, 	// ส -> s
	{0xe2b, 0x68}, 	// ห -> h
	{0xe2c, 0x6c}, 	// ฬ -> l
	{0xe2d, 0x60}, 	// อ -> `
	{0xe2e, 0x68}, 	// ฮ -> h
	{0xe2f, 0x7e}, 	// ฯ -> ~
	{0xe30, 0x61}, 	// ะ -> a
	{0xe31, 0x61}, 	// ั -> a
	{0xe34, 0x69}, 	// ิ -> i
	{0xe38, 0x75}, 	// ุ -> u
	{0xe3a, 0x27}, 	// ฺ -> '
	{0xe40, 0x65}, 	// เ -> e
	{0xe42, 0x6f}, 	// โ -> o
	{0xe46, 0x2b}, 	// ๆ -> +
	{0xe4d, 0x4d}, 	// ํ -> M
	{0xe50, 0x6f}, 	// ๐ -> o
	{0xe51, 0x31}, 	// ๑ -> 1
	{0xe52, 0x32}, 	// ๒ -> 2
	{0xe53, 0x33}, 	// ๓ -> 3
	{0xe54, 0x34}, 	// ๔ -> 4
	{0xe55, 0x35}, 	// ๕ -> 5
	{0xe56, 0x36}, 	// ๖ -> 6
	{0xe57, 0x37}, 	// ๗ -> 7
	{0xe58, 0x38}, 	// ๘ -> 8
	{0xe59, 0x39}, 	// ๙ -> 9
	{0xe81, 0x6b}, 	// ກ -> k
	{0xe8a, 0x73}, 	// ຊ -> s
	{0xe94, 0x64}, 	// ດ -> d
	{0xe95, 0x68}, 	// ຕ -> h
	{0xe99, 0x6e}, 	// ນ -> n
	{0xe9a, 0x62}, 	// ບ -> b
	{0xe9b, 0x70}, 	// ປ -> p
	{0xe9d, 0x66}, 	// ຝ -> f
	{0xe9f, 0x66}, 	// ຟ -> f
	{0xea1, 0x6d}, 	// ມ -> m
	{0xea2, 0x79}, 	// ຢ -> y
	{0xea3, 0x72}, 	// ຣ -> r
	{0xea5, 0x6c}, 	// ລ -> l
	{0xea7, 0x77}, 	// ວ -> w
	{0xeaa, 0x73}, 	// ສ -> s
	{0xeab, 0x68}, 	// ຫ -> h
	{0xead, 0x60}, 	// ອ -> `
	{0xeaf, 0x7e}, 	// ຯ -> ~
	{0xeb0, 0x61}, 	// ະ -> a
	{0xeb4, 0x69}, 	// ິ -> i
	{0xeb6, 0x79}, 	// ຶ -> y
	{0xeb8, 0x75}, 	// ຸ -> u
	{0xebb, 0x6f}, 	// ົ -> o
	{0xebc, 0x6c}, 	// ຼ -> l
	{0xec0, 0x65}, 	// ເ -> e
	{0xec2, 0x6f}, 	// ໂ -> o
	{0xec6, 0x2b}, 	// ໆ -> +
	{0xecd, 0x4d}, 	// ໍ -> M
	{0xed0, 0x6f}, 	// ໐ -> o
	{0xed1, 0x31}, 	// ໑ -> 1
	{0xed2, 0x32}, 	// ໒ -> 2
	{0xed3, 0x33}, 	// ໓ -> 3
	{0xed4, 0x34}, 	// ໔ -> 4
	{0xed5, 0x35}, 	// ໕ -> 5
	{0xed6, 0x36}, 	// ໖ -> 6
	{0xed7, 0x37}, 	// ໗ -> 7
	{0xed8, 0x38}, 	// ໘ -> 8
	{0xed9, 0x39}, 	// ໙ -> 9
	{0xf0b, 0x2d}, 	// ་ -> -
	{0xf20, 0x30}, 	// ༠ -> 0
	{0xf21, 0x31}, 	// ༡ -> 1
	{0xf22, 0x32}, 	// ༢ -> 2
	{0xf23, 0x33}, 	// ༣ -> 3
	{0xf24, 0x34}, 	// ༤ -> 4
	{0xf25, 0x35}, 	// ༥ -> 5
	{0xf26, 0x36}, 	// ༦ -> 6
	{0xf27, 0x37}, 	// ༧ -> 7
	{0xf28, 0x38}, 	// ༨ -> 8
	{0xf29, 0x39}, 	// ༩ -> 9
	{0xf34, 0x2b}, 	// ༴ -> +
	{0xf35, 0x2a}, 	// ༵ -> *
	{0xf36, 0x5e}, 	// ༶ -> ^
	{0xf37, 0x5f}, 	// ༷ -> _
	{0xf39, 0x7e}, 	// ༹ -> ~
	{0xf3b, 0x5d}, 	// ༻ -> ]
	{0xf40, 0x6b}, 	// ཀ -> k
	{0xf42, 0x67}, 	// ག -> g
	{0xf45, 0x63}, 	// ཅ -> c
	{0xf47, 0x6a}, 	// ཇ -> j
	{0xf4f, 0x74}, 	// ཏ -> t
	{0xf51, 0x64}, 	// ད -> d
	{0xf53, 0x6e}, 	// ན -> n
	{0xf54, 0x70}, 	// པ -> p
	{0xf56, 0x62}, 	// བ -> b
	{0xf58, 0x6d}, 	// མ -> m
	{0xf5d, 0x77}, 	// ཝ -> w
	{0xf5f, 0x7a}, 	// ཟ -> z
	{0xf60, 0x27}, 	// འ -> '
	{0xf61, 0x79}, 	// ཡ -> y
	{0xf62, 0x72}, 	// ར -> r
	{0xf63, 0x6c}, 	// ལ -> l
	{0xf66, 0x73}, 	// ས -> s
	{0xf67, 0x68}, 	// ཧ -> h
	{0xf68, 0x61}, 	// ཨ -> a
	{0xf6a, 0x72}, 	// ཪ -> r
	{0xf72, 0x69}, 	// ི -> i
	{0xf74, 0x75}, 	// ུ -> u
	{0xf76, 0x52}, 	// ྲྀ -> R
	{0xf78, 0x4c}, 	// ླྀ -> L
	{0xf7a, 0x65}, 	// ེ -> e
	{0xf7c, 0x6f}, 	// ོ -> o
	{0xf7e, 0x4d}, 	// ཾ -> M
	{0xf7f, 0x48}, 	// ཿ -> H
	{0xf80, 0x69}, 	// ྀ -> i
	{0xf90, 0x6b}, 	// ྐ -> k
	{0xf92, 0x67}, 	// ྒ -> g
	{0xf95, 0x63}, 	// ྕ -> c
	{0xf97, 0x6a}, 	// ྗ -> j
	{0xf9f, 0x74}, 	// ྟ -> t
	{0xfa1, 0x64}, 	// ྡ -> d
	{0xfa3, 0x6e}, 	// ྣ -> n
	{0xfa4, 0x70}, 	// ྤ -> p
	{0xfa6, 0x62}, 	// ྦ -> b
	{0xfa8, 0x6d}, 	// ྨ -> m
	{0xfad, 0x77}, 	// ྭ -> w
	{0xfaf, 0x7a}, 	// ྯ -> z
	{0xfb0, 0x27}, 	// ྰ -> '
	{0xfb1, 0x79}, 	// ྱ -> y
	{0xfb2, 0x72}, 	// ྲ -> r
	{0xfb3, 0x6c}, 	// ླ -> l
	{0xfb6, 0x73}, 	// ྶ -> s
	{0xfb7, 0x68}, 	// ྷ -> h
	{0xfb8, 0x61}, 	// ྸ -> a
	{0xfba, 0x77}, 	// ྺ -> w
	{0xfbb, 0x79}, 	// ྻ -> y
	{0xfbc, 0x72}, 	// ྼ -> r
	{0xfbe, 0x58}, 	// ྾ -> X
	{0x1000, 0x6b}, 	// က -> k
	{0x1002, 0x67}, 	// ဂ -> g
	{0x1005, 0x63}, 	// စ -> c
	{0x1007, 0x6a}, 	// ဇ -> j
	{0x1010, 0x6f}, 	// တ -> o
	{0x1012, 0x64}, 	// ဒ -> d
	{0x1014, 0x6e}, 	// န -> n
	{0x1015, 0x70}, 	// ပ -> p
	{0x1017, 0x62}, 	// ဗ -> b
	{0x1019, 0x6d}, 	// မ -> m
	{0x101a, 0x79}, 	// ယ -> y
	{0x101b, 0x72}, 	// ရ -> r
	{0x101c, 0x6c}, 	// လ -> l
	{0x101d, 0x6f}, 	// ဝ -> o
	{0x101e, 0x73}, 	// သ -> s
	{0x101f, 0x68}, 	// ဟ -> h
	{0x1021, 0x61}, 	// အ -> a
	{0x1023, 0x69}, 	// ဣ -> i
	{0x1025, 0x75}, 	// ဥ -> u
	{0x1027, 0x65}, 	// ဧ -> e
	{0x1029, 0x6f}, 	// ဩ -> o
	{0x102d, 0x69}, 	// ိ -> i
	{0x102f, 0x75}, 	// ု -> u
	{0x1031, 0x65}, 	// ေ -> e
	{0x1036, 0x4e}, 	// ံ -> N
	{0x1037, 0x27}, 	// ့ -> '
	{0x1038, 0x3a}, 	// း -> :
	{0x1040, 0x6f}, 	// ၀ -> o
	{0x1041, 0x31}, 	// ၁ -> 1
	{0x1042, 0x32}, 	// ၂ -> 2
	{0x1043, 0x33}, 	// ၃ -> 3
	{0x1044, 0x34}, 	// ၄ -> 4
	{0x1045, 0x35}, 	// ၅ -> 5
	{0x1046, 0x36}, 	// ၆ -> 6
	{0x1047, 0x37}, 	// ၇ -> 7
	{0x1048, 0x38}, 	// ၈ -> 8
	{0x1049, 0x39}, 	// ၉ -> 9
	{0x1052, 0x52}, 	// ၒ -> R
	{0x1054, 0x4c}, 	// ၔ -> L
	{0x1056, 0x52}, 	// ၖ -> R
	{0x1058, 0x4c}, 	// ၘ -> L
	{0x10a0, 0x41}, 	// Ⴀ -> A
	{0x10a1, 0x42}, 	// Ⴁ -> B
	{0x10a2, 0x47}, 	// Ⴂ -> G
	{0x10a3, 0x44}, 	// Ⴃ -> D
	{0x10a4, 0x45}, 	// Ⴄ -> E
	{0x10a5, 0x56}, 	// Ⴅ -> V
	{0x10a6, 0x5a}, 	// Ⴆ -> Z
	{0x10a8, 0x49}, 	// Ⴈ -> I
	{0x10a9, 0x4b}, 	// Ⴉ -> K
	{0x10aa, 0x4c}, 	// Ⴊ -> L
	{0x10ab, 0x4d}, 	// Ⴋ -> M
	{0x10ac, 0x4e}, 	// Ⴌ -> N
	{0x10ad, 0x4f}, 	// Ⴍ -> O
	{0x10ae, 0x50}, 	// Ⴎ -> P
	{0x10b0, 0x52}, 	// Ⴐ -> R
	{0x10b1, 0x53}, 	// Ⴑ -> S
	{0x10b2, 0x54}, 	// Ⴒ -> T
	{0x10b3, 0x55}, 	// Ⴓ -> U
	{0x10b7, 0x51}, 	// Ⴗ -> Q
	{0x10bc, 0x43}, 	// Ⴜ -> C
	{0x10be, 0x58}, 	// Ⴞ -> X
	{0x10bf, 0x4a}, 	// Ⴟ -> J
	{0x10c0, 0x48}, 	// Ⴠ -> H
	{0x10c1, 0x45}, 	// Ⴡ -> E
	{0x10c2, 0x59}, 	// Ⴢ -> Y
	{0x10c3, 0x57}, 	// Ⴣ -> W
	{0x10d0, 0x61}, 	// ა -> a
	{0x10d1, 0x62}, 	// ბ -> b
	{0x10d2, 0x67}, 	// გ -> g
	{0x10d3, 0x64}, 	// დ -> d
	{0x10d4, 0x65}, 	// ე -> e
	{0x10d5, 0x76}, 	// ვ -> v
	{0x10d6, 0x7a}, 	// ზ -> z
	{0x10d8, 0x69}, 	// ი -> i
	{0x10d9, 0x6b}, 	// კ -> k
	{0x10da, 0x6c}, 	// ლ -> l
	{0x10db, 0x6d}, 	// მ -> m
	{0x10dc, 0x6e}, 	// ნ -> n
	{0x10dd, 0x6f}, 	// ო -> o
	{0x10de, 0x70}, 	// პ -> p
	{0x10e0, 0x72}, 	// რ -> r
	{0x10e1, 0x73}, 	// ს -> s
	{0x10e2, 0x74}, 	// ტ -> t
	{0x10e3, 0x75}, 	// უ -> u
	{0x10e7, 0x79}, 	// ყ -> y
	{0x10ec, 0x63}, 	// წ -> c
	{0x10ee, 0x78}, 	// ხ -> x
	{0x10ef, 0x6a}, 	// ჯ -> j
	{0x10f0, 0x68}, 	// ჰ -> h
	{0x10f1, 0x65}, 	// ჱ -> e
	{0x10f2, 0x79}, 	// ჲ -> y
	{0x10f3, 0x77}, 	// ჳ -> w
	{0x10f6, 0x66}, 	// ჶ -> f
	{0x10ff, 0x6f}, 	// ჿ -> o
	{0x1100, 0x67}, 	// ᄀ -> g
	{0x1102, 0x6e}, 	// ᄂ -> n
	{0x1103, 0x64}, 	// ᄃ -> d
	{0x1105, 0x72}, 	// ᄅ -> r
	{0x1106, 0x6d}, 	// ᄆ -> m
	{0x1107, 0x62}, 	// ᄇ -> b
	{0x1109, 0x73}, 	// ᄉ -> s
	{0x110c, 0x6a}, 	// ᄌ -> j
	{0x110e, 0x63}, 	// ᄎ -> c
	{0x110f, 0x6b}, 	// ᄏ -> k
	{0x1110, 0x74}, 	// ᄐ -> t
	{0x1111, 0x70}, 	// ᄑ -> p
	{0x1112, 0x68}, 	// ᄒ -> h
	{0x1135, 0x73}, 	// ᄵ -> s
	{0x1140, 0x5a}, 	// ᅀ -> Z
	{0x1141, 0x67}, 	// ᅁ -> g
	{0x1142, 0x64}, 	// ᅂ -> d
	{0x1143, 0x6d}, 	// ᅃ -> m
	{0x1144, 0x62}, 	// ᅄ -> b
	{0x1145, 0x73}, 	// ᅅ -> s
	{0x1146, 0x5a}, 	// ᅆ -> Z
	{0x1148, 0x6a}, 	// ᅈ -> j
	{0x1149, 0x63}, 	// ᅉ -> c
	{0x114a, 0x74}, 	// ᅊ -> t
	{0x114b, 0x70}, 	// ᅋ -> p
	{0x114c, 0x4e}, 	// ᅌ -> N
	{0x114d, 0x6a}, 	// ᅍ -> j
	{0x1159, 0x51}, 	// ᅙ -> Q
	{0x1161, 0x61}, 	// ᅡ -> a
	{0x1166, 0x65}, 	// ᅦ -> e
	{0x1169, 0x6f}, 	// ᅩ -> o
	{0x116e, 0x75}, 	// ᅮ -> u
	{0x1175, 0x69}, 	// ᅵ -> i
	{0x119e, 0x55}, 	// ᆞ -> U
	{0x11a8, 0x67}, 	// ᆨ -> g
	{0x11ab, 0x6e}, 	// ᆫ -> n
	{0x11ae, 0x64}, 	// ᆮ -> d
	{0x11af, 0x6c}, 	// ᆯ -> l
	{0x11b7, 0x6d}, 	// ᆷ -> m
	{0x11b8, 0x62}, 	// ᆸ -> b
	{0x11ba, 0x73}, 	// ᆺ -> s
	{0x11bd, 0x6a}, 	// ᆽ -> j
	{0x11be, 0x63}, 	// ᆾ -> c
	{0x11bf, 0x6b}, 	// ᆿ -> k
	{0x11c0, 0x74}, 	// ᇀ -> t
	{0x11c1, 0x70}, 	// ᇁ -> p
	{0x11c2, 0x68}, 	// ᇂ -> h
	{0x11eb, 0x5a}, 	// ᇫ -> Z
	{0x11ec, 0x67}, 	// ᇬ -> g
	{0x11f0, 0x4e}, 	// ᇰ -> N
	{0x11f9, 0x51}, 	// ᇹ -> Q
	{0x1200, 0x55}, 	// ሀ -> U
	{0x12d0, 0x4f}, 	// ዐ -> O
	{0x1361, 0x20}, 	// ፡ ->  
	{0x1362, 0x2e}, 	// ። -> .
	{0x1363, 0x2c}, 	// ፣ -> ,
	{0x1364, 0x3b}, 	// ፤ -> ;
	{0x1365, 0x3a}, 	// ፥ -> :
	{0x1367, 0x3f}, 	// ፧ -> ?
	{0x1369, 0x31}, 	// ፩ -> 1
	{0x136a, 0x32}, 	// ፪ -> 2
	{0x136b, 0x33}, 	// ፫ -> 3
	{0x136c, 0x34}, 	// ፬ -> 4
	{0x136d, 0x35}, 	// ፭ -> 5
	{0x136e, 0x36}, 	// ፮ -> 6
	{0x136f, 0x37}, 	// ፯ -> 7
	{0x1370, 0x38}, 	// ፰ -> 8
	{0x1371, 0x39}, 	// ፱ -> 9
	{0x13a0, 0x44}, 	// Ꭰ -> D
	{0x13a1, 0x52}, 	// Ꭱ -> R
	{0x13a2, 0x54}, 	// Ꭲ -> T
	{0x13a3, 0x6f}, 	// Ꭳ -> o
	{0x13a4, 0x4f}, 	// Ꭴ -> O
	{0x13a5, 0x69}, 	// Ꭵ -> i
	{0x13a9, 0x59}, 	// Ꭹ -> Y
	{0x13aa, 0x41}, 	// Ꭺ -> A
	{0x13ab, 0x4a}, 	// Ꭻ -> J
	{0x13ac, 0x45}, 	// Ꭼ -> E
	{0x13b3, 0x57}, 	// Ꮃ -> W
	{0x13b7, 0x4d}, 	// Ꮇ -> M
	{0x13bb, 0x48}, 	// Ꮋ -> H
	{0x13bd, 0x59}, 	// Ꮍ -> Y
	{0x13be, 0x4f}, 	// Ꮎ -> O
	{0x13c0, 0x47}, 	// Ꮐ -> G
	{0x13c2, 0x68}, 	// Ꮒ -> h
	{0x13c3, 0x5a}, 	// Ꮓ -> Z
	{0x13cc, 0x55}, 	// Ꮜ -> U
	{0x13cd, 0x73}, 	// Ꮝ -> s
	{0x13cf, 0x62}, 	// Ꮟ -> b
	{0x13d2, 0x52}, 	// Ꮢ -> R
	{0x13d4, 0x57}, 	// Ꮤ -> W
	{0x13d5, 0x53}, 	// Ꮥ -> S
	{0x13d9, 0x56}, 	// Ꮩ -> V
	{0x13da, 0x53}, 	// Ꮪ -> S
	{0x13de, 0x4c}, 	// Ꮮ -> L
	{0x13df, 0x43}, 	// Ꮯ -> C
	{0x13e2, 0x50}, 	// Ꮲ -> P
	{0x13e6, 0x4b}, 	// Ꮶ -> K
	{0x13e7, 0x64}, 	// Ꮷ -> d
	{0x13eb, 0x4f}, 	// Ꮻ -> O
	{0x13f2, 0x68}, 	// Ᏺ -> h
	{0x13f3, 0x47}, 	// Ᏻ -> G
	{0x13f4, 0x42}, 	// Ᏼ -> B
	{0x1401, 0x65}, 	// ᐁ -> e
	{0x1403, 0x69}, 	// ᐃ -> i
	{0x1405, 0x6f}, 	// ᐅ -> o
	{0x1409, 0x69}, 	// ᐉ -> i
	{0x140a, 0x61}, 	// ᐊ -> a
	{0x141d, 0x77}, 	// ᐝ -> w
	{0x141e, 0x27}, 	// ᐞ -> '
	{0x141f, 0x74}, 	// ᐟ -> t
	{0x1420, 0x6b}, 	// ᐠ -> k
	{0x1422, 0x73}, 	// ᐢ -> s
	{0x1423, 0x6e}, 	// ᐣ -> n
	{0x1424, 0x77}, 	// ᐤ -> w
	{0x1425, 0x6e}, 	// ᐥ -> n
	{0x1427, 0x77}, 	// ᐧ -> w
	{0x1428, 0x63}, 	// ᐨ -> c
	{0x1429, 0x3f}, 	// ᐩ -> ?
	{0x142a, 0x6c}, 	// ᐪ -> l
	{0x142f, 0x56}, 	// ᐯ -> V
	{0x143a, 0x56}, 	// ᐺ -> V
	{0x143b, 0x56}, 	// ᐻ -> V
	{0x1449, 0x70}, 	// ᑉ -> p
	{0x144a, 0x70}, 	// ᑊ -> p
	{0x144b, 0x68}, 	// ᑋ -> h
	{0x144c, 0x55}, 	// ᑌ -> U
	{0x1457, 0x55}, 	// ᑗ -> U
	{0x1458, 0x55}, 	// ᑘ -> U
	{0x1466, 0x74}, 	// ᑦ -> t
	{0x1467, 0x55}, 	// ᑧ -> U
	{0x146d, 0x50}, 	// ᑭ -> P
	{0x146f, 0x64}, 	// ᑯ -> d
	{0x1472, 0x62}, 	// ᑲ -> b
	{0x1473, 0x62}, 	// ᑳ -> b
	{0x1476, 0x50}, 	// ᑶ -> P
	{0x1477, 0x70}, 	// ᑷ -> p
	{0x147a, 0x64}, 	// ᑺ -> d
	{0x147b, 0x64}, 	// ᑻ -> d
	{0x147e, 0x62}, 	// ᑾ -> b
	{0x147f, 0x62}, 	// ᑿ -> b
	{0x1480, 0x62}, 	// ᒀ -> b
	{0x1481, 0x62}, 	// ᒁ -> b
	{0x1483, 0x6b}, 	// ᒃ -> k
	{0x1486, 0x50}, 	// ᒆ -> P
	{0x1487, 0x64}, 	// ᒇ -> d
	{0x1488, 0x62}, 	// ᒈ -> b
	{0x148d, 0x4a}, 	// ᒍ -> J
	{0x1498, 0x4a}, 	// ᒘ -> J
	{0x1499, 0x4a}, 	// ᒙ -> J
	{0x14a1, 0x63}, 	// ᒡ -> c
	{0x14aa, 0x4c}, 	// ᒪ -> L
	{0x14b6, 0x4c}, 	// ᒶ -> L
	{0x14b7, 0x6c}, 	// ᒷ -> l
	{0x14bb, 0x6d}, 	// ᒻ -> m
	{0x14bc, 0x6d}, 	// ᒼ -> m
	{0x14be, 0x6d}, 	// ᒾ -> m
	{0x14bf, 0x6d}, 	// ᒿ -> m
	{0x14d0, 0x6e}, 	// ᓐ -> n
	{0x14ea, 0x6c}, 	// ᓪ -> l
	{0x14eb, 0x6c}, 	// ᓫ -> l
	{0x14ec, 0x6c}, 	// ᓬ -> l
	{0x1505, 0x73}, 	// ᔅ -> s
	{0x1506, 0x73}, 	// ᔆ -> s
	{0x1508, 0x73}, 	// ᔈ -> s
	{0x150e, 0x62}, 	// ᔎ -> b
	{0x153e, 0x79}, 	// ᔾ -> y
	{0x153f, 0x79}, 	// ᔿ -> y
	{0x1540, 0x79}, 	// ᕀ -> y
	{0x1541, 0x78}, 	// ᕁ -> x
	{0x1550, 0x72}, 	// ᕐ -> r
	{0x1551, 0x72}, 	// ᕑ -> r
	{0x1552, 0x72}, 	// ᕒ -> r
	{0x155d, 0x66}, 	// ᕝ -> f
	{0x157b, 0x68}, 	// ᕻ -> h
	{0x157c, 0x48}, 	// ᕼ -> H
	{0x157d, 0x78}, 	// ᕽ -> x
	{0x157f, 0x50}, 	// ᕿ -> P
	{0x1581, 0x64}, 	// ᖁ -> d
	{0x1583, 0x62}, 	// ᖃ -> b
	{0x1584, 0x62}, 	// ᖄ -> b
	{0x1585, 0x71}, 	// ᖅ -> q
	{0x1587, 0x52}, 	// ᖇ -> R
	{0x1591, 0x4a}, 	// ᖑ -> J
	{0x15af, 0x62}, 	// ᖯ -> b
	{0x15b0, 0x65}, 	// ᖰ -> e
	{0x15b1, 0x69}, 	// ᖱ -> i
	{0x15b2, 0x6f}, 	// ᖲ -> o
	{0x15b3, 0x61}, 	// ᖳ -> a
	{0x15b4, 0x46}, 	// ᖴ -> F
	{0x15c5, 0x41}, 	// ᗅ -> A
	{0x15de, 0x44}, 	// ᗞ -> D
	{0x15ea, 0x44}, 	// ᗪ -> D
	{0x15ee, 0x70}, 	// ᗮ -> p
	{0x15f0, 0x4d}, 	// ᗰ -> M
	{0x15f7, 0x42}, 	// ᗷ -> B
	{0x1646, 0x7a}, 	// ᙆ -> z
	{0x1647, 0x7a}, 	// ᙇ -> z
	{0x166d, 0x58}, 	// ᙭ -> X
	{0x166e, 0x78}, 	// ᙮ -> x
	{0x1673, 0x4a}, 	// ᙳ -> J
	{0x1680, 0x20}, 	//   ->  
	{0x1681, 0x62}, 	// ᚁ -> b
	{0x1682, 0x6c}, 	// ᚂ -> l
	{0x1683, 0x66}, 	// ᚃ -> f
	{0x1684, 0x73}, 	// ᚄ -> s
	{0x1685, 0x6e}, 	// ᚅ -> n
	{0x1686, 0x68}, 	// ᚆ -> h
	{0x1687, 0x64}, 	// ᚇ -> d
	{0x1688, 0x74}, 	// ᚈ -> t
	{0x1689, 0x63}, 	// ᚉ -> c
	{0x168a, 0x71}, 	// ᚊ -> q
	{0x168b, 0x6d}, 	// ᚋ -> m
	{0x168c, 0x67}, 	// ᚌ -> g
	{0x168e, 0x7a}, 	// ᚎ -> z
	{0x168f, 0x72}, 	// ᚏ -> r
	{0x1690, 0x61}, 	// ᚐ -> a
	{0x1691, 0x6f}, 	// ᚑ -> o
	{0x1692, 0x75}, 	// ᚒ -> u
	{0x1693, 0x65}, 	// ᚓ -> e
	{0x1694, 0x69}, 	// ᚔ -> i
	{0x1698, 0x70}, 	// ᚘ -> p
	{0x1699, 0x78}, 	// ᚙ -> x
	{0x169a, 0x70}, 	// ᚚ -> p
	{0x169b, 0x3c}, 	// ᚛ -> <
	{0x169c, 0x3e}, 	// ᚜ -> >
	{0x16a0, 0x66}, 	// ᚠ -> f
	{0x16a1, 0x76}, 	// ᚡ -> v
	{0x16a2, 0x75}, 	// ᚢ -> u
	{0x16a4, 0x79}, 	// ᚤ -> y
	{0x16a5, 0x77}, 	// ᚥ -> w
	{0x16a8, 0x61}, 	// ᚨ -> a
	{0x16a9, 0x6f}, 	// ᚩ -> o
	{0x16ac, 0x6f}, 	// ᚬ -> o
	{0x16ad, 0x6f}, 	// ᚭ -> o
	{0x16ae, 0x6f}, 	// ᚮ -> o
	{0x16b1, 0x72}, 	// ᚱ -> r
	{0x16b2, 0x6b}, 	// ᚲ -> k
	{0x16b3, 0x63}, 	// ᚳ -> c
	{0x16b4, 0x6b}, 	// ᚴ -> k
	{0x16b5, 0x67}, 	// ᚵ -> g
	{0x16b7, 0x58}, 	// ᚷ -> X
	{0x16b8, 0x67}, 	// ᚸ -> g
	{0x16b9, 0x77}, 	// ᚹ -> w
	{0x16ba, 0x68}, 	// ᚺ -> h
	{0x16bb, 0x68}, 	// ᚻ -> h
	{0x16bc, 0x68}, 	// ᚼ -> h
	{0x16bd, 0x68}, 	// ᚽ -> h
	{0x16be, 0x6e}, 	// ᚾ -> n
	{0x16bf, 0x6e}, 	// ᚿ -> n
	{0x16c0, 0x6e}, 	// ᛀ -> n
	{0x16c1, 0x6c}, 	// ᛁ -> l
	{0x16c2, 0x65}, 	// ᛂ -> e
	{0x16c3, 0x6a}, 	// ᛃ -> j
	{0x16c4, 0x67}, 	// ᛄ -> g
	{0x16c6, 0x61}, 	// ᛆ -> a
	{0x16c8, 0x70}, 	// ᛈ -> p
	{0x16c9, 0x7a}, 	// ᛉ -> z
	{0x16ca, 0x73}, 	// ᛊ -> s
	{0x16cb, 0x73}, 	// ᛋ -> s
	{0x16cc, 0x73}, 	// ᛌ -> s
	{0x16cd, 0x63}, 	// ᛍ -> c
	{0x16ce, 0x7a}, 	// ᛎ -> z
	{0x16cf, 0x74}, 	// ᛏ -> t
	{0x16d0, 0x74}, 	// ᛐ -> t
	{0x16d1, 0x64}, 	// ᛑ -> d
	{0x16d2, 0x62}, 	// ᛒ -> b
	{0x16d3, 0x62}, 	// ᛓ -> b
	{0x16d4, 0x70}, 	// ᛔ -> p
	{0x16d5, 0x4b}, 	// ᛕ -> K
	{0x16d6, 0x4d}, 	// ᛖ -> M
	{0x16d7, 0x6d}, 	// ᛗ -> m
	{0x16d8, 0x6d}, 	// ᛘ -> m
	{0x16d9, 0x6d}, 	// ᛙ -> m
	{0x16da, 0x6c}, 	// ᛚ -> l
	{0x16db, 0x6c}, 	// ᛛ -> l
	{0x16de, 0x64}, 	// ᛞ -> d
	{0x16df, 0x6f}, 	// ᛟ -> o
	{0x16e5, 0x73}, 	// ᛥ -> s
	{0x16e9, 0x71}, 	// ᛩ -> q
	{0x16ea, 0x78}, 	// ᛪ -> x
	{0x16eb, 0x2e}, 	// ᛫ -> .
	{0x16ec, 0x3a}, 	// ᛬ -> :
	{0x16ed, 0x2b}, 	// ᛭ -> +
	{0x1780, 0x6b}, 	// ក -> k
	{0x1782, 0x67}, 	// គ -> g
	{0x1785, 0x63}, 	// ច -> c
	{0x1787, 0x6a}, 	// ជ -> j
	{0x178a, 0x74}, 	// ដ -> t
	{0x178c, 0x64}, 	// ឌ -> d
	{0x178f, 0x74}, 	// ត -> t
	{0x1791, 0x64}, 	// ទ -> d
	{0x1793, 0x6e}, 	// ន -> n
	{0x1794, 0x70}, 	// ប -> p
	{0x1796, 0x62}, 	// ព -> b
	{0x1798, 0x6d}, 	// ម -> m
	{0x1799, 0x79}, 	// យ -> y
	{0x179a, 0x72}, 	// រ -> r
	{0x179b, 0x6c}, 	// ល -> l
	{0x179c, 0x76}, 	// វ -> v
	{0x179f, 0x73}, 	// ស -> s
	{0x17a0, 0x68}, 	// ហ -> h
	{0x17a1, 0x6c}, 	// ឡ -> l
	{0x17a2, 0x71}, 	// អ -> q
	{0x17a3, 0x61}, 	// ឣ -> a
	{0x17a5, 0x69}, 	// ឥ -> i
	{0x17a7, 0x75}, 	// ឧ -> u
	{0x17af, 0x65}, 	// ឯ -> e
	{0x17b4, 0x61}, 	// ឴ -> a
	{0x17b7, 0x69}, 	// ិ -> i
	{0x17b9, 0x79}, 	// ឹ -> y
	{0x17bb, 0x75}, 	// ុ -> u
	{0x17c1, 0x65}, 	// េ -> e
	{0x17c6, 0x4d}, 	// ំ -> M
	{0x17c7, 0x48}, 	// ះ -> H
	{0x17cc, 0x72}, 	// ៌ -> r
	{0x17ce, 0x21}, 	// ៎ -> !
	{0x17d4, 0x2e}, 	// ។ -> .
	{0x17d6, 0x3a}, 	// ៖ -> :
	{0x17d7, 0x2b}, 	// ៗ -> +
	{0x17dc, 0x27}, 	// ៜ -> '
	{0x17e0, 0x30}, 	// ០ -> 0
	{0x17e1, 0x31}, 	// ១ -> 1
	{0x17e2, 0x32}, 	// ២ -> 2
	{0x17e3, 0x33}, 	// ៣ -> 3
	{0x17e4, 0x34}, 	// ៤ -> 4
	{0x17e5, 0x35}, 	// ៥ -> 5
	{0x17e6, 0x36}, 	// ៦ -> 6
	{0x17e7, 0x37}, 	// ៧ -> 7
	{0x17e8, 0x38}, 	// ៨ -> 8
	{0x17e9, 0x39}, 	// ៩ -> 9
	{0x1807, 0x2d}, 	// ᠇ -> -
	{0x1810, 0x30}, 	// ᠐ -> 0
	{0x1811, 0x31}, 	// ᠑ -> 1
	{0x1812, 0x32}, 	// ᠒ -> 2
	{0x1813, 0x33}, 	// ᠓ -> 3
	{0x1814, 0x34}, 	// ᠔ -> 4
	{0x1815, 0x35}, 	// ᠕ -> 5
	{0x1816, 0x36}, 	// ᠖ -> 6
	{0x1817, 0x37}, 	// ᠗ -> 7
	{0x1818, 0x38}, 	// ᠘ -> 8
	{0x1819, 0x39}, 	// ᠙ -> 9
	{0x1820, 0x61}, 	// ᠠ -> a
	{0x1821, 0x65}, 	// ᠡ -> e
	{0x1822, 0x69}, 	// ᠢ -> i
	{0x1823, 0x6f}, 	// ᠣ -> o
	{0x1824, 0x75}, 	// ᠤ -> u
	{0x1825, 0x4f}, 	// ᠥ -> O
	{0x1826, 0x55}, 	// ᠦ -> U
	{0x1828, 0x6e}, 	// ᠨ -> n
	{0x182a, 0x62}, 	// ᠪ -> b
	{0x182b, 0x70}, 	// ᠫ -> p
	{0x182c, 0x71}, 	// ᠬ -> q
	{0x182d, 0x67}, 	// ᠭ -> g
	{0x182e, 0x6d}, 	// ᠮ -> m
	{0x182f, 0x6c}, 	// ᠯ -> l
	{0x1830, 0x73}, 	// ᠰ -> s
	{0x1832, 0x74}, 	// ᠲ -> t
	{0x1833, 0x64}, 	// ᠳ -> d
	{0x1835, 0x6a}, 	// ᠵ -> j
	{0x1836, 0x79}, 	// ᠶ -> y
	{0x1837, 0x72}, 	// ᠷ -> r
	{0x1838, 0x77}, 	// ᠸ -> w
	{0x1839, 0x66}, 	// ᠹ -> f
	{0x183a, 0x6b}, 	// ᠺ -> k
	{0x183d, 0x7a}, 	// ᠽ -> z
	{0x183e, 0x68}, 	// ᠾ -> h
	{0x1843, 0x2d}, 	// ᡃ -> -
	{0x1844, 0x65}, 	// ᡄ -> e
	{0x1845, 0x69}, 	// ᡅ -> i
	{0x1846, 0x6f}, 	// ᡆ -> o
	{0x1847, 0x75}, 	// ᡇ -> u
	{0x1848, 0x4f}, 	// ᡈ -> O
	{0x1849, 0x55}, 	// ᡉ -> U
	{0x184b, 0x62}, 	// ᡋ -> b
	{0x184c, 0x70}, 	// ᡌ -> p
	{0x184d, 0x71}, 	// ᡍ -> q
	{0x184e, 0x67}, 	// ᡎ -> g
	{0x184f, 0x6d}, 	// ᡏ -> m
	{0x1850, 0x74}, 	// ᡐ -> t
	{0x1851, 0x64}, 	// ᡑ -> d
	{0x1853, 0x6a}, 	// ᡓ -> j
	{0x1855, 0x79}, 	// ᡕ -> y
	{0x1856, 0x77}, 	// ᡖ -> w
	{0x1857, 0x6b}, 	// ᡗ -> k
	{0x1858, 0x67}, 	// ᡘ -> g
	{0x1859, 0x68}, 	// ᡙ -> h
	{0x185d, 0x65}, 	// ᡝ -> e
	{0x185e, 0x69}, 	// ᡞ -> i
	{0x1860, 0x55}, 	// ᡠ -> U
	{0x1861, 0x75}, 	// ᡡ -> u
	{0x1863, 0x6b}, 	// ᡣ -> k
	{0x1864, 0x67}, 	// ᡤ -> g
	{0x1865, 0x68}, 	// ᡥ -> h
	{0x1866, 0x70}, 	// ᡦ -> p
	{0x1868, 0x74}, 	// ᡨ -> t
	{0x1869, 0x64}, 	// ᡩ -> d
	{0x186a, 0x6a}, 	// ᡪ -> j
	{0x186b, 0x66}, 	// ᡫ -> f
	{0x186c, 0x67}, 	// ᡬ -> g
	{0x186d, 0x68}, 	// ᡭ -> h
	{0x186f, 0x7a}, 	// ᡯ -> z
	{0x1870, 0x72}, 	// ᡰ -> r
	{0x1873, 0x69}, 	// ᡳ -> i
	{0x1874, 0x6b}, 	// ᡴ -> k
	{0x1875, 0x72}, 	// ᡵ -> r
	{0x1876, 0x66}, 	// ᡶ -> f
	{0x1881, 0x48}, 	// ᢁ -> H
	{0x1882, 0x58}, 	// ᢂ -> X
	{0x1883, 0x57}, 	// ᢃ -> W
	{0x1884, 0x4d}, 	// ᢄ -> M
	{0x1887, 0x61}, 	// ᢇ -> a
	{0x1888, 0x69}, 	// ᢈ -> i
	{0x1889, 0x6b}, 	// ᢉ -> k
	{0x188b, 0x63}, 	// ᢋ -> c
	{0x1890, 0x74}, 	// ᢐ -> t
	{0x1891, 0x64}, 	// ᢑ -> d
	{0x1892, 0x70}, 	// ᢒ -> p
	{0x1896, 0x7a}, 	// ᢖ -> z
	{0x1897, 0x61}, 	// ᢗ -> a
	{0x1898, 0x74}, 	// ᢘ -> t
	{0x189c, 0x63}, 	// ᢜ -> c
	{0x18a0, 0x74}, 	// ᢠ -> t
	{0x18a5, 0x7a}, 	// ᢥ -> z
	{0x18a6, 0x75}, 	// ᢦ -> u
	{0x18a7, 0x79}, 	// ᢧ -> y
	{0x18a9, 0x27}, 	// ᢩ -> '
	{0x1d00, 0x41}, 	// ᴀ -> A
	{0x1d03, 0x42}, 	// ᴃ -> B
	{0x1d04, 0x63}, 	// ᴄ -> c
	{0x1d05, 0x44}, 	// ᴅ -> D
	{0x1d06, 0x44}, 	// ᴆ -> D
	{0x1d07, 0x45}, 	// ᴇ -> E
	{0x1d08, 0x65}, 	// ᴈ -> e
	{0x1d09, 0x69}, 	// ᴉ -> i
	{0x1d0a, 0x4a}, 	// ᴊ -> J
	{0x1d0b, 0x6b}, 	// ᴋ -> k
	{0x1d0c, 0x4c}, 	// ᴌ -> L
	{0x1d0d, 0x4d}, 	// ᴍ -> M
	{0x1d0e, 0x4e}, 	// ᴎ -> N
	{0x1d0f, 0x6f}, 	// ᴏ -> o
	{0x1d11, 0x6f}, 	// ᴑ -> o
	{0x1d13, 0x4f}, 	// ᴓ -> O
	{0x1d18, 0x50}, 	// ᴘ -> P
	{0x1d19, 0x52}, 	// ᴙ -> R
	{0x1d1a, 0x52}, 	// ᴚ -> R
	{0x1d1b, 0x54}, 	// ᴛ -> T
	{0x1d1c, 0x75}, 	// ᴜ -> u
	{0x1d1d, 0x75}, 	// ᴝ -> u
	{0x1d1e, 0x75}, 	// ᴞ -> u
	{0x1d1f, 0x6d}, 	// ᴟ -> m
	{0x1d20, 0x76}, 	// ᴠ -> v
	{0x1d21, 0x77}, 	// ᴡ -> w
	{0x1d22, 0x7a}, 	// ᴢ -> z
	{0x1d26, 0x72}, 	// ᴦ -> r
	{0x1d2c, 0x41}, 	// ᴬ -> A
	{0x1d2e, 0x42}, 	// ᴮ -> B
	{0x1d2f, 0x42}, 	// ᴯ -> B
	{0x1d30, 0x44}, 	// ᴰ -> D
	{0x1d31, 0x45}, 	// ᴱ -> E
	{0x1d32, 0x45}, 	// ᴲ -> E
	{0x1d33, 0x47}, 	// ᴳ -> G
	{0x1d34, 0x48}, 	// ᴴ -> H
	{0x1d35, 0x49}, 	// ᴵ -> I
	{0x1d36, 0x4a}, 	// ᴶ -> J
	{0x1d37, 0x4b}, 	// ᴷ -> K
	{0x1d38, 0x4c}, 	// ᴸ -> L
	{0x1d39, 0x4d}, 	// ᴹ -> M
	{0x1d3a, 0x4e}, 	// ᴺ -> N
	{0x1d3b, 0x4e}, 	// ᴻ -> N
	{0x1d3c, 0x4f}, 	// ᴼ -> O
	{0x1d3e, 0x50}, 	// ᴾ -> P
	{0x1d3f, 0x52}, 	// ᴿ -> R
	{0x1d40, 0x54}, 	// ᵀ -> T
	{0x1d41, 0x55}, 	// ᵁ -> U
	{0x1d42, 0x57}, 	// ᵂ -> W
	{0x1d43, 0x61}, 	// ᵃ -> a
	{0x1d44, 0x61}, 	// ᵄ -> a
	{0x1d45, 0x61}, 	// ᵅ -> a
	{0x1d47, 0x62}, 	// ᵇ -> b
	{0x1d48, 0x64}, 	// ᵈ -> d
	{0x1d49, 0x65}, 	// ᵉ -> e
	{0x1d4b, 0x65}, 	// ᵋ -> e
	{0x1d4c, 0x65}, 	// ᵌ -> e
	{0x1d4d, 0x67}, 	// ᵍ -> g
	{0x1d4e, 0x69}, 	// ᵎ -> i
	{0x1d4f, 0x6b}, 	// ᵏ -> k
	{0x1d50, 0x6d}, 	// ᵐ -> m
	{0x1d52, 0x6f}, 	// ᵒ -> o
	{0x1d56, 0x70}, 	// ᵖ -> p
	{0x1d57, 0x74}, 	// ᵗ -> t
	{0x1d58, 0x75}, 	// ᵘ -> u
	{0x1d59, 0x75}, 	// ᵙ -> u
	{0x1d5a, 0x6d}, 	// ᵚ -> m
	{0x1d5b, 0x76}, 	// ᵛ -> v
	{0x1d5d, 0x62}, 	// ᵝ -> b
	{0x1d5e, 0x67}, 	// ᵞ -> g
	{0x1d5f, 0x64}, 	// ᵟ -> d
	{0x1d60, 0x66}, 	// ᵠ -> f
	{0x1d62, 0x69}, 	// ᵢ -> i
	{0x1d63, 0x72}, 	// ᵣ -> r
	{0x1d64, 0x75}, 	// ᵤ -> u
	{0x1d65, 0x76}, 	// ᵥ -> v
	{0x1d66, 0x62}, 	// ᵦ -> b
	{0x1d67, 0x67}, 	// ᵧ -> g
	{0x1d68, 0x72}, 	// ᵨ -> r
	{0x1d69, 0x66}, 	// ᵩ -> f
	{0x1d6b, 0x75}, 	// ᵫ -> u
	{0x1d6c, 0x62}, 	// ᵬ -> b
	{0x1d6d, 0x64}, 	// ᵭ -> d
	{0x1d6e, 0x66}, 	// ᵮ -> f
	{0x1d6f, 0x72}, 	// ᵯ -> r
	{0x1d70, 0x6e}, 	// ᵰ -> n
	{0x1d71, 0x70}, 	// ᵱ -> p
	{0x1d72, 0x72}, 	// ᵲ -> r
	{0x1d73, 0x72}, 	// ᵳ -> r
	{0x1d74, 0x73}, 	// ᵴ -> s
	{0x1d75, 0x74}, 	// ᵵ -> t
	{0x1d76, 0x7a}, 	// ᵶ -> z
	{0x1d77, 0x67}, 	// ᵷ -> g
	{0x1d7b, 0x69}, 	// ᵻ -> i
	{0x1d7c, 0x69}, 	// ᵼ -> i
	{0x1d7d, 0x70}, 	// ᵽ -> p
	{0x1d7e, 0x75}, 	// ᵾ -> u
	{0x1d80, 0x62}, 	// ᶀ -> b
	{0x1d81, 0x64}, 	// ᶁ -> d
	{0x1d82, 0x66}, 	// ᶂ -> f
	{0x1d83, 0x67}, 	// ᶃ -> g
	{0x1d84, 0x6b}, 	// ᶄ -> k
	{0x1d85, 0x6c}, 	// ᶅ -> l
	{0x1d86, 0x6d}, 	// ᶆ -> m
	{0x1d87, 0x6e}, 	// ᶇ -> n
	{0x1d88, 0x70}, 	// ᶈ -> p
	{0x1d89, 0x72}, 	// ᶉ -> r
	{0x1d8a, 0x73}, 	// ᶊ -> s
	{0x1d8c, 0x79}, 	// ᶌ -> y
	{0x1d8d, 0x78}, 	// ᶍ -> x
	{0x1d8e, 0x7a}, 	// ᶎ -> z
	{0x1e00, 0x41}, 	// Ḁ -> A
	{0x1e01, 0x61}, 	// ḁ -> a
	{0x1e02, 0x42}, 	// Ḃ -> B
	{0x1e03, 0x62}, 	// ḃ -> b
	{0x1e04, 0x42}, 	// Ḅ -> B
	{0x1e05, 0x62}, 	// ḅ -> b
	{0x1e06, 0x42}, 	// Ḇ -> B
	{0x1e07, 0x62}, 	// ḇ -> b
	{0x1e08, 0x43}, 	// Ḉ -> C
	{0x1e09, 0x63}, 	// ḉ -> c
	{0x1e0a, 0x44}, 	// Ḋ -> D
	{0x1e0b, 0x64}, 	// ḋ -> d
	{0x1e0c, 0x44}, 	// Ḍ -> D
	{0x1e0d, 0x64}, 	// ḍ -> d
	{0x1e0e, 0x44}, 	// Ḏ -> D
	{0x1e0f, 0x64}, 	// ḏ -> d
	{0x1e10, 0x44}, 	// Ḑ -> D
	{0x1e11, 0x64}, 	// ḑ -> d
	{0x1e12, 0x44}, 	// Ḓ -> D
	{0x1e13, 0x64}, 	// ḓ -> d
	{0x1e14, 0x45}, 	// Ḕ -> E
	{0x1e15, 0x65}, 	// ḕ -> e
	{0x1e16, 0x45}, 	// Ḗ -> E
	{0x1e17, 0x65}, 	// ḗ -> e
	{0x1e18, 0x45}, 	// Ḙ -> E
	{0x1e19, 0x65}, 	// ḙ -> e
	{0x1e1a, 0x45}, 	// Ḛ -> E
	{0x1e1b, 0x65}, 	// ḛ -> e
	{0x1e1c, 0x45}, 	// Ḝ -> E
	{0x1e1d, 0x65}, 	// ḝ -> e
	{0x1e1e, 0x46}, 	// Ḟ -> F
	{0x1e1f, 0x66}, 	// ḟ -> f
	{0x1e20, 0x47}, 	// Ḡ -> G
	{0x1e21, 0x67}, 	// ḡ -> g
	{0x1e22, 0x48}, 	// Ḣ -> H
	{0x1e23, 0x68}, 	// ḣ -> h
	{0x1e24, 0x48}, 	// Ḥ -> H
	{0x1e25, 0x68}, 	// ḥ -> h
	{0x1e26, 0x48}, 	// Ḧ -> H
	{0x1e27, 0x68}, 	// ḧ -> h
	{0x1e28, 0x48}, 	// Ḩ -> H
	{0x1e29, 0x68}, 	// ḩ -> h
	{0x1e2a, 0x48}, 	// Ḫ -> H
	{0x1e2b, 0x68}, 	// ḫ -> h
	{0x1e2c, 0x49}, 	// Ḭ -> I
	{0x1e2d, 0x69}, 	// ḭ -> i
	{0x1e2e, 0x49}, 	// Ḯ -> I
	{0x1e2f, 0x69}, 	// ḯ -> i
	{0x1e30, 0x4b}, 	// Ḱ -> K
	{0x1e31, 0x6b}, 	// ḱ -> k
	{0x1e32, 0x4b}, 	// Ḳ -> K
	{0x1e33, 0x6b}, 	// ḳ -> k
	{0x1e34, 0x4b}, 	// Ḵ -> K
	{0x1e35, 0x6b}, 	// ḵ -> k
	{0x1e36, 0x4c}, 	// Ḷ -> L
	{0x1e37, 0x6c}, 	// ḷ -> l
	{0x1e38, 0x4c}, 	// Ḹ -> L
	{0x1e39, 0x6c}, 	// ḹ -> l
	{0x1e3a, 0x4c}, 	// Ḻ -> L
	{0x1e3b, 0x6c}, 	// ḻ -> l
	{0x1e3c, 0x4c}, 	// Ḽ -> L
	{0x1e3d, 0x6c}, 	// ḽ -> l
	{0x1e3e, 0x4d}, 	// Ḿ -> M
	{0x1e3f, 0x6d}, 	// ḿ -> m
	{0x1e40, 0x4d}, 	// Ṁ -> M
	{0x1e41, 0x6d}, 	// ṁ -> m
	{0x1e42, 0x4d}, 	// Ṃ -> M
	{0x1e43, 0x6d}, 	// ṃ -> m
	{0x1e44, 0x4e}, 	// Ṅ -> N
	{0x1e45, 0x6e}, 	// ṅ -> n
	{0x1e46, 0x4e}, 	// Ṇ -> N
	{0x1e47, 0x6e}, 	// ṇ -> n
	{0x1e48, 0x4e}, 	// Ṉ -> N
	{0x1e49, 0x6e}, 	// ṉ -> n
	{0x1e4a, 0x4e}, 	// Ṋ -> N
	{0x1e4b, 0x6e}, 	// ṋ -> n
	{0x1e4c, 0x4f}, 	// Ṍ -> O
	{0x1e4d, 0x6f}, 	// ṍ -> o
	{0x1e4e, 0x4f}, 	// Ṏ -> O
	{0x1e4f, 0x6f}, 	// ṏ -> o
	{0x1e50, 0x4f}, 	// Ṑ -> O
	{0x1e51, 0x6f}, 	// ṑ -> o
	{0x1e52, 0x4f}, 	// Ṓ -> O
	{0x1e53, 0x6f}, 	// ṓ -> o
	{0x1e54, 0x50}, 	// Ṕ -> P
	{0x1e55, 0x70}, 	// ṕ -> p
	{0x1e56, 0x50}, 	// Ṗ -> P
	{0x1e57, 0x70}, 	// ṗ -> p
	{0x1e58, 0x52}, 	// Ṙ -> R
	{0x1e59, 0x72}, 	// ṙ -> r
	{0x1e5a, 0x52}, 	// Ṛ -> R
	{0x1e5b, 0x72}, 	// ṛ -> r
	{0x1e5c, 0x52}, 	// Ṝ -> R
	{0x1e5d, 0x72}, 	// ṝ -> r
	{0x1e5e, 0x52}, 	// Ṟ -> R
	{0x1e5f, 0x72}, 	// ṟ -> r
	{0x1e60, 0x53}, 	// Ṡ -> S
	{0x1e61, 0x73}, 	// ṡ -> s
	{0x1e62, 0x53}, 	// Ṣ -> S
	{0x1e63, 0x73}, 	// ṣ -> s
	{0x1e64, 0x53}, 	// Ṥ -> S
	{0x1e65, 0x73}, 	// ṥ -> s
	{0x1e66, 0x53}, 	// Ṧ -> S
	{0x1e67, 0x73}, 	// ṧ -> s
	{0x1e68, 0x53}, 	// Ṩ -> S
	{0x1e69, 0x73}, 	// ṩ -> s
	{0x1e6a, 0x54}, 	// Ṫ -> T
	{0x1e6b, 0x74}, 	// ṫ -> t
	{0x1e6c, 0x54}, 	// Ṭ -> T
	{0x1e6d, 0x74}, 	// ṭ -> t
	{0x1e6e, 0x54}, 	// Ṯ -> T
	{0x1e6f, 0x74}, 	// ṯ -> t
	{0x1e70, 0x54}, 	// Ṱ -> T
	{0x1e71, 0x74}, 	// ṱ -> t
	{0x1e72, 0x55}, 	// Ṳ -> U
	{0x1e73, 0x75}, 	// ṳ -> u
	{0x1e74, 0x55}, 	// Ṵ -> U
	{0x1e75, 0x75}, 	// ṵ -> u
	{0x1e76, 0x55}, 	// Ṷ -> U
	{0x1e77, 0x75}, 	// ṷ -> u
	{0x1e78, 0x55}, 	// Ṹ -> U
	{0x1e79, 0x75}, 	// ṹ -> u
	{0x1e7a, 0x55}, 	// Ṻ -> U
	{0x1e7b, 0x75}, 	// ṻ -> u
	{0x1e7c, 0x56}, 	// Ṽ -> V
	{0x1e7d, 0x76}, 	// ṽ -> v
	{0x1e7e, 0x56}, 	// Ṿ -> V
	{0x1e7f, 0x76}, 	// ṿ -> v
	{0x1e80, 0x57}, 	// Ẁ -> W
	{0x1e81, 0x77}, 	// ẁ -> w
	{0x1e82, 0x57}, 	// Ẃ -> W
	{0x1e83, 0x77}, 	// ẃ -> w
	{0x1e84, 0x57}, 	// Ẅ -> W
	{0x1e85, 0x77}, 	// ẅ -> w
	{0x1e86, 0x57}, 	// Ẇ -> W
	{0x1e87, 0x77}, 	// ẇ -> w
	{0x1e88, 0x57}, 	// Ẉ -> W
	{0x1e89, 0x77}, 	// ẉ -> w
	{0x1e8a, 0x58}, 	// Ẋ -> X
	{0x1e8b, 0x78}, 	// ẋ -> x
	{0x1e8c, 0x58}, 	// Ẍ -> X
	{0x1e8d, 0x78}, 	// ẍ -> x
	{0x1e8e, 0x59}, 	// Ẏ -> Y
	{0x1e8f, 0x79}, 	// ẏ -> y
	{0x1e90, 0x5a}, 	// Ẑ -> Z
	{0x1e91, 0x7a}, 	// ẑ -> z
	{0x1e92, 0x5a}, 	// Ẓ -> Z
	{0x1e93, 0x7a}, 	// ẓ -> z
	{0x1e94, 0x5a}, 	// Ẕ -> Z
	{0x1e95, 0x7a}, 	// ẕ -> z
	{0x1e96, 0x68}, 	// ẖ -> h
	{0x1e97, 0x74}, 	// ẗ -> t
	{0x1e98, 0x77}, 	// ẘ -> w
	{0x1e99, 0x79}, 	// ẙ -> y
	{0x1e9a, 0x61}, 	// ẚ -> a
	{0x1e9b, 0x66}, 	// ẛ -> f
	{0x1e9d, 0x66}, 	// ẝ -> f
	{0x1ea0, 0x41}, 	// Ạ -> A
	{0x1ea1, 0x61}, 	// ạ -> a
	{0x1ea2, 0x41}, 	// Ả -> A
	{0x1ea3, 0x61}, 	// ả -> a
	{0x1ea4, 0x41}, 	// Ấ -> A
	{0x1ea5, 0x61}, 	// ấ -> a
	{0x1ea6, 0x41}, 	// Ầ -> A
	{0x1ea7, 0x61}, 	// ầ -> a
	{0x1ea8, 0x41}, 	// Ẩ -> A
	{0x1ea9, 0x61}, 	// ẩ -> a
	{0x1eaa, 0x41}, 	// Ẫ -> A
	{0x1eab, 0x61}, 	// ẫ -> a
	{0x1eac, 0x41}, 	// Ậ -> A
	{0x1ead, 0x61}, 	// ậ -> a
	{0x1eae, 0x41}, 	// Ắ -> A
	{0x1eaf, 0x61}, 	// ắ -> a
	{0x1eb0, 0x41}, 	// Ằ -> A
	{0x1eb1, 0x61}, 	// ằ -> a
	{0x1eb2, 0x41}, 	// Ẳ -> A
	{0x1eb3, 0x61}, 	// ẳ -> a
	{0x1eb4, 0x41}, 	// Ẵ -> A
	{0x1eb5, 0x61}, 	// ẵ -> a
	{0x1eb6, 0x41}, 	// Ặ -> A
	{0x1eb7, 0x61}, 	// ặ -> a
	{0x1eb8, 0x45}, 	// Ẹ -> E
	{0x1eb9, 0x65}, 	// ẹ -> e
	{0x1eba, 0x45}, 	// Ẻ -> E
	{0x1ebb, 0x65}, 	// ẻ -> e
	{0x1ebc, 0x45}, 	// Ẽ -> E
	{0x1ebd, 0x65}, 	// ẽ -> e
	{0x1ebe, 0x45}, 	// Ế -> E
	{0x1ebf, 0x65}, 	// ế -> e
	{0x1ec0, 0x45}, 	// Ề -> E
	{0x1ec1, 0x65}, 	// ề -> e
	{0x1ec2, 0x45}, 	// Ể -> E
	{0x1ec3, 0x65}, 	// ể -> e
	{0x1ec4, 0x45}, 	// Ễ -> E
	{0x1ec5, 0x65}, 	// ễ -> e
	{0x1ec6, 0x45}, 	// Ệ -> E
	{0x1ec7, 0x65}, 	// ệ -> e
	{0x1ec8, 0x49}, 	// Ỉ -> I
	{0x1ec9, 0x69}, 	// ỉ -> i
	{0x1eca, 0x49}, 	// Ị -> I
	{0x1ecb, 0x69}, 	// ị -> i
	{0x1ecc, 0x4f}, 	// Ọ -> O
	{0x1ecd, 0x6f}, 	// ọ -> o
	{0x1ece, 0x4f}, 	// Ỏ -> O
	{0x1ecf, 0x6f}, 	// ỏ -> o
	{0x1ed0, 0x4f}, 	// Ố -> O
	{0x1ed1, 0x6f}, 	// ố -> o
	{0x1ed2, 0x4f}, 	// Ồ -> O
	{0x1ed3, 0x6f}, 	// ồ -> o
	{0x1ed4, 0x4f}, 	// Ổ -> O
	{0x1ed5, 0x6f}, 	// ổ -> o
	{0x1ed6, 0x4f}, 	// Ỗ -> O
	{0x1ed7, 0x6f}, 	// ỗ -> o
	{0x1ed8, 0x4f}, 	// Ộ -> O
	{0x1ed9, 0x6f}, 	// ộ -> o
	{0x1eda, 0x4f}, 	// Ớ -> O
	{0x1edb, 0x6f}, 	// ớ -> o
	{0x1edc, 0x4f}, 	// Ờ -> O
	{0x1edd, 0x6f}, 	// ờ -> o
	{0x1ede, 0x4f}, 	// Ở -> O
	{0x1edf, 0x6f}, 	// ở -> o
	{0x1ee0, 0x4f}, 	// Ỡ -> O
	{0x1ee1, 0x6f}, 	// ỡ -> o
	{0x1ee2, 0x4f}, 	// Ợ -> O
	{0x1ee3, 0x6f}, 	// ợ -> o
	{0x1ee4, 0x55}, 	// Ụ -> U
	{0x1ee5, 0x75}, 	// ụ -> u
	{0x1ee6, 0x55}, 	// Ủ -> U
	{0x1ee7, 0x75}, 	// ủ -> u
	{0x1ee8, 0x55}, 	// Ứ -> U
	{0x1ee9, 0x75}, 	// ứ -> u
	{0x1eea, 0x55}, 	// Ừ -> U
	{0x1eeb, 0x75}, 	// ừ -> u
	{0x1eec, 0x55}, 	// Ử -> U
	{0x1eed, 0x75}, 	// ử -> u
	{0x1eee, 0x55}, 	// Ữ -> U
	{0x1eef, 0x75}, 	// ữ -> u
	{0x1ef0, 0x55}, 	// Ự -> U
	{0x1ef1, 0x75}, 	// ự -> u
	{0x1ef2, 0x59}, 	// Ỳ -> Y
	{0x1ef3, 0x79}, 	// ỳ -> y
	{0x1ef4, 0x59}, 	// Ỵ -> Y
	{0x1ef5, 0x79}, 	// ỵ -> y
	{0x1ef6, 0x59}, 	// Ỷ -> Y
	{0x1ef7, 0x79}, 	// ỷ -> y
	{0x1ef8, 0x59}, 	// Ỹ -> Y
	{0x1ef9, 0x79}, 	// ỹ -> y
	{0x1eff, 0x79}, 	// ỿ -> y
	{0x1f00, 0x61}, 	// ἀ -> a
	{0x1f01, 0x61}, 	// ἁ -> a
	{0x1f02, 0x61}, 	// ἂ -> a
	{0x1f03, 0x61}, 	// ἃ -> a
	{0x1f04, 0x61}, 	// ἄ -> a
	{0x1f05, 0x61}, 	// ἅ -> a
	{0x1f06, 0x61}, 	// ἆ -> a
	{0x1f07, 0x61}, 	// ἇ -> a
	{0x1f08, 0x41}, 	// Ἀ -> A
	{0x1f09, 0x41}, 	// Ἁ -> A
	{0x1f0a, 0x41}, 	// Ἂ -> A
	{0x1f0b, 0x41}, 	// Ἃ -> A
	{0x1f0c, 0x41}, 	// Ἄ -> A
	{0x1f0d, 0x41}, 	// Ἅ -> A
	{0x1f0e, 0x41}, 	// Ἆ -> A
	{0x1f0f, 0x41}, 	// Ἇ -> A
	{0x1f10, 0x65}, 	// ἐ -> e
	{0x1f11, 0x65}, 	// ἑ -> e
	{0x1f12, 0x65}, 	// ἒ -> e
	{0x1f13, 0x65}, 	// ἓ -> e
	{0x1f14, 0x65}, 	// ἔ -> e
	{0x1f15, 0x65}, 	// ἕ -> e
	{0x1f18, 0x45}, 	// Ἐ -> E
	{0x1f19, 0x45}, 	// Ἑ -> E
	{0x1f1a, 0x45}, 	// Ἒ -> E
	{0x1f1b, 0x45}, 	// Ἓ -> E
	{0x1f1c, 0x45}, 	// Ἔ -> E
	{0x1f1d, 0x45}, 	// Ἕ -> E
	{0x1f20, 0x65}, 	// ἠ -> e
	{0x1f21, 0x65}, 	// ἡ -> e
	{0x1f22, 0x65}, 	// ἢ -> e
	{0x1f23, 0x65}, 	// ἣ -> e
	{0x1f24, 0x65}, 	// ἤ -> e
	{0x1f25, 0x65}, 	// ἥ -> e
	{0x1f26, 0x65}, 	// ἦ -> e
	{0x1f27, 0x6e}, 	// ἧ -> n
	{0x1f28, 0x45}, 	// Ἠ -> E
	{0x1f29, 0x45}, 	// Ἡ -> E
	{0x1f2a, 0x45}, 	// Ἢ -> E
	{0x1f2b, 0x45}, 	// Ἣ -> E
	{0x1f2c, 0x45}, 	// Ἤ -> E
	{0x1f2d, 0x45}, 	// Ἥ -> E
	{0x1f2e, 0x45}, 	// Ἦ -> E
	{0x1f2f, 0x45}, 	// Ἧ -> E
	{0x1f30, 0x69}, 	// ἰ -> i
	{0x1f31, 0x69}, 	// ἱ -> i
	{0x1f32, 0x69}, 	// ἲ -> i
	{0x1f33, 0x69}, 	// ἳ -> i
	{0x1f34, 0x69}, 	// ἴ -> i
	{0x1f35, 0x69}, 	// ἵ -> i
	{0x1f36, 0x69}, 	// ἶ -> i
	{0x1f37, 0x69}, 	// ἷ -> i
	{0x1f38, 0x49}, 	// Ἰ -> I
	{0x1f39, 0x49}, 	// Ἱ -> I
	{0x1f3a, 0x49}, 	// Ἲ -> I
	{0x1f3b, 0x49}, 	// Ἳ -> I
	{0x1f3c, 0x49}, 	// Ἴ -> I
	{0x1f3d, 0x49}, 	// Ἵ -> I
	{0x1f3e, 0x49}, 	// Ἶ -> I
	{0x1f3f, 0x49}, 	// Ἷ -> I
	{0x1f40, 0x6f}, 	// ὀ -> o
	{0x1f41, 0x6f}, 	// ὁ -> o
	{0x1f42, 0x6f}, 	// ὂ -> o
	{0x1f43, 0x6f}, 	// ὃ -> o
	{0x1f44, 0x6f}, 	// ὄ -> o
	{0x1f45, 0x6f}, 	// ὅ -> o
	{0x1f48, 0x4f}, 	// Ὀ -> O
	{0x1f49, 0x4f}, 	// Ὁ -> O
	{0x1f4a, 0x4f}, 	// Ὂ -> O
	{0x1f4b, 0x4f}, 	// Ὃ -> O
	{0x1f4c, 0x4f}, 	// Ὄ -> O
	{0x1f4d, 0x4f}, 	// Ὅ -> O
	{0x1f50, 0x75}, 	// ὐ -> u
	{0x1f51, 0x75}, 	// ὑ -> u
	{0x1f52, 0x75}, 	// ὒ -> u
	{0x1f53, 0x75}, 	// ὓ -> u
	{0x1f54, 0x75}, 	// ὔ -> u
	{0x1f55, 0x75}, 	// ὕ -> u
	{0x1f56, 0x75}, 	// ὖ -> u
	{0x1f57, 0x75}, 	// ὗ -> u
	{0x1f59, 0x55}, 	// Ὑ -> U
	{0x1f5b, 0x55}, 	// Ὓ -> U
	{0x1f5d, 0x55}, 	// Ὕ -> U
	{0x1f5f, 0x55}, 	// Ὗ -> U
	{0x1f60, 0x6f}, 	// ὠ -> o
	{0x1f61, 0x6f}, 	// ὡ -> o
	{0x1f62, 0x6f}, 	// ὢ -> o
	{0x1f63, 0x6f}, 	// ὣ -> o
	{0x1f64, 0x6f}, 	// ὤ -> o
	{0x1f65, 0x6f}, 	// ὥ -> o
	{0x1f66, 0x6f}, 	// ὦ -> o
	{0x1f67, 0x6f}, 	// ὧ -> o
	{0x1f68, 0x4f}, 	// Ὠ -> O
	{0x1f69, 0x4f}, 	// Ὡ -> O
	{0x1f6a, 0x4f}, 	// Ὢ -> O
	{0x1f6b, 0x4f}, 	// Ὣ -> O
	{0x1f6c, 0x4f}, 	// Ὤ -> O
	{0x1f6d, 0x4f}, 	// Ὥ -> O
	{0x1f6e, 0x4f}, 	// Ὦ -> O
	{0x1f6f, 0x4f}, 	// Ὧ -> O
	{0x1f70, 0x61}, 	// ὰ -> a
	{0x1f71, 0x61}, 	// ά -> a
	{0x1f72, 0x65}, 	// ὲ -> e
	{0x1f73, 0x65}, 	// έ -> e
	{0x1f74, 0x65}, 	// ὴ -> e
	{0x1f75, 0x65}, 	// ή -> e
	{0x1f76, 0x69}, 	// ὶ -> i
	{0x1f77, 0x69}, 	// ί -> i
	{0x1f78, 0x6f}, 	// ὸ -> o
	{0x1f79, 0x6f}, 	// ό -> o
	{0x1f7a, 0x75}, 	// ὺ -> u
	{0x1f7b, 0x75}, 	// ύ -> u
	{0x1f7c, 0x6f}, 	// ὼ -> o
	{0x1f7d, 0x6f}, 	// ώ -> o
	{0x1f80, 0x61}, 	// ᾀ -> a
	{0x1f81, 0x61}, 	// ᾁ -> a
	{0x1f82, 0x61}, 	// ᾂ -> a
	{0x1f83, 0x61}, 	// ᾃ -> a
	{0x1f84, 0x61}, 	// ᾄ -> a
	{0x1f85, 0x61}, 	// ᾅ -> a
	{0x1f86, 0x61}, 	// ᾆ -> a
	{0x1f87, 0x61}, 	// ᾇ -> a
	{0x1f88, 0x41}, 	// ᾈ -> A
	{0x1f89, 0x41}, 	// ᾉ -> A
	{0x1f8a, 0x41}, 	// ᾊ -> A
	{0x1f8b, 0x41}, 	// ᾋ -> A
	{0x1f8c, 0x41}, 	// ᾌ -> A
	{0x1f8d, 0x41}, 	// ᾍ -> A
	{0x1f8e, 0x41}, 	// ᾎ -> A
	{0x1f8f, 0x41}, 	// ᾏ -> A
	{0x1f90, 0x65}, 	// ᾐ -> e
	{0x1f91, 0x65}, 	// ᾑ -> e
	{0x1f92, 0x65}, 	// ᾒ -> e
	{0x1f93, 0x65}, 	// ᾓ -> e
	{0x1f94, 0x65}, 	// ᾔ -> e
	{0x1f95, 0x65}, 	// ᾕ -> e
	{0x1f96, 0x65}, 	// ᾖ -> e
	{0x1f97, 0x65}, 	// ᾗ -> e
	{0x1f98, 0x45}, 	// ᾘ -> E
	{0x1f99, 0x45}, 	// ᾙ -> E
	{0x1f9a, 0x45}, 	// ᾚ -> E
	{0x1f9b, 0x45}, 	// ᾛ -> E
	{0x1f9c, 0x45}, 	// ᾜ -> E
	{0x1f9d, 0x45}, 	// ᾝ -> E
	{0x1f9e, 0x45}, 	// ᾞ -> E
	{0x1f9f, 0x45}, 	// ᾟ -> E
	{0x1fa0, 0x6f}, 	// ᾠ -> o
	{0x1fa1, 0x6f}, 	// ᾡ -> o
	{0x1fa2, 0x6f}, 	// ᾢ -> o
	{0x1fa3, 0x6f}, 	// ᾣ -> o
	{0x1fa4, 0x6f}, 	// ᾤ -> o
	{0x1fa5, 0x6f}, 	// ᾥ -> o
	{0x1fa6, 0x6f}, 	// ᾦ -> o
	{0x1fa7, 0x6f}, 	// ᾧ -> o
	{0x1fa8, 0x4f}, 	// ᾨ -> O
	{0x1fa9, 0x4f}, 	// ᾩ -> O
	{0x1faa, 0x4f}, 	// ᾪ -> O
	{0x1fab, 0x4f}, 	// ᾫ -> O
	{0x1fac, 0x4f}, 	// ᾬ -> O
	{0x1fad, 0x4f}, 	// ᾭ -> O
	{0x1fae, 0x4f}, 	// ᾮ -> O
	{0x1faf, 0x4f}, 	// ᾯ -> O
	{0x1fb0, 0x61}, 	// ᾰ -> a
	{0x1fb1, 0x61}, 	// ᾱ -> a
	{0x1fb2, 0x61}, 	// ᾲ -> a
	{0x1fb3, 0x61}, 	// ᾳ -> a
	{0x1fb4, 0x61}, 	// ᾴ -> a
	{0x1fb6, 0x61}, 	// ᾶ -> a
	{0x1fb7, 0x61}, 	// ᾷ -> a
	{0x1fb8, 0x41}, 	// Ᾰ -> A
	{0x1fb9, 0x41}, 	// Ᾱ -> A
	{0x1fba, 0x41}, 	// Ὰ -> A
	{0x1fbb, 0x41}, 	// Ά -> A
	{0x1fbc, 0x41}, 	// ᾼ -> A
	{0x1fbd, 0x27}, 	// ᾽ -> '
	{0x1fbe, 0x69}, 	// ι -> i
	{0x1fbf, 0x27}, 	// ᾿ -> '
	{0x1fc0, 0x7e}, 	// ῀ -> ~
	{0x1fc2, 0x65}, 	// ῂ -> e
	{0x1fc3, 0x65}, 	// ῃ -> e
	{0x1fc4, 0x65}, 	// ῄ -> e
	{0x1fc6, 0x65}, 	// ῆ -> e
	{0x1fc7, 0x65}, 	// ῇ -> e
	{0x1fc8, 0x45}, 	// Ὲ -> E
	{0x1fc9, 0x45}, 	// Έ -> E
	{0x1fca, 0x45}, 	// Ὴ -> E
	{0x1fcb, 0x45}, 	// Ή -> E
	{0x1fcc, 0x45}, 	// ῌ -> E
	{0x1fd0, 0x69}, 	// ῐ -> i
	{0x1fd1, 0x69}, 	// ῑ -> i
	{0x1fd2, 0x69}, 	// ῒ -> i
	{0x1fd3, 0x69}, 	// ΐ -> i
	{0x1fd6, 0x69}, 	// ῖ -> i
	{0x1fd7, 0x69}, 	// ῗ -> i
	{0x1fd8, 0x49}, 	// Ῐ -> I
	{0x1fd9, 0x49}, 	// Ῑ -> I
	{0x1fda, 0x49}, 	// Ὶ -> I
	{0x1fdb, 0x49}, 	// Ί -> I
	{0x1fe0, 0x75}, 	// ῠ -> u
	{0x1fe1, 0x75}, 	// ῡ -> u
	{0x1fe2, 0x75}, 	// ῢ -> u
	{0x1fe3, 0x75}, 	// ΰ -> u
	{0x1fe4, 0x70}, 	// ῤ -> p
	{0x1fe5, 0x70}, 	// ῥ -> p
	{0x1fe6, 0x75}, 	// ῦ -> u
	{0x1fe7, 0x75}, 	// ῧ -> u
	{0x1fe8, 0x55}, 	// Ῠ -> U
	{0x1fe9, 0x55}, 	// Ῡ -> U
	{0x1fea, 0x55}, 	// Ὺ -> U
	{0x1feb, 0x55}, 	// Ύ -> U
	{0x1fec, 0x52}, 	// Ῥ -> R
	{0x1fef, 0x60}, 	// ` -> `
	{0x1ff2, 0x6f}, 	// ῲ -> o
	{0x1ff3, 0x6f}, 	// ῳ -> o
	{0x1ff4, 0x6f}, 	// ῴ -> o
	{0x1ff6, 0x6f}, 	// ῶ -> o
	{0x1ff7, 0x6f}, 	// ῷ -> o
	{0x1ff8, 0x4f}, 	// Ὸ -> O
	{0x1ff9, 0x4f}, 	// Ό -> O
	{0x1ffa, 0x4f}, 	// Ὼ -> O
	{0x1ffb, 0x4f}, 	// Ώ -> O
	{0x1ffc, 0x4f}, 	// ῼ -> O
	{0x1ffd, 0x27}, 	// ´ -> '
	{0x1ffe, 0x60}, 	// ῾ -> `
	{0x2000, 0x20}, 	//   ->  
	{0x2001, 0x20}, 	//   ->  
	{0x2002, 0x20}, 	//   ->  
	{0x2003, 0x20}, 	//   ->  
	{0x2004, 0x20}, 	//   ->  
	{0x2005, 0x20}, 	//   ->  
	{0x2006, 0x20}, 	//   ->  
	{0x2007, 0x20}, 	//   ->  
	{0x2008, 0x20}, 	//   ->  
	{0x2009, 0x20}, 	//   ->  
	{0x200a, 0x20}, 	//   ->  
	{0x200b, 0x20}, 	// ​ ->  
	{0x2010, 0x2d}, 	// ‐ -> -
	{0x2011, 0x2d}, 	// ‑ -> -
	{0x2012, 0x2d}, 	// ‒ -> -
	{0x2013, 0x2d}, 	// – -> -
	{0x2016, 0x6c}, 	// ‖ -> l
	{0x2017, 0x5f}, 	// ‗ -> _
	{0x2018, 0x27}, 	// ‘ -> '
	{0x2019, 0x27}, 	// ’ -> '
	{0x201a, 0x2c}, 	// ‚ -> ,
	{0x201b, 0x27}, 	// ‛ -> '
	{0x201c, 0x22}, 	// “ -> "
	{0x201d, 0x22}, 	// ” -> "
	{0x201f, 0x22}, 	// ‟ -> "
	{0x2020, 0x2b}, 	// † -> +
	{0x2022, 0x2a}, 	// • -> *
	{0x2024, 0x2e}, 	// ․ -> .
	{0x2027, 0x2e}, 	// ‧ -> .
	{0x202f, 0x20}, 	//   ->  
	{0x2032, 0x27}, 	// ′ -> '
	{0x2035, 0x60}, 	// ‵ -> `
	{0x2038, 0x5e}, 	// ‸ -> ^
	{0x2039, 0x3c}, 	// ‹ -> <
	{0x203a, 0x3e}, 	// › -> >
	{0x203b, 0x2a}, 	// ※ -> *
	{0x203e, 0x2d}, 	// ‾ -> -
	{0x203f, 0x5f}, 	// ‿ -> _
	{0x2040, 0x2d}, 	// ⁀ -> -
	{0x2041, 0x5e}, 	// ⁁ -> ^
	{0x2044, 0x2f}, 	// ⁄ -> /
	{0x204a, 0x26}, 	// ⁊ -> &
	{0x204e, 0x2a}, 	// ⁎ -> *
	{0x2052, 0x25}, 	// ⁒ -> %
	{0x2053, 0x7e}, 	// ⁓ -> ~
	{0x205f, 0x20}, 	//   ->  
	{0x2070, 0x30}, 	// ⁰ -> 0
	{0x2071, 0x69}, 	// ⁱ -> i
	{0x2074, 0x34}, 	// ⁴ -> 4
	{0x2075, 0x35}, 	// ⁵ -> 5
	{0x2076, 0x36}, 	// ⁶ -> 6
	{0x2077, 0x37}, 	// ⁷ -> 7
	{0x2078, 0x38}, 	// ⁸ -> 8
	{0x2079, 0x39}, 	// ⁹ -> 9
	{0x207a, 0x2b}, 	// ⁺ -> +
	{0x207b, 0x2d}, 	// ⁻ -> -
	{0x207c, 0x3d}, 	// ⁼ -> =
	{0x207d, 0x28}, 	// ⁽ -> (
	{0x207e, 0x29}, 	// ⁾ -> )
	{0x207f, 0x6e}, 	// ⁿ -> n
	{0x2080, 0x30}, 	// ₀ -> 0
	{0x2081, 0x31}, 	// ₁ -> 1
	{0x2082, 0x32}, 	// ₂ -> 2
	{0x2083, 0x33}, 	// ₃ -> 3
	{0x2084, 0x34}, 	// ₄ -> 4
	{0x2085, 0x35}, 	// ₅ -> 5
	{0x2086, 0x36}, 	// ₆ -> 6
	{0x2087, 0x37}, 	// ₇ -> 7
	{0x2088, 0x38}, 	// ₈ -> 8
	{0x2089, 0x39}, 	// ₉ -> 9
	{0x208a, 0x2b}, 	// ₊ -> +
	{0x208b, 0x2d}, 	// ₋ -> -
	{0x208c, 0x3d}, 	// ₌ -> =
	{0x208d, 0x28}, 	// ₍ -> (
	{0x208e, 0x29}, 	// ₎ -> )
	{0x2090, 0x61}, 	// ₐ -> a
	{0x2091, 0x65}, 	// ₑ -> e
	{0x2092, 0x6f}, 	// ₒ -> o
	{0x2093, 0x78}, 	// ₓ -> x
	{0x2095, 0x68}, 	// ₕ -> h
	{0x2096, 0x6b}, 	// ₖ -> k
	{0x2097, 0x6c}, 	// ₗ -> l
	{0x2098, 0x6d}, 	// ₘ -> m
	{0x2099, 0x6e}, 	// ₙ -> n
	{0x209a, 0x70}, 	// ₚ -> p
	{0x209b, 0x73}, 	// ₛ -> s
	{0x209c, 0x74}, 	// ₜ -> t
	{0x20a1, 0x43}, 	// ₡ -> C
	{0x20a4, 0x4c}, 	// ₤ -> L
	{0x20a5, 0x72}, 	// ₥ -> r
	{0x20a6, 0x4e}, 	// ₦ -> N
	{0x20a8, 0x52}, 	// ₨ -> R
	{0x20a9, 0x57}, 	// ₩ -> W
	{0x20ab, 0x64}, 	// ₫ -> d
	{0x20ad, 0x4b}, 	// ₭ -> K
	{0x20ae, 0x54}, 	// ₮ -> T
	{0x20b1, 0x50}, 	// ₱ -> P
	{0x20b2, 0x47}, 	// ₲ -> G
	{0x20b3, 0x41}, 	// ₳ -> A
	{0x20b6, 0x6c}, 	// ₶ -> l
	{0x20b8, 0x54}, 	// ₸ -> T
	{0x20ba, 0x4c}, 	// ₺ -> L
	{0x20bb, 0x4d}, 	// ₻ -> M
	{0x20bc, 0x6d}, 	// ₼ -> m
	{0x20bd, 0x52}, 	// ₽ -> R
	{0x20be, 0x6c}, 	// ₾ -> l
	{0x2100, 0x61}, 	// ℀ -> a
	{0x2101, 0x61}, 	// ℁ -> a
	{0x2102, 0x43}, 	// ℂ -> C
	{0x2103, 0x43}, 	// ℃ -> C
	{0x2105, 0x63}, 	// ℅ -> c
	{0x2106, 0x63}, 	// ℆ -> c
	{0x2109, 0x46}, 	// ℉ -> F
	{0x210a, 0x67}, 	// ℊ -> g
	{0x210b, 0x48}, 	// ℋ -> H
	{0x210c, 0x48}, 	// ℌ -> H
	{0x210d, 0x48}, 	// ℍ -> H
	{0x210e, 0x68}, 	// ℎ -> h
	{0x210f, 0x68}, 	// ℏ -> h
	{0x2110, 0x6c}, 	// ℐ -> l
	{0x2111, 0x6c}, 	// ℑ -> l
	{0x2112, 0x4c}, 	// ℒ -> L
	{0x2113, 0x6c}, 	// ℓ -> l
	{0x2115, 0x4e}, 	// ℕ -> N
	{0x2116, 0x4e}, 	// № -> N
	{0x2119, 0x50}, 	// ℙ -> P
	{0x211a, 0x51}, 	// ℚ -> Q
	{0x211b, 0x52}, 	// ℛ -> R
	{0x211c, 0x52}, 	// ℜ -> R
	{0x211d, 0x52}, 	// ℝ -> R
	{0x2121, 0x54}, 	// ℡ -> T
	{0x2124, 0x5a}, 	// ℤ -> Z
	{0x2128, 0x5a}, 	// ℨ -> Z
	{0x212a, 0x4b}, 	// K -> K
	{0x212b, 0x41}, 	// Å -> A
	{0x212c, 0x42}, 	// ℬ -> B
	{0x212d, 0x43}, 	// ℭ -> C
	{0x212e, 0x65}, 	// ℮ -> e
	{0x212f, 0x65}, 	// ℯ -> e
	{0x2130, 0x45}, 	// ℰ -> E
	{0x2131, 0x46}, 	// ℱ -> F
	{0x2132, 0x46}, 	// Ⅎ -> F
	{0x2133, 0x4d}, 	// ℳ -> M
	{0x2134, 0x6f}, 	// ℴ -> o
	{0x2139, 0x69}, 	// ℹ -> i
	{0x213b, 0x46}, 	// ℻ -> F
	{0x213c, 0x6e}, 	// ℼ -> n
	{0x213d, 0x79}, 	// ℽ -> y
	{0x2145, 0x44}, 	// ⅅ -> D
	{0x2146, 0x64}, 	// ⅆ -> d
	{0x2147, 0x65}, 	// ⅇ -> e
	{0x2148, 0x69}, 	// ⅈ -> i
	{0x2149, 0x6a}, 	// ⅉ -> j
	{0x214e, 0x46}, 	// ⅎ -> F
	{0x2160, 0x6c}, 	// Ⅰ -> l
	{0x2161, 0x6c}, 	// Ⅱ -> l
	{0x2162, 0x6c}, 	// Ⅲ -> l
	{0x2163, 0x6c}, 	// Ⅳ -> l
	{0x2164, 0x56}, 	// Ⅴ -> V
	{0x2165, 0x56}, 	// Ⅵ -> V
	{0x2166, 0x56}, 	// Ⅶ -> V
	{0x2167, 0x56}, 	// Ⅷ -> V
	{0x2168, 0x6c}, 	// Ⅸ -> l
	{0x2169, 0x58}, 	// Ⅹ -> X
	{0x216a, 0x58}, 	// Ⅺ -> X
	{0x216b, 0x58}, 	// Ⅻ -> X
	{0x216c, 0x4c}, 	// Ⅼ -> L
	{0x216d, 0x43}, 	// Ⅽ -> C
	{0x216e, 0x44}, 	// Ⅾ -> D
	{0x216f, 0x4d}, 	// Ⅿ -> M
	{0x2170, 0x69}, 	// ⅰ -> i
	{0x2171, 0x69}, 	// ⅱ -> i
	{0x2172, 0x69}, 	// ⅲ -> i
	{0x2173, 0x69}, 	// ⅳ -> i
	{0x2174, 0x76}, 	// ⅴ -> v
	{0x2175, 0x76}, 	// ⅵ -> v
	{0x2176, 0x76}, 	// ⅶ -> v
	{0x2177, 0x76}, 	// ⅷ -> v
	{0x2178, 0x69}, 	// ⅸ -> i
	{0x2179, 0x78}, 	// ⅹ -> x
	{0x217a, 0x78}, 	// ⅺ -> x
	{0x217b, 0x78}, 	// ⅻ -> x
	{0x217c, 0x6c}, 	// ⅼ -> l
	{0x217d, 0x63}, 	// ⅽ -> c
	{0x217e, 0x64}, 	// ⅾ -> d
	{0x217f, 0x72}, 	// ⅿ -> r
	{0x2183, 0x29}, 	// Ↄ -> )
	{0x2190, 0x2d}, 	// ← -> -
	{0x2191, 0x7c}, 	// ↑ -> |
	{0x2192, 0x2d}, 	// → -> -
	{0x2193, 0x7c}, 	// ↓ -> |
	{0x2194, 0x2d}, 	// ↔ -> -
	{0x2195, 0x7c}, 	// ↕ -> |
	{0x2196, 0x5c}, 	// ↖ -> backslash
	{0x2197, 0x2f}, 	// ↗ -> /
	{0x2198, 0x5c}, 	// ↘ -> backslash
	{0x2199, 0x2f}, 	// ↙ -> /
	{0x219a, 0x2d}, 	// ↚ -> -
	{0x219b, 0x2d}, 	// ↛ -> -
	{0x219c, 0x7e}, 	// ↜ -> ~
	{0x219d, 0x7e}, 	// ↝ -> ~
	{0x219e, 0x2d}, 	// ↞ -> -
	{0x219f, 0x7c}, 	// ↟ -> |
	{0x21a0, 0x2d}, 	// ↠ -> -
	{0x21a1, 0x7c}, 	// ↡ -> |
	{0x21a2, 0x2d}, 	// ↢ -> -
	{0x21a3, 0x2d}, 	// ↣ -> -
	{0x21a4, 0x2d}, 	// ↤ -> -
	{0x21a5, 0x7c}, 	// ↥ -> |
	{0x21a6, 0x2d}, 	// ↦ -> -
	{0x21a7, 0x7c}, 	// ↧ -> |
	{0x21a8, 0x7c}, 	// ↨ -> |
	{0x21a9, 0x2d}, 	// ↩ -> -
	{0x21aa, 0x2d}, 	// ↪ -> -
	{0x21ab, 0x2d}, 	// ↫ -> -
	{0x21ac, 0x2d}, 	// ↬ -> -
	{0x21ad, 0x2d}, 	// ↭ -> -
	{0x21ae, 0x2d}, 	// ↮ -> -
	{0x21af, 0x7c}, 	// ↯ -> |
	{0x21b0, 0x7c}, 	// ↰ -> |
	{0x21b1, 0x7c}, 	// ↱ -> |
	{0x21b2, 0x7c}, 	// ↲ -> |
	{0x21b3, 0x7c}, 	// ↳ -> |
	{0x21b4, 0x7c}, 	// ↴ -> |
	{0x21b5, 0x7c}, 	// ↵ -> |
	{0x21b6, 0x5e}, 	// ↶ -> ^
	{0x21b7, 0x56}, 	// ↷ -> V
	{0x21b8, 0x5c}, 	// ↸ -> backslash
	{0x21b9, 0x3d}, 	// ↹ -> =
	{0x21ba, 0x56}, 	// ↺ -> V
	{0x21bb, 0x5e}, 	// ↻ -> ^
	{0x21bc, 0x2d}, 	// ↼ -> -
	{0x21bd, 0x2d}, 	// ↽ -> -
	{0x21be, 0x7c}, 	// ↾ -> |
	{0x21bf, 0x7c}, 	// ↿ -> |
	{0x21c0, 0x2d}, 	// ⇀ -> -
	{0x21c1, 0x2d}, 	// ⇁ -> -
	{0x21c2, 0x7c}, 	// ⇂ -> |
	{0x21c3, 0x7c}, 	// ⇃ -> |
	{0x21c4, 0x3d}, 	// ⇄ -> =
	{0x21c5, 0x7c}, 	// ⇅ -> |
	{0x21c6, 0x3d}, 	// ⇆ -> =
	{0x21c7, 0x3d}, 	// ⇇ -> =
	{0x21c8, 0x7c}, 	// ⇈ -> |
	{0x21c9, 0x3d}, 	// ⇉ -> =
	{0x21ca, 0x7c}, 	// ⇊ -> |
	{0x21cb, 0x3d}, 	// ⇋ -> =
	{0x21cc, 0x3d}, 	// ⇌ -> =
	{0x21cd, 0x3d}, 	// ⇍ -> =
	{0x21ce, 0x3d}, 	// ⇎ -> =
	{0x21cf, 0x3d}, 	// ⇏ -> =
	{0x21d0, 0x3d}, 	// ⇐ -> =
	{0x21d1, 0x7c}, 	// ⇑ -> |
	{0x21d2, 0x3d}, 	// ⇒ -> =
	{0x21d3, 0x7c}, 	// ⇓ -> |
	{0x21d4, 0x3d}, 	// ⇔ -> =
	{0x21d5, 0x7c}, 	// ⇕ -> |
	{0x21d6, 0x5c}, 	// ⇖ -> backslash
	{0x21d7, 0x2f}, 	// ⇗ -> /
	{0x21d8, 0x5c}, 	// ⇘ -> backslash
	{0x21d9, 0x2f}, 	// ⇙ -> /
	{0x21da, 0x3d}, 	// ⇚ -> =
	{0x21db, 0x3d}, 	// ⇛ -> =
	{0x21dc, 0x7e}, 	// ⇜ -> ~
	{0x21dd, 0x7e}, 	// ⇝ -> ~
	{0x21de, 0x7c}, 	// ⇞ -> |
	{0x21df, 0x7c}, 	// ⇟ -> |
	{0x21e0, 0x2d}, 	// ⇠ -> -
	{0x21e1, 0x7c}, 	// ⇡ -> |
	{0x21e2, 0x2d}, 	// ⇢ -> -
	{0x21e3, 0x7c}, 	// ⇣ -> |
	{0x21e4, 0x2d}, 	// ⇤ -> -
	{0x21e5, 0x2d}, 	// ⇥ -> -
	{0x21e6, 0x2d}, 	// ⇦ -> -
	{0x21e7, 0x7c}, 	// ⇧ -> |
	{0x21e8, 0x2d}, 	// ⇨ -> -
	{0x21e9, 0x7c}, 	// ⇩ -> |
	{0x21ea, 0x7c}, 	// ⇪ -> |
	{0x21eb, 0x7c}, 	// ⇫ -> |
	{0x21ec, 0x7c}, 	// ⇬ -> |
	{0x21ed, 0x7c}, 	// ⇭ -> |
	{0x21ee, 0x7c}, 	// ⇮ -> |
	{0x21ef, 0x7c}, 	// ⇯ -> |
	{0x21f0, 0x2d}, 	// ⇰ -> -
	{0x21f1, 0x5c}, 	// ⇱ -> backslash
	{0x21f2, 0x5c}, 	// ⇲ -> backslash
	{0x21f3, 0x7c}, 	// ⇳ -> |
	{0x2212, 0x2d}, 	// − -> -
	{0x2215, 0x2f}, 	// ∕ -> /
	{0x2216, 0x5c}, 	// ∖ -> backslash
	{0x2217, 0x2a}, 	// ∗ -> *
	{0x221e, 0x6f}, 	// ∞ -> o
	{0x2223, 0x6c}, 	// ∣ -> l
	{0x2225, 0x6c}, 	// ∥ -> l
	{0x2228, 0x76}, 	// ∨ -> v
	{0x222a, 0x55}, 	// ∪ -> U
	{0x2236, 0x3a}, 	// ∶ -> :
	{0x223c, 0x7e}, 	// ∼ -> ~
	{0x2296, 0x4f}, 	// ⊖ -> O
	{0x229d, 0x4f}, 	// ⊝ -> O
	{0x22a4, 0x54}, 	// ⊤ -> T
	{0x22c1, 0x76}, 	// ⋁ -> v
	{0x22c3, 0x55}, 	// ⋃ -> U
	{0x22ff, 0x45}, 	// ⋿ -> E
	{0x2303, 0x5e}, 	// ⌃ -> ^
	{0x2329, 0x3c}, 	// 〈 -> <
	{0x2361, 0x54}, 	// ⍡ -> T
	{0x236c, 0x4f}, 	// ⍬ -> O
	{0x2373, 0x69}, 	// ⍳ -> i
	{0x2374, 0x70}, 	// ⍴ -> p
	{0x2376, 0x61}, 	// ⍶ -> a
	{0x2378, 0x69}, 	// ⍸ -> i
	{0x237a, 0x61}, 	// ⍺ -> a
	{0x23fd, 0x6c}, 	// ⏽ -> l
	{0x2460, 0x31}, 	// ① -> 1
	{0x2461, 0x32}, 	// ② -> 2
	{0x2462, 0x33}, 	// ③ -> 3
	{0x2463, 0x34}, 	// ④ -> 4
	{0x2464, 0x35}, 	// ⑤ -> 5
	{0x2465, 0x36}, 	// ⑥ -> 6
	{0x2466, 0x37}, 	// ⑦ -> 7
	{0x2467, 0x38}, 	// ⑧ -> 8
	{0x2468, 0x39}, 	// ⑨ -> 9
	{0x2474, 0x6c}, 	// ⑴ -> l
	{0x247d, 0x6c}, 	// ⑽ -> l
	{0x247e, 0x6c}, 	// ⑾ -> l
	{0x247f, 0x6c}, 	// ⑿ -> l
	{0x2480, 0x6c}, 	// ⒀ -> l
	{0x2481, 0x6c}, 	// ⒁ -> l
	{0x2482, 0x6c}, 	// ⒂ -> l
	{0x2483, 0x6c}, 	// ⒃ -> l
	{0x2484, 0x6c}, 	// ⒄ -> l
	{0x2485, 0x6c}, 	// ⒅ -> l
	{0x2486, 0x6c}, 	// ⒆ -> l
	{0x2487, 0x4f}, 	// ⒇ -> O
	{0x2488, 0x6c}, 	// ⒈ -> l
	{0x2491, 0x6c}, 	// ⒑ -> l
	{0x2492, 0x6c}, 	// ⒒ -> l
	{0x2493, 0x6c}, 	// ⒓ -> l
	{0x2494, 0x6c}, 	// ⒔ -> l
	{0x2495, 0x6c}, 	// ⒕ -> l
	{0x2496, 0x6c}, 	// ⒖ -> l
	{0x2497, 0x6c}, 	// ⒗ -> l
	{0x2498, 0x6c}, 	// ⒘ -> l
	{0x2499, 0x6c}, 	// ⒙ -> l
	{0x249a, 0x6c}, 	// ⒚ -> l
	{0x249b, 0x4f}, 	// ⒛ -> O
	{0x249c, 0x61}, 	// ⒜ -> a
	{0x249d, 0x62}, 	// ⒝ -> b
	{0x249e, 0x63}, 	// ⒞ -> c
	{0x249f, 0x64}, 	// ⒟ -> d
	{0x24a0, 0x65}, 	// ⒠ -> e
	{0x24a1, 0x66}, 	// ⒡ -> f
	{0x24a2, 0x67}, 	// ⒢ -> g
	{0x24a3, 0x68}, 	// ⒣ -> h
	{0x24a4, 0x69}, 	// ⒤ -> i
	{0x24a5, 0x6a}, 	// ⒥ -> j
	{0x24a6, 0x6b}, 	// ⒦ -> k
	{0x24a7, 0x6c}, 	// ⒧ -> l
	{0x24a8, 0x72}, 	// ⒨ -> r
	{0x24a9, 0x6e}, 	// ⒩ -> n
	{0x24aa, 0x6f}, 	// ⒪ -> o
	{0x24ab, 0x70}, 	// ⒫ -> p
	{0x24ac, 0x71}, 	// ⒬ -> q
	{0x24ad, 0x72}, 	// ⒭ -> r
	{0x24ae, 0x73}, 	// ⒮ -> s
	{0x24af, 0x74}, 	// ⒯ -> t
	{0x24b0, 0x75}, 	// ⒰ -> u
	{0x24b1, 0x76}, 	// ⒱ -> v
	{0x24b2, 0x77}, 	// ⒲ -> w
	{0x24b3, 0x78}, 	// ⒳ -> x
	{0x24b4, 0x79}, 	// ⒴ -> y
	{0x24b5, 0x7a}, 	// ⒵ -> z
	{0x24b6, 0x41}, 	// Ⓐ -> A
	{0x24b7, 0x42}, 	// Ⓑ -> B
	{0x24b8, 0x43}, 	// Ⓒ -> C
	{0x24b9, 0x44}, 	// Ⓓ -> D
	{0x24ba, 0x45}, 	// Ⓔ -> E
	{0x24bb, 0x46}, 	// Ⓕ -> F
	{0x24bc, 0x47}, 	// Ⓖ -> G
	{0x24bd, 0x48}, 	// Ⓗ -> H
	{0x24be, 0x49}, 	// Ⓘ -> I
	{0x24bf, 0x4a}, 	// Ⓙ -> J
	{0x24c0, 0x4b}, 	// Ⓚ -> K
	{0x24c1, 0x4c}, 	// Ⓛ -> L
	{0x24c2, 0x4d}, 	// Ⓜ -> M
	{0x24c3, 0x4e}, 	// Ⓝ -> N
	{0x24c4, 0x4f}, 	// Ⓞ -> O
	{0x24c5, 0x50}, 	// Ⓟ -> P
	{0x24c6, 0x51}, 	// Ⓠ -> Q
	{0x24c7, 0x52}, 	// Ⓡ -> R
	{0x24c8, 0x53}, 	// Ⓢ -> S
	{0x24c9, 0x54}, 	// Ⓣ -> T
	{0x24ca, 0x55}, 	// Ⓤ -> U
	{0x24cb, 0x56}, 	// Ⓥ -> V
	{0x24cc, 0x57}, 	// Ⓦ -> W
	{0x24cd, 0x58}, 	// Ⓧ -> X
	{0x24ce, 0x59}, 	// Ⓨ -> Y
	{0x24cf, 0x5a}, 	// Ⓩ -> Z
	{0x24d0, 0x61}, 	// ⓐ -> a
	{0x24d1, 0x62}, 	// ⓑ -> b
	{0x24d2, 0x63}, 	// ⓒ -> c
	{0x24d3, 0x64}, 	// ⓓ -> d
	{0x24d4, 0x65}, 	// ⓔ -> e
	{0x24d5, 0x66}, 	// ⓕ -> f
	{0x24d6, 0x67}, 	// ⓖ -> g
	{0x24d7, 0x68}, 	// ⓗ -> h
	{0x24d8, 0x69}, 	// ⓘ -> i
	{0x24d9, 0x6a}, 	// ⓙ -> j
	{0x24da, 0x6b}, 	// ⓚ -> k
	{0x24db, 0x6c}, 	// ⓛ -> l
	{0x24dc, 0x6d}, 	// ⓜ -> m
	{0x24dd, 0x6e}, 	// ⓝ -> n
	{0x24de, 0x6f}, 	// ⓞ -> o
	{0x24df, 0x70}, 	// ⓟ -> p
	{0x24e0, 0x71}, 	// ⓠ -> q
	{0x24e1, 0x72}, 	// ⓡ -> r
	{0x24e2, 0x73}, 	// ⓢ -> s
	{0x24e3, 0x74}, 	// ⓣ -> t
	{0x24e4, 0x75}, 	// ⓤ -> u
	{0x24e5, 0x76}, 	// ⓥ -> v
	{0x24e6, 0x77}, 	// ⓦ -> w
	{0x24e7, 0x78}, 	// ⓧ -> x
	{0x24e8, 0x79}, 	// ⓨ -> y
	{0x24e9, 0x7a}, 	// ⓩ -> z
	{0x24ea, 0x30}, 	// ⓪ -> 0
	{0x24f5, 0x31}, 	// ⓵ -> 1
	{0x24f6, 0x32}, 	// ⓶ -> 2
	{0x24f7, 0x33}, 	// ⓷ -> 3
	{0x24f8, 0x34}, 	// ⓸ -> 4
	{0x24f9, 0x35}, 	// ⓹ -> 5
	{0x24fa, 0x36}, 	// ⓺ -> 6
	{0x24fb, 0x37}, 	// ⓻ -> 7
	{0x24fc, 0x38}, 	// ⓼ -> 8
	{0x24fd, 0x39}, 	// ⓽ -> 9
	{0x24ff, 0x30}, 	// ⓿ -> 0
	{0x2500, 0x2d}, 	// ─ -> -
	{0x2501, 0x2d}, 	// ━ -> -
	{0x2502, 0x7c}, 	// │ -> |
	{0x2503, 0x7c}, 	// ┃ -> |
	{0x2504, 0x2d}, 	// ┄ -> -
	{0x2505, 0x2d}, 	// ┅ -> -
	{0x2506, 0x7c}, 	// ┆ -> |
	{0x2507, 0x7c}, 	// ┇ -> |
	{0x2508, 0x2d}, 	// ┈ -> -
	{0x2509, 0x2d}, 	// ┉ -> -
	{0x250a, 0x7c}, 	// ┊ -> |
	{0x250b, 0x7c}, 	// ┋ -> |
	{0x250c, 0x2b}, 	// ┌ -> +
	{0x250d, 0x2b}, 	// ┍ -> +
	{0x250e, 0x2b}, 	// ┎ -> +
	{0x250f, 0x2b}, 	// ┏ -> +
	{0x2510, 0x2b}, 	// ┐ -> +
	{0x2511, 0x2b}, 	// ┑ -> +
	{0x2512, 0x2b}, 	// ┒ -> +
	{0x2513, 0x2b}, 	// ┓ -> +
	{0x2514, 0x2b}, 	// └ -> +
	{0x2515, 0x2b}, 	// ┕ -> +
	{0x2516, 0x2b}, 	// ┖ -> +
	{0x2517, 0x2b}, 	// ┗ -> +
	{0x2518, 0x2b}, 	// ┘ -> +
	{0x2519, 0x2b}, 	// ┙ -> +
	{0x251a, 0x2b}, 	// ┚ -> +
	{0x251b, 0x2b}, 	// ┛ -> +
	{0x251c, 0x2b}, 	// ├ -> +
	{0x251d, 0x2b}, 	// ┝ -> +
	{0x251e, 0x2b}, 	// ┞ -> +
	{0x251f, 0x2b}, 	// ┟ -> +
	{0x2520, 0x2b}, 	// ┠ -> +
	{0x2521, 0x2b}, 	// ┡ -> +
	{0x2522, 0x2b}, 	// ┢ -> +
	{0x2523, 0x2b}, 	// ┣ -> +
	{0x2524, 0x2b}, 	// ┤ -> +
	{0x2525, 0x2b}, 	// ┥ -> +
	{0x2526, 0x2b}, 	// ┦ -> +
	{0x2527, 0x2b}, 	// ┧ -> +
	{0x2528, 0x2b}, 	// ┨ -> +
	{0x2529, 0x2b}, 	// ┩ -> +
	{0x252a, 0x2b}, 	// ┪ -> +
	{0x252b, 0x2b}, 	// ┫ -> +
	{0x252c, 0x2b}, 	// ┬ -> +
	{0x252d, 0x2b}, 	// ┭ -> +
	{0x252e, 0x2b}, 	// ┮ -> +
	{0x252f, 0x2b}, 	// ┯ -> +
	{0x2530, 0x2b}, 	// ┰ -> +
	{0x2531, 0x2b}, 	// ┱ -> +
	{0x2532, 0x2b}, 	// ┲ -> +
	{0x2533, 0x2b}, 	// ┳ -> +
	{0x2534, 0x2b}, 	// ┴ -> +
	{0x2535, 0x2b}, 	// ┵ -> +
	{0x2536, 0x2b}, 	// ┶ -> +
	{0x2537, 0x2b}, 	// ┷ -> +
	{0x2538, 0x2b}, 	// ┸ -> +
	{0x2539, 0x2b}, 	// ┹ -> +
	{0x253a, 0x2b}, 	// ┺ -> +
	{0x253b, 0x2b}, 	// ┻ -> +
	{0x253c, 0x2b}, 	// ┼ -> +
	{0x253d, 0x2b}, 	// ┽ -> +
	{0x253e, 0x2b}, 	// ┾ -> +
	{0x253f, 0x2b}, 	// ┿ -> +
	{0x2540, 0x2b}, 	// ╀ -> +
	{0x2541, 0x2b}, 	// ╁ -> +
	{0x2542, 0x2b}, 	// ╂ -> +
	{0x2543, 0x2b}, 	// ╃ -> +
	{0x2544, 0x2b}, 	// ╄ -> +
	{0x2545, 0x2b}, 	// ╅ -> +
	{0x2546, 0x2b}, 	// ╆ -> +
	{0x2547, 0x2b}, 	// ╇ -> +
	{0x2548, 0x2b}, 	// ╈ -> +
	{0x2549, 0x2b}, 	// ╉ -> +
	{0x254a, 0x2b}, 	// ╊ -> +
	{0x254b, 0x2b}, 	// ╋ -> +
	{0x254c, 0x2d}, 	// ╌ -> -
	{0x254d, 0x2d}, 	// ╍ -> -
	{0x254e, 0x7c}, 	// ╎ -> |
	{0x254f, 0x7c}, 	// ╏ -> |
	{0x2550, 0x2d}, 	// ═ -> -
	{0x2551, 0x7c}, 	// ║ -> |
	{0x2552, 0x2b}, 	// ╒ -> +
	{0x2553, 0x2b}, 	// ╓ -> +
	{0x2554, 0x2b}, 	// ╔ -> +
	{0x2555, 0x2b}, 	// ╕ -> +
	{0x2556, 0x2b}, 	// ╖ -> +
	{0x2557, 0x2b}, 	// ╗ -> +
	{0x2558, 0x2b}, 	// ╘ -> +
	{0x2559, 0x2b}, 	// ╙ -> +
	{0x255a, 0x2b}, 	// ╚ -> +
	{0x255b, 0x2b}, 	// ╛ -> +
	{0x255c, 0x2b}, 	// ╜ -> +
	{0x255d, 0x2b}, 	// ╝ -> +
	{0x255e, 0x2b}, 	// ╞ -> +
	{0x255f, 0x2b}, 	// ╟ -> +
	{0x2560, 0x2b}, 	// ╠ -> +
	{0x2561, 0x2b}, 	// ╡ -> +
	{0x2562, 0x2b}, 	// ╢ -> +
	{0x2563, 0x2b}, 	// ╣ -> +
	{0x2564, 0x2b}, 	// ╤ -> +
	{0x2565, 0x2b}, 	// ╥ -> +
	{0x2566, 0x2b}, 	// ╦ -> +
	{0x2567, 0x2b}, 	// ╧ -> +
	{0x2568, 0x2b}, 	// ╨ -> +
	{0x2569, 0x2b}, 	// ╩ -> +
	{0x256a, 0x2b}, 	// ╪ -> +
	{0x256b, 0x2b}, 	// ╫ -> +
	{0x256c, 0x2b}, 	// ╬ -> +
	{0x256d, 0x2b}, 	// ╭ -> +
	{0x256e, 0x2b}, 	// ╮ -> +
	{0x256f, 0x2b}, 	// ╯ -> +
	{0x2570, 0x2b}, 	// ╰ -> +
	{0x2571, 0x2f}, 	// ╱ -> /
	{0x2572, 0x5c}, 	// ╲ -> backslash
	{0x2573, 0x58}, 	// ╳ -> X
	{0x2574, 0x2d}, 	// ╴ -> -
	{0x2575, 0x7c}, 	// ╵ -> |
	{0x2576, 0x2d}, 	// ╶ -> -
	{0x2577, 0x7c}, 	// ╷ -> |
	{0x2578, 0x2d}, 	// ╸ -> -
	{0x2579, 0x7c}, 	// ╹ -> |
	{0x257a, 0x2d}, 	// ╺ -> -
	{0x257b, 0x7c}, 	// ╻ -> |
	{0x257c, 0x2d}, 	// ╼ -> -
	{0x257d, 0x7c}, 	// ╽ -> |
	{0x257e, 0x2d}, 	// ╾ -> -
	{0x257f, 0x7c}, 	// ╿ -> |
	{0x2580, 0x23}, 	// ▀ -> #
	{0x2581, 0x23}, 	// ▁ -> #
	{0x2582, 0x23}, 	// ▂ -> #
	{0x2583, 0x23}, 	// ▃ -> #
	{0x2584, 0x23}, 	// ▄ -> #
	{0x2585, 0x23}, 	// ▅ -> #
	{0x2586, 0x23}, 	// ▆ -> #
	{0x2587, 0x23}, 	// ▇ -> #
	{0x2588, 0x23}, 	// █ -> #
	{0x2589, 0x23}, 	// ▉ -> #
	{0x258a, 0x23}, 	// ▊ -> #
	{0x258b, 0x23}, 	// ▋ -> #
	{0x258c, 0x23}, 	// ▌ -> #
	{0x258d, 0x23}, 	// ▍ -> #
	{0x258e, 0x23}, 	// ▎ -> #
	{0x258f, 0x23}, 	// ▏ -> #
	{0x2590, 0x23}, 	// ▐ -> #
	{0x2591, 0x23}, 	// ░ -> #
	{0x2592, 0x23}, 	// ▒ -> #
	{0x2593, 0x23}, 	// ▓ -> #
	{0x2594, 0x2d}, 	// ▔ -> -
	{0x2595, 0x7c}, 	// ▕ -> |
	{0x25a0, 0x23}, 	// ■ -> #
	{0x25a1, 0x23}, 	// □ -> #
	{0x25a2, 0x23}, 	// ▢ -> #
	{0x25a3, 0x23}, 	// ▣ -> #
	{0x25a4, 0x23}, 	// ▤ -> #
	{0x25a5, 0x23}, 	// ▥ -> #
	{0x25a6, 0x23}, 	// ▦ -> #
	{0x25a7, 0x23}, 	// ▧ -> #
	{0x25a8, 0x23}, 	// ▨ -> #
	{0x25a9, 0x23}, 	// ▩ -> #
	{0x25aa, 0x23}, 	// ▪ -> #
	{0x25ab, 0x23}, 	// ▫ -> #
	{0x25ac, 0x23}, 	// ▬ -> #
	{0x25ad, 0x23}, 	// ▭ -> #
	{0x25ae, 0x23}, 	// ▮ -> #
	{0x25af, 0x23}, 	// ▯ -> #
	{0x25b0, 0x23}, 	// ▰ -> #
	{0x25b1, 0x23}, 	// ▱ -> #
	{0x25b2, 0x5e}, 	// ▲ -> ^
	{0x25b3, 0x5e}, 	// △ -> ^
	{0x25b4, 0x5e}, 	// ▴ -> ^
	{0x25b5, 0x5e}, 	// ▵ -> ^
	{0x25b6, 0x3e}, 	// ▶ -> >
	{0x25b7, 0x3e}, 	// ▷ -> >
	{0x25b8, 0x3e}, 	// ▸ -> >
	{0x25b9, 0x3e}, 	// ▹ -> >
	{0x25ba, 0x3e}, 	// ► -> >
	{0x25bb, 0x3e}, 	// ▻ -> >
	{0x25bc, 0x56}, 	// ▼ -> V
	{0x25bd, 0x56}, 	// ▽ -> V
	{0x25be, 0x56}, 	// ▾ -> V
	{0x25bf, 0x56}, 	// ▿ -> V
	{0x25c0, 0x3c}, 	// ◀ -> <
	{0x25c1, 0x3c}, 	// ◁ -> <
	{0x25c2, 0x3c}, 	// ◂ -> <
	{0x25c3, 0x3c}, 	// ◃ -> <
	{0x25c4, 0x3c}, 	// ◄ -> <
	{0x25c5, 0x3c}, 	// ◅ -> <
	{0x25c6, 0x2a}, 	// ◆ -> *
	{0x25c7, 0x2a}, 	// ◇ -> *
	{0x25c8, 0x2a}, 	// ◈ -> *
	{0x25c9, 0x2a}, 	// ◉ -> *
	{0x25ca, 0x2a}, 	// ◊ -> *
	{0x25cb, 0x2a}, 	// ○ -> *
	{0x25cc, 0x2a}, 	// ◌ -> *
	{0x25cd, 0x2a}, 	// ◍ -> *
	{0x25ce, 0x2a}, 	// ◎ -> *
	{0x25cf, 0x2a}, 	// ● -> *
	{0x25d0, 0x2a}, 	// ◐ -> *
	{0x25d1, 0x2a}, 	// ◑ -> *
	{0x25d2, 0x2a}, 	// ◒ -> *
	{0x25d3, 0x2a}, 	// ◓ -> *
	{0x25d4, 0x2a}, 	// ◔ -> *
	{0x25d5, 0x2a}, 	// ◕ -> *
	{0x25d6, 0x2a}, 	// ◖ -> *
	{0x25d7, 0x2a}, 	// ◗ -> *
	{0x25d8, 0x2a}, 	// ◘ -> *
	{0x25d9, 0x2a}, 	// ◙ -> *
	{0x25da, 0x2a}, 	// ◚ -> *
	{0x25db, 0x2a}, 	// ◛ -> *
	{0x25dc, 0x2a}, 	// ◜ -> *
	{0x25dd, 0x2a}, 	// ◝ -> *
	{0x25de, 0x2a}, 	// ◞ -> *
	{0x25df, 0x2a}, 	// ◟ -> *
	{0x25e0, 0x2a}, 	// ◠ -> *
	{0x25e1, 0x2a}, 	// ◡ -> *
	{0x25e2, 0x2a}, 	// ◢ -> *
	{0x25e3, 0x2a}, 	// ◣ -> *
	{0x25e4, 0x2a}, 	// ◤ -> *
	{0x25e5, 0x2a}, 	// ◥ -> *
	{0x25e6, 0x2a}, 	// ◦ -> *
	{0x25e7, 0x23}, 	// ◧ -> #
	{0x25e8, 0x23}, 	// ◨ -> #
	{0x25e9, 0x23}, 	// ◩ -> #
	{0x25ea, 0x23}, 	// ◪ -> #
	{0x25eb, 0x23}, 	// ◫ -> #
	{0x25ec, 0x5e}, 	// ◬ -> ^
	{0x25ed, 0x5e}, 	// ◭ -> ^
	{0x25ee, 0x5e}, 	// ◮ -> ^
	{0x25ef, 0x4f}, 	// ◯ -> O
	{0x25f0, 0x23}, 	// ◰ -> #
	{0x25f1, 0x23}, 	// ◱ -> #
	{0x25f2, 0x23}, 	// ◲ -> #
	{0x25f3, 0x23}, 	// ◳ -> #
	{0x25f4, 0x23}, 	// ◴ -> #
	{0x25f5, 0x23}, 	// ◵ -> #
	{0x25f6, 0x23}, 	// ◶ -> #
	{0x25f7, 0x23}, 	// ◷ -> #
	{0x266f, 0x23}, 	// ♯ -> #
	{0x2731, 0x2a}, 	// ✱ -> *
	{0x2758, 0x7c}, 	// ❘ -> |
	{0x275c, 0x27}, 	// ❜ -> '
	{0x275d, 0x22}, 	// ❝ -> "
	{0x275e, 0x22}, 	// ❞ -> "
	{0x275f, 0x2c}, 	// ❟ -> ,
	{0x2762, 0x21}, 	// ❢ -> !
	{0x27d9, 0x54}, 	// ⟙ -> T
	{0x27e6, 0x5b}, 	// ⟦ -> [
	{0x27e8, 0x3c}, 	// ⟨ -> <
	{0x2800, 0x20}, 	// ⠀ ->  
	{0x2801, 0x61}, 	// ⠁ -> a
	{0x2802, 0x31}, 	// ⠂ -> 1
	{0x2803, 0x62}, 	// ⠃ -> b
	{0x2804, 0x27}, 	// ⠄ -> '
	{0x2805, 0x6b}, 	// ⠅ -> k
	{0x2806, 0x32}, 	// ⠆ -> 2
	{0x2807, 0x6c}, 	// ⠇ -> l
	{0x2808, 0x40}, 	// ⠈ -> @
	{0x2809, 0x63}, 	// ⠉ -> c
	{0x280a, 0x69}, 	// ⠊ -> i
	{0x280b, 0x66}, 	// ⠋ -> f
	{0x280c, 0x2f}, 	// ⠌ -> /
	{0x280d, 0x6d}, 	// ⠍ -> m
	{0x280e, 0x73}, 	// ⠎ -> s
	{0x280f, 0x70}, 	// ⠏ -> p
	{0x2810, 0x22}, 	// ⠐ -> "
	{0x2811, 0x65}, 	// ⠑ -> e
	{0x2812, 0x33}, 	// ⠒ -> 3
	{0x2813, 0x68}, 	// ⠓ -> h
	{0x2814, 0x39}, 	// ⠔ -> 9
	{0x2815, 0x6f}, 	// ⠕ -> o
	{0x2816, 0x36}, 	// ⠖ -> 6
	{0x2817, 0x72}, 	// ⠗ -> r
	{0x2818, 0x5e}, 	// ⠘ -> ^
	{0x2819, 0x64}, 	// ⠙ -> d
	{0x281a, 0x6a}, 	// ⠚ -> j
	{0x281b, 0x67}, 	// ⠛ -> g
	{0x281c, 0x3e}, 	// ⠜ -> >
	{0x281d, 0x6e}, 	// ⠝ -> n
	{0x281e, 0x74}, 	// ⠞ -> t
	{0x281f, 0x71}, 	// ⠟ -> q
	{0x2820, 0x2c}, 	// ⠠ -> ,
	{0x2821, 0x2a}, 	// ⠡ -> *
	{0x2822, 0x35}, 	// ⠢ -> 5
	{0x2823, 0x3c}, 	// ⠣ -> <
	{0x2824, 0x2d}, 	// ⠤ -> -
	{0x2825, 0x75}, 	// ⠥ -> u
	{0x2826, 0x38}, 	// ⠦ -> 8
	{0x2827, 0x76}, 	// ⠧ -> v
	{0x2828, 0x2e}, 	// ⠨ -> .
	{0x2829, 0x25}, 	// ⠩ -> %
	{0x282a, 0x5b}, 	// ⠪ -> [
	{0x282b, 0x24}, 	// ⠫ -> $
	{0x282c, 0x2b}, 	// ⠬ -> +
	{0x282d, 0x78}, 	// ⠭ -> x
	{0x282e, 0x21}, 	// ⠮ -> !
	{0x282f, 0x26}, 	// ⠯ -> &
	{0x2830, 0x3b}, 	// ⠰ -> ;
	{0x2831, 0x3a}, 	// ⠱ -> :
	{0x2832, 0x34}, 	// ⠲ -> 4
	{0x2833, 0x5c}, 	// ⠳ -> backslash
	{0x2834, 0x30}, 	// ⠴ -> 0
	{0x2835, 0x7a}, 	// ⠵ -> z
	{0x2836, 0x37}, 	// ⠶ -> 7
	{0x2837, 0x28}, 	// ⠷ -> (
	{0x2838, 0x5f}, 	// ⠸ -> _
	{0x2839, 0x3f}, 	// ⠹ -> ?
	{0x283a, 0x77}, 	// ⠺ -> w
	{0x283b, 0x5d}, 	// ⠻ -> ]
	{0x283c, 0x23}, 	// ⠼ -> #
	{0x283d, 0x79}, 	// ⠽ -> y
	{0x283e, 0x29}, 	// ⠾ -> )
	{0x283f, 0x3d}, 	// ⠿ -> =
	{0x292b, 0x78}, 	// ⤫ -> x
	{0x292c, 0x78}, 	// ⤬ -> x
	{0x2983, 0x7b}, 	// ⦃ -> {
	{0x2a2f, 0x78}, 	// ⨯ -> x
	{0x2a30, 0x78}, 	// ⨰ -> x
	{0x2c60, 0x4c}, 	// Ⱡ -> L
	{0x2c61, 0x6c}, 	// ⱡ -> l
	{0x2c62, 0x4c}, 	// Ɫ -> L
	{0x2c63, 0x50}, 	// Ᵽ -> P
	{0x2c64, 0x52}, 	// Ɽ -> R
	{0x2c65, 0x61}, 	// ⱥ -> a
	{0x2c66, 0x74}, 	// ⱦ -> t
	{0x2c67, 0x48}, 	// Ⱨ -> H
	{0x2c68, 0x68}, 	// ⱨ -> h
	{0x2c69, 0x4b}, 	// Ⱪ -> K
	{0x2c6a, 0x6b}, 	// ⱪ -> k
	{0x2c6b, 0x5a}, 	// Ⱬ -> Z
	{0x2c6c, 0x7a}, 	// ⱬ -> z
	{0x2c6e, 0x4d}, 	// Ɱ -> M
	{0x2c6f, 0x41}, 	// Ɐ -> A
	{0x2c85, 0x72}, 	// ⲅ -> r
	{0x2c8e, 0x48}, 	// Ⲏ -> H
	{0x2c92, 0x6c}, 	// Ⲓ -> l
	{0x2c94, 0x4b}, 	// Ⲕ -> K
	{0x2c95, 0x6b}, 	// ⲕ -> k
	{0x2c98, 0x4d}, 	// Ⲙ -> M
	{0x2c9a, 0x4e}, 	// Ⲛ -> N
	{0x2c9e, 0x4f}, 	// Ⲟ -> O
	{0x2c9f, 0x6f}, 	// ⲟ -> o
	{0x2ca2, 0x50}, 	// Ⲣ -> P
	{0x2ca3, 0x70}, 	// ⲣ -> p
	{0x2ca4, 0x43}, 	// Ⲥ -> C
	{0x2ca5, 0x63}, 	// ⲥ -> c
	{0x2ca6, 0x54}, 	// Ⲧ -> T
	{0x2ca8, 0x59}, 	// Ⲩ -> Y
	{0x2cac, 0x58}, 	// Ⲭ -> X
	{0x2cd0, 0x4c}, 	// Ⳑ -> L
	{0x2d31, 0x4f}, 	// ⴱ -> O
	{0x2d38, 0x56}, 	// ⴸ -> V
	{0x2d39, 0x45}, 	// ⴹ -> E
	{0x2d41, 0x4f}, 	// ⵁ -> O
	{0x2d4f, 0x6c}, 	// ⵏ -> l
	{0x2d54, 0x4f}, 	// ⵔ -> O
	{0x2d55, 0x51}, 	// ⵕ -> Q
	{0x2d5d, 0x58}, 	// ⵝ -> X
	{0x2e00, 0x72}, 	// ⸀ -> r
	{0x2e06, 0x54}, 	// ⸆ -> T
	{0x2e09, 0x73}, 	// ⸉ -> s
	{0x2e0c, 0x5c}, 	// ⸌ -> backslash
	{0x2e0d, 0x2f}, 	// ⸍ -> /
	{0x2e12, 0x3e}, 	// ⸒ -> >
	{0x2e13, 0x25}, 	// ⸓ -> %
	{0x2e16, 0x3e}, 	// ⸖ -> >
	{0x2e17, 0x3d}, 	// ⸗ -> =
	{0x2e19, 0x2f}, 	// ⸙ -> /
	{0x2e1a, 0x2d}, 	// ⸚ -> -
	{0x2e1b, 0x7e}, 	// ⸛ -> ~
	{0x2e1c, 0x5c}, 	// ⸜ -> backslash
	{0x2e1d, 0x2f}, 	// ⸝ -> /
	{0x2e1e, 0x7e}, 	// ⸞ -> ~
	{0x2e1f, 0x7e}, 	// ⸟ -> ~
	{0x2e2e, 0x3f}, 	// ⸮ -> ?
	{0x2e2f, 0x27}, 	// ⸯ -> '
	{0x2e30, 0x6f}, 	// ⸰ -> o
	{0x2e31, 0x2e}, 	// ⸱ -> .
	{0x2e32, 0x2c}, 	// ⸲ -> ,
	{0x2e33, 0x2e}, 	// ⸳ -> .
	{0x2e34, 0x2c}, 	// ⸴ -> ,
	{0x2e35, 0x3b}, 	// ⸵ -> ;
	{0x2e3c, 0x78}, 	// ⸼ -> x
	{0x2e3d, 0x7c}, 	// ⸽ -> |
	{0x2e40, 0x3d}, 	// ⹀ -> =
	{0x2e41, 0x2c}, 	// ⹁ -> ,
	{0x2e42, 0x22}, 	// ⹂ -> "
	{0x3000, 0x20}, 	// 　 ->  
	{0x3003, 0x22}, 	// 〃 -> "
	{0x3005, 0x22}, 	// 々 -> "
	{0x3006, 0x2f}, 	// 〆 -> /
	{0x3007, 0x4f}, 	// 〇 -> O
	{0x3008, 0x3c}, 	// 〈 -> <
	{0x300c, 0x5b}, 	// 「 -> [
	{0x300e, 0x7b}, 	// 『 -> {
	{0x3012, 0x40}, 	// 〒 -> @
	{0x3014, 0x5b}, 	// 〔 -> [
	{0x3020, 0x40}, 	// 〠 -> @
	{0x3021, 0x31}, 	// 〡 -> 1
	{0x3022, 0x32}, 	// 〢 -> 2
	{0x3023, 0x33}, 	// 〣 -> 3
	{0x3024, 0x34}, 	// 〤 -> 4
	{0x3025, 0x35}, 	// 〥 -> 5
	{0x3026, 0x36}, 	// 〦 -> 6
	{0x3027, 0x37}, 	// 〧 -> 7
	{0x3028, 0x38}, 	// 〨 -> 8
	{0x3029, 0x39}, 	// 〩 -> 9
	{0x3030, 0x7e}, 	// 〰 -> ~
	{0x3031, 0x2b}, 	// 〱 -> +
	{0x3032, 0x2b}, 	// 〲 -> +
	{0x3033, 0x2b}, 	// 〳 -> +
	{0x3034, 0x2b}, 	// 〴 -> +
	{0x3036, 0x40}, 	// 〶 -> @
	{0x3041, 0x61}, 	// ぁ -> a
	{0x3042, 0x61}, 	// あ -> a
	{0x3043, 0x69}, 	// ぃ -> i
	{0x3044, 0x69}, 	// い -> i
	{0x3045, 0x75}, 	// ぅ -> u
	{0x3046, 0x75}, 	// う -> u
	{0x3047, 0x65}, 	// ぇ -> e
	{0x3048, 0x65}, 	// え -> e
	{0x3049, 0x6f}, 	// ぉ -> o
	{0x304a, 0x6f}, 	// お -> o
	{0x3093, 0x6e}, 	// ん -> n
	{0x309d, 0x22}, 	// ゝ -> "
	{0x309e, 0x22}, 	// ゞ -> "
	{0x30a0, 0x3d}, 	// ゠ -> =
	{0x30a1, 0x61}, 	// ァ -> a
	{0x30a2, 0x61}, 	// ア -> a
	{0x30a3, 0x69}, 	// ィ -> i
	{0x30a4, 0x69}, 	// イ -> i
	{0x30a5, 0x75}, 	// ゥ -> u
	{0x30a6, 0x75}, 	// ウ -> u
	{0x30a7, 0x65}, 	// ェ -> e
	{0x30a8, 0x65}, 	// エ -> e
	{0x30a9, 0x6f}, 	// ォ -> o
	{0x30aa, 0x6f}, 	// オ -> o
	{0x30f3, 0x6e}, 	// ン -> n
	{0x30fb, 0x2a}, 	// ・ -> *
	{0x30fc, 0x2d}, 	// ー -> -
	{0x30fd, 0x22}, 	// ヽ -> "
	{0x30fe, 0x22}, 	// ヾ -> "
	{0x3105, 0x42}, 	// ㄅ -> B
	{0x3106, 0x50}, 	// ㄆ -> P
	{0x3107, 0x4d}, 	// ㄇ -> M
	{0x3108, 0x46}, 	// ㄈ -> F
	{0x3109, 0x44}, 	// ㄉ -> D
	{0x310a, 0x54}, 	// ㄊ -> T
	{0x310b, 0x4e}, 	// ㄋ -> N
	{0x310c, 0x4c}, 	// ㄌ -> L
	{0x310d, 0x47}, 	// ㄍ -> G
	{0x310e, 0x4b}, 	// ㄎ -> K
	{0x310f, 0x48}, 	// ㄏ -> H
	{0x3110, 0x4a}, 	// ㄐ -> J
	{0x3111, 0x51}, 	// ㄑ -> Q
	{0x3112, 0x58}, 	// ㄒ -> X
	{0x3116, 0x52}, 	// ㄖ -> R
	{0x3117, 0x5a}, 	// ㄗ -> Z
	{0x3118, 0x43}, 	// ㄘ -> C
	{0x3119, 0x53}, 	// ㄙ -> S
	{0x311a, 0x41}, 	// ㄚ -> A
	{0x311b, 0x4f}, 	// ㄛ -> O
	{0x311c, 0x45}, 	// ㄜ -> E
	{0x3127, 0x49}, 	// ㄧ -> I
	{0x3128, 0x55}, 	// ㄨ -> U
	{0x312a, 0x56}, 	// ㄪ -> V
	{0x3131, 0x67}, 	// ㄱ -> g
	{0x3134, 0x6e}, 	// ㄴ -> n
	{0x3137, 0x64}, 	// ㄷ -> d
	{0x3139, 0x72}, 	// ㄹ -> r
	{0x3141, 0x6d}, 	// ㅁ -> m
	{0x3142, 0x62}, 	// ㅂ -> b
	{0x3145, 0x73}, 	// ㅅ -> s
	{0x3148, 0x6a}, 	// ㅈ -> j
	{0x314a, 0x63}, 	// ㅊ -> c
	{0x314b, 0x6b}, 	// ㅋ -> k
	{0x314c, 0x74}, 	// ㅌ -> t
	{0x314d, 0x70}, 	// ㅍ -> p
	{0x314e, 0x68}, 	// ㅎ -> h
	{0x314f, 0x61}, 	// ㅏ -> a
	{0x3154, 0x65}, 	// ㅔ -> e
	{0x3157, 0x6f}, 	// ㅗ -> o
	{0x315c, 0x75}, 	// ㅜ -> u
	{0x3163, 0x69}, 	// ㅣ -> i
	{0x317f, 0x5a}, 	// ㅿ -> Z
	{0x3181, 0x4e}, 	// ㆁ -> N
	{0x3186, 0x51}, 	// ㆆ -> Q
	{0x318d, 0x55}, 	// ㆍ -> U
	{0x31b4, 0x50}, 	// ㆴ -> P
	{0x31b5, 0x54}, 	// ㆵ -> T
	{0x31b6, 0x4b}, 	// ㆶ -> K
	{0x31b7, 0x48}, 	// ㆷ -> H
	{0x32d0, 0x61}, 	// ㋐ -> a
	{0x32d1, 0x69}, 	// ㋑ -> i
	{0x32d2, 0x75}, 	// ㋒ -> u
	{0x32d3, 0x75}, 	// ㋓ -> u
	{0x32d4, 0x6f}, 	// ㋔ -> o
	{0xa002, 0x69}, 	// ꀂ -> i
	{0xa00a, 0x61}, 	// ꀊ -> a
	{0xa011, 0x6f}, 	// ꀑ -> o
	{0xa014, 0x65}, 	// ꀔ -> e
	{0xa4d0, 0x42}, 	// ꓐ -> B
	{0xa4d1, 0x50}, 	// ꓑ -> P
	{0xa4d2, 0x64}, 	// ꓒ -> d
	{0xa4d3, 0x44}, 	// ꓓ -> D
	{0xa4d4, 0x54}, 	// ꓔ -> T
	{0xa4d6, 0x47}, 	// ꓖ -> G
	{0xa4d7, 0x4b}, 	// ꓗ -> K
	{0xa4d9, 0x4a}, 	// ꓙ -> J
	{0xa4da, 0x43}, 	// ꓚ -> C
	{0xa4dc, 0x5a}, 	// ꓜ -> Z
	{0xa4dd, 0x46}, 	// ꓝ -> F
	{0xa4df, 0x4d}, 	// ꓟ -> M
	{0xa4e0, 0x4e}, 	// ꓠ -> N
	{0xa4e1, 0x4c}, 	// ꓡ -> L
	{0xa4e2, 0x53}, 	// ꓢ -> S
	{0xa4e3, 0x52}, 	// ꓣ -> R
	{0xa4e6, 0x56}, 	// ꓦ -> V
	{0xa4e7, 0x48}, 	// ꓧ -> H
	{0xa4ea, 0x57}, 	// ꓪ -> W
	{0xa4eb, 0x58}, 	// ꓫ -> X
	{0xa4ec, 0x59}, 	// ꓬ -> Y
	{0xa4ee, 0x41}, 	// ꓮ -> A
	{0xa4f0, 0x45}, 	// ꓰ -> E
	{0xa4f2, 0x6c}, 	// ꓲ -> l
	{0xa4f3, 0x4f}, 	// ꓳ -> O
	{0xa4f4, 0x55}, 	// ꓴ -> U
	{0xa647, 0x69}, 	// ꙇ -> i
	{0xa650, 0x6c}, 	// Ꙑ -> l
	{0xa651, 0x62}, 	// ꙑ -> b
	{0xa695, 0x68}, 	// ꚕ -> h
	{0xa698, 0x4f}, 	// Ꚙ -> O
	{0xa699, 0x6f}, 	// ꚙ -> o
	{0xa6df, 0x56}, 	// ꛟ -> V
	{0xa728, 0x54}, 	// Ꜩ -> T
	{0xa729, 0x74}, 	// ꜩ -> t
	{0xa731, 0x73}, 	// ꜱ -> s
	{0xa732, 0x41}, 	// Ꜳ -> A
	{0xa733, 0x61}, 	// ꜳ -> a
	{0xa734, 0x41}, 	// Ꜵ -> A
	{0xa735, 0x61}, 	// ꜵ -> a
	{0xa736, 0x41}, 	// Ꜷ -> A
	{0xa737, 0x61}, 	// ꜷ -> a
	{0xa738, 0x41}, 	// Ꜹ -> A
	{0xa739, 0x61}, 	// ꜹ -> a
	{0xa73a, 0x41}, 	// Ꜻ -> A
	{0xa73b, 0x61}, 	// ꜻ -> a
	{0xa73c, 0x41}, 	// Ꜽ -> A
	{0xa73d, 0x61}, 	// ꜽ -> a
	{0xa740, 0x4b}, 	// Ꝁ -> K
	{0xa74a, 0x4f}, 	// Ꝋ -> O
	{0xa74b, 0x6f}, 	// ꝋ -> o
	{0xa74e, 0x4f}, 	// Ꝏ -> O
	{0xa74f, 0x6f}, 	// ꝏ -> o
	{0xa761, 0x77}, 	// ꝡ -> w
	{0xa777, 0x74}, 	// ꝷ -> t
	{0xa798, 0x46}, 	// Ꞙ -> F
	{0xa799, 0x66}, 	// ꞙ -> f
	{0xa79f, 0x75}, 	// ꞟ -> u
	{0xa7b2, 0x4a}, 	// Ʝ -> J
	{0xa7b3, 0x58}, 	// Ꭓ -> X
	{0xa7b4, 0x42}, 	// Ꞵ -> B
	{0xab32, 0x65}, 	// ꬲ -> e
	{0xab35, 0x66}, 	// ꬵ -> f
	{0xab3d, 0x6f}, 	// ꬽ -> o
	{0xab3e, 0x6f}, 	// ꬾ -> o
	{0xab47, 0x72}, 	// ꭇ -> r
	{0xab48, 0x72}, 	// ꭈ -> r
	{0xab4e, 0x75}, 	// ꭎ -> u
	{0xab52, 0x75}, 	// ꭒ -> u
	{0xab5a, 0x79}, 	// ꭚ -> y
	{0xab63, 0x75}, 	// ꭣ -> u
	{0xab74, 0x6f}, 	// ꭴ -> o
	{0xab75, 0x69}, 	// ꭵ -> i
	{0xab81, 0x72}, 	// ꮁ -> r
	{0xab83, 0x77}, 	// ꮃ -> w
	{0xab8e, 0x6f}, 	// ꮎ -> o
	{0xab93, 0x7a}, 	// ꮓ -> z
	{0xab9c, 0x75}, 	// ꮜ -> u
	{0xaba9, 0x76}, 	// ꮩ -> v
	{0xabaa, 0x73}, 	// ꮪ -> s
	{0xabaf, 0x63}, 	// ꮯ -> c
	{0xabb6, 0x6b}, 	// ꮶ -> k
	{0xabbb, 0x6f}, 	// ꮻ -> o
	{0xc544, 0x61}, 	// 아 -> a
	{0xc5d0, 0x65}, 	// 에 -> e
	{0xc624, 0x6f}, 	// 오 -> o
	{0xc6b0, 0x75}, 	// 우 -> u
	{0xc774, 0x69}, 	// 이 -> i
	{0xfb00, 0x66}, 	// ﬀ -> f
	{0xfb01, 0x66}, 	// ﬁ -> f
	{0xfb02, 0x66}, 	// ﬂ -> f
	{0xfb03, 0x66}, 	// ﬃ -> f
	{0xfb04, 0x66}, 	// ﬄ -> f
	{0xfb06, 0x73}, 	// ﬆ -> s
	{0xfb1d, 0x69}, 	// יִ -> i
	{0xfb20, 0x60}, 	// ﬠ -> `
	{0xfb21, 0x41}, 	// ﬡ -> A
	{0xfb22, 0x64}, 	// ﬢ -> d
	{0xfb23, 0x68}, 	// ﬣ -> h
	{0xfb25, 0x6c}, 	// ﬥ -> l
	{0xfb26, 0x6d}, 	// ﬦ -> m
	{0xfb27, 0x72}, 	// ﬧ -> r
	{0xfb28, 0x74}, 	// ﬨ -> t
	{0xfb29, 0x2b}, 	// ﬩ -> +
	{0xfb2b, 0x53}, 	// שׂ -> S
	{0xfb2d, 0x53}, 	// שּׂ -> S
	{0xfb2e, 0x61}, 	// אַ -> a
	{0xfb2f, 0x61}, 	// אָ -> a
	{0xfb30, 0x41}, 	// אּ -> A
	{0xfb31, 0x62}, 	// בּ -> b
	{0xfb32, 0x67}, 	// גּ -> g
	{0xfb33, 0x64}, 	// דּ -> d
	{0xfb34, 0x68}, 	// הּ -> h
	{0xfb35, 0x76}, 	// וּ -> v
	{0xfb36, 0x7a}, 	// זּ -> z
	{0xfb38, 0x74}, 	// טּ -> t
	{0xfb39, 0x79}, 	// יּ -> y
	{0xfb3c, 0x6c}, 	// לּ -> l
	{0xfb3e, 0x6d}, 	// מּ -> m
	{0xfb40, 0x6e}, 	// נּ -> n
	{0xfb41, 0x73}, 	// סּ -> s
	{0xfb43, 0x70}, 	// ףּ -> p
	{0xfb44, 0x70}, 	// פּ -> p
	{0xfb47, 0x6b}, 	// קּ -> k
	{0xfb48, 0x72}, 	// רּ -> r
	{0xfb4a, 0x74}, 	// תּ -> t
	{0xfb4b, 0x6f}, 	// וֹ -> o
	{0xfb4c, 0x76}, 	// בֿ -> v
	{0xfb4e, 0x66}, 	// פֿ -> f
	{0xfba6, 0x6f}, 	// ﮦ -> o
	{0xfba7, 0x6f}, 	// ﮧ -> o
	{0xfba8, 0x6f}, 	// ﮨ -> o
	{0xfba9, 0x6f}, 	// ﮩ -> o
	{0xfbaa, 0x6f}, 	// ﮪ -> o
	{0xfbab, 0x6f}, 	// ﮫ -> o
	{0xfbac, 0x6f}, 	// ﮬ -> o
	{0xfbad, 0x6f}, 	// ﮭ -> o
	{0xfbea, 0x6c}, 	// ﯪ -> l
	{0xfbeb, 0x6c}, 	// ﯫ -> l
	{0xfbec, 0x6f}, 	// ﯬ -> o
	{0xfbed, 0x6f}, 	// ﯭ -> o
	{0xfc37, 0x6c}, 	// ﰷ -> l
	{0xfc51, 0x6f}, 	// ﱑ -> o
	{0xfc52, 0x6f}, 	// ﱒ -> o
	{0xfc53, 0x6f}, 	// ﱓ -> o
	{0xfc54, 0x6f}, 	// ﱔ -> o
	{0xfc80, 0x6c}, 	// ﲀ -> l
	{0xfc88, 0x6c}, 	// ﲈ -> l
	{0xfc9b, 0x6f}, 	// ﲛ -> o
	{0xfca0, 0x6f}, 	// ﲠ -> o
	{0xfca5, 0x6f}, 	// ﲥ -> o
	{0xfccd, 0x6f}, 	// ﳍ -> o
	{0xfcd6, 0x6f}, 	// ﳖ -> o
	{0xfcd7, 0x6f}, 	// ﳗ -> o
	{0xfcd8, 0x6f}, 	// ﳘ -> o
	{0xfcd9, 0x6f}, 	// ﳙ -> o
	{0xfcde, 0x6f}, 	// ﳞ -> o
	{0xfce0, 0x6f}, 	// ﳠ -> o
	{0xfce2, 0x6f}, 	// ﳢ -> o
	{0xfce4, 0x6f}, 	// ﳤ -> o
	{0xfce6, 0x6f}, 	// ﳦ -> o
	{0xfce8, 0x6f}, 	// ﳨ -> o
	{0xfcea, 0x6f}, 	// ﳪ -> o
	{0xfcef, 0x6f}, 	// ﳯ -> o
	{0xfcf1, 0x6f}, 	// ﳱ -> o
	{0xfd31, 0x6f}, 	// ﴱ -> o
	{0xfd32, 0x6f}, 	// ﴲ -> o
	{0xfd3c, 0x6c}, 	// ﴼ -> l
	{0xfd3d, 0x6c}, 	// ﴽ -> l
	{0xfd93, 0x6f}, 	// ﶓ -> o
	{0xfd94, 0x6f}, 	// ﶔ -> o
	{0xfdf2, 0x6c}, 	// ﷲ -> l
	{0xfdf3, 0x6c}, 	// ﷳ -> l
	{0xfdf7, 0x6f}, 	// ﷷ -> o
	{0xfdfa, 0x6c}, 	// ﷺ -> l
	{0xfdfb, 0x6c}, 	// ﷻ -> l
	{0xfdfc, 0x6c}, 	// ﷼ -> l
	{0xfe23, 0x7e}, 	// ︣ -> ~
	{0xfe32, 0x2d}, 	// ︲ -> -
	{0xfe33, 0x5f}, 	// ︳ -> _
	{0xfe34, 0x5f}, 	// ︴ -> _
	{0xfe35, 0x28}, 	// ︵ -> (
	{0xfe37, 0x7b}, 	// ︷ -> {
	{0xfe39, 0x5b}, 	// ︹ -> [
	{0xfe3f, 0x3c}, 	// ︿ -> <
	{0xfe41, 0x5b}, 	// ﹁ -> [
	{0xfe43, 0x7b}, 	// ﹃ -> {
	{0xfe44, 0x7d}, 	// ﹄ -> }
	{0xfe50, 0x2c}, 	// ﹐ -> ,
	{0xfe51, 0x2c}, 	// ﹑ -> ,
	{0xfe52, 0x2e}, 	// ﹒ -> .
	{0xfe54, 0x3b}, 	// ﹔ -> ;
	{0xfe55, 0x3a}, 	// ﹕ -> :
	{0xfe56, 0x3f}, 	// ﹖ -> ?
	{0xfe57, 0x21}, 	// ﹗ -> !
	{0xfe58, 0x2d}, 	// ﹘ -> -
	{0xfe59, 0x28}, 	// ﹙ -> (
	{0xfe5a, 0x29}, 	// ﹚ -> )
	{0xfe5b, 0x7b}, 	// ﹛ -> {
	{0xfe5c, 0x7d}, 	// ﹜ -> }
	{0xfe5d, 0x7b}, 	// ﹝ -> {
	{0xfe5e, 0x7d}, 	// ﹞ -> }
	{0xfe5f, 0x23}, 	// ﹟ -> #
	{0xfe60, 0x26}, 	// ﹠ -> &
	{0xfe61, 0x2a}, 	// ﹡ -> *
	{0xfe62, 0x2b}, 	// ﹢ -> +
	{0xfe63, 0x2d}, 	// ﹣ -> -
	{0xfe64, 0x3c}, 	// ﹤ -> <
	{0xfe65, 0x3e}, 	// ﹥ -> >
	{0xfe66, 0x3d}, 	// ﹦ -> =
	{0xfe68, 0x5c}, 	// ﹨ -> backslash
	{0xfe69, 0x24}, 	// ﹩ -> $
	{0xfe6a, 0x25}, 	// ﹪ -> %
	{0xfe6b, 0x40}, 	// ﹫ -> @
	{0xfe83, 0x6c}, 	// ﺃ -> l
	{0xfe84, 0x6c}, 	// ﺄ -> l
	{0xfe87, 0x6c}, 	// ﺇ -> l
	{0xfe88, 0x6c}, 	// ﺈ -> l
	{0xfe8d, 0x6c}, 	// ﺍ -> l
	{0xfe8e, 0x6c}, 	// ﺎ -> l
	{0xfee9, 0x6f}, 	// ﻩ -> o
	{0xfeea, 0x6f}, 	// ﻪ -> o
	{0xfeeb, 0x6f}, 	// ﻫ -> o
	{0xfeec, 0x6f}, 	// ﻬ -> o
	{0xfef7, 0x6c}, 	// ﻷ -> l
	{0xfef8, 0x6c}, 	// ﻸ -> l
	{0xfef9, 0x6c}, 	// ﻹ -> l
	{0xfefa, 0x6c}, 	// ﻺ -> l
	{0xfefb, 0x6c}, 	// ﻻ -> l
	{0xfefc, 0x6c}, 	// ﻼ -> l
	{0xff01, 0x21}, 	// ！ -> !
	{0xff02, 0x22}, 	// ＂ -> "
	{0xff03, 0x23}, 	// ＃ -> #
	{0xff04, 0x24}, 	// ＄ -> $
	{0xff05, 0x25}, 	// ％ -> %
	{0xff06, 0x26}, 	// ＆ -> &
	{0xff07, 0x27}, 	// ＇ -> '
	{0xff08, 0x28}, 	// （ -> (
	{0xff09, 0x29}, 	// ） -> )
	{0xff0a, 0x2a}, 	// ＊ -> *
	{0xff0b, 0x2b}, 	// ＋ -> +
	{0xff0c, 0x2c}, 	// ， -> ,
	{0xff0d, 0x2d}, 	// － -> -
	{0xff0e, 0x2e}, 	// ． -> .
	{0xff0f, 0x2f}, 	// ／ -> /
	{0xff10, 0x30}, 	// ０ -> 0
	{0xff11, 0x31}, 	// １ -> 1
	{0xff12, 0x32}, 	// ２ -> 2
	{0xff13, 0x33}, 	// ３ -> 3
	{0xff14, 0x34}, 	// ４ -> 4
	{0xff15, 0x35}, 	// ５ -> 5
	{0xff16, 0x36}, 	// ６ -> 6
	{0xff17, 0x37}, 	// ７ -> 7
	{0xff18, 0x38}, 	// ８ -> 8
	{0xff19, 0x39}, 	// ９ -> 9
	{0xff1a, 0x3a}, 	// ： -> :
	{0xff1b, 0x3b}, 	// ； -> ;
	{0xff1c, 0x3c}, 	// ＜ -> <
	{0xff1d, 0x3d}, 	// ＝ -> =
	{0xff1e, 0x3e}, 	// ＞ -> >
	{0xff1f, 0x3f}, 	// ？ -> ?
	{0xff20, 0x40}, 	// ＠ -> @
	{0xff21, 0x41}, 	// Ａ -> A
	{0xff22, 0x42}, 	// Ｂ -> B
	{0xff23, 0x43}, 	// Ｃ -> C
	{0xff24, 0x44}, 	// Ｄ -> D
	{0xff25, 0x45}, 	// Ｅ -> E
	{0xff26, 0x46}, 	// Ｆ -> F
	{0xff27, 0x47}, 	// Ｇ -> G
	{0xff28, 0x48}, 	// Ｈ -> H
	{0xff29, 0x6c}, 	// Ｉ -> l
	{0xff2a, 0x4a}, 	// Ｊ -> J
	{0xff2b, 0x4b}, 	// Ｋ -> K
	{0xff2c, 0x4c}, 	// Ｌ -> L
	{0xff2d, 0x4d}, 	// Ｍ -> M
	{0xff2e, 0x4e}, 	// Ｎ -> N
	{0xff2f, 0x4f}, 	// Ｏ -> O
	{0xff30, 0x50}, 	// Ｐ -> P
	{0xff31, 0x51}, 	// Ｑ -> Q
	{0xff32, 0x52}, 	// Ｒ -> R
	{0xff33, 0x53}, 	// Ｓ -> S
	{0xff34, 0x54}, 	// Ｔ -> T
	{0xff35, 0x55}, 	// Ｕ -> U
	{0xff36, 0x56}, 	// Ｖ -> V
	{0xff37, 0x57}, 	// Ｗ -> W
	{0xff38, 0x58}, 	// Ｘ -> X
	{0xff39, 0x59}, 	// Ｙ -> Y
	{0xff3a, 0x5a}, 	// Ｚ -> Z
	{0xff3b, 0x5b}, 	// ［ -> [
	{0xff3c, 0x5c}, 	// ＼ -> backslash
	{0xff3d, 0x5d}, 	// ］ -> ]
	{0xff3e, 0x5e}, 	// ＾ -> ^
	{0xff3f, 0x5f}, 	// ＿ -> _
	{0xff40, 0x60}, 	// ｀ -> `
	{0xff41, 0x61}, 	// ａ -> a
	{0xff42, 0x62}, 	// ｂ -> b
	{0xff43, 0x63}, 	// ｃ -> c
	{0xff44, 0x64}, 	// ｄ -> d
	{0xff45, 0x65}, 	// ｅ -> e
	{0xff46, 0x66}, 	// ｆ -> f
	{0xff47, 0x67}, 	// ｇ -> g
	{0xff48, 0x68}, 	// ｈ -> h
	{0xff49, 0x69}, 	// ｉ -> i
	{0xff4a, 0x6a}, 	// ｊ -> j
	{0xff4b, 0x6b}, 	// ｋ -> k
	{0xff4c, 0x6c}, 	// ｌ -> l
	{0xff4d, 0x6d}, 	// ｍ -> m
	{0xff4e, 0x6e}, 	// ｎ -> n
	{0xff4f, 0x6f}, 	// ｏ -> o
	{0xff50, 0x70}, 	// ｐ -> p
	{0xff51, 0x71}, 	// ｑ -> q
	{0xff52, 0x72}, 	// ｒ -> r
	{0xff53, 0x73}, 	// ｓ -> s
	{0xff54, 0x74}, 	// ｔ -> t
	{0xff55, 0x75}, 	// ｕ -> u
	{0xff56, 0x76}, 	// ｖ -> v
	{0xff57, 0x77}, 	// ｗ -> w
	{0xff58, 0x78}, 	// ｘ -> x
	{0xff59, 0x79}, 	// ｙ -> y
	{0xff5a, 0x7a}, 	// ｚ -> z
	{0xff5b, 0x7b}, 	// ｛ -> {
	{0xff5c, 0x7c}, 	// ｜ -> |
	{0xff5d, 0x7d}, 	// ｝ -> }
	{0xff5e, 0x7e}, 	// ～ -> ~
	{0xff61, 0x2e}, 	// ｡ -> .
	{0xff62, 0x5b}, 	// ｢ -> [
	{0xff63, 0x5d}, 	// ｣ -> ]
	{0xff64, 0x2c}, 	// ､ -> ,
	{0xff65, 0x2a}, 	// ･ -> *
	{0xff67, 0x61}, 	// ｧ -> a
	{0xff68, 0x69}, 	// ｨ -> i
	{0xff69, 0x75}, 	// ｩ -> u
	{0xff6a, 0x65}, 	// ｪ -> e
	{0xff6b, 0x6f}, 	// ｫ -> o
	{0xff70, 0x2b}, 	// ｰ -> +
	{0xff71, 0x61}, 	// ｱ -> a
	{0xff72, 0x69}, 	// ｲ -> i
	{0xff73, 0x75}, 	// ｳ -> u
	{0xff74, 0x65}, 	// ｴ -> e
	{0xff75, 0x6f}, 	// ｵ -> o
	{0xff9d, 0x6e}, 	// ﾝ -> n
	{0xff9e, 0x3a}, 	// ﾞ -> :
	{0xff9f, 0x3b}, 	// ﾟ -> ;
	{0xffa1, 0x67}, 	// ﾡ -> g
	{0xffa4, 0x6e}, 	// ﾤ -> n
	{0xffa7, 0x64}, 	// ﾧ -> d
	{0xffa9, 0x72}, 	// ﾩ -> r
	{0xffb1, 0x6d}, 	// ﾱ -> m
	{0xffb2, 0x62}, 	// ﾲ -> b
	{0xffb5, 0x73}, 	// ﾵ -> s
	{0xffb8, 0x6a}, 	// ﾸ -> j
	{0xffba, 0x63}, 	// ﾺ -> c
	{0xffbb, 0x6b}, 	// ﾻ -> k
	{0xffbc, 0x74}, 	// ﾼ -> t
	{0xffbd, 0x70}, 	// ﾽ -> p
	{0xffbe, 0x68}, 	// ﾾ -> h
	{0xffc2, 0x61}, 	// ￂ -> a
	{0xffc7, 0x65}, 	// ￇ -> e
	{0xffcc, 0x6f}, 	// ￌ -> o
	{0xffd3, 0x75}, 	// ￓ -> u
	{0xffdc, 0x69}, 	// ￜ -> i
	{0xffe2, 0x21}, 	// ￢ -> !
	{0xffe3, 0x2d}, 	// ￣ -> -
	{0xffe4, 0x7c}, 	// ￤ -> |
	{0xffe8, 0x6c}, 	// ￨ -> l
	{0xffe9, 0x2d}, 	// ￩ -> -
	{0xffea, 0x7c}, 	// ￪ -> |
	{0xffeb, 0x2d}, 	// ￫ -> -
	{0xffec, 0x7c}, 	// ￬ -> |
	{0xffed, 0x23}, 	// ￭ -> #
	{0xffee, 0x4f}, 	// ￮ -> O
	{0xfff9, 0x7b}, 	// ￹ -> {
	{0xfffa, 0x7c}, 	// ￺ -> |
	{0xfffb, 0x7d}, 	// ￻ -> }
	{0x1018e, 0x4e}, 	// 𐆎 -> N
	{0x10196, 0x58}, 	// 𐆖 -> X
	{0x10197, 0x56}, 	// 𐆗 -> V
	{0x10198, 0x6c}, 	// 𐆘 -> l
	{0x10199, 0x6c}, 	// 𐆙 -> l
	{0x10282, 0x42}, 	// 𐊂 -> B
	{0x10286, 0x45}, 	// 𐊆 -> E
	{0x10287, 0x46}, 	// 𐊇 -> F
	{0x1028a, 0x6c}, 	// 𐊊 -> l
	{0x10290, 0x58}, 	// 𐊐 -> X
	{0x10292, 0x4f}, 	// 𐊒 -> O
	{0x10295, 0x50}, 	// 𐊕 -> P
	{0x10296, 0x53}, 	// 𐊖 -> S
	{0x10297, 0x54}, 	// 𐊗 -> T
	{0x102a0, 0x41}, 	// 𐊠 -> A
	{0x102a1, 0x42}, 	// 𐊡 -> B
	{0x102a2, 0x43}, 	// 𐊢 -> C
	{0x102a5, 0x46}, 	// 𐊥 -> F
	{0x102ab, 0x4f}, 	// 𐊫 -> O
	{0x102b0, 0x4d}, 	// 𐊰 -> M
	{0x102b1, 0x54}, 	// 𐊱 -> T
	{0x102b2, 0x59}, 	// 𐊲 -> Y
	{0x102b4, 0x58}, 	// 𐊴 -> X
	{0x102cf, 0x48}, 	// 𐋏 -> H
	{0x102f5, 0x5a}, 	// 𐋵 -> Z
	{0x10301, 0x42}, 	// 𐌁 -> B
	{0x10302, 0x43}, 	// 𐌂 -> C
	{0x10309, 0x6c}, 	// 𐌉 -> l
	{0x10311, 0x4d}, 	// 𐌑 -> M
	{0x10315, 0x54}, 	// 𐌕 -> T
	{0x10317, 0x58}, 	// 𐌗 -> X
	{0x10320, 0x6c}, 	// 𐌠 -> l
	{0x10322, 0x58}, 	// 𐌢 -> X
	{0x10404, 0x4f}, 	// 𐐄 -> O
	{0x10415, 0x43}, 	// 𐐕 -> C
	{0x1041b, 0x4c}, 	// 𐐛 -> L
	{0x10420, 0x53}, 	// 𐐠 -> S
	{0x1042c, 0x6f}, 	// 𐐬 -> o
	{0x1043d, 0x63}, 	// 𐐽 -> c
	{0x10448, 0x73}, 	// 𐑈 -> s
	{0x104b4, 0x52}, 	// 𐒴 -> R
	{0x104c2, 0x4f}, 	// 𐓂 -> O
	{0x104ce, 0x55}, 	// 𐓎 -> U
	{0x104ea, 0x6f}, 	// 𐓪 -> o
	{0x104f6, 0x75}, 	// 𐓶 -> u
	{0x10513, 0x4e}, 	// 𐔓 -> N
	{0x10516, 0x4f}, 	// 𐔖 -> O
	{0x10518, 0x4b}, 	// 𐔘 -> K
	{0x1051c, 0x43}, 	// 𐔜 -> C
	{0x1051d, 0x56}, 	// 𐔝 -> V
	{0x10525, 0x46}, 	// 𐔥 -> F
	{0x10526, 0x4c}, 	// 𐔦 -> L
	{0x10527, 0x58}, 	// 𐔧 -> X
	{0x114c5, 0x77}, 	// 𑓅 -> w
	{0x114d0, 0x4f}, 	// 𑓐 -> O
	{0x11700, 0x72}, 	// 𑜀 -> r
	{0x11706, 0x76}, 	// 𑜆 -> v
	{0x1170a, 0x77}, 	// 𑜊 -> w
	{0x1170e, 0x77}, 	// 𑜎 -> w
	{0x1170f, 0x77}, 	// 𑜏 -> w
	{0x118a0, 0x56}, 	// 𑢠 -> V
	{0x118a2, 0x46}, 	// 𑢢 -> F
	{0x118a3, 0x4c}, 	// 𑢣 -> L
	{0x118a4, 0x59}, 	// 𑢤 -> Y
	{0x118a6, 0x45}, 	// 𑢦 -> E
	{0x118a9, 0x5a}, 	// 𑢩 -> Z
	{0x118ae, 0x45}, 	// 𑢮 -> E
	{0x118b2, 0x4c}, 	// 𑢲 -> L
	{0x118b5, 0x4f}, 	// 𑢵 -> O
	{0x118b8, 0x55}, 	// 𑢸 -> U
	{0x118bc, 0x54}, 	// 𑢼 -> T
	{0x118c0, 0x76}, 	// 𑣀 -> v
	{0x118c1, 0x73}, 	// 𑣁 -> s
	{0x118c2, 0x46}, 	// 𑣂 -> F
	{0x118c3, 0x69}, 	// 𑣃 -> i
	{0x118c4, 0x7a}, 	// 𑣄 -> z
	{0x118c8, 0x6f}, 	// 𑣈 -> o
	{0x118d7, 0x6f}, 	// 𑣗 -> o
	{0x118d8, 0x75}, 	// 𑣘 -> u
	{0x118dc, 0x79}, 	// 𑣜 -> y
	{0x118e0, 0x4f}, 	// 𑣠 -> O
	{0x118e3, 0x72}, 	// 𑣣 -> r
	{0x118e5, 0x5a}, 	// 𑣥 -> Z
	{0x118e6, 0x57}, 	// 𑣦 -> W
	{0x118e9, 0x43}, 	// 𑣩 -> C
	{0x118ec, 0x58}, 	// 𑣬 -> X
	{0x118ef, 0x57}, 	// 𑣯 -> W
	{0x118f2, 0x43}, 	// 𑣲 -> C
	{0x16f08, 0x56}, 	// 𖼈 -> V
	{0x16f0a, 0x54}, 	// 𖼊 -> T
	{0x16f16, 0x4c}, 	// 𖼖 -> L
	{0x16f28, 0x6c}, 	// 𖼨 -> l
	{0x16f35, 0x52}, 	// 𖼵 -> R
	{0x16f3a, 0x53}, 	// 𖼺 -> S
	{0x16f40, 0x41}, 	// 𖽀 -> A
	{0x16f42, 0x55}, 	// 𖽂 -> U
	{0x16f43, 0x59}, 	// 𖽃 -> Y
	{0x1ccd6, 0x41}, 	// 𜳖 -> A
	{0x1ccd7, 0x42}, 	// 𜳗 -> B
	{0x1ccd8, 0x43}, 	// 𜳘 -> C
	{0x1ccd9, 0x44}, 	// 𜳙 -> D
	{0x1ccda, 0x45}, 	// 𜳚 -> E
	{0x1ccdb, 0x46}, 	// 𜳛 -> F
	{0x1ccdc, 0x47}, 	// 𜳜 -> G
	{0x1ccdd, 0x48}, 	// 𜳝 -> H
	{0x1ccde, 0x6c}, 	// 𜳞 -> l
	{0x1ccdf, 0x4a}, 	// 𜳟 -> J
	{0x1cce0, 0x4b}, 	// 𜳠 -> K
	{0x1cce1, 0x4c}, 	// 𜳡 -> L
	{0x1cce2, 0x4d}, 	// 𜳢 -> M
	{0x1cce3, 0x4e}, 	// 𜳣 -> N
	{0x1cce4, 0x4f}, 	// 𜳤 -> O
	{0x1cce5, 0x50}, 	// 𜳥 -> P
	{0x1cce6, 0x51}, 	// 𜳦 -> Q
	{0x1cce7, 0x52}, 	// 𜳧 -> R
	{0x1cce8, 0x53}, 	// 𜳨 -> S
	{0x1cce9, 0x54}, 	// 𜳩 -> T
	{0x1ccea, 0x55}, 	// 𜳪 -> U
	{0x1cceb, 0x56}, 	// 𜳫 -> V
	{0x1ccec, 0x57}, 	// 𜳬 -> W
	{0x1cced, 0x58}, 	// 𜳭 -> X
	{0x1ccee, 0x59}, 	// 𜳮 -> Y
	{0x1ccef, 0x5a}, 	// 𜳯 -> Z
	{0x1ccf0, 0x4f}, 	// 𜳰 -> O
	{0x1ccf1, 0x6c}, 	// 𜳱 -> l
	{0x1d20d, 0x56}, 	// 𝈍 -> V
	{0x1d213, 0x46}, 	// 𝈓 -> F
	{0x1d216, 0x52}, 	// 𝈖 -> R
	{0x1d21a, 0x4f}, 	// 𝈚 -> O
	{0x1d22a, 0x4c}, 	// 𝈪 -> L
	{0x1d400, 0x41}, 	// 𝐀 -> A
	{0x1d401, 0x42}, 	// 𝐁 -> B
	{0x1d402, 0x43}, 	// 𝐂 -> C
	{0x1d403, 0x44}, 	// 𝐃 -> D
	{0x1d404, 0x45}, 	// 𝐄 -> E
	{0x1d405, 0x46}, 	// 𝐅 -> F
	{0x1d406, 0x47}, 	// 𝐆 -> G
	{0x1d407, 0x48}, 	// 𝐇 -> H
	{0x1d408, 0x6c}, 	// 𝐈 -> l
	{0x1d409, 0x4a}, 	// 𝐉 -> J
	{0x1d40a, 0x4b}, 	// 𝐊 -> K
	{0x1d40b, 0x4c}, 	// 𝐋 -> L
	{0x1d40c, 0x4d}, 	// 𝐌 -> M
	{0x1d40d, 0x4e}, 	// 𝐍 -> N
	{0x1d40e, 0x4f}, 	// 𝐎 -> O
	{0x1d40f, 0x50}, 	// 𝐏 -> P
	{0x1d410, 0x51}, 	// 𝐐 -> Q
	{0x1d411, 0x52}, 	// 𝐑 -> R
	{0x1d412, 0x53}, 	// 𝐒 -> S
	{0x1d413, 0x54}, 	// 𝐓 -> T
	{0x1d414, 0x55}, 	// 𝐔 -> U
	{0x1d415, 0x56}, 	// 𝐕 -> V
	{0x1d416, 0x57}, 	// 𝐖 -> W
	{0x1d417, 0x58}, 	// 𝐗 -> X
	{0x1d418, 0x59}, 	// 𝐘 -> Y
	{0x1d419, 0x5a}, 	// 𝐙 -> Z
	{0x1d41a, 0x61}, 	// 𝐚 -> a
	{0x1d41b, 0x62}, 	// 𝐛 -> b
	{0x1d41c, 0x63}, 	// 𝐜 -> c
	{0x1d41d, 0x64}, 	// 𝐝 -> d
	{0x1d41e, 0x65}, 	// 𝐞 -> e
	{0x1d41f, 0x66}, 	// 𝐟 -> f
	{0x1d420, 0x67}, 	// 𝐠 -> g
	{0x1d421, 0x68}, 	// 𝐡 -> h
	{0x1d422, 0x69}, 	// 𝐢 -> i
	{0x1d423, 0x6a}, 	// 𝐣 -> j
	{0x1d424, 0x6b}, 	// 𝐤 -> k
	{0x1d425, 0x6c}, 	// 𝐥 -> l
	{0x1d426, 0x72}, 	// 𝐦 -> r
	{0x1d427, 0x6e}, 	// 𝐧 -> n
	{0x1d428, 0x6f}, 	// 𝐨 -> o
	{0x1d429, 0x70}, 	// 𝐩 -> p
	{0x1d42a, 0x71}, 	// 𝐪 -> q
	{0x1d42b, 0x72}, 	// 𝐫 -> r
	{0x1d42c, 0x73}, 	// 𝐬 -> s
	{0x1d42d, 0x74}, 	// 𝐭 -> t
	{0x1d42e, 0x75}, 	// 𝐮 -> u
	{0x1d42f, 0x76}, 	// 𝐯 -> v
	{0x1d430, 0x77}, 	// 𝐰 -> w
	{0x1d431, 0x78}, 	// 𝐱 -> x
	{0x1d432, 0x79}, 	// 𝐲 -> y
	{0x1d433, 0x7a}, 	// 𝐳 -> z
	{0x1d434, 0x41}, 	// 𝐴 -> A
	{0x1d435, 0x42}, 	// 𝐵 -> B
	{0x1d436, 0x43}, 	// 𝐶 -> C
	{0x1d437, 0x44}, 	// 𝐷 -> D
	{0x1d438, 0x45}, 	// 𝐸 -> E
	{0x1d439, 0x46}, 	// 𝐹 -> F
	{0x1d43a, 0x47}, 	// 𝐺 -> G
	{0x1d43b, 0x48}, 	// 𝐻 -> H
	{0x1d43c, 0x6c}, 	// 𝐼 -> l
	{0x1d43d, 0x4a}, 	// 𝐽 -> J
	{0x1d43e, 0x4b}, 	// 𝐾 -> K
	{0x1d43f, 0x4c}, 	// 𝐿 -> L
	{0x1d440, 0x4d}, 	// 𝑀 -> M
	{0x1d441, 0x4e}, 	// 𝑁 -> N
	{0x1d442, 0x4f}, 	// 𝑂 -> O
	{0x1d443, 0x50}, 	// 𝑃 -> P
	{0x1d444, 0x51}, 	// 𝑄 -> Q
	{0x1d445, 0x52}, 	// 𝑅 -> R
	{0x1d446, 0x53}, 	// 𝑆 -> S
	{0x1d447, 0x54}, 	// 𝑇 -> T
	{0x1d448, 0x55}, 	// 𝑈 -> U
	{0x1d449, 0x56}, 	// 𝑉 -> V
	{0x1d44a, 0x57}, 	// 𝑊 -> W
	{0x1d44b, 0x58}, 	// 𝑋 -> X
	{0x1d44c, 0x59}, 	// 𝑌 -> Y
	{0x1d44d, 0x5a}, 	// 𝑍 -> Z
	{0x1d44e, 0x61}, 	// 𝑎 -> a
	{0x1d44f, 0x62}, 	// 𝑏 -> b
	{0x1d450, 0x63}, 	// 𝑐 -> c
	{0x1d451, 0x64}, 	// 𝑑 -> d
	{0x1d452, 0x65}, 	// 𝑒 -> e
	{0x1d453, 0x66}, 	// 𝑓 -> f
	{0x1d454, 0x67}, 	// 𝑔 -> g
	{0x1d456, 0x69}, 	// 𝑖 -> i
	{0x1d457, 0x6a}, 	// 𝑗 -> j
	{0x1d458, 0x6b}, 	// 𝑘 -> k
	{0x1d459, 0x6c}, 	// 𝑙 -> l
	{0x1d45a, 0x72}, 	// 𝑚 -> r
	{0x1d45b, 0x6e}, 	// 𝑛 -> n
	{0x1d45c, 0x6f}, 	// 𝑜 -> o
	{0x1d45d, 0x70}, 	// 𝑝 -> p
	{0x1d45e, 0x71}, 	// 𝑞 -> q
	{0x1d45f, 0x72}, 	// 𝑟 -> r
	{0x1d460, 0x73}, 	// 𝑠 -> s
	{0x1d461, 0x74}, 	// 𝑡 -> t
	{0x1d462, 0x75}, 	// 𝑢 -> u
	{0x1d463, 0x76}, 	// 𝑣 -> v
	{0x1d464, 0x77}, 	// 𝑤 -> w
	{0x1d465, 0x78}, 	// 𝑥 -> x
	{0x1d466, 0x79}, 	// 𝑦 -> y
	{0x1d467, 0x7a}, 	// 𝑧 -> z
	{0x1d468, 0x41}, 	// 𝑨 -> A
	{0x1d469, 0x42}, 	// 𝑩 -> B
	{0x1d46a, 0x43}, 	// 𝑪 -> C
	{0x1d46b, 0x44}, 	// 𝑫 -> D
	{0x1d46c, 0x45}, 	// 𝑬 -> E
	{0x1d46d, 0x46}, 	// 𝑭 -> F
	{0x1d46e, 0x47}, 	// 𝑮 -> G
	{0x1d46f, 0x48}, 	// 𝑯 -> H
	{0x1d470, 0x6c}, 	// 𝑰 -> l
	{0x1d471, 0x4a}, 	// 𝑱 -> J
	{0x1d472, 0x4b}, 	// 𝑲 -> K
	{0x1d473, 0x4c}, 	// 𝑳 -> L
	{0x1d474, 0x4d}, 	// 𝑴 -> M
	{0x1d475, 0x4e}, 	// 𝑵 -> N
	{0x1d476, 0x4f}, 	// 𝑶 -> O
	{0x1d477, 0x50}, 	// 𝑷 -> P
	{0x1d478, 0x51}, 	// 𝑸 -> Q
	{0x1d479, 0x52}, 	// 𝑹 -> R
	{0x1d47a, 0x53}, 	// 𝑺 -> S
	{0x1d47b, 0x54}, 	// 𝑻 -> T
	{0x1d47c, 0x55}, 	// 𝑼 -> U
	{0x1d47d, 0x56}, 	// 𝑽 -> V
	{0x1d47e, 0x57}, 	// 𝑾 -> W
	{0x1d47f, 0x58}, 	// 𝑿 -> X
	{0x1d480, 0x59}, 	// 𝒀 -> Y
	{0x1d481, 0x5a}, 	// 𝒁 -> Z
	{0x1d482, 0x61}, 	// 𝒂 -> a
	{0x1d483, 0x62}, 	// 𝒃 -> b
	{0x1d484, 0x63}, 	// 𝒄 -> c
	{0x1d485, 0x64}, 	// 𝒅 -> d
	{0x1d486, 0x65}, 	// 𝒆 -> e
	{0x1d487, 0x66}, 	// 𝒇 -> f
	{0x1d488, 0x67}, 	// 𝒈 -> g
	{0x1d489, 0x68}, 	// 𝒉 -> h
	{0x1d48a, 0x69}, 	// 𝒊 -> i
	{0x1d48b, 0x6a}, 	// 𝒋 -> j
	{0x1d48c, 0x6b}, 	// 𝒌 -> k
	{0x1d48d, 0x6c}, 	// 𝒍 -> l
	{0x1d48e, 0x72}, 	// 𝒎 -> r
	{0x1d48f, 0x6e}, 	// 𝒏 -> n
	{0x1d490, 0x6f}, 	// 𝒐 -> o
	{0x1d491, 0x70}, 	// 𝒑 -> p
	{0x1d492, 0x71}, 	// 𝒒 -> q
	{0x1d493, 0x72}, 	// 𝒓 -> r
	{0x1d494, 0x73}, 	// 𝒔 -> s
	{0x1d495, 0x74}, 	// 𝒕 -> t
	{0x1d496, 0x75}, 	// 𝒖 -> u
	{0x1d497, 0x76}, 	// 𝒗 -> v
	{0x1d498, 0x77}, 	// 𝒘 -> w
	{0x1d499, 0x78}, 	// 𝒙 -> x
	{0x1d49a, 0x79}, 	// 𝒚 -> y
	{0x1d49b, 0x7a}, 	// 𝒛 -> z
	{0x1d49c, 0x41}, 	// 𝒜 -> A
	{0x1d49e, 0x43}, 	// 𝒞 -> C
	{0x1d49f, 0x44}, 	// 𝒟 -> D
	{0x1d4a2, 0x47}, 	// 𝒢 -> G
	{0x1d4a5, 0x4a}, 	// 𝒥 -> J
	{0x1d4a6, 0x4b}, 	// 𝒦 -> K
	{0x1d4a9, 0x4e}, 	// 𝒩 -> N
	{0x1d4aa, 0x4f}, 	// 𝒪 -> O
	{0x1d4ab, 0x50}, 	// 𝒫 -> P
	{0x1d4ac, 0x51}, 	// 𝒬 -> Q
	{0x1d4ae, 0x53}, 	// 𝒮 -> S
	{0x1d4af, 0x54}, 	// 𝒯 -> T
	{0x1d4b0, 0x55}, 	// 𝒰 -> U
	{0x1d4b1, 0x56}, 	// 𝒱 -> V
	{0x1d4b2, 0x57}, 	// 𝒲 -> W
	{0x1d4b3, 0x58}, 	// 𝒳 -> X
	{0x1d4b4, 0x59}, 	// 𝒴 -> Y
	{0x1d4b5, 0x5a}, 	// 𝒵 -> Z
	{0x1d4b6, 0x61}, 	// 𝒶 -> a
	{0x1d4b7, 0x62}, 	// 𝒷 -> b
	{0x1d4b8, 0x63}, 	// 𝒸 -> c
	{0x1d4b9, 0x64}, 	// 𝒹 -> d
	{0x1d4bb, 0x66}, 	// 𝒻 -> f
	{0x1d4bd, 0x68}, 	// 𝒽 -> h
	{0x1d4be, 0x69}, 	// 𝒾 -> i
	{0x1d4bf, 0x6a}, 	// 𝒿 -> j
	{0x1d4c0, 0x6b}, 	// 𝓀 -> k
	{0x1d4c1, 0x6c}, 	// 𝓁 -> l
	{0x1d4c2, 0x72}, 	// 𝓂 -> r
	{0x1d4c3, 0x6e}, 	// 𝓃 -> n
	{0x1d4c5, 0x70}, 	// 𝓅 -> p
	{0x1d4c6, 0x71}, 	// 𝓆 -> q
	{0x1d4c7, 0x72}, 	// 𝓇 -> r
	{0x1d4c8, 0x73}, 	// 𝓈 -> s
	{0x1d4c9, 0x74}, 	// 𝓉 -> t
	{0x1d4ca, 0x75}, 	// 𝓊 -> u
	{0x1d4cb, 0x76}, 	// 𝓋 -> v
	{0x1d4cc, 0x77}, 	// 𝓌 -> w
	{0x1d4cd, 0x78}, 	// 𝓍 -> x
	{0x1d4ce, 0x79}, 	// 𝓎 -> y
	{0x1d4cf, 0x7a}, 	// 𝓏 -> z
	{0x1d4d0, 0x41}, 	// 𝓐 -> A
	{0x1d4d1, 0x42}, 	// 𝓑 -> B
	{0x1d4d2, 0x43}, 	// 𝓒 -> C
	{0x1d4d3, 0x44}, 	// 𝓓 -> D
	{0x1d4d4, 0x45}, 	// 𝓔 -> E
	{0x1d4d5, 0x46}, 	// 𝓕 -> F
	{0x1d4d6, 0x47}, 	// 𝓖 -> G
	{0x1d4d7, 0x48}, 	// 𝓗 -> H
	{0x1d4d8, 0x6c}, 	// 𝓘 -> l
	{0x1d4d9, 0x4a}, 	// 𝓙 -> J
	{0x1d4da, 0x4b}, 	// 𝓚 -> K
	{0x1d4db, 0x4c}, 	// 𝓛 -> L
	{0x1d4dc, 0x4d}, 	// 𝓜 -> M
	{0x1d4dd, 0x4e}, 	// 𝓝 -> N
	{0x1d4de, 0x4f}, 	// 𝓞 -> O
	{0x1d4df, 0x50}, 	// 𝓟 -> P
	{0x1d4e0, 0x51}, 	// 𝓠 -> Q
	{0x1d4e1, 0x52}, 	// 𝓡 -> R
	{0x1d4e2, 0x53}, 	// 𝓢 -> S
	{0x1d4e3, 0x54}, 	// 𝓣 -> T
	{0x1d4e4, 0x55}, 	// 𝓤 -> U
	{0x1d4e5, 0x56}, 	// 𝓥 -> V
	{0x1d4e6, 0x57}, 	// 𝓦 -> W
	{0x1d4e7, 0x58}, 	// 𝓧 -> X
	{0x1d4e8, 0x59}, 	// 𝓨 -> Y
	{0x1d4e9, 0x5a}, 	// 𝓩 -> Z
	{0x1d4ea, 0x61}, 	// 𝓪 -> a
	{0x1d4eb, 0x62}, 	// 𝓫 -> b
	{0x1d4ec, 0x63}, 	// 𝓬 -> c
	{0x1d4ed, 0x64}, 	// 𝓭 -> d
	{0x1d4ee, 0x65}, 	// 𝓮 -> e
	{0x1d4ef, 0x66}, 	// 𝓯 -> f
	{0x1d4f0, 0x67}, 	// 𝓰 -> g
	{0x1d4f1, 0x68}, 	// 𝓱 -> h
	{0x1d4f2, 0x69}, 	// 𝓲 -> i
	{0x1d4f3, 0x6a}, 	// 𝓳 -> j
	{0x1d4f4, 0x6b}, 	// 𝓴 -> k
	{0x1d4f5, 0x6c}, 	// 𝓵 -> l
	{0x1d4f6, 0x72}, 	// 𝓶 -> r
	{0x1d4f7, 0x6e}, 	// 𝓷 -> n
	{0x1d4f8, 0x6f}, 	// 𝓸 -> o
	{0x1d4f9, 0x70}, 	// 𝓹 -> p
	{0x1d4fa, 0x71}, 	// 𝓺 -> q
	{0x1d4fb, 0x72}, 	// 𝓻 -> r
	{0x1d4fc, 0x73}, 	// 𝓼 -> s
	{0x1d4fd, 0x74}, 	// 𝓽 -> t
	{0x1d4fe, 0x75}, 	// 𝓾 -> u
	{0x1d4ff, 0x76}, 	// 𝓿 -> v
	{0x1d500, 0x77}, 	// 𝔀 -> w
	{0x1d501, 0x78}, 	// 𝔁 -> x
	{0x1d502, 0x79}, 	// 𝔂 -> y
	{0x1d503, 0x7a}, 	// 𝔃 -> z
	{0x1d504, 0x41}, 	// 𝔄 -> A
	{0x1d505, 0x42}, 	// 𝔅 -> B
	{0x1d507, 0x44}, 	// 𝔇 -> D
	{0x1d508, 0x45}, 	// 𝔈 -> E
	{0x1d509, 0x46}, 	// 𝔉 -> F
	{0x1d50a, 0x47}, 	// 𝔊 -> G
	{0x1d50d, 0x4a}, 	// 𝔍 -> J
	{0x1d50e, 0x4b}, 	// 𝔎 -> K
	{0x1d50f, 0x4c}, 	// 𝔏 -> L
	{0x1d510, 0x4d}, 	// 𝔐 -> M
	{0x1d511, 0x4e}, 	// 𝔑 -> N
	{0x1d512, 0x4f}, 	// 𝔒 -> O
	{0x1d513, 0x50}, 	// 𝔓 -> P
	{0x1d514, 0x51}, 	// 𝔔 -> Q
	{0x1d516, 0x53}, 	// 𝔖 -> S
	{0x1d517, 0x54}, 	// 𝔗 -> T
	{0x1d518, 0x55}, 	// 𝔘 -> U
	{0x1d519, 0x56}, 	// 𝔙 -> V
	{0x1d51a, 0x57}, 	// 𝔚 -> W
	{0x1d51b, 0x58}, 	// 𝔛 -> X
	{0x1d51c, 0x59}, 	// 𝔜 -> Y
	{0x1d51e, 0x61}, 	// 𝔞 -> a
	{0x1d51f, 0x62}, 	// 𝔟 -> b
	{0x1d520, 0x63}, 	// 𝔠 -> c
	{0x1d521, 0x64}, 	// 𝔡 -> d
	{0x1d522, 0x65}, 	// 𝔢 -> e
	{0x1d523, 0x66}, 	// 𝔣 -> f
	{0x1d524, 0x67}, 	// 𝔤 -> g
	{0x1d525, 0x68}, 	// 𝔥 -> h
	{0x1d526, 0x69}, 	// 𝔦 -> i
	{0x1d527, 0x6a}, 	// 𝔧 -> j
	{0x1d528, 0x6b}, 	// 𝔨 -> k
	{0x1d529, 0x6c}, 	// 𝔩 -> l
	{0x1d52a, 0x72}, 	// 𝔪 -> r
	{0x1d52b, 0x6e}, 	// 𝔫 -> n
	{0x1d52c, 0x6f}, 	// 𝔬 -> o
	{0x1d52d, 0x70}, 	// 𝔭 -> p
	{0x1d52e, 0x71}, 	// 𝔮 -> q
	{0x1d52f, 0x72}, 	// 𝔯 -> r
	{0x1d530, 0x73}, 	// 𝔰 -> s
	{0x1d531, 0x74}, 	// 𝔱 -> t
	{0x1d532, 0x75}, 	// 𝔲 -> u
	{0x1d533, 0x76}, 	// 𝔳 -> v
	{0x1d534, 0x77}, 	// 𝔴 -> w
	{0x1d535, 0x78}, 	// 𝔵 -> x
	{0x1d536, 0x79}, 	// 𝔶 -> y
	{0x1d537, 0x7a}, 	// 𝔷 -> z
	{0x1d538, 0x41}, 	// 𝔸 -> A
	{0x1d539, 0x42}, 	// 𝔹 -> B
	{0x1d53b, 0x44}, 	// 𝔻 -> D
	{0x1d53c, 0x45}, 	// 𝔼 -> E
	{0x1d53d, 0x46}, 	// 𝔽 -> F
	{0x1d53e, 0x47}, 	// 𝔾 -> G
	{0x1d540, 0x6c}, 	// 𝕀 -> l
	{0x1d541, 0x4a}, 	// 𝕁 -> J
	{0x1d542, 0x4b}, 	// 𝕂 -> K
	{0x1d543, 0x4c}, 	// 𝕃 -> L
	{0x1d544, 0x4d}, 	// 𝕄 -> M
	{0x1d546, 0x4f}, 	// 𝕆 -> O
	{0x1d54a, 0x53}, 	// 𝕊 -> S
	{0x1d54b, 0x54}, 	// 𝕋 -> T
	{0x1d54c, 0x55}, 	// 𝕌 -> U
	{0x1d54d, 0x56}, 	// 𝕍 -> V
	{0x1d54e, 0x57}, 	// 𝕎 -> W
	{0x1d54f, 0x58}, 	// 𝕏 -> X
	{0x1d550, 0x59}, 	// 𝕐 -> Y
	{0x1d552, 0x61}, 	// 𝕒 -> a
	{0x1d553, 0x62}, 	// 𝕓 -> b
	{0x1d554, 0x63}, 	// 𝕔 -> c
	{0x1d555, 0x64}, 	// 𝕕 -> d
	{0x1d556, 0x65}, 	// 𝕖 -> e
	{0x1d557, 0x66}, 	// 𝕗 -> f
	{0x1d558, 0x67}, 	// 𝕘 -> g
	{0x1d559, 0x68}, 	// 𝕙 -> h
	{0x1d55a, 0x69}, 	// 𝕚 -> i
	{0x1d55b, 0x6a}, 	// 𝕛 -> j
	{0x1d55c, 0x6b}, 	// 𝕜 -> k
	{0x1d55d, 0x6c}, 	// 𝕝 -> l
	{0x1d55e, 0x72}, 	// 𝕞 -> r
	{0x1d55f, 0x6e}, 	// 𝕟 -> n
	{0x1d560, 0x6f}, 	// 𝕠 -> o
	{0x1d561, 0x70}, 	// 𝕡 -> p
	{0x1d562, 0x71}, 	// 𝕢 -> q
	{0x1d563, 0x72}, 	// 𝕣 -> r
	{0x1d564, 0x73}, 	// 𝕤 -> s
	{0x1d565, 0x74}, 	// 𝕥 -> t
	{0x1d566, 0x75}, 	// 𝕦 -> u
	{0x1d567, 0x76}, 	// 𝕧 -> v
	{0x1d568, 0x77}, 	// 𝕨 -> w
	{0x1d569, 0x78}, 	// 𝕩 -> x
	{0x1d56a, 0x79}, 	// 𝕪 -> y
	{0x1d56b, 0x7a}, 	// 𝕫 -> z
	{0x1d56c, 0x41}, 	// 𝕬 -> A
	{0x1d56d, 0x42}, 	// 𝕭 -> B
	{0x1d56e, 0x43}, 	// 𝕮 -> C
	{0x1d56f, 0x44}, 	// 𝕯 -> D
	{0x1d570, 0x45}, 	// 𝕰 -> E
	{0x1d571, 0x46}, 	// 𝕱 -> F
	{0x1d572, 0x47}, 	// 𝕲 -> G
	{0x1d573, 0x48}, 	// 𝕳 -> H
	{0x1d574, 0x6c}, 	// 𝕴 -> l
	{0x1d575, 0x4a}, 	// 𝕵 -> J
	{0x1d576, 0x4b}, 	// 𝕶 -> K
	{0x1d577, 0x4c}, 	// 𝕷 -> L
	{0x1d578, 0x4d}, 	// 𝕸 -> M
	{0x1d579, 0x4e}, 	// 𝕹 -> N
	{0x1d57a, 0x4f}, 	// 𝕺 -> O
	{0x1d57b, 0x50}, 	// 𝕻 -> P
	{0x1d57c, 0x51}, 	// 𝕼 -> Q
	{0x1d57d, 0x52}, 	// 𝕽 -> R
	{0x1d57e, 0x53}, 	// 𝕾 -> S
	{0x1d57f, 0x54}, 	// 𝕿 -> T
	{0x1d580, 0x55}, 	// 𝖀 -> U
	{0x1d581, 0x56}, 	// 𝖁 -> V
	{0x1d582, 0x57}, 	// 𝖂 -> W
	{0x1d583, 0x58}, 	// 𝖃 -> X
	{0x1d584, 0x59}, 	// 𝖄 -> Y
	{0x1d585, 0x5a}, 	// 𝖅 -> Z
	{0x1d586, 0x61}, 	// 𝖆 -> a
	{0x1d587, 0x62}, 	// 𝖇 -> b
	{0x1d588, 0x63}, 	// 𝖈 -> c
	{0x1d589, 0x64}, 	// 𝖉 -> d
	{0x1d58a, 0x65}, 	// 𝖊 -> e
	{0x1d58b, 0x66}, 	// 𝖋 -> f
	{0x1d58c, 0x67}, 	// 𝖌 -> g
	{0x1d58d, 0x68}, 	// 𝖍 -> h
	{0x1d58e, 0x69}, 	// 𝖎 -> i
	{0x1d58f, 0x6a}, 	// 𝖏 -> j
	{0x1d590, 0x6b}, 	// 𝖐 -> k
	{0x1d591, 0x6c}, 	// 𝖑 -> l
	{0x1d592, 0x72}, 	// 𝖒 -> r
	{0x1d593, 0x6e}, 	// 𝖓 -> n
	{0x1d594, 0x6f}, 	// 𝖔 -> o
	{0x1d595, 0x70}, 	// 𝖕 -> p
	{0x1d596, 0x71}, 	// 𝖖 -> q
	{0x1d597, 0x72}, 	// 𝖗 -> r
	{0x1d598, 0x73}, 	// 𝖘 -> s
	{0x1d599, 0x74}, 	// 𝖙 -> t
	{0x1d59a, 0x75}, 	// 𝖚 -> u
	{0x1d59b, 0x76}, 	// 𝖛 -> v
	{0x1d59c, 0x77}, 	// 𝖜 -> w
	{0x1d59d, 0x78}, 	// 𝖝 -> x
	{0x1d59e, 0x79}, 	// 𝖞 -> y
	{0x1d59f, 0x7a}, 	// 𝖟 -> z
	{0x1d5a0, 0x41}, 	// 𝖠 -> A
	{0x1d5a1, 0x42}, 	// 𝖡 -> B
	{0x1d5a2, 0x43}, 	// 𝖢 -> C
	{0x1d5a3, 0x44}, 	// 𝖣 -> D
	{0x1d5a4, 0x45}, 	// 𝖤 -> E
	{0x1d5a5, 0x46}, 	// 𝖥 -> F
	{0x1d5a6, 0x47}, 	// 𝖦 -> G
	{0x1d5a7, 0x48}, 	// 𝖧 -> H
	{0x1d5a8, 0x6c}, 	// 𝖨 -> l
	{0x1d5a9, 0x4a}, 	// 𝖩 -> J
	{0x1d5aa, 0x4b}, 	// 𝖪 -> K
	{0x1d5ab, 0x4c}, 	// 𝖫 -> L
	{0x1d5ac, 0x4d}, 	// 𝖬 -> M
	{0x1d5ad, 0x4e}, 	// 𝖭 -> N
	{0x1d5ae, 0x4f}, 	// 𝖮 -> O
	{0x1d5af, 0x50}, 	// 𝖯 -> P
	{0x1d5b0, 0x51}, 	// 𝖰 -> Q
	{0x1d5b1, 0x52}, 	// 𝖱 -> R
	{0x1d5b2, 0x53}, 	// 𝖲 -> S
	{0x1d5b3, 0x54}, 	// 𝖳 -> T
	{0x1d5b4, 0x55}, 	// 𝖴 -> U
	{0x1d5b5, 0x56}, 	// 𝖵 -> V
	{0x1d5b6, 0x57}, 	// 𝖶 -> W
	{0x1d5b7, 0x58}, 	// 𝖷 -> X
	{0x1d5b8, 0x59}, 	// 𝖸 -> Y
	{0x1d5b9, 0x5a}, 	// 𝖹 -> Z
	{0x1d5ba, 0x61}, 	// 𝖺 -> a
	{0x1d5bb, 0x62}, 	// 𝖻 -> b
	{0x1d5bc, 0x63}, 	// 𝖼 -> c
	{0x1d5bd, 0x64}, 	// 𝖽 -> d
	{0x1d5be, 0x65}, 	// 𝖾 -> e
	{0x1d5bf, 0x66}, 	// 𝖿 -> f
	{0x1d5c0, 0x67}, 	// 𝗀 -> g
	{0x1d5c1, 0x68}, 	// 𝗁 -> h
	{0x1d5c2, 0x69}, 	// 𝗂 -> i
	{0x1d5c3, 0x6a}, 	// 𝗃 -> j
	{0x1d5c4, 0x6b}, 	// 𝗄 -> k
	{0x1d5c5, 0x6c}, 	// 𝗅 -> l
	{0x1d5c6, 0x72}, 	// 𝗆 -> r
	{0x1d5c7, 0x6e}, 	// 𝗇 -> n
	{0x1d5c8, 0x6f}, 	// 𝗈 -> o
	{0x1d5c9, 0x70}, 	// 𝗉 -> p
	{0x1d5ca, 0x71}, 	// 𝗊 -> q
	{0x1d5cb, 0x72}, 	// 𝗋 -> r
	{0x1d5cc, 0x73}, 	// 𝗌 -> s
	{0x1d5cd, 0x74}, 	// 𝗍 -> t
	{0x1d5ce, 0x75}, 	// 𝗎 -> u
	{0x1d5cf, 0x76}, 	// 𝗏 -> v
	{0x1d5d0, 0x77}, 	// 𝗐 -> w
	{0x1d5d1, 0x78}, 	// 𝗑 -> x
	{0x1d5d2, 0x79}, 	// 𝗒 -> y
	{0x1d5d3, 0x7a}, 	// 𝗓 -> z
	{0x1d5d4, 0x41}, 	// 𝗔 -> A
	{0x1d5d5, 0x42}, 	// 𝗕 -> B
	{0x1d5d6, 0x43}, 	// 𝗖 -> C
	{0x1d5d7, 0x44}, 	// 𝗗 -> D
	{0x1d5d8, 0x45}, 	// 𝗘 -> E
	{0x1d5d9, 0x46}, 	// 𝗙 -> F
	{0x1d5da, 0x47}, 	// 𝗚 -> G
	{0x1d5db, 0x48}, 	// 𝗛 -> H
	{0x1d5dc, 0x6c}, 	// 𝗜 -> l
	{0x1d5dd, 0x4a}, 	// 𝗝 -> J
	{0x1d5de, 0x4b}, 	// 𝗞 -> K
	{0x1d5df, 0x4c}, 	// 𝗟 -> L
	{0x1d5e0, 0x4d}, 	// 𝗠 -> M
	{0x1d5e1, 0x4e}, 	// 𝗡 -> N
	{0x1d5e2, 0x4f}, 	// 𝗢 -> O
	{0x1d5e3, 0x50}, 	// 𝗣 -> P
	{0x1d5e4, 0x51}, 	// 𝗤 -> Q
	{0x1d5e5, 0x52}, 	// 𝗥 -> R
	{0x1d5e6, 0x53}, 	// 𝗦 -> S
	{0x1d5e7, 0x54}, 	// 𝗧 -> T
	{0x1d5e8, 0x55}, 	// 𝗨 -> U
	{0x1d5e9, 0x56}, 	// 𝗩 -> V
	{0x1d5ea, 0x57}, 	// 𝗪 -> W
	{0x1d5eb, 0x58}, 	// 𝗫 -> X
	{0x1d5ec, 0x59}, 	// 𝗬 -> Y
	{0x1d5ed, 0x5a}, 	// 𝗭 -> Z
	{0x1d5ee, 0x61}, 	// 𝗮 -> a
	{0x1d5ef, 0x62}, 	// 𝗯 -> b
	{0x1d5f0, 0x63}, 	// 𝗰 -> c
	{0x1d5f1, 0x64}, 	// 𝗱 -> d
	{0x1d5f2, 0x65}, 	// 𝗲 -> e
	{0x1d5f3, 0x66}, 	// 𝗳 -> f
	{0x1d5f4, 0x67}, 	// 𝗴 -> g
	{0x1d5f5, 0x68}, 	// 𝗵 -> h
	{0x1d5f6, 0x69}, 	// 𝗶 -> i
	{0x1d5f7, 0x6a}, 	// 𝗷 -> j
	{0x1d5f8, 0x6b}, 	// 𝗸 -> k
	{0x1d5f9, 0x6c}, 	// 𝗹 -> l
	{0x1d5fa, 0x72}, 	// 𝗺 -> r
	{0x1d5fb, 0x6e}, 	// 𝗻 -> n
	{0x1d5fc, 0x6f}, 	// 𝗼 -> o
	{0x1d5fd, 0x70}, 	// 𝗽 -> p
	{0x1d5fe, 0x71}, 	// 𝗾 -> q
	{0x1d5ff, 0x72}, 	// 𝗿 -> r
	{0x1d600, 0x73}, 	// 𝘀 -> s
	{0x1d601, 0x74}, 	// 𝘁 -> t
	{0x1d602, 0x75}, 	// 𝘂 -> u
	{0x1d603, 0x76}, 	// 𝘃 -> v
	{0x1d604, 0x77}, 	// 𝘄 -> w
	{0x1d605, 0x78}, 	// 𝘅 -> x
	{0x1d606, 0x79}, 	// 𝘆 -> y
	{0x1d607, 0x7a}, 	// 𝘇 -> z
	{0x1d608, 0x41}, 	// 𝘈 -> A
	{0x1d609, 0x42}, 	// 𝘉 -> B
	{0x1d60a, 0x43}, 	// 𝘊 -> C
	{0x1d60b, 0x44}, 	// 𝘋 -> D
	{0x1d60c, 0x45}, 	// 𝘌 -> E
	{0x1d60d, 0x46}, 	// 𝘍 -> F
	{0x1d60e, 0x47}, 	// 𝘎 -> G
	{0x1d60f, 0x48}, 	// 𝘏 -> H
	{0x1d610, 0x6c}, 	// 𝘐 -> l
	{0x1d611, 0x4a}, 	// 𝘑 -> J
	{0x1d612, 0x4b}, 	// 𝘒 -> K
	{0x1d613, 0x4c}, 	// 𝘓 -> L
	{0x1d614, 0x4d}, 	// 𝘔 -> M
	{0x1d615, 0x4e}, 	// 𝘕 -> N
	{0x1d616, 0x4f}, 	// 𝘖 -> O
	{0x1d617, 0x50}, 	// 𝘗 -> P
	{0x1d618, 0x51}, 	// 𝘘 -> Q
	{0x1d619, 0x52}, 	// 𝘙 -> R
	{0x1d61a, 0x53}, 	// 𝘚 -> S
	{0x1d61b, 0x54}, 	// 𝘛 -> T
	{0x1d61c, 0x55}, 	// 𝘜 -> U
	{0x1d61d, 0x56}, 	// 𝘝 -> V
	{0x1d61e, 0x57}, 	// 𝘞 -> W
	{0x1d61f, 0x58}, 	// 𝘟 -> X
	{0x1d620, 0x59}, 	// 𝘠 -> Y
	{0x1d621, 0x5a}, 	// 𝘡 -> Z
	{0x1d622, 0x61}, 	// 𝘢 -> a
	{0x1d623, 0x62}, 	// 𝘣 -> b
	{0x1d624, 0x63}, 	// 𝘤 -> c
	{0x1d625, 0x64}, 	// 𝘥 -> d
	{0x1d626, 0x65}, 	// 𝘦 -> e
	{0x1d627, 0x66}, 	// 𝘧 -> f
	{0x1d628, 0x67}, 	// 𝘨 -> g
	{0x1d629, 0x68}, 	// 𝘩 -> h
	{0x1d62a, 0x69}, 	// 𝘪 -> i
	{0x1d62b, 0x6a}, 	// 𝘫 -> j
	{0x1d62c, 0x6b}, 	// 𝘬 -> k
	{0x1d62d, 0x6c}, 	// 𝘭 -> l
	{0x1d62e, 0x72}, 	// 𝘮 -> r
	{0x1d62f, 0x6e}, 	// 𝘯 -> n
	{0x1d630, 0x6f}, 	// 𝘰 -> o
	{0x1d631, 0x70}, 	// 𝘱 -> p
	{0x1d632, 0x71}, 	// 𝘲 -> q
	{0x1d633, 0x72}, 	// 𝘳 -> r
	{0x1d634, 0x73}, 	// 𝘴 -> s
	{0x1d635, 0x74}, 	// 𝘵 -> t
	{0x1d636, 0x75}, 	// 𝘶 -> u
	{0x1d637, 0x76}, 	// 𝘷 -> v
	{0x1d638, 0x77}, 	// 𝘸 -> w
	{0x1d639, 0x78}, 	// 𝘹 -> x
	{0x1d63a, 0x79}, 	// 𝘺 -> y
	{0x1d63b, 0x7a}, 	// 𝘻 -> z
	{0x1d63c, 0x41}, 	// 𝘼 -> A
	{0x1d63d, 0x42}, 	// 𝘽 -> B
	{0x1d63e, 0x43}, 	// 𝘾 -> C
	{0x1d63f, 0x44}, 	// 𝘿 -> D
	{0x1d640, 0x45}, 	// 𝙀 -> E
	{0x1d641, 0x46}, 	// 𝙁 -> F
	{0x1d642, 0x47}, 	// 𝙂 -> G
	{0x1d643, 0x48}, 	// 𝙃 -> H
	{0x1d644, 0x6c}, 	// 𝙄 -> l
	{0x1d645, 0x4a}, 	// 𝙅 -> J
	{0x1d646, 0x4b}, 	// 𝙆 -> K
	{0x1d647, 0x4c}, 	// 𝙇 -> L
	{0x1d648, 0x4d}, 	// 𝙈 -> M
	{0x1d649, 0x4e}, 	// 𝙉 -> N
	{0x1d64a, 0x4f}, 	// 𝙊 -> O
	{0x1d64b, 0x50}, 	// 𝙋 -> P
	{0x1d64c, 0x51}, 	// 𝙌 -> Q
	{0x1d64d, 0x52}, 	// 𝙍 -> R
	{0x1d64e, 0x53}, 	// 𝙎 -> S
	{0x1d64f, 0x54}, 	// 𝙏 -> T
	{0x1d650, 0x55}, 	// 𝙐 -> U
	{0x1d651, 0x56}, 	// 𝙑 -> V
	{0x1d652, 0x57}, 	// 𝙒 -> W
	{0x1d653, 0x58}, 	// 𝙓 -> X
	{0x1d654, 0x59}, 	// 𝙔 -> Y
	{0x1d655, 0x5a}, 	// 𝙕 -> Z
	{0x1d656, 0x61}, 	// 𝙖 -> a
	{0x1d657, 0x62}, 	// 𝙗 -> b
	{0x1d658, 0x63}, 	// 𝙘 -> c
	{0x1d659, 0x64}, 	// 𝙙 -> d
	{0x1d65a, 0x65}, 	// 𝙚 -> e
	{0x1d65b, 0x66}, 	// 𝙛 -> f
	{0x1d65c, 0x67}, 	// 𝙜 -> g
	{0x1d65d, 0x68}, 	// 𝙝 -> h
	{0x1d65e, 0x69}, 	// 𝙞 -> i
	{0x1d65f, 0x6a}, 	// 𝙟 -> j
	{0x1d660, 0x6b}, 	// 𝙠 -> k
	{0x1d661, 0x6c}, 	// 𝙡 -> l
	{0x1d662, 0x72}, 	// 𝙢 -> r
	{0x1d663, 0x6e}, 	// 𝙣 -> n
	{0x1d664, 0x6f}, 	// 𝙤 -> o
	{0x1d665, 0x70}, 	// 𝙥 -> p
	{0x1d666, 0x71}, 	// 𝙦 -> q
	{0x1d667, 0x72}, 	// 𝙧 -> r
	{0x1d668, 0x73}, 	// 𝙨 -> s
	{0x1d669, 0x74}, 	// 𝙩 -> t
	{0x1d66a, 0x75}, 	// 𝙪 -> u
	{0x1d66b, 0x76}, 	// 𝙫 -> v
	{0x1d66c, 0x77}, 	// 𝙬 -> w
	{0x1d66d, 0x78}, 	// 𝙭 -> x
	{0x1d66e, 0x79}, 	// 𝙮 -> y
	{0x1d66f, 0x7a}, 	// 𝙯 -> z
	{0x1d670, 0x41}, 	// 𝙰 -> A
	{0x1d671, 0x42}, 	// 𝙱 -> B
	{0x1d672, 0x43}, 	// 𝙲 -> C
	{0x1d673, 0x44}, 	// 𝙳 -> D
	{0x1d674, 0x45}, 	// 𝙴 -> E
	{0x1d675, 0x46}, 	// 𝙵 -> F
	{0x1d676, 0x47}, 	// 𝙶 -> G
	{0x1d677, 0x48}, 	// 𝙷 -> H
	{0x1d678, 0x6c}, 	// 𝙸 -> l
	{0x1d679, 0x4a}, 	// 𝙹 -> J
	{0x1d67a, 0x4b}, 	// 𝙺 -> K
	{0x1d67b, 0x4c}, 	// 𝙻 -> L
	{0x1d67c, 0x4d}, 	// 𝙼 -> M
	{0x1d67d, 0x4e}, 	// 𝙽 -> N
	{0x1d67e, 0x4f}, 	// 𝙾 -> O
	{0x1d67f, 0x50}, 	// 𝙿 -> P
	{0x1d680, 0x51}, 	// 𝚀 -> Q
	{0x1d681, 0x52}, 	// 𝚁 -> R
	{0x1d682, 0x53}, 	// 𝚂 -> S
	{0x1d683, 0x54}, 	// 𝚃 -> T
	{0x1d684, 0x55}, 	// 𝚄 -> U
	{0x1d685, 0x56}, 	// 𝚅 -> V
	{0x1d686, 0x57}, 	// 𝚆 -> W
	{0x1d687, 0x58}, 	// 𝚇 -> X
	{0x1d688, 0x59}, 	// 𝚈 -> Y
	{0x1d689, 0x5a}, 	// 𝚉 -> Z
	{0x1d68a, 0x61}, 	// 𝚊 -> a
	{0x1d68b, 0x62}, 	// 𝚋 -> b
	{0x1d68c, 0x63}, 	// 𝚌 -> c
	{0x1d68d, 0x64}, 	// 𝚍 -> d
	{0x1d68e, 0x65}, 	// 𝚎 -> e
	{0x1d68f, 0x66}, 	// 𝚏 -> f
	{0x1d690, 0x67}, 	// 𝚐 -> g
	{0x1d691, 0x68}, 	// 𝚑 -> h
	{0x1d692, 0x69}, 	// 𝚒 -> i
	{0x1d693, 0x6a}, 	// 𝚓 -> j
	{0x1d694, 0x6b}, 	// 𝚔 -> k
	{0x1d695, 0x6c}, 	// 𝚕 -> l
	{0x1d696, 0x72}, 	// 𝚖 -> r
	{0x1d697, 0x6e}, 	// 𝚗 -> n
	{0x1d698, 0x6f}, 	// 𝚘 -> o
	{0x1d699, 0x70}, 	// 𝚙 -> p
	{0x1d69a, 0x71}, 	// 𝚚 -> q
	{0x1d69b, 0x72}, 	// 𝚛 -> r
	{0x1d69c, 0x73}, 	// 𝚜 -> s
	{0x1d69d, 0x74}, 	// 𝚝 -> t
	{0x1d69e, 0x75}, 	// 𝚞 -> u
	{0x1d69f, 0x76}, 	// 𝚟 -> v
	{0x1d6a0, 0x77}, 	// 𝚠 -> w
	{0x1d6a1, 0x78}, 	// 𝚡 -> x
	{0x1d6a2, 0x79}, 	// 𝚢 -> y
	{0x1d6a3, 0x7a}, 	// 𝚣 -> z
	{0x1d6a4, 0x69}, 	// 𝚤 -> i
	{0x1d6a5, 0x6a}, 	// 𝚥 -> j
	{0x1d6a8, 0x41}, 	// 𝚨 -> A
	{0x1d6a9, 0x42}, 	// 𝚩 -> B
	{0x1d6ac, 0x45}, 	// 𝚬 -> E
	{0x1d6ad, 0x5a}, 	// 𝚭 -> Z
	{0x1d6ae, 0x48}, 	// 𝚮 -> H
	{0x1d6af, 0x4f}, 	// 𝚯 -> O
	{0x1d6b0, 0x6c}, 	// 𝚰 -> l
	{0x1d6b1, 0x4b}, 	// 𝚱 -> K
	{0x1d6b3, 0x4d}, 	// 𝚳 -> M
	{0x1d6b4, 0x4e}, 	// 𝚴 -> N
	{0x1d6b6, 0x4f}, 	// 𝚶 -> O
	{0x1d6b8, 0x50}, 	// 𝚸 -> P
	{0x1d6b9, 0x4f}, 	// 𝚹 -> O
	{0x1d6bb, 0x54}, 	// 𝚻 -> T
	{0x1d6bc, 0x59}, 	// 𝚼 -> Y
	{0x1d6be, 0x58}, 	// 𝚾 -> X
	{0x1d6c2, 0x61}, 	// 𝛂 -> a
	{0x1d6c4, 0x79}, 	// 𝛄 -> y
	{0x1d6c8, 0x6e}, 	// 𝛈 -> n
	{0x1d6c9, 0x4f}, 	// 𝛉 -> O
	{0x1d6ca, 0x69}, 	// 𝛊 -> i
	{0x1d6cb, 0x6b}, 	// 𝛋 -> k
	{0x1d6ce, 0x76}, 	// 𝛎 -> v
	{0x1d6d0, 0x6f}, 	// 𝛐 -> o
	{0x1d6d1, 0x6e}, 	// 𝛑 -> n
	{0x1d6d2, 0x70}, 	// 𝛒 -> p
	{0x1d6d4, 0x6f}, 	// 𝛔 -> o
	{0x1d6d5, 0x74}, 	// 𝛕 -> t
	{0x1d6d6, 0x75}, 	// 𝛖 -> u
	{0x1d6dd, 0x4f}, 	// 𝛝 -> O
	{0x1d6de, 0x6b}, 	// 𝛞 -> k
	{0x1d6e0, 0x70}, 	// 𝛠 -> p
	{0x1d6e2, 0x41}, 	// 𝛢 -> A
	{0x1d6e3, 0x42}, 	// 𝛣 -> B
	{0x1d6e6, 0x45}, 	// 𝛦 -> E
	{0x1d6e7, 0x5a}, 	// 𝛧 -> Z
	{0x1d6e8, 0x48}, 	// 𝛨 -> H
	{0x1d6e9, 0x4f}, 	// 𝛩 -> O
	{0x1d6ea, 0x6c}, 	// 𝛪 -> l
	{0x1d6eb, 0x4b}, 	// 𝛫 -> K
	{0x1d6ed, 0x4d}, 	// 𝛭 -> M
	{0x1d6ee, 0x4e}, 	// 𝛮 -> N
	{0x1d6f0, 0x4f}, 	// 𝛰 -> O
	{0x1d6f2, 0x50}, 	// 𝛲 -> P
	{0x1d6f3, 0x4f}, 	// 𝛳 -> O
	{0x1d6f5, 0x54}, 	// 𝛵 -> T
	{0x1d6f6, 0x59}, 	// 𝛶 -> Y
	{0x1d6f8, 0x58}, 	// 𝛸 -> X
	{0x1d6fc, 0x61}, 	// 𝛼 -> a
	{0x1d6fe, 0x79}, 	// 𝛾 -> y
	{0x1d702, 0x6e}, 	// 𝜂 -> n
	{0x1d703, 0x4f}, 	// 𝜃 -> O
	{0x1d704, 0x69}, 	// 𝜄 -> i
	{0x1d705, 0x6b}, 	// 𝜅 -> k
	{0x1d708, 0x76}, 	// 𝜈 -> v
	{0x1d70a, 0x6f}, 	// 𝜊 -> o
	{0x1d70b, 0x6e}, 	// 𝜋 -> n
	{0x1d70c, 0x70}, 	// 𝜌 -> p
	{0x1d70e, 0x6f}, 	// 𝜎 -> o
	{0x1d70f, 0x74}, 	// 𝜏 -> t
	{0x1d710, 0x75}, 	// 𝜐 -> u
	{0x1d717, 0x4f}, 	// 𝜗 -> O
	{0x1d718, 0x6b}, 	// 𝜘 -> k
	{0x1d71a, 0x70}, 	// 𝜚 -> p
	{0x1d71c, 0x41}, 	// 𝜜 -> A
	{0x1d71d, 0x42}, 	// 𝜝 -> B
	{0x1d720, 0x45}, 	// 𝜠 -> E
	{0x1d721, 0x5a}, 	// 𝜡 -> Z
	{0x1d722, 0x48}, 	// 𝜢 -> H
	{0x1d723, 0x4f}, 	// 𝜣 -> O
	{0x1d724, 0x6c}, 	// 𝜤 -> l
	{0x1d725, 0x4b}, 	// 𝜥 -> K
	{0x1d727, 0x4d}, 	// 𝜧 -> M
	{0x1d728, 0x4e}, 	// 𝜨 -> N
	{0x1d72a, 0x4f}, 	// 𝜪 -> O
	{0x1d72c, 0x50}, 	// 𝜬 -> P
	{0x1d72d, 0x4f}, 	// 𝜭 -> O
	{0x1d72f, 0x54}, 	// 𝜯 -> T
	{0x1d730, 0x59}, 	// 𝜰 -> Y
	{0x1d732, 0x58}, 	// 𝜲 -> X
	{0x1d736, 0x61}, 	// 𝜶 -> a
	{0x1d738, 0x79}, 	// 𝜸 -> y
	{0x1d73c, 0x6e}, 	// 𝜼 -> n
	{0x1d73d, 0x4f}, 	// 𝜽 -> O
	{0x1d73e, 0x69}, 	// 𝜾 -> i
	{0x1d73f, 0x6b}, 	// 𝜿 -> k
	{0x1d742, 0x76}, 	// 𝝂 -> v
	{0x1d744, 0x6f}, 	// 𝝄 -> o
	{0x1d745, 0x6e}, 	// 𝝅 -> n
	{0x1d746, 0x70}, 	// 𝝆 -> p
	{0x1d748, 0x6f}, 	// 𝝈 -> o
	{0x1d74a, 0x75}, 	// 𝝊 -> u
	{0x1d751, 0x4f}, 	// 𝝑 -> O
	{0x1d752, 0x6b}, 	// 𝝒 -> k
	{0x1d754, 0x70}, 	// 𝝔 -> p
	{0x1d756, 0x41}, 	// 𝝖 -> A
	{0x1d757, 0x42}, 	// 𝝗 -> B
	{0x1d75a, 0x45}, 	// 𝝚 -> E
	{0x1d75b, 0x5a}, 	// 𝝛 -> Z
	{0x1d75c, 0x48}, 	// 𝝜 -> H
	{0x1d75d, 0x4f}, 	// 𝝝 -> O
	{0x1d75e, 0x6c}, 	// 𝝞 -> l
	{0x1d75f, 0x4b}, 	// 𝝟 -> K
	{0x1d761, 0x4d}, 	// 𝝡 -> M
	{0x1d762, 0x4e}, 	// 𝝢 -> N
	{0x1d764, 0x4f}, 	// 𝝤 -> O
	{0x1d766, 0x50}, 	// 𝝦 -> P
	{0x1d767, 0x4f}, 	// 𝝧 -> O
	{0x1d769, 0x54}, 	// 𝝩 -> T
	{0x1d76a, 0x59}, 	// 𝝪 -> Y
	{0x1d76c, 0x58}, 	// 𝝬 -> X
	{0x1d770, 0x61}, 	// 𝝰 -> a
	{0x1d772, 0x79}, 	// 𝝲 -> y
	{0x1d776, 0x6e}, 	// 𝝶 -> n
	{0x1d777, 0x4f}, 	// 𝝷 -> O
	{0x1d778, 0x69}, 	// 𝝸 -> i
	{0x1d779, 0x6b}, 	// 𝝹 -> k
	{0x1d77c, 0x76}, 	// 𝝼 -> v
	{0x1d77e, 0x6f}, 	// 𝝾 -> o
	{0x1d77f, 0x6e}, 	// 𝝿 -> n
	{0x1d780, 0x70}, 	// 𝞀 -> p
	{0x1d782, 0x6f}, 	// 𝞂 -> o
	{0x1d784, 0x75}, 	// 𝞄 -> u
	{0x1d78b, 0x4f}, 	// 𝞋 -> O
	{0x1d78c, 0x6b}, 	// 𝞌 -> k
	{0x1d78e, 0x70}, 	// 𝞎 -> p
	{0x1d790, 0x41}, 	// 𝞐 -> A
	{0x1d791, 0x42}, 	// 𝞑 -> B
	{0x1d794, 0x45}, 	// 𝞔 -> E
	{0x1d795, 0x5a}, 	// 𝞕 -> Z
	{0x1d796, 0x48}, 	// 𝞖 -> H
	{0x1d797, 0x4f}, 	// 𝞗 -> O
	{0x1d798, 0x6c}, 	// 𝞘 -> l
	{0x1d799, 0x4b}, 	// 𝞙 -> K
	{0x1d79b, 0x4d}, 	// 𝞛 -> M
	{0x1d79c, 0x4e}, 	// 𝞜 -> N
	{0x1d79e, 0x4f}, 	// 𝞞 -> O
	{0x1d7a0, 0x50}, 	// 𝞠 -> P
	{0x1d7a1, 0x4f}, 	// 𝞡 -> O
	{0x1d7a3, 0x54}, 	// 𝞣 -> T
	{0x1d7a4, 0x59}, 	// 𝞤 -> Y
	{0x1d7a6, 0x58}, 	// 𝞦 -> X
	{0x1d7aa, 0x61}, 	// 𝞪 -> a
	{0x1d7ac, 0x79}, 	// 𝞬 -> y
	{0x1d7b0, 0x6e}, 	// 𝞰 -> n
	{0x1d7b1, 0x4f}, 	// 𝞱 -> O
	{0x1d7b2, 0x69}, 	// 𝞲 -> i
	{0x1d7b3, 0x6b}, 	// 𝞳 -> k
	{0x1d7b6, 0x76}, 	// 𝞶 -> v
	{0x1d7b8, 0x6f}, 	// 𝞸 -> o
	{0x1d7ba, 0x70}, 	// 𝞺 -> p
	{0x1d7bc, 0x6f}, 	// 𝞼 -> o
	{0x1d7bd, 0x74}, 	// 𝞽 -> t
	{0x1d7be, 0x75}, 	// 𝞾 -> u
	{0x1d7c5, 0x4f}, 	// 𝟅 -> O
	{0x1d7c6, 0x6b}, 	// 𝟆 -> k
	{0x1d7c8, 0x70}, 	// 𝟈 -> p
	{0x1d7ca, 0x46}, 	// 𝟊 -> F
	{0x1d7ce, 0x4f}, 	// 𝟎 -> O
	{0x1d7cf, 0x6c}, 	// 𝟏 -> l
	{0x1d7d0, 0x32}, 	// 𝟐 -> 2
	{0x1d7d1, 0x33}, 	// 𝟑 -> 3
	{0x1d7d2, 0x34}, 	// 𝟒 -> 4
	{0x1d7d3, 0x35}, 	// 𝟓 -> 5
	{0x1d7d4, 0x36}, 	// 𝟔 -> 6
	{0x1d7d5, 0x37}, 	// 𝟕 -> 7
	{0x1d7d6, 0x38}, 	// 𝟖 -> 8
	{0x1d7d7, 0x39}, 	// 𝟗 -> 9
	{0x1d7d8, 0x4f}, 	// 𝟘 -> O
	{0x1d7d9, 0x6c}, 	// 𝟙 -> l
	{0x1d7da, 0x32}, 	// 𝟚 -> 2
	{0x1d7db, 0x33}, 	// 𝟛 -> 3
	{0x1d7dc, 0x34}, 	// 𝟜 -> 4
	{0x1d7dd, 0x35}, 	// 𝟝 -> 5
	{0x1d7de, 0x36}, 	// 𝟞 -> 6
	{0x1d7df, 0x37}, 	// 𝟟 -> 7
	{0x1d7e0, 0x38}, 	// 𝟠 -> 8
	{0x1d7e1, 0x39}, 	// 𝟡 -> 9
	{0x1d7e2, 0x4f}, 	// 𝟢 -> O
	{0x1d7e3, 0x6c}, 	// 𝟣 -> l
	{0x1d7e4, 0x32}, 	// 𝟤 -> 2
	{0x1d7e5, 0x33}, 	// 𝟥 -> 3
	{0x1d7e6, 0x34}, 	// 𝟦 -> 4
	{0x1d7e7, 0x35}, 	// 𝟧 -> 5
	{0x1d7e8, 0x36}, 	// 𝟨 -> 6
	{0x1d7e9, 0x37}, 	// 𝟩 -> 7
	{0x1d7ea, 0x38}, 	// 𝟪 -> 8
	{0x1d7eb, 0x39}, 	// 𝟫 -> 9
	{0x1d7ec, 0x4f}, 	// 𝟬 -> O
	{0x1d7ed, 0x6c}, 	// 𝟭 -> l
	{0x1d7ee, 0x32}, 	// 𝟮 -> 2
	{0x1d7ef, 0x33}, 	// 𝟯 -> 3
	{0x1d7f0, 0x34}, 	// 𝟰 -> 4
	{0x1d7f1, 0x35}, 	// 𝟱 -> 5
	{0x1d7f2, 0x36}, 	// 𝟲 -> 6
	{0x1d7f3, 0x37}, 	// 𝟳 -> 7
	{0x1d7f4, 0x38}, 	// 𝟴 -> 8
	{0x1d7f5, 0x39}, 	// 𝟵 -> 9
	{0x1d7f6, 0x4f}, 	// 𝟶 -> O
	{0x1d7f7, 0x6c}, 	// 𝟷 -> l
	{0x1d7f8, 0x32}, 	// 𝟸 -> 2
	{0x1d7f9, 0x33}, 	// 𝟹 -> 3
	{0x1d7fa, 0x34}, 	// 𝟺 -> 4
	{0x1d7fb, 0x35}, 	// 𝟻 -> 5
	{0x1d7fc, 0x36}, 	// 𝟼 -> 6
	{0x1d7fd, 0x37}, 	// 𝟽 -> 7
	{0x1d7fe, 0x38}, 	// 𝟾 -> 8
	{0x1d7ff, 0x39}, 	// 𝟿 -> 9
	{0x1e8c7, 0x6c}, 	// 𞣇 -> l
	{0x1ee00, 0x6c}, 	// 𞸀 -> l
	{0x1ee24, 0x6f}, 	// 𞸤 -> o
	{0x1ee64, 0x6f}, 	// 𞹤 -> o
	{0x1ee80, 0x6c}, 	// 𞺀 -> l
	{0x1ee84, 0x6f}, 	// 𞺄 -> o
	{0x1f100, 0x4f}, 	// 🄀 -> O
	{0x1f101, 0x4f}, 	// 🄁 -> O
	{0x1f102, 0x6c}, 	// 🄂 -> l
	{0x1f110, 0x41}, 	// 🄐 -> A
	{0x1f111, 0x42}, 	// 🄑 -> B
	{0x1f112, 0x43}, 	// 🄒 -> C
	{0x1f113, 0x44}, 	// 🄓 -> D
	{0x1f114, 0x45}, 	// 🄔 -> E
	{0x1f115, 0x46}, 	// 🄕 -> F
	{0x1f116, 0x47}, 	// 🄖 -> G
	{0x1f117, 0x48}, 	// 🄗 -> H
	{0x1f118, 0x6c}, 	// 🄘 -> l
	{0x1f119, 0x4a}, 	// 🄙 -> J
	{0x1f11a, 0x4b}, 	// 🄚 -> K
	{0x1f11b, 0x4c}, 	// 🄛 -> L
	{0x1f11c, 0x4d}, 	// 🄜 -> M
	{0x1f11d, 0x4e}, 	// 🄝 -> N
	{0x1f11e, 0x4f}, 	// 🄞 -> O
	{0x1f11f, 0x50}, 	// 🄟 -> P
	{0x1f120, 0x51}, 	// 🄠 -> Q
	{0x1f121, 0x52}, 	// 🄡 -> R
	{0x1f122, 0x53}, 	// 🄢 -> S
	{0x1f123, 0x54}, 	// 🄣 -> T
	{0x1f124, 0x55}, 	// 🄤 -> U
	{0x1f125, 0x56}, 	// 🄥 -> V
	{0x1f126, 0x57}, 	// 🄦 -> W
	{0x1f127, 0x58}, 	// 🄧 -> X
	{0x1f128, 0x59}, 	// 🄨 -> Y
	{0x1f129, 0x5a}, 	// 🄩 -> Z
	{0x1f12a, 0x53}, 	// 🄪 -> S
	{0x1f16e, 0x43}, 	// 🅮 -> C
	{0x1f674, 0x26}, 	// 🙴 -> &
	{0x1f675, 0x26}, 	// 🙵 -> &
	{0x1f676, 0x22}, 	// 🙶 -> "
	{0x1f677, 0x22}, 	// 🙷 -> "
	{0x1f700, 0x51}, 	// 🜀 -> Q
	{0x1f707, 0x41}, 	// 🜇 -> A
	{0x1f708, 0x56}, 	// 🜈 -> V
	{0x1f714, 0x4f}, 	// 🜔 -> O
	{0x1f74c, 0x43}, 	// 🝌 -> C
	{0x1f75c, 0x73}, 	// 🝜 -> s
	{0x1f768, 0x54}, 	// 🝨 -> T
	{0x1f76b, 0x4d}, 	// 🝫 -> M
	{0x1f76c, 0x56}, 	// 🝬 -> V
	{0x1fbf0, 0x4f}, 	// 🯰 -> O
	{0x1fbf1, 0x6c}, 	// 🯱 -> l
};

#define IS_IN_RANGE(c, f, l)    (((c) >= (f)) && ((c) <= (l)))

#define SCRIPT_UNDEFINED -1

/* Forward declarations */
char *_utf8_convert_confusables(const char *i, char *obuf, int olen);
int utf8_text_analysis(Client *client, const char *text, TextAnalysis *e);
const char *_utf8_get_block_name(int i);
int _utf8_get_block_number(const char *name);

MOD_TEST()
{
	MARK_AS_OFFICIAL_MODULE(modinfo);
	EfunctionAddString(modinfo->handle, EFUNC_UTF8_CONVERT_CONFUSABLES, _utf8_convert_confusables);
	EfunctionAddConstString(modinfo->handle, EFUNC_UTF8_GET_BLOCK_NAME, _utf8_get_block_name);
	EfunctionAdd(modinfo->handle, EFUNC_UTF8_GET_BLOCK_NUMBER, _utf8_get_block_number);
	return MOD_SUCCESS;
}

MOD_INIT()
{
	MARK_AS_OFFICIAL_MODULE(modinfo);
	HookAdd(modinfo->handle, HOOKTYPE_ANALYZE_TEXT, -1, utf8_text_analysis);

	return MOD_SUCCESS;
}

MOD_LOAD()
{
	return MOD_SUCCESS;
}

MOD_UNLOAD()
{
	return MOD_SUCCESS;
}

/** Convert UTF8 sequence to UTF32.
 * This function should be safe to run on untrusted user-input.
 * @param t	The UTF8 byte sequence, of which 1 byte can be used, but also 2, 3 or 4.
 * @param bytes	In this we will store the number of bytes that was used (1-4).
 * @note A) In case of invalid UTF8 sequence, 0 is returned and *bytes is set to 1.
 *          Only if the UTF8 sequence is valid, *bytes can become 2/3/4.
 * @note B) This function does not check if the unicode code point actually exists
 *          (eg is undefined/emtpy or private).
 * @returns The UTF32 value, or 0 if invalid.
 */
uint32_t utf8_to_utf32(const char *t, int *bytes)
{
	char c1, c2;
	const char *ptr = t;
	uint32_t uc = 0;
	int i;
	char seqlen = 0;

	/* Set a sane default, as 'bytes' is usually used
	 * for advancing/skipping to next character, so
	 * setting it to 1 makes sense here.
	 */
	*bytes = 1;

	c1 = ptr[0];
	if( (c1 & 0x80) == 0 )
	{
		uc = (u_long) (c1 & 0x7F);
		seqlen = 1;
	}
	else if( (c1 & 0xE0) == 0xC0 )
	{
		uc = (u_long) (c1 & 0x1F);
		seqlen = 2;
	}
	else if( (c1 & 0xF0) == 0xE0 )
	{
		uc = (u_long) (c1 & 0x0F);
		seqlen = 3;
	}
	else if( (c1 & 0xF8) == 0xF0 )
	{
		uc = (u_long) (c1 & 0x07);
		seqlen = 4;
	} else
		return -1; /* should be impossible */

	for (i = 1; i < seqlen; ++i)
	{
		c1 = ptr[i];

		if( (c1 & 0xC0) != 0x80 )
			return 0; /* Invalid UTF8 */
	}

	switch( seqlen )
	{
		case 2:
		{
			c1 = ptr[0];

			if( !IS_IN_RANGE(c1, 0xC2, 0xDF) )
				return 0; /* Invalid UTF8 */

			break;
		}

		case 3:
		{
			c1 = ptr[0];
			c2 = ptr[1];

			switch (c1)
			{
				case 0xE0:
					if (!IS_IN_RANGE(c2, 0xA0, 0xBF))
						return 0; /* Invalid UTF8 */
					break;

				case 0xED:
					if (!IS_IN_RANGE(c2, 0x80, 0x9F))
						return 0; /* Invalid UTF8 */
					break;

				default:
					if (!IS_IN_RANGE(c1, 0xE1, 0xEC) && !IS_IN_RANGE(c1, 0xEE, 0xEF))
						return 0; /* Invalid UTF8 */
					break;
			}

			break;
		}

		case 4:
		{
			c1 = ptr[0];
			c2 = ptr[1];

			switch (c1)
			{
				case 0xF0:
					if (!IS_IN_RANGE(c2, 0x90, 0xBF))
						return 0; /* Invalid UTF8 */
					break;

				case 0xF4:
					if (!IS_IN_RANGE(c2, 0x80, 0x8F))
						return 0; /* Invalid UTF8 */
					break;

				default:
					if (!IS_IN_RANGE(c1, 0xF1, 0xF3))
						return 0; /* Invalid UTF8 */
					break;
			}

			break;
		}
	}

	for (i = 1; i < seqlen; ++i)
		uc = ((uc << 6) | (uint32_t)(ptr[i] & 0x3F));

	*bytes = seqlen;

	return uc;
}

int utf32_to_utf8(char *buf, uint32_t code)
{
	if (code <= 0x7F) {
		buf[0] = code;
		return 1;
	}
	if (code <= 0x7FF) {
		buf[0] = 0xC0 | (code >> 6);		/* 110xxxxx */
		buf[1] = 0x80 | (code & 0x3F);		/* 10xxxxxx */
		return 2;
	}
	if (code <= 0xFFFF) {
		buf[0] = 0xE0 | (code >> 12);		/* 1110xxxx */
		buf[1] = 0x80 | ((code >> 6) & 0x3F);   /* 10xxxxxx */
		buf[2] = 0x80 | (code & 0x3F);		/* 10xxxxxx */
		return 3;
	}
	if (code <= 0x10FFFF) {
		buf[0] = 0xF0 | (code >> 18);		/* 11110xxx */
		buf[1] = 0x80 | ((code >> 12) & 0x3F);  /* 10xxxxxx */
		buf[2] = 0x80 | ((code >> 6) & 0x3F);   /* 10xxxxxx */
		buf[3] = 0x80 | (code & 0x3F);		/* 10xxxxxx */
		return 4;
	}
	return 0;
}


/** Detect which script the current character is,
 * such as latin script or cyrillic script.
 * @retval See SCRIPT_*
 */
int detect_script(uint32_t utfchar)
{
	int i;

	/* Special handling for ASCII:
	 * - Consider only a-z/A-Z as latin1
	 * - All the rest, like spaces or numbers and stuff is
	 *   seen as SCRIPT_UNDEFINED, since it can and will
	 *   likely be used a lot with other unicode blocks too.
	 */
	if (utfchar <= 127)
	{
		if (((utfchar >= 'a') && (utfchar <= 'z')) ||
		    ((utfchar >= 'A') && (utfchar <= 'Z')))
		{
			return 0;
		}
		return SCRIPT_UNDEFINED;
	}

	for (i=0; i < elementsof(unicode_blocks); i++)
	{
		if ((utfchar >= unicode_blocks[i].start) &&
		    (utfchar <= unicode_blocks[i].end))
		{
			return i;
		}
	}

	return SCRIPT_UNDEFINED;
}

/** Do UTF8 text analysis.
 * This builds on the work of antimixedutf8 from earlier.
 * @param text		The input text.
 * @param points	The number of points given (number of transitions
 *			between unicode blocks, possibly score adjusted,
 *			antimixedutf8 style).
 * @param num_blocks	Number of different blocks in use (this is or can
 *			be different than the number of transitions).
 * @param blockmap	A char array (not a string!) with counts of how
 *			many times a character was encounted in that
 *			particular unicode block. Eg blockmap[0] would be
 *			"Basic Latin". Since it is a char, counts are
 *			capped at 255.
 * @param blockmaplen	Length of the blockmap array (since blockmap is
 *			not a string it is full of zeroes and not zero
 *			terminated).
 * @returns If this utf8functions module is loaded (so the code you are
 *          looking at) we return 1. The default handler, used when this
 *          module is not loaded, will always return 0.
 */
int utf8_text_analysis(Client *client, const char *text, TextAnalysis *e)
{
	const char *p;
	int last_script = 0;
	int current_script;
	int last_character_was_word_separator = 0;
	int utf8len;
	uint32_t utfchar;
	int ublks = 0;

	for (p = text; *p; p += utf8len)
	{
		utfchar = utf8_to_utf32(p, &utf8len);
		current_script = detect_script(utfchar);

		e->num_bytes += utf8len;
		e->num_unicode_characters++;

		if (current_script != SCRIPT_UNDEFINED)
		{
			/* Two things to be careful about:
			 * 1) If current_script is SCRIPT_UNDEFINED it is -1 so we would OOB,
			 *    but this case is covered two lines up.
			 * 2) This is of type 'char', so stop at the maximum value, 255.
			 */
			if (!e->unicode_blockmap[current_script])
				e->unicode_blocks++;
			if (e->unicode_blockmap[current_script] < 255)
				e->unicode_blockmap[current_script]++;

			/* For antimixedutf8 scores (script changes) we do some
			 * remapping. This so several blocks are treated as the same.
			 * At the moment this is only:
			 * * Latin-1 Supplement, Latin Extended-A, Latin Extended-B
			 *   => Basic Latin
			 */
			if (current_script <= 3)
				current_script = 0;

			if ((current_script != last_script) && (last_script != SCRIPT_UNDEFINED))
			{
				/* Script change: add X point(s) */
				int add_points = unicode_blocks[current_script].score;
				e->antimixedutf8_points += add_points;

#if 0
				/* Give an extra point if the script change happened
				 * within the same word, as that would be rather unusual
				 * in normal cases. (Unless .score is 0 points)
				 */
				if (add_points && !last_character_was_word_separator)
					e->antimixedutf8_points++;
#endif
			}
			last_script = current_script;
		}

		if (strchr("., ", *p))
			last_character_was_word_separator = 1;
		else
			last_character_was_word_separator = 0;
	}

	/* For performance reasons it would be logical to implement the
	 * conversion of confusables in the loop from above.
	 * For consistency, or as long as we provide utf8_convert_confusables(),
	 * however it makes more sense to keep the function separate.
	 */
	*e->deconfused = '\0'; // Initialize, in case next fails.
	_utf8_convert_confusables(text, e->deconfused, sizeof(e->deconfused));

	return 1;
}

/** Returns length of an (UTF8) character. May return <1 for error conditions.
 * Made by i <info@servx.org>
 */
static int utf8_charlen(const char *str)
{
	struct { char mask; char val; } t[4] =
	{ { 0x80, 0x00 }, { 0xE0, 0xC0 }, { 0xF0, 0xE0 }, { 0xF8, 0xF0 } };
	unsigned k, j;

	for (k = 0; k < 4; k++)
	{
		if ((*str & t[k].mask) == t[k].val)
		{
			for (j = 0; j < k; j++)
			{
				if ((*(++str) & 0xC0) != 0x80)
					return -1;
			}
			return k + 1;
		}
	}
	return 1;
}

uint32_t utf8_lookup_confusable(uint32_t c)
{
	int start = 0;
	int stop = ARRAY_SIZEOF(confusables_table)-1;
	int mid;

	while (start <= stop) {
		mid = (start+stop)/2;
		if (c < confusables_table[mid].from)
			stop = mid-1;
		else if (c == confusables_table[mid].from)
			return confusables_table[mid].to;
		else
			start = mid+1;
	}
	return 0;
}

char *_utf8_convert_confusables(const char *i, char *obuf, int olen)
{
	int utf8charlen, x;
	char *o = obuf;
	uint32_t utfchar, conv;

	if ((i == NULL) || (olen <= 0))
		return NULL;
	olen--; /* reserve room for \0 right now */

	for (; *i && olen; i += utf8charlen)
	{
		utfchar = utf8_to_utf32(i, &utf8charlen);
		if (utfchar == 0)
		{
			/* Invalid UTF8. Dilemma: use as-is or skip it? */
			*o++ = *i;
			olen--;
			continue;
		}
		conv = utf8_lookup_confusable(utfchar);
		if (conv == 0)
		{
			/* use as-is */
			if (olen < utf8charlen)
				break; /* cut off */
			for (x=0; x < utf8charlen; x++)
				*o++ = i[x];
			olen -= utf8charlen;
		} else {
			/* convert */
			int replacelen;
			if (olen < 4) // yeah too lazy to lookup replacelen in advance :D
				break; /* cut off */
			replacelen = utf32_to_utf8(o, conv);
			o += replacelen;
			olen -= replacelen;
		}
	}

	*o = '\0';
	return obuf;
}

/** Get UTF8 name for a block number (eg 0 returns "Basic Latin") */
const char *_utf8_get_block_name(int i)
{
	if ((i < 0) || (i > ARRAY_SIZEOF(unicode_blocks)-1))
		return NULL;
	return unicode_blocks[i].name;
}

/** Get UTF8 block number by name (eg "Basic Latin" returns 0) */
int _utf8_get_block_number(const char *name)
{
	int i;
	for (i = 0; i < ARRAY_SIZEOF(unicode_blocks); i++)
		if (!strcasecmp(unicode_blocks[i].name, name))
			return i;
	return -1;
}
