"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsKmsMrkAwareSymmetricKeyringClass = void 0;
const material_management_1 = require("@aws-crypto/material-management");
const helpers_1 = require("./helpers");
const arn_parsing_1 = require("./arn_parsing");
function AwsKmsMrkAwareSymmetricKeyringClass(BaseKeyring) {
    class AwsKmsMrkAwareSymmetricKeyring
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.5
    //# MUST implement the AWS Encryption SDK Keyring interface (../keyring-
    //# interface.md#interface)
     extends BaseKeyring {
        keyId;
        client;
        grantTokens;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.6
        //# On initialization the caller MUST provide:
        constructor({ client, keyId, grantTokens, }) {
            super();
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.6
            //# The AWS KMS key identifier MUST NOT be null or empty.
            (0, material_management_1.needs)(keyId && typeof keyId === 'string', 'An AWS KMS key identifier is required.');
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.6
            //# The AWS KMS
            //# key identifier MUST be a valid identifier (aws-kms-key-arn.md#a-
            //# valid-aws-kms-identifier).
            (0, material_management_1.needs)((0, arn_parsing_1.validAwsKmsIdentifier)(keyId), `Key id ${keyId} is not a valid identifier.`);
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.6
            //# The AWS KMS
            //# SDK client MUST NOT be null.
            (0, material_management_1.needs)(!!client, 'An AWS SDK client is required');
            (0, material_management_1.readOnlyProperty)(this, 'client', client);
            (0, material_management_1.readOnlyProperty)(this, 'keyId', keyId);
            (0, material_management_1.readOnlyProperty)(this, 'grantTokens', grantTokens);
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
        //# OnEncrypt MUST take encryption materials (structures.md#encryption-
        //# materials) as input.
        async _onEncrypt(material) {
            const { client, keyId, grantTokens } = this;
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
            //# If the input encryption materials (structures.md#encryption-
            //# materials) do not contain a plaintext data key OnEncrypt MUST attempt
            //# to generate a new plaintext data key by calling AWS KMS
            //# GenerateDataKey (https://docs.aws.amazon.com/kms/latest/APIReference/
            //# API_GenerateDataKey.html).
            if (!material.hasUnencryptedDataKey) {
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# The keyring MUST call
                //# AWS KMS GenerateDataKeys with a request constructed as follows:
                //
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If the call to AWS KMS GenerateDataKey
                //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_GenerateDataKey.html) does not succeed, OnEncrypt MUST NOT modify
                //# the encryption materials (structures.md#encryption-materials) and
                //# MUST fail.
                const dataKey = await (0, helpers_1.generateDataKey)(
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If the keyring calls AWS KMS GenerateDataKeys, it MUST use the
                //# configured AWS KMS client to make the call.
                client, material.suite.keyLengthBytes, keyId, material.encryptionContext, grantTokens);
                /* This should be impossible given that generateDataKey only returns false if the client supplier does. */
                (0, material_management_1.needs)(dataKey, 'Generator KMS key did not generate a data key');
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# The Generate Data Key response's "KeyId" MUST be A valid AWS
                //# KMS key ARN (aws-kms-key-arn.md#identifying-an-aws-kms-multi-region-
                //# key).
                (0, material_management_1.needs)((0, arn_parsing_1.parseAwsKmsKeyArn)(dataKey.KeyId), 'Malformed arn.');
                const flags = material_management_1.KeyringTraceFlag.WRAPPING_KEY_GENERATED_DATA_KEY |
                    material_management_1.KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX |
                    material_management_1.KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY;
                const trace = {
                    keyNamespace: helpers_1.KMS_PROVIDER_ID,
                    keyName: dataKey.KeyId,
                    flags,
                };
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If verified,
                //# OnEncrypt MUST do the following with the response from AWS KMS
                //# GenerateDataKey (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_GenerateDataKey.html):
                material
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                    //# If the Generate Data Key call succeeds, OnEncrypt MUST verify that
                    //# the response "Plaintext" length matches the specification of the
                    //# algorithm suite (algorithm-suites.md)'s Key Derivation Input Length
                    //# field.
                    //
                    // setUnencryptedDataKey will throw if the plaintext does not match the algorithm suite requirements.
                    .setUnencryptedDataKey(dataKey.Plaintext, trace)
                    .addEncryptedDataKey((0, helpers_1.kmsResponseToEncryptedDataKey)(dataKey), material_management_1.KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
                    material_management_1.KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX);
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# *  OnEncrypt MUST output the modified encryption materials
                //# (structures.md#encryption-materials)
                return material;
            }
            else {
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# Given a plaintext data key in the encryption materials
                //# (structures.md#encryption-materials), OnEncrypt MUST attempt to
                //# encrypt the plaintext data key using the configured AWS KMS key
                //# identifier.
                const unencryptedDataKey = (0, material_management_1.unwrapDataKey)(material.getUnencryptedDataKey());
                const flags = material_management_1.KeyringTraceFlag.WRAPPING_KEY_ENCRYPTED_DATA_KEY |
                    material_management_1.KeyringTraceFlag.WRAPPING_KEY_SIGNED_ENC_CTX;
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If the call to AWS KMS Encrypt
                //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_Encrypt.html) does not succeed, OnEncrypt MUST fail.
                //
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# The keyring
                //# MUST AWS KMS Encrypt call with a request constructed as follows:
                const kmsEDK = await (0, helpers_1.encrypt)(
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# The keyring MUST call AWS KMS Encrypt
                //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_Encrypt.html) using the configured AWS KMS client.
                client, unencryptedDataKey, keyId, material.encryptionContext, grantTokens);
                /* This should be impossible given that encrypt only returns false if the client supplier does. */
                (0, material_management_1.needs)(kmsEDK, 'AwsKmsMrkAwareSymmetricKeyring failed to encrypt data key');
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If the Encrypt call succeeds The response's "KeyId" MUST be A valid
                //# AWS KMS key ARN (aws-kms-key-arn.md#identifying-an-aws-kms-multi-
                //# region-key).
                (0, material_management_1.needs)((0, arn_parsing_1.parseAwsKmsKeyArn)(kmsEDK.KeyId), 'Malformed arn.');
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If verified, OnEncrypt MUST do the following with the response from
                //# AWS KMS Encrypt (https://docs.aws.amazon.com/kms/latest/APIReference/
                //# API_Encrypt.html):
                material.addEncryptedDataKey((0, helpers_1.kmsResponseToEncryptedDataKey)(kmsEDK), flags);
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.7
                //# If all Encrypt calls succeed, OnEncrypt MUST output the modified
                //# encryption materials (structures.md#encryption-materials).
                return material;
            }
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
        //# OnDecrypt MUST take decryption materials (structures.md#decryption-
        //# materials) and a list of encrypted data keys
        //# (structures.md#encrypted-data-key) as input.
        async _onDecrypt(material, encryptedDataKeys) {
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
            //# If the decryption materials (structures.md#decryption-materials)
            //# already contained a valid plaintext data key OnDecrypt MUST
            //# immediately return the unmodified decryption materials
            //# (structures.md#decryption-materials).
            if (material.hasValidKey())
                return material;
            const { client, keyId, grantTokens } = this;
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
            //# The set of encrypted data keys MUST first be filtered to match this
            //# keyring's configuration.
            const decryptableEDKs = encryptedDataKeys.filter(filterEDKs(keyId));
            const cmkErrors = [];
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
            //# For each encrypted data key in the filtered set, one at a time, the
            //# OnDecrypt MUST attempt to decrypt the data key.
            for (const edk of decryptableEDKs) {
                const { providerId, encryptedDataKey } = edk;
                try {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# When calling AWS KMS Decrypt
                    //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html), the keyring MUST call with a request constructed
                    //# as follows:
                    const dataKey = await (0, helpers_1.decrypt)(
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# To attempt to decrypt a particular encrypted data key
                    //# (structures.md#encrypted-data-key), OnDecrypt MUST call AWS KMS
                    //# Decrypt (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html) with the configured AWS KMS client.
                    client, 
                    // For MRKs the key identifier MUST be the configured key identifer.
                    { providerId, encryptedDataKey, providerInfo: this.keyId }, material.encryptionContext, grantTokens);
                    /* This should be impossible given that decrypt only returns false if the client supplier does
                     * or if the providerId is not "aws-kms", which we have already filtered out
                     */
                    (0, material_management_1.needs)(dataKey, 'decrypt did not return a data key.');
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# *  The "KeyId" field in the response MUST equal the configured AWS
                    //# KMS key identifier.
                    (0, material_management_1.needs)(dataKey.KeyId === this.keyId, 'KMS Decryption key does not match the requested key id.');
                    const flags = material_management_1.KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY |
                        material_management_1.KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX;
                    const trace = {
                        keyNamespace: helpers_1.KMS_PROVIDER_ID,
                        keyName: dataKey.KeyId,
                        flags,
                    };
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# If the response does satisfies these requirements then OnDecrypt MUST
                    //# do the following with the response:
                    //
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# *  The length of the response's "Plaintext" MUST equal the key
                    //# derivation input length (algorithm-suites.md#key-derivation-input-
                    //# length) specified by the algorithm suite (algorithm-suites.md)
                    //# included in the input decryption materials
                    //# (structures.md#decryption-materials).
                    //
                    // setUnencryptedDataKey will throw if the plaintext does not match the algorithm suite requirements.
                    material.setUnencryptedDataKey(dataKey.Plaintext, trace);
                    return material;
                }
                catch (e) {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# If this attempt
                    //# results in an error, then these errors MUST be collected.
                    //
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
                    //# If the response does not satisfies these requirements then an error
                    //# MUST be collected and the next encrypted data key in the filtered set
                    //# MUST be attempted.
                    cmkErrors.push({ errPlus: e });
                }
            }
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
            //# If OnDecrypt fails to successfully decrypt any encrypted data key
            //# (structures.md#encrypted-data-key), then it MUST yield an error that
            //# includes all the collected errors.
            (0, material_management_1.needs)(material.hasValidKey(), [
                `Unable to decrypt data key${!decryptableEDKs.length ? ': No EDKs supplied' : ''}.`,
                ...cmkErrors.map((e, i) => `Error #${i + 1}  \n${e.errPlus.stack}`),
            ].join('\n'));
            return material;
        }
    }
    (0, material_management_1.immutableClass)(AwsKmsMrkAwareSymmetricKeyring);
    return AwsKmsMrkAwareSymmetricKeyring;
}
exports.AwsKmsMrkAwareSymmetricKeyringClass = AwsKmsMrkAwareSymmetricKeyringClass;
function filterEDKs(keyringKeyId) {
    return function filter({ providerId, providerInfo }) {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
        //# *  Its provider ID MUST exactly match the value "aws-kms".
        if (providerId !== helpers_1.KMS_PROVIDER_ID)
            return false;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
        //# *  The provider info MUST be a valid AWS KMS ARN (aws-kms-key-
        //# arn.md#a-valid-aws-kms-arn) with a resource type of "key" or
        //# OnDecrypt MUST fail.
        const arnInfo = (0, arn_parsing_1.parseAwsKmsKeyArn)(providerInfo);
        (0, material_management_1.needs)(arnInfo && arnInfo.ResourceType === 'key', 'Unexpected EDK ProviderInfo for AWS KMS EDK');
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-keyring.txt#2.8
        //# *  The the function AWS KMS MRK Match for Decrypt (aws-kms-mrk-match-
        //# for-decrypt.md#implementation) called with the configured AWS KMS
        //# key identifier and the provider info MUST return "true".
        return (0, arn_parsing_1.mrkAwareAwsKmsKeyIdCompare)(keyringKeyId, providerInfo);
    };
}
//# sourceMappingURL=data:application/json;base64,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