from typing import Dict

from .. import Provider as LoremProvider


class Provider(LoremProvider):
    """Implement lorem provider for ``en_US`` locale.

    Word list is based on the source(s) below, and some words have been removed
    to make the word list appropriate for public testing.

    Sources:

    - https://www.educall.com.tr/blog/post/500-most-common-english-verbs
    - http://www.ef.edu/english-resources/english-vocabulary/top-1000-words/
    - https://www.talkenglish.com/vocabulary/top-1500-nouns.aspx
    - https://www.talkenglish.com/vocabulary/top-250-adverbs.aspx
    - https://www.talkenglish.com/vocabulary/top-500-adjectives.aspx
    """

    word_list = (
        "a",
        "ability",
        "able",
        "about",
        "above",
        "accept",
        "according",
        "account",
        "across",
        "act",
        "action",
        "activity",
        "actually",
        "add",
        "address",
        "administration",
        "admit",
        "adult",
        "affect",
        "after",
        "again",
        "against",
        "age",
        "agency",
        "agent",
        "ago",
        "agree",
        "agreement",
        "ahead",
        "air",
        "all",
        "allow",
        "almost",
        "alone",
        "along",
        "already",
        "also",
        "although",
        "always",
        "American",
        "among",
        "amount",
        "analysis",
        "and",
        "animal",
        "another",
        "answer",
        "any",
        "anyone",
        "anything",
        "appear",
        "apply",
        "approach",
        "area",
        "argue",
        "arm",
        "around",
        "arrive",
        "art",
        "article",
        "artist",
        "as",
        "ask",
        "assume",
        "at",
        "attack",
        "attention",
        "attorney",
        "audience",
        "author",
        "authority",
        "available",
        "avoid",
        "away",
        "baby",
        "back",
        "bad",
        "bag",
        "ball",
        "bank",
        "bar",
        "base",
        "be",
        "beat",
        "beautiful",
        "because",
        "become",
        "bed",
        "before",
        "begin",
        "behavior",
        "behind",
        "believe",
        "benefit",
        "best",
        "better",
        "between",
        "beyond",
        "big",
        "bill",
        "billion",
        "bit",
        "black",
        "blood",
        "blue",
        "board",
        "body",
        "book",
        "born",
        "both",
        "box",
        "boy",
        "break",
        "bring",
        "brother",
        "budget",
        "build",
        "building",
        "business",
        "but",
        "buy",
        "by",
        "call",
        "camera",
        "campaign",
        "can",
        "candidate",
        "capital",
        "car",
        "card",
        "care",
        "career",
        "carry",
        "case",
        "catch",
        "cause",
        "cell",
        "center",
        "central",
        "century",
        "certain",
        "certainly",
        "chair",
        "challenge",
        "chance",
        "change",
        "character",
        "charge",
        "check",
        "child",
        "choice",
        "choose",
        "church",
        "citizen",
        "city",
        "civil",
        "claim",
        "class",
        "clear",
        "clearly",
        "close",
        "coach",
        "cold",
        "collection",
        "college",
        "color",
        "commercial",
        "common",
        "community",
        "company",
        "compare",
        "computer",
        "concern",
        "condition",
        "conference",
        "Congress",
        "consider",
        "consumer",
        "contain",
        "continue",
        "control",
        "cost",
        "could",
        "country",
        "couple",
        "course",
        "court",
        "cover",
        "create",
        "crime",
        "cultural",
        "culture",
        "cup",
        "current",
        "customer",
        "cut",
        "dark",
        "data",
        "daughter",
        "day",
        "deal",
        "debate",
        "decade",
        "decide",
        "decision",
        "deep",
        "defense",
        "degree",
        "Democrat",
        "democratic",
        "describe",
        "design",
        "despite",
        "detail",
        "determine",
        "develop",
        "development",
        "difference",
        "different",
        "difficult",
        "dinner",
        "direction",
        "director",
        "discover",
        "discuss",
        "discussion",
        "do",
        "doctor",
        "dog",
        "door",
        "down",
        "draw",
        "dream",
        "drive",
        "drop",
        "drug",
        "during",
        "each",
        "early",
        "east",
        "easy",
        "eat",
        "economic",
        "economy",
        "edge",
        "education",
        "effect",
        "effort",
        "eight",
        "either",
        "election",
        "else",
        "employee",
        "end",
        "energy",
        "enjoy",
        "enough",
        "enter",
        "entire",
        "environment",
        "environmental",
        "especially",
        "establish",
        "even",
        "evening",
        "event",
        "ever",
        "every",
        "everybody",
        "everyone",
        "everything",
        "evidence",
        "exactly",
        "example",
        "executive",
        "exist",
        "expect",
        "experience",
        "expert",
        "explain",
        "eye",
        "face",
        "fact",
        "factor",
        "fall",
        "family",
        "far",
        "fast",
        "father",
        "fear",
        "federal",
        "feel",
        "feeling",
        "few",
        "field",
        "fight",
        "figure",
        "fill",
        "film",
        "final",
        "finally",
        "financial",
        "find",
        "fine",
        "finish",
        "fire",
        "firm",
        "first",
        "fish",
        "five",
        "floor",
        "fly",
        "focus",
        "follow",
        "food",
        "foot",
        "for",
        "force",
        "foreign",
        "forget",
        "form",
        "former",
        "forward",
        "four",
        "free",
        "friend",
        "from",
        "front",
        "full",
        "fund",
        "future",
        "game",
        "garden",
        "gas",
        "general",
        "generation",
        "get",
        "girl",
        "give",
        "glass",
        "go",
        "goal",
        "good",
        "government",
        "great",
        "green",
        "ground",
        "group",
        "grow",
        "growth",
        "guess",
        "gun",
        "guy",
        "hair",
        "half",
        "hand",
        "happen",
        "happy",
        "hard",
        "have",
        "he",
        "head",
        "health",
        "hear",
        "heart",
        "heavy",
        "help",
        "her",
        "here",
        "herself",
        "high",
        "him",
        "himself",
        "his",
        "history",
        "hit",
        "hold",
        "home",
        "hope",
        "hospital",
        "hot",
        "hotel",
        "hour",
        "house",
        "how",
        "however",
        "huge",
        "human",
        "hundred",
        "husband",
        "I",
        "idea",
        "identify",
        "if",
        "image",
        "imagine",
        "impact",
        "important",
        "improve",
        "in",
        "include",
        "including",
        "increase",
        "indeed",
        "indicate",
        "individual",
        "industry",
        "information",
        "inside",
        "instead",
        "institution",
        "interest",
        "interesting",
        "international",
        "interview",
        "into",
        "investment",
        "involve",
        "issue",
        "it",
        "item",
        "its",
        "itself",
        "job",
        "join",
        "just",
        "keep",
        "key",
        "kid",
        "kind",
        "kitchen",
        "know",
        "knowledge",
        "land",
        "language",
        "large",
        "last",
        "late",
        "later",
        "laugh",
        "law",
        "lawyer",
        "lay",
        "lead",
        "leader",
        "learn",
        "least",
        "leave",
        "left",
        "leg",
        "less",
        "let",
        "letter",
        "level",
        "life",
        "light",
        "like",
        "likely",
        "line",
        "list",
        "listen",
        "little",
        "live",
        "local",
        "long",
        "look",
        "lose",
        "loss",
        "lot",
        "low",
        "machine",
        "magazine",
        "main",
        "maintain",
        "major",
        "majority",
        "make",
        "man",
        "manage",
        "management",
        "manager",
        "many",
        "market",
        "marriage",
        "material",
        "matter",
        "may",
        "maybe",
        "me",
        "mean",
        "measure",
        "media",
        "medical",
        "meet",
        "meeting",
        "member",
        "memory",
        "mention",
        "message",
        "method",
        "middle",
        "might",
        "military",
        "million",
        "mind",
        "minute",
        "miss",
        "mission",
        "model",
        "modern",
        "moment",
        "money",
        "month",
        "more",
        "morning",
        "most",
        "mother",
        "mouth",
        "move",
        "movement",
        "movie",
        "Mr",
        "Mrs",
        "much",
        "music",
        "must",
        "my",
        "myself",
        "name",
        "nation",
        "national",
        "natural",
        "nature",
        "near",
        "nearly",
        "necessary",
        "need",
        "network",
        "never",
        "new",
        "news",
        "newspaper",
        "next",
        "nice",
        "night",
        "no",
        "none",
        "nor",
        "north",
        "not",
        "note",
        "nothing",
        "notice",
        "now",
        "number",
        "occur",
        "of",
        "off",
        "offer",
        "office",
        "officer",
        "official",
        "often",
        "oil",
        "ok",
        "old",
        "on",
        "once",
        "one",
        "only",
        "onto",
        "open",
        "operation",
        "opportunity",
        "option",
        "or",
        "order",
        "organization",
        "other",
        "others",
        "our",
        "out",
        "outside",
        "over",
        "own",
        "owner",
        "page",
        "painting",
        "paper",
        "parent",
        "part",
        "participant",
        "particular",
        "particularly",
        "partner",
        "party",
        "pass",
        "past",
        "pattern",
        "pay",
        "peace",
        "people",
        "per",
        "perform",
        "performance",
        "perhaps",
        "person",
        "personal",
        "phone",
        "physical",
        "pick",
        "picture",
        "piece",
        "place",
        "plan",
        "plant",
        "play",
        "player",
        "PM",
        "point",
        "police",
        "policy",
        "political",
        "politics",
        "poor",
        "popular",
        "population",
        "position",
        "positive",
        "possible",
        "power",
        "practice",
        "prepare",
        "present",
        "president",
        "pressure",
        "pretty",
        "prevent",
        "price",
        "probably",
        "process",
        "produce",
        "product",
        "production",
        "professional",
        "professor",
        "program",
        "project",
        "property",
        "protect",
        "prove",
        "provide",
        "public",
        "pull",
        "purpose",
        "push",
        "put",
        "quality",
        "question",
        "quickly",
        "quite",
        "race",
        "radio",
        "raise",
        "range",
        "rate",
        "rather",
        "reach",
        "read",
        "ready",
        "real",
        "reality",
        "realize",
        "really",
        "reason",
        "receive",
        "recent",
        "recently",
        "recognize",
        "record",
        "red",
        "reduce",
        "reflect",
        "region",
        "relate",
        "relationship",
        "religious",
        "remain",
        "remember",
        "report",
        "represent",
        "Republican",
        "require",
        "research",
        "resource",
        "respond",
        "response",
        "responsibility",
        "rest",
        "result",
        "return",
        "reveal",
        "rich",
        "right",
        "rise",
        "risk",
        "road",
        "rock",
        "role",
        "room",
        "rule",
        "run",
        "safe",
        "same",
        "save",
        "say",
        "scene",
        "school",
        "science",
        "scientist",
        "score",
        "sea",
        "season",
        "seat",
        "second",
        "section",
        "security",
        "see",
        "seek",
        "seem",
        "sell",
        "send",
        "senior",
        "sense",
        "series",
        "serious",
        "serve",
        "service",
        "set",
        "seven",
        "several",
        "shake",
        "share",
        "she",
        "short",
        "should",
        "shoulder",
        "show",
        "side",
        "sign",
        "significant",
        "similar",
        "simple",
        "simply",
        "since",
        "sing",
        "single",
        "sister",
        "sit",
        "site",
        "situation",
        "six",
        "size",
        "skill",
        "skin",
        "small",
        "smile",
        "so",
        "social",
        "society",
        "soldier",
        "some",
        "somebody",
        "someone",
        "something",
        "sometimes",
        "son",
        "song",
        "soon",
        "sort",
        "sound",
        "source",
        "south",
        "southern",
        "space",
        "speak",
        "special",
        "specific",
        "speech",
        "spend",
        "sport",
        "spring",
        "staff",
        "stage",
        "stand",
        "standard",
        "star",
        "start",
        "state",
        "statement",
        "station",
        "stay",
        "step",
        "still",
        "stock",
        "stop",
        "store",
        "story",
        "strategy",
        "street",
        "strong",
        "structure",
        "student",
        "study",
        "stuff",
        "style",
        "subject",
        "success",
        "successful",
        "such",
        "suddenly",
        "suffer",
        "suggest",
        "summer",
        "support",
        "sure",
        "surface",
        "system",
        "table",
        "take",
        "talk",
        "task",
        "tax",
        "teach",
        "teacher",
        "team",
        "technology",
        "television",
        "tell",
        "ten",
        "tend",
        "term",
        "test",
        "than",
        "thank",
        "that",
        "the",
        "their",
        "them",
        "themselves",
        "then",
        "theory",
        "there",
        "these",
        "they",
        "thing",
        "think",
        "third",
        "this",
        "those",
        "though",
        "thought",
        "thousand",
        "threat",
        "three",
        "through",
        "throughout",
        "throw",
        "thus",
        "time",
        "to",
        "today",
        "together",
        "tonight",
        "too",
        "top",
        "total",
        "tough",
        "toward",
        "town",
        "trade",
        "traditional",
        "training",
        "travel",
        "treat",
        "treatment",
        "tree",
        "trial",
        "trip",
        "trouble",
        "true",
        "truth",
        "try",
        "turn",
        "TV",
        "two",
        "type",
        "under",
        "understand",
        "unit",
        "until",
        "up",
        "upon",
        "us",
        "use",
        "usually",
        "value",
        "various",
        "very",
        "view",
        "visit",
        "voice",
        "vote",
        "wait",
        "walk",
        "wall",
        "want",
        "war",
        "watch",
        "water",
        "way",
        "we",
        "wear",
        "week",
        "weight",
        "well",
        "west",
        "western",
        "what",
        "whatever",
        "when",
        "where",
        "whether",
        "which",
        "while",
        "white",
        "who",
        "whole",
        "whom",
        "whose",
        "why",
        "wide",
        "wife",
        "will",
        "win",
        "wind",
        "window",
        "wish",
        "with",
        "within",
        "without",
        "woman",
        "wonder",
        "word",
        "work",
        "worker",
        "world",
        "worry",
        "would",
        "write",
        "writer",
        "wrong",
        "yard",
        "yeah",
        "year",
        "yes",
        "yet",
        "you",
        "young",
        "your",
        "yourself",
    )

    parts_of_speech: Dict[str, tuple] = {
        "verb": (
            "be",
            "have",
            "do",
            "say",
            "get",
            "make",
            "go",
            "see",
            "know",
            "take",
            "think",
            "come",
            "give",
            "look",
            "use",
            "find",
            "want",
            "tell",
            "put",
            "mean",
            "become",
            "leave",
            "work",
            "need",
            "feel",
            "seem",
            "ask",
            "show",
            "try",
            "call",
            "keep",
            "provide",
            "hold",
            "turn",
            "follow",
            "begin",
            "bring",
            "like",
            "going",
            "help",
            "start",
            "run",
            "write",
            "set",
            "move",
            "play",
            "pay",
            "hear",
            "include",
            "believe",
            "allow",
            "meet",
            "lead",
            "live",
            "stand",
            "happen",
            "carry",
            "talk",
            "appear",
            "produce",
            "sit",
            "offer",
            "consider",
            "expect",
            "let",
            "read",
            "require",
            "continue",
            "lose",
            "add",
            "change",
            "fall",
            "remain",
            "remember",
            "buy",
            "speak",
            "stop",
            "send",
            "receive",
            "decide",
            "win",
            "understand",
            "describe",
            "develop",
            "agree",
            "open",
            "reach",
            "build",
            "involve",
            "spend",
            "return",
            "draw",
            "die",
            "hope",
            "create",
            "walk",
            "sell",
            "wait",
            "cause",
            "pass",
            "lie",
            "accept",
            "watch",
            "raise",
            "base",
            "apply",
            "break",
            "explain",
            "learn",
            "increase",
            "cover",
            "grow",
            "claim",
            "report",
            "support",
            "cut",
            "form",
            "stay",
            "contain",
            "reduce",
            "establish",
            "join",
            "wish",
            "seek",
            "choose",
            "deal",
            "face",
            "fail",
            "serve",
            "end",
            "kill",
            "occur",
            "drive",
            "represent",
            "rise",
            "discuss",
            "love",
            "pick",
            "place",
            "argue",
            "prove",
            "wear",
            "catch",
            "enjoy",
            "eat",
            "introduce",
            "enter",
            "present",
            "arrive",
            "ensure",
            "point",
            "plan",
            "pull",
            "refer",
            "act",
            "relate",
            "affect",
            "close",
            "identify",
            "manage",
            "thank",
            "compare",
            "announce",
            "obtain",
            "note",
            "forget",
            "indicate",
            "wonder",
            "maintain",
            "publish",
            "suffer",
            "avoid",
            "express",
            "suppose",
            "finish",
            "determine",
            "design",
            "listen",
            "save",
            "tend",
            "treat",
            "control",
            "share",
            "remove",
            "throw",
            "visit",
            "exist",
            "force",
            "reflect",
            "admit",
            "assume",
            "smile",
            "prepare",
            "replace",
            "fill",
            "improve",
            "mention",
            "fight",
            "intend",
            "miss",
            "discover",
            "drop",
            "hit",
            "push",
            "prevent",
            "refuse",
            "regard",
            "lay",
            "reveal",
            "teach",
            "answer",
            "operate",
            "state",
            "depend",
            "enable",
            "record",
            "check",
            "complete",
            "cost",
            "sound",
            "laugh",
            "realise",
            "extend",
            "arise",
            "notice",
            "define",
            "examine",
            "fit",
            "study",
            "bear",
            "hang",
            "recognise",
            "shake",
            "sign",
            "attend",
            "fly",
            "gain",
            "result",
            "travel",
            "adopt",
            "confirm",
            "protect",
            "demand",
            "stare",
            "imagine",
            "attempt",
            "beat",
            "born",
            "associate",
            "care",
            "marry",
            "collect",
            "voice",
            "employ",
            "issue",
            "release",
            "emerge",
            "mind",
            "aim",
            "deny",
            "mark",
            "shoot",
            "appoint",
            "order",
            "supply",
            "drink",
            "observe",
            "reply",
            "ignore",
            "link",
            "propose",
            "ring",
            "settle",
            "strike",
            "press",
            "respond",
            "arrange",
            "survive",
            "concentrate",
            "lift",
            "approach",
            "cross",
            "test",
            "charge",
            "experience",
            "touch",
            "acquire",
            "commit",
            "demonstrate",
            "grant",
            "prefer",
            "repeat",
            "sleep",
            "threaten",
            "feed",
            "insist",
            "launch",
            "limit",
            "promote",
            "deliver",
            "measure",
            "own",
            "retain",
            "attract",
            "belong",
            "consist",
            "contribute",
            "hide",
            "promise",
            "reject",
            "cry",
            "impose",
            "invite",
            "sing",
            "vary",
            "warn",
            "address",
            "declare",
            "destroy",
            "worry",
            "divide",
            "head",
            "name",
            "stick",
            "nod",
            "recognize",
            "train",
            "attack",
            "clear",
            "combine",
            "handle",
            "influence",
            "realize",
            "recommend",
            "shout",
            "spread",
            "undertake",
            "account",
            "select",
            "climb",
            "contact",
            "recall",
            "secure",
            "step",
            "transfer",
            "welcome",
            "conclude",
            "disappear",
            "display",
            "dress",
            "illustrate",
            "imply",
            "organise",
            "direct",
            "escape",
            "generate",
            "remind",
            "advise",
            "afford",
            "earn",
            "hand",
            "inform",
            "rely",
            "succeed",
            "approve",
            "burn",
            "fear",
            "vote",
            "conduct",
            "cope",
            "derive",
            "elect",
            "gather",
            "jump",
            "last",
            "match",
            "matter",
            "persuade",
            "ride",
            "shut",
            "blow",
            "estimate",
            "recover",
            "score",
            "slip",
            "count",
            "hate",
            "attach",
            "exercise",
            "house",
            "lean",
            "roll",
            "wash",
            "accuse",
            "bind",
            "judge",
            "rest",
            "steal",
            "comment",
            "exclude",
            "focus",
            "hurt",
            "stretch",
            "withdraw",
            "back",
            "fix",
            "justify",
            "knock",
            "pursue",
            "switch",
            "benefit",
            "lack",
            "list",
            "occupy",
            "permit",
            "surround",
            "abandon",
            "blame",
            "complain",
            "connect",
            "construct",
            "dominate",
            "engage",
            "paint",
            "quote",
            "view",
            "incorporate",
            "interpret",
            "proceed",
            "search",
            "separate",
            "stress",
            "alter",
            "analyse",
            "arrest",
            "bother",
            "defend",
            "expand",
            "implement",
            "possess",
            "review",
            "suit",
            "tie",
            "assist",
            "calculate",
            "glance",
            "mix",
            "question",
            "resolve",
            "rule",
            "suspect",
            "wake",
            "appeal",
            "challenge",
            "clean",
            "damage",
            "guess",
            "reckon",
            "restore",
            "restrict",
            "specify",
            "constitute",
            "convert",
            "distinguish",
            "submit",
            "trust",
            "urge",
            "feature",
            "land",
            "locate",
            "predict",
            "preserve",
            "solve",
            "sort",
            "struggle",
            "cast",
            "cook",
            "dance",
            "invest",
            "lock",
            "owe",
            "pour",
            "shift",
            "kick",
            "kiss",
            "light",
            "purchase",
            "race",
            "retire",
        ),
        "noun": (
            "people",
            "history",
            "way",
            "art",
            "world",
            "information",
            "map",
            "family",
            "government",
            "health",
            "system",
            "computer",
            "meat",
            "year",
            "thanks",
            "music",
            "person",
            "reading",
            "method",
            "data",
            "food",
            "understanding",
            "theory",
            "law",
            "bird",
            "literature",
            "problem",
            "software",
            "control",
            "knowledge",
            "power",
            "ability",
            "economics",
            "love",
            "internet",
            "television",
            "science",
            "library",
            "nature",
            "fact",
            "product",
            "idea",
            "temperature",
            "investment",
            "area",
            "society",
            "activity",
            "story",
            "industry",
            "media",
            "thing",
            "oven",
            "community",
            "definition",
            "safety",
            "quality",
            "development",
            "language",
            "management",
            "player",
            "variety",
            "video",
            "week",
            "security",
            "country",
            "exam",
            "movie",
            "organization",
            "equipment",
            "physics",
            "analysis",
            "policy",
            "series",
            "thought",
            "basis",
            "boyfriend",
            "direction",
            "strategy",
            "technology",
            "army",
            "camera",
            "freedom",
            "paper",
            "environment",
            "child",
            "instance",
            "month",
            "truth",
            "marketing",
            "university",
            "writing",
            "article",
            "department",
            "difference",
            "goal",
            "news",
            "audience",
            "fishing",
            "growth",
            "income",
            "marriage",
            "user",
            "combination",
            "failure",
            "meaning",
            "medicine",
            "philosophy",
            "teacher",
            "communication",
            "night",
            "chemistry",
            "disease",
            "disk",
            "energy",
            "nation",
            "road",
            "role",
            "soup",
            "advertising",
            "location",
            "success",
            "addition",
            "apartment",
            "education",
            "math",
            "moment",
            "painting",
            "politics",
            "attention",
            "decision",
            "event",
            "property",
            "shopping",
            "student",
            "wood",
            "competition",
            "distribution",
            "entertainment",
            "office",
            "population",
            "president",
            "unit",
            "category",
            "cigarette",
            "context",
            "introduction",
            "opportunity",
            "performance",
            "driver",
            "flight",
            "length",
            "magazine",
            "newspaper",
            "relationship",
            "teaching",
            "cell",
            "dealer",
            "finding",
            "lake",
            "member",
            "message",
            "phone",
            "scene",
            "appearance",
            "association",
            "concept",
            "customer",
            "death",
            "discussion",
            "housing",
            "inflation",
            "insurance",
            "mood",
            "woman",
            "advice",
            "blood",
            "effort",
            "expression",
            "importance",
            "opinion",
            "payment",
            "reality",
            "responsibility",
            "situation",
            "skill",
            "statement",
            "wealth",
            "application",
            "city",
            "county",
            "depth",
            "estate",
            "foundation",
            "grandmother",
            "heart",
            "perspective",
            "photo",
            "recipe",
            "studio",
            "topic",
            "collection",
            "depression",
            "imagination",
            "passion",
            "percentage",
            "resource",
            "setting",
            "ad",
            "agency",
            "college",
            "connection",
            "criticism",
            "debt",
            "description",
            "memory",
            "patience",
            "secretary",
            "solution",
            "administration",
            "aspect",
            "attitude",
            "director",
            "personality",
            "psychology",
            "recommendation",
            "response",
            "selection",
            "storage",
            "version",
            "alcohol",
            "argument",
            "complaint",
            "contract",
            "emphasis",
            "highway",
            "loss",
            "membership",
            "possession",
            "preparation",
            "steak",
            "union",
            "agreement",
            "cancer",
            "currency",
            "employment",
            "engineering",
            "entry",
            "interaction",
            "mixture",
            "preference",
            "region",
            "republic",
            "tradition",
            "virus",
            "actor",
            "classroom",
            "delivery",
            "device",
            "difficulty",
            "drama",
            "election",
            "engine",
            "football",
            "guidance",
            "hotel",
            "owner",
            "priority",
            "protection",
            "suggestion",
            "tension",
            "variation",
            "anxiety",
            "atmosphere",
            "awareness",
            "bath",
            "bread",
            "candidate",
            "climate",
            "comparison",
            "confusion",
            "construction",
            "elevator",
            "emotion",
            "employee",
            "employer",
            "guest",
            "height",
            "leadership",
            "mall",
            "manager",
            "operation",
            "recording",
            "sample",
            "transportation",
            "charity",
            "cousin",
            "disaster",
            "editor",
            "efficiency",
            "excitement",
            "extent",
            "feedback",
            "guitar",
            "homework",
            "leader",
            "mom",
            "outcome",
            "permission",
            "presentation",
            "promotion",
            "reflection",
            "refrigerator",
            "resolution",
            "revenue",
            "session",
            "singer",
            "tennis",
            "basket",
            "bonus",
            "cabinet",
            "childhood",
            "church",
            "clothes",
            "coffee",
            "dinner",
            "drawing",
            "hair",
            "hearing",
            "initiative",
            "judgment",
            "lab",
            "measurement",
            "mode",
            "mud",
            "orange",
            "poetry",
            "police",
            "possibility",
            "procedure",
            "queen",
            "ratio",
            "relation",
            "restaurant",
            "satisfaction",
            "sector",
            "signature",
            "significance",
            "song",
            "tooth",
            "town",
            "vehicle",
            "volume",
            "wife",
            "accident",
            "airport",
            "appointment",
            "arrival",
            "assumption",
            "baseball",
            "chapter",
            "committee",
            "conversation",
            "database",
            "enthusiasm",
            "error",
            "explanation",
            "farmer",
            "gate",
            "girl",
            "hall",
            "historian",
            "hospital",
            "injury",
            "instruction",
            "maintenance",
            "manufacturer",
            "meal",
            "perception",
            "pie",
            "poem",
            "presence",
            "proposal",
            "reception",
            "replacement",
            "revolution",
            "river",
            "son",
            "speech",
            "tea",
            "village",
            "warning",
            "winner",
            "worker",
            "writer",
            "assistance",
            "breath",
            "buyer",
            "chest",
            "chocolate",
            "conclusion",
            "contribution",
            "cookie",
            "courage",
            "desk",
            "drawer",
            "establishment",
            "examination",
            "garbage",
            "grocery",
            "honey",
            "impression",
            "improvement",
            "independence",
            "insect",
            "inspection",
            "inspector",
            "king",
            "ladder",
            "menu",
            "penalty",
            "piano",
            "potato",
            "profession",
            "professor",
            "quantity",
            "reaction",
            "requirement",
            "salad",
            "sister",
            "supermarket",
            "tongue",
            "weakness",
            "wedding",
            "affair",
            "ambition",
            "analyst",
            "apple",
            "assignment",
            "assistant",
            "bathroom",
            "bedroom",
            "beer",
            "birthday",
            "celebration",
            "championship",
            "cheek",
            "client",
            "consequence",
            "departure",
            "diamond",
            "dirt",
            "ear",
            "fortune",
            "friendship",
            "funeral",
            "gene",
            "girlfriend",
            "hat",
            "indication",
            "intention",
            "lady",
            "midnight",
            "negotiation",
            "obligation",
            "passenger",
            "pizza",
            "platform",
            "poet",
            "pollution",
            "recognition",
            "reputation",
            "shirt",
            "sir",
            "speaker",
            "stranger",
            "surgery",
            "sympathy",
            "tale",
            "throat",
            "trainer",
            "uncle",
            "youth",
            "time",
            "work",
            "film",
            "water",
            "money",
            "example",
            "while",
            "business",
            "study",
            "game",
            "life",
            "form",
            "air",
            "day",
            "place",
            "number",
            "part",
            "field",
            "fish",
            "back",
            "process",
            "heat",
            "hand",
            "experience",
            "job",
            "book",
            "end",
            "point",
            "type",
            "home",
            "economy",
            "value",
            "body",
            "market",
            "guide",
            "interest",
            "state",
            "radio",
            "course",
            "company",
            "price",
            "size",
            "card",
            "list",
            "mind",
            "trade",
            "line",
            "care",
            "group",
            "risk",
            "word",
            "fat",
            "force",
            "key",
            "light",
            "training",
            "name",
            "school",
            "top",
            "amount",
            "level",
            "order",
            "practice",
            "research",
            "sense",
            "service",
            "piece",
            "web",
            "boss",
            "sport",
            "fun",
            "house",
            "page",
            "term",
            "test",
            "answer",
            "sound",
            "focus",
            "matter",
            "kind",
            "soil",
            "board",
            "oil",
            "picture",
            "access",
            "garden",
            "range",
            "rate",
            "reason",
            "future",
            "site",
            "demand",
            "exercise",
            "image",
            "case",
            "cause",
            "coast",
            "action",
            "age",
            "bad",
            "boat",
            "record",
            "result",
            "section",
            "building",
            "mouse",
            "cash",
            "class",
            "nothing",
            "period",
            "plan",
            "store",
            "tax",
            "side",
            "subject",
            "space",
            "rule",
            "stock",
            "weather",
            "chance",
            "figure",
            "man",
            "model",
            "source",
            "beginning",
            "earth",
            "program",
            "chicken",
            "design",
            "feature",
            "head",
            "material",
            "purpose",
            "question",
            "rock",
            "salt",
            "act",
            "birth",
            "car",
            "dog",
            "object",
            "scale",
            "sun",
            "note",
            "profit",
            "rent",
            "speed",
            "style",
            "war",
            "bank",
            "craft",
            "half",
            "inside",
            "outside",
            "standard",
            "bus",
            "exchange",
            "eye",
            "fire",
            "position",
            "pressure",
            "stress",
            "advantage",
            "benefit",
            "box",
            "frame",
            "issue",
            "step",
            "cycle",
            "face",
            "item",
            "metal",
            "paint",
            "review",
            "room",
            "screen",
            "structure",
            "view",
            "account",
            "ball",
            "discipline",
            "medium",
            "share",
            "balance",
            "bit",
            "black",
            "bottom",
            "choice",
            "gift",
            "impact",
            "machine",
            "shape",
            "tool",
            "wind",
            "address",
            "average",
            "career",
            "culture",
            "morning",
            "pot",
            "sign",
            "table",
            "task",
            "condition",
            "contact",
            "credit",
            "egg",
            "hope",
            "ice",
            "network",
            "north",
            "square",
            "attempt",
            "date",
            "effect",
            "link",
            "post",
            "star",
            "voice",
            "capital",
            "challenge",
            "friend",
            "self",
            "shot",
            "brush",
            "couple",
            "debate",
            "exit",
            "front",
            "function",
            "lack",
            "living",
            "plant",
            "plastic",
            "spot",
            "summer",
            "taste",
            "theme",
            "track",
            "wing",
            "brain",
            "button",
            "click",
            "desire",
            "foot",
            "gas",
            "influence",
            "notice",
            "rain",
            "wall",
            "base",
            "damage",
            "distance",
            "feeling",
            "pair",
            "savings",
            "staff",
            "sugar",
            "target",
            "text",
            "animal",
            "author",
            "budget",
            "discount",
            "file",
            "ground",
            "lesson",
            "minute",
            "officer",
            "phase",
            "reference",
            "register",
            "sky",
            "stage",
            "stick",
            "title",
            "trouble",
            "bowl",
            "bridge",
            "campaign",
            "character",
            "club",
            "edge",
            "evidence",
            "fan",
            "letter",
            "lock",
            "maximum",
            "novel",
            "option",
            "pack",
            "park",
            "plenty",
            "quarter",
            "skin",
            "sort",
            "weight",
            "baby",
            "background",
            "carry",
            "dish",
            "factor",
            "fruit",
            "glass",
            "joint",
            "master",
            "muscle",
            "red",
            "strength",
            "traffic",
            "trip",
            "vegetable",
            "appeal",
            "chart",
            "gear",
            "ideal",
            "kitchen",
            "land",
            "log",
            "mother",
            "net",
            "party",
            "principle",
            "relative",
            "sale",
            "season",
            "signal",
            "spirit",
            "street",
            "tree",
            "wave",
            "belt",
            "bench",
            "commission",
            "copy",
            "drop",
            "minimum",
            "path",
            "progress",
            "project",
            "sea",
            "south",
            "status",
            "stuff",
            "ticket",
            "tour",
            "angle",
            "blue",
            "breakfast",
            "confidence",
            "daughter",
            "degree",
            "doctor",
            "dot",
            "dream",
            "duty",
            "essay",
            "father",
            "fee",
            "finance",
            "hour",
            "juice",
            "limit",
            "luck",
            "milk",
            "mouth",
            "peace",
            "pipe",
            "seat",
            "stable",
            "storm",
            "substance",
            "team",
            "trick",
            "afternoon",
            "bat",
            "beach",
            "blank",
            "catch",
            "chain",
            "consideration",
            "cream",
            "crew",
            "detail",
            "gold",
            "interview",
            "kid",
            "mark",
            "match",
            "mission",
            "pain",
            "pleasure",
            "score",
            "screw",
            "sex",
            "shop",
            "shower",
            "suit",
            "tone",
            "window",
            "agent",
            "band",
            "block",
            "bone",
            "calendar",
            "cap",
            "coat",
            "contest",
            "corner",
            "court",
            "cup",
            "district",
            "door",
            "east",
            "finger",
            "garage",
            "guarantee",
            "hole",
            "hook",
            "implement",
            "layer",
            "lecture",
            "lie",
            "manner",
            "meeting",
            "nose",
            "parking",
            "partner",
            "profile",
            "respect",
            "rice",
            "routine",
            "schedule",
            "swimming",
            "telephone",
            "tip",
            "winter",
            "airline",
            "bag",
            "battle",
            "bed",
            "bill",
            "bother",
            "cake",
            "code",
            "curve",
            "designer",
            "dimension",
            "dress",
            "ease",
            "emergency",
            "evening",
            "extension",
            "farm",
            "fight",
            "gap",
            "grade",
            "holiday",
            "horror",
            "horse",
            "host",
            "husband",
            "loan",
            "mistake",
            "mountain",
            "nail",
            "noise",
            "occasion",
            "package",
            "patient",
            "pause",
            "phrase",
            "proof",
            "race",
            "relief",
            "sand",
            "sentence",
            "shoulder",
            "smoke",
            "stomach",
            "string",
            "tourist",
            "towel",
            "vacation",
            "west",
            "wheel",
            "wine",
            "arm",
            "aside",
            "associate",
            "bet",
            "blow",
            "border",
            "branch",
            "breast",
            "brother",
            "buddy",
            "bunch",
            "chip",
            "coach",
            "cross",
            "document",
            "draft",
            "dust",
            "expert",
            "floor",
            "god",
            "golf",
            "habit",
            "iron",
            "judge",
            "knife",
            "landscape",
            "league",
            "mail",
            "mess",
            "native",
            "opening",
            "parent",
            "pattern",
            "pin",
            "pool",
            "pound",
            "request",
            "salary",
            "shame",
            "shelter",
            "shoe",
            "silver",
            "tackle",
            "tank",
            "trust",
            "assist",
            "bake",
            "bar",
            "bell",
            "bike",
            "blame",
            "boy",
            "brick",
            "chair",
            "closet",
            "clue",
            "collar",
            "comment",
            "conference",
            "devil",
            "diet",
            "fear",
            "fuel",
            "glove",
            "jacket",
            "lunch",
            "monitor",
            "mortgage",
            "nurse",
            "pace",
            "panic",
            "peak",
            "plane",
            "reward",
            "row",
            "sandwich",
            "shock",
            "spite",
            "spray",
            "surprise",
            "till",
            "transition",
            "weekend",
            "welcome",
            "yard",
            "alarm",
            "bend",
            "bicycle",
            "bite",
            "blind",
            "bottle",
            "cable",
            "candle",
            "clerk",
            "cloud",
            "concert",
            "counter",
            "flower",
            "grandfather",
            "harm",
            "knee",
            "lawyer",
            "leather",
            "load",
            "mirror",
            "neck",
            "pension",
            "plate",
            "purple",
            "ruin",
            "ship",
            "skirt",
            "slice",
            "snow",
            "specialist",
            "stroke",
            "switch",
            "trash",
            "tune",
            "zone",
            "anger",
            "award",
            "bid",
            "bitter",
            "boot",
            "bug",
            "camp",
            "candy",
            "carpet",
            "cat",
            "champion",
            "channel",
            "clock",
            "comfort",
            "cow",
            "crack",
            "engineer",
            "entrance",
            "fault",
            "grass",
            "guy",
            "hell",
            "highlight",
            "incident",
            "island",
            "joke",
            "jury",
            "leg",
            "lip",
            "mate",
            "motor",
            "nerve",
            "passage",
            "pen",
            "pride",
            "priest",
            "prize",
            "promise",
            "resident",
            "resort",
            "ring",
            "roof",
            "rope",
            "sail",
            "scheme",
            "script",
            "sock",
            "station",
            "toe",
            "tower",
            "truck",
            "witness",
        ),
        "adverb": (
            "not",
            "also",
            "very",
            "often",
            "however",
            "too",
            "usually",
            "really",
            "early",
            "never",
            "always",
            "sometimes",
            "together",
            "likely",
            "simply",
            "generally",
            "instead",
            "actually",
            "again",
            "rather",
            "almost",
            "especially",
            "ever",
            "quickly",
            "probably",
            "already",
            "below",
            "directly",
            "therefore",
            "else",
            "thus",
            "easily",
            "eventually",
            "exactly",
            "certainly",
            "normally",
            "currently",
            "extremely",
            "finally",
            "constantly",
            "properly",
            "soon",
            "specifically",
            "ahead",
            "daily",
            "highly",
            "immediately",
            "relatively",
            "slowly",
            "fairly",
            "primarily",
            "completely",
            "ultimately",
            "widely",
            "recently",
            "seriously",
            "frequently",
            "fully",
            "mostly",
            "naturally",
            "nearly",
            "occasionally",
            "carefully",
            "clearly",
            "essentially",
            "possibly",
            "slightly",
            "somewhat",
            "equally",
            "greatly",
            "necessarily",
            "personally",
            "rarely",
            "regularly",
            "similarly",
            "basically",
            "closely",
            "effectively",
            "initially",
            "literally",
            "mainly",
            "merely",
            "gently",
            "hopefully",
            "originally",
            "roughly",
            "significantly",
            "totally",
            "twice",
            "elsewhere",
            "everywhere",
            "obviously",
            "perfectly",
            "physically",
            "successfully",
            "suddenly",
            "truly",
            "virtually",
            "altogether",
            "anyway",
            "automatically",
            "deeply",
            "definitely",
            "deliberately",
            "hardly",
            "readily",
            "terribly",
            "unfortunately",
            "forth",
            "briefly",
            "moreover",
            "strongly",
            "honestly",
            "previously",
            "as",
            "there",
            "when",
            "how",
            "so",
            "up",
            "out",
            "only",
            "well",
            "then",
            "first",
            "where",
            "why",
            "now",
            "around",
            "once",
            "down",
            "off",
            "here",
            "tonight",
            "away",
            "today",
            "far",
            "quite",
            "later",
            "above",
            "yet",
            "maybe",
            "otherwise",
            "near",
            "forward",
            "somewhere",
            "anywhere",
            "please",
            "forever",
            "somehow",
            "absolutely",
            "abroad",
            "yeah",
            "nowhere",
            "tomorrow",
            "yesterday",
        ),
        "adjective": (
            "different",
            "used",
            "important",
            "every",
            "large",
            "available",
            "popular",
            "able",
            "basic",
            "known",
            "various",
            "difficult",
            "several",
            "united",
            "historical",
            "hot",
            "useful",
            "mental",
            "scared",
            "additional",
            "emotional",
            "old",
            "political",
            "similar",
            "healthy",
            "financial",
            "medical",
            "traditional",
            "federal",
            "entire",
            "strong",
            "actual",
            "significant",
            "successful",
            "electrical",
            "expensive",
            "pregnant",
            "intelligent",
            "interesting",
            "poor",
            "happy",
            "responsible",
            "cute",
            "helpful",
            "recent",
            "willing",
            "nice",
            "wonderful",
            "impossible",
            "serious",
            "huge",
            "rare",
            "technical",
            "typical",
            "competitive",
            "critical",
            "electronic",
            "immediate",
            "aware",
            "educational",
            "environmental",
            "global",
            "legal",
            "relevant",
            "accurate",
            "capable",
            "dangerous",
            "dramatic",
            "efficient",
            "powerful",
            "foreign",
            "hungry",
            "practical",
            "psychological",
            "severe",
            "suitable",
            "numerous",
            "sufficient",
            "unusual",
            "consistent",
            "cultural",
            "existing",
            "famous",
            "pure",
            "afraid",
            "obvious",
            "careful",
            "latter",
            "unhappy",
            "acceptable",
            "aggressive",
            "boring",
            "distinct",
            "eastern",
            "logical",
            "reasonable",
            "strict",
            "administrative",
            "automatic",
            "civil",
            "former",
            "massive",
            "southern",
            "unfair",
            "visible",
            "alive",
            "angry",
            "desperate",
            "exciting",
            "friendly",
            "lucky",
            "realistic",
            "sorry",
            "ugly",
            "unlikely",
            "anxious",
            "comprehensive",
            "curious",
            "impressive",
            "informal",
            "inner",
            "pleasant",
            "sexual",
            "sudden",
            "terrible",
            "unable",
            "weak",
            "wooden",
            "asleep",
            "confident",
            "conscious",
            "decent",
            "embarrassed",
            "guilty",
            "lonely",
            "mad",
            "nervous",
            "odd",
            "remarkable",
            "substantial",
            "suspicious",
            "tall",
            "tiny",
            "more",
            "some",
            "all",
            "many",
            "most",
            "other",
            "such",
            "even",
            "new",
            "just",
            "good",
            "any",
            "each",
            "much",
            "own",
            "great",
            "another",
            "same",
            "few",
            "free",
            "right",
            "still",
            "best",
            "public",
            "human",
            "both",
            "local",
            "sure",
            "better",
            "general",
            "specific",
            "enough",
            "long",
            "small",
            "less",
            "high",
            "certain",
            "little",
            "common",
            "next",
            "simple",
            "hard",
            "past",
            "big",
            "possible",
            "particular",
            "real",
            "major",
            "personal",
            "current",
            "left",
            "national",
            "least",
            "natural",
            "physical",
            "short",
            "last",
            "single",
            "individual",
            "main",
            "potential",
            "professional",
            "international",
            "lower",
            "open",
            "according",
            "alternative",
            "special",
            "working",
            "true",
            "whole",
            "clear",
            "dry",
            "easy",
            "cold",
            "commercial",
            "full",
            "low",
            "primary",
            "worth",
            "necessary",
            "positive",
            "present",
            "close",
            "creative",
            "green",
            "late",
            "fit",
            "glad",
            "proper",
            "complex",
            "content",
            "due",
            "effective",
            "middle",
            "regular",
            "fast",
            "independent",
            "original",
            "wide",
            "beautiful",
            "complete",
            "active",
            "negative",
            "safe",
            "visual",
            "wrong",
            "ago",
            "quick",
            "ready",
            "straight",
            "white",
            "direct",
            "excellent",
            "extra",
            "junior",
            "pretty",
            "unique",
            "classic",
            "final",
            "overall",
            "private",
            "separate",
            "western",
            "alone",
            "familiar",
            "official",
            "perfect",
            "bright",
            "broad",
            "comfortable",
            "flat",
            "rich",
            "warm",
            "young",
            "heavy",
            "valuable",
            "correct",
            "leading",
            "slow",
            "clean",
            "fresh",
            "normal",
            "secret",
            "tough",
            "brown",
            "cheap",
            "deep",
            "objective",
            "secure",
            "thin",
            "chemical",
            "cool",
            "extreme",
            "exact",
            "fair",
            "fine",
            "formal",
            "opposite",
            "remote",
            "total",
            "vast",
            "lost",
            "smooth",
            "dark",
            "double",
            "equal",
            "firm",
            "frequent",
            "internal",
            "sensitive",
            "constant",
            "minor",
            "previous",
            "raw",
            "soft",
            "solid",
            "weird",
            "amazing",
            "annual",
            "busy",
            "dead",
            "false",
            "round",
            "sharp",
            "thick",
            "wise",
            "equivalent",
            "initial",
            "narrow",
            "nearby",
            "proud",
            "spiritual",
            "wild",
            "adult",
            "apart",
            "brief",
            "crazy",
            "prior",
            "rough",
            "sad",
            "sick",
            "strange",
            "external",
            "illegal",
            "loud",
            "mobile",
            "nasty",
            "ordinary",
            "royal",
            "senior",
            "super",
            "tight",
            "upper",
            "yellow",
            "dependent",
            "funny",
            "gross",
            "ill",
            "spare",
            "sweet",
            "upstairs",
            "usual",
            "brave",
            "calm",
            "dirty",
            "downtown",
            "grand",
            "honest",
            "loose",
            "male",
            "quiet",
            "brilliant",
            "dear",
            "drunk",
            "empty",
            "female",
            "inevitable",
            "neat",
            "ok",
            "representative",
            "silly",
            "slight",
            "smart",
            "stupid",
            "temporary",
            "weekly",
        ),
    }
