"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FileSystemProvider = void 0;
const fs_1 = require("fs");
const util_1 = require("util");
const vscode_css_languageservice_1 = require("vscode-css-languageservice");
const utils_1 = require("../../utils");
class FileSystemProvider {
    constructor() {
        // TODO use fs/promises after we bumps the target nodejs versions
        this.promisifyStat = (0, util_1.promisify)(fs_1.stat);
        this.promisifyReaddir = (0, util_1.promisify)(fs_1.readdir);
        this.readDirectory = this.readDirectory.bind(this);
        this.stat = this.stat.bind(this);
    }
    async stat(uri) {
        const path = (0, utils_1.urlToPath)(uri);
        if (!path) {
            return this.unknownStat();
        }
        let stat;
        try {
            stat = await this.promisifyStat(path);
        }
        catch (error) {
            if (error != null &&
                typeof error === 'object' &&
                'code' in error &&
                error.code === 'ENOENT') {
                return {
                    type: vscode_css_languageservice_1.FileType.Unknown,
                    ctime: -1,
                    mtime: -1,
                    size: -1
                };
            }
            throw error;
        }
        return {
            ctime: stat.ctimeMs,
            mtime: stat.mtimeMs,
            size: stat.size,
            type: this.getFileType(stat)
        };
    }
    unknownStat() {
        return {
            type: vscode_css_languageservice_1.FileType.Unknown,
            ctime: -1,
            mtime: -1,
            size: -1
        };
    }
    getFileType(stat) {
        return stat.isDirectory()
            ? vscode_css_languageservice_1.FileType.Directory
            : stat.isFile()
                ? vscode_css_languageservice_1.FileType.File
                : stat.isSymbolicLink()
                    ? vscode_css_languageservice_1.FileType.SymbolicLink
                    : vscode_css_languageservice_1.FileType.Unknown;
    }
    async readDirectory(uri) {
        const path = (0, utils_1.urlToPath)(uri);
        if (!path) {
            return [];
        }
        const files = await this.promisifyReaddir(path, {
            withFileTypes: true
        });
        return files.map((file) => [file.name, this.getFileType(file)]);
    }
}
exports.FileSystemProvider = FileSystemProvider;
//# sourceMappingURL=FileSystemProvider.js.map