// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/FinalState.hh"
#include "Rivet/Projections/UnstableParticles.hh"

namespace Rivet {


  /// @brief e+e- -> → D* D* pi0
  class BESIII_2015_I1382146 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(BESIII_2015_I1382146);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      // set the PDG code
      _pid = getOption<double>("PID", 9030443);
      // projections
      declare(FinalState(), "FS");
      // histograms
      book(_sigma, 1, 1, 1);
      book(_h["avg"], 2, 1, 1);
      size_t ih = 1;
      for (double eVal : allowedEnergies()) {
        const string en = toString(round(eVal/MeV));
        if (isCompatibleWithSqrtS(eVal))  _sqs = en;
        book(_h[en], 2, 1, 1+ih);
        ++ih;
      }
      raiseBeamErrorIf(_sqs.empty());
      _edge = (_sqs == "4230"s)? "4.23" : "4.26";
    }


    /// Perform the per-event analysis
    void analyze(const Event& event) {
      Particles fs = apply<FinalState>(event, "FS").particles();
      Particles DD,other;
      for (const Particle& p : fs) {
        Particle parent=p;
        while (!parent.parents().empty()) {
          parent=parent.parents()[0];
          if (parent.abspid()==413 || parent.abspid()==423) break;
        }
        if (parent.abspid()==413 || parent.abspid()==423) {
          bool found=false;
          for (const auto& D : DD) {
            // D already in list
            if (fuzzyEquals(D.mom(), parent.mom())) {
              found=true;
              break;
            }
          }
          if (!found) DD.push_back(parent);
        }
        else {
          other.push_back(p);
        }
      }
      // D Dbar + neutral pion
      if (DD.size()!=2 || other.size()!=1) vetoEvent;
      if (DD[0].pid()!=-DD[1].pid()) vetoEvent;
      if (other[0].pid()!=111) vetoEvent;
      const double mass = (DD[0].mom()+DD[1].mom()).mass();
      _h[_sqs]->fill(mass/GeV);
      _h["avg"]->fill(mass/GeV);
      // parent Z0
      if (DD[0].parents()[0].pid()==_pid &&
          DD[1].parents()[0].pid()==_pid &&
          fuzzyEquals(DD[0].parents()[0].mom(),DD[1].parents()[0].mom()) ) _sigma->fill(_edge);
    }


    /// Normalise histograms etc., after the run
    void finalize() {
      // distributions
      normalize(_h, 1.0, false);
      // cross section
      scale(_sigma, crossSection()/ sumOfWeights() /picobarn);
    }
    /// @}


    /// @name Histograms
    /// @{
    BinnedHistoPtr<string> _sigma;
    map<string,Histo1DPtr> _h;
    string _sqs = "", _edge = "";
    int _pid;
    /// @}


  };


  RIVET_DECLARE_PLUGIN(BESIII_2015_I1382146);

}
