#
# Copyright (C) 2017 J.F.Dockes
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

# Manage the [folders] section of the tree.
#
# Object Id prefix: 0$uprcl$folders
#
# Data structure:
#
# The _rcldocs list has one entry for each document in the index (mime:* search), it is the result
# of a 'mime:*' recoll query. Each entry is, or is similar to, a recoll.Doc(), a kind of dict.
#
# The _dirvec list has one entry for each directory. Each entry is a Python dict, mapping the
# directory entries' names to a pair (diridx,docidx), where:
#
#  - diridx is an index into _dirvec if the name is a directory, else -1
#  - docidx is an index into _rcldocs, or -1 if:
#     - There is no _rcldocs entry, which happens if there is no corresponding recoll doc for an
#       intermediary element in a path, or because this is a synthetic 'Group' entry.
#     - Or, while we build the structure, temporarily, if the doc was not yet seen. The value will
#       then be updated when we see it.
#
# Directories are created as needed by splitting the paths/urls from _rcldocs (and possibly adding
# some for groupings defined by the Group tag). Directories have no direct relation to their
# possible recoll index objects (if any), they are identified by their _dirvec index
#
# Note: docidx is usually set in the pair for a directory, if there is a Doc entry, but I don't
# think that it is ever used. The Recoll Doc for a directory has nothing very interesting in it.
#
# Note: We could probably use a single value, with a convention saying, e.g., that > 0 is for docs
# and < -1 for folders. Check if this saves a significant amount of memory.
#
# Each directory has a special ".." entry with a diridx pointing to the parent directory. This
# allows building a path from a container id (aka pwd).
#
# Only playlists have a "." entry (needed during init)
#
# Entry 0 in _dirvec is special: it holds the 'topdirs' from the recoll configuration. The entries
# are paths instead of simple names, and their docidx is 0. The diridx points to a regular dirvec
# entry.
#
# Object ids inside the section:
#    Container: $d<diridx> where <diridx> indexes into _dirvec
#    Item: $i<docidx> where <docidx> indexes into _rcldocs
# Note: this is very different from what Minim does. Minim uses actual objid paths as objids.
# E.g. 0$folders$f1589$f1593$f1604$*i11609. Must make pwd and any walk up the tree much easier.
# 
# We used to build an _xid2idx xdocid->objidx map to allow a Recoll item search result to be
# connected back to the folders tree, but this was not actually useful (bogus objids for items in
# search results are quite probably ok). Also quite probably, this could also be done using the URL,
# as it is what we use to build the folders tree in the first place. _xid2idx is currently
# desactivated (see comment). See objidfordoc() in this file for how we compute objids for
# search results which are actual directories (which need to be browsable).
#
#

import os
import shlex
import sys
import time

from recoll import recoll
from recoll import qresultstore
from recoll import rclconfig
from upmplgutils import uplog, direntry, getOptionValue
from uprclutils import audiomtypes, rcldoctoentry, cmpentries
import uprclutils
import uprclinit

# All Doc fields which we may want to access (reserve slots in the
# resultstore). We use an inclusion list, and end up with a smaller
# store than by using an exclusion list, but it's a bit more difficult
# to manage.
#
# +  possibly 'xdocid' and/or 'rcludi' if/when needed?
#
_otherneededfields = [
    "albumartist",
    "allartists",
    "comment",
    "composer",
    "conductor",
    "contentgroup",
    "date",
    "dmtime",
    "discnumber",
    "embdimg",
    "filename",
    "genre",
    "group",
    "label",
    "lyricist",
    "orchestra",
    "performer",
]


# All standard cover art file names:
_artexts = (".jpg", ".png")
def _artnamegen(base):
    for ext in _artexts:
        yield base + ext
_folderartbases = ("cover", "folder")
_folderartnames = []
for base in _folderartbases:
    for path in _artnamegen(base):
        _folderartnames.append(path)


class Folders(object):

    # Initialize (read recoll data and build tree).
    def __init__(self, confdir, httphp, pathprefix):
        self._idprefix = "0$uprcl$folders"
        self._httphp = httphp
        self._pprefix = pathprefix
        # Debug : limit processed recoll entries for speed
        self._maxrclcnt = 0
        # Overflow storage for synthetic records created for playlists
        # url entries. Uses docidx values starting at len(_rcldocs),
        # with actual index (value - len(_rcldocs))
        self._moredocs = []
        self._fetchalldocs(confdir)
        self._rcl2folders(confdir)
        self._enabletags = uprclinit.g_minimconfig.getboolvalue("showExtras", True)
        self._notagview = getOptionValue("uprclnotagview", False)

    def rcldocs(self):
        return self._rcldocs

    # Create new directory entry: insert in father and append dirvec slot
    # (with ".." entry)
    def _createdir(self, fathidx, docidx, nm):
        self._dirvec.append({})
        thisidx = len(self._dirvec) - 1
        self._dirvec[fathidx][nm] = (thisidx, docidx)
        self._dirvec[-1][".."] = (fathidx, -1)
        self._dirvec[-1]["."] = (thisidx, docidx)
        return len(self._dirvec) - 1

    # Create directory for playlist. Create + populate. The docs which
    # are pointed by the playlist entries may not be in the tree yet,
    # so we don't know how to find them (can't walk the tree yet).
    # Just store the diridx and populate all playlists at the end
    def _createpldir(self, fathidx, docidx, doc, nm):
        myidx = self._createdir(fathidx, docidx, nm)
        # We need a "." entry
        self._dirvec[myidx]["."] = (myidx, docidx)
        self._playlists.append(myidx)
        return myidx


    # Find the doc index for a filesystem path.
    # We use a temporary doc to call _stat()
    def statpath(self, path, verbose=False):
        doc = dict();
        doc["group"] = None
        doc["url"] = "file://" + path
        fathidx, docidx = self._stat(doc, verbose)
        if docidx >= 0 and docidx < len(self._rcldocs):
            return docidx
        return -1


    # Initialize all playlists after the tree is otherwise complete
    def _initplaylists(self, confdir):
        # We use a recoll db connection for creating bogus rcl docs
        rcldb = recoll.connect(confdir=confdir)
        for diridx in self._playlists:
            pldocidx = self._dirvec[diridx]["."][1]
            pldoc = self._rcldocs[pldocidx]
            plpath = uprclutils.docpath(pldoc)
            try:
                m3u = uprclutils.M3u(plpath)
            except Exception as ex:
                uplog(f"M3u open failed: plpath [{plpath}] : {ex}")
                continue
            for urlorpath in m3u:
                if m3u.urlRE.match(urlorpath):
                    # Actual URL (usually http). Create bogus doc
                    doc = uprclutils.docforurl(rcldb, urlorpath)
                    self._moredocs.append(doc)
                    docidx = len(self._rcldocs) + len(self._moredocs) - 1
                    tt = doc["title"]
                    self._dirvec[diridx][tt] = (-1, docidx)
                else:
                    if not os.path.isabs(urlorpath):
                        urlorpath = os.path.join(os.path.dirname(plpath), urlorpath)
                    docidx = self.statpath(urlorpath)
                    if docidx >= 0:
                        #uplog(f"Track OK for playlist [{plpath}] entry [{urlorpath}]")
                        elt = os.path.split(urlorpath)[1]
                        self._dirvec[diridx][elt] = (-1, docidx)
                    else:
                        uplog(f"No track for playlist [{plpath}] entry [{urlorpath}]")
                        #self.statpath(urlorpath, verbose=True)

    # Compute and return the index in root of the topdir we're a child and the rest of the path
    # split as a list.
    # The root entry (diridx 0) is special because its keys are the
    # topdirs paths, not simple names. We look with what topdir path
    # this doc belongs to, then return the appropriate diridx and the
    # split remainder of the path
    def _pathbeyondtopdirs(self, doc):
        path = doc["url"][7:].rstrip("/")

        # Determine the root entry (topdirs element). Special because its path is not a simple
        # name. Fathidx is its index in _dirvec
        firstdiridx = -1
        for rtpath, idx in self._dirvec[0].items():
            if path.startswith(rtpath):
                firstdiridx = idx[0]
                break
        if firstdiridx == -1:
            #uplog("No parent in topdirs: %s" % path)
            return None, None

        # Compute rest of path. If there is none, we're not interested.
        path1 = path[len(rtpath) :]
        if len(path1) == 0:
            return None, None

        # If there is a Group field, just add it as a virtual
        # directory in the path. This only affects the visible tree,
        # not the 'real' PATHs of course.
        try:
            if doc["group"]:
                a = os.path.dirname(path1)
                b = os.path.basename(path1)
                path1 = os.path.join(a, doc["group"], b)
        except:
            pass

        # Split path. The caller will walk the list (possibly creating
        # directory entries as needed, or doing something else).
        path = path1.split("/")[1:]
        return firstdiridx, path


    # Main folders build method: walk the recoll docs array and split
    # the URLs paths to build the [folders] data structure
    def _rcl2folders(self, confdir):
        self._dirvec = []
        self._xid2idx = {}
        # This is used to store the diridx for the playlists during
        # the initial walk, for initialization when the tree is
        # complete.
        self._playlists = []

        start = time.time()

        rclconf = rclconfig.RclConfig(confdir)
        topdirs = [os.path.expanduser(d) for d in shlex.split(rclconf.getConfParam("topdirs"))]
        topdirs = [d.rstrip("/") for d in topdirs]

        # Create the 1st entry. This is special because it holds the
        # recoll topdirs, which are paths instead of simple names. There
        # does not seem any need to build the tree between a topdir and /
        self._dirvec.append({})
        self._dirvec[0][".."] = (0, -1)
        for d in topdirs:
            self._dirvec.append({})
            self._dirvec[0][d] = (len(self._dirvec) - 1, -1)
            self._dirvec[-1][".."] = (0, -1)

        # Walk the doc list and update the directory tree according to the url: create intermediary
        # directories if needed, create leaf entry.
        #
        # Binary path issue: at the moment the python rclconfig can't handle binary (the underlying
        # conftree.py can, we'd need a binary stringToStrings). So the topdirs entries have to be
        # strings, and so we decode the binurl too. This probably could be changed we wanted to
        # support binary, (non utf-8) paths. For now, all dir/file names in the tree are str
        for docidx in range(len(self._rcldocs)):
            doc = self._rcldocs[docidx]

            # For linking item search results to the main array. Deactivated for now as it does not
            # seem to be needed (and we would need to add xdocid to the resultstore fields).
            # self._xid2idx[doc["xdocid"]] = docidx

            fathidx, path = self._pathbeyondtopdirs(doc)
            if not fathidx:
                continue

            # uplog("%s"%path, file=sys.stderr)
            for idx in range(len(path)):
                elt = path[idx]
                if elt in self._dirvec[fathidx]:
                    # This path element was already seen
                    # If this is the last entry in the path, maybe update
                    # the doc idx (previous entries were created for
                    # intermediate elements without a Doc).
                    if idx == len(path) - 1:
                        self._dirvec[fathidx][elt] = (self._dirvec[fathidx][elt][0], docidx)
                    # Update fathidx for next iteration
                    fathidx = self._dirvec[fathidx][elt][0]
                else:
                    # Element has no entry in father directory (hence no
                    # self._dirvec entry either).
                    if idx != len(path) - 1:
                        # This is an intermediate element. Create a
                        # Doc-less directory
                        fathidx = self._createdir(fathidx, -1, elt)
                    else:
                        # Last element. If directory, needs a self._dirvec entry
                        if doc["mtype"] == "inode/directory":
                            fathidx = self._createdir(fathidx, docidx, elt)
                        elif doc["mtype"] == "audio/x-mpegurl":
                            fathidx = self._createpldir(fathidx, docidx, doc, elt)
                        else:
                            self._dirvec[fathidx][elt] = (-1, docidx)

        if False:
            for ent in self._dirvec:
                uplog("%s" % ent)

        self._initplaylists(confdir)

        end = time.time()
        uplog("_rcl2folders took %.2f Seconds" % (end - start))

    # Fetch all the docs by querying Recoll with an empty query (needs recoll 1.43.10, else use
    # [mime:*]), which is guaranteed to match every doc.
    # This creates the main doc array, which is then used by all modules.
    #
    # Because we are using the resultstore, the records are not modifyable and the aliastags
    # processing is performed at indexing time by rclaudio. Cf. minimtagfixer.py
    def _fetchalldocs(self, confdir):
        # uplog("_fetchalldocs: has_resultstore: %s" % _has_resultstore)
        start = time.time()

        rcldb = recoll.connect(confdir=confdir)
        rclq = rcldb.query()
        rclq.execute("", stemming=0)
        # rclq.execute('album:a* OR album:b* OR album:c*', stemming=0)
        uplog("Estimated alldocs query results: %d" % (rclq.rowcount))

        fields = [r[1] for r in uprclutils.upnp2rclfields.items()]
        fields += _otherneededfields
        fields += uprclinit.allMinimTags()
        fields = list(set(fields))
        #uplog(f"_fetchalldocs: store fields: {fields}")
        self._rcldocs = qresultstore.QResultStore()
        self._rcldocs.storeQuery(rclq, fieldspec=fields, isinc=True)

        end = time.time()
        uplog("Retrieved %d docs in %.2f Seconds" % (len(self._rcldocs), end - start))


    ##############
    # Browsing the initialized [folders] hierarchy

    # Extract diridx and further path (leading to tags) from objid.
    def _objidtoidx(self, pid):
        if not pid.startswith(self._idprefix):
            raise Exception(f"folders.browse: bad pid [{pid}]")

        if len(self._rcldocs) == 0:
            raise Exception("folders:browse: no docs")

        isitem = False
        dirpth = pid[len(self._idprefix) :]
        if not dirpth:
            idx = 0
            pathremain = ""
        else:
            if dirpth[0:2] == "$d":
                isitem = False
            elif dirpth[0:2] == "$i":
                isitem = True
            else:
                raise Exception("folders:browse: called on non dir objid %s" % pid)
            # Other $sign?
            nextdol = dirpth.find("$", 1)
            if nextdol > 0:
                idx = int(dirpth[2:nextdol])
            else:
                idx = int(dirpth[2:])
            pathremain = None
            if nextdol > 0:
                pathremain = dirpth[nextdol:]

        if isitem:
            if idx >= len(self._rcldocs):
                raise Exception(f"folders:browse: bad pid exceeds rcldocs size [{pid}]")
        else:
            if idx >= len(self._dirvec):
                raise Exception(f"folders:browse: bad pid exceeds dirvec size [{pid}]")

        return (isitem, idx, pathremain)


    # Tell the top module what entries we define in the root
    def rootentries(self, pid):
        return [
            direntry(pid + "folders", pid, "[folders]"),
        ]


    def _docforidx(self, docidx):
        if docidx < len(self._rcldocs):
            return self._rcldocs[docidx]
        else:
            return self._moredocs[docidx - len(self._rcldocs)]


    # Look all non-directory docs inside directory, and return the cover art we find.
    # 
    # TBD:
    # - In the case where this is a Group directory, we'd need to go look into the file system for a
    #   group.xxx image.  As it is, things work if the tracks rely on the group pic (instead of
    #   having an embedded pic or track pic)
    # - playlists: need to look at the physical dir for a e.g. playlistname.jpg.
    #
    # We used to only look for art for direct children tracks, but we now also look at
    # subdirs. This will yield an image from the first subdir which has an image file in
    # it, so somewhat random, but nice anyway.
    def _arturifordironedoc(self, diridx, docidx):
        # Look for art for one object, track or directory.
        # Directories only have doc entries if they are also albums. Else we fake a doc.
        if docidx >= 0 and docidx < len(self._rcldocs):
            doc = self._rcldocs[docidx]
        else:
            doc = {}
            doc["mtype"] = "inode/directory"
            doc["url"] = "file://" + self.dirpath("", diridx)
            doc["group"] = None
            doc["embdimg"] = None
        return self.docarturi(doc, preferfolder=True)

    # Look for art for the directory itself, then its children.
    def _arturifordir(self, thisdiridx, thisdocidx=-1):
        # First look at the directory itself.
        arturi = self._arturifordironedoc(thisdiridx, thisdocidx)
        if arturi:
            return arturi
        # Then look at children.
        for nm, ids in self._dirvec[thisdiridx].items():
            diridx = ids[0]
            docidx = ids[1]
            arturi = self._arturifordironedoc(diridx, docidx)
            if arturi:
                return arturi


    def _browsemeta(self, pid, isitem, idx):
        docidx = -1
        if isitem:
            docidx = idx
        else:
            try:
                ids = self._dirvec[idx]
                docidx = ids[1]
            except:
                pass
        if docidx != -1:
            doc = self._docforidx(docidx)
            id = self._idprefix + "$i" + str(docidx)
            e = rcldoctoentry(id, pid, self._httphp, self._pprefix, doc)
            return [
                e,
            ]


    # Folder hierarchy browse method.
    # @param pid objid is like folders$index
    # @param flag: "meta" or "children".
    def browse(self, pid, flag, offset, count):

        isitem, idx, pthremain = self._objidtoidx(pid)

        # If pthremain is set, this is pointing to 'Tag View'. Pass
        # the request along to the tags browser.
        if pthremain:
            if not pthremain.find("$tagview.0") == 0:
                raise Exception(f"uprclfolders:browse: pid [{pid}]. bad pthremain")
            return uprclinit.getTree("tags").browseFolder(pid, flag, pthremain, self.dirpath(pid))

        # If there is only one entry in root, skip it. This means that 0 and 1 point to the same
        # dir, but this does not seem to be an issue
        if not isitem and idx == 0 and len(self._dirvec[0]) == 2:
            idx = 1

        if flag == "meta":
            if not isitem:
                raise Exception(f"uprclfolders:browse: browsemeta on non-item pid [{pid}]")
            return self._browsemeta(pid, isitem, idx)

        # uplog(f"Folders browse: idx [{idx}] content: [{self._dirvec[idx]}]")
        entries = []
        showtopart = True
        # The basename call is just for diridx==0 (topdirs). Remove it if
        # this proves a performance issue
        
        for nm, ids in self._dirvec[idx].items():
            if nm == ".." or nm == ".":
                continue
            thisdiridx = ids[0]
            thisdocidx = ids[1]
            if thisdiridx >= 0:
                # Skip empty directories
                if len(self._dirvec[thisdiridx]) == 1:
                    continue
                # If there are directories, don't show art for the Tags top entries, this would
                # show one of the subdir's art and looks weird
                showtopart = False
                id = self._idprefix + "$d" + str(thisdiridx)
                arturi = self._arturifordir(thisdiridx, thisdocidx)
                entries.append(direntry(id, pid, os.path.basename(nm), arturi=arturi))
            else:
                # Not a directory. docidx had better been set
                if thisdocidx == -1:
                    uplog("folders:docidx -1 for non-dir entry %s" % nm)
                    continue
                doc = self._docforidx(thisdocidx)

                id = self._idprefix + "$i" + str(thisdocidx)
                e = rcldoctoentry(id, pid, self._httphp, self._pprefix, doc)
                if e:
                    entries.append(e)

        if idx not in self._playlists:
            entries.sort(key=cmpentries)

        # Add "Browse subtree by tags" entry
        if (not self._notagview) and pid != self._idprefix and self._enabletags:
            arturi = None
            if showtopart:
                arturi = self._arturifordir(idx)
            id = pid + "$tagview.0"
            entries.insert(0, direntry(id, pid, ">> Tag View", arturi=arturi))

        return entries


    # Return path for objid, which has to be a container.This is good old
    # pwd... It is called from the search module for generating a 'dir:'
    # recoll filtering directive.
    def dirpath(self, objid, diridx=None):
        # We may get called from search, on the top dir (above [folders]). Return empty in this case
        if not diridx:
            try:
                isitem, diridx, pthremain = self._objidtoidx(objid)
            except:
                return ""
            if isitem:
                raise Exception(f"uprclfolders:dirpath: called on item pid [{pid}]")

        if diridx == 0:
            return "/"

        lpath = []
        while True:
            fathidx = self._dirvec[diridx][".."][0]
            found = False
            for nm, ids in self._dirvec[fathidx].items():
                if ids[0] == diridx:
                    lpath.append(nm)
                    found = True
                    break
            # End for
            if not found:
                uplog(f"uprclfolders: pwd failed for {objid} (father not found), returning /")
                return "/"
            if len(lpath) > 200:
                uplog(f"uprclfolders: pwd failed for {objid} (looping), returning /")
                return "/"

            diridx = fathidx
            if diridx == 0:
                break

        if not lpath:
            path = "/"
        else:
            path = ""
        for elt in reversed(lpath):
            path += elt + "/"

        return path


    # Compute object id for doc out of recoll search. Not used at the
    # moment, and _xid2idx is not built.
    def _objidforxdocid(self, doc):
        if doc["xdocid"] not in self._xid2idx:
            return None
        return self._idprefix + "$i" + str(self._xid2idx[doc["xdocid"]])


    # Given a doc, we walk its url down from the part in root to find its directory entry, and
    # return the _dirvec and _rcldocs indices it holds, either of which can be -1
    def _stat(self, doc, verbose=False):
        # _pathbeyond... returns the _dirvec index of the topdirs entry in root that we start from
        # and the split rest of path.  That is if the doc url has /av/mp3/classique/bach/ and the
        # root entry is /av/mp3, we get the _dirvec entry index for /av/mp3 and [classique, bach]
        fathidx, pathl = self._pathbeyondtopdirs(doc)
        if verbose:
            uplog(f"_stat: pbtd returns fathidx {fathidx} pathl {pathl}")
        if not fathidx:
            return -1, -1
        docidx = -1
        for elt in pathl:
            if not elt in self._dirvec[fathidx]:
                if verbose:
                    uplog(f"_stat: element [{elt}] has no entry in {fathidx} "
                          "[{self._dirvec[fathidx]}.keys()]")
                return -1, -1
            if verbose:
                uplog(f"_stat: element [{elt}] entry in {fathidx} [{self._dirvec[fathidx][elt]}]")
            fathidx, docidx = self._dirvec[fathidx][elt]

        return fathidx, docidx


    # Only works for directories but we do not check. Caller beware.
    def _objidforpath(self, doc):
        fathidx, docidx = self._stat(doc)
        return self._idprefix + "$d" + str(fathidx)


    def objidfordoc(self, doc):
        id = None
        if doc["mtype"] == "inode/directory":
            id = self._objidforpath(doc)
        else:
            # Note: we should have something like objidforxdocid (above) for using consistent
            # objids, but it's not currently doing anything, see method comments. Use unique but
            # different id instead for now.
            # id = self._objidforxdocid(doc)
            id = self._idprefix + "$xdocid" + doc.xdocid
        # uplog(f"objidfordoc: returning {id}")
        return id

    ##########################
    # Find cover art for doc
    #
    # We return a special uri if the file has embedded image data, else an
    # uri for for the directory cover art (if any).

    # Track-specific art.
    def _trackarturi(self, doc, objpath):
        # Check for an image specific to the track file
        base, ext = os.path.splitext(objpath)
        for artpath in _artnamegen(base):
            #uplog(f"_trackarturi: checking existence:[{artpath}]")
            artdoc = dict();
            artdoc["url"] = "file://" + artpath
            artdoc["group"] = None
            fathidx,docidx = self._stat(artdoc)
            if docidx >= 0:
                return uprclutils.httpurl(self._httphp, os.path.join(self._pprefix, artpath))

        # Else try to use an embedded img
        if doc["embdimg"]:
            arturi = uprclutils.embdimgurl(doc, self._httphp, self._pprefix)
            if arturi:
                # uplog("docarturi: embedded: %s"%printable(arturi))
                return arturi
        return None


    # Return folder-level art uri (e.g. /path/to/folder.jpg) if it exists
    def _folderart(self, doc, albtitle=None):
        # If doc is a directory, this returns it own path, else the father path
        folderpath = uprclutils.docfolder(doc)
        dirdoc = {}
        dirdoc["url"] = "file://" + folderpath
        dirdoc["group"] = None
        folderidx,_ = self._stat(dirdoc)
        if folderidx < 0:
            uplog(f"_folderart: folder not found: {folderpath}")
            return None
        foldercontents = self._dirvec[folderidx].keys()
        #uplog(f"_folderart: path [{folderpath}] idx {folderidx} contents [{foldercontents}]")

        # If albtitle is set check for an image of the same name
        if albtitle:
            for fsimple in _artnamegen(albtitle):
                if fsimple in foldercontents:
                    return uprclutils.httpurl(self._httphp, os.path.join(folderpath, fsimple))

        # Look for an appropriate image in the file folder. Generating the charcase combinations
        # would be complicated so we list the folder and look for a case-insensitive match. 
        arturi = None
        for f in foldercontents:
            flowersimple = f.lower()
            if flowersimple in _folderartnames:
                path = os.path.join(self._pprefix, folderpath, f)
                arturi = uprclutils.httpurl(self._httphp, path)
                break

        #uplog(f"folderart: returning {arturi}")
        return arturi


    def docarturi(self, doc, preferfolder=False, albtitle=None):
        objpath = doc["url"][7:]
        #uplog(f"docarturi: preferfolder {preferfolder} docpath {objpath}")
        
        if not preferfolder:
            arturi = self._trackarturi(doc, objpath)
            if arturi:
                return arturi
        
        # won't work for the virtual group directory itself: it has no doc
        if doc["group"]:
            base = os.path.join(os.path.dirname(objpath), uprclutils.tag2fn(doc["group"]))
            for artpath in _artnamegen(base):
                #uplog(f"docarturi:calling os.path.exist({artpath})")
                if os.path.exists(artpath):
                    return uprclutils.httpurl(self._httphp, os.path.join(self._pprefix, artpath))
        
        # TBD Here minimserver would look for album disc before album art (which is taken care of by
        # _folderart() with albtitle set)
        # Look for folder level image file (e.g. cover.jpg)
        arturi = self._folderart(doc, albtitle)
        if arturi:
            return arturi
        
        # If preferfolder is set, we did not look at the track-specific art, do it last.
        if preferfolder:
            arturi = self._trackarturi(doc, objpath)
        
        return arturi
