/*  job_finddiagramshiftsalt.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_finddiagramshiftsalt.h"
#include "functions.h"
#include "ginacutils.h"
#include "yamlutils.h"
#include "files.h"
#include "diagram.h"
#include "graph.h"
#include "integralfamily.h"
#include "sectormappings.h"
#include "int.h"

using namespace std;

namespace Reduze {

// register job type at JobFactory
namespace {
JobProxy<FindDiagramShiftsAlt> dummy;
}

void FindDiagramShiftsAlt::run_serial() {
	using GiNaC::exmap;
	using GiNaC::lst;

	list<Diagram> alldias;
	read_diagrams(alldias, qgraf_filename_, names_);

	string diag_tmp_filename = output_filename_ + ".tmp";
	ofstream os_diag(diag_tmp_filename.c_str());
	if (!os_diag)
		ABORT("Can't open file " << diag_tmp_filename);

	bool write_info_file = !info_filename_form_.empty();
	string info_tmp_filename = info_filename_form_ + ".tmp";
	ofstream os_info;
	if (write_info_file)
		os_info.open(info_tmp_filename.c_str());

	set<Sector> secs; // direct target sectors
	set<Sector> minsecs; // minimal target sectors (use crossings, relations)
	size_t num_matched = 0;
	// status:
	// 'O' direct match to zero sector
	// 'X' direct match to non-zero sector
	// 'o' match via crossed diagram with zero sector
	// 'x' match via crossed diagram with non-zero sector
	// '.' no match, no crossed partner
	map<string, char> status_of_dianame;
	map<int, list<string> > dias_by_t; // dias indexed by number of propagators
	map<string, lst> loop_moms_by_dianame; // loop momenta of the diagram
	int col = 0;
	LOGN("\nAnalyzing topologies of " << alldias.size() << " diagrams:");
	for (list<Diagram>::iterator d = alldias.begin(); d != alldias.end(); ++d) {
		Diagram& dia = *d;
		LOGX("\nAnalyzing diagram " << dia.name());
		if ((col++) % 50 == 0)
			LOG("");
		Graph graph = dia.get_Graph();
		dias_by_t[graph.num_loop_edges()].push_back(dia.name());
		loop_moms_by_dianame[dia.name()] = dia.loop_momenta();
		map<int, map<Sector, exmap> > shifts;
		list<IntegralFamily*> l = Files::instance()->integralfamilies();
		list<IntegralFamily*>::iterator ic;
		// try basic integral families
		for (ic = l.begin(); ic != l.end(); ++ic)
			if (!(*ic)->is_crossed() && graph.match_to_integralfamily(*ic))
				break;
		// try crossed integral families if no match yet
		if (ic == l.end())
			for (ic = l.begin(); ic != l.end(); ++ic)
				if ((*ic)->is_crossed() && graph.match_to_integralfamily(*ic))
					break;

		if (ic == l.end()) {
			status_of_dianame[dia.name()] = '.';
			LOGN(status_of_dianame[dia.name()]);
		} else {
			const pair<const Sector, exmap>* match = graph.shift_to_sector();
			Sector sec = match->first;
			exmap shift = match->second;
			LOGX("  => matched diagram " << dia.name() << " to " << sec);
			LOGX("     via shift: " << shift);
			Sector minsec = sec;
			const IntegralFamily* secic = sec.integralfamily();
			if (secic->is_crossed()) {
				const CrossedIntegralFamily* cic =
						dynamic_cast<const CrossedIntegralFamily*> (secic);
				minsec = Sector(cic->source_integralfamily(), sec.id());
				minsec = SectorMappings::get_shifted_sector(minsec);
				status_of_dianame[dia.name()]
						= (SectorMappings::is_zero(minsec) ? 'o' : 'x');
				LOGX("     target sector equivalent to " <<
						cic->crossing().transform(minsec) << "\n");
			} else {
				minsec = SectorMappings::get_shifted_sector(minsec);
				status_of_dianame[dia.name()]
						= (SectorMappings::is_zero(minsec) ? 'O' : 'X');
				LOGX("     target sector equivalent to " << minsec << "\n");
			}
			LOGN(status_of_dianame[dia.name()]);
			GiNaC::lst new_syms = sec.integralfamily()->loop_momenta();
			dia.substitute_momenta(shift);
			dia.set_loop_momenta(new_syms);
			dia.set_sector(sec);
			// write to info file
			os_info << "id DiaMatch(" << dia.name() << ") = " //
					<< "Sector(" << sec.integralfamily()->name()//
					<< ", " << sec.t()//
					<< ", " << sec.id() << ")"//
					<< " * Shift(";
			for (exmap::const_iterator i = shift.begin(); i != shift.end();) {
				os_info << i->first << ", " << i->second << ", []";
				if (++i != shift.end())
					os_info << ", ";
			}
			os_info << ");" << endl;
			// remember matches
			secs.insert(sec);
			minsecs.insert(minsec);
			++num_matched;
		}
	}
	if (write_info_file) {
		os_info.close();
		rename(info_tmp_filename, info_filename_form_);
	}

	list<Diagram>::iterator dia;

	// write out the new diagram file
	LOG("\n\nWriting new diagram file: " << output_filename_);
	for (dia = alldias.begin(); dia != alldias.end(); ++dia) {
		YAML::Emitter ye;
		ye << YAML::BeginMap;
		ye << YAML::Key << "diagram" << YAML::Value << *dia;
		ye << YAML::EndMap;
		os_diag << ye.c_str() << endl;
	}
	//for (dia = alldias.begin(); dia != alldias.end(); ++dia)
	//	dia->write_stream(os_diag);
	os_diag.close();
	rename(diag_tmp_filename, output_filename_);

	LOG("\nMatched diagrams: " << num_matched);

	map<int, list<string> >::const_iterator it;
	list<string>::const_iterator n;

	LOGN("\nUnmatched diagrams: " << alldias.size() - num_matched);
	size_t num_unmatched = 0;
	for (it = dias_by_t.begin(); it != dias_by_t.end(); ++it) {
		bool first = true;
		col = 0;
		for (n = it->second.begin(); n != it->second.end(); ++n)
			if (status_of_dianame[*n] == '.') {
				if (first)
					LOGN("\nt = " << it->first << " : ");
				++num_unmatched;
				if ((++col) % 10 == 0)
					LOGN("\n        ");
				LOGN(*n << " ");
				first = false;
			}
	}
	VERIFY(num_matched + num_unmatched == alldias.size());
	LOG("");

	LOGN("\nSectors: " << secs.size());
	col = 0;
	int last_t = -1;
	const IntegralFamily* last_ic = 0;
	for (set<Sector>::const_iterator s = secs.begin(); s != secs.end(); ++s) {
		if (last_ic == 0 || *s->integralfamily() != *last_ic) {
			LOGN("\nfamily " << s->integralfamily()->name() << ":");
			last_ic = s->integralfamily();
			last_t = -1;
		}
		if (s->t() != last_t)
			LOGN("\nt = " << s->t() << ": ");
		last_t = s->t();
		LOGN(s->id() << " ");
	}
	LOG("");

	SectorSelection sectree =
			SectorSelection::find_compact_recursive_selection(minsecs, true);
	YAML::Emitter ye;
	ye << YAML::BeginMap << YAML::Key << "sector_selection" << YAML::Value
			<< sectree << YAML::EndMap;
	LOG("\nThis selection of sectors should be reduced for the matched diagrams:");
	LOG(ye.c_str());
}

bool FindDiagramShiftsAlt::find_dependencies(const set<string>& outothers,
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	//find_dependencies_all_sectormappings(outothers, in, auxjobs);
	in.push_back(qgraf_filename_);
	out.push_back(output_filename_);
	if (!info_filename_form_.empty())
		out.push_back(info_filename_form_);
	if (Files::instance()->integralfamilies().empty())
		ABORT("Can't match any diagrams since no integral families defined");
	return true;
}

std::string FindDiagramShiftsAlt::get_description() const {
	return "find diagram shifts (alt) for " + short_filename(qgraf_filename_);
}

}
