/*  job_findfiniteintegrals.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_findfiniteintegrals.h"
#include "functions.h"
#include "int.h"
#include "filedata.h"
#include "equation.h"
#include <ginac/ginac.h>
#include "ginacutils.h"
#include "sector.h"
#include "integralfamily.h"
#include "files.h"
#include "globalsymbols.h"
#include "kinematics.h"

using namespace std;
using namespace GiNaC;

namespace Reduze {

namespace {
JobProxy<FindFiniteIntegrals> dummy;
}

bool FindFiniteIntegrals::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	in.push_back(input_file_);
	out.push_back(output_file_);
	return true;
}

void FindFiniteIntegrals::init() {
}

std::string FindFiniteIntegrals::get_description() const {
	return "find finite integrals";
}

class JKSet {
public:
	JKSet(unsigned j, unsigned k, unsigned nedges) : j_(j), k_(k) {
		VERIFY(is_valid_set(j,k,nedges));
	}
	inline unsigned j() const { return j_; }
	inline unsigned k() const { return k_; }
	inline bool is_in_J(unsigned p) const { return (1 << p) & j_; }
	inline bool is_in_K(unsigned p) const { return (1 << p) & k_; }
	static inline bool is_valid_set(unsigned j, unsigned k, unsigned nedges) {
		// J and K must be disjoint sets, their union must not be maximal
		unsigned max = 1 << nedges;
		return (/*j < (1 << nedges) && k < (1 << nedges) &&*/
				(j & k) == 0 && (j | k) > 0 && (j | k) < max - 1);
	}
	static list<JKSet> generate_all_sets(unsigned nedges) {
		list<JKSet> res;
		unsigned max = 1 << nedges;
//		unsigned k = 0;
		for (unsigned k = 0; k < max; ++k) {
			for (unsigned j = 0; j < max; ++j) {
				if (is_valid_set(j, k, nedges)) {
					res.push_back(JKSet(j,k,nedges));
				}
			}
		}
		return res;
	}
private:
	// we denote n-tuples of Feynman parameters to scale by an ID j similar to sector ID:
	// j = sum_{i=1}^t if propagator i scales with lambda then 2^(i-1) else 0
	// same story for k
	unsigned j_, k_;
};

// returns whether integral is quasi-finite (for omegalimit = 0)
// or whether it is convergent enough for the chosen omegalimit > 0
bool is_quasi_finite(const INT& i, const vector<pair<JKSet, pair<int, int> > >& ufscale,
		int dim, int omegalimit) {
	const IntegralFamily* fam = i.integralfamily();
    Sector sec = i.get_sector();
    int l = fam->loop_momenta().nops();
    int r = i.r();
	int uexp = r - (dim / 2) * (l + 1);
	int fexp = -r + (dim / 2) * l;
//	bool ksignal = false;
	for (unsigned m = 0; m < ufscale.size(); ++m) {
		const JKSet& jk = ufscale[m].first;
//		if (jk.k() != 0 && !ksignal) {
//			LOG("check also K for " << i);
//			ksignal = true;
//		}
		const pair<int, int>& uf = ufscale[m].second;
		int omegajk = uf.first * uexp + uf.second * fexp;
		unsigned p = 0;
		for (unsigned n = 0; n < fam->num_propagators(); ++n) {
			if (sec.id() & (1 << n)) { // belongs to topology
				if (jk.is_in_J(p)) {
					omegajk += i.v_i(n) - 1; // for additional dots
					++omegajk; // for size(J) summand (measure)
				} else if (jk.is_in_K(p)) {
					omegajk -= i.v_i(n) - 1; // for additional dots
					--omegajk; // for size(K) summand (measure)
				}
				++p;
			}
		}
		if (omegajk <= omegalimit) { // not quasi-finite
			if (omegalimit == 0 && jk.k() > 0)
				LOG("found divergences at infinity");
			return false;
		}
	}
//	if (ksignal)
//		LOG("extended check done");
	return true;
}

vector<pair<JKSet, pair<int, int> > >
calculate_UF_scalings(ex u, ex f, lst xi, unsigned t) {
//cout << "prep scalings" << endl;
//int bla=0;
	list<JKSet> sets = JKSet::generate_all_sets(t);
	LOGX("prepare scalings for " << sets.size() << " possible subdivergences");
	vector<pair<JKSet, pair<int, int> > > res;
	res.reserve(sets.size());
	symbol lam("lam");
	list<JKSet>::const_iterator jk;
	for (jk = sets.begin(); jk != sets.end(); ++jk) {
		exmap scale;
		unsigned numj = 0;
		for (unsigned p = 0; p < t; ++p) {
			if (jk->is_in_J(p)) {
				scale[xi.op(p)] = xi.op(p) * lam;
			} else if (jk->is_in_K(p)) {
				scale[xi.op(p)] = xi.op(p) / lam;
				++numj;
			}
		}
		// in typical applications, the following two lines take the full time
		// of a search (actual candidate tests later on don't matter)
		ex uln = normal(u.subs(scale,subs_options::no_pattern)).ldegree(lam);
		ex fln = normal(f.subs(scale,subs_options::no_pattern)).ldegree(lam);
//		cout << "{ " << ++bla << ", " << uln  << ", " << u.subs(scale,subs_options::no_pattern) << "}," << endl;
//		cout << "{ " << ++bla << ", " << fln  << ", " << f.subs(scale,subs_options::no_pattern) << "}," << endl;
		VERIFY(uln.info(info_flags::integer) && fln.info(info_flags::integer));
		int uldeg = ex_to<numeric>(uln).to_int();
		int fldeg = ex_to<numeric>(fln).to_int();
		pair<int, int> ufdeg = make_pair(uldeg, fldeg);
		res.push_back(make_pair(*jk, ufdeg));
	}
	return res;
}

void FindFiniteIntegrals::run_serial() {
	LOG("Reading integrals from file \"" << input_file_ << '\"');
	InFileINTs in(input_file_);
	set<INT> ints;
	in.get_all(ints);
	in.close();
	LOG("Writing quasi-finite integrals to file \"" << output_file_ << '\"');
	OutFileINTs out(output_file_);
	set<Sector> secs;
	for (set<INT>::const_iterator i = ints.begin(); i != ints.end(); ++i)
		secs.insert(i->get_sector());
	LOG("Analyzing " << secs.size() << " sectors (from " << ints.size() << " integrals)\n");
	int maxt = 0;
	for (set<INT>::const_iterator i = ints.begin(); i != ints.end(); ++i)
		maxt = max(maxt, (int)i->t());
	lst xi = create_symbols("x", maxt);
	unsigned ntot = 0;
	for (set<Sector>::const_iterator sec = secs.begin(); sec != secs.end(); ++sec) {
		++ntot;
		LOG("\n\n" << ntot << ")  sector " << *sec << ": ");
		const IntegralFamily* fam = sec->integralfamily();
		ex u, f;
		sec->find_UF_polynomials(xi, u, f);
		//LOG(" U = " << u);
		//LOG(" F = " << f);
		unsigned t = sec->t();
		// prepare scaling of default integrand (default dim, no dots) for each element in (J,K)
		LOGX("prepare scalings");
		vector<pair<JKSet, pair<int,int> > > ufscale = calculate_UF_scalings(u, f, xi, t);
		LOGX("done prepare scalings");
		SeedGeneratorOptions sgopts;
		sgopts.set_reduce_to_equivalent(true);
		SeedGenerator seedgenerator(fam, sgopts);
		// search for quasi-finite integrals (need several per sector in general)
		int ntogo = num_candidates_;
		const GiNaC::symbol& d = Files::instance()->globalsymbols()->d();
		GiNaC::ex ds = fam->kinematics()->dimension().subs(d == reference_dimension_);
		VERIFY(ds.info(info_flags::integer));
		int dimsource = ex_to<numeric>(ds).to_int();
		for (unsigned ndd = 0; ntogo > 0 && ndd < 1000; ++ndd) {
			for (int shift = 2 * ndd - 2; ntogo > 0 && shift >= 0; shift -= 2) {
				int dim = dimsource + shift;
				unsigned numdots = ndd - shift/2 - 1;
				int r = t + numdots;
				LOGNX("  dim = " << dim << ", dots = " << numdots);
				int l = fam->loop_momenta().nops();
				int gammaarg = r - l*dim/2; // divergence from prefactor ?
				if (require_finiteness_ && gammaarg <= 0)
					continue;
				set<INT> cands;
				seedgenerator.generate(sec->id(), r, 0, cands);
				LOGX(", nseeds = " << cands.size());
				set<INT>::const_iterator c;
				for (c = cands.begin(); ntogo > 0 && c != cands.end(); ++c) {
					if (is_quasi_finite(*c, ufscale, dim, min_omega_)) {
						--ntogo;
						LOG("=>  dim = " << dim << ", dots = " << numdots << ", integral = " << *c);
						string famname = Kinematics::dimension_shifted_label(
								c->integralfamily()->name(), shift);
						out << INT(Files::instance()->integralfamily(famname), c->v());
					}
				}
			}
		}
	}
	out.finalize();
}

}
