/*  job_reducefiles.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef JOB_REDUCEFILES_H_
#define JOB_REDUCEFILES_H_

#include "job.h"
#include "reduzer.h"

namespace Reduze {

/// Job to reduce files
class ReduceFiles: public Reduze::Job {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("reduce_files");
		s.set_short_description("Reduces identities in given files.");
		s.set_long_description(""//
					"Job to reduce identities in given files.");
		s.add_option("equation_files", true, "sequence of strings",
				"Names of files containing the equations to be reduced.");
		s.add_option("substitution_files", false, "sequence of strings",
				"Names of files containing equations to be used for substitutions only.");
		s.add_option("preferred_masters_file", false, "string",
				"Input file with preferred master integrals.");
		s.add_option("output_file", true, "string", "Output file name.");
		s.add_option("reduzer_options", false, "reduzer_options", ""//
					"Options for the reduction algorithm.");
#ifdef HAVE_MPI
		s.add_option("use_full_parallelization", false, "boolean",//
				"Whether to reduce the files in a distributed"
				" way employing worker processes. "
				" A distributed reduction job requires one MPI process"
				" for the manager of the job plus at least one MPI process"
				" for a worker (in addition to the MPI process"
				" for the JobCenter).");
		s.add_option("min_workers", false, "integer",
				" Only relevant if use_full_parallelization is true."
				" Minimum number of worker processes, must be 1 or higher.");
		s.add_option("max_workers", false, "integer",
				" Only relevant if use_full_parallelization is true."
				" Maximum number of worker processes, must be 1 or higher.");
#endif
		s.add_options(Job::yaml_spec());
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	ReduceFiles()
#ifdef HAVE_MPI
	: use_full_parallelization_(true), min_workers_(1), max_workers_(1000)
#endif
	{
		add_auto_options();
	}

	virtual ~ReduceFiles() {
	}
	virtual void run_serial();
	virtual bool find_dependencies(const std::set<std::string>&,//
			std::list<std::string>&, std::list<std::string>&, std::list<Job*>&);
	virtual std::string get_description() const;

protected:
	virtual void add_auto_options() {
		add_auto_io("equation_files", equation_files_);
		add_auto_io("substitution_files", substitution_files_);
		add_auto_io("preferred_masters_file", preferred_masters_file_);
		add_auto_io("output_file", output_file_);
		add_auto_io("reduzer_options", reduzer_options_);
#ifdef HAVE_MPI
		add_auto_io("use_full_parallelization", use_full_parallelization_);
		add_auto_io("min_workers", min_workers_);
		add_auto_io("max_workers", max_workers_);
#endif
	}
	virtual void init();
	std::string get_tmp_directory() const;

private:
	std::list<std::string> equation_files_, substitution_files_;
	std::string preferred_masters_file_;
	std::string output_file_;
	ReduzerOptions reduzer_options_;
#ifdef HAVE_MPI
	bool use_full_parallelization_;
	int min_workers_, max_workers_;
#endif
};

}

#endif /* JOB_REDUCEFILES_H_ */
