/*  job_runreduction.cpp
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#include "job_runreduction.h"
#include "sectormappings.h"
#include "files.h"
#include "filedata.h"
#include "reduzer.h"
#include "yamlutils.h"
#include "job_printreductioninfo.h"

#ifdef HAVE_MPI
#include "reduzermpi.h"
#endif

using namespace std;

namespace Reduze {

// register job types at JobFactory
namespace {
JobProxy<RunReduction> dummy;
JobProxy<PrintReductionInfoFile> dummy2;
}

RunReduction::RunReduction() {
	add_auto_options();
}

RunReduction::RunReduction(const Sector& sector, //
		const std::string& tmp_directory, //
		const std::list<std::string>& identity_filenames, //
		const std::list<std::string>& subst_filenames, //
		const std::string& output_file, //
		const std::string& output_file_external, //
		const ReductionGenericOptions& reduction_generic_options, //
		const ReduzerOptions& reduzer_options, //
		const std::string preferred_masters_file, //
		const std::list<std::string>& delete_files, //
		const std::list<std::string>& delete_directories) :
	tmp_directory_(tmp_directory), /**/
	identity_filenames_(identity_filenames), /**/
	subst_filenames_(subst_filenames), /**/
	output_file_(output_file), /**/
	output_file_external_(output_file_external), /**/
	reduction_generic_options_(reduction_generic_options), /**/
	reduzer_options_(reduzer_options), /**/
	preferred_masters_file_(preferred_masters_file), /**/
	delete_files_(delete_files), /**/
	delete_directories_(delete_directories), /**/
	subst_subsectors_(true) {
	sector_.push_back(sector);
	reduzer_options_.add_auto_options();
	add_auto_options();
}

RunReduction::RunReduction(const std::string& tmp_directory, //
		const std::list<std::string>& identity_filenames, //
		const std::list<std::string>& subst_filenames,//
		const std::string& output_file, //
		const ReduzerOptions& reduzer_options, //
		const std::string preferred_masters_file, //
		const std::list<std::string>& delete_files, //
		const std::list<std::string>& delete_directories) :
	tmp_directory_(tmp_directory), /**/
	identity_filenames_(identity_filenames), /**/
	subst_filenames_(subst_filenames), /**/
	output_file_(output_file), /**/
	reduzer_options_(reduzer_options), /**/
	preferred_masters_file_(preferred_masters_file), /**/
	delete_files_(delete_files), /**/
	delete_directories_(delete_directories), /**/
	subst_subsectors_(true) {
	reduzer_options_.add_auto_options();
	add_auto_options();
}

RunReduction::~RunReduction() {
}

void RunReduction::run_serial() {
#ifdef HAVE_DATABASE
	if (reduzer_options_.use_database) {
		ReduzerDb r(reduzer_options_);
		compute(&r);
	} else
#endif // HAVE_DATABASE
	{
		ReduzerMem r(reduzer_options_);
		compute(&r);
	}
}

#ifdef HAVE_MPI
void RunReduction::run_manager(MPI::Intracomm* comm, int jobcenter_rank) {
#ifdef HAVE_DATABASE
	if (reduzer_options_.use_database) {
		ReduzerDbMPIRoot r(reduzer_options_, comm, jobcenter_rank);
		compute(&r);
	} else
#endif // HAVE_DATABASE
	{
		ReduzerMemMPIRoot r(reduzer_options_, comm, jobcenter_rank);
		compute(&r);
	}
}

void RunReduction::run_worker(MPI::Intracomm* comm, int manager_rank) {
	ReduzerMPILeave l(comm, manager_rank);
	l.run();
}
#endif // HAVE_MPI
//

void RunReduction::compute(ReduzerMem* r) {
	ASSERT(r != 0);
	Timer timer;
	make_directory(tmp_directory_);
	// remove any previous database
	remove_invalid_database(tmp_directory_, true);

	LOG("Setting up in-memory reduction");
	LOGX("Determining integrals: all_ints, external_ints, helper_ints, critical_ints, discard_ints");
	set<INT> all_ints, external_ints, helper_ints, critical_ints, discard_ints;
	unord_set_INT identity_ints;
	find_integrals_and_substitutions(all_ints, identity_ints, external_ints,
			helper_ints, critical_ints, discard_ints);
    LOG("Initialize reduzer");
	r->setup();
	r->load(identity_filenames_, subst_filenames_, all_ints, identity_ints,
			external_ints, helper_ints, critical_ints, discard_ints);
	LOG("Setup of reduction done " << timer.get_wall_time_nice_string());
	double time_setup = timer.restart();

	string sec_string;
	if (!sector_.empty())
		sec_string = " for sector " + to_string(sector_.back());
	LOG("Running Reduction" << sec_string);
	r->print_status();
	r->run();
	r->print_status();
	r->write_result(output_file_, output_file_external_);
	r->clear();
	LOG("Reduction done " << timer.get_wall_time_nice_string() << "\n");
	double time_reduction = timer.restart();

	LOG("Cleaning up");
	remove(delete_files_);
	remove_empty_dir(delete_directories_);
	remove_empty_dir(tmp_directory_);

	LOG("Analyzing reduction result");
	PrintReductionInfoFile info_job;
	info_job.input_file = output_file_;
	info_job.run_serial();
	LOG("Check for masters done " << timer.get_wall_time_nice_string() << "\n");
	double time_write = timer.get_wall_time();

	double time_total = time_setup + time_reduction + time_write;
	double nt = (time_total > 0 ? 100. / time_total : 0.);
	LOG("Done [" << fixed << setprecision(0)
			<< time_total << " s]" << resetiosflags(ios::fixed));
	LOGX(fixed << setprecision(0)
			<< "Setup:         " << time_setup << " s  (" << time_setup*nt << " %)\n"
			<< "Reduction:     " << time_reduction << " s  (" << time_reduction*nt << " %)\n"
			<< "Write/analyze: " << time_write << " s  (" << time_write*nt << " %)\n"
			<< resetiosflags(ios::fixed));

}

#ifdef HAVE_DATABASE
void RunReduction::compute(ReduzerDb* r) {
	ASSERT(r != 0);
	Timer timer;
	make_directory(tmp_directory_);
	// Check for valid aborted run (with transactions turned on and not aborted during setup)
	bool resume_reduction = RunReduction::resume_reduction(tmp_directory_,
			reduzer_options_.use_transactions, is_conditional());
	remove_invalid_database(tmp_directory_, resume_reduction);
	string db_file = db_file_name();
	string db_home = db_dir();

	if (resume_reduction) {
		LOG("Opening existing database");
		LOGX("Found valid aborted run");
		LOG("Resuming reduction");
		r->open(db_home, db_file);
		LOG("Open database done " << timer.get_wall_time_nice_string());
	} else {
		LOG("Setting up database");
		LOGX("No valid aborted run found");
		LOGX("Determining integrals: all_ints, external_ints, helper_ints, critical_ints, discard_ints");
		set<INT> all_ints, external_ints, helper_ints, critical_ints, discard_ints;
		unord_set_INT identity_ints;
		find_integrals_and_substitutions(all_ints, identity_ints, external_ints,
				helper_ints, critical_ints, discard_ints);
	    LOG("Initialize reduzer");
		make_directory(db_home);
		string db_file_tmp = db_file + ".tmp";
		r->setup(db_home, db_file_tmp);
		r->load(identity_filenames_, subst_filenames_, all_ints, identity_ints, external_ints,
				helper_ints, critical_ints, discard_ints);
		r->dbrename(db_file_tmp, db_file);
		LOG("Setup database done " << timer.get_wall_time_nice_string());
	}
	double time_setup = timer.restart();

	bool remove_identities = true && reduzer_options_.use_transactions;
	if (remove_identities) {
		remove(delete_files_);
		remove_empty_dir(delete_directories_);
	}

	string sec_string;
	if (!sector_.empty()) {
		sec_string = " for sector " + to_string(sector_.back());
	}
	LOG("Running Reduction" << sec_string);
	r->print_status();
	r->run();
	r->print_status();
	r->write_result(output_file_, output_file_external_);
	r->clear();
	LOG("Reduction done " << timer.get_wall_time_nice_string() << "\n");
	double time_reduction = timer.restart();

	LOG("Cleaning up");
	remove_identities = true && !reduzer_options_.use_transactions;
	if (remove_identities) {
		remove(delete_files_);
		remove_empty_dir(delete_directories_);
	}
	remove_directory_with_files(db_home);// remove database
	remove_empty_dir(tmp_directory_);

	LOG("Analyzing reduction result");
	PrintReductionInfoFile info_job;
	info_job.input_file = output_file_;
	info_job.run_serial();
	LOG("Check for masters done " << timer.get_wall_time_nice_string() << "\n");
	double time_write = timer.get_wall_time();

	double time_total = time_setup + time_reduction + time_write;
	double nt = (time_total > 0 ? 100. / time_total : 0.);
	LOG("Done [" << fixed << setprecision(0)
			<< time_total << " s]" << resetiosflags(ios::fixed));
	LOGX(fixed << setprecision(0)
			<< "Setup:         " << time_setup << " s  (" << time_setup*nt << " %)\n"
			<< "Reduction:     " << time_reduction << " s  (" << time_reduction*nt << " %)\n"
			<< "Write/analyze: " << time_write << " s  (" << time_write*nt << " %)\n"
			<< resetiosflags(ios::fixed));

}
#endif // HAVE_DATABASE
bool RunReduction::find_dependencies(const set<string>& outothers,//
		list<string>& in, list<string>& out, list<Job*>& auxjobs) {
	out.push_back(output_file_);

#ifdef HAVE_DATABASE
	bool resume_reduction = RunReduction::resume_reduction(tmp_directory_,
			reduzer_options_.use_transactions, is_conditional());
#else // HAVE_DATABASE
	bool resume_reduction = false;
#endif // HAVE_DATABASE
	if (resume_reduction
			|| (is_conditional() && is_readable_file(output_file_)))
		return true;

	in.insert(in.end(), subst_filenames_.begin(), subst_filenames_.end());
	if (!sector_.empty() && !reduction_generic_options_.set_subsectors_to_zero_) {
	    // for dependencies assume all sub-sector reduction is needed
		// (at run-time it will be decided which of these need to be loaded)
		set<Sector> sub = SectorSelection::find_prerequisite_sectors(
				sector_.front(), true);
		for (set<Sector>::const_iterator s = sub.begin(); s != sub.end(); ++s) {
			const string fn = Files::instance()->get_filename_sectorreduction(*s);
			// add to be generated (by reduction) or already available sub-sector results
			if (outothers.find(get_canonical_filename(fn)) != outothers.end()
					|| is_readable_file(fn)) {
				in.push_back(fn);
			}
		}
	}
	in.insert(in.end(), identity_filenames_.begin(), identity_filenames_.end());
	if (!preferred_masters_file_.empty())
		in.push_back(preferred_masters_file_);
	return true;
}

std::string RunReduction::get_description() const {
	ostringstream ss;
	ss << "run reduction";
	if (!sector_.empty()) {
		ss << " for sector " << sector_.front();
	} else if (!identity_filenames_.empty()) {
		ss << " of files " << short_filename(identity_filenames_.front())
				<< " + ...";
	} else {
		ss << " of nothing";
	}
	return ss.str();
}

void RunReduction::print_manual_options(YAML::Emitter& os) const {
	using namespace YAML;
	os << Key << "sector" << Value;
	if (!sector_.empty())
		os << sector_.front();
	else
		// print empty list (for doc)
		os << sector_;
}

void RunReduction::read_manual_options(const YAML::Node& node) {
	if (node.FindValue("sector")) {
		try {
			Sector s(node["sector"]);
			sector_.push_back(s);
		} catch (exception& e) {
			sector_.clear();
		}
	}
}

std::string RunReduction::db_file_name() { // static member
	return "database.db";
}

std::string RunReduction::db_dir_name(bool use_transactions) {// static member
	return (use_transactions ? "dbhome/" : "dbhome_tr_off/");
}

#ifdef HAVE_DATABASE
std::string RunReduction::db_dir() const {
	if (reduzer_options_.use_transactions)
	return tmp_directory_ + db_dir_name(true);
	else
	return tmp_directory_ + db_dir_name(false);
}
#endif

bool RunReduction::resume_reduction(const std::string& tmp_directory,
		bool use_transactions, bool conditional) {
	if (!use_transactions || !conditional)
		return false;
	string db_file = db_file_name();
	string db_home = tmp_directory + db_dir_name(use_transactions); // directory with transactions on
	if (FILE* f = fopen((db_home + db_file).c_str(), "r")) {
		fclose(f);
		return true;
	}
	return false;
}

void RunReduction::find_integrals_and_substitutions(std::set<INT>& all_ints,
		unord_set_INT& identity_ints, std::set<INT>& external_ints,
		std::set<INT>& helper_ints, std::set<INT>& critical_ints,
		std::set<INT>& discard_ints) {
	INT::load_preferred(preferred_masters_file_);

	unord_set_INT all;

	LOG("Find integrals in identities");
	list<string>::const_iterator f;
	Timer timer;
	for (f = identity_filenames_.begin(); f != identity_filenames_.end(); ++f) {
		LOGNX("  scanning " + *f);
		InFileLinearCombinations::find_INT(*f, identity_ints);
		LOGX(" (" << timer.get_wall_time() << "s)");
		timer.restart();
	}
	if (!sector_.empty() && subst_subsectors_) {
		LOG("Add optimal set of sub-sector substitution files");
		unord_set_INT::iterator i;
		list<Sector> secs;
		for (i = identity_ints.begin(); i != identity_ints.end(); ++i)
			if (i->get_sector() != sector_.front())
				secs.push_back(i->get_sector());
		secs.sort();
		secs.unique();
		for (list<Sector>::iterator s = secs.begin(); s != secs.end(); ++s) {
			string f = Files::instance()->get_filename_sectorreduction(*s);
			if (is_readable_file(f)) {
				LOGX("  using " << f);
				subst_filenames_.push_back(f);
			} else {
				LOG("  CAN'T FIND " << f);
			}
			f = Files::instance()->get_filename_sectorexternal(*s);
			if (is_readable_file(f)) {
				LOGX("  using " << f);
				subst_filenames_.push_back(f);
			}
		}
		string shared = Files::instance()->get_filename_sectorexternal_shared();
		if (is_readable_file(shared)) {
			LOGX("  using " << shared);
			subst_filenames_.push_back(shared);
		}
	}
	all.insert(identity_ints.begin(), identity_ints.end());
	LOG("Find integrals in substitutions");
	for (f = subst_filenames_.begin(); f != subst_filenames_.end(); ++f) {
		LOGNX("  scanning " + *f);
		InFileLinearCombinations::find_INT(*f, all);
		LOGX(" (" << timer.get_wall_time() << "s)");
		timer.restart();
	}
#ifdef USE_HASH_TABLE
	all_ints.insert(all.begin(), all.end());
#else
	all_ints.swap(all);
#endif

	if (sector_.empty()) {
		LOGX("No sector specified:");
		LOGX("  Setting critical_ints to all_ints");
		LOGX("  No external integrals");
		critical_ints = all_ints;
		return;
	}

	// for runs with sector specified
	const Sector& sector = sector_.front();
	int t = sector.t();

	LOGX("Find external and helper integrals");
	for (set<INT>::const_iterator i = all_ints.begin(); i != all_ints.end(); ++i) {
		if (i->get_sector() != sector)
			external_ints.insert(external_ints.end(), *i);
		if (i->get_sector() < sector) {
			if (reduction_generic_options_.discard_new_subsector_identities_)
				discard_ints.insert(discard_ints.end(), *i);
		}
		if (i->get_sector() > sector) {
			helper_ints.insert(helper_ints.end(), *i);
		} else {
			list<RSGenericSelection>::const_iterator rs;
			rs = reduction_generic_options_.requested_solutions_.begin();
			for (; rs != reduction_generic_options_.requested_solutions_.end(); ++rs)
				if (rs->contains(t, *i))
					break;
			if (rs == reduction_generic_options_.requested_solutions_.end())
				helper_ints.insert(helper_ints.end(), *i); // not requested, set to helper
		}
	}

	LOGX("Find critical integrals");
	try { // select border of requested solutions if those are finite
		// this might throw
		list<RSFiniteGenericSelection> fin_req_sol(
				reduction_generic_options_.requested_solutions_.begin(),
				reduction_generic_options_.requested_solutions_.end());
		LOG("Selecting border as critical integrals");
		set<RSPoint> crit_rs;
		RSFiniteGenericSelection::find_upper_border(fin_req_sol, t, crit_rs);
		for (set<INT>::const_iterator i = all_ints.begin(); i != all_ints.end(); ++i) {
			if (i->get_sector() != sector)
				continue;
			set<RSPoint>::const_iterator p = crit_rs.begin();
			for (; p != crit_rs.end(); ++p)
				if (i->r() == p->r() && i->s() == p->s())
					break;
			if (p != crit_rs.end())
				critical_ints.insert(critical_ints.end(), *i);
		}
	} catch (exception&) {
		LOGX("Setting critical_ints to all_ints");
		critical_ints = all_ints;
	}
}

void RunReduction::remove_invalid_database(const std::string& tmp_directory,
		bool resume_reduction) {
	LOGX("Checking for aborted previous run");
	string db_path_tr_off = tmp_directory + db_dir_name(false);
	string db_path_tr_on = tmp_directory + db_dir_name(true);
	// remove invalid database (no transactions)
	if (FILE* f = fopen(db_path_tr_off.c_str(), "r")) {
		LOG("Remove invalid aborted run (transactions turned off) from database");
		fclose(f);
		remove_directory_with_files(db_path_tr_off);
	}
	// delete previous aborted run with transactions on if not resuming a reduction
	if (!resume_reduction) {
		if (FILE * f = fopen(db_path_tr_on.c_str(), "r")) {
			LOG("Remove previous old database, which was set up with transactions on");
			fclose(f);
			remove_directory_with_files(db_path_tr_on);
		}
	}
}

} // namespace Reduze


