/*  reduzer.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef REDUZER_H_
#define REDUZER_H_

#include <iostream>
#include <set>
#include <map>
#include "equationlist.h"

#ifdef HAVE_DATABASE
#include <dbstl_map.h>
#include <dbstl_set.h>
#endif

#ifdef USE_HASH_TABLE
typedef std::tr1::unordered_map<Reduze::INT, Reduze::INTIndex> unord_map_INT;
typedef std::tr1::unordered_set<Reduze::INT> unord_set_INT;
#else
typedef std::map<Reduze::INT, Reduze::INTIndex> unord_map_INT;
typedef std::set<Reduze::INT> unord_set_INT;
#endif

namespace Reduze {

template<class > struct SecJob;
template<class > struct SecJobResult;
typedef SecJob<EquationList> SectorJob;
typedef SecJob<EquationLightList> SectorJobLight;
typedef SecJobResult<EquationList> SectorJobResult;
typedef SecJobResult<EquationLightList> SectorJobLightResult;

// abbreviation function for masking of coefficients
DECLARE_FUNCTION_1P(Coeff)

/// Storage types for Reduzer classes

#ifdef HAVE_DATABASE
enum StorageType {
	MEMORY, DATABASE
};
#else
enum StorageType {
	MEMORY
};
#endif /* HAVE_DATABASE */

/// wrapper to access DBSTL and STL maps in a uniform way
template<StorageType storage_type, class key_type, class val_type>
struct MapTraits;
/// wrapper to access DBSTL and STL sets in a uniform way
template<StorageType storage_type, class key_type>
struct SetTraits;

// specializations for STL
template<class key_type, class val_type>
struct MapTraits<MEMORY, key_type, val_type> {
	typedef std::map<key_type, val_type> type;
};
template<class key_type>
struct SetTraits<MEMORY, key_type> {
#ifdef USE_HASH_TABLE
	typedef std::tr1::unordered_set<key_type> type;
#else
	typedef std::set<key_type> type;
#endif
};

// specializations for DBSTL
#ifdef HAVE_DATABASE
template<class key_type, class val_type>
struct MapTraits<DATABASE, key_type, val_type> {
	typedef dbstl::db_map<key_type, val_type> type;
};
template<class key_type>
struct SetTraits<DATABASE, key_type> {
	typedef dbstl::db_set<key_type, dbstl::ElementHolder<key_type> > type;
};
#endif /* HAVE_DATABASE */

/// options for Reduzer
/** note: if you copy ReduzerOptions, call add_auto_options() for the copy*/
class ReduzerOptions: public YAMLAutoConfigurable {
public:
	static YAMLSpec yaml_spec() {
		YAMLSpec s;
		s.set_keyword("reduzer_options");
		s.set_short_description(""//
					"Options for the reduction algorithm.");
		s.set_long_description(s.short_description());
		/* don't confuse user with too many details
		 s.add_option("back_substitute_early", false, "boolean", ""//
		 "Whether to back eliminate prior to forward elimination.");
		 s.add_option("back_substitute_recursively", false, "boolean", ""//
		 "Mainly relevant for parallel execution."
		 " Whether to directly back substitute integrals in "
		 " back substitutions.");
		 s.add_option("back_substitute_inprocess", false, "boolean", ""//
		 "Only relevant for parallel execution."
		 " Whether to substitute integrals for which a forward"
		 " is currently in process.");
		 s.add_option("back_substitute_helpers", false, "boolean", ""//
		 "Whether to back eliminate in equations for helper integrals.");
		 */
		// incomplete: s.add_option("mask_substitutions", false, "boolean", "");
		s.add_option("num_equations_per_subjob", false, "integer >= 1", ""//
					"Minimal number of equations to be processed per subjob sent"
					" to a worker.");
		s.add_option("num_blocks_per_subjob", false, "integer >= 1", ""//
					"Minimal number of blocks (collection of equations with the"
					" same leading integral) to be processed per subjob sent"
					" to a worker.");
		s.add_option("only_substitute", false, "boolean", ""//
					"Whether only substitutions are performed.");
#ifdef HAVE_DATABASE
		s.add_option("use_database", false, "boolean", ""//
				"Whether to use a database for the reduction."
				" Otherwise, everything is stored in memory only."
				" Note that without a database, large reductions might "
				" run out of memory and fail.");
		s.add_option("use_transactions", false, "boolean", ""//
				"Only relevant if use_database is true."
				" Whether to use transactions and logging for the database."
				" Only with transactions turned on, it is safe to resume"
				" a reduction using the database from an aborted run.");
		s.add_option("cache_size", false, "integer", ""//
				"Only relevant if use_database is true."
				" Size of the database RAM cache in bytes,"
				" must be smaller than 4294967296.");
		s.add_option("compress_coefficients", false, "boolean", ""//
				"Only relevant if use_database is true."
				" Whether to use compression for storage of coefficients"
				" in the database.");
#endif // HAVE_DATABASE
		return s;
	}
	virtual YAMLSpec yaml_spec_link() const {
		return yaml_spec();
	}

	ReduzerOptions() :
		back_substitute_early(true),//
				back_substitute_recursively(false), //
				back_substitute_inprocess(true), //
				back_substitute_helpers(true), //
				delete_helpers_in_result(false), //
				mask_substitutions(false), //
				num_equations_per_subjob(1), //
				num_blocks_per_subjob(1), //
				only_substitute(false)//
#ifdef HAVE_DATABASE
	,use_database(true),//
	use_transactions(true),//
	cache_size(512 * 1024 * 1024), //
	compress_coefficients(false)
#endif // HAVE_DATABASE
	{
		add_auto_options();
	}

	/// whether to consider identities with requested leading integrals first
	/** if false, reduction order is determined by integral ordering only **/
	//bool consider_requested_first;
	/// whether early elimination should be forced
	bool back_substitute_early;
	/// whether elimination of sub-block integrals should be done recursively
	bool back_substitute_recursively;
	/// whether equations which are in process should be used for back elimination
	bool back_substitute_inprocess;
	/// whether to back eliminate in equations for unrequested integrals
	bool back_substitute_helpers;
	/// whether helpers identities should be deleted in result file
	bool delete_helpers_in_result;
	/// whether coefficients of pure substitutions should first be masked
	/** permanently set to false since not fully implemented */
	bool mask_substitutions;
	/// whether r+1 or s+1 helper equations should be removed early
	/** this option relies on the assumption that solutions for r,s need at most
	 ** helper equations with r+1 and s+1 */
	//bool drop_helper_eqs_early;
	/// number of equations per subjob
	unsigned num_equations_per_subjob;
	/// number of blocks per subjob
	unsigned num_blocks_per_subjob;
	bool only_substitute;
#ifdef HAVE_DATABASE
	/// whether to use a database
	bool use_database;
	/// whether to enable transactions for the database
	bool use_transactions;
	/// cache size in bytes
	long cache_size; // e.g. 512 * 1024 * 1024, converted to u_int32_t later on
	/// whether coefficients should be compressed in the database
	bool compress_coefficients;
#endif // HAVE_DATABASE
	//
	virtual void add_auto_options() {
		/*
		 add_auto_io("back_substitute_early", back_substitute_early);
		 add_auto_io("back_substitute_recursively", back_substitute_recursively);
		 add_auto_io("back_substitute_inprocess", back_substitute_inprocess);
		 add_auto_io("back_substitute_helpers", back_substitute_helpers);
		 */
		//add_auto_io("mask_substitutions", mask_substitutions);
		add_auto_io("num_equations_per_subjob", num_equations_per_subjob);
		add_auto_io("num_blocks_per_subjob", num_blocks_per_subjob);
		add_auto_io("only_substitute", only_substitute);
#ifdef HAVE_DATABASE
		add_auto_io("use_database", use_database);
		add_auto_io("use_transactions", use_transactions);
		add_auto_io("cache_size", cache_size);
		add_auto_io("compress_coefficients", compress_coefficients);
#endif // HAVE_DATABASE
	}

protected:
	virtual void init();
};

inline void operator>>(const YAML::Node& n, ReduzerOptions& r) {
	r.read(n);
}

inline YAML::Emitter& operator<<(YAML::Emitter& ye, const ReduzerOptions& r) {
	r.print(ye);
	return ye;
}

/// algorithm for reduction of system of equations using small jobs
/** Usage: call setup() or open() method of derived class, then request jobs,
 ** and optionally clear() at the end. */
template<StorageType storage_type>
class ReduzerBase {
public:
	typedef INTIndex integral_type;
	typedef CoeffIndex coeff_type;
	typedef EquationXLight eq_type;
	typedef EquationXLightList eqlist_type;

	ReduzerBase(const ReduzerOptions& opts = ReduzerOptions());
	virtual ~ReduzerBase();

	/// fills the Reduzer with equations (call setup() from derived class first)
	virtual void load(const std::list<std::string>& eq_files,//
			const std::list<std::string>& subst_files,//
			const std::set<INT>& all_integrals,//
			const unord_set_INT& identity, //
			const std::set<INT>& external,//
			const std::set<INT>& helpers,//
			const std::set<INT>& criticals,//
			const std::set<INT>& substints);
	/// creates next job to compute and mark it as in process
	/** returns zero if no job found, nonzero result must be deleted by caller
	 ** tries to fill num_eqs into the job */
	virtual SectorJobLight* create_next_job();
	/// returns the approximate fraction of reduced integrals
	virtual double progress();
	/// print some info
	virtual void print_status() const;
	/// insert results
	virtual void insert(SectorJobLightResult& result, ProgressBar* pbar = 0);
	/// writes the result to a file
	/** second file name is for new subsector identities,
	 ** should be const, is not because of funny dbstl behaviour **/
	virtual void write_result(const std::string& filename,
			const std::string& filenameexternal);
	/// writes substitutions to a file (for tests)
	virtual void write_substitutions(const std::string& filename) const;
	/// writes abbreviations for integrals and coefficients to a file (for tests)
	virtual void write_abbreviations(const std::string& filename) const;
	/// whether the reduction is completed
	virtual bool completed();
	/// frees memory and resets the instance (closes database if any)
	/** derived classes should override this and call parents clear **/
	virtual void clear();
	/// perform the full reduction (serial usage only)
	virtual void run();

	/// equations with the same leading integral
	// should be private, is not because of extern "C" functions
	struct Block {
	public:
		Block() {
		}
		/// tells if the equations are reduced
		inline bool reduced() const {
			return !in_process() && eqs.size() <= 1;
		}
		/// tells if someone is working on the block
		inline bool in_process() const {
			return !eqs_inprocess.empty();
		}
		eqlist_type eqs; // the equations
		eqlist_type eqs_inprocess; // equations in process
	};

protected:
	typedef typename MapTraits<storage_type, integral_type, Block>::type Blocks;

	/// size of integral abbreviations
	size_t integ_size() const {
		if (integ == 0) {
			return 0;
		}
		return integ->size();
	}
	/// inserts substitutions in file to database, integrals must be known
	/** inserts only those substitutions which have a leading integral in 'integrals'*/
	void insert_substitutions(const std::string& filename,
			const unord_map_INT& integ_abbr, const unord_set_INT& integrals);
	/// inserts equations to be reduced to database, integrals must be known
	void insert_equations(const std::string& filename,
			const unord_map_INT& integ_abbr);

	/// begins a transaction if transactions enabled, does nothing otherwise
	virtual void begin_transaction() {
	}
	/// commits a transaction if transactions enabled, does nothing otherwise
	virtual void commit_transaction() {
	}
	/// aborts a transaction if transactions enabled, does nothing otherwise
	virtual void abort_transaction() {
	}

	// user controlled options
	ReduzerOptions options_;

	// maps and sets are allocated and freed by derived classes

	/// abbreviation map for integrals
	typename MapTraits<storage_type, integral_type, std::string>::type* integ;
	/// abbreviation map for coefficients
	typename MapTraits<storage_type, coeff_type, std::string>::type* coeff;
	/// substitutions which are not part of the system to be reduced
	typename MapTraits<storage_type, integral_type, eq_type>::type* subst;
	/// equations divided into blocks, key for each block is leading integral
	Blocks* blocks;
	/// critical integrals determining when helpers are not reduced anymore
	typename SetTraits<storage_type, integral_type>::type * criticals;
	/// helper integrals
	typename SetTraits<storage_type, integral_type>::type * helpers;
	/// external integrals (for new subsector identities)
	typename SetTraits<storage_type, integral_type>::type * external;
	/// integrals for which new reductions are discarded
	typename SetTraits<storage_type, integral_type>::type * substinteg;

private:
	enum TriBool {
		True, False, Unknown
	};

	/// finds next blocks to compute
	/** - flag specifies whether back elimination should have high priority
	 ** - returns zero if no job found, nonzero result must be deleted by caller
	 ** - updates 'done'  */
	typename Blocks::iterator find_next_block(bool back_elim_first);
	/// inserts equations (removing them from argument)
	void insert_equations(eqlist_type& eqs);
	/// checks if any substitutions are available for subleading INTs of eqs
	bool have_subst(const eqlist_type& eqs) const;
	/// finds substitutions available for subleading INTs of eqs
	/** if recurse == true substitutions are searched recursively */
	void
	find_subst(const eqlist_type& eqs, eqlist_type& subst, bool recurse) const;

	/// converts the equation and stores the abbreviated coefficients
	EquationXLight abbreviate(const EquationHLight& eq,
			const unord_map_INT& integ_abbr);
	EquationXLight abbreviate(const EquationLight& eq);
	EquationXLightList abbreviate(const EquationLightList& eqs);
	EquationLight unabbreviate_coefficients(const EquationXLight& eq) const;
	EquationLightList
	unabbreviate_coefficients(const EquationXLightList& eqs) const;
	EquationHLight unabbreviate(const EquationXLight& eq) const;
	EquationHLightList unabbreviate(const EquationXLightList& eqs) const;
	void erase_coefficient_abbreviations(const EquationXLight& eq);
	void erase_coefficient_abbreviations(const EquationXLightList& eqs);

	/// maximal integral which was processed so far for progress report
	integral_type maxint_processed;

	/// whether the reduction is completed
	TriBool done;
	/// hint where to start search for next block
	integral_type findnextbegin;
};

/// in-memory storage Reduzer
class ReduzerMem: public ReduzerBase<MEMORY> {
public:
	ReduzerMem(const ReduzerOptions& opts = ReduzerOptions()) :
			ReduzerBase<MEMORY>(opts) {
	}
	virtual ~ReduzerMem() {
		clear();
	}
	/// sets up the storage
	void setup();
	/// resets the instance and frees memory
	virtual void clear();
};

#ifdef HAVE_DATABASE
/// database storage Reduzer
class ReduzerDb: public ReduzerBase<DATABASE> {
public:
	ReduzerDb(const ReduzerOptions& opts = ReduzerOptions());
	virtual ~ReduzerDb() {
		clear();
	}
	/// fills the database with equations
	void setup(const std::string& db_home, const std::string& db_filename);
	/// opens an existing database
	void open(const std::string& db_home, const std::string& db_filename);
	/// close the database and reset the instance
	/** it is never necessary to explicitly close() the database except
	 ** to free memory */
	virtual void clear();
	/// renames an open database
	/** closes the databases (without closing the environment),
	 *  renames the database file and opens the databases again,
	 *  throws if env == 0
	 */
	void dbrename(const std::string& old_name, const std::string& new_name);
	/// sync to disk
	void sync();
	/// clean up files associated with current or last opened environment
	void remove_files();

protected:
	virtual void begin_transaction();
	virtual void commit_transaction();
	virtual void abort_transaction();

private:
	/// sets up an environment
	void init_env(const std::string& home);
	/// sets up a new database from scratch and closes it
	void init_db(const std::string& filename);
	/// closes only the environment (close_db() must be called before)
	void close_env();
	/// closes only the databases
	void close_db();
	/// opens only the environment
	void open_env(const std::string& db_home);
	/// opens only the database (open_env(...) must be called before)
	void open_db(const std::string& db_filename);

	/// the various databases (all reside in one file)
	Db *db_integ, *db_coeff, *db_subst, *db_blocks, *db_crit, *db_help,
	*db_external, *db_substinteg;
	/// the database environment
	DbEnv* env;
	/// database directory
	std::string dbhome;

};
#endif // HAVE_DATABASE
template<>
struct SecJob<EquationList> { // alias SectorJob
	std::list<INTIndex> maxints; // only for book-keeping who is in process
	EquationList upper_eqs, subst_eqs;
	bool forward_eliminate;
	bool only_substitute;
	SecJob();
	SecJob(const SecJob<EquationLightList>& job_light);
	/// perform the job
	SecJobResult<EquationList> run();
};

template<>
struct SecJob<EquationLightList> { // alias SectorJobLight
	SecJob() :
		forward_eliminate(true), only_substitute(false) {
	}
	std::list<INTIndex> maxints; // only for book-keeping who is in process
	EquationLightList upper_eqs, subst_eqs;
	bool forward_eliminate;
	bool only_substitute;
};

template<>
struct SecJobResult<EquationList> { // alias SectorJobResult
	std::list<EquationList::integral_type> maxints;
	EquationList upper_eqs, lower_eqs;
};

template<>
struct SecJobResult<EquationLightList> { // alias SectorJobLightResult
	SecJobResult();
	SecJobResult(const SecJobResult<EquationList>& full);
	std::list<EquationLightList::integral_type> maxints;
	EquationLightList upper_eqs, lower_eqs;
};

} // namespace Reduze

#endif /* REDUZER_H_ */
