/*  yamlconfigurable.h
 *
 *  Copyright (C) 2010-2012 Andreas von Manteuffel
 *  Copyright (C) 2010-2012 Cedric Studerus
 *
 *  This file is part of the package Reduze 2.
 *  It is distributed under the GNU General Public License version 3
 *  (see the file GPL-3.0.txt or http://www.gnu.org/licenses/gpl-3.0.txt).
 */

#ifndef YAMLCONFIGURABLE_H_
#define YAMLCONFIGURABLE_H_

#include "../yaml/include/yaml.h" // forward decl. not enough for template impl.
#include "yamlutils.h" // to make clang on Mac happy for YAMLReader::read() impl. below
#include <vector>
#include <set>

namespace Reduze {

/// specification of a YAML interface
/** defines a list of options via keywords and descriptions,
 ** can be used by YAMLFactory to:
 ** - display help texts
 ** - check if all keywords in a YAML map are known
 ** - check if a YAML map contains all mandatory keywords
 */
class YAMLSpec {
public:
	void set_keyword(const std::string&);
	void set_short_description(const std::string&);
	void set_long_description(const std::string&);
	const std::string& keyword() const;
	const std::string& short_description() const;
	const std::string& long_description() const;
	void add_option(const std::string& keyword, bool mandatory,
			const std::string& type, const std::string& help);
	void add_options(const YAMLSpec&);
	bool has_option(const std::string&) const;
	const std::set<std::string>& mandatory_options() const;
	void print_help(std::ostream&) const;
private:
	std::string keyword_, short_description_, long_description_;
	std::set<std::string> mandatory_options_;
	std::map<std::string, int> optindex_;
	struct OptHelp {
		std::string keyword, type, help;
		bool mandatory;
	};
	std::vector<OptHelp> opts_;
};

/// interface for a type which implements a YAML interface
class YAMLConfigurable {
public:
	virtual ~YAMLConfigurable() {
	}
	// provides access to static type info of children
	virtual YAMLSpec yaml_spec_link() const = 0;
	/// prints object (as a map of options, no header)
	virtual void print(YAML::Emitter& os) const = 0;
	/// reads object (as a map of options, no header)
	virtual void read(const YAML::Node& n) = 0;
	/// verifies YAML::Node fulfills the spec, should be called in read()
	virtual void verify_yaml_spec(const YAML::Node& n);
	/// prints object as value of a one-element map with its keyword as key
	void print_with_type(YAML::Emitter& os) const;
	/// returns a YAML representation of the object with its current settings
	/** the caller should take care of deleting the node */
	YAML::Node* create_yaml_rep() const;
};

// provide types with a YAML interface via registration to a factory
//
// the design pattern is based on:
// Jim Beveridge, August 01, 1998: "Self-Registering Objects in C++"
// http://drdobbs.com/cpp/184410633
// and a modified implementation posted in:
// http://stackoverflow.com/questions/1260954/how-can-i-keep-track-of-enumerate-all-classes-that-implement-an-interface

/// interface between YAMLFactory and YAMLProxy
class IYAMLProxy {
public:
	virtual ~IYAMLProxy() {
	}
	virtual YAMLConfigurable* create_object() = 0;
	virtual YAMLSpec yaml_spec() const = 0;
};

/// factory and help for objects having a YAML interface
class YAMLFactory {
public:
	static YAMLFactory& instance() {
		static YAMLFactory instance;
		return instance;
	}
	// add a YAML specification
	void add(IYAMLProxy* p) {
		specs_[p->yaml_spec().keyword()] = p;
	}
	size_t num_jobs() {
		return specs_.size();
	}
	YAMLConfigurable* create_object(const std::string& keyword);
	void print_help_topics(std::ostream&) const;
	void print_help(const std::string& keyword, std::ostream&) const;
	std::set<std::string> keywords() const;
private:
	std::map<std::string, IYAMLProxy*> specs_;
};

/// YAML factory proxy template class
template<typename T>
class YAMLProxy: public IYAMLProxy {
public:
	YAMLProxy() {
		YAMLFactory::instance().add(this);
	}
	virtual ~YAMLProxy() {
	}
	YAMLConfigurable* create_object() {
		return new T;
	}
	virtual YAMLSpec yaml_spec() const {
		return T::yaml_spec();
	}
};

class IYAMLReader {
public:
	virtual ~IYAMLReader() {
	}
	virtual void read(const YAML::Node& n) = 0;
};

class IYAMLPrinter {
public:
	virtual ~IYAMLPrinter() {
	}
	virtual void print(YAML::Emitter&) = 0;
};

template<typename T>
class YAMLReader: public IYAMLReader {
public:
	YAMLReader(T& val) :
		val_(val) {
	}
	virtual ~YAMLReader() {
	}
	virtual void read(const YAML::Node& n) {
		n >> val_;
	}
private:
	T& val_;
};

template<typename T>
class YAMLPrinter: public IYAMLPrinter {
public:
	YAMLPrinter(const T& val) :
		val_(val) {
	}
	virtual ~YAMLPrinter() {
	}
	virtual void print(YAML::Emitter& ye) {
		ye << val_;
	}
private:
	const T& val_;
};

/// convenience class for automatic read and print of options
/** class is instance specific: coupling of options to non-static members,
 ** note: take care with copy/assignment of derived classes:
 ** no readers/printers are copied automatically, set_add_auto_io() should
 ** be called explicitely in derived classes copy/assignment operators */
class YAMLAutoConfigurable: public YAMLConfigurable {
public:
	YAMLAutoConfigurable() {
	}
	/// copy ctor will _not_ copy any readers/printers
	/** use set_add_auto_io again in derived classes */
	YAMLAutoConfigurable(const YAMLAutoConfigurable&);
	/// assignment operator will _clear_ any readers/printers
	/** use set_add_auto_io again in derived classes */
	YAMLAutoConfigurable& operator=(const YAMLAutoConfigurable&);
	virtual ~YAMLAutoConfigurable();
	virtual void print(YAML::Emitter& os) const;
	virtual void read(const YAML::Node& n);
protected:
	/// derived classes should implement their add_auto_io() calls here
	/** it is up to the derived class to call this function once
	 ** in its constructors **/
	virtual void add_auto_options() = 0;
	/// called after read to validate and postprocess input
	virtual void init() {
	}
	/// reads additional options and validates YAML input
	/** options of class Job and options handled by AutoOptions (add_auto_options())
	 ** will be read before this method is called, derived classes should
	 ** implement this to read more complicated structures */
	virtual void print_manual_options(YAML::Emitter& os) const {
	}
	/// prints additional options
	/** options of class Job and options handled by AutoOptions (add_auto_options())
	 ** will be written before this method is called, derived classes should
	 ** implement this to print more complicated structures */
	virtual void read_manual_options(const YAML::Node&) {
	}
	/// associates an option to a variable for automatic read/print
	/** should be called exclusively from add_auto_options() */
	template<class T>
	void add_auto_io(const std::string& keyword, T& t) {
		readers_.push_back(make_pair(keyword, new YAMLReader<T> (t)));
		printers_.push_back(make_pair(keyword, new YAMLPrinter<T> (t)));
	}
	void read_auto_options(const YAML::Node& n);
	void print_auto_options(YAML::Emitter& os) const;
private:
	void clear_auto_io();
	std::list<std::pair<std::string, IYAMLReader*> > readers_;
	std::list<std::pair<std::string, IYAMLPrinter*> > printers_;
};

} // namespace Reduze

#endif /* YAMLCONFIGURABLE_H_ */
