/*  Pasang Emas. Enjoy a unique traditional game of Brunei.
    Copyright (C) 2010  Nor Jaidi Tuah

    This file is part of Pasang Emas.
      
    Pasang Emas is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
namespace Pasang {

/**
 * Dimensions of board, including sentinnel rows and columns.
 * The actual playing area is 11 by 11.
 *
 * Visually, the board is 13x13, allowing for decorative borders
 */
public const int BOARD_WIDTH = 13;
public const int BOARD_SIZE = 13*13;

/**
 * Number of each type of pieces.
 */
public const int NUM_WHITES = 60;
public const int NUM_BLACKS = 60;

/**
 * Relative directions
 */
public enum Delta {
    UP = -13,
    DOWN = 13,
    RIGHT = 1,
    LEFT = -1
}

/**
 * Possible contents of a board.
 * When cast to int, the value is used to index to image portions.
 * WHITE and BLACK should actually correspond to visually 
 * black/dark and white/light pieces (like in chess).
 *
 * A int can also be cast into a Piece to convert player number 
 * (either 0 or 1) into either KAS_0 or KAS_1.
 */
public enum Piece {
    NULL = -999,          // Uninitialized
    KAS_0 = 0,            // Kas for player 0
    KAS_1 = 1,            // Kas for player 1
    BLACK = 2,            // Black piece, worth 1 point
    WHITE = 3,            // White piece, worth 2 point
    COUNT = 4,            // Number of different piece types
    BOARD = 4,            // Not really a piece
    GUARD = 5,            // Sentinnel piece, used for border and centre
    GUIDE = 6,            // Guide placed in "Guided play"
    EMPTY = 100;          // Empty, not occupied by any piece

    public bool is_real () {
        return this >= KAS_0 && this <= WHITE;
    }

    public bool is_kas () {
        return this == KAS_0 || this == KAS_1;
    }
}

public enum Stage {
    NULL = -999,          // Uninitialized
    OPENING = 0,          // The stages where players remove a column,
    SELECT = 1,           // ... select a piece for kas (mobile piece),
    SUB_SELECT = 2,       // ... disambiguate (which line to capture),
    MOVE = 3,             // ... move kas,
    SUB_MOVE = 4,         // ... disambiguate (which to capture, blacks or whites).
    GAME_OVER = 5
}

public class Move : Object {
    /**
     * At Stage.OPEN, this is the column selected (1-11).
     * At Stage.SELECT, this is the source position.
     * At Stage.MOVE, this is the destination position.
     */
    public int position {get; private set;}

    /**
     * Positions of tokens captured by this move.
     */
    public int[] booties;

    /**
     * Standard string representation for this move.
     */
    public string notation {get; private set;}

    /**
     * Not used by this class. Provided here for the program's AI.
     */
    public int @value {get; set;}
      
    public Move (Game game, int position, int[] booties) {
        assert (booties.length % 2 == 1);  // Capture odd number of pieces
        this.position = position;
        this.booties = booties;
        int worth = 0;
        foreach (int pos in booties) {
            assert (pos >= 0 && pos < BOARD_SIZE);
            int piece = game.board[pos];
            assert (piece == Piece.BLACK || piece == Piece.WHITE);
            worth += (piece == Piece.BLACK) ? 1 : 2;
        }
        char col = 'A' - 1 + (12 - position % BOARD_WIDTH);
        int  row = position / BOARD_WIDTH;
        bool capture_vertical = (booties[0] / BOARD_WIDTH) != row;
        if (game.stage == Stage.OPENING) {
            notation = "%cx%d".printf (col, worth);
        }
        else if (game.stage == Stage.SELECT && capture_vertical) {
            notation = "%c%dx%d*".printf (col, row, worth);
        }
        else {
            notation = "%c%dx%d".printf (col, row, worth);
        }
    }
}//class Move

public class Game : Object {
    public Stage stage {get; private set; default = Stage.NULL;}

    public Piece[] board = new Piece[BOARD_SIZE];

    /**
     * Move sequence number.
     * Initially -1. 0 after start (). +1 after perform ().
     * Used in NETWORK mode to verify that clients are synchronized.
     * (Too paranoid?)
     */
    public int seq {get; private set; default = -1;}

    /**
     * Player to move next. Player 0 always moves first.
     */
    public int player {get; private set;}

    /**
     * The kas owned by the player to move next.
     */
    public Piece kas {get {return (Piece) player;}}

    public int value_of_uncaptured_pieces {get; private set;}

    /**
     * Moves available from the current position.
     */
    Move[] moves = {};

    public int[] score = {0, 0};

    public int[] kas_position = {0, 0};

    /**
     * Set during Stage.OPENING and looked up during Stage.SELECT.
     * Respective players' first moves. This is needed (1) to make sure that
     * Player 1 does not copy Player 0's first move, (2) to generate the 
     * player's kas selection move, and (3) to determine onto which
     * column a kas candidate should be dropped.
     */
    public int[] first_move= {0, 0}; // {get; private set; default = {0, 0};}

    /**
     * Clone a game, except for its list of moves.
     */
    public Game.clone (Game source) {
        board = source.board;
        player = source.player;
        value_of_uncaptured_pieces = source.value_of_uncaptured_pieces;
        stage = source.stage;
        score = source.score;
        kas_position = source.kas_position;
        first_move = source.first_move;
        seq = source.seq;
        // moves doesn't have to be cloned
    }

    /**
     * Check that a pattern is valid.
     */
    public static bool is_valid_pattern (string pattern) {
        var blacks = 0;
        var whites = 0;
        for (int i=0; i < pattern.length; i++) {
            if (pattern[i] == 'P') whites++;
            else if (pattern[i] == 'H') blacks++;
            else if (pattern[i] == ' ') continue;
            else return false;
        }
        return blacks == 30 && whites == 30;
    }

    /**
     * Compile a string of H's and P's into an array of Pieces representing 
     * an intial board arrangment.
     */
    public static void compile (string pattern, ref Piece[] b) 
        requires (b.length == BOARD_SIZE) {
        for (int i=0; i< BOARD_SIZE; i++) b[i] = Piece.GUARD;
        var whites = 0;
        var blacks = 0;
        var n = 0;
        for (int i=0; i < pattern.length; i++) {
            var code = pattern[i];
            if (code == ' ') continue;
            // pos = map n to skip borders
            var row = n / (BOARD_WIDTH - 2) + 1;
            var col = n % (BOARD_WIDTH - 2) + 1;
            var pos = row * BOARD_WIDTH + col;
            if (code == 'H') {
                b[pos] = b[BOARD_SIZE - 1 - pos] = Piece.BLACK;
                blacks += 2;
            }
            else {
                assert (code == 'P');
                b[pos] = b[BOARD_SIZE - 1 - pos] = Piece.WHITE;
                whites += 2;
            }
            n++;
        }
        assert (whites == NUM_WHITES && blacks == NUM_BLACKS);
    }

    /**
     * Start the game with the given starting board (in textual form).
     * Player 0 always plays first and the game always begins with Stage.OPEN. 
     */
    public void start (string starting_pattern) {
        compile (starting_pattern, ref board);
        player = 0;
        score = {0, 0};
        kas_position = {0, 0};
        value_of_uncaptured_pieces = 2 * NUM_WHITES + NUM_BLACKS;
        stage = Stage.OPENING;
        seq = 0;
        generate_moves ();
    }
         
    /**
     * Perform the given move. The move can be partial, e.g., the user has moved
     * her kas, but has yet to decide either to capture blacks or whites.
     *
     * full_move should be true for AI players, since AI players are not
     * constrained to use a sequential input device to make their moves.
     * full_move should also be true for remote players, since moves from remote players
     * are always marshalled.
     *
     * Unchecked precondition: if full_move==false, generate_moves should be called first.
     */
    public void perform (Move move, bool full_move) {
        assert (stage != Stage.NULL);
        switch (stage) {
        case Stage.OPENING :
        case Stage.SUB_SELECT :
        case Stage.SUB_MOVE :
            if (stage == Stage.OPENING) first_move[player] = move.position;
            perform_capture (move);
            next_player ();
            generate_moves ();
            break;
        case Stage.SELECT :
        case Stage.MOVE :
            perform_move_kas (move);
            if (!full_move && count_submoves (move) > 1) {
                // Partial move. Needs another mouse click to decide capture.
                stage = (stage == Stage.MOVE) ? Stage.SUB_MOVE : Stage.SUB_SELECT;
                Move[] submoves = {};
                foreach (Move m in moves) {
                    if (m.position == move.position) submoves += m;
                }
                moves = submoves;
                assert (moves.length == 2 || moves.length == 3);
            }
            else {
                // Complete the move with a capture. Next player.
                perform_capture (move);
                next_player ();
                generate_moves ();
            }
            break;
        }
        
        if (moves.length == 0) {
            stage = Stage.GAME_OVER;
            if (value_of_uncaptured_pieces > 0) { // Out of moves (suntuk)
                score[1 - player] += score[player];
                score[player] = 0;
            }
        }
        if (stage != Stage.SUB_MOVE && stage != Stage.SUB_SELECT) seq++;
    }

    /**
     * Perform the given text-encoded move (see Move.notation).
     * Unchecked precondition: generate_moves should be called first.
     */
    public void perform_from_notation (string move_notation) {
        perform (get_move (move_notation), true);
    }

    /**
     * Get the move represented by the given notation (see Move.notation).
     */
    public Move? get_move (string move_notation) {
        foreach (Move m in moves) {
            if (m.notation == move_notation) {
                return m;
            }
        }
        return null;
    }

    /**
     * Move kas, but do not capture yet because a further submove may be required.
     */
    private void perform_move_kas (Move move) {
        var position = move.position;
        if (stage == Stage.SELECT) {
            // From move.position, a piece is moved onto kas_position[player]
            // to become the player's kas.
            int col = first_move[player];
            int row = position / BOARD_WIDTH;
            kas_position[player] = col + row * BOARD_WIDTH;
            // Since the promoted token is not capturable, its value is lost
            value_of_uncaptured_pieces -= (board[position] == Piece.WHITE) ? 2 : 1;
            // Update the board
            board[position] = Piece.EMPTY;
            board[kas_position[player]] = (Piece) player;
        }
        else { 
            // Normal move
            board[kas_position[player]] = Piece.EMPTY;
            board[position] = (Piece) player;
            kas_position[player] = position;
        }
    }

    /**
     * Capture the pieces listed in the move's list of booties, making 
     * those positions empty. The player's score is updated.
     */
    private void perform_capture (Move move) {
        foreach (int pos in move.booties) {
            if (board[pos] == Piece.WHITE) {
                value_of_uncaptured_pieces -= 2;
                score[player] += 2;
            }
            else if (board[pos] == Piece.BLACK) {
                value_of_uncaptured_pieces --;
                score[player] ++;
            }
            else {//if error
                for (int i=0; i < BOARD_SIZE; i++) {
                    if (i % BOARD_WIDTH == 0) stderr.printf("\n");
                    switch (board[i]) {
                    case Piece.WHITE : 
                        stderr.printf("2"); break;
                    case Piece.BLACK : 
                        stderr.printf("1"); break;
                    case Piece.KAS_0 : 
                        stderr.printf("R"); break;
                    case Piece.KAS_1 : 
                        stderr.printf("B"); break;
                    case Piece.EMPTY : 
                        stderr.printf("."); break;
                    default          : 
                        stderr.printf(" "); break;
                    }
              }
              stderr.printf ("\nError trying to perform %s\n", move.notation);
              assert_not_reached (); // Invalid capture
            }//endif error
            board[pos] = Piece.EMPTY;
        }//endforeach
    }

    /**
     * Set up for the next player's turn, advancing the stage of the game.
     */
    private void next_player () {
        if (stage == Stage.SUB_SELECT)
            stage = Stage.SELECT;
        else if (stage == Stage.SUB_MOVE)
            stage = Stage.MOVE;
          
        if (player == 1) {
            if (stage == Stage.OPENING)
                stage = Stage.SELECT;
            else if (stage == Stage.SELECT)
                stage = Stage.MOVE;
        }       
        player = 1 - player;
    }

    /**
     * Count the number of moves having the same "position" (but different capture).
     * After the kas is moved to (or selected from) this position, the capture
     * must be resolved (i.e., a submove must be made).
     */
    public int count_submoves (Move move) {
        int count = 0;
        foreach (Move m in moves) {
            if (move.position == m.position) count++;
        }
        return count;
    }
              
    /** 
     * Sort moves in descending values.
     */
    public void sort_moves () {
       Posix.qsort (moves, moves.length, sizeof(Move), 
        (m1, m2) => {return (*((Move**)m2))->@value - (*((Move**)m1))->@value;}); // Ugly!!!
    }

    /**
     * Delete every move except the given one
     */
    public void leave_one_move (Move move) {
        moves = {move};
    }
    
    /**
     * Generate a list of all possible moves.
     */
    public void generate_moves () {
        moves = {};
        if (stage == Stage.OPENING) {
            generate_opening_moves ();
        }
        else if (stage == Stage.SELECT) {
            generate_kas_selection_moves ();
        }
        else if (stage==Stage.MOVE) {
            generate_kas_moving_moves ();
        }
        else {
            stderr.printf ("Stage : %d\n", stage);
            assert_not_reached (); // Cannot generate moves
        }
    }

    /**
     * Generate moves each of which captures either these 5 pieces
     *   (x,1), (x,2), (x,3), (x,4), (x,5)    -- for player 0
     * or these 5
     *   (x,7), (x,8), (x,9), (x,10), (x,11)  -- for player 1
     * where x is an element of [1,2,...,11]-[forbidden_column]
     */
    private void generate_opening_moves () {
        // Recall player 0's first move. Make sure that player 1 does not copy this move.
        var forbidden_column = (player == 1) ? (12 - first_move[0]) : -1;
        var ref_pos =  ((player == 0) ? 1 : 7) * BOARD_WIDTH;
        for (int col=1; col <= 11; col++) {
            if (col == forbidden_column) continue;
            int[] booties = {};
            int capture_position = ref_pos + col;
            for (int n=0;  n < 5;  n++) {
                booties += capture_position;
                capture_position += Delta.DOWN;
            }
        	moves += new Move (this, col, booties);
        }
    }

    /**
     * The kas can be selected from a piece on the banks, each two-column wide, astriding 
     * the empty column that was captured during the opening move. Such a piece is either 
     * one grid or two grid away from the empty column (see figure below where O represent 
     * a candidate kas).
     *
     *                 OOeOO
     *                 OOmOO
     *                 OOpOO
     *                 OOtOO
     *                 OOyOO
     *
     * <diversion>
     *   Note that the test "candidate != Piece.EMPTY" was missing from the original
     *   Pasang Emas (non-free, and this bug will live forever in there). As a result, 
     *   a captured piece can still be used for the kas!. (e.g. Select "galap" pattern. 
     *   Player 0 capture column 3, Player 1 capture column 2. Now Player 0 can 
     *   capture the three black pieces, two of which are player 1's potential kas).
     * </diversion>
     */
    private void generate_kas_selection_moves () {
      int row = (player == 0) ? 1 : 7;
      int col = first_move[player];
      int[] distances = {-2, -1, 1, 2};
      for (int j=0; j < 5; j++) { // for each of the 5 grid points in the empty column
          int empty_position = (row + j) * BOARD_WIDTH + col;
          foreach (int distance in distances) {
              if ((col + distance < 1) || (col + distance > 11)) continue;  // Don't go outside the board
              int candidate_position = empty_position + distance;
              Piece piece = board[candidate_position];
              if (piece == Piece.EMPTY) continue;
              board[candidate_position] = Piece.EMPTY;                      // Lift the candidate
              add_move (candidate_position, empty_position, Delta.RIGHT);   // Capture horizontally
              add_move (candidate_position, empty_position, Delta.DOWN);    // Capture vertically
              board[candidate_position] = piece;                            // Put back the candidate
          }
      }
    }

    /**
     * Move the kas up and down, capturing horizontally,
     * or left and right, capturing vertically.
     */
    private void generate_kas_moving_moves () {
        int[] kas_directions     = {Delta.UP,    Delta.DOWN,  Delta.LEFT, Delta.RIGHT};
        int[] capture_directions = {Delta.RIGHT, Delta.RIGHT, Delta.DOWN, Delta.DOWN};
        for (int i=0; i < 4; i++) {
            int position = kas_position[player] + kas_directions[i];
            while (board[position] == Piece.EMPTY) {
                add_move (position, position, capture_directions[i]);
                position += kas_directions[i];
            }
        }
    }
        
    /**
     * Construct a move for each possible capture in the given capture_direction
     * (Delta.RIGHT for horizontal capture, Delta.DOWN for vertical capture). Only an odd 
     * number of pieces of the same colour can be captured. These pieces must be
     * in line with capture_position.
     */
    private void add_move (int position, int capture_position, int capture_direction)
        requires (capture_direction == Delta.RIGHT || capture_direction == Delta.DOWN) {
        Piece[] booty_types = {Piece.NULL, Piece.NULL};
        int[] booties0 = {};   // TODO: this should have been int[][], but vala currently doesn't support this
        int[] booties1 = {};
        for (int i=0; i < 2; i++) {
            int delta = i==0 ? -capture_direction : capture_direction;  // Two opposite directions to scan
            int cursor = capture_position;
            while (true) {
                cursor += delta;
                Piece piece = board[cursor];
                if (piece == Piece.EMPTY) continue;
                if (piece != Piece.BLACK && piece != Piece.WHITE) break;
                if (booty_types[i] != Piece.NULL && piece != booty_types[i]) break;
                if (i == 0) booties0 += cursor;  // TODO: booties[i] += cursor
                else        booties1 += cursor;
                booty_types[i] = piece;
            }
        }//for
        if (booty_types[0] == booty_types[1]) {         // Both directions have pieces of the same colour?
            foreach (int p in booties1) booties0 += p;  // If so, accumulate all booties
            booties1 = {};
        }
        if (booties0.length % 2 == 1) {                 // Capture odd number of pieces only
            moves += new Move (this, position, booties0);
        }
        if (booties1.length % 2 == 1) {
            moves += new Move (this, position, booties1);
        }
    }

    /**
     * Iterator to support foreach(Move m in game) {...}
     */
    public Iterator iterator() {
        return new Iterator(this);
    }

    public class Iterator {
        Game game;
        int index;
        public Iterator(Game game) {
            this.game = game;
            index = 0;
        }

        public Move? next_value() {
            return index == game.moves.length ? null : game.moves[index++];
        }
    }//class Iterator

    /**
     * Access to moves
     */
    public int numMoves() {
        return moves.length;
    }

    public Move getMove(int n) {
        return moves[n];
    }
}//class Game
}//namespace
// vim: tabstop=4: expandtab: textwidth=100: autoindent:
