/*******************************************************************************
*                                                                              *
* align_comments.c -- align programming language comments                      *
*                                                                              *
* Copyright (C) 1998 Steve Haehn                                               *
*                                                                              *
* This is free software; you can redistribute it and/or modify it under the    *
* terms of the GNU General Public License as published by the Free Software    *
* Foundation; either version 2 of the License, or (at your option) any later   *
* version.                                                                     *
*                                                                              *
* This software is distributed in the hope that it will be useful, but WITHOUT *
* ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or        *
* FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License *
* for more details.                                                            *
*                                                                              *
* You should have received a copy of the GNU General Public License along with *
* software; if not, write to the Free Software Foundation, Inc., 59 Temple     *
* Place, Suite 330, Boston, MA  02111-1307 USA                                 *
*                                                                              *
*******************************************************************************/

/*============================================================================*/
/*====================  OPERATING SYSTEM INCLUDE FILES  ======================*/
/*============================================================================*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

/*===========================================================================*/
/*                             SYMBOL DEFINITIONS                            */
/*===========================================================================*/

#define MAX_LINES 300
#define EOS '\0'

#define FPATH_SEPARATOR '/'
#define FALSE (bool_t)0
#define TRUE  (bool_t)!FALSE
#define MAXLINE 150

/*==========================================================================*/
/*                        VARIABLE TYPE DEFINITIONS                         */
/*==========================================================================*/

typedef int bool_t;           /* boolean, legal values are TRUE and FALSE */

/*============================================================================*/
/*                             PROGRAM PROTOTYPES                             */
/*============================================================================*/

#ifdef NEED_GETOPT_PROTO_TYPE
int   getopt( int, char **, char * );
#endif

/*===========================================================================*/
/*================================= PROGRAMS ================================*/
/*===========================================================================*/

char *trim_space(

    char * line
)
{
   char * end = line + strlen( line ) - 1;
   
   /*------------------------
   * trim trailing whitespace
   *-------------------------*/
   while( line <= end  && isspace( *end  ) ) end--;
   *( end+1 ) = EOS;
   
   return line;
}

/*----------------------------------------------------------------------------*/

static char * get_s( char * line, int limit )
{
    char * what = fgets( line, limit, stdin );
    strtok( line, "\n" );  /* make this feel like a real gets call */
    return what;
}

/*----------------------------------------------------------------------------*/

void align_comments( 

    char * comment_begin,
    char * comment_end
)
{
    char * beg_comment_position;
    char * end_comment_position;
    char *newLines[MAX_LINES];
    int  comment_start[MAX_LINES];
    int  comment_size[MAX_LINES];
    char line[MAXLINE];
    int lineCnt = 0;
    int i, crntLine;
    int lineLen;
    int max_begin = 0;
    int max_end = 0;
    bool_t comment_open = FALSE;
    int c_end_size = ( comment_end != NULL ) ? strlen( comment_end ): 0;
    
    for( i=0; i< MAX_LINES; i++ ) comment_size[i] = 0;
    
    while( get_s( line, MAXLINE ) && lineCnt < MAX_LINES )
    {
        int size_adjust = 0;
        
        /*--------------------------------------
        * Allocate space for a copy of the line.
        *--------------------------------------*/
        lineLen = strlen( line );
        newLines[lineCnt] = calloc( 1, lineLen+1 );
        strcpy( newLines[lineCnt], trim_space( line ) );
        lineLen = strlen( newLines[lineCnt] );
        
        beg_comment_position = strstr( line, comment_begin );
        
        end_comment_position = 
           ( comment_end == NULL ) ? line+lineLen : strstr( line, comment_end );
        
        if( beg_comment_position == NULL && comment_open )
        {
	    int invisibleLen     = strspn( line, " \t" );
	    beg_comment_position = line + invisibleLen;
            size_adjust         += strlen( comment_begin ) + 1;
        }
        
        /*------------------------------------
        * Determine closure status of comment.
        *------------------------------------*/
        if( beg_comment_position != NULL && end_comment_position == NULL )
            comment_open = TRUE;

        if( end_comment_position != NULL )
            comment_open = FALSE;

        /*------------------------------
        * Save start of comment in line.
        *------------------------------*/
        if( beg_comment_position == NULL )
            beg_comment_position = line+lineLen;  /* no comment on line */

        comment_start[ lineCnt ] = beg_comment_position - line;
        
        if( comment_start[ lineCnt ] > max_begin )
            max_begin = comment_start[ lineCnt ];
	
        /*----------------------------
        * Save end of comment in line.
        *----------------------------*/
        if( end_comment_position == NULL )
        {
            end_comment_position = line+lineLen;  /* no comment on line */
            
            if( comment_open )
                size_adjust += strlen( comment_end ) + 1;
        }

        comment_size[ lineCnt ] =  end_comment_position - beg_comment_position;
        
        if( comment_size[ lineCnt ] + size_adjust > max_end )
            max_end = comment_size[ lineCnt ] + size_adjust;
            
        lineCnt++;
    }


    if( lineCnt )
    {
        for( crntLine = 0; crntLine < lineCnt; crntLine++ )
        {
            int sans_size  = c_end_size;
            int pad        = max_begin - comment_start[ crntLine ];
            int pad_adjust = 0;
            int near_end;
            
            /*----------------------
            * Output front of line.
            *----------------------*/
            for( i = 0; i < comment_start[ crntLine ]; i++ )
                putchar( newLines[ crntLine ][i] );
            
            if( comment_size[ crntLine ] > 0 )
            {
                /*---------------------------------
                * Adjust padding for open comments.
                *---------------------------------*/
                if( strstr( newLines[ crntLine ], comment_begin ) == NULL )
                    pad_adjust = strlen( comment_begin ) + 1;

                /*----------------
                * Pad with spaces.
                *----------------*/
                for( i = pad + pad_adjust; i > 0; i-- )
                    putchar( ' ' );
            }

            /*-------------------------------------------------
            * Output remainder of line sans comment terminator.
            *-------------------------------------------------*/
            near_end = comment_start[ crntLine ] + comment_size[ crntLine ];
               
            for( i = comment_start[ crntLine ]; 
                 i < near_end && newLines[ crntLine ][i] != EOS; i++ )
                putchar( newLines[ crntLine ][i] );

            /*--------------------------------
            * Comment terminator on this line?
            *--------------------------------*/
            if( c_end_size > 0 )
                if( strstr( newLines[ crntLine ], comment_end ) == NULL )
                    sans_size = 0;  /* No! */

            /*----------------------------------------------------
            * Fill out comment with spaces, and terminate comment.
            *----------------------------------------------------*/
            if( sans_size > 0 )
            {
                i = max_end - comment_size[ crntLine ] - pad_adjust;
                while( i-- > 0 )
                    putchar( ' ' );
                printf( "%s", comment_end );
            }
            putchar( '\n' );
        }
    }
}

/*----------------------------------------------------------------------------*/

char *
remove_file_path
(
    char * inp_file_path
)
{
    char * file_name = strrchr( inp_file_path, FPATH_SEPARATOR );
    
    return ( file_name ) ? file_name+1 : inp_file_path;
}

/*----------------------------------------------------------------------------*/

int main( int argc, char * argv[] )
{
    extern char *optarg;

    char * program_nm    = remove_file_path( *argv );
    char * comment_begin = getenv( "BOX_COMMENT_BEGIN" );
    char * comment_end   = getenv( "BOX_COMMENT_END" );
    char   c;   
        
    while( ( c = getopt( argc, argv, "e:s:" )) != -1 )
    {
        switch( c )
        {
          case 'e':   /* comment end characters */
            comment_end = optarg;
            break;

          case 's':   /* comment starting characters */
            comment_begin = optarg;
            break;
          
          default:
            printf( "\nUsage: %s [options]\n", program_nm );
            printf( "\n" );
            printf( "  where options are:\n" );
            printf( "\n" );
            printf( "    -s<tarting comment characters>\n" );
            printf( "    -e<nding comment characters>\n" );
            printf( "\n" );
            printf( "  The following environment variables can also be used\n" );
            printf( "  to affect the results of the program.\n" );
            printf( "\n" );
            printf( "    BOX_COMMENT_BEGIN    (same as -s)\n" );
            printf( "    BOX_COMMENT_END      (same as -e)\n" );
            printf( "\n" );
            printf( "  Program options will override the environment variables.\n" );
            printf( "  The default settings for the program are for 'C' language comments.\n" );
            printf( "\n" );
            printf( "  Hint: By supplying only the starting comment character,\n" );
            printf( "        the program assumes that a newline terminates\n" );
            printf( "        the comment (Good for shell programming).\n" );
            printf( "\n" );
            exit( 1 );
            break;
        }
    }
    /*-------------------------------------------
    * If invalid comment end given, don't use it.
    *-------------------------------------------*/
    if( comment_end != NULL && *comment_end == EOS )
        comment_end = NULL;
        
    /*--------------------------------------
    * Default to 'C' language comment style.
    *--------------------------------------*/
    if( comment_begin == NULL || *comment_begin == EOS )
    {
        comment_begin = "/*";
        comment_end   = "*/";
    }
    
    align_comments( comment_begin, comment_end );

    return 0;
}
