/*
  Copyright (C) 2004-2005 Tommi Tervonen, Petteri Klemola, Pasi Orovuo, Marko Mattila

  This file is part of Kajaani Kombat.

  Kajaani Kombat is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  Kajaani Kombat is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with Kajaani Kombat; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "client_net.h"

client_tcpnet::client_tcpnet(IPaddress *server_ip) throw(string &)
{
  socket = SDLNet_TCP_Open (server_ip);
  if (socket == NULL)
    throw string ("Error connecting to server: ") + SDLNet_GetError();
  socketset = SDLNet_AllocSocketSet(1);

  int ret = SDLNet_TCP_AddSocket(socketset, socket);
  assert (ret != -1);

  cond = SDL_CreateCond();
  if (cond <0)
    {
      fprintf (stderr, "Error creating cond variable\n");
      exit(2);
    }
  mutex = SDL_CreateMutex();
  if (mutex < 0)
    {
      fprintf (stderr, "Error creating mutex\n");
      exit(2);
    }

  active = true;
  trans_th = SDL_CreateThread (&client_tcpnet::transf_func, (void *) this);
  if (trans_th < 0)
    {
      fprintf(stderr, "Error starting thread: %s\n", SDL_GetError());
      exit(2);
    }
  input_th = SDL_CreateThread (&client_tcpnet::input_func, (void *) this);
  if (input_th < 0)
    {
      fprintf(stderr, "Error starting thread: %s\n", SDL_GetError());
      exit(2);
    }
}

void client_tcpnet::disconnect()
{
  active = false;
  SDL_CondSignal(cond);
  SDL_WaitThread(trans_th, NULL);
  SDL_WaitThread(input_th, NULL);

  SDLNet_TCP_Close(socket);
  SDLNet_TCP_DelSocket(socketset, socket);
  SDLNet_FreeSocketSet (socketset);
}

client_tcpnet::~client_tcpnet()
{
  disconnect();
  SDL_DestroyCond(cond);
  SDL_DestroyMutex(mutex);
}

msg * client_tcpnet::recv() throw (string &)
{
  while (in.empty())
    SDL_Delay(10);
  if (rec_err.length() > 0) throw rec_err;
  return in.pop();
}

void client_tcpnet::send(msg *m) throw (string &)
{
  out.push(m->deep_copy());
  if (send_err.length() > 0) throw send_err;

  if (SDL_CondSignal(cond) < 0)
    {
      fprintf (stderr, "Error signalling cond variable\n");
      exit(2);
    }
}

bool client_tcpnet::has_msgs() throw (string &)
{
  return !in.empty(); //SDLNet_CheckSockets (socketset, 0) > 0;
}


msg * client_localnet::recv() throw (string &)
{
  return in.pop();
}

void client_localnet::send(msg * m) throw (string &)
{
  out.push(m->deep_copy());
}

bool client_localnet::has_msgs() throw (string &)
{
  return !in.empty();
}

monitor_queue<msg *> * client_localnet::get_in_queue()
{
  return &in;
}

monitor_queue<msg *> * client_localnet::get_out_queue()
{
  return &out;
}

client_localnet::client_localnet()
{

}

void client_localnet::disconnect()
{

}
client_localnet::~client_localnet()
{

}

int client_tcpnet::transf_func(void *data)
{
  client_tcpnet *cl = (client_tcpnet *) data;

  while (cl->active)
    {
      // if queue empty, wait on cond variable
      if (cl->out.empty())
	{
	  if (SDL_mutexP(cl->mutex) == -1)
	    {
	      fprintf (stderr, "Error locking mutex!\n");
	      exit(2);
	    }
	  SDL_CondWait(cl->cond, cl->mutex);
	}
      while (!cl->out.empty() && cl->active)
	{
	  msg *m = 0;
	  if (cl->socket == 0) return 0;
	  try{
	    m = cl->out.pop();
	    m->send(cl->socket);
	  }
	  catch(string &s)
	    {
	      if (m) delete m;
	      m = 0;
	      if (cl->active)
		cl->send_err = s;
	      return 0;
	    }
	  if (m) delete m;
	}
    }
  return 0;
}

int client_tcpnet::input_func(void *data)
{
  client_tcpnet *cl = (client_tcpnet *) data;

  while (cl->active)
    {
      while (SDLNet_CheckSockets(cl->socketset, CHECK_WAIT) > 0 && cl->active)
	{
	  if (!cl->active || cl->socket == 0) return 0;
	  if (SDLNet_SocketReady(cl->socket) == 0) continue;
	  msg *m = 0;
	  try {
	    msg *m = msg::recv(cl->socket);
	    cl->in.push(m);
	  }
	  catch (string &s)
	    {
	      if (m) delete m;
	      if (cl->active)
		cl->rec_err = s;
	      return 0;	      
	    }
	  if (m) delete m;
	}
    }
  return 0;
}
