/*
  Copyright(C) 2002-2007 Pierre Mazire
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

/*
  amlxml.c

AUTHORS  XML Advanced MAME List format files functions
*/

#include "common.h"
#include <stdio.h>
#include <mamory/readinfo.h>
#include <mamory/modeconv.h>
#include <expat.h>
#include <mamory/amlxml.h>

#include <sortbox/sortbox.h>


#define ROMONLY 0
#define TOTALGAME 1

#define NBR_TOKENS 114
#define BUFSIZx2 (BUFSIZ*4)

//unsigned int NbrUnknownTokens=0;
//s_SortBox **SamplesSortBox=NULL;

typedef enum e_Token 
  {
    tk_unknown=0,
    tk_mamory,tk_source,tk_target,tk_targetver,tk_comment,tk_manager,
    tk_forcestoragemode,
    tk_file, tk_exec, tk_directory, tk_admameli,
    tk_clrmamepro, tk_category, tk_version, tk_author,tk_storagemode,
    tk_forcemerging,tk_forcestorageformat, tk_forcezipping,
    tk_game,tk_resource,tk_rom, tk_history,tk_disk,tk_name,tk_merge,
    tk_compsize,tk_size,tk_crc,tk_md5,tk_sample,tk_clone,tk_chip,tk_type,
    tk_flags,tk_clock,tk_video,tk_screen,tk_orientation,tk_vertical,
    tk_horizontal,tk_x,tk_y,tk_colors,tk_freq,tk_sound,
    tk_channels,tk_input,tk_players,tk_control,tk_buttons,
    tk_coins,tk_service,tk_tilt,tk_dipswitch,tk_entry,
    tk_default,tk_driver,tk_status,tk_color,tk_palettesize,
    tk_description,tk_year,tk_manufacturer,tk_cloneof,
    tk_romof,tk_sampleof,tk_opened_par,tk_closed_par,
    tk_region,tk_offs,tk_aspectx,tk_aspecty,tk_blit,tk_index,
    tk_baddump,tk_nodump,tk_sha1,tk_soundonly,
    tk_refresh,tk_yes,tk_no,tk_dipvalue,tk_width,tk_height,
    tk_offset,tk_audio, tk_dispose, tk_mame, tk_good,
    tk_cpu, tk_raster, tk_vector, tk_preliminary, tk_test,
    tk_imperfect, tk_runnable,tk_biosset,tk_bios,
    tk_sourcefile, tk_display, tk_rotate, tk_flipx, tk_0,
    tk_90, tk_180, tk_270, tk_minimum, tk_maximum, tk_sensitivity, 
    tk_keydelta, tk_reverse, tk_isbios
  } e_Token;

unsigned char *XMLTokens[NBR_TOKENS]=
  {
    NULL,
    "mamory","source","target","targetver","comment","manager",
    "forcestoragemode",
    "file","exec","directory","admameli",
    "clrmamepro","category","version","author","storagemode",
    "forcemerging","forcestorageformat","forcezipping",
    "game","resource","rom","history","disk","name","merge","compsize",
    "size","crc","md5","sample","clone","chip","type","flags","clock",
    "video","screen","orientation","vertical","horizontal",
    "x","y","colors","freq","sound","channels",
    "input","players","control","buttons","coins",
    "service","tilt","dipswitch","entry","default",
    "driver","status","color","palettesize","description",
    "year","manufacturer","cloneof","romof","sampleof",
    "(",")","region","offs","aspectx","aspecty","blit","index",
    "baddump","nodump","sha1","soundonly",
    "refresh","yes","no","dipvalue","width","height",
    "offset","audio","dispose","mame","good",
    "cpu","raster","vector","preliminary","test",
    "imperfect","runnable","biosset","bios",
    "sourcefile", "display", "rotate","flipx", "0", 
    "90","180","270","minimum","maximum","sensitivity",
    "keydelta","reverse","isbios"
  };


e_Token GetTokenID(const char *Token)
{
  unsigned int i;

//  printf("--%s--\n",Token);
  for(i=1;i<NBR_TOKENS;i++)
    if(strcmp(Token,XMLTokens[i])==0)
      break;

  if(i!=NBR_TOKENS)
    return i;

  return 0;
  
};

unsigned char *XMLAddSampleToMameSamples(unsigned char *Sample,
					 s_GamesList *GamesList)
{
  unsigned int i;

 for(i=0;i<GamesList->NbrSoundSamples;i++)
  if(strcmp(GamesList->SoundSamples[i],Sample)==0)
      {
	XFREE(Sample);
	return GamesList->SoundSamples[i];
      };

  GamesList->SoundSamples=XREALLOC(GamesList->SoundSamples,
				  unsigned char*,
				  GamesList->NbrSoundSamples+1);

  GamesList->SoundSamples[GamesList->NbrSoundSamples]=Sample;
  GamesList->NbrSoundSamples++;
  return Sample;
};

void XMLCheckGamesListResources(s_GamesList *GamesList,
			     s_GameInfo *GameInfo)
{
  unsigned int i;

  for(i=0;i<GamesList->NbrResources;i++)
    {
      if(strcmp(GamesList->Resources[i]->Name,GameInfo->More->Resource)==0)
	{
	  GamesList->Resources[i]->AssociatedGames=
	    XREALLOC(GamesList->Resources[i]->AssociatedGames,
		    s_GameInfo*,
		    GamesList->Resources[i]->NbrAssociatedGames+1);

	  GamesList->Resources[i]->AssociatedGames[GamesList->Resources[i]->NbrAssociatedGames]=GameInfo;
	  GamesList->Resources[i]->NbrAssociatedGames++;
	  break;
	};
    };
  
  if(i==GamesList->NbrResources)
    {
      GamesList->Resources=XREALLOC(GamesList->Resources,
				    s_ResourceInfo*,
				    GamesList->NbrResources+1);

      GamesList->Resources[GamesList->NbrResources]=XCALLOC(s_ResourceInfo,1);

      GamesList->Resources[GamesList->NbrResources]->Name=XSTRDUP(GameInfo->More->Resource);
      GamesList->Resources[GamesList->NbrResources]->AssociatedGames=
	XREALLOC(GamesList->Resources[GamesList->NbrResources]->AssociatedGames,
		 s_GameInfo*,GamesList->Resources[GamesList->NbrResources]->NbrAssociatedGames+1);

      GamesList->Resources[GamesList->NbrResources]->AssociatedGames[GamesList->Resources[GamesList->NbrResources]->NbrAssociatedGames]=GameInfo;
      GamesList->Resources[GamesList->NbrResources]->NbrAssociatedGames++;
      GamesList->NbrResources++;
    };
};

void
DocTypeStartHandler(void  *userdata,
		    const char *doctypeName,
		    const char *sysid,
		    const char *pubid,
		    int has_internal_subset)
{
  int *Data=(int*)userdata;

  if(strcmp(doctypeName,"mame")==0)
    *Data=TRUE;
  else
    *Data=FALSE;
};

int isAMLXMLFile(const char *Path)
{
  FILE *GamesSourceFile;
  char Buffer[BUFSIZ];
  XML_Parser parser=XML_ParserCreate(NULL);
  int done;
  int Data=-1;

  XML_SetUserData(parser,&Data);
  XML_SetStartDoctypeDeclHandler(parser,DocTypeStartHandler);

  GamesSourceFile=fopen(Path,"r");
  
  do
    {
      size_t len = fread(Buffer,1,sizeof(Buffer),GamesSourceFile);
      done=len<sizeof(Buffer);
      if(XML_Parse(parser,Buffer,len,done)==XML_STATUS_ERROR)
	Data=FALSE;
    } while(Data==-1);

  fclose(GamesSourceFile);
  XML_ParserFree(parser); 
  return Data;
};


static void
inEmptyStartElementHandler(void *userdata, const char *name, const char **attr)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  
  Data->Depth++;

  TokenID=GetTokenID(name);
};

static void
EmptyEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);
};

static void
PCDATAEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;
  
  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);

  switch(TokenID)
    {
    case tk_description:
      Data->GameInfo->More->Description=XSTRDUP(Data->String);
      break;
    case tk_year:
      if (Data->Type!=ROMONLY)
	Data->GameInfo->More->Misc->Year=(short)atoi(Data->String);
      break;
    case tk_manufacturer:
      if (Data->Type!=ROMONLY)
	Data->GameInfo->More->Misc->Manufacturer=XSTRDUP(Data->String);
      break;
    case tk_history:
      break;
    default:
      LPRINTF(DEBUG,"Unknown Element \"%s\"",name);
      Data->GamesList->NbrUnknownTokens++;
      break;
    };
  XFREE(Data->String);
};

static void
RomEndElementHandler(void *userdata, const char *name)
{
  s_RomContent **Contents=NULL;
  unsigned int nbr=0,i,j;
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);  
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);

  if (Data->Rom->Content->CRC!=0)
    Contents=(s_RomContent**)SortBox_FindTheUint(Data->RomContentsSortBox,
						 Data->Rom->Content->CRC,
						 (int*)&nbr);
  if(Contents!=NULL)
    for(i=0;i<nbr;i++)
      if (Contents[i]->Size==Data->Rom->Content->Size)
	break;

  if(Contents==NULL || (Contents!=NULL && i==nbr))
    {
      if(Data->Rom->Content->CRC!=0)
	SortBox_AddOneUint(Data->RomContentsSortBox,
			   Data->Rom->Content->CRC,
			   Data->Rom->Content); 
      if (Data->Rom->Content->CRC!=0)
	{
	  Data->GamesList->RomContents=XREALLOC(Data->GamesList->RomContents,
						s_RomContent*,
						Data->GamesList->NbrRomContents+1);

	  Data->GamesList->RomContents[Data->GamesList->NbrRomContents]=
	    Data->Rom->Content;
	  Data->GamesList->NbrRomContents++;
	};
    }
  else
    {
      for (j=0;j<Data->Rom->Content->NbrAssociatedGames;j++)
	{
	  Data->Rom->Content->AssociatedGames[j]->Game=NULL;
	  XFREE(Data->Rom->Content->AssociatedGames[j]);
	};
      XFREE(Data->Rom->Content->AssociatedGames);
      XFREE(Data->Rom->Content);
      Data->Rom->Content=Contents[i];
    };

  for(i=0;i<nbr;i++)
    Contents[i]=NULL;
  XFREE(Contents);

  if(Data->Rom!=NULL)
    {
      if(Data->Rom->Merge!=NULL || Data->Rom->GameName!=NULL)
	{
	  if(Data->Rom->GameName==NULL)
	    {
	      if(Data->GameInfo->More->CloneOf!=NULL)
		Data->Rom->GameName=XSTRDUP(Data->GameInfo->More->CloneOf);
	      else
		Data->Rom->GameName=XSTRDUP(Data->GameInfo->Name);
	    };
	  switch(Data->Rom->Source->StorageMode)
	    {
	    case STMODE_SPLIT: 
	      if(Data->Rom->Merge==NULL &&
		 strcmp(Data->Rom->GameName,Data->GameInfo->Name)!=0)
		Data->Rom->Merge=XSTRDUP(Data->Rom->Name);
	      AddRomToGameMergedRoms(Data->GameInfo,Data->Rom);
	      break;
	    case STMODE_MERGE: 
	      if(Data->GameInfo->More->Resource!=NULL &&
		 strcmp(Data->Rom->GameName,Data->GameInfo->More->Resource)==0)
		AddRomToGameMergedRoms(Data->GameInfo,Data->Rom);
	      else
		{
		  Data->Rom=AddRomToRomSharedRoms(Data->GameInfo,Data->Rom,NULL);
		  if(Data->Rom!=NULL)
		    AddRomToGameRoms(Data->GameInfo,Data->Rom);
		}
	      break;
	    case STMODE_FULL: 
	      if(Data->Rom->Merge==NULL &&
		 strcmp(Data->Rom->GameName,Data->GameInfo->Name)!=0)
		Data->Rom->Merge=XSTRDUP(Data->Rom->Name);
	      AddRomToGameRoms(Data->GameInfo,Data->Rom);
	      break;
	    };
	}
      else
	{
	  if(Data->Rom->GameName==NULL)
	    Data->Rom->GameName=XSTRDUP(Data->GameInfo->Name);
	  switch(Data->Rom->Source->StorageMode)
	    {
	    case STMODE_MERGE:
	      //		      if (strcmp(Rom->GameName,GameInfo->Name)!=0)
	      //			{
	      Data->Rom=AddRomToRomSharedRoms(Data->GameInfo,Data->Rom,NULL);
	      if(Data->Rom==NULL)
		break;
	      //			};
	    case STMODE_SPLIT:
	    case STMODE_FULL:
	      AddRomToGameRoms(Data->GameInfo,Data->Rom);
	      break;
	    };
	};      
    };

  Data->Rom=NULL;
};

static void
DiskEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);  
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);

  Data->GameInfo->Disks=XREALLOC(Data->GameInfo->Disks,s_DiskInfo*,
				 Data->GameInfo->NbrDisks+1);
  Data->GameInfo->Disks[Data->GameInfo->NbrDisks]=Data->Disk;
  Data->GameInfo->NbrDisks++;  

  Data->Disk=NULL;
};

static void
DriverEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);

  if(Data->Type!=ROMONLY)
    Data->GameInfo->More->Misc->Driver=Data->Driver;

  Data->Driver=NULL;
};

static void
ChipEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);

  if(Data->Type!=ROMONLY)
    {
      Data->GameInfo->More->Misc->Chips=
	XREALLOC(Data->GameInfo->More->Misc->Chips,
		 s_ChipInfo*,
		 Data->GameInfo->More->Misc->NbrChips+1);
      
      Data->GameInfo->More->Misc->Chips[Data->GameInfo->More->Misc->NbrChips]=
	Data->Chip;
      Data->GameInfo->More->Misc->NbrChips++;     
    };

  Data->Chip=NULL;
};  

static void
DisplayEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);
  if(Data->Type!=ROMONLY)
    {
      Data->GameInfo->More->Misc->Displays=
	XREALLOC(Data->GameInfo->More->Misc->Displays,
		 s_DisplayInfo*,
		 Data->GameInfo->More->Misc->NbrDisplays+1);
      Data->GameInfo->More->Misc->Displays[Data->GameInfo->More->Misc->NbrDisplays]=Data->Display;
      Data->GameInfo->More->Misc->NbrDisplays++;
    };

  Data->Display=NULL;
};

static void
SoundEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);

  if(Data->Type!=ROMONLY)
    Data->GameInfo->More->Misc->Sound=Data->Sound;

  Data->Sound=NULL;
};

static void
InputEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);

  if(Data->Type!=ROMONLY)
    Data->GameInfo->More->Misc->Input=Data->Input;

  Data->Input=NULL;
};

static void
inInputStartElementHandler(void *userdata, const char *name, const char **attr)
{
  unsigned int i;
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  
  Data->Depth++;

  TokenID=GetTokenID(name);
  switch(TokenID)
    {
    case tk_control:
      if(Data->Type!=ROMONLY)
	{
	  Data->Input->Controls=XREALLOC(Data->Input->Controls,
					    s_ControlInfo*,
					    Data->Input->NbrControls+1);
	  for(i=0;attr[i];i+=2)
	    {
	      TokenID=GetTokenID(attr[i]);
	      switch(TokenID)
		{
		case tk_type:
		  Data->Input->Controls[Data->Input->NbrControls]->Type=
		    XSTRDUP(attr[i+1]);
		  break;
		case tk_minimum:
		  Data->Input->Controls[Data->Input->NbrControls]->Minimum=
		    atoi(attr[i+1]);
		  break;
		case tk_maximum:
		  Data->Input->Controls[Data->Input->NbrControls]->Maximum=
		    atoi(attr[i+1]);
		  break;
		case tk_sensitivity:
		  Data->Input->Controls[Data->Input->NbrControls]->Sensitivity=
		    atoi(attr[i+1]);
		  break;
		case tk_keydelta:
		  Data->Input->Controls[Data->Input->NbrControls]->Keydelta=
		    atoi(attr[i+1]);
		  break;
		case tk_reverse:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_yes:
		      Data->Input->Controls[Data->Input->NbrControls]->Reverse=
			TRUE;
		      break;
		    case tk_no:
		      Data->Input->Controls[Data->Input->NbrControls]->Reverse=
			FALSE;
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in DipSwitch - default",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Input - control",attr[i]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	    };
	  Data->Input->NbrControls++;
	};   
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,EmptyEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=EmptyEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_control;
      break;
    default:
      LPRINTF(DEBUG,"Unknown Token \"%s\" in Input",name);
      Data->GamesList->NbrUnknownTokens++;
      break;
    };
}

static void
inDipSwitchStartElementHandler(void *userdata, const char *name, const char **attr)
{
  unsigned int i;
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  
  Data->Depth++;

  TokenID=GetTokenID(name);
  switch(TokenID)
    {
    case tk_dipvalue:
      if(Data->Type!=ROMONLY)
	{
	  Data->DipSwitch->Entries=XREALLOC(Data->DipSwitch->Entries,
					    unsigned char*,
					    Data->DipSwitch->NbrEntries+1);
	  for(i=0;attr[i];i+=2)
	    {
	      TokenID=GetTokenID(attr[i]);
	      switch(TokenID)
		{
		case tk_name:
		  Data->DipSwitch->Entries[Data->DipSwitch->NbrEntries]=
		    XSTRDUP(attr[i+1]); 
		  break;
		case tk_default:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_yes:
		      Data->DipSwitch->Default=XSTRDUP(Data->DipSwitch->Entries[Data->DipSwitch->NbrEntries]);
		      break;
		    case tk_no:
		      Data->DipSwitch->Default=NULL;
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in DipSwitch - default",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in DipSwitch - dipvalue",attr[i]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	    };
	  Data->DipSwitch->NbrEntries++;
	};   
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,EmptyEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=EmptyEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_dipvalue;
      break;
    default:
      LPRINTF(DEBUG,"Unknown Token \"%s\" in DipSwitch",name);
      Data->GamesList->NbrUnknownTokens++;
      break;
    };
};

static void
DipSwitchEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);

  if(Data->Type!=ROMONLY)
    {
      Data->GameInfo->More->Misc->DipSwitches=
	XREALLOC(Data->GameInfo->More->Misc->DipSwitches,
		 s_DipSwitchInfo*,
		 Data->GameInfo->More->Misc->NbrDipSwitches+1);
      
      Data->GameInfo->More->Misc->DipSwitches[Data->GameInfo->More->Misc->NbrDipSwitches]=Data->DipSwitch;
      Data->GameInfo->More->Misc->NbrDipSwitches++;
    };

  Data->DipSwitch=NULL;
};



static void
inGameStartElementHandler(void *userdata, const char *name, const char **attr)
{
  e_Token TokenID;
  unsigned int i;
  s_XMLUserData *Data=userdata;


  Data->Depth++;
  TokenID=GetTokenID(name);
  switch(TokenID)
    {
    case tk_description:
    case tk_year:
    case tk_manufacturer:
    case tk_history:
      XFREE(Data->String);
      XML_SetEndElementHandler(Data->Parser,PCDATAEndElementHandler);
      //Data->EndHandlerStack[Data->Depth]=PCDATAEndElementHandler;
      break;
    case tk_biosset:
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,EmptyEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=EmptyEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_biosset;
      break;        
    case tk_rom:
      Data->Rom=XCALLOC(s_RomInfo,1);
      Data->Rom->Source=Data->GamesList->Sources[0];
      //  Data->Rom->StorageFormat=STFMT_UNSET;
      Data->Rom->Status=UNKNOWN_DUMP;
      if (Data->Type==TOTALGAME)
	{
	  Data->Rom->Hardware=XCALLOC(s_RomHardwareInfo,1);
	  
	  Data->Rom->Hardware->Region=NULL;
	  Data->Rom->Hardware->Flags=NULL;
	  Data->Rom->Hardware->Offset=0;
	}
      else
	Data->Rom->Hardware=NULL;
      
      Data->Rom->Content=XCALLOC(s_RomContent,1);
      
      for(i=0;attr[i];i+=2)
	{
	  TokenID=GetTokenID(attr[i]);
	  switch(TokenID)
	    {
	    case tk_name:
	      Data->Rom->Name=XSTRDUP(attr[i+1]);
	      break;
	    case tk_bios:
	      break;
	    case tk_size:
	      Data->Rom->Content->Size=atoi(attr[i+1]);
	      break;
	    case tk_crc:
	      Data->Rom->Content->CRC=strtoul(attr[i+1],NULL,16);
	      if(Data->Rom->Content->CRC==0)
		Data->Rom->Status=NEED_REDUMP;
	      break;
	    case tk_md5:
	      break;
	    case tk_sha1:
	      break;
	    case tk_merge:
	      Data->Rom->Merge=XSTRDUP(attr[i+1]);
	      break;
	    case tk_region:
	      if (Data->Type==TOTALGAME)
		Data->Rom->Hardware->Region=XSTRDUP(attr[i+1]);
	      break;
	    case tk_offset:
	      if (Data->Type==TOTALGAME)
		Data->Rom->Hardware->Offset=atoi(attr[i+1]);
	      break;
	    case tk_status:
	      TokenID=GetTokenID(attr[i+1]);
	      switch(TokenID)
		{
		case tk_good:
		  break;
		case tk_nodump:
		  Data->Rom->Content->CRC=0;
		  Data->Rom->Status=NEED_REDUMP;
		  break;
		case tk_baddump:
		  Data->Rom->Status=NO_GOOD_DUMP_KNOWN;
		  break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Rom - status",attr[i+1]);
		  Data->GamesList->NbrUnknownTokens++;
		};
	      break;
	    case tk_dispose:
	      if(Data->Type==TOTALGAME)
		{
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_yes:
		      Data->Rom->Hardware->Flags=XSTRDUP(attr[i]);
		      break;
		    case tk_no:
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Rom - dispose",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		};
	      break;
	    case tk_soundonly:
	      if(Data->Type==TOTALGAME)
		{
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_yes:
		      Data->Rom->Hardware->Flags=XSTRDUP(attr[i]);
		      break;
		    case tk_no:
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Rom - soundonly",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		};
	      break;
	    default:
	      LPRINTF(DEBUG,"Unknown Token \"%s\" in Rom",attr[i]);
	      Data->GamesList->NbrUnknownTokens++;
	      break;
	    };
	};	
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,RomEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=RomEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_rom;
      break;  
    case tk_disk:
      Data->Disk=XCALLOC(s_DiskInfo,1);
      if(Data->Type==TOTALGAME)
	Data->Disk->Hardware=XCALLOC(s_DiskHardwareInfo,1);

      for(i=0;attr[i];i+=2)
	{
	  TokenID=GetTokenID(attr[i]);
	  switch(TokenID)
	    {
	    case tk_name:
	      Data->Disk->Name=XSTRDUP(attr[i+1]);
	      break;
	    case tk_md5:
	      Data->Disk->md5=XSTRDUP(attr[i+1]);
	      break;
	    case tk_sha1:
	      break;
	    case tk_region:
	      if(Data->Type==TOTALGAME)
		Data->Disk->Hardware->Region=XSTRDUP(attr[i+1]);
	      break;
	    case tk_index:
	      if(Data->Type==TOTALGAME)
		Data->Disk->Hardware->Index=atoi(attr[i+1]);
	      break;
	     case tk_status:
	     case tk_merge:
	       break;
	    default:
	      LPRINTF(DEBUG,"Unknown Token \"%s\" in Disk",attr[i]);
	      Data->GamesList->NbrUnknownTokens++;
	      break;
	    };
	};
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,DiskEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=DiskEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_disk;
      break;
    case tk_sample:
      for(i=0;attr[i];i+=2)
	{
	  TokenID=GetTokenID(attr[i]);
	  switch(TokenID)
	    {
	    case tk_name:
	      {
		unsigned char *sample;
		sample=XSTRDUP(attr[i+1]);
		sample=XMLAddSampleToMameSamples(sample,Data->GamesList);
		Data->GameInfo->More->Samples=XREALLOC(Data->GameInfo->More->Samples,
						 unsigned char*,
						 Data->GameInfo->More->NbrSamples+1);
		
		Data->GameInfo->More->Samples[Data->GameInfo->More->NbrSamples]=sample;
		Data->GameInfo->More->NbrSamples++;
		break;
	      };
	    default:
	      LPRINTF(DEBUG,"Unknown Token \"%s\" in Sample",attr[i]);
	      Data->GamesList->NbrUnknownTokens++;
	      break;
	    };
	};
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,EmptyEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=EmptyEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_sample;
      break;
    case tk_chip:
      if(Data->Type!=ROMONLY)
	{
	  Data->Chip=XCALLOC(s_ChipInfo,1);
	  for(i=0;attr[i];i+=2)
	    {
	      TokenID=GetTokenID(attr[i]);
	      switch(TokenID)
		{
		case tk_name:
		  Data->Chip->Name=XSTRDUP(attr[i+1]);
		  break;		  
		case tk_type:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_cpu:
		    case tk_audio:
		      Data->Chip->Type=XSTRDUP(attr[i+1]);
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Chip - type",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    }
		  break;
		case tk_soundonly:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_yes:
		      Data->Chip->Flags=XSTRDUP(attr[i]);
		      break;
		    case tk_no:
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Chip - soundonly",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_clock:
		  Data->Chip->Clock=atoi(attr[i+1]);
		  break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Chip",attr[i]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	    };
	};	  
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,ChipEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=ChipEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_chip;
      break;
    case tk_video: /* replaced by display in mame 0.107 */
      if(Data->Type!=ROMONLY)
	{
	  Data->Display=XCALLOC(s_DisplayInfo,1);
	  for(i=0;attr[i];i+=2)
	    {
	      TokenID=GetTokenID(attr[i]);
	      switch(TokenID)
		{
		case tk_screen:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_raster:
		    case tk_vector:		      
		      Data->Display->Type=XSTRDUP(attr[i+1]);
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Video - screen",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_orientation:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_horizontal:
		      Data->Display->Rotate=HORIZONTAL;
		      break;
		    case tk_vertical:
		      Data->Display->Rotate=VERTICAL;
		      break; 
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Video - orientation",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_width:
		  Data->Display->X=atoi(attr[i+1]);	
		  break;
		case tk_height:
		  Data->Display->Y=atoi(attr[i+1]);      
		  break;
		case tk_aspectx:
		  Data->Display->AspectX=(unsigned char)atoi(attr[i+1]); 
		  break;
		case tk_aspecty:
		  Data->Display->AspectY=(unsigned char)atoi(attr[i+1]); 
		  break;
		case tk_refresh:
		  Data->Display->Freq=atof(attr[i+1]);
		  break;		
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Video",attr[i]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	    };
	};	  
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,DisplayEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=DisplayEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_video;
      break;
    case tk_display: /* replaces video in mame 0.107 */
      if(Data->Type!=ROMONLY)
	{
	  Data->Display=XCALLOC(s_DisplayInfo,1);
	  for(i=0;attr[i];i+=2)
	    {
	      TokenID=GetTokenID(attr[i]);
	      switch(TokenID)
		{
		case tk_type:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_raster:
		    case tk_vector:		      
		      Data->Display->Type=XSTRDUP(attr[i+1]);
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Video - screen",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_rotate:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_0:
		      Data->Display->Rotate=ROTATE_0;
		      break;
		    case tk_90:
		      Data->Display->Rotate=ROTATE_90;
		      break; 
		    case tk_180:
		      Data->Display->Rotate=ROTATE_180;
		      break;
		    case tk_270:
		      Data->Display->Rotate=ROTATE_270;
		      break; 		      
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Video - orientation",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_width:
		  Data->Display->X=atoi(attr[i+1]);	
		  break;
		case tk_height:
		  Data->Display->Y=atoi(attr[i+1]);      
		  break;
		case tk_flipx:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_yes:
		      Data->Display->Flipx=TRUE;
		      break;
		    case tk_no:
		       Data->Display->Flipx=FALSE;
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Video - orientation",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_refresh:
		  Data->Display->Freq=atof(attr[i+1]);
		  break;		
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Video",attr[i]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	    };
	};	  
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,DisplayEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=DisplayEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_display;
      break;      
    case tk_sound:
      if(Data->Type!=ROMONLY)
	{
	  Data->Sound=XCALLOC(s_SoundInfo,1);
	  for(i=0;attr[i];i+=2)
	    {
	      TokenID=GetTokenID(attr[i]);
	      switch(TokenID)
		{
		case tk_channels:
		  Data->Sound->Channels=(unsigned char)atoi(attr[i+1]);
		  break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Sound ",attr[i]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	    };
	};

      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,SoundEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=SoundEndElementHandler;	  
      Data->ElementIDStack[Data->Depth]=tk_sound;
      break;
    case tk_input:
      if(Data->Type!=ROMONLY)
	{
	  Data->Input=XCALLOC(s_InputInfo,1);
	  for(i=0;attr[i];i+=2)
	    {
	      TokenID=GetTokenID(attr[i]);
	      switch(TokenID)
		{
		case tk_service:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_yes:
		    case tk_no:		      
		      Data->Input->Service=XSTRDUP(attr[i+1]);
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Input - service",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_tilt:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_yes:
		    case tk_no:		      
		      Data->Input->Tilt=XSTRDUP(attr[i+1]);
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Input - tilt",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_players:
		  Data->Input->Players=(unsigned char)atoi(attr[i+1]);
		  break;
		case tk_control: 
		  /* replaced by control element in mame 0.107 */
		  Data->Input->Controls=XCALLOC(s_ControlInfo*,1);
		  Data->Input->Controls[0]=XCALLOC(s_ControlInfo,1);
		  Data->Input->Controls[0]->Type=XSTRDUP(attr[i+1]); 
		  Data->Input->NbrControls=1;
		  break;
		case tk_buttons:
		  Data->Input->Buttons=(unsigned char)atoi(attr[i+1]);
		  break;
		case tk_coins:
		  Data->Input->Coins=(unsigned char)atoi(attr[i+1]);
		  break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Input",attr[i]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	    };
	};	  
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,InputEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=InputEndElementHandler;      
      Data->ElementIDStack[Data->Depth]=tk_input;
      break;
    case tk_dipswitch:
      if(Data->Type!=ROMONLY)
	{
	  Data->DipSwitch=XCALLOC(s_DipSwitchInfo,1);
	  for(i=0;attr[i];i+=2)
	    {
	      TokenID=GetTokenID(attr[i]);
	      switch(TokenID)
		{
		case tk_name:
		   Data->DipSwitch->Name=XSTRDUP(attr[i+1]);
		   break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in DipSwitch",attr[i]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	    };
	};
      XML_SetStartElementHandler(Data->Parser,inDipSwitchStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inDipSwitchStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,DipSwitchEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=DipSwitchEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_dipswitch;
      break;
    case tk_driver:
      if(Data->Type!=ROMONLY)
	{
	  Data->Driver=XCALLOC(s_DriverInfo,1);
	  for(i=0;attr[i];i+=2)
	    {
	      TokenID=GetTokenID(attr[i]);
	      switch(TokenID)
		{
		case tk_status:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_good:
		    case tk_preliminary:
		    case tk_test:
		      Data->Driver->Status=XSTRDUP(attr[i+1]);
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Driver - status",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_color:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_good:
		    case tk_imperfect:
		    case tk_preliminary:
		      Data->Driver->Color=XSTRDUP(attr[i+1]);
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Driver - color",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_sound:
		  TokenID=GetTokenID(attr[i+1]);
		  switch(TokenID)
		    {
		    case tk_good:
		    case tk_imperfect:
		    case tk_preliminary:		  
		      Data->Driver->Sound=XSTRDUP(attr[i+1]);
		      break;
		    default:
		      LPRINTF(DEBUG,"Unknown Token \"%s\" in Driver - sound",attr[i+1]);
		      Data->GamesList->NbrUnknownTokens++;
		      break;
		    };
		  break;
		case tk_palettesize:
		  Data->Driver->PaletteSize=(unsigned char)atoi(attr[i+1]);
		  break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Driver",attr[i]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	    };
	};
      XML_SetStartElementHandler(Data->Parser,inEmptyStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inEmptyStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,DriverEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=DriverEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_driver;
      break;
    default:
      LPRINTF(DEBUG,"Unknown Token \"%s\" in Game",name);
      Data->GamesList->NbrUnknownTokens++;
      break;
    };
};

static void
GameEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_GameInfo **Games=NULL;
  unsigned int nbrdata=0,i;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);

  Games=(s_GameInfo**)SortBox_FindTheWord(Data->GamesSortBox,
					  Data->GameInfo->Name,
					  (int*)&nbrdata);
  if(nbrdata==0)
    {
      Data->GamesList->Games=XREALLOC(Data->GamesList->Games,
				      s_GameInfo*,
				      Data->GamesList->NbrGames+1);
      
      Data->GamesList->Games[Data->GamesList->NbrGames]=Data->GameInfo;
      Data->GamesList->NbrGames++;
      SortBox_AddOneWord(Data->GamesSortBox,Data->GameInfo->Name,Data->GameInfo);
    }
  for(i=0;i<nbrdata;i++)
    Games[i]=NULL;
  XFREE(Games);

  Data->GameInfo=NULL;
};

static void
ResourceEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  unsigned int i;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);

  Data->GamesList->Games=XREALLOC(Data->GamesList->Games,
				  s_GameInfo*,
				  Data->GamesList->NbrGames+1);
  
  Data->GamesList->Games[Data->GamesList->NbrGames]=Data->GameInfo;
  Data->GamesList->NbrGames++;
  SortBox_AddOneWord(Data->GamesSortBox,Data->GameInfo->Name,Data->GameInfo);
  
  for(i=0;i<Data->GamesList->NbrResources;i++)
    if(strcmp(Data->GameInfo->Name,Data->GamesList->Resources[i]->Name)==0)
      {
	Data->GamesList->Resources[i]->Game=Data->GameInfo;
	break;
      };
  
  if(i==Data->GamesList->NbrResources)
    {
      Data->GamesList->Resources=XREALLOC(Data->GamesList->Resources,
					  s_ResourceInfo*,
					  Data->GamesList->NbrResources+1);
      
      Data->GamesList->Resources[Data->GamesList->NbrResources]=
	XCALLOC(s_ResourceInfo,1);
      Data->GamesList->Resources[Data->GamesList->NbrResources]->Name=
	XSTRDUP(Data->GameInfo->Name);
      Data->GamesList->Resources[Data->GamesList->NbrResources]->Game=
	Data->GameInfo;
      Data->GamesList->NbrResources++;
    };
  
  Data->GameInfo=NULL;
};

static void
inGameListStartElementHandler(void *userdata, const char *name, const char **attr)
{
  e_Token TokenID;
  s_GameInfo **Games=NULL;
  unsigned int i,nbrdata=0;
  s_XMLUserData *Data=userdata;


  Data->Depth++;
  
  TokenID=GetTokenID(name);
  switch(TokenID)
    {
    case tk_game:
      Data->GameInfo=XCALLOC(s_GameInfo,1);
      Data->GameInfo->More=XCALLOC(s_MoreGameInfo,1);
      
      if(Data->Type==TOTALGAME)
	Data->GameInfo->More->Misc=XCALLOC(s_MiscGameInfo,1);
      for(i=0;attr[i];i+=2)
	{
	  TokenID=GetTokenID(attr[i]);
	  switch(TokenID)
	    {
	    case tk_name:
	      Data->GameInfo->Name=XSTRDUP(attr[i+1]);
	      break;
	    case tk_sourcefile:
	      break;
        case tk_isbios:
          TokenID=GetTokenID(attr[i+1]);
	      switch(TokenID)
		{
		case tk_yes:
		  XML_SetEndElementHandler(Data->Parser,ResourceEndElementHandler);
		  Data->EndHandlerStack[Data->Depth]=ResourceEndElementHandler;
		  break;
		case tk_no:
		  XML_SetEndElementHandler(Data->Parser,GameEndElementHandler);
		  Data->EndHandlerStack[Data->Depth]=GameEndElementHandler;
		  break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Game",attr[i+1]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	      break;
    case tk_runnable:
	      TokenID=GetTokenID(attr[i+1]);
          /* in 0.117u2, isbios and runnable are both defined in the DTD,
           * but only isbios is used. This mean that runnable attribute
           * is set to "yes" for all games */
          if(Data->EndHandlerStack[Data->Depth]==ResourceEndElementHandler)
            break;
	      switch(TokenID)
		{
		case tk_no :
		  XML_SetEndElementHandler(Data->Parser,ResourceEndElementHandler);
		  Data->EndHandlerStack[Data->Depth]=ResourceEndElementHandler;
		  break;
		case tk_yes:
		  XML_SetEndElementHandler(Data->Parser,GameEndElementHandler);
		  Data->EndHandlerStack[Data->Depth]=GameEndElementHandler;
		  break;
		default:
		  LPRINTF(DEBUG,"Unknown Token \"%s\" in Game",attr[i+1]);
		  Data->GamesList->NbrUnknownTokens++;
		  break;
		};
	      break;
	    case tk_cloneof:
	      Data->GameInfo->More->CloneOf=XSTRDUP(attr[i+1]);
	      break;
	    case tk_romof:
	      Data->GameInfo->More->RomOf=XSTRDUP(attr[i+1]);
	      if(Data->GameInfo->More->Resource==NULL)
		{
		  if(Data->GameInfo->More->CloneOf==NULL)
		    {
		      Data->GameInfo->More->Resource=XSTRDUP(attr[i+1]); 
		      XMLCheckGamesListResources(Data->GamesList,Data->GameInfo);	  
		    }
		  else
		    {
		      nbrdata=1;
		      Games=SortBox_FindTheWord(Data->GamesSortBox,
						Data->GameInfo->More->CloneOf,
						(int*)&nbrdata);
		      if(nbrdata==1)
			{
			  if(Games[0]->More->Resource!=NULL)
			    {
			      Data->GameInfo->More->Resource=XSTRDUP(Games[0]->More->Resource);
			      XMLCheckGamesListResources(Data->GamesList,Data->GameInfo);
			    };
			  
			}
		      else
			{
			  // stocker les games dans liste temporaire
			  Data->TempGames=XREALLOC(Data->TempGames,s_GameInfo*,Data->NbrTempGames+1);
			  
			  Data->TempGames[Data->NbrTempGames]=Data->GameInfo;
			  Data->NbrTempGames++;
			};
		      XFREE(Games);
		    };
		}; 
	      break;
	    case tk_sampleof:
	      Data->GameInfo->More->SampleOf=XSTRDUP(attr[i+1]);
	      break;
	    default:
	      LPRINTF(DEBUG,"Unknown Token \"%s\" in Game",attr[i]);
	      Data->GamesList->NbrUnknownTokens++;
	      break;
	    };
	};
      XML_SetStartElementHandler(Data->Parser,inGameStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inGameStartElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_game;
      break;
    default:
      LPRINTF(DEBUG,"Unknown Token \"%s\" in Game",name);
      Data->GamesList->NbrUnknownTokens++;
      break;
    };
};

static void
GameListEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

  XML_SetEndElementHandler(Data->Parser,Data->EndHandlerStack[Data->Depth]);
  XML_SetStartElementHandler(Data->Parser,Data->StartHandlerStack[Data->Depth]);
};

static void
inRootStartElementHandler(void *userdata, const char *name, const char **attr)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;
  s_GamesListSource *DefaultSource;

  Data->Depth++;

  TokenID=GetTokenID(name);

  switch(TokenID)
    {
    case tk_mame:
      Data->GamesList=XMALLOC(s_GamesList,1);
      
      Data->GamesList->MergedGamesList=FALSE;
      Data->GamesList->NbrSources=0;
      Data->GamesList->Sources=NULL;
      DefaultSource=XMALLOC(s_GamesListSource,1);
      
      DefaultSource->GamesList=Data->GamesList;
      DefaultSource->Type=Data->SourceType;
      DefaultSource->Source=XSTRDUP(Data->Source);
      DefaultSource->Target=NULL;
      DefaultSource->TargetVersion=NULL;
      DefaultSource->Author=NULL;
      DefaultSource->Version=NULL;
      DefaultSource->Comment=NULL;
      DefaultSource->StorageMode=STMODE_SPLIT;
      DefaultSource->ForceStorageMode=STMODE_UNSET;
      //  DefaultSource->StorageFormat=STFMT_UNSET;
      DefaultSource->ForceStorageFormat=STFMT_UNSET;
      Data->GamesList->NbrUnknownTokens=0;

      Data->GamesList->NbrResources=0;
      Data->GamesList->Resources=NULL;
      Data->GamesList->NbrGames=0;
      Data->GamesList->Games=NULL;
      Data->GamesList->NbrRomContents=0;
      Data->GamesList->RomContents=NULL;
      Data->GamesList->NbrSoundSamples=0;
      Data->GamesList->SoundSamples=NULL;
      
      Data->GamesSortBox=InitSortBox(SB_ALPHANUM,NULL);
      Data->RomContentsSortBox=InitSortBox(SB_NUM,NULL);
      //  SamplesSortBox=InitSortBox(NULL);
      
      Data->GamesList->Sources=XREALLOC(Data->GamesList->Sources,
				  s_GamesListSource*,
				  Data->GamesList->NbrSources+1);
      
      Data->GamesList->Sources[Data->GamesList->NbrSources]=DefaultSource;
      Data->GamesList->NbrSources++;      

      XML_SetStartElementHandler(Data->Parser,inGameListStartElementHandler);
      Data->StartHandlerStack[Data->Depth]=inGameListStartElementHandler;
      XML_SetEndElementHandler(Data->Parser,GameListEndElementHandler);
      Data->EndHandlerStack[Data->Depth]=GameListEndElementHandler;
      Data->ElementIDStack[Data->Depth]=tk_mame;
      break;
    default:
      LPRINTF(DEBUG,"Unknown Token \"%s\" in Root",name);
      Data->GamesList->NbrUnknownTokens++;
      break;
    };
}

static void
RootEndElementHandler(void *userdata, const char *name)
{
  e_Token TokenID;
  s_XMLUserData *Data=userdata;  

  Data->Depth--;
  TokenID=GetTokenID(name);

};

static void
CharacterDataHandler(void *userdata,const char *s,int len)
{
  s_XMLUserData *Data=userdata;
  unsigned int stringlen;

//  printf("--%s--",Data->String);
  if(!Data->String)
    stringlen=0;
  else
    stringlen=strlen(Data->String);

  while(len!=0 && s[len-1]==' ')
    len--;
  Data->String=XREALLOC(Data->String,unsigned char,stringlen + len +1);
  strncpy(&Data->String[stringlen],s,len);
  Data->String[stringlen+len]=0;
};


s_GamesList *GetGamesListFromAMLXMLFile(unsigned char *File,
					unsigned char *Source,
					unsigned int SourceType)
{
  FILE *GamesSourceFile;
  s_GamesList *GamesList=NULL;
  //s_GamesListSource *NewSource=NULL,*DefaultSource=NULL;
  s_GameInfo **Games=NULL;
  s_RomInfo** roms;
  unsigned int nbrdata=1,i,j,k,l,m;
  //  e_Token TokenID=0;
  unsigned char *Token=NULL;

  char Buffer[BUFSIZ];
  XML_Parser parser=XML_ParserCreate(NULL);
  s_XMLUserData *Data;
  int done;


  if(!parser)
    {
      LPRINTF(WARNING,"Cannot allocate memory for XML parser");
      return NULL;
    };

  Data=XCALLOC(s_XMLUserData,1);
  Data->Parser=parser;
  Data->Depth=0;
  Data->String=NULL;
  Data->Source=Source;
  Data->SourceType=SourceType;
  Data->StartHandlerStack[Data->Depth]=NULL;
  Data->EndHandlerStack[Data->Depth]=RootEndElementHandler;
  Data->ElementIDStack[Data->Depth]=tk_unknown;
  Data->Type=ROMONLY;
  //Data->Type=TOTALGAME;

  XML_SetUserData(parser,Data);
  XML_SetElementHandler(parser,inRootStartElementHandler,RootEndElementHandler);
  XML_SetCharacterDataHandler(parser,CharacterDataHandler);

  GamesSourceFile=fopen(File,"r");
  
  do
    {
      size_t len = fread(Buffer,1,sizeof(Buffer),GamesSourceFile);
      done=len<sizeof(Buffer);
      if(XML_Parse(parser,Buffer,len,done)==XML_STATUS_ERROR)
	{
	  LPRINTF(WARNING,"%s at line %d",
		  XML_ErrorString(XML_GetErrorCode(parser)),
		  XML_GetCurrentLineNumber(parser));
	  return NULL;
	};
    } while(!done);
  XML_ParserFree(parser);
  Data->Parser=NULL;
  
  GamesList=Data->GamesList;
  
  for(i=0;i<Data->NbrTempGames;i++)
    {
      //      printf("Game %s",Data->TempGames[i]->Name);
      nbrdata=1;
      Games=SortBox_FindTheWord(Data->GamesSortBox,
				Data->TempGames[i]->More->CloneOf,
				(int*)(&nbrdata));
      if(nbrdata==1)
	{
	  if(Games[0]->More->Resource!=NULL)
	    {
	      Data->TempGames[i]->More->Resource=XSTRDUP(Games[0]->More->Resource);
	      XMLCheckGamesListResources(GamesList,Data->TempGames[i]);
	    };	      
	};
//      printf("\n");
      for(j=0;j<nbrdata;j++)
	Games[j]=NULL;
      XFREE(Games);
    };

  XFREE(Data->TempGames);
  Data->NbrTempGames=0;


  for(i=0;i<GamesList->NbrResources;i++)
    {
      if(GamesList->Resources[i]->Game==NULL)
	{
	  LPRINTF(WARNING,"Data for resource '%s' are missing.",
		  GamesList->Resources[i]->Name);
	  GamesList->Resources[i]->Game=XCALLOC(s_GameInfo,1);
	  GamesList->Resources[i]->Game->Name=
	    XSTRDUP(GamesList->Resources[i]->Name);
	  GamesList->Resources[i]->Game->More=XCALLOC(s_MoreGameInfo,1);
	  GamesList->Games=XREALLOC(GamesList->Games,
				    s_GameInfo*,
				    GamesList->NbrGames+1);
	  GamesList->Games[GamesList->NbrGames]=
	    GamesList->Resources[i]->Game;
	  GamesList->NbrGames++;

	  // Transfert resource rom into the game (Merged -> Rom)
	  for(j=0;j<GamesList->Resources[i]->NbrAssociatedGames;j++)
	    {
	      for(k=0;k<GamesList->Resources[i]->AssociatedGames[j]->More->NbrMergedRoms;k++)
		{
		  if(strcmp(GamesList->Resources[i]->AssociatedGames[j]->Name,
			    GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]->GameName)==0)
		    {
		      GamesList->Resources[i]->AssociatedGames[j]->Roms=
			XREALLOC(GamesList->Resources[i]->AssociatedGames[j]->Roms,
				 s_RomInfo*,
				 GamesList->Resources[i]->AssociatedGames[j]->NbrRoms+1);
		      GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]=GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k];
		      
		      // Add the game to the rom associated games
		      for(l=0;l<GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames;l++)
			if(GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[l]->Game==GamesList->Resources[i]->AssociatedGames[j])
			  break;
		      
		      if(l==GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames)
			{		      
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames=
			    XREALLOC(GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames,
				     s_RomAssociatedGame*,
				     GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames+1);
			  
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames]=
			    XCALLOC(s_RomAssociatedGame,1);
			  
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames]->NbFiles=1;
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames]->Game=GamesList->Resources[i]->AssociatedGames[j];
			  GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->NbrAssociatedGames++;
			}
		      else
			GamesList->Resources[i]->AssociatedGames[j]->Roms[GamesList->Resources[i]->AssociatedGames[j]->NbrRoms]->Content->AssociatedGames[l]->NbFiles++;
		      
		      GamesList->Resources[i]->AssociatedGames[j]->NbrRoms++;
		      
		      // remove the rom from the Merged Roms
		      GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]=NULL;
		      GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms=
			XREALLOC(GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms,
				 s_RomInfo*,
				 GamesList->Resources[i]->AssociatedGames[j]->More->NbrMergedRoms-1);
		      GamesList->Resources[i]->AssociatedGames[j]->More->NbrMergedRoms--;
		      if(GamesList->Resources[i]->AssociatedGames[j]->More->NbrMergedRoms==0)
			GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms=NULL;
		    };
		};		
	    };
	};
    };


  if(GamesList->Sources[0]->StorageMode==STMODE_SPLIT)
    {
      for(i=0;i<GamesList->NbrGames;i++)
	{
	  if(GamesList->Games[i]->More->RomOf!=NULL)
	    {
	      nbrdata=1;
	      Games=SortBox_FindTheWord(Data->GamesSortBox,
				GamesList->Games[i]->More->RomOf,
				(int*)(&nbrdata));
	      if(nbrdata==0)
		{
		  LPRINTF(WARNING,"Parent game '%s' for '%s' not found",
			  GamesList->Games[i]->More->RomOf,
			  GamesList->Games[i]->Name);
		  continue;
		};

	      if(GamesList->Games[i]->More->NbrMergedRoms==0)
		{
		  for(j=0;j<GamesList->Games[i]->NbrRoms;j++)
		    {
		      for(k=0;k<Games[0]->NbrRoms;k++)
			{
			  if(GamesList->Games[i]->Roms[j]->Content->Size==Games[0]->Roms[k]->Content->Size &&
			     GamesList->Games[i]->Roms[j]->Content->CRC==Games[0]->Roms[k]->Content->CRC)
			    {
			      GamesList->Games[i]->More->MergedRoms=
				XREALLOC(GamesList->Games[i]->More->MergedRoms,
					 s_RomInfo*,
					 GamesList->Games[i]->More->NbrMergedRoms+1);

			      GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]=GamesList->Games[i]->Roms[j];
			      XFREE(GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]->GameName);
			      GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]->GameName=XSTRDUP(GamesList->Games[i]->More->CloneOf);
			      GamesList->Games[i]->More->NbrMergedRoms++;
			      for(l=0;l<GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames;l++)
				if(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->Game==GamesList->Games[i])
				  break;

			      if(l!=GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames)
				{
				  GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->NbFiles--;
				  if(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->NbFiles==0)
				    {
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->Game=NULL;
				      XFREE(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]);
				      for(m=l;m<GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames-1;m++)
					GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m]=GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m+1];
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m]=NULL;
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames=XREALLOC(GamesList->Games[i]->Roms[j]->Content->AssociatedGames,s_RomAssociatedGame*,GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames-1);
				      GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames--;
				    };
				}
			      else
				{
				  LPRINTF(WARNING,"Game '%s' not found as a romcontent associated game of %x/%i (CRC/size)",
					  GamesList->Games[i]->Name,
					  GamesList->Games[i]->Roms[j]->Content->CRC,
					  GamesList->Games[i]->Roms[j]->Content->Size);
				};
			      GamesList->Games[i]->Roms[j]=NULL;
			      break;
			    };
			};
		    };
		  l=0;
		  for(k=0;k<GamesList->Games[i]->NbrRoms;k++)
		    if(GamesList->Games[i]->Roms[k]!=NULL)
		      l++;
		  roms=XCALLOC(s_RomInfo*,l);
		  l=0;
		  for(k=0;k<GamesList->Games[i]->NbrRoms;k++)
		    if(GamesList->Games[i]->Roms[k]!=NULL)
		      {
			roms[l]=GamesList->Games[i]->Roms[k];
			l++;
		      };
		  XFREE(GamesList->Games[i]->Roms);
		  GamesList->Games[i]->Roms=roms;
		  GamesList->Games[i]->NbrRoms=l;
		};
	      XFREE(Games);
	    };

	  if(GamesList->Games[i]->More->Resource!=NULL &&
	     strcmp(GamesList->Games[i]->More->Resource,
		    GamesList->Games[i]->More->RomOf)!=0)
	    {
	      nbrdata=1;
	      Games=SortBox_FindTheWord(Data->GamesSortBox,
				GamesList->Games[i]->More->Resource,
				(int*)(&nbrdata));
	      if(nbrdata==0)
		{
		  LPRINTF(WARNING,"Resource '%s' for '%s' not found",
			  GamesList->Games[i]->More->Resource,
			  GamesList->Games[i]->Name);
		  continue;
		}; 
//	      if(GamesList->Games[i]->More->NbrMergedRoms==0)
		{
		  for(j=0;j<GamesList->Games[i]->NbrRoms;j++)
		    {
		      for(k=0;k<Games[0]->NbrRoms;k++)
			{
			  if(GamesList->Games[i]->Roms[j]->Content->Size==Games[0]->Roms[k]->Content->Size &&
			     GamesList->Games[i]->Roms[j]->Content->CRC==Games[0]->Roms[k]->Content->CRC)
			    {
			      GamesList->Games[i]->More->MergedRoms=
				XREALLOC(GamesList->Games[i]->More->MergedRoms,
					 s_RomInfo*,
					 GamesList->Games[i]->More->NbrMergedRoms+1);

			      GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]=GamesList->Games[i]->Roms[j];
			      XFREE(GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]->GameName);
			      GamesList->Games[i]->More->MergedRoms[GamesList->Games[i]->More->NbrMergedRoms]->GameName=XSTRDUP(GamesList->Games[i]->More->Resource);
			      GamesList->Games[i]->More->NbrMergedRoms++;
			      for(l=0;l<GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames;l++)
				if(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->Game==GamesList->Games[i])
				  break;

			      if(l!=GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames)
				{
				  GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->NbFiles--;
				  if(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->NbFiles==0)
				    {
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]->Game=NULL;
				      XFREE(GamesList->Games[i]->Roms[j]->Content->AssociatedGames[l]);
				      for(m=l;m<GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames-1;m++)
					GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m]=GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m+1];
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames[m]=NULL;
				      GamesList->Games[i]->Roms[j]->Content->AssociatedGames=XREALLOC(GamesList->Games[i]->Roms[j]->Content->AssociatedGames,s_RomAssociatedGame*,GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames-1);
				      GamesList->Games[i]->Roms[j]->Content->NbrAssociatedGames--;
				    };
				}
			      else
				{
				  LPRINTF(WARNING,"Game '%s' not found as a romcontent associated game of %x/%i (CRC/size)",
					  GamesList->Games[i]->Name,
					  GamesList->Games[i]->Roms[j]->Content->CRC,
					  GamesList->Games[i]->Roms[j]->Content->Size);
				};
			      GamesList->Games[i]->Roms[j]=NULL;
			      break;
			    };
			};
		    };
		  l=0;
		  for(k=0;k<GamesList->Games[i]->NbrRoms;k++)
		    if(GamesList->Games[i]->Roms[k]!=NULL)
		      l++;
		  roms=XCALLOC(s_RomInfo*,l);
		  l=0;
		  for(k=0;k<GamesList->Games[i]->NbrRoms;k++)
		    if(GamesList->Games[i]->Roms[k]!=NULL)
		      {
			roms[l]=GamesList->Games[i]->Roms[k];
			l++;
		      };
		  XFREE(GamesList->Games[i]->Roms);
		  GamesList->Games[i]->Roms=roms;
		  GamesList->Games[i]->NbrRoms=l;
		};
	      XFREE(Games);
	    };
	};
    };

  for(i=0;i<GamesList->NbrResources;i++)
    {  
      for(j=0;j<GamesList->Resources[i]->NbrAssociatedGames;j++)
	{
	  for(k=0;k<GamesList->Resources[i]->AssociatedGames[j]->More->NbrMergedRoms;k++)
	    {
	      for(l=0;l<GamesList->Resources[i]->Game->NbrRoms;l++)
		if(strcmp(GamesList->Resources[i]->Game->Roms[l]->Name,
			  GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]->Name)==0)
		  {
		    XFREE(GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]->GameName);
		    GamesList->Resources[i]->AssociatedGames[j]->More->MergedRoms[k]->GameName=XSTRDUP(GamesList->Resources[i]->Name);
		  };
	    };
	};
    };
  

  fclose(GamesSourceFile);
  FreeSortBox(Data->RomContentsSortBox);
  Data->RomContentsSortBox=NULL;
  FreeSortBox(Data->GamesSortBox);
  Data->GamesSortBox=NULL;
  XFREE(Data->String);
  XFREE(Data);
//  GamesList->NbrUnknownTokens=NbrUnknownTokens;
  XFREE(Token);
  return GamesList;
};
