/* GDC -- D front-end for GCC
   Copyright (C) 2004 David Friedman
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*
  This file is based on dmd/tocsym.c.  Original copyright:

// Copyright (c) 1999-2002 by Digital Mars
// All Rights Reserved
// written by Walter Bright
// www.digitalmars.com
// License for redistribution is by either the Artistic License
// in artistic.txt, or the GNU General Public License in gnu.txt.
// See the included readme.txt for details.

*/

#include "d-gcc-includes.h"
#include <assert.h>

#include "total.h"
#include "mars.h"
#include "statement.h"
#include "aggregate.h"
#include "init.h"
#include "attrib.h"

#include "symbol.h"
#include "d-lang.h"
#include "d-codegen.h"

void slist_add(Symbol */*s*/)
{
}
void slist_reset()
{
}

/********************************* SymbolDeclaration ****************************/

SymbolDeclaration::SymbolDeclaration(Loc loc, Symbol *s)
    : Declaration(new Identifier(s->Sident, TOKidentifier))
{
    this->loc = loc;
    sym = s;
    storage_class |= STCconst;
}

Symbol *SymbolDeclaration::toSymbol()
{
    // Create the actual back-end value if not yet done
    if (! sym->Stree) {
	AggregateDeclaration * agg_decl = sym->Sspecial.aggDecl;
	agg_decl->toInitializer(); // Try to create the VAR_DECL...
	// ... If it didn't work, then the returned Symbol* is invalid
	assert(sym->Stree);
    }
    return sym;
}

/*************************************
 * Helper
 */

Symbol *Dsymbol::toSymbolX(const char *prefix, int sclass, type *t)
{
    Symbol *s;
    char *id;
    char *n;

    n = mangle(); //ident->toChars();
    id = (char *) alloca(strlen(prefix) + strlen(n) + 1);
    sprintf(id,"%s%s", prefix, n);
    s = symbol_name(id, sclass, t);
    return s;
}

/*************************************
 */

Symbol *Dsymbol::toSymbol()
{
    printf("Dsymbol::toSymbol() '%s', kind = '%s'\n", toChars(), kind());
    assert(0);		// BUG: implement
    return NULL;
}

/*********************************
 * Generate import symbol from symbol.
 */

Symbol *Dsymbol::toImport()
{
    if (!isym)
    {
	if (!csym)
	    csym = toSymbol();
	isym = toImport(csym);
    }
    return isym;
}

/*************************************
 */

Symbol *Dsymbol::toImport(Symbol * /*sym*/)
{
    // not used in GCC (yet?)
    return 0;
}


// Workaround for: If there is a declaration in a function, another
// other declarations with the same identifier in overloads of the
// functions will have the same mangled name.  DMD gets away with this
// because it produces its own object files and the declarations are
// essentially private.  Assemblers won't accept this, so unique names
// must be generated.

// Unresolved issues:
// * What if these are referenced from another module when inlining?
// * Okay to always SET_DECL_ASSEMBLER_NAME?
//
// Use set_decl_assembler_name hook instead?
//
// Assumes 'd' is static storage.

static StringTable * uniqueNames = 0;
static /*char **/void uniqueName(Dsymbol * d, tree t, const char * asm_name) {
    // First, check if the symbol is declared in a function
    Dsymbol * p = d->parent; 
    const char * out_name = asm_name;
    char * alloc_name = 0;

    while (p) {
	if (p->isFuncDeclaration()) {
	    StringValue * sv;

	    // Assumes one assembler output file
	    if (! uniqueNames)
		uniqueNames = new StringTable;
	    sv = uniqueNames->update(asm_name, strlen(asm_name));
	    if (sv->intvalue) {
		out_name = alloc_name = d_asm_format_private_name(asm_name, sv->intvalue );
	    }
	    sv->intvalue++;

	    //return out_name;
	}
	p = p->parent;
    }
    // return asm_name;

    SET_DECL_ASSEMBLER_NAME(t, get_identifier(out_name));

    if (alloc_name)
	free(alloc_name);
}


/*************************************
 */

Symbol *VarDeclaration::toSymbol()
{
    if (! csym) {
	tree var_decl;

	// For field declaration, it is possible for toSymbol to be called
	// before the parent's toCtype()
	if (storage_class & STCfield) {
	    AggregateDeclaration * parent_decl = toParent()->isAggregateDeclaration();
	    assert(parent_decl);
	    parent_decl->type->toCtype();
	    assert(csym);
	    return csym;
	}

	/*	
	tree ident_to_use;
	if (isDataseg())
	    ident_to_use = get_identifier( mangle() );
	else
	    ident_to_use = get_identifier( ident->string );
	*/
	const char * ident_to_use;
	if (isDataseg())
	    ident_to_use = mangle();
	else
	    ident_to_use = ident->string;
	    
	var_decl = build_decl(storage_class & STCparameter ? PARM_DECL : VAR_DECL,
	    get_identifier(ident_to_use), gen.trueDeclarationType( this ));
	if (isDataseg())
	    uniqueName(this, var_decl, ident_to_use);
	dkeep(var_decl);
	gen.setDeclLoc(var_decl, this);
	if ( TREE_CODE( var_decl ) == VAR_DECL ) {
	    gen.setupSymbolStorage(this, var_decl);
	} else {
	    /* from gcc code: Some languages have different nominal and real types.  */
	    // %% What about DECL_ORIGINAL_TYPE, DECL_ARG_TYPE_AS_WRITTEN, DECL_ARG_TYPE ?
	    DECL_ARG_TYPE( var_decl ) = TREE_TYPE (var_decl);
	    
	    DECL_CONTEXT( var_decl ) = gen.declContext(this);
	    assert( TREE_CODE(DECL_CONTEXT( var_decl )) == FUNCTION_DECL );
	}

	// Can't set TREE_STATIC, etc. until we get to toObjFile as this could be
	// called from a varaible in an imported module
	// %% (out const X x) doesn't mean the reference is const...
	if ( isConst() && ! gen.isDeclarationReferenceType( this )) {
	    // %% CONST_DECLS don't have storage, so we can't use those,
	    // but it would be nice to get the benefit of them (could handle in
	    // VarExp -- makeAddressOf could switch back to the VAR_DECL

	    // if ( typs->isscalar() ) CONST_DECL...
	    
	    TREE_READONLY( var_decl ) = 1;
	    // can at least do this...
	    //  const doesn't seem to matter for aggregates, so prevent problems..
	    if ( type->isscalar() ) {
		TREE_CONSTANT( var_decl ) = 1;
	    }
	}

	if (nestedref) {
	    DECL_NONLOCAL( var_decl ) = 1;
	    TREE_ADDRESSABLE( var_decl ) = 1;
	}

	// %%EXPER%% TREE_ADDRESSABLE( var_decl ) = 1;
	TREE_USED( var_decl ) = 1;
	
	csym = new Symbol();
	csym->Stree = var_decl;

    }
    return csym;
}

/*************************************
 */

Symbol *ClassInfoDeclaration::toSymbol()
{
    return cd->toSymbol();
}

/*************************************
 */

Symbol *ModuleInfoDeclaration::toSymbol()
{
    return mod->toSymbol();
}

/*************************************
 */

Symbol *TypeInfoDeclaration::toSymbol()
{
    if ( ! csym ) {
	VarDeclaration::toSymbol();
	
	// This variable is the static initialization for the
	// given TypeInfo.  It is the actual data, not a reference
	assert( TREE_CODE( TREE_TYPE( csym->Stree )) == REFERENCE_TYPE );
	TREE_TYPE( csym->Stree ) = TREE_TYPE( TREE_TYPE( csym->Stree ));
	D_DECL_ONE_ONLY( csym->Stree ) = 1;
    }
    return csym;
}

/*************************************
 */

// returns a FUNCTION_DECL tree
Symbol *FuncDeclaration::toSymbol()
{
    if (! csym) {
	tree id;
	//struct prod_token_parm_item* parm;
	//tree type_node;
	tree fn_decl;
	char * mangled_ident_str = 0;
	AggregateDeclaration * agg_decl;

	if (ident) {
	    id = get_identifier(ident->string);
	} else {
	    // This happens for assoc array foreach bodies
	    
	    // Not sure if idents are strictly necc., but announce_function
	    //  dies without them.

	    // %% better: use parent name
	    
	    static unsigned unamed_seq = 0;
	    char buf[64];
	    sprintf(buf, "___unamed_%u", ++unamed_seq);//%% sprintf
	    id = get_identifier(buf);
	}

	tree fn_type = type->toCtype();
	dkeep(fn_type); /* TODO: fix this. we need to keep the type because
			   the creation of a method type below leaves this fn_type
			   unreferenced. maybe lang_specific.based_on */
	
	tree vindex = NULL_TREE;
	if (isNested()) {
	    // Nested functions take an extra argument to be compatible with delegates.
	    // The stack frame will come from a trampoline.
	    fn_type = build_function_type(TREE_TYPE(fn_type),
		tree_cons(NULL_TREE, ptr_type_node, TYPE_ARG_TYPES(fn_type)));
	    TYPE_LANG_SPECIFIC( fn_type ) = build_d_type_lang_specific(type);
	    TYPE_LANG_SPECIFIC( fn_type )->is_nested_function = 1;
	}
	else if ( ( agg_decl = isMember() ) ) {
	    // Do this even if there is no debug info.  It is needed to maker
	    // sure member functions are not called statically
	    if (isThis()) {
		tree method_type = build_method_type(TREE_TYPE(agg_decl->handle->toCtype()), fn_type);
		TYPE_ATTRIBUTES( method_type ) = TYPE_ATTRIBUTES( fn_type );
		fn_type = method_type;

		/* hopefully, this->type->toCtype()->lang_specific->dtype != this->type
		   won't be a problem */
		TYPE_LANG_SPECIFIC( fn_type ) = build_d_type_lang_specific(this->type);

		if (isVirtual())
		    vindex = size_int(vtblIndex);
	    }
	}

	// %%CHECK: is it okay for static nested functions to have a FUNC_DECL context?
	// seems okay so far...
	
	fn_decl = build_decl( FUNCTION_DECL, id, fn_type );
	dkeep(fn_decl);
	if (ident) {
	    mangled_ident_str = mangle();
	    uniqueName(this, fn_decl, mangled_ident_str);
	    // old : SET_DECL_ASSEMBLER_NAME (fn_decl, get_identifier( mangled_ident_str ));
	}
	// %% What about DECL_SECTION_NAME ?
	//DECL_ARGUMENTS(fn_decl) = NULL_TREE; // Probably don't need to do this until toObjFile
	DECL_CONTEXT (fn_decl) = gen.declContext(this); //context;
	if (vindex) {
	    DECL_VINDEX    (fn_decl) = vindex;
	    DECL_VIRTUAL_P (fn_decl) = 1;
	}
	if (! isNested()) {
	    // Prevent backend from thinking this is a nested function.
	    DECL_NO_STATIC_CHAIN( fn_decl ) = 1;
	}

#ifdef TARGET_DLLIMPORT_DECL_ATTRIBUTES
	// Have to test for import first
	if (isImportedSymbol())
	    gen.addDeclAttribute( fn_decl, "dllimport" );
	else if (isExport())
	    gen.addDeclAttribute( fn_decl, "dllexport" );
#endif
	    
	gen.setDeclLoc(fn_decl, this);
	gen.setupSymbolStorage(this, fn_decl);
	if (! ident)
	    TREE_PUBLIC( fn_decl ) = 0;
	
	TREE_USED (fn_decl) = 1; // %% Probably should be a little more intelligent about this

	// if -mrtd is passed, how to handle this? handle in parsing or do
	// we go back and find out if linkage was specified
	switch (linkage)
	{
	    case LINKwindows:
		gen.addDeclAttribute(fn_decl, "stdcall");
		// The stdcall attribute also needs to be set on the function type.
		assert( ((TypeFunction *) type)->linkage == LINKwindows );
		break;
	    case LINKpascal:
		// this is just stdcall without mangling, right?
		break;
	    case LINKc:
		// %% hack: on darwin (at least) using a DECL_EXTERNAL (IRState::getLibCallDecl)
		// and TREE_STATIC FUNCTION_DECLs causes the stub label to be output twice.  This
		// is a work around.  This doesn't handle the case in which the normal
		// getLibCallDecl has already be created an used.  Note that the problem only
		// occurs with function inlining is used.
		gen.replaceLibCallDecl(this);
		break;
	    case LINKd:
		// %% If x86, regparm(1)
		// not sure if reg struct return 
		break;
	    case LINKcpp:
		break;
	    default:
		printf("linkage = %d\n", linkage);
		assert(0);
	}
	
	csym  = new Symbol();
	csym->Sident = mangled_ident_str; // save for making thunks
	csym->Stree = fn_decl;

	gen.maybeSetUpBuiltin(this);
    }
    return csym;
}

/*************************************
 */

Symbol *FuncDeclaration::toThunkSymbol(int offset)
{
    Symbol *sthunk;
    Thunk * thunk;
    bool is_extern;

    toSymbol();
    is_extern = DECL_EXTERNAL( csym->Stree ); // shortcut for shouldEmit, isTemplate, etc.
    
    /* If the thunk is to be static (that is, it is being emitted in this
       module, there can only be one FUNCTION_DECL for it.   Thus, there
       is a list of all thunks for a given function. */
    if ( ! csym->Sspecial.thunks )
	csym->Sspecial.thunks = new Array;
    Array & thunks = * csym->Sspecial.thunks;
    bool found = false;
	
    for (unsigned i = 0; i < thunks.dim; i++) {
	thunk = (Thunk *) thunks.data[i];
	if (thunk->offset == offset) {
	    found = true;
	    break;
	}
    }

    if (! found) {
	thunk = new Thunk;
	thunk->offset = offset;
	thunks.push(thunk);
    }

    if ( ! thunk->symbol ) {
	char *id;
	char *n;
	//type *t;

	n = csym->Sident; // dmd uses 'sym' -- not sure what that is...
	id = (char *) alloca(8 + 5 + strlen(n) + 1);
	sprintf(id,"_thunk%d__%s", offset, n);
	sthunk = symbol_calloc(id);
	slist_add(sthunk);
	//sthunk = symbol_generate(SCstatic, csym->Stype);
	//sthunk->Sflags |= SFLimplem;

	tree target_func_decl = csym->Stree;
	tree thunk_decl = build_decl(FUNCTION_DECL, get_identifier(id), TREE_TYPE( target_func_decl ));
	dkeep(thunk_decl);
	sthunk->Stree = thunk_decl;

	SET_DECL_ASSEMBLER_NAME(thunk_decl, DECL_NAME(thunk_decl));
	DECL_CONTEXT(thunk_decl) = DECL_CONTEXT(target_func_decl);  // from c++...
	TREE_READONLY(thunk_decl) = TREE_READONLY(target_func_decl);
	TREE_THIS_VOLATILE(thunk_decl) = TREE_THIS_VOLATILE(target_func_decl);
	TREE_PUBLIC(thunk_decl) = TREE_PUBLIC(target_func_decl);
	DECL_ARTIFICIAL(thunk_decl) = 1;
	DECL_NO_STATIC_CHAIN(thunk_decl) = 1;
	DECL_INLINE(thunk_decl) = 0;
#if D_GCC_VER >= 34
	DECL_DECLARED_INLINE_P(thunk_decl) = 0;
#endif
	D_DECL_ONE_ONLY(thunk_decl) = D_DECL_ONE_ONLY(target_func_decl);
	D_DECL_IS_TEMPLATE(thunk_decl) = D_DECL_IS_TEMPLATE(target_func_decl); // This does not do anything now.

	TREE_ADDRESSABLE(thunk_decl) = 1;
	TREE_USED (thunk_decl) = 1; // assuming only called if it will be used

	gen.prepareSymbolOutput(sthunk);

	if (is_extern) {
	    DECL_EXTERNAL( thunk_decl ) = 1;
	} else {
	    gen.doThunk(thunk_decl, target_func_decl, offset);
	}

	thunk->symbol = sthunk;
    }
    return thunk->symbol;
}

/****************************************
 * Create a static symbol we can hang DT initializers onto.
 */

Symbol *static_sym()
{
    Symbol * s = symbol_tree(NULL_TREE);
    s->Sfl = FLstatic_sym;
    slist_add(s);
    return s;
}

/**************************************
 * Fake a struct symbol.
 */

// Not used in GCC
Classsym *fake_classsym(char * /*name*/)
{
    return 0;
}

/*************************************
 * Create the "ClassInfo" symbol
 */

Symbol *ClassDeclaration::toSymbol()
{
    if (! csym)
    {
	tree decl;
	csym = toSymbolX("_Class_", SCextern, 0);
	slist_add(csym);
	decl = build_decl( VAR_DECL, get_identifier( csym->Sident ),
	    TREE_TYPE( ClassDeclaration::classinfo->type->toCtype() )); // want the RECORD_TYPE, not the REFERENCE_TYPE
	csym->Stree = decl;
	dkeep(decl);
 
	gen.setupStaticStorage(this, decl);

	TREE_CONSTANT( decl ) = 0; // DMD puts this into .data, not .rodata...
	TREE_READONLY( decl ) = 1; // Non-constant data, but still won't be assigned to.
    }
    return csym;
}

/*************************************
 * Create the "InterfaceInfo" symbol
 */

Symbol *InterfaceDeclaration::toSymbol()
{
    if (!csym)
    {
	csym = ClassDeclaration::toSymbol();
	tree decl = csym->Stree;
	
	Symbol * temp_sym = toSymbolX("_Interface_", SCextern, 0);
	DECL_NAME( decl ) = get_identifier( temp_sym->Sident );
	delete temp_sym;

	TREE_CONSTANT( decl ) = 1; // Interface ClassInfo images are in .rodata, but classes arent..?
    }
    return csym;
}

/*************************************
 * Create the "ModuleInfo" symbol
 */

Symbol *Module::toSymbol()
{
    if (!csym)
    {
	Type * some_type;

	/* PAIN -- causes problems with zomg_t3h_rei .. workaround
	   is to call moduleinfo->toCtype() before we start processing
	   decls in genobjfile */
	/* We don't, in general, have moduleinfo anyway, so don't bother */
	/*
	if (moduleinfo) {
	    some_type = moduleinfo->type;
	} else {
	    some_type = gen.getObjectType();
	}
	*/
	some_type = gen.getObjectType();
	
	csym = toSymbolX("_ModuleInfo_", SCextern, 0);
	slist_add(csym);

	tree decl = build_decl(VAR_DECL, get_identifier(csym->Sident),
	    TREE_TYPE(some_type->toCtype())); // want the RECORD_TYPE, not the REFERENCE_TYPE
	csym->Stree = decl;
#if D_GCC_VER >= 34
	// EXPER
	/*
	csym->ScontextDecl = build_decl(TRANSLATION_UNIT_DECL, NULL, NULL);
	dkeep(csym->ScontextDecl);
	*/
#endif
	
	dkeep(decl);
	
	gen.setupStaticStorage(this, decl);
	TREE_CONSTANT( decl ) = 0; // *not* readonly, moduleinit depends on this
	TREE_READONLY( decl ) = 0; // Not an lvalue, tho
    }
    return csym;
}

/*************************************
 * This is accessible via the ClassData, but since it is frequently
 * needed directly (like for rtti comparisons), make it directly accessible.
 */

Symbol *ClassDeclaration::toVtblSymbol()
{
    if (!vtblsym)
    {
	tree decl;
	
	vtblsym = toSymbolX("_vtbl_", SCextern, 0);
	slist_add(vtblsym);

	/* The DECL_INITIAL value will have a different type object from the
	   VAR_DECL.  The back end seems to accept this. */
	TypeSArray * vtbl_type = new TypeSArray(Type::tvoid->pointerTo(), new IntegerExp(vtbl.dim));

	decl = build_decl( VAR_DECL, get_identifier( vtblsym->Sident ), vtbl_type->toCtype() );
	vtblsym->Stree = decl;
	dkeep(decl);

	gen.setupStaticStorage(this, decl);

	TREE_CONSTANT( decl ) = 1;
	TREE_ADDRESSABLE( decl ) = 1;
	// from cp/class.c
	DECL_CONTEXT (decl) =  TREE_TYPE( type->toCtype() );
	DECL_VIRTUAL_P (decl) = 1;
	DECL_ALIGN (decl) = TARGET_VTABLE_ENTRY_ALIGN;
    }
    return vtblsym;
}

/**********************************
 * Create the static initializer for the struct/class.
 */

/* Because this is called from the front end (mtype.cc:TypeStruct::defaultInit()),
   we need to hold off using back-end stuff until the toobjfile phase.
*/

Symbol *AggregateDeclaration::toInitializer()
{
    char *id;
    char *n;
    Symbol *s;
    Classsym *stag;

    if (!sinit)
    {
	n = mangle();
	stag = fake_classsym(n);

	id = (char *) alloca(6 + strlen(n) + 1);
	sprintf(id,"_init_%s",n);
	s = symbol_calloc(id);
	s->Sspecial.aggDecl = this;
	slist_add(s);
	sinit = s;
    }
    if (! sinit->Stree) {
	if (getCurrentModule()) {
	    tree struct_type = type->toCtype();
	    if ( POINTER_TYPE_P( struct_type ) )
		struct_type = TREE_TYPE( struct_type ); // for TypeClass, want the RECORD_TYPE, not the REFERENCE_TYPE
	    tree t = build_decl(VAR_DECL, get_identifier(sinit->Sident), struct_type);
	    sinit->Stree = t;
	    dkeep(t);

	    gen.setupStaticStorage(this, t);

	    // %% what's the diff between setting this stuff on the DECL and the
	    // CONSTRUCTOR itself?

	    TREE_ADDRESSABLE( t ) = 1;
	    TREE_CONSTANT( t ) = 1;
	    TREE_READONLY( t ) = 1;
	    DECL_CONTEXT( t ) = 0; // These are always global
	}
    }
    return sinit;
}


/******************************************
 */

Symbol *Module::toModuleAssert()
{
    // Not used in GCC
    return 0;
}

/******************************************
 */

Symbol *Module::toModuleArray()
{
    // Not used in GCC (all array bounds checks are inlined)
    return 0;
}

/********************************************
 * Determine the right symbol to look up
 * an associative array element.
 * Input:
 *	flags	0	don't add value signature
 *		1	add value signature
 */

Symbol *TypeAArray::aaGetSymbol(char *func, int flags)
{
    // This is not used in GCC (yet?)
    return 0;
}

