/* GDC -- D front-end for GCC
   Copyright (C) 2004 David Friedman
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
 
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef GCC_DCMPLR_IRSTATE_H
#define GCC_DCMPLR_IRSTATE_H

// Due to the inlined functions, "dc-gcc-includes.h" needs to
// be included before this header is included.

#include "mars.h"
#include "root.h"
#include "lexer.h"
#include "mtype.h"
#include "declaration.h"
#include "statement.h"
#include "expression.h"
#include "aggregate.h"
#include "symbol.h"

// IRBase contains the core functionality of IRState.  The actual IRState class
// extends this with lots of code generation utilities.
//
// Currently, each function gets its own IRState when emitting code.  There is
// also a global IRState.
//
// Most toElem calls don't actually need the IRState because they create GCC
// expression trees rather than emit instructions.

struct IRBase : Object
{
    IRBase * parentIRState;

    IRBase();

    // ** Functions
    
    // This is used by LabelStatement to find the LabelDsymbol that
    // GotoStatements refer to.
    FuncDeclaration * thisFunction; // %% make this a stack

    FuncDeclaration * getCurrentFunction() { return thisFunction; }
    IRState * startFunction(FuncDeclaration * decl);
protected:
    IRState * endFunction();
public:
    Array deferedFuncDecls;
    bool shouldDeferFunction(FuncDeclaration * decl);

    static void initFunctionStart(tree fn_decl, const Loc & loc);

    // ** Labels

    // It is only valid to call this while the function in which the label is defined
    // is being compiled.
    tree getLabelTree(LabelDsymbol * label);

    // ** Loops (and case statements)

    typedef struct
    {
	Statement * statement;
	// expand_start_case doesn't return a nesting structure, so
	// we have to generate our own label for 'break'
	nesting * loop;
	tree      exitLabel;
    } Flow;

    Array loops; // of Flow
    
    // These routines don't generate code.  They are for tracking labeled loops.
    Flow *    getLoopForLabel(Identifier * ident, bool want_continue = false);
    void      beginLoop(Statement * stmt, nesting * loop);
    void      endLoop();
    void      continueLoop(Identifier * ident);
    void      exitLoop(Identifier * ident);


    // ** DECL_CONTEXT support

    // TODO: separate stacks for variables, functions, blocks, etc.
    Array declContextStack;

    bool isToplevelDeclContext() { return getDeclContext() ? false : true; }
    tree getDeclContext(); 
    void pushDeclContext(tree context);
    void popDeclContext();

    // ** "Binding contours"

    /* Definitions for IRBase scope code:
       "Scope": A container for binding contours.  Each user-declared
       function has a toplevel scope.  Every ScopeStatement creates
       a new scope. (And for now, until the emitLocalVar crash is
       solved, this also creates a default binding contour.)
       
       "Binding countour": Same as GCC's definition, whatever that is.
       Each user-declared variable will have a binding contour that begins
       where the variable is declared and ends at it's containing scope.
    */
    Array scopes; // of unsigned*
    
    void startScope();
    void endScope();

    void startBindings();
    void endBindings();

    // ** Temporaries (currently just SAVE_EXPRs)

    // Create a SAVE_EXPR if it is not safe (or effecient*) to reevaluate
    // the expression. * -- Actually can product inefficient code because
    // it prevents cse.
    Expression * maybeMakeTemp(Expression * e);
    tree maybeMakeTemp(tree t);

    // ** Volatile state

    unsigned volatileDepth;
    bool inVolatile() { return volatileDepth != 0; }
    void pushVolatile() { ++volatileDepth; }
    void popVolatile() { --volatileDepth; }

};


#endif
