/* GNOME DB library
 * Copyright (C) 1999-2002 The GNOME Foundation.
 *
 * AUTHORS:
 *      Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gtk/gtklabel.h>
#include <libgnomedb/gnome-db-grid.h>
#include <libgnomedb/gnome-db-i18n.h>
#include <libgnomedb/gnome-db-query-builder.h>
#include <libgnomedb/gnome-db-util.h>

#include <gtk/gtktable.h>

struct _GnomeDbQueryBuilderPrivate {
	GdaConnection *cnc;

	GtkWidget *table_selector;
	GtkWidget *table_fields;
};

static void gnome_db_query_builder_class_init (GnomeDbQueryBuilderClass *klass);
static void gnome_db_query_builder_init (GnomeDbQueryBuilder *builder, GnomeDbQueryBuilderClass *klass);
static void gnome_db_query_builder_finalize (GObject *object);

static GObjectClass *parent_class = NULL;

static void
gnome_db_query_builder_class_init (GnomeDbQueryBuilderClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = gnome_db_query_builder_finalize;
}

static void
gnome_db_query_builder_init (GnomeDbQueryBuilder *builder, GnomeDbQueryBuilderClass *klass)
{
	GtkWidget *table, *label;
	gchar *str;

	builder->priv = g_new0 (GnomeDbQueryBuilderPrivate, 1);
	builder->priv->cnc = NULL;

	/* create main table */
	table = gnome_db_new_table_widget (6, 6, FALSE);
	gtk_box_pack_start (GTK_BOX (builder), table, TRUE, TRUE, 0);

	/* create field selection area */
	label = gnome_db_new_label_widget (NULL);
	str = g_strdup_printf ("<b>%s</b>", _("Fields"));
	gtk_label_set_markup (GTK_LABEL (label), str);
	g_free (str);
	gtk_table_attach (GTK_TABLE (table), label, 0, 6, 0, 1, GTK_FILL, GTK_FILL, 6, 6);

	builder->priv->table_selector = gnome_db_new_option_menu_widget ();
	gtk_table_attach (GTK_TABLE (table), builder->priv->table_selector, 1, 2, 1, 2,
			  GTK_FILL, GTK_FILL, 6, 6);

	builder->priv->table_fields = gnome_db_grid_new ();
	gtk_widget_show (builder->priv->table_fields);
	gtk_table_attach (GTK_TABLE (table), builder->priv->table_fields, 1, 2, 2, 3,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK,
			  GTK_FILL | GTK_EXPAND | GTK_SHRINK, 6, 6);
}

static void
gnome_db_query_builder_finalize (GObject *object)
{
	GnomeDbQueryBuilder *builder = (GnomeDbQueryBuilder *) object;

	g_return_if_fail (GNOME_DB_IS_QUERY_BUILDER (builder));

	/* free memory */
	if (builder->priv) {
		if (GDA_IS_CONNECTION (builder->priv->cnc)) {
			g_object_unref (builder->priv->cnc);
			builder->priv->cnc = NULL;
		}

		builder->priv = NULL;
	}

	parent_class->finalize (object);
}

GType
gnome_db_query_builder_get_type (void)
{
	static GType type = 0;

	if (!type) {
		static const GTypeInfo info = {
			sizeof (GnomeDbQueryBuilderClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) gnome_db_query_builder_class_init,
			NULL,
			NULL,
			sizeof (GnomeDbQueryBuilder),
			0,
			(GInstanceInitFunc) gnome_db_query_builder_init
		};
		type = g_type_register_static (GTK_TYPE_VBOX, "GnomeDbQueryBuilder", &info, 0);
	}
	return type;
}

/**
 * gnome_db_query_builder_new
 * @cnc: A #GdaConnection object.
 *
 * Create a new #GnomeDbQueryBuilder widget, which presents a visual way
 * of building SQL queries.
 *
 * Returns: the newly-created widget.
 */
GtkWidget *
gnome_db_query_builder_new (GdaConnection *cnc)
{
	GnomeDbQueryBuilder *builder;

	builder = g_object_new (GNOME_DB_TYPE_QUERY_BUILDER, NULL);

	if (GDA_IS_CONNECTION (cnc))
		gnome_db_query_builder_set_connection (builder, cnc);

	return (GtkWidget *) builder;
}

/**
 * gnome_db_query_builder_get_connection
 * @builder: A #GnomeDbQueryBuilder widget.
 *
 * Get the #GdaConnection associated with the given query builder widget.
 *
 * Returns: the associated connection.
 */
GdaConnection *
gnome_db_query_builder_get_connection (GnomeDbQueryBuilder *builder)
{
	g_return_val_if_fail (GNOME_DB_IS_QUERY_BUILDER (builder), NULL);
	return builder->priv->cnc;
}

/**
 * gnome_db_query_builder_set_connection
 * @builder: A #GnomeDbQueryBuilder widget.
 * @cnc: The connection to use in the widget.
 *
 * Associate a #GdaConnection with the given query builder widget.
 */
void
gnome_db_query_builder_set_connection (GnomeDbQueryBuilder *builder, GdaConnection *cnc)
{
	g_return_if_fail (GNOME_DB_IS_QUERY_BUILDER (builder));

	if (GDA_IS_CONNECTION (builder->priv->cnc))
		g_object_unref (builder->priv->cnc);

	builder->priv->cnc = cnc;
	if (GDA_IS_CONNECTION (cnc)) {
		g_object_ref (cnc);

		/* FIXME: refresh all widgets */
	}
}
