/*
 * loadsave.c
 * Thomas Nemeth, le 15.10.2003
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include "defines.h"
#include <ncurses.h>
#include <readline/readline.h>
#include "windows.h"
#include "commands.h"
#include "control.h"
#include "tmms.h"
#include "xmem.h"
#include "playlist.h"
#include "playerwin.h"
#include "playlistwin.h"
#include "loadsave.h"


static WINDOW *loadsave_win = NULL;
static const char *prompt = "Playlist : ";


/* Display functions */

void display_loadsave(char *title) {
    if (loadsave_win) destroy_win(loadsave_win);
    loadsave_win = create_newwin(4, COLS, 0, 0);
    display_loadsavetitle(title);
}


void display_loadsavetitle(char *title) {
    int j;
    wmove(loadsave_win, 0, 2);
    wattron(loadsave_win, A_BOLD);
    set_color(loadsave_win, White);
    waddch(loadsave_win, ACS_RTEE);
    set_color(loadsave_win, Yellow);
    wprintw(loadsave_win, " %s ", title);
    set_color(loadsave_win, White);
    waddch(loadsave_win, ACS_LTEE);
    wattroff(loadsave_win, A_BOLD);
    wattron(loadsave_win, A_BOLD);
    curs_set(1);
    wmove(loadsave_win, 1, 2);
    wprintw(loadsave_win, "%s", prompt);
    wattron(loadsave_win, A_REVERSE);
    for (j = strlen(prompt) + 2 ; j < COLS - 2; j++)
        wprintw(loadsave_win, " ");
    wmove(loadsave_win, 1, 2 + strlen(prompt));
    wattroff(loadsave_win, A_REVERSE);
    wrefresh(loadsave_win);
}


static void close_loadsave() {
    curs_set(0);
    destroy_win(loadsave_win);
    loadsave_win = NULL;
    player_display = TRUE;
    display_player();
}


/* Commands functions */


static char *stripwhite(char *string) {
    register char *s, *t;

    for (s = string; whitespace(*s); s++);
    if (*s == 0) return (s);

    t = s + strlen(s) - 1;
    while ((t > s) && whitespace(*t)) t--;
    *++t = '\0';

    return s;
}


static void display_rlinput() {
    int   j, x = 0, y = 0, dpos;
    int   len = rl_line_buffer ? strlen(rl_line_buffer) : 0;
    char *buffer = NULL;

    getmaxyx(stdscr, y, x);
    if ((int) strlen(prompt) + len > x - 5) {
        dpos = len - x + 5 + strlen(prompt);
        buffer = rl_line_buffer + dpos;
        len = strlen(buffer);
    } else {
        dpos = 0;
        buffer = rl_line_buffer;
    }
    wmove(loadsave_win, 1, 2);
    wprintw(loadsave_win, "%s", prompt);
    wattron(loadsave_win, A_REVERSE);
    wprintw(loadsave_win, "%s", buffer ? buffer : "");
    for (j = len + strlen(prompt) + 2 ; j < x - 2; j++)
        wprintw(loadsave_win, " ");
    wmove(loadsave_win, 1, 2 + strlen(prompt) + rl_point - dpos);
    wattroff(loadsave_win, A_REVERSE);
    wrefresh(loadsave_win);
}


static void multi_matches(char **matches, int num_matches, int max_length) {
/*
    rl_delete_text(0, rl_end);
    rl_insert_text(matches[1]);
    display_rlinput();
*/
    return;
}


static void init_readline() {
    rl_readline_name = PROGRAM;
    rl_redisplay_function = display_rlinput;
    rl_completion_display_matches_hook = multi_matches;
    rl_completion_append_character = '\0';
    rl_basic_word_break_characters = "\t\n\"\\'`@$><=;|&{(";
    rl_filename_quote_characters = " ";
    //rl_catch_signals = 0;
}


static char *get_playlist(char *title) {
    char *line, *filename = NULL;

    player_display = FALSE;
    display_loadsave(title);
    init_readline();
    line = readline(NULL);
    if (line && (line[0] != 0)) {
        filename = xstrdup(stripwhite(line));
        if (strncmp(filename, "~/", 2) == 0) {
            char *Home     = robust_home(NULL);
            int   len      = strlen(filename) + strlen(Home) + 3;
            char *expanded = xmalloc(len);
            snprintf(expanded, len - 1, "%s/%s", Home, filename + 2);
            free(filename);
            filename = expanded;
        } else if (filename[0] == '~') {
            /* XXX should be replaced by the real user's home directory */
            char *Home     = "/home";
            int   len      = strlen(filename) + strlen(Home) + 3;
            char *expanded = xmalloc(len);
            snprintf(expanded, len - 1, "%s/%s", Home, filename + 1);
            free(filename);
            filename = expanded;
        }
        free(line);
    }

    return filename;
}


int load_fileordir() {
    char *pl = get_playlist("Load playlist...");

    if (pl) {
        playlist_free();
        set_command(STOPPED);
        load_playlist(pl);
        free(pl);
        reinit_playlistcursor();
    }
    close_loadsave();
    display_playlist();

    return CONT;
}


int save_fileordir() {
    char *pl = get_playlist("Save playlist...");

    if (pl) {
        struct stat s;
        int ok = TRUE;
        if (stat(pl, &s) != -1) {
            ok = FALSE;
            if (S_ISREG(s.st_mode)) {
                int rep;
                mvwprintw(loadsave_win, 2, 2,
                        "File exists. Overwrite (O/N) ? ");
                wrefresh(loadsave_win);
                do {
                    rep = wgetch(stdscr);
                    rep = toupper(rep);
                } while ((rep != 'Y') && (rep != 'N'));
                if (rep == 'Y') ok = TRUE;
            }
        }
        if (ok) save_playlist(pl);
        free(pl);
    }
    close_loadsave();

    return CONT;
}


int add_fileorstream() {
    char *pl = get_playlist("Add title/stream...");

    if (pl) {
        struct stat s;
        if (((stat(pl, &s) != -1) && (S_ISREG(s.st_mode))) ||
            (strncmp(pl, "http://", 7) == 0))
            playlist_add(pl);
        free(pl);
    }
    close_loadsave();

    return CONT;
}

