/*
 * playerwin.c
 * Thomas Nemeth, le 15.10.2003
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include "defines.h"
#include <ncurses.h>
#include "windows.h"
#include "commands.h"
#include "tmms.h"
#include "xmem.h"
#include "playlist.h"
#include "playerwin.h"
#include "playlistwin.h"
#include "modules.h"
#include "control.h"


int            player_display = TRUE;
static WINDOW *player_win = NULL;



/* Display functions */

void display_player() {
    if (player_win) destroy_win(player_win);
    player_win = create_newwin(4, COLS, 0, 0);
    display_playertitle();
    display_time(0, 0, 0, 0);
    display_status();
    display_song();
}


void display_playertitle() {
    pthread_mutex_lock(&mutex_display);
    wmove(player_win, 0, 2);
    wattron(player_win, A_BOLD);
    set_color(player_win, White);
    waddch(player_win, ACS_RTEE);
    set_color(player_win, Yellow);
    wprintw(player_win, " "PACKAGE" ");
    set_color(player_win, White);
    waddch(player_win, ACS_LTEE);
    wattroff(player_win, A_BOLD);
    wrefresh(player_win);
    pthread_mutex_unlock(&mutex_display);
}


void display_time(int mins, int secs, int rms, int rss) {
    char text[10];

    if (! player_display) return;
    bzero(text, 10);
    pthread_mutex_lock(&mutex_display);
    switch (timedata) {
        case NORMAL: snprintf(text, 9, "%02d:%02d  ", mins, secs); break;
        case REMAIN: snprintf(text, 9, "%02d:%02d  ", rms, rss);   break;
        default:     snprintf(text, 9, "%02d:%02d  ", mins, secs); break;
    }
    wattron(player_win, A_BOLD);
    set_color(player_win, Green);
    mvwprintw(player_win, 1, 2, text);
    wattroff(player_win, A_BOLD);
    wrefresh(player_win);
    pthread_mutex_unlock(&mutex_display);
}


void display_status() {
    Status s = get_status();
    char *status_str;
    char *mode_str;
    char *play_str;

    if (! player_display) return;
    pthread_mutex_lock(&mutex_display);
    switch (s) {
        case PLAYING : status_str = "|>"; break;
        case PAUSED  : status_str = "||"; break;
        case STOPPED : status_str = "[]"; break;
        case FFW     : status_str = ">>"; break;
        case RWD     : status_str = "<<"; break;
        case NEXT    : status_str = ">|"; break;
        case PREV    : status_str = "|<"; break;
        default      : status_str = "[]"; break;
    }

    switch (mode) {
        case REPEAT_ONE : mode_str = "R1"; break;
        case REPEAT_ALL : mode_str = "RA"; break;
        case REPEAT_NUL : mode_str = "  "; break;
        default         : mode_str = "  "; break;
    }

    switch (get_playmode()) {
        case NRM_PLAY : play_str = "   "; break;
        case RND_PLAY : play_str = "RND"; break;
        default       : play_str = "   "; break;
    }

    wattron(player_win, A_BOLD);
    set_color(player_win, Cyan);
    mvwprintw(player_win, 1, 12, status_str);
    mvwprintw(player_win, 1, 18, mode_str);
    mvwprintw(player_win, 1, 24, play_str);
    wattroff(player_win, A_BOLD);
    wrefresh(player_win);
    pthread_mutex_unlock(&mutex_display);
}


void display_song() {
    char *file_title;
    char *spaces;
    unsigned int j;

    if (! player_display) return;
    spaces = xmalloc(COLS - 4);
    pthread_mutex_lock(&mutex_display);
    set_color(player_win, White);
    if (playlist_current())
        file_title = strrchr(playlist_current(), '/') + 1;
    else
        file_title = "NO FILE...";

    if ((! file_title) || (file_title[0] == 0)) file_title = playlist_current();

    for (j = 0 ; j < COLS - 4 - strlen(file_title) ; j++) spaces[j] = ' ';
    spaces[j] = 0;
    mvwprintw(player_win, 2, 2, "%s%s", file_title, spaces);
    wrefresh(player_win);
    free(spaces);
    pthread_mutex_unlock(&mutex_display);
}


/* Commands functions */

int play_pause() {
    Status s1 = get_status();
    Status s2;
    struct timespec ts = { 0, 1000 };

    if (playlist_max() == 0) return CONT;
    if (s1 == PLAYING)
        s2 = PAUSED;
    else if ((s1 == PAUSED) || (s1 == STOPPED))
        s2 = PLAYING;
    else
        s2 = STOPPED;

    set_command(s2);
    while (s1 != s2) {
        s1 = get_status();
        nanosleep(&ts, NULL);
    }

    return CONT;
}


int stop() {
    Status s = get_status();
    struct timespec ts = { 0, 1000 };

    if (playlist_max() == 0) return CONT;
    set_command(STOPPED);
    while (s != STOPPED) {
        s = get_status();
        nanosleep(&ts, NULL);
    }

    return CONT;
}


int fforward() {
    if (playlist_max() != 0) set_command(FFW);
    return CONT;
}


int frewind() {
    if (playlist_max() != 0) set_command(RWD);
    return CONT;
}


int next_file() {
    if (playlist_max() != 0) set_command(NEXT);
    return CONT;
}


int prev_file() {
    if (playlist_max() != 0) set_command(PREV);
    return CONT;
}


int repeat_mode() {
    switch(mode) {
        case REPEAT_NUL: mode = REPEAT_ONE; break;
        case REPEAT_ONE: mode = REPEAT_ALL; break;
        case REPEAT_ALL: mode = REPEAT_NUL; break;
        default:         mode = REPEAT_NUL;
    }
    display_status();
    return CONT;
}


int time_data() {
    if (timedata == NORMAL)
        timedata = REMAIN;
    else
        timedata = NORMAL;
    return CONT;
}


int toggle_pm() {
    toggle_playmode();
    display_status();
    return CONT;
}
