/*
 * playlistwin.c
 * Thomas Nemeth, le 15.10.2003
 *
 *   Copyright (C) 1999  Thomas Nemeth
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#include "config.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include "defines.h"
#include <ncurses.h>
#include "windows.h"
#include "commands.h"
#include "tmms.h"
#include "xmem.h"
#include "playlist.h"
#include "playerwin.h"
#include "playlistwin.h"
#include "printlog.h"


static int mv_up();
static int mv_dn();
static int mv_pu();
static int mv_pd();
static int mv_home();
static int mv_end();
static int mv_done();



int            playlist_display = TRUE;
int            mv_mode = FALSE;
static WINDOW *playlist_win = NULL;
static ListID  start = 0, cursor = 0;



static Cmdslst move_cmds[] = {
    {KEY_UP,     mv_up},
    {KEY_DOWN,   mv_dn},
    {KEY_PPAGE,  mv_pu},
    {KEY_NPAGE,  mv_pd},
    {KEY_HOME,   mv_home},
    {KEY_END,    mv_end},
    {KEY_ENTER,  mv_done},
    {10,         mv_done},
    {KEY_RESIZE, redisp},
    {0,          NULL}
};


void display_playlist() {
    int x, y;

    if (! playlist_display) return;
    if (playlist_win) destroy_win(playlist_win);
    getmaxyx(stdscr, y, x);
    playlist_win = create_newwin(y - 4, x, 4, 0);
    printlog(1, "Taille cran : %dLx%dC\n", y, x);
    cursor = get_current();
    display_playlisttitle();
    display_list();
}


void display_playlisttitle() {
    char *playlist_name;
    pthread_mutex_lock(&mutex_display);
    if (playlist_display) {
        playlist_name = get_playlist_name() ? get_playlist_name() : "none";
        wattron(playlist_win, A_BOLD);
        set_color(playlist_win, White);
        wmove(playlist_win, 0, 2);
        waddch(playlist_win, ACS_RTEE);
        wprintw(playlist_win, " Playlist : "); // 12 cars
        if ((int)strlen(playlist_name) + 12 + 7 > (int)COLS) {
            playlist_name = strrchr(get_playlist_name(), '/');
            if (playlist_name) playlist_name++;
            if ((playlist_name && (
                    ((int)strlen(playlist_name) + 12 + 7 > (int)COLS))) ||
                    (! playlist_name))
                playlist_name = "";
        }
        wprintw(playlist_win, "%s ", playlist_name);
        waddch(playlist_win, ACS_LTEE);
        wattroff(playlist_win, A_BOLD);
        wrefresh(playlist_win);
    }
    pthread_mutex_unlock(&mutex_display);
}


void display_entry(ListID i, ListID current, char *text, char *spaces) {
    if (cursor == i) wattron(playlist_win, A_REVERSE);
    if (current == i) wattron(playlist_win, A_BOLD);
    set_color(playlist_win, Green);
    wmove(playlist_win, i - start + 1, 1);
    if (current == i)
        wprintw(playlist_win, ">%3d. ",
                playlist_get_sel(i) != 0 ? playlist_get_sel(i) : i);
    else
        wprintw(playlist_win, " %3d. ",
                playlist_get_sel(i) != 0 ? playlist_get_sel(i) : i);
    if (mv_mode && (cursor == i)) {
        set_color(playlist_win, Cyan);
    } else {
        if (playlist_get_sel(i) != 0)
            set_color(playlist_win, Magenta);
        else
            set_color(playlist_win, White);
    }
    wmove(playlist_win, i - start + 1, 7); 
    wprintw(playlist_win, "%s%s", text, spaces);
    if (current == i) wattroff(playlist_win, A_BOLD);
    if (cursor == i) wattroff(playlist_win, A_REVERSE);
}


void clean_lastlines(ListID last, unsigned int x, unsigned int y) {
    ListID i;
    for (i = last - start + 2 ; i < y - 5 ; i++) {
        unsigned int j;
        char        *spaces = xmalloc(x - 3);

        for (j = 0 ; j < x - 4; j++) spaces[j] = ' ';
        spaces[j] = 0;
        wmove(playlist_win, i, 1); 
        wprintw(playlist_win, "%s", spaces);
        free(spaces);
    }
}


void get_display_entry(ListID i, unsigned int x, char **entry, char **spaces) {
    unsigned int j;
    char *text = strrchr(playlist_get(i), '/');
    char *s = NULL, *e = NULL;

    if ((! text) || (text[0] == 0))
        text = playlist_get(i);
    else
        text++;

    s = xmalloc(x - 7);
    bzero(s, x - 7);
    if (strlen(text) > x - 8) {
        e = xmalloc(x - 6);
        bzero(e, x - 6);
        strncpy(e, text, x - 8);
    } else {
        e = xstrdup(text);
        for (j = 0 ; j < x - 8 - strlen(text) ; j++) s[j] = ' ';
    }
    *entry = e;
    *spaces = s;
}


void display_list() {
    ListID       i, last, current = get_current();
    unsigned int x, y;

    if (! playlist_display) return;
    pthread_mutex_lock(&mutex_display);
    if (start == 0) start = 1;
    getmaxyx(stdscr, y, x);
    last = (start + y - 7 > playlist_max() ? playlist_max() : start + y - 7);

    for (i = start ; i <= last ; i++) {
        char *entry = NULL, *spaces = NULL;

        get_display_entry(i, x, &entry, &spaces);
        display_entry(i, current, entry, spaces);
        FREE(entry);
        FREE(spaces);
    }
    clean_lastlines(last, x, y);
    wrefresh(playlist_win);
    pthread_mutex_unlock(&mutex_display);
}


void reinit_playlistcursor() {
    cursor = 0;
    start = 0;
}


/* Common commands */

static void up() {
    int x, y;

    getmaxyx(stdscr, y, x);
    cursor--;
    if (cursor == 0) cursor = playlist_max();
    while (cursor < start) start--;
    while (cursor > start + y - 7) start++;
}


static void down() {
    int x, y;

    getmaxyx(stdscr, y, x);
    cursor++;
    if (cursor > playlist_max()) cursor = 1;
    while (cursor > start + y - 7) start++;
    while (cursor < start) start--;
}


static void home() {
    cursor = 1;
    start = 1;
}


static void end() {
    int x, y;

    getmaxyx(stdscr, y, x);
    cursor = playlist_max();
    start = cursor - y + 7;
    if (start <= 0) start = 1;
}


static void page_up() {
    int x, y;

    getmaxyx(stdscr, y, x);
    cursor -= y - 7;
    if (cursor <= 0) cursor = 1;
    start = cursor;
}


static void page_down() {
    int x, y;

    getmaxyx(stdscr, y, x);
    cursor += y - 7;
    if (cursor > playlist_max()) cursor = playlist_max();
    start = cursor - y + 7;
    if (start <= 0) start = 1;
}


/* Commands */

int previous() {
    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return prev_file();
    up();
    display_list();
    return CONT;
}


int next() {
    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return next_file();
    down();
    display_list();
    return CONT;
}


int first() {
    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    home();
    display_list();
    return CONT;
}


int last() {
    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    end();
    display_list();
    return CONT;
}


int pgup() {
    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    page_up();
    display_list();
    return CONT;
}


int pgdn() {
    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    page_down();
    display_list();
    return CONT;
}


int enter() {
    int cont;
    int nbs = playlist_nb_sel();
    int sel = playlist_get_sel(cursor);
    int title = cursor;

    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return play_pause();
    if ((nbs != 0) && (sel == 0)) title = playlist_first_sel();
    set_current(title);
    cont = stop();
    return play_pause();
}


int remove_entry() {
    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    playlist_del(cursor);
    display_list();
    return CONT;
}


int select_entry() {
    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    playlist_tsel(cursor);
    return next();
}


int sel_all() {
    ListID i;

    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    for (i = 1 ; i <= playlist_max() ; i++) {
        playlist_sel(i, TRUE);
    }
    display_list();
    return CONT;
}


int unsel_all() {
    ListID i;

    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    for (i = 1 ; i <= playlist_max() ; i++) {
        playlist_sel(i, FALSE);
    }
    display_list();
    return CONT;
}


int invert_sel() {
    ListID i;

    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    for (i = 1 ; i <= playlist_max() ; i++) {
        playlist_tsel(i);
    }
    display_list();
    return CONT;
}


int move_entry() {
    int cont = CONT;

    if (playlist_max() == 0) return CONT;
    if (! playlist_display) return CONT;
    mv_mode = TRUE;
    display_list();
    while (cont == CONT) {
        cont = command_loop(move_cmds);
    }

    return CONT;
}


static int mv_up() {
    int pos = cursor;

    up();
    playlist_move(pos, cursor);
    display_list();
    return CONT;
}


static int mv_dn() {
    int pos = cursor;

    down();
    playlist_move(pos, cursor);
    display_list();
    return CONT;
}


static int mv_pu() {
    int pos = cursor;

    page_up();
    playlist_move(pos, cursor);
    display_list();
    return CONT;
}


static int mv_pd() {
    int pos = cursor;

    page_down();
    playlist_move(pos, cursor);
    display_list();
    return CONT;
}


static int mv_home() {
    int pos = cursor;

    home();
    playlist_move(pos, cursor);
    display_list();
    return CONT;
}


static int mv_end() {
    int pos = cursor;

    end();
    playlist_move(pos, cursor);
    display_list();
    return CONT;
}


static int mv_done() {
    mv_mode = FALSE;
    display_list();
    return STOP;
}

