/* $USAGI: ipxfrm.c,v 1.4 2004/04/27 10:56:34 nakam Exp $ */

/*
 * A frontend for xfrm like iproute2 command
 */
/*
 * Authors:
 *	Noriaki TAKAMIYA @USAGI
 *	Masahide NAKAMURA @USAGI
 */
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#define _GNU_SOURCE
#include <getopt.h>

#include <time.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <asm/types.h>

#include <net/if.h>

#include <linux/xfrm.h>

#include <mip6.h>
#include <libxfnl.h>
#include <libutil.h>

#define CMD_NAME	"ipxfrm"
#define CMD_VERSION	"0.0.1"

static const char opt_str[] = "f:v";

static const struct option opt_table[] = {
	{"family", 1, 0, 'f'},

	{"verbose", 0, 0, 'v'},

	{"version", 0, 0, 0},
	{"help", 0, 0, 0},

	{0, 0, 0, 0}
};

static const char *opt_arg_name[] = {
	"family",

	NULL,

	NULL,
	NULL,

	NULL
};

static const char *opt_desc[] = {
	"address family := { inet | inet6 }(default=inet6)",

	"verbose output",

	"show version",
	"show this help",

	NULL
};

#if 0
int preferred_family = AF_UNSPEC;
#else
int preferred_family = AF_INET6;
#endif
static int is_verbose_mode = 0; /* XXX: quick hacking... */
static int error_code = 0; /* XXX: quick hacking... */

static void version()
{
	printf("%s %s\n", CMD_NAME, CMD_VERSION);
}

static void help_base()
{
	char opt_buf[1024];
	char lopt_buf[1024];
	int i;

	/* XXX: global options */
	printf("Usage: %s [options]\n", CMD_NAME);

	for (i = 0; ; i ++) {
		if (opt_table[i].name == 0)
			break;

		if (opt_table[i].val != 0)
			sprintf(opt_buf, "-%c,", (char)opt_table[i].val);
		else
			sprintf(opt_buf, "   ");

		if (opt_table[i].has_arg == 1)
			sprintf(lopt_buf, "--%s=<%s>",
				opt_table[i].name, opt_arg_name[i]);
		else
			sprintf(lopt_buf, "--%s", opt_table[i].name);

		printf("  %s %-*s %s\n",
			opt_buf,
			21, /* format-length for long option name */
			lopt_buf,
			opt_desc[i]);
	}

	/* basic syntax */
	printf("Usage: %s [ OPTION ] OBJECT COMMAND\n", CMD_NAME);
	printf("OBJECT := { state | policy }\n");
}

static void help_state()
{
	printf("Usage: %s state { add | change } DADDR SADDR PROTO\n", CMD_NAME);
#ifdef USE_IPSEC_SA
	printf("           [ mode MODE ] [ spi SPI ] [ PROTO_OPT ] [ sel SELECTOR ]\n");
#else
	printf("           [ coa COADDR ] [ sel SELECTOR ]\n");
#endif
	printf("       %s state { del | get } DADDR SADDR PROTO\n", CMD_NAME);
	printf("       %s state { flush }\n", CMD_NAME);

	printf("DADDR - destination address\n");
	printf("SADDR - source address\n");

#ifdef USE_IPSEC_SA
	printf("PROTO := esp | ah | route2 | hao | ipv6\n");

 	printf("MODE := transport | tunnel (default=transport)\n");

	printf("SPI - security parameter index(default=0)\n");

	printf("PROTO_OPT :=  algo ALGO [algo ALGO...] | coa COADDR\n");

	printf("ALGO := ALGOTYPE ALGONAME ALGOKEY\n");
	printf("ALGOTYPE := A | E | C\n");
	printf("ALGONAME - algorithm name\n");
	printf("ALGOKEY - algorithm key\n");
#else
	printf("PROTO := route2 | hao | ipv6\n");
#endif
	printf("COADDR - care-of address\n");

	printf("SELECTOR := DADDR SADDR [ upspec UPSPEC [ UPSPEC_OPT ] ] [ dev DEV ]\n");

	printf("UPSPEC - upper layer protocol name or number(default=0)\n");
	printf("UPSPEC_OPT - port DPORT SPORT | type TYPE (UPSPEC=icmp6,mh)\n");
	printf("DEV - device name(default=none)\n");
}

static void help_policy()
{
	printf("Usage: %s policy { add | change } DIR SELECTOR [ priority PRIORITY ] TMPLS\n", CMD_NAME);
	printf("       %s policy { del | get } DIR SELECTOR\n", CMD_NAME);
	printf("       %s policy { flush }\n", CMD_NAME);

	printf("DIR := in | out | fwd [ ACTION ]\n");
	printf("SELECTOR := DADDR SADDR [ upspec UPSPEC [ UPSPEC_OPT ] ] [ dev DEV ]\n");
	printf("PRIORITY - priority value(default=0)\n");
	printf("TMPLS := tmpl TMPL [ tmpl TMPL ...](max count=%d)\n", TMPL_MAX);

	printf("ACTION := allow | block(default=allow)\n");
	printf("DADDR - destination address\n");
	printf("SADDR - source address\n");
	printf("UPSPEC - upper layer protocol name or number(default=0)\n");
	printf("UPSPEC_OPT - port DPORT SPORT | type TYPE (icmp6,mh)\n");	printf("DEV - device name(default=none)\n");

	printf("TMPL := PROTO DADDR SADDR [ mode MODE ] [ level LEVEL ]\n");
	printf("PROTO := esp | ah | route2 | hao | ipv6\n");
 	printf("MODE := transport | tunnel (default=transport)\n");
	printf("LEVEL := required | use (default=required)\n");
}

static void help(char *object)
{
	if (object) {
		if (!strcmp(object, "state")) {
			help_state();
			return;
		} else if (!strcmp(object, "policy")) {
			help_policy();
			return;
		}
	}
	help_base();
}

static void usage()
{
#if 0
	printf("Try `%s --help` for more information.\n", CMD_NAME);
#endif
}

static void dump_selector(const struct xfrm_selector *sel, const char *prefix)
{
	char dev_str[IF_NAMESIZE];

	if (prefix)
		printf(prefix);
	printf("%s", strxfaddr(&sel->daddr, preferred_family));
	printf("/%d", sel->prefixlen_d);
	if (sel->proto != IPPROTO_ICMPV6 && sel->proto != IPPROTO_MH)
		printf("[%d]", sel->dport);
	printf(" ");

	printf("%s", strxfaddr(&sel->saddr, preferred_family));
	printf("/%d", sel->prefixlen_s);
	if (sel->proto != IPPROTO_ICMPV6 && sel->proto != IPPROTO_MH)
		printf("[%d]", sel->sport);

	printf("\n");
	if (prefix)
		printf(prefix);
	printf("\t");

	printf("upspec=%s", strproto(sel->proto));
	if (sel->proto == IPPROTO_ICMPV6)
		;
	else if(sel->proto == IPPROTO_MH)
		printf(" type=%s", strmh(sel->mh_type));
	printf(" ");

	//printf("ifindex=%d", sel->ifindex);
	if (sel->ifindex == 0)
		strcpy(dev_str, "none");
	else {
		memset(dev_str, '\0', sizeof(dev_str));
		if_indextoname(sel->ifindex, dev_str);
	}
	printf("dev=%s", dev_str);

	printf(" ");
	printf("uid=%d", sel->user);

	printf("\n");
}

static void dump_lifetime(const struct xfrm_lifetime_cfg *cfg,
			  const struct xfrm_lifetime_cur *cur,
			  const char *prefix)
{
	if (cfg) {
		if (prefix)
			printf(prefix);
		printf("lifetime config:\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("limit: ");
		printf("soft=");
		printf(strxflimit(cfg->soft_byte_limit));
		printf(" bytes, hard=");
		printf(strxflimit(cfg->hard_byte_limit));
		printf(" bytes\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("limit: ");
		printf("soft=");
		printf(strxflimit(cfg->soft_packet_limit));
		printf(" packets, hard=");
		printf(strxflimit(cfg->hard_packet_limit));
		printf(" packets\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("expire add: ");
		printf("soft=");
		printf("%llu", cfg->soft_add_expires_seconds);
		printf(" sec, hard=");
		printf("%llu", cfg->hard_add_expires_seconds);
		printf(" sec\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("expire use: ");
		printf("soft=");
		printf("%llu", cfg->soft_use_expires_seconds);
		printf(" sec, hard=");
		printf("%llu", cfg->hard_use_expires_seconds);
		printf(" sec\n");
	}
	if (cur) {
		if (prefix)
			printf(prefix);
		printf("lifetime current:\n");

		if (prefix)
			printf(prefix);
		printf("  ");
		printf("%llu bytes, ", cur->bytes);
		printf("%llu packets\n", cur->packets);
		if (prefix)
			printf(prefix);
		printf("  ");
		printf("add %s ", strxftime(cur->add_time));
		printf("use %s", strxftime(cur->use_time));
		printf("\n");
	}
}

static void dump_stats(const struct xfrm_stats *stats, const char *prefix)
{
	if (prefix)
		printf(prefix);
	printf("stats:\n");

	if (prefix)
		printf(prefix);
	printf("  ");
	printf("replay-window=%d ", stats->replay_window);
	printf("replay=%d ", stats->replay);
	printf("failed=%d", stats->integrity_failed);
	printf("\n");
}

static void dump_id(const struct xfrm_id *id, const xfrm_address_t *saddr,
		    __u8 mode, const char *prefix)
{
	if (prefix)
		printf(prefix);
	printf("%s", strxfaddr(&id->daddr, preferred_family));
	printf(" ");
	printf("%s", strxfaddr(saddr, preferred_family));

	printf("\n");
	if (prefix)
		printf(prefix);
	printf("\t");

	printf("spi=0x%08u", id->spi);
	printf(" ");
	printf("%s", mip6_strproto(id->proto));
	printf(" ");

	if (mode)
		printf("tunnel");
	else
		printf("transport");
	printf("\n");
}

static void dump_tmpl(const struct xfrm_user_tmpl *tmpl, const char *prefix)
{
	dump_id(&tmpl->id, &tmpl->saddr, tmpl->mode, prefix);

	if (is_verbose_mode) {
		if (prefix)
			printf(prefix);
		printf("\t");

		if (tmpl->optional)
			printf("optional");
		else
			printf("required");
		printf(" ");

		printf("reqid=%d ", tmpl->reqid);
		printf("share=%s", strxfshare(tmpl->share));
		printf("\n");
	}
}

static void dump_policy(const struct xfrm_userpolicy_info *pol,
			const struct xfrm_user_tmpl *tmpls, int num_tmpls)
{
	int i;

	dump_selector(&pol->sel, NULL);

	printf("\t");

	printf("%s ", strxfpolicy_dir(pol->dir));
	printf("%s ", strxfpolicy_action(pol->action));

	printf("index=%d ", pol->index);
	printf("priority=%d ", pol->priority);

	printf("flags=%s", strxfpolicy_flags(pol->flags));

	printf("\n");

	if (is_verbose_mode)
		dump_lifetime(&pol->lft, &pol->curlft, "\t");

	/* Dump xfrm_templates */
	for(i=0; i < num_tmpls; i++){
		printf("\ttemplate %d:\n", i+1);
		dump_tmpl(&tmpls[i], "\t  ");
	}
}

static int dump_policy_handler(const struct xfnl_handle *xh,
			       const struct xfrm_userpolicy_info *pol, 
			       const struct xfrm_user_tmpl *tmpls,
			       int num_tmpls)
{
	dump_policy(pol, tmpls, num_tmpls);
	return 0;
}

static void dump_algo(const struct xfrm_algo *algo, __u16 type,
		      const char *prefix)
{
	int len;
	int i;

	if (prefix)
		printf(prefix);
	switch (type) {
	case XFRMA_ALG_AUTH:
		printf("A");
		break;
	case XFRMA_ALG_CRYPT:
		printf("E");
		break;
	case XFRMA_ALG_COMP:
		printf("C");
		break;
	default:
		printf("?");
		break;
	}

	printf("=%s", algo->alg_name);

	len = algo->alg_key_len / 8;
	for (i = 0; i < len; i ++) {
		if (i % 4 == 0)
			printf(" ");
		printf("%x", algo->alg_key[i]);
	}

	printf("\n");
}

static void dump_att(const struct xfnl_att *att, const char *prefix)
{
	if (prefix)
		printf(prefix);
	switch (att->type) {
	case XFRMA_ALG_AUTH:
	case XFRMA_ALG_CRYPT:
	case XFRMA_ALG_COMP:
		dump_algo(&att->algo, att->type, NULL);
		break;

#ifdef USE_XFRM_ENHANCEMENT
	case XFRMA_ADDR:
		/* Dump coa */
		printf("coa=%s\n", strxfaddr(&att->coaddr, preferred_family));
		break;
#endif

	default:
		printf("unknown-data type=%d\n", att->type);
		break;
	}
}

static void dump_state(const struct xfrm_usersa_info *st,
		       const struct xfnl_att *atts, int natts)
{
	int i;

	/* xfrm_id */
	dump_id(&st->id, &st->saddr, st->mode, NULL);

	if (is_verbose_mode) {

		printf("\t");
		printf("seq=0x%08u ", st->seq);
		printf("reqid=%d ", st->reqid);
		printf("replay-window=%d", st->replay_window);

		printf("\n");

		printf("\t");
		printf("selector:");
		printf("\n");
		dump_selector(&st->sel, "\t  ");

		/* xfrm_lifetime_cfg/cur */
		dump_lifetime(&st->lft, &st->curlft, "\t");

		/* xfrm_stats */
		dump_stats(&st->stats, "\t");
	}

	for (i = 0; i < natts; i++)
		dump_att(&atts[i], "\t");

}

static int dump_state_handler(const struct xfnl_handle *xh,
			      const struct xfrm_usersa_info *st,
			      const struct xfnl_att *atts, int natts)
{
	dump_state(st, atts, natts);
	return 0;
}

static int del_policy(const struct xfnl_handle *xh,
		      const struct xfrm_userpolicy_info *pol)
{
	struct xfnl_handle xh2;

	memset(&xh2, 0, sizeof(xh2));

	xh2.sock = xh->sock;
	xh2.family = xh->family;

	xh2.type = XFRM_MSG_DELPOLICY;

	xh2.error_handler = NULL;
	xh2.probe_policy_handler = NULL;
	xh2.probe_state_handler = NULL;

	memcpy(&xh2.sel, &pol->sel, sizeof(xh2.sel));
	xh2.pol.index = pol->index;
	xh2.pol.dir = pol->dir;

	return xfnl_talk(&xh2);
}

static int del_policy_handler(const struct xfnl_handle *xh,
			      const struct xfrm_userpolicy_info *pol,
			      const struct xfrm_user_tmpl *tmpls,
			      int num_tmpls)
{
	int ret;
	ret = del_policy(xh, pol); /* XXX: ignore each error when flushing */
	return 0;
}


static int del_state(const struct xfnl_handle *xh,
		     const struct xfrm_usersa_info *st)
{
	struct xfnl_handle xh2;

	memset(&xh2, 0, sizeof(xh2));

	xh2.sock = xh->sock;
	xh2.family = xh->family;

	xh2.type = XFRM_MSG_DELSA;

	xh2.error_handler = NULL;
	xh2.probe_policy_handler = NULL;
	xh2.probe_state_handler = NULL;

	xh2.st.id.spi = st->id.spi;
	xh2.st.id.proto = st->id.proto;
	memcpy(&xh2.st.id.daddr, &st->id.daddr, sizeof(xh2.st.id.daddr));
	memcpy(&xh2.st.id.saddr, &st->saddr, sizeof(xh2.st.id.saddr));

	return xfnl_talk(&xh2);
}

static int del_state_handler(const struct xfnl_handle *xh,
			     const struct xfrm_usersa_info *st,
			     const struct xfnl_att *atts, int natts)
{
	int ret;
	ret = del_state(xh, st); /* XXX: ignore each error when flushing */
	return 0;
}

static void error_handler(const struct xfnl_handle *xh,
			  const struct nlmsghdr *hdr, int error,
			  unsigned int len)
{
	error_code = error;
}

static int parse_dir(int argc, char **argv, int idx, __u8 *dir)
{
	char *arg;

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	if (strcmp(arg, "in") == 0)
		*dir = XFRM_POLICY_IN;
	else if (strcmp(arg, "out") == 0)
		*dir = XFRM_POLICY_OUT;
	else if (strcmp(arg, "fwd") == 0)
		*dir = XFRM_POLICY_FWD;
	else {
		printf("dir doesn't know: %s\n", arg);
		goto error;
	}

	return idx;
 error:
	return -EINVAL;
}

static int try_parse_action(int argc, char **argv, int idx, __u8 *action)
{
	char *arg;

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	if (strcmp(arg, "allow") == 0)
		*action = XFRM_POLICY_ALLOW;
	else if (strcmp(arg, "block") == 0)
		*action = XFRM_POLICY_BLOCK;
	else {
		/* back track */
		idx --;
	}

	return idx;
 error:
	return -EINVAL;
}

static int parse_address(int argc, char **argv, int idx, xfrm_address_t *addr,
			 __u8 *prefix)
{
	char *arg;
	char buf[1024];
	char *p;
	const xfrm_address_t *xp;

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];

	if (prefix) {
		if (strlen(arg) > sizeof(buf) - 1) {
			printf("buffer overflow:%s\n", arg);
			return -EINVAL;
		}
		strcpy(buf, arg);

		p = strrchr(arg, '/');
		if (p) {
			p[0] = '\0';
			p ++;
			*prefix = strtol(p, NULL, 0);
		} else
			*prefix = 128;
		p = buf;
	} else
		p = arg;

	xp = getxfaddr(p, preferred_family);
	if (!xp)
		goto error;
	memcpy(addr, xp, sizeof(*xp));

	return idx;
 error:
	return -EINVAL;
}

static int parse_addresses(int argc, char **argv, int idx,
			   xfrm_address_t *daddr, __u8 *dprefix,
			   xfrm_address_t *saddr, __u8 *sprefix)
{
	int ret;

	/* DADDR */
	ret = parse_address(argc, argv, idx, daddr, dprefix);
	if (ret < 0)
		goto error;
	idx = ret;

	/* SADDR */
	ret = parse_address(argc, argv, idx, saddr, sprefix);
	if (ret < 0)
		goto error;
	idx = ret;

	return idx;
 error:
	return -EINVAL;
}

static int parse_selector(int argc, char **argv, int idx,
			  struct xfnl_sel *sel)
{
	char *arg;
	int ifindex = 0;
	int ret;

	/* DADDR, SADDR */
	ret = parse_addresses(argc, argv, idx, &sel->daddr, &sel->prefixlen_d,
			      &sel->saddr, &sel->prefixlen_s);
	if (ret < 0)
		goto error;
	idx = ret;

	/* UPSPEC */
	if (idx >= argc)
		goto fin;
	arg = argv[idx ++];
	if (strcmp(arg, "upspec") != 0) {
		/* back track */
		idx --;
	} else {
		if (idx >= argc) {
			printf("missing arguments: %s\n", arg);
			goto error;
		}
		arg = argv[idx ++];
		sel->proto = getproto(arg);

		/* UPSPEC_OPT */
		if (sel->proto == IPPROTO_ICMPV6)
			;
		else if (sel->proto == IPPROTO_MH) {
			/* type TYPE */
			if (idx >= argc)
				goto fin;
			arg = argv[idx ++];
			if (strcmp(arg, "type") != 0) {
				/* back track */
				idx --;
			} else {
				if (idx >= argc)
					goto fin;
				arg = argv[idx ++];
				sel->mh_type = strtol(arg, NULL, 0);
			}
		} else {
			/* port DPORT SPORT */
			/*
			 * this had better be moved to address parsing phase.
			 */
			if (idx >= argc)
				goto fin;
			arg = argv[idx ++];
			if (strcmp(arg, "port") != 0) {
				/* back track */
				idx --;
			} else {
				if (idx >= argc)
					goto fin;
				arg = argv[idx ++];
				sel->dport = strtol(arg, NULL, 0);
				if (idx >= argc)
					goto fin;
				arg = argv[idx ++];
				sel->sport = strtol(arg, NULL, 0);
			}
		}
	}

	/* DEV */
	if (idx >= argc)
		goto fin;
	arg = argv[idx ++];
	if (strcmp(arg, "dev") != 0) {
		/* back track */
		idx --;
		goto fin;
	}
	if (idx >= argc) {
		printf("missing arguments: %s\n", arg);
		goto error;
	}
	arg = argv[idx ++];
	if (strcmp(arg, "none") == 0)
		ifindex = 0;
	else {
		ifindex = if_nametoindex(arg);
		if (ifindex <= 0) {
			printf("if_nametoindex(\"%s\"): %s\n", arg, strerror(errno));
			goto error;
		}
	}
	sel->ifindex = ifindex;

 fin:
	return idx;
 error:
	return -EINVAL;
}

static int parse_proto(int argc, char **argv, int idx,
		       struct xfnl_id *id)
{
	char *arg;

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];

	id->proto = mip6_getproto(arg);

	switch (id->proto) {
	case IPPROTO_ESP:
	case IPPROTO_AH:
	case IPPROTO_ROUTING:
	case IPPROTO_DSTOPTS:
	case IPPROTO_IPV6:
		break;
	default:
		printf("invalid proto: %s\n", arg);
		goto error;
	}

	return idx;
 error:
	return -EINVAL;
}

static int try_parse_mode(int argc, char **argv, int idx,
			  struct xfnl_id *id)
{
	char *arg;

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	if (strcmp(arg, "mode") != 0) {
		/* back track */
		idx --;
		goto fin;
	}

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	if (strcmp(arg, "transport") == 0)
		id->mode = 0;
	else if (strcmp(arg, "tunnel") == 0)
		id->mode = 1;
	else {
		printf("unknown mode: %s.\n", arg);
		goto error;
	}

 fin:
	return idx;
 error:
	return -EINVAL;
}

static int try_parse_level(int argc, char **argv, int idx,
			   struct xfnl_id *id)
{
	char *arg;

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	if (strcmp(arg, "level") != 0) {
		/* back track */
		idx --;
		goto fin;
	}

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	if (strcmp(arg, "required") == 0)
		id->level = 0;
	else if (strcmp(arg, "use") == 0)
		id->level = 1;
	else {
		printf("unknown level: %s.\n", arg);
		goto error;
	}

 fin:
	return idx;
 error:
	return -EINVAL;
}

#ifdef USE_IPSEC_SA
static int try_parse_algo(int argc, char **argv, int idx,
			  struct xfnl_state *st)
{
	char *arg;

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	if (strcmp(arg, "algo") != 0) {
		/* back track */
		idx --;
		goto fin;
	}

#if 1
	/* XXX: not implemented yet. */
	printf("not supported yet: ALGO\n");
	if (1) {
		idx --; /* back track for "algo" */
		goto error;
	}
#else
	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	; /* ALGOTYPE */

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	; /* ALGONAME */

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	; /* ALGOKEY */

	//st->atts[st->natts].type = XFRMA_ALG_{AUTH,CRYPT,COMP};
	st->natts ++;
#endif
 fin:
	return idx;
 error:
	return -EINVAL;
}
#endif

static int try_parse_coa(int argc, char **argv, int idx,
			 struct xfnl_state *st)
{
#ifdef USE_XFRM_ENHANCEMENT
	char *arg;
	int ret;

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	if (strcmp(arg, "coa") != 0) {
		/* back track */
		idx --;
		goto fin;
	}

	ret = parse_address(argc, argv, idx,
			    &st->atts[st->natts].coaddr, NULL);
	if (ret < 0)
		goto error;
	idx = ret;

	st->atts[st->natts].type = XFRMA_ADDR;
	st->natts ++;

	if (st->id.proto != IPPROTO_ROUTING &&
	    st->id.proto != IPPROTO_DSTOPTS) {
		printf("coa must be used with PROTO as ");
		printf("%s or ", mip6_strproto(IPPROTO_ROUTING));
		printf("%s.\n", mip6_strproto(IPPROTO_DSTOPTS));
		goto error;
	}

 fin:
	return idx;
#else /* USE_XFRM_ENHANCEMENT */
	printf("not compiled to handle coa.\n");
	goto error;
#endif /* USE_XFRM_ENHANCEMENT */
 error:
	return -EINVAL;
}

static int try_parse_priority(int argc, char **argv, int idx, __u32 *priority)
{
	char *arg;

	if (idx >= argc)
		goto error;
	arg = argv[idx ++];
	if (strcmp(arg, "priority") != 0) {
		/* back track */
		idx --;
		goto fin;
	}

	if (idx >= argc) {
		printf("missing arguments: %s\n", arg);
		goto error;
	}
	arg = argv[idx ++];
	*priority = strtol(arg, NULL, 0);

 fin:
	return idx;
 error:
	return -EINVAL;
}

static int policy_main(int argc, char **argv, int cur_idx)
{
	char *act;
	int is_flush = 0;
	int tmpl_idx;
	int err = -EINVAL;
	int ret;
	struct xfnl_handle handle;

	memset(&handle, 0, sizeof(handle));
	if (xfnl_open(&handle) < 0){
		printf("failed to open netlink socket: %s\n", strerror(errno));
		err = errno;
		goto usage_exit;
	}
	handle.family = preferred_family;

	if (cur_idx == argc) {
		handle.type = XFRM_MSG_GETPOLICY;
		handle.flag |= NLM_F_DUMP;
		goto talk;
	} else if (cur_idx > argc)
		goto usage_exit;
	/* COMMAND */
	act = argv[cur_idx ++];

	if (strcmp(act, "help") == 0) {
		err = 0;
		goto usage_exit;
	} else if (strcmp(act, "add") == 0)
		handle.type = XFRM_MSG_NEWPOLICY;
	else if (strcmp(act, "del") == 0)
		handle.type = XFRM_MSG_DELPOLICY;
	else if (strcmp(act, "get") == 0)
		handle.type = XFRM_MSG_GETPOLICY;
	else if (strcmp(act, "change") == 0)
		handle.type = XFRM_MSG_UPDPOLICY;
	else if (strcmp(act, "flush") == 0) {
		handle.type = XFRM_MSG_GETPOLICY;
		handle.flag |= NLM_F_DUMP;
		is_flush = 1;
		goto talk;
	} else {
		printf("invalid arguments: %s: unknown command.\n", act);
		goto usage_exit;
	}

	/* DIR */
	ret = parse_dir(argc, argv, cur_idx, &handle.pol.dir);
	if (ret < 0)
		goto usage_exit;
	cur_idx = ret;

	/* ACTION */
	ret = try_parse_action(argc, argv, cur_idx, &handle.pol.action);
	if (ret < 0)
		goto usage_exit;
	cur_idx = ret;

	/* SELECTOR */
	ret = parse_selector(argc, argv, cur_idx, &handle.sel);
	if (ret < 0)
		goto usage_exit;
	cur_idx = ret;

	/* PRIORITY */
	if (handle.type == XFRM_MSG_NEWPOLICY ||
	    handle.type == XFRM_MSG_UPDPOLICY) {
		if (cur_idx < argc) {
			ret = try_parse_priority(argc, argv, cur_idx,
						 &handle.pol.priority);
			if (ret < 0)
				goto usage_exit;
			cur_idx = ret;
		}
	}

	/* TMPLS */
	tmpl_idx = 0;
	while (cur_idx < argc) {
		int pre_idx = cur_idx;
		struct xfnl_id *tmpl = NULL;
		char *arg;

		/* tmpl TMPL */
		arg = argv[cur_idx ++];
		if (strcmp(arg, "tmpl") != 0) {
			/* back track */
			cur_idx = pre_idx;
			break;
		}

		if (tmpl_idx >= TMPL_MAX) {
			printf("too many TMPLs.\n");
			goto usage_exit;
		}
		tmpl = &handle.pol.tmpls[tmpl_idx ++];

		/* PROTO */
		ret = parse_proto(argc, argv, cur_idx, tmpl);
		if (ret < 0)
			break;
		cur_idx = ret;

		/* DADDR, SADDR */
		ret = parse_addresses(argc, argv, cur_idx,
				      &tmpl->daddr, NULL, &tmpl->saddr, NULL);
		if (ret < 0)
			goto usage_exit;
		cur_idx = ret;

		/* MODE */
		if (cur_idx >= argc)
			break;
		ret = try_parse_mode(argc, argv, cur_idx, tmpl);
		if (ret < 0)
			break;
		cur_idx = ret;

		/* LEVEL */
		if (cur_idx >= argc)
			break;
		ret = try_parse_level(argc, argv, cur_idx, tmpl);
		if (ret < 0)
			break;
		cur_idx = ret;
	}
	handle.pol.ntmpls = tmpl_idx;

	if (cur_idx < argc) {
		printf("too many arguments=%d (parsed=%d)\n", argc, cur_idx);
		goto usage_exit;
	} else if (cur_idx > argc) {
		printf("internal error: arguments=%d, parsed=%d\n", argc, cur_idx);
		goto usage_exit;
	}

 talk:
	handle.error_handler = error_handler;
	if (is_flush)
		handle.probe_policy_handler = del_policy_handler;
	else
		handle.probe_policy_handler = dump_policy_handler;

	err = xfnl_talk(&handle);

	xfnl_close(&handle);

	return err;

 usage_exit:
	usage();
	help_policy();
	exit((err > 0) ? err : -err);
}

static int state_main(int argc, char **argv, int cur_idx)
{
	char *act;
	int is_flush = 0;
	int err = -EINVAL;
	int ret;
	struct xfnl_handle handle;

	memset(&handle, 0, sizeof(handle));
	if (xfnl_open(&handle) < 0){
		printf("failed to open netlink socket: %s\n", strerror(errno));
		err = errno;
		goto usage_exit;
	}
	handle.family = preferred_family;

	if (cur_idx == argc) {
		handle.type = XFRM_MSG_GETSA;
		handle.flag |= NLM_F_DUMP;
		goto talk;
	} else if (cur_idx > argc)
		goto usage_exit;
	/* COMMAND */
	act = argv[cur_idx ++];

	if (strcmp(act, "help") == 0) {
		err = 0;
		goto usage_exit;
	} else if (strcmp(act, "add") == 0)
		handle.type = XFRM_MSG_NEWSA;
	else if (strcmp(act, "del") == 0)
		handle.type = XFRM_MSG_DELSA;
	else if (strcmp(act, "get") == 0)
		handle.type = XFRM_MSG_GETSA;
	else if (strcmp(act, "change") == 0)
		handle.type = XFRM_MSG_UPDSA;
	else if (strcmp(act, "flush") == 0) {
		handle.type = XFRM_MSG_GETSA;
		handle.flag |= NLM_F_DUMP;
		is_flush = 1;
		goto talk;
	} else {
		printf("invalid arguments: %s: unknown command.\n", act);
		goto usage_exit;
	}

	/* DADDR, SADDR */
	ret = parse_addresses(argc, argv, cur_idx,
			      &handle.st.id.daddr, NULL, &handle.st.id.saddr, NULL);
	if (ret < 0)
		goto usage_exit;
	cur_idx = ret;

	/* PROTO */
	ret = parse_proto(argc, argv, cur_idx, &handle.st.id);
	if (ret < 0)
		goto usage_exit;
	cur_idx = ret;

#ifndef USE_IPSEC_SA
	if (handle.st.id.proto != IPPROTO_ROUTING &&
	    handle.st.id.proto != IPPROTO_DSTOPTS) {
		printf("PROTO must be either ");
		printf("%s or ", mip6_strproto(IPPROTO_ROUTING));
		printf("%s.\n", mip6_strproto(IPPROTO_DSTOPTS));
		goto usage_exit;
	}
#endif

	if (handle.type == XFRM_MSG_NEWSA || handle.type == XFRM_MSG_UPDSA) {
		char *arg;

		/* MODE */
		if (cur_idx >= argc)
			goto fin;
		ret = try_parse_mode(argc, argv, cur_idx, &handle.st.id);
		if (ret < 0)
			goto usage_exit;
		cur_idx = ret;

#ifdef USE_IPSEC_SA
		/* SPI */
		if (cur_idx >= argc)
			goto fin;
		arg = argv[cur_idx ++];
		if (strcmp(arg, "spi") != 0) {
			/* back track */
			cur_idx --;
		} else {
			char *val;
			if (cur_idx >= argc) {
				printf("missing spi.\n");
				goto usage_exit;
			}
			val = argv[cur_idx ++];
			handle.st.id.spi = strtol(val, NULL, 0);
		}

		/* PROTO_OPT */
		/* algo ALGO ALGOKEY */
		if (cur_idx >= argc)
			goto fin;
		ret = try_parse_algo(argc, argv, cur_idx, &handle.st);
		if (ret < 0)
			goto usage_exit;
		cur_idx = ret;		
#endif

		/* coa COADDR */
		if (cur_idx >= argc)
			goto fin;
		ret = try_parse_coa(argc, argv, cur_idx, &handle.st);
		if (ret < 0)
			goto usage_exit;
		cur_idx = ret;

		/* SELECTOR */
		if (cur_idx >= argc)
			goto fin;
		arg = argv[cur_idx ++];
		if (strcmp(arg, "sel") != 0) {
			/* back track */
			cur_idx --;
		} else {
			ret = parse_selector(argc, argv, cur_idx, &handle.sel);
			if (ret < 0)
				goto usage_exit;
			cur_idx = ret;
		}
	}
 fin:
	if (cur_idx < argc) {
		printf("too many arguments=%d (parsed=%d)\n", argc, cur_idx);
		goto usage_exit;
	} else if (cur_idx > argc) {
		printf("internal error: arguments=%d, parsed=%d\n", argc, cur_idx);
		goto usage_exit;
	}

 talk:
	handle.error_handler = error_handler;
	if (is_flush)
		handle.probe_state_handler = del_state_handler;
	else
		handle.probe_state_handler = dump_state_handler;

	err = xfnl_talk(&handle);

	xfnl_close(&handle);

	return err;

 usage_exit:
	usage();
	help_state();
	exit((err > 0) ? err : -err);
}

int main(int argc, char **argv)
{
	int opt;
	int cur_idx;
	char *object = NULL;
	int err = EINVAL;

	while (1) {
		/*int this_option_optind = optind ? optind : 1;*/
		int option_index = 0;

		opt = getopt_long(argc, argv, opt_str,
				  opt_table, &option_index);
		if (opt == -1)
			break;

		switch (opt) {
		case 0: /* option has no short one */
			if (strcmp(opt_table[option_index].name, "version") == 0) {
				version();
				exit(0);
			} else if (strcmp(opt_table[option_index].name, "help") == 0) {
				help(object);
				exit(0);
			} else
				goto usage_exit;

		case 'f':
			if (strcmp(optarg, "inet6") == 0)
				preferred_family = AF_INET6;
			else if (strcmp(optarg, "inet") == 0)
				preferred_family = AF_INET;
			break;
		case 'v':
			is_verbose_mode = 1;
			break;
		case '?':
			break;
		default:
			printf("unknown opt=%d\n", opt);
			goto usage_exit;
		}
	}

	cur_idx = optind;

	if (cur_idx >= argc)
		goto usage_exit;

	/* OBJECT */
	object = argv[cur_idx ++];

	if (strcmp(object, "state") == 0 || strcmp(object, "st") == 0)
		err = state_main(argc, argv, cur_idx);
	else if (strcmp(object, "policy") == 0 || strcmp(object, "pol") == 0)
		err = policy_main(argc, argv, cur_idx);
	else {
		if (strcmp(object, "help") != 0)
			printf("invalid argument: %s: uknown object.\n", object);
		goto usage_exit;
	}

	if (err != 0)
		return (err > 0) ? err : -err;
	else
		return error_code;
		
 usage_exit:
	usage();
	help(object);
	exit((err > 0) ? err : -err);
}
