/* $USAGI: libxfnl.c,v 1.5 2004/04/24 15:10:05 nakam Exp $ */

/*
 * Copyright (C)2004 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * Authors:
 *	Masahide NAKAMURA @USAGI
 */
#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>
#include <linux/xfrm.h>

#include "libxfnl.h"

#define	BUFLEN	8192
#define NELEMSOF(array) (sizeof(array) / sizeof(array[0]))

#ifdef DEBUG
#define DPRINTF(format, arg...) \
	__dprintf(stderr,"<D>",__FILE__,__LINE__,__FUNCTION__,format,##arg)
#define EPRINTF(format, arg...) \
	__dprintf(stderr,"<E>",__FILE__,__LINE__,__FUNCTION__,format,##arg)
#define PERROR(s) \
	__derrno(stderr, "<E>",__FILE__,__LINE__,__FUNCTION__,s)
#else /* DEBUG */
#define DPRINTF(format, arg...)   do { ; } while(0)
#define EPRINTF(format, arg...) \
	fprintf(stderr, "(%s)" format, __FUNCTION__, ##arg)
#define PERROR(s) perror(s)
#endif /* DEBUG */

#ifdef DEBUG
#include <stdarg.h>

static inline void __dprintf(FILE *stream, const char *prefix,
			     const char *file, int line, const char *func,
			     const char *format, ...)
{
	char buf[2048];
	va_list ap;

	va_start(ap, format);
	vsprintf(buf, format, ap);
	va_end(ap);

	fprintf(stream, "%s%s:%d(%s)%s", prefix, file, line, func, buf);
}

static inline void __derrno(FILE *stream, const char *prefix,
			    const char *file, int line, const char *func,
			    const char *s)
{
	__dprintf(stream, prefix, file, line, func,
		  "%s:%s(%d)\n", s, strerror(errno), errno);
}
#endif

static int xfrm_addr_any(xfrm_address_t *a, int family)
{
	switch (family) {
	case AF_INET:
		return (a->a4 == 0); 
	case AF_INET6:
		return ((a->a6[0] | a->a6[1] | a->a6[2] | a->a6[3] ) == 0); 
	case AF_UNSPEC:
		return 1;
	default:
		EPRINTF("unsupported family: %d\n", family);
		break;
	}
	return -1;
}

static int set_selector(struct xfrm_selector *sel, const struct xfnl_handle *xh)
{
	memset(sel, 0, sizeof(*sel));
	memcpy(&sel->daddr, &xh->sel.daddr, sizeof(sel->daddr));
	memcpy(&sel->saddr, &xh->sel.saddr, sizeof(sel->saddr));
	if (!xfrm_addr_any(&sel->daddr, xh->family))
		sel->prefixlen_d = 128;
	else
		sel->prefixlen_d = xh->sel.prefixlen_d;
	if (!xfrm_addr_any(&sel->saddr, xh->family))
		sel->prefixlen_s = 128;
	else
		sel->prefixlen_s = xh->sel.prefixlen_s;
	sel->mh_type = xh->sel.mh_type;
	sel->dport = xh->sel.dport;
	sel->sport = xh->sel.sport;
	if (sel->dport)
		sel->dport_mask = ~0;
	if (sel->sport)
		sel->sport_mask = ~0;
	sel->proto = xh->sel.proto;
	sel->ifindex = xh->sel.ifindex;
	sel->user = getuid();
	sel->family = xh->family;

	return 0;
}

static int set_lifetime(struct xfrm_lifetime_cfg *lft)
{
	lft->soft_byte_limit = XFRM_INF;
	lft->hard_byte_limit = XFRM_INF;
	lft->soft_packet_limit = XFRM_INF;
	lft->hard_packet_limit = XFRM_INF;
	lft->soft_add_expires_seconds = 0;
	lft->hard_add_expires_seconds = 0;
	lft->soft_use_expires_seconds = 0;
	lft->hard_use_expires_seconds = 0;

	return 0;
}


static int set_template(struct xfrm_user_tmpl *tmpl, __u32 spi, __u8 proto,
			__u8 mode, const xfrm_address_t *id_daddr,
			const xfrm_address_t *id_saddr)
{
	memset(tmpl, 0, sizeof(*tmpl));

	/* id */	
	if (id_daddr)
		memcpy(&tmpl->id.daddr, id_daddr, sizeof(tmpl->id.daddr));
	tmpl->id.spi = spi;
	tmpl->id.proto = proto;

	if (id_saddr)
		memcpy(&tmpl->saddr, id_saddr, sizeof(tmpl->saddr));

	tmpl->reqid = 0;
	tmpl->mode = mode;
	tmpl->share = 0;
	/* XXX: this is maybe a flag ignored if state is not found. */
	/* XXX: see xfrm_tmpl_resolve() */
	tmpl->optional = 0;
	tmpl->aalgos = 0;
	tmpl->ealgos = 0;
	tmpl->calgos = 0;

	return 0;
}

static int make_msg_policy_info(const struct xfnl_handle *xh, void *buf, unsigned short int *len)
{
	struct xfrm_userpolicy_info *pol = (struct xfrm_userpolicy_info *)buf;
	struct xfrm_user_tmpl *tmpl;
	struct rtattr *rta;
	int rta_len;
	int cur_len = 0;
	int i;

	set_selector(&pol->sel, xh);

	/* lifetime_cfg/cur */
	set_lifetime(&pol->lft);
	/* XXX: pol->curlft is needless from userland. */

	pol->priority = xh->pol.priority;
	pol->index = xh->pol.index;
	pol->dir = xh->pol.dir;
	pol->action = xh->pol.action;
	pol->flags = xh->pol.flags;

	pol->share = XFRM_SHARE_ANY; /* XXX: no idea */

	cur_len = NLMSG_ALIGN(sizeof(*pol));

	if (xh->pol.ntmpls > TMPL_MAX) {
		EPRINTF("templates overflow: %d (>%d)\n", xh->pol.ntmpls,
			  TMPL_MAX);
		return -EINVAL;
	} else if (xh->pol.ntmpls > 0) {
		rta = (struct rtattr *)((void *)pol + NLMSG_ALIGN(sizeof(*pol)));
		rta->rta_type = XFRMA_TMPL;
		tmpl = (struct xfrm_user_tmpl *)RTA_DATA(rta);

		rta_len = 0;
		for (i = 0; i < xh->pol.ntmpls; ++i) {
			set_template(tmpl,
				     xh->pol.tmpls[i].spi,
				     xh->pol.tmpls[i].proto,
				     xh->pol.tmpls[i].mode,
				     &xh->pol.tmpls[i].daddr,
				     &xh->pol.tmpls[i].saddr);

			rta_len += sizeof(*tmpl);
			tmpl ++;
		}
		rta->rta_len = RTA_LENGTH(rta_len);
		cur_len += RTA_SPACE(rta_len);
	}

	*len = cur_len;

	return 0;
}

static int make_msg_policy_id(const struct xfnl_handle *xh, void *buf, unsigned short int *len)
{
	struct xfrm_userpolicy_id *pol;

	pol = (struct xfrm_userpolicy_id *)buf;

	set_selector(&pol->sel, xh);

	pol->index = xh->pol.index;
	pol->dir = xh->pol.dir;

	*len = NLMSG_ALIGN(sizeof(*pol));

	return 0;
}

static int make_msg_state_info(const struct xfnl_handle *xh, void *buf, unsigned short int *len)
{
	struct xfrm_usersa_info *st = (struct xfrm_usersa_info *)buf;
	struct rtattr *rta;
	int cur_len = 0;
	int i;

	memset(st, 0, sizeof(*st));

	set_selector(&st->sel, xh);

	/* id */
	memcpy(&st->id.daddr, &xh->st.id.daddr, sizeof(st->id.daddr));
	st->id.spi = xh->st.id.spi;
	st->id.proto = xh->st.id.proto;

	memcpy(&st->saddr, &xh->st.id.saddr, sizeof(st->saddr));

	/* lifetime_cfg/cur */
	set_lifetime(&st->lft);
	/* st->curlft is needless from userland. */

	/* stats */
	;

	st->seq = 0;
	st->family = xh->family;
	st->reqid = 0;
	st->mode = xh->st.id.mode;
	st->replay_window = 0;
	st->flags = 0;

	cur_len = NLMSG_ALIGN(sizeof(*st));

	rta = (struct rtattr *)((void *)st + NLMSG_ALIGN(sizeof(*st)));

	for (i = 0; i < xh->st.natts; i++) {
		const struct xfnl_att *att = &xh->st.atts[i];
		void *rta_data = RTA_DATA(rta);
		int rta_len = 0;

		switch (att->type) {
		case XFRMA_ALG_AUTH:
		case XFRMA_ALG_CRYPT:
		case XFRMA_ALG_COMP:
		{
			struct xfrm_algo *algo = (struct xfrm_algo *)rta_data;
			int key_len = att->algo.alg_key_len;

			if (key_len > sizeof(att->algo_key_buf)) {
				EPRINTF("algo-key over flow: type=%d\n", att->type);
				return -ENOMEM;
			}
			key_len /= 8;

			memcpy(algo, &att->algo, sizeof(*algo));
			memcpy(&algo->alg_key, &att->algo.alg_key, key_len);
			rta_len = sizeof(*algo) + key_len;

			break;
		}
#ifdef USE_XFRM_ENHANCEMENT
		case XFRMA_ADDR:
		{
			xfrm_address_t *coa = (xfrm_address_t *)rta_data;

			memcpy(coa, &att->coaddr, sizeof(*coa));
			rta_len = sizeof(*coa);

			break;
		}
#endif /* USE_XFRM_ENHANCEMENT */
		default:
			EPRINTF("unknown type: %d\n", att->type);
			break;
		}

		rta->rta_type = att->type;
		rta->rta_len = RTA_LENGTH(rta_len);
		cur_len += RTA_SPACE(rta_len);

		rta = (struct rtattr *)((void *)rta + RTA_SPACE(rta_len));
	}

	*len = cur_len;

	return 0;
}

static int make_msg_state_id(const struct xfnl_handle *xh, void *buf, unsigned short int *len)
{
	struct xfrm_usersa_id *sa;

	sa = (struct xfrm_usersa_id *)buf;
	memset(sa, 0, sizeof(*sa));

	memcpy(&sa->daddr, &xh->st.id.daddr, sizeof(sa->daddr));
#ifdef USE_XFRM_ENHANCEMENT
	memcpy(&sa->saddr, &xh->st.id.saddr, sizeof(sa->saddr));
#endif

	sa->spi = xh->st.id.spi;
	sa->family = xh->family;
	sa->proto = xh->st.id.proto;

	*len = NLMSG_ALIGN(sizeof(*sa));

	return 0;
}

static int make_msg_one(const struct xfnl_handle *xh, void *buf, unsigned short int *len)
{
	struct nlmsghdr	*hdr = (struct nlmsghdr *)buf;
	unsigned short int data_len = 0;
	int ret;

	switch (xh->type) {
	case XFRM_MSG_NEWPOLICY:
	case XFRM_MSG_UPDPOLICY:
		ret = make_msg_policy_info(xh, NLMSG_DATA(hdr), &data_len);
		break;
	case XFRM_MSG_DELPOLICY:
	case XFRM_MSG_GETPOLICY:
		ret = make_msg_policy_id(xh, NLMSG_DATA(hdr), &data_len);
		break;

	case XFRM_MSG_NEWSA:
	case XFRM_MSG_UPDSA:
		ret = make_msg_state_info(xh, NLMSG_DATA(hdr), &data_len);
		break;

	case XFRM_MSG_DELSA:
	case XFRM_MSG_GETSA:
		ret = make_msg_state_id(xh, NLMSG_DATA(hdr), &data_len);
		break;

	default:
		ret = -EINVAL;
		EPRINTF("unrecognized type: %d\n", xh->type);
		break;
	}

	if (!ret) {
		assert(data_len > 0);

		hdr->nlmsg_len = NLMSG_LENGTH(data_len);
		hdr->nlmsg_type = xh->type; /* XFRM_MSG_... */
		hdr->nlmsg_flags =  NLM_F_REQUEST | xh->flag;

		*len = NLMSG_SPACE(data_len);
	}
	return ret;
}

int xfnl_make_msg(const struct xfnl_handle *xh, void *buf, unsigned short int *len)
{
	return make_msg_one(xh, buf, len);
}

static int probe_msg_policy(const struct xfnl_handle *xh, void *buf, int len)
{
	int rest_len;
	struct xfrm_userpolicy_info *pol;
	struct rtattr *rta;
	struct xfrm_user_tmpl tmpls[TMPL_MAX*2]; /* to read more than writable */
	struct xfrm_user_tmpl *tmpl;
	int num_tmpls = 0;
	int i;
	int ret = 0;

	memset(&tmpls, 0, sizeof(tmpls));

	if (len < sizeof(struct xfrm_userpolicy_info)) {
		EPRINTF("nlmsg data is too short:%d\n", len);
		ret = -EINVAL;
		goto error;
	}
	if (len <= NLMSG_ALIGN(sizeof(struct xfrm_userpolicy_info)))
		goto fin;

	pol = (struct xfrm_userpolicy_info *)buf;
	rest_len = len - NLMSG_ALIGN(sizeof(*pol));

	if (rest_len < sizeof(struct rtattr))
		goto fin;
	rta = (struct rtattr *)((void *)pol + NLMSG_ALIGN(sizeof(*pol)));

	if (!RTA_OK(rta, rest_len))
		goto fin;
	if (rta->rta_type != XFRMA_TMPL) {
		EPRINTF("rta type is not tmpl:%d\n", rta->rta_type);
		goto fin;
	}
	rest_len = (rest_len > rta->rta_len) ? rta->rta_len : rest_len;

	memset(&tmpls, 0, sizeof(tmpls));
	i = 0;
	for (tmpl = (struct xfrm_user_tmpl *)RTA_DATA(rta);
	     tmpl; tmpl ++) {
		if (rest_len < sizeof(struct xfrm_user_tmpl))
			break;
		if (i >= NELEMSOF(tmpls)) {
			EPRINTF("too many templates\n");
			break;
		}

		memcpy(&tmpls[i], tmpl, sizeof(tmpls[i]));

		rest_len -= sizeof(*tmpl);
		i ++;
	}
	num_tmpls = i;

 fin:
	if (rest_len > 0)
		DPRINTF("message remains %u\n", rest_len);

	if (xh->probe_policy_handler) {
		const struct xfrm_user_tmpl *tp = (num_tmpls > 0) ? tmpls : NULL;
		ret = xh->probe_policy_handler(xh, pol, tp, num_tmpls);
	}

	return ret;
 error:
	return ret;
}

static int probe_msg_state(const struct xfnl_handle *xh, void *buf, int len)
{
	struct xfrm_usersa_info *st = (struct xfrm_usersa_info *)buf;
	struct rtattr *rta;
	void *rta_data;
	int rest_len;
	struct xfnl_att atts[ATT_MAX*2]; /* to read more than writable */
	struct xfnl_att *att;
	int num_atts = 0;
	int i;
	int ret = 0;

	memset(atts, 0, sizeof(atts));

	if (len < sizeof(struct xfrm_usersa_info)) {
		EPRINTF("nlmsg data is too short:%u\n", len);
		return -EINVAL;
	}

	st = (struct xfrm_usersa_info *)buf;

	rest_len = len - NLMSG_ALIGN(sizeof(*st));

	if (rest_len < sizeof(struct rtattr))
		goto fin;
	rta = (struct rtattr *)((void *)st + NLMSG_ALIGN(sizeof(*st)));

	i = 0;
	while (RTA_OK(rta, rest_len)) {
		rta_data = RTA_DATA(rta);

		if (i >= NELEMSOF(atts)) {
			EPRINTF("too many attributes\n");
			goto fin;
		}
		att = &atts[i];

		att->type = rta->rta_type;
		switch (att->type) {
		case XFRMA_ALG_AUTH:
		case XFRMA_ALG_CRYPT:
		case XFRMA_ALG_COMP:
		{
			int rta_pl = RTA_PAYLOAD(rta);
			struct xfrm_algo *algo;
			int len = sizeof(*algo);
			int key_len;
			if (len > rta_pl) {
				EPRINTF("algo too short: type=%d\n", att->type);
				break;
			}

			algo = (struct xfrm_algo *)rta_data;
			key_len = algo->alg_key_len;
			if (key_len > sizeof(att->algo_key_buf)) {
				EPRINTF("algo-key over flow: type=%d\n", att->type);
				break;
			}
			key_len /= 8;

			if (len + key_len > rta_pl) {
				EPRINTF("algo too short with key: type=%d\n", att->type);
				break;
			}
			memcpy(&att->algo, algo, sizeof(att->algo));
			memcpy(att->algo.alg_key, algo->alg_key, key_len);
			break;
		}

#ifdef USE_XFRM_ENHANCEMENT
		case XFRMA_ADDR:
			if (RTA_PAYLOAD(rta) < sizeof(att->coaddr)) {
				EPRINTF("too short: %d\n", att->type);
				break;
			}

			memcpy(&att->coaddr, rta_data, sizeof(att->coaddr));
			break;
#endif /* USE_XFRM_ENHANCEMENT */

		default:
			DPRINTF("ignored type: %d\n", att->type);
			break;
		}
		i++;

		rta = RTA_NEXT(rta, rest_len);
	}
	num_atts = i;

 fin:
	if (rest_len > 0)
		DPRINTF("message remains %u\n", rest_len);
	if (xh->probe_state_handler) {
		const struct xfnl_att *ap = (num_atts > 0) ? atts : NULL;
		ret = xh->probe_state_handler(xh, st, ap, num_atts);
	}
	return ret;
}

static int probe_msg_one(const struct xfnl_handle *xh, void *buf, int len)
{
	int ret = 0;

	switch (xh->type) {
	case XFRM_MSG_NEWPOLICY:
	case XFRM_MSG_DELPOLICY:
	case XFRM_MSG_GETPOLICY:
	case XFRM_MSG_UPDPOLICY:
		ret = probe_msg_policy(xh, buf, len);
		break;
	case XFRM_MSG_NEWSA:
	case XFRM_MSG_DELSA:
	case XFRM_MSG_GETSA:
	case XFRM_MSG_UPDSA:
		ret = probe_msg_state(xh, buf, len);
		break;
	default:
		break;
	}

	return ret;
}

/*
 * buf is pointed to nlmsghdr, lenp is length of buf including current
 * nlmsghdr's length.
 */
int xfnl_probe_msg(const struct xfnl_handle *xh, void *buf, int *lenp)
{
	struct nlmsghdr	*hdr = (struct nlmsghdr *)buf;
	void *nextbuf;
	int len = *lenp;
	int ret;

	if (len == 0) {
		DPRINTF("no more buffer(len=0)\n");
		goto done;
	}
	if (len < sizeof(*hdr)) {
		EPRINTF("nlmsg cannot be parsed: len=%u\n", len);
		return -EINVAL;
	}
	if (!NLMSG_OK(hdr, len)) {
		EPRINTF("nlmsg is not ok: len=%d, type=%u\n", len, hdr->nlmsg_type);

		 /* XXX: type == 0 is able to be occured... */
		if (hdr->nlmsg_type == NLMSG_DONE || !hdr->nlmsg_type) {
			EPRINTF("ignored error. this is ok.\n");
			goto done;
		}

		return -EINVAL;
	}

#if 0
	DPRINTF("nlmsg len=%u\n", len);
#endif

	switch (hdr->nlmsg_type) {

	case NLMSG_ERROR:	/* Error		*/
	{
		struct nlmsgerr *err_hdr = (struct nlmsgerr *)NLMSG_DATA(hdr);
		ret = -err_hdr->error;
		if (ret != 0)
			EPRINTF("nlmsg code: %d (%s)\n", ret, strerror(ret));
		else
			DPRINTF("nlmsg code: %d (%s)\n", ret, strerror(ret));

		if (xh->error_handler)
			xh->error_handler(xh, &err_hdr->msg, ret, len);

		*lenp -= len;
		break;
	}
	case NLMSG_OVERRUN:	/* Data lost		*/
		ret = 0;
		EPRINTF("nlmsg data lost; type=OVERRUN\n");
		break;

	case NLMSG_NOOP:	/* Nothing.		*/
		ret = 0;
		EPRINTF("nlmsg data is nothing; type=NOOP\n");
		break;

	case NLMSG_DONE:	/* End of a dump	*/
		DPRINTF("nlmsg data is done; type=DONE\n");

		if (hdr->nlmsg_len == sizeof(struct nlmsgerr)) {
			struct nlmsgerr *err_hdr;
			err_hdr = (struct nlmsgerr *)NLMSG_DATA(hdr);
			*lenp -= sizeof(struct nlmsgerr);
			ret = -err_hdr->error;
			DPRINTF("nlmsg code: %d (%s)\n", ret, strerror(ret));

			break;
		}

		goto done;
	default:
		DPRINTF("nlmsg data type=%d\n", hdr->nlmsg_type);

		ret = probe_msg_one(xh, NLMSG_DATA(hdr), hdr->nlmsg_len);
		if (ret != 0)
			break;

		/* NLMSG_NEXT split length of next nlmsghdr. */
		nextbuf = NLMSG_NEXT(hdr, len);

		*lenp = len;
		/* recursive call */
		ret = xfnl_probe_msg(xh, nextbuf, lenp);

		break;
	}

	DPRINTF("returns=%d; message remains len=%lld\n", ret, (long long int)*lenp);
	return ret;

 done:
	DPRINTF("returns=0; message remains len=%lld\n", (long long int)*lenp);
	return 0;
}

int xfnl_verify(const struct xfnl_handle *xh)
{
	if (!xh) {
		EPRINTF("NULL\n");
		return -EINVAL;
	}

	if (xh->sock < 0) {
		EPRINTF("socket is not available\n");
		return -EINVAL;
	}
	return 0;
}

int xfnl_talk(const struct xfnl_handle *xh)
{
	char sendbuf[BUFLEN];
	char recvbuf[BUFLEN];
	unsigned short int data_len = 0;
	int err = 0;
	int len = 0;

	err = xfnl_verify(xh);
	if (err)
		goto fin;

	memset(sendbuf, 0, sizeof(sendbuf));
	memset(recvbuf, 0, sizeof(recvbuf));

	DPRINTF("type = %s (%d)\n",
		  ((xh->type == XFRM_MSG_NEWPOLICY) ? "new policy" :
		   (xh->type == XFRM_MSG_DELPOLICY) ? "del policy" :
		   (xh->type == XFRM_MSG_UPDPOLICY) ? "upd policy" :
		   (xh->type == XFRM_MSG_NEWSA) ? "new sa" :
		   (xh->type == XFRM_MSG_DELSA) ? "del sa" :
		   (xh->type == XFRM_MSG_UPDSA) ? "upd sa" :
		   (xh->type == XFRM_MSG_MIP6NOTIFY) ? "mip6notify" :
		   "unknown"),
		  xh->type);

	err = xfnl_make_msg(xh, (void *)sendbuf, &data_len);
	if (err)
		goto fin;

	assert(data_len > 0);
	assert(data_len < sizeof(sendbuf));

	err = nl_talk_sr(xh->sock, sendbuf, data_len, recvbuf, sizeof(recvbuf), &len);
	if (err)
		goto fin;

	assert(len <=  sizeof(recvbuf));

	err = xfnl_probe_msg(xh, recvbuf, &len);
	if (err)
		goto fin;

 fin:
	return err;
}

void xfnl_init(struct xfnl_handle *xh, int sock)
{
	xh->sock = sock;
	xh->flag |= NLM_F_ACK;
}

static int sock_open(int *sockp)
{
	int sock;
	struct sockaddr_nl nladdr;
	int err = 0;

	sock = socket(PF_NETLINK, SOCK_RAW, NETLINK_XFRM);
	if (sock < 0) {
		perror("socket");
		return -errno;
	}

	memset(&nladdr, 0, sizeof(nladdr));
	nladdr.nl_family = AF_NETLINK;
	err = bind(sock, (struct sockaddr *)&nladdr, sizeof(nladdr));
	if (err != 0) {
		perror("bind");
		err = -errno;
		close(sock);
		return err;
	}

	*sockp = sock;
	return 0;
}

int xfnl_open(struct xfnl_handle *xh)
{
	int sock = -1;
	int err;

	memset(xh, 0, sizeof(xh));

	err = sock_open(&sock);
	if (err < 0)
		return err;

	xfnl_init(xh, sock);

	return 0;
}

void xfnl_close(struct xfnl_handle *xh)
{
	if (!xh)
		return;
	if (xh->sock >= 0)
		close(xh->sock);
}

/*
 * send and receive message with netlink socket.
 * @sock: netlink socket.
 * @nlmsg: message buffer to send.
 * @nlmsg_len: length of message to send.
 * @rnlmsg: message buffer to receive. (if NULL, recvmsg() is not performed.)
 * @rnlmsg_max: max size of message buffer to receive.
 * @rnlmsg_len: length of received message. (if NULL, not to be specified.)
 */
int nl_talk_sr(int sock, void *nlmsg, int nlmsg_len,
	       void *rnlmsg, int rnlmsg_max, int *rnlmsg_len)
{
	struct sockaddr_nl nladdr;
	struct msghdr msg;
	struct iovec iov;
	int err;
	int ret;

	memset(&nladdr, 0, sizeof(nladdr));
	nladdr.nl_family = AF_NETLINK;

	memset(&msg, 0, sizeof(msg));
	msg.msg_name = &nladdr;
	msg.msg_namelen = sizeof(nladdr);
	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;
	iov.iov_base = nlmsg;
	iov.iov_len = nlmsg_len;

	DPRINTF("sending...\n");

	ret = sendmsg(sock, &msg, 0);
	if (ret < 0) {
		PERROR("sendmsg");
		err = -errno;
		goto error;
	}

	DPRINTF("sending...done.\n");

	if (rnlmsg) {
		struct msghdr rmsg;
		struct iovec riov;

		memset(&rmsg, 0, sizeof(rmsg));

		rmsg.msg_name = &nladdr;
		rmsg.msg_namelen = sizeof(nladdr);

		rmsg.msg_iov = &riov;
		rmsg.msg_iovlen = 1;

		riov.iov_base = rnlmsg;
		riov.iov_len = rnlmsg_max;

		DPRINTF("receiving...\n");

		ret = recvmsg(sock, &rmsg, 0);
		if (ret < 0) {
			PERROR("recvmsg");
			err = -errno;
			goto error;
		}
		if (rnlmsg_len)
			*rnlmsg_len = ret;
		DPRINTF("receiving...done.\n");
	}
	return 0;
 error:
	return err;
}

int nl_talk(void *nlmsg, int nlmsg_len,
	    void *rnlmsg, int rnlmsg_max, int *rnlmsg_len)
{
	int sock = -1;
	int err;

	err = sock_open(&sock);
	if (err < 0)
		goto error;

	err = nl_talk_sr(sock, nlmsg, nlmsg_len, rnlmsg,
			 rnlmsg_max, rnlmsg_len);
	if (err != 0)
		goto error;

	close(sock);
	return 0;
 error:
	if (sock >= 0)
		close(sock);
	return err;
}
