/* $USAGI: bc.c,v 1.6 2004/04/24 08:37:12 nakam Exp $ */

/*
 * Copyright (C)2003 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * Authors:
 *	Masahide NAKAMURA @USAGI
 */

/*
#include <linux/skbuff.h>
#include <linux/spinlock.h>
#include <net/ipv6.h>
#include <net/addrconf.h>
#include <net/mip6.h>
*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <assert.h>

#include <mip6.h>
#include "mip6d.h"
#include "bc.h"


static struct mip6_bc mip6_bc_ctl = {
	.lock = RW_LOCK_UNLOCKED,
	.entry_count = 0,
};

static unsigned char mip6_bc_hash(const struct in6_addr *addr)
{
	return (ipv6_addr_hash(addr) % MIP6_BC_HASH_SIZE);
}

/*
 * hoa is search key for binding cache.
 * if bcep is not NULL, it is stored suitable area for search key.
 * returns: <0 error
 *           0 not found
 *          >0 found(bcepp is pointed target area)
 */
int mip6_bc_lookup(struct in6_addr *hoa, struct mip6_bc_entry **bcep)
{
	struct mip6_bc_entry *bce = NULL;
	int table_id;

	assert(hoa != NULL);
#ifdef DEBUG
	__dprintf("hoa: %x:%x:%x:%x:%x:%x:%x:%x\n", NIP6(*hoa));
#endif

	table_id = mip6_bc_hash(hoa);

	read_lock(&mip6_bc_ctl.lock);

	list_for_each_entry(bce, &mip6_bc_ctl.entry_table[table_id], list) {
		int cmp = ipv6_addr_cmp(hoa, &bce->hoa);
		if (cmp == 0)
			goto found;

	}

	read_unlock(&mip6_bc_ctl.lock);

	/* not found */
	return 0;

 found:
	read_unlock(&mip6_bc_ctl.lock);

	if (bcep != NULL)
		*bcep = bce;

#ifdef DEBUG
	__dprintf("found\n");
#endif
	return 1;
}

static int mip6_bc_set_one(struct mip6_bc_entry *bce,
			   struct in6_addr *coa, __u16 lifetime, __u8 flags, __u16 seq)
{
	if (coa == NULL) {
		__eprintf("%s: coa is NULL\n", __FUNCTION__);
		return -EINVAL;
	}
	if (lifetime == 0) {
		__eprintf("%s: lifetime is 0\n", __FUNCTION__);
		return -EINVAL;
	}

	ipv6_addr_copy(&(bce->coa), coa);
	bce->lifetime = lifetime;
	bce->flags = flags;
	bce->seq = seq;

	return 0;
}

int mip6_bc_set(struct mip6_bc_entry *bce,
		struct in6_addr *coa,__u16 lifetime, __u8 flags, __u16 seq)
{
	int ret;

	write_lock(&mip6_bc_ctl.lock);

	ret = mip6_bc_set_one(bce, coa, lifetime, flags, seq);

#ifdef CONFIG_IPV6_MIP6_DEBUG
	mip6_bc_dump();
#endif
	write_unlock(&mip6_bc_ctl.lock);

	return ret;
}

int mip6_bc_add(struct in6_addr *hoa, struct in6_addr *coa,
		__u16 lifetime, __u8 flags, __u16 seq, struct mip6_bc_entry **bcep)
{
	struct mip6_bc_entry *bce;
	unsigned int h = mip6_bc_hash(hoa);
	int ret;

	bce = (struct mip6_bc_entry *)malloc(sizeof(struct mip6_bc_entry));
	if (!bce) {
		__perror("malloc");
		return -ENOMEM;
	}

	ipv6_addr_copy(&(bce->hoa), hoa);

	ret = mip6_bc_set_one(bce, coa, lifetime, flags, seq);
	if (ret < 0) {
		free(bce);
		return ret;
	}

	write_lock(&mip6_bc_ctl.lock);

	list_add(&bce->list, &mip6_bc_ctl.entry_table[h]);
	mip6_bc_ctl.entry_count ++;

	if (bcep)
		*bcep = bce;

#ifdef CONFIG_IPV6_MIP6_DEBUG
	mip6_bc_dump();
#endif
	write_unlock(&mip6_bc_ctl.lock);

	return 0;
}

int mip6_bc_del(struct mip6_bc_entry *bce)
{
	write_lock(&mip6_bc_ctl.lock);

	list_del(&bce->list);
	free(bce);
	mip6_bc_ctl.entry_count --;
#ifdef CONFIG_IPV6_MIP6_DEBUG
	mip6_bc_dump();
#endif
	write_unlock(&mip6_bc_ctl.lock);

	return 0;
}

#ifdef CONFIG_IPV6_MIP6_DEBUG
static void mip6_bc_dump_one(struct mip6_bc_entry *bce)
{
	if (!bce)
		return;

	/*printf("%*c", 3*depth, ' ');*/
	printf("  ");
	printf("%x:%x:%x:%x:%x:%x:%x:%x\n", NIP6(bce->hoa));
	printf("  coa=%x:%x:%x:%x:%x:%x:%x:%x\n", NIP6(bce->coa));
	printf("  ");
	printf("  lifetime=%d", bce->lifetime);
	printf("  seq=%d", bce->seq);
	printf("  flags[A|H|L|K]=[%d|%d|%d|%d]",
	       !!(bce->flags & MIP6_BU_F_ACK),
	       !!(bce->flags & MIP6_BU_F_HR),
	       !!(bce->flags & MIP6_BU_F_LL),
	       !!(bce->flags & MIP6_BU_F_KM));

	printf("\n");
}

void mip6_bc_dump()
{
	int i;

	read_lock(&mip6_bc_ctl.lock);

	printf("====== Binding Cache ======\n");

	for (i = 0; i < MIP6_BC_HASH_SIZE; i++) {
		struct mip6_bc_entry *bce;

		printf("bc hash=%d\n", i);

		list_for_each_entry(bce, &mip6_bc_ctl.entry_table[i], list) {
			mip6_bc_dump_one(bce);
		}
	}

	printf("===========================\n");

	read_unlock(&mip6_bc_ctl.lock);
}
#endif

static void mip6_bc_free(struct mip6_bc_entry *bce)
{
	memset(bce, 0, sizeof(struct mip6_bc_entry));
	free(bce);
}

/* XXX: quick hacking... you must try to call this function every second. */
void mip6_bc_update()
{
	int i;

	write_lock(&mip6_bc_ctl.lock);

	for (i = 0; i < MIP6_BC_HASH_SIZE; i++) {
		struct list_head *bc_list = &mip6_bc_ctl.entry_table[i];
		struct list_head *entry, *tmp;

		if (!bc_list)
			continue;

		list_for_each_safe(entry, tmp, bc_list) {
			struct mip6_bc_entry *bce = list_entry(entry, struct mip6_bc_entry, list);
			bce->lifetime --;
			if (bce->lifetime > 0)
				continue;

#ifdef DEBUG
			__dprintf("died bc: %x:%x:%x:%x:%x:%x:%x:%x\n", NIP6(bce->hoa));
#endif
			/* delete if lifetime is zero */
			list_del(&bce->list);
			free(bce);
		}
	}

	write_unlock(&mip6_bc_ctl.lock);
}

void mip6_bc_clear()
{
	int i;

	write_lock(&mip6_bc_ctl.lock);

	for (i = 0; i < MIP6_BC_HASH_SIZE; i++) {
		struct list_head *bc_list = &mip6_bc_ctl.entry_table[i];
		struct list_head *entry, *tmp;

		if (!bc_list)
			continue;

		list_for_each_safe(entry, tmp, bc_list) {
			struct mip6_bc_entry *bce = list_entry(entry, struct mip6_bc_entry, list);
			list_del(entry);
			mip6_bc_free(bce);
		}

		INIT_LIST_HEAD(&mip6_bc_ctl.entry_table[i]);
	}
	mip6_bc_ctl.entry_count = 0;

#ifdef CONFIG_IPV6_MIP6_DEBUG
	mip6_bc_dump();
#endif
	write_unlock(&mip6_bc_ctl.lock);
}

int mip6_bc_may_overflow()
{
	int ov;

	read_lock(&mip6_bc_ctl.lock);
	ov = (mip6_bc_ctl.entry_count > MIP6_BC_ENTRY_MAX - 1);
	read_unlock(&mip6_bc_ctl.lock);

	return ov;
}

void mip6_bc_fini()
{
	mip6_bc_clear();
}

int mip6_bc_init()
{
	int i;

	/*write_lock(&mip6_bc_ctl.lock);*/

	for (i = 0; i < MIP6_BC_HASH_SIZE; i++)
		INIT_LIST_HEAD(&mip6_bc_ctl.entry_table[i]);

	mip6_bc_ctl.entry_count = 0;

	/*write_unlock(&mip6_bc_ctl.lock);*/

	return 0;
}
