/* $USAGI: util.c,v 1.8 2003/11/13 17:18:21 takamiya Exp $ */

/*
 * Copyright (C)2003 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <errno.h>
#include <ifaddrs.h>

#include <stdio.h>
#include <strings.h>
#include <netinet/in.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>


#include <mip6.h>
#include <mip6log.h>
#include "util.h"

const char *strmh(__u8 type)
{
	static char str[32];

	switch (type) {
	case MIP6_MH_BRR:
		/* Binding Refresh Request */
		strcpy(str, "brr");
		break;
	case MIP6_MH_HOTI:
		/* Home Test Init */
		strcpy(str, "hoti");
		break;
	case MIP6_MH_COTI:
		/* Care-of Test Init */
		strcpy(str, "coti");
		break;
	case MIP6_MH_HOT:
		/* Home Test */
		strcpy(str, "ht");
		break;
	case MIP6_MH_COT:
		/* Care-of Test */
		strcpy(str, "cot");
		break;
	case MIP6_MH_BU:
		/* Binding Update */
		strcpy(str, "bu");
		break;
	case MIP6_MH_BA:
		/* Binding Acknowledgement */
		strcpy(str, "ba");
		break;
	case MIP6_MH_BE:
		/* Binding Error */
		strcpy(str, "be");
		break;
	default:
		sprintf(str, "(%d)", type);
		break;
	}

	return str;
}

/* XXX: currently mip6 related types only. */
const char *stricmp6(__u8 type)
{
	static char str[32];

	switch(type) {
	case ICMPV6_DHAAD_REQUEST:
		strcpy(str, "dhaad request");
		break;
	case ICMPV6_DHAAD_REPLY:
		strcpy(str, "dhaad reply");
		break;
	case ICMPV6_MOBILE_PREFIX_SOLICIT:
		strcpy(str, "mps");
		break;
	case ICMPV6_MOBILE_PREFIX_ADVERT:
		strcpy(str, "mpa");
		break;
	default:
		sprintf(str, "(%d)", type);
		break;
	}

	return str;
}

const char *in6_straddr(const struct in6_addr *addr)
{
	static char str[INET6_ADDRSTRLEN];
	struct sockaddr_in6 sa;
	int ret;

	if (memcmp(addr, &in6addr_any, sizeof(*addr)) == 0) {
		strcpy(str, "(any)");
		return str;
	}

	memset(&sa, 0, sizeof(sa));
	sa.sin6_family = PF_INET6;
	memcpy(&sa.sin6_addr, addr, sizeof(sa.sin6_addr));

	ret = getnameinfo((struct sockaddr *)&sa, sizeof(sa), str, sizeof(str), NULL, 0, NI_NUMERICHOST);
	if (ret != 0) {
		__eprintf("getnameinfo():%s\n", gai_strerror(ret));
		return NULL;
	}

	return str;
}

int in6_addr_get(struct in6_addr *addr, const char *str)
{
	struct addrinfo hints, *res = NULL;
	int ret;

	if (strcmp(str, "any") == 0) {
		memcpy(addr, &in6addr_any, sizeof(addr));
		goto fin;
	}

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_INET6;
	hints.ai_socktype = SOCK_RAW;
	hints.ai_flags = AI_PASSIVE;

	ret = getaddrinfo(str, NULL, &hints, &res);
	if (ret != 0) {
		__eprintf("getaddrinfo(\"%s\", ...):%s\n", str, gai_strerror(ret));
		goto fin;
	}
	if (res->ai_addrlen != sizeof(struct sockaddr_in6)) {
		__eprintf("getaddrinfo(\"%s\", ...) contains invalid length:%d\n",
			  str, res->ai_addrlen);
		goto fin;
	}
	memcpy(addr,
	       &((struct sockaddr_in6 *)(res->ai_addr))->sin6_addr,
	       sizeof(struct in6_addr));

 fin:
	if (res)
		freeaddrinfo(res);
	return ret;
}

int in6_lladdr_get(struct in6_addr *addr, const char *ifname)
{
	struct ifaddrs *ifap = NULL, *x = NULL;
	struct sockaddr_in6 *candidate;
	int ret = 0;

	__dprintf("getifaddrs...\n");

	ret = getifaddrs(&ifap);
	if (ret != 0) {
		__perror("getifaddrs");
		goto fin;
	}

	__dprintf("getifaddrs... ok\n");

	for(x = ifap; x; x = x->ifa_next) {
		if (!strcmp(x->ifa_name, ifname)) {
			candidate = (struct sockaddr_in6 *)x->ifa_addr;
			if (candidate->sin6_family != PF_INET6)
				continue;
			if (ntohs(candidate->sin6_addr.s6_addr16[0]) == 0xfe80) {
				memcpy(addr, &candidate->sin6_addr, sizeof(struct in6_addr));
				goto fin;
			}
		}
	}
	/* not found */
	__eprintf("not found: %s\n", ifname);
	ret = -ENODEV;

 fin:
	if (ifap)
		freeifaddrs(ifap);
	return ret;
}

int mip6_addr_unicast_routable(struct in6_addr *a)
{
	/* XXX: not yet. i need function/code like addrconf's... */
	/* XXX: this might be in include/linux/{some}.h ... */
	return 1; /* XXX: currently every address is unicast routable. */
}

/* XXX: this is not mip6 specific... */
int mip6_fill_pad(unsigned char *data, int len, int optlen)
{
	unsigned char *p = data;
	int padding = (16 - (len%8) - (optlen%8))%8;

	if (padding == 1) {
		*p = MIP6_OPT_PAD_1;
	} else {
		*p++ = MIP6_OPT_PAD_N;
		*p++ = padding - 2;
		memset(p, 0, padding - 2);
	}
	return padding;
}
