/* $USAGI: miping6.c,v 1.4 2003/10/22 06:10:48 nakam Exp $ */

/*
 * Copyright (C)2003 USAGI/WIDE Project
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/*
 * Authors:
 *	Noriaki TAKAMIYA @USAGI
 *	Masahide NAKAMURA @USAGI
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#define _GNU_SOURCE
#include <getopt.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>

#include <mip6.h>
#include "miping6.h"

#define CMD_NAME	"miping6"
#define CMD_VERSION	"0.0.1"

#if 0
#define STRCPY_SAFE(dst, src) \
	strncpy(dst, src, sizeof(dst)); \
	dst[sizeof(dst)-1] = '\0';
#endif

static const char opt_str[] = "s:h:r:AHLKa:q:l:c:";

static const struct option opt_table[] = {
	{"source", 0, 0, 's'},

	{"home-addr-opt", 0, 0, 'h'},
	{"routing-hdr-addr", 0, 0, 'r'},

	{"ack", 0, 0, 'A'},
	{"home-reg", 0, 0, 'H'},
	{"linklocal-compat", 0, 0, 'L'},
	{"key-mng-mob-capab", 0, 0, 'K'},

	{"status", 1, 0, 'a'},
	{"sequence", 1, 0, 'q'},
	{"lifetime", 1, 0, 'l'},
	{"cookie", 1, 0, 'c'},

	{"version", 0, 0, 0},
	{"help", 0, 0, 0},

	{0, 0, 0, 0}
};

static const char *opt_arg_name[] = {
	"addr",

	"addr",
	"addr",

	NULL,
	NULL,
	NULL,
	NULL,

	"status",
	"#",
	"seconds",
	"cookie",

	"auto,yes,no",

	NULL,
	NULL,

	NULL
};

static const char *opt_desc[] = {
	"source address",

	"address in Destination Options Header(Home Address Option)",
	"address in Routing Header type 2",

	"set Acknowledge(A) bit[bu]",
	"set Home Registration(H) bit[bu]",
	"set Link-Local Address Compatibility(L) bit[bu]",
	"set Key Management Mobility Capability(K) bit[bu,ba]",

	"status value[ba,be]",
	"sequence number[bu,ba]",
	"lifetime[bu,ba]",
	"cookie(e.g. 0x1234ff, 3210fe)(NOTICE: corrently support only 32-bit)[hot,cot,hoti,coti]",

	"show version",
	"show this help",

	NULL
};

static const struct mh_type_entry mh_type_table[] = {
	{ MIP6_MH_BRR,	"brr",	"Binding Refresh Request" },
	{ MIP6_MH_HOTI,	"hoti",	"Home Test Init" },
	{ MIP6_MH_COTI,	"coti",	"Care-of Test Init" },
	{ MIP6_MH_HOT,	"hot",	"Home Test" },
	{ MIP6_MH_COT,	"cot",	"Care-of Test" },
	{ MIP6_MH_BU,	"bu",	"Binding Update" },
	{ MIP6_MH_BA,	"ba",	"Binding Acknowledgement" },
	{ MIP6_MH_BE,	"be",	"Binding Error" },
	{ -1,		NULL,	NULL }
};

struct opt_parms opt_parms;

static void version()
{
	printf("%s %s\n", CMD_NAME, CMD_VERSION);
}

static void help()
{
	char opt_buf[1024];
	char lopt_buf[1024];
	int i;

	version();
	printf("%s [options] type destination\n", CMD_NAME);

	for (i = 0; ; i ++) {
		if (opt_table[i].name == 0)
			break;

		if (opt_table[i].val != 0)
			sprintf(opt_buf, "-%c,", (char)opt_table[i].val);
		else
			sprintf(opt_buf, "   ");

		if (opt_table[i].has_arg == 1)
			sprintf(lopt_buf, "--%s=<%s>",
				opt_table[i].name, opt_arg_name[i]);
		else
			sprintf(lopt_buf, "--%s", opt_table[i].name);

		printf("  %s %-*s %s\n",
			opt_buf,
			21, /* format-length for long option name */
			lopt_buf,
			opt_desc[i]);
	}

	printf("\n");
	printf("  type:\n");

	for (i = 0; ; i++) {
		if (mh_type_table[i].name == NULL)
			break;

		printf("    %-*s %s\n",
			8, /* format-length for type name */
			mh_type_table[i].name,
			mh_type_table[i].desc);
	}

	return;

}
static void usage()
{
	printf("Try `%s --help` for more information.\n", CMD_NAME);
}

static int in6_addr_get(struct in6_addr *addr, const char *str)
{
	struct addrinfo hints, *res = NULL;
	int ret;

	if (strcmp(str, "any") == 0) {
		memcpy(addr, &in6addr_any, sizeof(addr));
		goto fin;
	}

	memset(&hints, 0, sizeof(hints));
	hints.ai_family = PF_INET6;
	hints.ai_socktype = SOCK_RAW;
	hints.ai_flags = AI_PASSIVE;

	ret = getaddrinfo(str, NULL, &hints, &res);
	if (ret != 0) {
		printf("getaddrinfo(\"%s\", ...):%s\n", str, gai_strerror(ret));
		goto fin;
	}
	if (res->ai_addrlen != sizeof(struct sockaddr_in6)) {
		printf("getaddrinfo(\"%s\", ...) contains invalid length:%d\n",
		       str, res->ai_addrlen);
		goto fin;
	}
	memcpy(addr,
	       &((struct sockaddr_in6 *)(res->ai_addr))->sin6_addr,
	       sizeof(struct in6_addr));

 fin:
	if (res)
		freeaddrinfo(res);
	return ret;
}


int main(int argc, char **argv)
{
	int opt;
	char *type_str;
	int i;

	memset(&opt_parms, 0, sizeof(opt_parms));

	while (1) {
		/*int this_option_optind = optind ? optind : 1;*/
		int option_index = 0;

		opt = getopt_long(argc, argv, opt_str,
				opt_table, &option_index);
		if (opt == -1)
			break;

		switch (opt) {
		case 0: /* option has no short one */
			if (strcmp(opt_table[option_index].name, "version") == 0) {
				version();
				exit(0);
			} else if (strcmp(opt_table[option_index].name, "help") == 0) {
				help();
				exit(0);
			} else
				goto invalid_arg;

			break;
		case 's':
			if (in6_addr_get(&opt_parms.saddr, optarg) < 0)
				goto invalid_arg;
			break;
		case 'h':
			if (in6_addr_get(&opt_parms.dsto_addr, optarg) < 0)
				goto invalid_arg;
			break;
		case 'r':
			if (1)
				goto not_supported;

			if (in6_addr_get(&opt_parms.rt_addr, optarg) < 0)
				goto invalid_arg;
			break;
		case 'A':
			opt_parms.bu_flags |= MIP6_BU_F_ACK;
			break;
		case 'H':
			opt_parms.bu_flags |= MIP6_BU_F_HR;
			break;
		case 'L':
			opt_parms.bu_flags |= MIP6_BU_F_LL;
			break;
		case 'K':
			opt_parms.bu_flags |= MIP6_BU_F_KM;
			opt_parms.ba_flags |= MIP6_BA_F_KM;
			break;
		case 'a':
			opt_parms.status = (__u8)atoi(optarg);
			break;
		case 'q':
			opt_parms.seq = (__u16)atoi(optarg);
			break;
		case 'l':
			opt_parms.lifetime = (__u16)atoi(optarg);
			break;
		case 'c':
		{
			/*
			 * XXX: As spec, cookie value is 64-bit integer, but this program
			 * supports only 32-bit. Fix me.
			 */
			__u32 val = strtol(optarg, (char **)NULL, 16);
			if (val == LONG_MAX || val == LONG_MIN) {
				perror(optarg);
				goto invalid_arg;
			}
			val = htonl(val);
			memcpy(&opt_parms.cookie, &val, sizeof(opt_parms.cookie));
			break;
		}
		case '?':
			goto invalid_arg;
		default:
			goto invalid_arg;
		}
	}

	/* the rest arguments are type and destination. */
	if (argc - optind != 2) {
		printf ("invalid argments\n");
		goto invalid_arg;
	}

	type_str = argv[optind];
	for (i = 0; ; i++) {
		if (mh_type_table[i].name == NULL)
			break;

		if (strcmp(type_str, mh_type_table[i].name) == 0) {
			opt_parms.mte = mh_type_table[i];
			break;
		}
	}
	if (opt_parms.mh_type_name == NULL) {
		printf ("no such type:%s\n", type_str);
		goto invalid_arg;
	}


	if (in6_addr_get(&opt_parms.daddr, argv[optind+1]) < 0)
		goto invalid_arg;

	return miping6_send();

 invalid_arg:
	usage();
	exit(EINVAL);
 not_supported:
	printf("currently not suppored yet.\n");
	exit(0);
}
