//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License Version 2
//  as published by the Free Software Foundation.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifndef BMP_LIBRARY_CLASS_HH
#define BMP_LIBRARY_CLASS_HH

#include <set>

#include <glibmm.h>
#include <glibmm/i18n.h>

#include <boost/optional.hpp>
#include <boost/format.hpp>
#include <boost/tuple/tuple.hpp>
#include <boost/tuple/tuple_comparison.hpp>

#ifndef BMP_PLUGIN_BUILD
#  include <taglib/tag.h>
#endif //!BMP_PLUGIN_BUILD

#include "database.hh"

#include <bmp/library_types.hh>

namespace TagLib
{
    class File;
}

namespace Bmp
{
  // Database flags
  enum DBFlags
  {
    DB_FLAG_NONE        = 0,
    DB_FLAG_USING_HAL   = 1 << 0,
  };

  enum LibraryAddOp
  {
    LIBRARY_ADD_ERROR     = -1,
    LIBRARY_ADD_NOOP      =  0,
    LIBRARY_ADD_IMPORT    =  1,
    LIBRARY_ADD_UPDATE    =  2,
  };

  namespace UiPart { class Albums; class CDDA; } 

  namespace Library
  {

#include "exception.hh"

    EXCEPTION(MetadataWriteError)
    EXCEPTION(NoMetadataError)

#ifdef HAVE_HAL
    EXCEPTION(NoHALInformationError)
#endif //HAVE_HAL

    EXCEPTION(DatabaseError)

    Bmp::DB::VAttributes album_to_attrs (Bmp::Library::Album const& album, bool include_tracks = true);
  
    class Library
    {
      public:

          struct TaglibPlugin
          {
            bool (*set) (std::string const& filename, Bmp::Library::UTrack const& track);
            bool (*get) (TagLib::File * p, Bmp::DB::Row & row, std::string const& name);
          };

          typedef std::map < std::string, TaglibPlugin > TaglibPlugins;

      private:

#include "exception.hh"

          EXCEPTION(NoMetadataTaglibError)
          EXCEPTION(NoMetadataGstError)

          void get_metadata_row      (Glib::ustring const& location,
                                      Bmp::DB::Row&        row) const;

          bool load_taglib_plugin (std::string const& path);

          TaglibPlugins m_plugins;

      public:

          TaglibPlugins const&
          get_plugins () { return m_plugins; }

          typedef sigc::signal <void, int> ModifyStart;
          typedef sigc::signal <void>      ModifyStep;
          typedef sigc::signal <void>      ModifyEnd;
          typedef sigc::signal <void>      SignalProcessing;
          typedef sigc::signal <void, Glib::ustring, Bmp::Library::UTrack > SignalRowUpdated;

          SignalProcessing& signal_processing_start ();
          SignalProcessing& signal_processing_end   ();
          SignalRowUpdated& signal_row_updated      ();

          ModifyStart& signal_modify_start ();
          ModifyStep&  signal_modify_step  ();
          ModifyEnd&   signal_modify_end   ();

          Library ();
          ~Library ();

          // Init stuff for GUI feedback
          bool    open            ();
          bool    check_version   ();
          void    init            ();
          void    recreate_db     ();

          DBFlags get_db_flags    ();

          void query (DB::VAttributes const&  attrv,
                      DB::VRows            &  vector) const;

          void new_approve  ();
          void new_approve  (Album const& a);
          void new_drop_all (); 

          bool new_exist    (); 

          void
          del (DB::VAttributes const& attrv);

          LibraryAddOp
          add (Glib::ustring const& location);

          void mod      (UTrackV & tracks,
                         bool retag     = true,
                         bool db_update = true,
                         bool relocate  = true);

          void get      (Glib::ustring const& location,
                         Track & track) const;

          void project  (std::string const& project,
                         DB::VRows & vector,
                         DB::VAttributes const& attributes) const;
 
          bool  album_exists            (Album const& album) const;
          Album album_refetch           (Album const& album) const;
          bool  album_has_new_items     (Album const& album) const;

          void  all_albums (DB::VRows & vector) const;
          void  new_albums (DB::VRows & vector) const;

#ifdef HAVE_HAL
          bool          volume_exists    (std::string const& volume_udi, std::string const& device_udi) const;
#endif //HAVE_HAL

          GHashTable *  get_hashtable    (Glib::ustring const& location) const;

    protected:

          friend class Bmp::UiPart::Albums;
          friend class Bmp::UiPart::CDDA;

          void metadata_get_taglib   (Glib::ustring const& location, DB::Row & row) const;
          void metadata_set_taglib   (Glib::ustring const& location, UTrack  & track);

          void metadata_get_gst      (Glib::ustring const& location, DB::Row &row) const;
          
    private:

          bool m_has_new_items;
          bool m_new_items_modified;

          SignalProcessing signal_processing_start_;
          SignalProcessing signal_processing_end_;
          SignalRowUpdated signal_row_updated_;

          ModifyStart  signal_modify_start_;
          ModifyStep   signal_modify_step_;
          ModifyEnd    signal_modify_end_;

          DB::ValueMap  map_main;
          DB::ValueMap  map_albums;
          DB::ValueMap  map_va;
          DB::ValueMap  map_volumes;
          DB::Database *db_main;

    }; 

  } // namespace Library 
} // namespace Bmp 

#endif // !BMP_LIBRARY_CLASS_HH
