//  BMPx - The Dumb Music Player
//  Copyright (C) 2005 BMPx development team.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//  --
//
//  The BMPx project hereby grants permission for non-GPL compatible GStreamer
//  plugins to be used and distributed together with GStreamer and BMPx. This
//  permission is above and beyond the permissions granted by the GPL license
//  BMPx is covered by.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif //HAVE_CONFIG_H

#include <boost/format.hpp>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/xmlreader.h>
#include <libxml/xpath.h>
#include <libxml/xpathInternals.h>

#include <glib/gstdio.h>
#include <glibmm.h>
#include <glibmm/i18n.h>
#include <glibmm/markup.h>
#include <iostream>
#include <fstream>

#include "paths.hh"

#include "util.hh"
#include "uri++.hh"
#include "uuid.h"

#include "neon++/session.hh"
#include "neon++/request.hh"

#include "podcast.hh"
#include "podcast-libxml2-sax.hh"
#include "x_vfs.hh"

namespace
{
  const char * A_URL = "url";
  const char * A_LENGTH = "length";
  const char * A_IS_PERMA_LINK = "isPermaLink";
  const char * A_TYPE = "type";

  std::string
  cast_filename (Bmp::PodCastTools::PodCast const& cast)
  {
    std::string filename;
    filename = Glib::build_filename (BMP_PATH_PODCAST_CACHE_DIR, cast.cast_uuid);
    return filename;
  }

  std::string
  cast_image_filename (Bmp::PodCastTools::PodCast const& cast)
  {
    return cast_filename (cast) + ".png";
  }
 
  std::string
  cast_overlay_filename (Bmp::PodCastTools::PodCast const& cast)
  {
    return cast_filename (cast) + ".overlay";
  }

  void
  item_overlay (Bmp::PodCastTools::PodCastItem & item, Bmp::PodCastTools::PodCastOverlayItem const& overlay_item)
  {
    item.listened_to = overlay_item.listened_to;
    item.downloaded = overlay_item.downloaded;
    item.localfilename = overlay_item.localfilename;
  }

}

namespace Bmp
{
  namespace PodCastTools
  {
    void
    overlay_from_item (PodCastItem const& item, PodCastOverlayItem & overlay_item)
    {
      overlay_item.listened_to = item.listened_to;
      overlay_item.downloaded = item.downloaded;
      overlay_item.localfilename = item.localfilename;
    }

    PodCastManager::PodCastManager ()
    {
      using namespace Glib;
      using namespace Gtk;

      std::string filename = Glib::build_filename (BMP_PATH_USER_DIR, "feedlist.opml");
      if (!Glib::file_test (filename, Glib::FILE_TEST_EXISTS))
        return;

      OPMLParser parser (m_casts, this);
      try{
        Markup::ParseContext context (parser);
        std::string data (Glib::file_get_contents (filename)); 
        if (!data.size())
          return;
        context.parse (data);
        context.end_parse ();
        }
      catch (...) {}
  
      if (!parser.check_sanity ())
        throw Bmp::PodCastTools::ParsingError();
    }

    void
    PodCastManager::pod_cast_overlay_save (PodCast const& cast)
    {
      xmlDocPtr doc = xmlNewDoc  (BAD_CAST "1.0");
        
      xmlNodePtr ovrl = xmlNewNode (NULL, BAD_CAST "bmp-cast-overlay");
      xmlSetProp (ovrl, BAD_CAST "version", BAD_CAST "1.0"); 

      xmlDocSetRootElement (doc, ovrl);

      for (VPodCastItems::const_iterator i = cast.items.begin() ; i != cast.items.end() ; ++i)
      {
        xmlNodePtr o = xmlNewNode (NULL, BAD_CAST "cast-item");
        xmlAddChild (ovrl, o);

        xmlSetProp (o, BAD_CAST "item-guid",
                       BAD_CAST  i->guid_value.get().c_str());

        xmlSetProp (o, BAD_CAST "listened-to",
                       BAD_CAST (i->listened_to? "1" : "0"));

        xmlSetProp (o, BAD_CAST "downloaded",
                       BAD_CAST (i->downloaded? "1" : "0"));

        if (i->downloaded)
            xmlSetProp (o, BAD_CAST "localfilename",
                           BAD_CAST (Glib::filename_to_utf8 (i->localfilename).c_str()));
        else
            xmlSetProp (o, BAD_CAST "localfilename",
                           BAD_CAST ""); 
      }

      xmlKeepBlanksDefault (0);
      xmlChar * data;
      int size;
      xmlDocDumpFormatMemoryEnc (doc, &data, &size, "UTF-8", 1);

      std::ofstream o (cast_overlay_filename (cast).c_str());
      o << data;
      o.close ();
      xmlFreeDoc (doc);
      g_free (data);
    }

    void
    PodCastManager::save_overlays ()
    {
      for (MPodCasts::const_iterator i = m_casts.begin() ; i != m_casts.end() ; ++i)
      {
        PodCast const& cast (i->second);
        pod_cast_overlay_save (cast);
      }
    }

    void
    PodCastManager::save_state ()
    {
      save_opml ();
      save_overlays ();
    }

    void
    PodCastManager::save_opml ()
    {
      xmlDocPtr doc = xmlNewDoc  (BAD_CAST "1.0");
        
      xmlNodePtr opml = xmlNewNode (NULL, BAD_CAST "opml");
      xmlNodePtr head = xmlNewNode (NULL, BAD_CAST "head");
      xmlNodePtr body = xmlNewNode (NULL, BAD_CAST "body");

      xmlSetProp (opml, BAD_CAST "version", BAD_CAST "1.0"); 
      xmlNewTextChild (head, NULL, BAD_CAST "title", BAD_CAST "BMP2 PodCast List");
      xmlDocSetRootElement (doc, opml);

      xmlAddChild (opml, head);
      xmlAddChild (opml, body);

      for (MPodCasts::const_iterator i = m_casts.begin() ; i != m_casts.end() ; ++i)
      {
        PodCast const& cast (i->second);

        std::cerr << " Writing Cast " << cast.cast_uuid << std::endl;

        xmlNodePtr o = xmlNewNode (NULL, BAD_CAST "outline");
        xmlAddChild (body, o);

        xmlSetProp (o, BAD_CAST "text",
                       BAD_CAST  cast.title.get().c_str());

        xmlSetProp (o, BAD_CAST "title",
                       BAD_CAST  cast.title.get().c_str());

        xmlSetProp (o, BAD_CAST "type",  
                       BAD_CAST "rss");

        xmlSetProp (o, BAD_CAST "description", 
                       BAD_CAST  (cast.description? cast.description.get().c_str():""));

        xmlSetProp (o, BAD_CAST "htmlUrl", 
                       BAD_CAST  (cast.link? cast.link.get().c_str():""));

        xmlSetProp (o, BAD_CAST "xmlUrl", 
                       BAD_CAST  cast.uri.c_str()); 

        xmlSetProp (o, BAD_CAST "updateInterval", 
                       BAD_CAST "-1"); 

        xmlSetProp (o, BAD_CAST "id", 
                       BAD_CAST  cast.cast_uuid.c_str()); 

        static boost::format ftime ("%llu");

        xmlSetProp (o, BAD_CAST "lastPollTime", 
                       BAD_CAST  (ftime % cast.last_poll_time).str().c_str()); 

        xmlSetProp (o, BAD_CAST "lastFaviconPollTime", 
                       BAD_CAST "0"); 

        xmlSetProp (o, BAD_CAST "sortColumn", 
                       BAD_CAST "time"); 
      }

      xmlKeepBlanksDefault (0);
      xmlChar * data;
      int size;
      xmlDocDumpFormatMemoryEnc (doc, &data, &size, "UTF-8", 1);

      std::string filename = Glib::build_filename (BMP_PATH_USER_DIR, "feedlist.opml");
      std::ofstream o (filename.c_str());
      o << data;
      o.close ();
      xmlFreeDoc (doc);
      g_free (data);
    }

    PodCastManager::~PodCastManager ()
    {
      save_state ();
    }

    void
    PodCastManager::pod_cast_delete (Glib::ustring const& uri)
    {
      PodCast const& cast (pod_cast_fetch (uri));

      g_unlink (cast_filename (cast).c_str());
      g_unlink (cast_overlay_filename (cast).c_str());
      if (cast.image_url)
        {
          std::string filename = cast_image_filename (cast);
          g_unlink (filename.c_str());
        }

      m_casts.erase (uri);
    }

    void
    PodCastManager::pod_cast_get_list (PodCastList & list)
    {
      for (MPodCasts::const_iterator i = m_casts.begin() ; i != m_casts.end() ; ++i)
      {
        list.push_back (i->second.uri);
      }
    }

    PodCast const&
    PodCastManager::pod_cast_fetch (Glib::ustring const& uri, Glib::RefPtr<Gdk::Pixbuf> & pixbuf, bool load_cover) const
                                   
    {
      PodCast const& cast (m_casts.find (uri)->second);

      if (cast.image_url)
        {
          pixbuf = Gdk::Pixbuf::create_from_file (cast_image_filename (cast)); 
        }
      else
        {
          pixbuf = Glib::RefPtr<Gdk::Pixbuf>(0);
        }

      return cast; 
    }

    PodCast const&
    PodCastManager::pod_cast_fetch (Glib::ustring const& uri) const
    {
      return m_casts.find (uri)->second;
    }

    void
    PodCastManager::pod_cast_load (std::string const& cast_uuid, PodCast & cast) const
    {
      using namespace Glib;

      cast.cast_uuid = cast_uuid; 

      try{

          if (Glib::file_test (cast_overlay_filename (cast), Glib::FILE_TEST_EXISTS))
            {
              try{
                  PodCastOverlayParser parser (cast.overlay_items);
                  Markup::ParseContext context (parser);

                  std::cerr << " Parsing Overlay w/ Filename '" << cast_overlay_filename (cast) << "'" 
                            << std::endl;

                  context.parse (Glib::file_get_contents (cast_overlay_filename (cast)));
                  context.end_parse ();

                  if (!parser.check_sanity ())
                    {
                      static boost::format message (_("Error parsing overlay for item %s"));
                      throw Bmp::PodCastTools::ParsingError((message % cast_uuid.c_str()).str());
                    }

                  std::cerr << " Parsing Overlay w/ Filename '" << cast_overlay_filename (cast) 
                            << "' SUCCESS " << std::endl;
                }
             catch (ConvertError & cxe)
                {
                  std::cerr << " Parsing Overlay w/ Filename '" << cast_overlay_filename (cast) 
                            << "' FAILED: " << cxe.what() << std::endl;

                  throw Bmp::PodCastTools::ParsingError(cxe.what()); 
                }
             catch (MarkupError & cxe)
                {
                  std::cerr << " Parsing Overlay w/ Filename '" << cast_overlay_filename (cast) 
                            << "' FAILED: " << cxe.what() << std::endl;
                  throw Bmp::PodCastTools::ParsingError(cxe.what()); 
                }
              catch (...)
                {
                  std::cerr << " Parsing Overlay w/ Filename '" << cast_overlay_filename (cast) 
                            << "' FAILED " << std::endl;
                  throw Bmp::PodCastTools::ParsingError();
                }
            }

          std::cerr << " Parsing Cast w/ Filename '" << cast_filename (cast) << "'" << std::endl;

          if (podcast_rss_parse (cast, Glib::file_get_contents (cast_filename (cast))) < 0)
            {
              static boost::format message (_("Error parsing RSS from item %s"));
              throw Bmp::PodCastTools::ParsingError((message % cast_uuid.c_str()).str());
            }

          std::cerr << " Parsing Cast w/ Filename '" << cast_filename (cast) << "' SUCCESS " << std::endl;

#if 0
          PodCastParser parser (cast);
          Markup::ParseContext context (parser);

          std::cerr << " Parsing Cast w/ Filename '" << cast_filename (cast) << "'" << std::endl;

          context.parse (Glib::file_get_contents (cast_filename (cast)));
          context.end_parse ();

          if (!parser.check_sanity ())
            {
              static boost::format message (_("Error parsing RSS from item %s"));
              throw Bmp::PodCastTools::ParsingError((message % cast_uuid.c_str()).str());
            }
#endif

        }
     catch (Bmp::URI::ParseError & cxe)
        {
          throw Bmp::PodCastTools::InvalidUriError(cxe.what()); 
        }
     catch (Bmp::VFS::Exception & cxe)
        {
          throw Bmp::PodCastTools::NetworkError(cxe.what()); 
        }
     catch (ConvertError & cxe)
        {
          throw Bmp::PodCastTools::ParsingError(cxe.what()); 
        }
    }

    void
    PodCastManager::pod_cast_update (Glib::ustring const& uri)
    {
      PodCast const& cast (m_casts.find(uri)->second);

      g_unlink (cast_filename (cast).c_str());
      if (cast.image_url)
        {
          std::string filename = cast_image_filename (cast);
          g_unlink (filename.c_str());
        }

      pod_cast_overlay_save (cast);
      pod_cast_cache (uri, true, cast.cast_uuid); 
    }

    void
    PodCastManager::pod_cast_cache (Glib::ustring const&  uri,
                                    bool                  update,
                                    std::string const&    uuid_)
    {
      using namespace Glib;

      std::string uuid;

      if (!update)
      {
        if (m_casts.find (uri) != m_casts.end())
          {
            throw Bmp::PodCastTools::PodCastExistsError();
          }

          // generate unique filename
          char s[37];
          do {
              uuid_t uu;
              uuid_generate (uu);  
              uuid_unparse (uu, s);
              uuid = s;
          } while (Glib::file_test (Glib::build_filename (BMP_PATH_PODCAST_CACHE_DIR, s),
                                    Glib::FILE_TEST_EXISTS));
      }
      else
      {
        uuid = uuid_;
        m_casts.erase (uri);
      }
    
      std::string data; 
      PodCast cast;
      cast.cast_uuid = uuid;

      if(update)
        {
          if(Glib::file_test (cast_overlay_filename (cast), Glib::FILE_TEST_EXISTS))
          {
            try{
                PodCastOverlayParser parser (cast.overlay_items);
                Markup::ParseContext context (parser);

                std::cerr << " Parsing Overlay w/ Filename for UPDATE '" << cast_overlay_filename (cast) << "'" 
                          << std::endl;

                context.parse (Glib::file_get_contents (cast_overlay_filename (cast)));
                context.end_parse ();

                if (!parser.check_sanity ())
                  {
                    static boost::format message (_("Error parsing overlay for item %s"));
                    throw Bmp::PodCastTools::ParsingError((message % cast.cast_uuid.c_str()).str());
                  }

                std::cerr << " Parsing Overlay w/ Filename for UPDATE '" << cast_overlay_filename (cast) 
                          << "' SUCCESS " << std::endl;
              }
           catch (ConvertError & cxe)
              {
                std::cerr << " Parsing Overlay w/ Filename for UPDATE '" << cast_overlay_filename (cast) 
                          << "' FAILED: " << cxe.what() << std::endl;

                throw Bmp::PodCastTools::ParsingError(cxe.what()); 
              }
           catch (MarkupError & cxe)
              {
                std::cerr << " Parsing Overlay w/ Filename for UPDATE '" << cast_overlay_filename (cast) 
                          << "' FAILED: " << cxe.what() << std::endl;
                throw Bmp::PodCastTools::ParsingError(cxe.what()); 
              }
            catch (...)
              {
                std::cerr << " Parsing Overlay w/ Filename for UPDATE '" << cast_overlay_filename (cast) 
                          << "' FAILED " << std::endl;
                throw Bmp::PodCastTools::ParsingError();
              }
          }
        else
          {
              std::cerr << " Parsing Overlay w/ Filename for UPDATE '" << cast_overlay_filename (cast) 
                        << "': No such file " << std::endl;
          }
        }

      try{

          Bmp::URI u (uri);

          Neon::Request r (u.hostname, u.fullpath(), (u.port > 0) ? u.port : 80);
          
          r >> data; 
          r.clear ();

          if (podcast_rss_parse (cast, data) < 0)
            {
              static boost::format message (_("Error parsing RSS from URI %s"));
              throw Bmp::PodCastTools::ParsingError((message % uri.c_str()).str());
            }

#if 0
          PodCastParser parser (cast);
          Markup::ParseContext context (parser);
          context.parse (data);
          context.end_parse ();

          if (!parser.check_sanity ())
            throw Bmp::PodCastTools::ParsingError();
#endif

        }
     catch (Bmp::URI::ParseError& cxe)
        {
          throw Bmp::PodCastTools::InvalidUriError(cxe.what()); 
        }
     catch (Bmp::VFS::Exception& cxe)
        {
          throw Bmp::PodCastTools::NetworkError(cxe.what()); 
        }
     catch (ConvertError& cxe)
        {
          throw Bmp::PodCastTools::ParsingError(cxe.what()); 
        }

      if (cast.items.empty())
        throw PodCastInvalidError();
    
      cast.cast_uuid = uuid; 

      std::ofstream o (cast_filename (cast).c_str()); 
      o << data.c_str();  
      o.close ();

      cast.uri = uri;
      cast.last_poll_time = time(NULL);

      if (cast.image_url)
        {
          Glib::RefPtr<Gdk::Pixbuf> cast_image = Util::get_image_from_uri (cast.image_url.get());
          cast_image->save (cast_image_filename (cast), "png");
        }

      m_casts.insert (std::make_pair (uri, cast));
    }

    void
    PodCastManager::pod_cast_item_change (Glib::ustring const& uri,
                                          Glib::ustring const& guid_value,
                                          PodCastOverlayItem const& overlay_item)
    {
      VPodCastItems & items (m_casts.find (uri)->second.items);

      for (VPodCastItems::iterator item = items.begin () ; item != items.end() ; ++item)
      {
        if (item->guid_value && (item->guid_value.get() == guid_value)) 
        {
          item_overlay ((*item), overlay_item);
          save_state ();
          return;
        }
      }
    }

    //////////////////////////////////////////////////////////////////////////////////////////////

#define STATE(e) ((state & e) != 0)
#define SET_STATE(e) ((state |= e))
#define CLEAR_STATE(e) ((state &= ~ e))

    bool
    OPMLParser::check_sanity  ()
    {
      if (state)
        {
          g_warning (G_STRLOC ": State should be 0, but is %d", state);
          return false;
        }
      return true;
    }

    OPMLParser::OPMLParser (MPodCasts & casts, const PodCastManager * manager)
        : m_casts (casts), m_manager (manager), state (0)
    {
    }

    OPMLParser::~OPMLParser () 
    {
    }
 
    void
    OPMLParser::on_start_element  (Glib::Markup::ParseContext& context,
                                      Glib::ustring const& name,
                                      AttributeMap const& attributes)
	  {
        if (name == "opml") 
        {
          SET_STATE(E_OPML);
          return;
        }

        if (name == "head") 
        {
          SET_STATE(E_HEAD);
          return;
        }

        if (name == "body") 
        {
          SET_STATE(E_BODY);
          return;
        }

        if (name == "outline") 
        {
          SET_STATE(E_OUTLINE);
          if (attributes.find ("xmlUrl") != attributes.end())
            {
              try{

                PodCast cast;

                std::cerr << " Loading Cast with ID " << attributes.find("id")->second << std::endl;

                m_manager->pod_cast_load (attributes.find("id")->second.c_str(), cast);

                cast.uri = attributes.find ("xmlUrl")->second; 

                if (attributes.find ("lastPollTime") != attributes.end())
                  {
                    cast.last_poll_time = strtoull (attributes.find ("lastPollTime")->second.c_str(), NULL, 10);
                  }

                m_casts.insert (std::make_pair (cast.uri, cast)); 

              }
            catch (Bmp::PodCastTools::ParsingError & cxe)
              {
                std::cerr << " Loading Cast with ID " << attributes.find("id")->second << " FAILED: " << cxe.what() << std::endl;
              }
            catch (...)
              {
                std::cerr << " Loading Cast with ID " << attributes.find("id")->second << " FAILED " << std::endl;
              }
            }
          return;
        }
    }

    void
    OPMLParser::on_end_element    (Glib::Markup::ParseContext& context,
                                      Glib::ustring const& name)
	  {
        if (name == "opml") 
        {
          CLEAR_STATE(E_OPML);
          return;
        }

        if (name == "head") 
        {
          CLEAR_STATE(E_HEAD);
          return;
        }

        if (name == "body") 
        {
          CLEAR_STATE(E_BODY);
          return;
        }

        if (name == "outline") 
        {
          CLEAR_STATE(E_OUTLINE);
          return;
        }
	  }

    void
    OPMLParser::on_text       (Glib::Markup::ParseContext& context,
                                  Glib::ustring const& text)
    {
    }

    void
    OPMLParser::on_passtrough (Glib::Markup::ParseContext& context,
                                  Glib::ustring const& text) {}

    void
    OPMLParser::on_error      (Glib::Markup::ParseContext& context,
                                  Glib::MarkupError const& error) {}


    //////////////////////////////////////////////////////////////////////////////////////////////

    bool
    PodCastOverlayParser::check_sanity  ()
    {
      if (state)
        {
          g_warning (G_STRLOC ": State should be 0, but is %d", state);
          return false;
        }
      return true;
    }

    PodCastOverlayParser::PodCastOverlayParser (MOverlayItems & overlays)
        : m_overlays (overlays), state (0)
    {
    }

    PodCastOverlayParser::~PodCastOverlayParser () 
    {
    }
 
    void
    PodCastOverlayParser::on_start_element  (Glib::Markup::ParseContext& context,
                                             Glib::ustring const& name,
                                             AttributeMap const& attributes)
	  {
      if (name == "bmp-cast-overlay")
        {
          SET_STATE(E_BMP_CAST_OVERLAY);
          return;
        }

      if (name == "cast-item")
        {
          SET_STATE(E_CAST_ITEM);
          PodCastOverlayItem overlay_item;

          // fill item with overlay attributes
          int _i;

          _i = g_ascii_strtoull (attributes.find ("listened-to")->second.c_str(), NULL, 10);
          overlay_item.listened_to = ((_i > 0) ? true : false); 

          _i = g_ascii_strtoull (attributes.find ("downloaded")->second.c_str(), NULL, 10);
          overlay_item.downloaded =  ((_i > 0) ? true : false); 

          if (overlay_item.downloaded)
            {
              overlay_item.localfilename = Glib::filename_from_utf8 (attributes.find ("localfilename")->second);

              std::cerr << "Local filename: '" << overlay_item.localfilename << "'" << std::endl;

              if (!Glib::file_test (overlay_item.localfilename, Glib::FILE_TEST_EXISTS))
                {
                  overlay_item.downloaded = false;
                  overlay_item.localfilename = std::string(); 
                }
            } 
     
          m_overlays.insert (std::make_pair (attributes.find ("item-guid")->second, overlay_item));
          return;
        }
    }

    void
    PodCastOverlayParser::on_end_element    (Glib::Markup::ParseContext& context,
                                             Glib::ustring const& name)
	  {
      if (name == "bmp-cast-overlay")
        {
          CLEAR_STATE(E_BMP_CAST_OVERLAY);
          return;
        }

      if (name == "cast-item")
        {
          CLEAR_STATE(E_CAST_ITEM);
          return;
        }
	  }

    void
    PodCastOverlayParser::on_text       (Glib::Markup::ParseContext& context,
                                         Glib::ustring const& text)
    {
    }

    void
    PodCastOverlayParser::on_passtrough (Glib::Markup::ParseContext& context,
                                  Glib::ustring const& text) {}

    void
    PodCastOverlayParser::on_error      (Glib::Markup::ParseContext& context,
                                  Glib::MarkupError const& error) {}

  }
}
