#ifndef _AUSB_H
#define _AUSB_H

/* Wrapper/Extension code to libusb-0.1 to support asynchronous requests
 * on Linux platforns 
 *
 * (C) 2004 by Harald Welte <laforge@gnumonks.org>
 *
 * Distributed under the terms of GNU LGPL, Version 2.1
 */


#include <usb.h>

#ifdef CONFIG_COMPAT
#undef CONFIG_COMPAT
#endif
#ifndef __user
# define __user
#endif
#include <linux/usbdevice_fs.h>

#define AUSB_USBDEVFS_URB_TYPES	4

/* structures */
struct ausb_callback {
  void (*handler)(struct usbdevfs_urb *uurb, void *userdata);
  void *userdata;
};

struct ausb_dev_handle {
  usb_dev_handle *uh;
  struct ausb_callback cb[AUSB_USBDEVFS_URB_TYPES];

  struct usbdevfs_urb *intUrb;
  int inBulkRead;
};

typedef struct ausb_dev_handle ausb_dev_handle;

/* intitialization */ 
int ausb_init(void);

/**
 * Open the given USB device. This creates the necessary
 * ausb handle which must be free'd when it is no longer
 * needed by calling @ref ausb_close.
 * @return ausb handle to be used by other functions in this group
 * @param dev usb device object obtained from libusb
 *
 */
ausb_dev_handle *ausb_open(struct usb_device *dev);
int ausb_close(ausb_dev_handle *ah);

/**
 * Register a callback which is called as soon as an URB request is
 * finished.
 * @param ah ausb handle obtained via @ref ausb_open
 * @param type type of the URB for which this callback is to be registerd:
 *  (USBDEVFS_URB_TYPE_INTERRUPT or USBDEVFS_URB_TYPE_BULK)
 * @param callback callback function to be used. This function receives a
 *  pointer to the finished URB and caller-specified user data
 * @param userdata userdata to be passed to the callback function
 */
int ausb_register_callback(ausb_dev_handle *ah, unsigned char type,
			   void (*callback)(struct usbdevfs_urb *uurb,
					    void *userdata),
			   void *userdata);

/* asynchronous URB related functions */
void ausb_dump_urb(struct usbdevfs_urb *uurb);


/**
 * Prepares an interrupt URB before submission.
 * Interrupt URBs in this sublibrary lead to POSIX signals as soon as they
 * are finished. This allows for receiption of interrupt URBs even if the
 * application does not expect them.
 * @param uurb urb to be filled
 * @param endpoint endpoint id
 * @param buffer buffer for the data to be received
 * @param buffer_length length of the buffer pointed to by buffer
 */
void ausb_fill_int_urb(struct usbdevfs_urb *uurb, unsigned char endpoint,
                       void *buffer, int buffer_length);

/**
 * Prepares a bulk URB before submission.
 * @param uurb urb to be filled
 * @param endpoint endpoint id
 * @param buffer buffer for the data to be received/sent
 * @param buffer_length length of the buffer pointed to by buffer
 */
void ausb_fill_bulk_urb(struct usbdevfs_urb *uurb, unsigned char endpoint,
                        void *buffer, int buffer_length);

/**
 * Submits an URB. As soon as this URB has been finished by the kernel it is
 * available via @ref ausb_get_urb.
 * @param ah ausb handle obtained via @ref ausb_open
 * @param uurb urb for submission
 */
int ausb_submit_urb(ausb_dev_handle *ah, struct usbdevfs_urb *uurb);

/**
 * Discards an URB which has been submitted using @ref ausb_submit_urb but
 * has not yet been finished (i.e. @ref ausb_get_urb did not yet return this
 * URB)
 * @param ah ausb handle obtained via @ref ausb_open
 * @param uurb urb to discard
 */
int ausb_discard_urb(ausb_dev_handle *ah, struct usbdevfs_urb *uurb);

/**
 * This function returns the next URB which has been finished. It is up to the caller
 * to decide whether the urb returned (if any) is an interrupt URB or a bulk URB and
 * to handle that URB accordingly.
 */
struct usbdevfs_urb *ausb_get_urb(ausb_dev_handle *ah);

/**
 * This function enables handling of received interrupt URBs. Without this function
 * beeing called all interrupt URBs are just ignored.
 * With the old cyberjack this function will always be called because that reader
 * randomly sends unimportant data via interrupt pipe.
 * With the new cyberjack (0x300) we only call this function while we are waiting for the
 * response to a command, because this reader sends keypresses via interrupt pipe which we
 * only accept while a command is active (because otherwise a keypress has no context).
 */
void ausb_start_accept_int(ausb_dev_handle *ah);

/**
 * Stop handling of interrupt URBs (see @ref ausb_start_accept_int).
 */
void ausb_end_accept_int(ausb_dev_handle *ah);


/* synchronous functions, mostly wrappers for libusb */
int ausb_claim_interface(ausb_dev_handle *ah, int interface);
int ausb_release_interface(ausb_dev_handle *ah, int interface);
int ausb_set_configuration(ausb_dev_handle *dev, int configuration);
int ausb_clear_halt(ausb_dev_handle *dev, unsigned int ep);
int ausb_reset(ausb_dev_handle *dev);
int ausb_resetep(ausb_dev_handle *dev, int ep);
int ausb_bulk_write(ausb_dev_handle *ah, int ep, char *bytes, int size, 
		    int timeout);
int ausb_bulk_read(ausb_dev_handle *ah, int ep, char *bytes, int size, 
		   int timeout);
#ifdef LIBUSB_HAS_GET_DRIVER_NP
int ausb_get_driver_np(ausb_dev_handle *ah, int interface, char *name,
		       unsigned int namelen);
#endif
#ifdef LIBUSB_HAS_DETACH_KERNEL_DRIVER_NP
int ausb_detach_kernel_driver_np(ausb_dev_handle *dev, int interface);
int ausb_reattach_kernel_driver_np(ausb_dev_handle *dev, int interface);
#endif

#endif /* _AUSB_H */
