/* log.c - 2007/03/25 */
/*
 *  EasyTAG - Tag editor for MP3 and Ogg Vorbis files
 *  Copyright (C) 2000-2007  Jerome Couderc <easytag@gmail.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <glib/gi18n-lib.h>
#include <string.h>
#include <stdio.h>

#include "log.h"
#include "easytag.h"
#include "bar.h"



/****************
 * Declarations *
 ****************/

GtkWidget    *LogList          = NULL;
GtkListStore *logListModel;
GList        *LogPrintTmpList  = NULL; // Temporary list to store messages for the LogList when this control wasn't yet created

enum
{
    LOG_TEXT,
    LOG_ROW_BACKGROUND,
    LOG_ROW_FOREGROUND,
    LOG_COLUMN_COUNT
};

// File for log
gchar *LOG_FILE = ".easytag/easytag.log";


/**************
 * Prototypes *
 **************/
gboolean Log_Popup_Menu_Handler (GtkMenu *menu, GdkEventButton *event);
void     Log_Print_Tmp_List (void);



/*************
 * Functions *
 *************/

GtkWidget *Create_Log_Area (void)
{
    GtkWidget *Frame;
    GtkWidget *ScrollWindowLogList;
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;
    GtkWidget *PopupMenu;


    Frame = gtk_frame_new(_("Log"));
    gtk_container_set_border_width(GTK_CONTAINER(Frame), 2);

    /*
     * The ScrollWindow and the List
     */
    ScrollWindowLogList = gtk_scrolled_window_new(NULL,NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(ScrollWindowLogList),
                                   GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
    gtk_container_add(GTK_CONTAINER(Frame),ScrollWindowLogList);

    /* The file list */
    logListModel = gtk_list_store_new(LOG_COLUMN_COUNT,
                                      G_TYPE_STRING,
                                      GDK_TYPE_COLOR,
                                      GDK_TYPE_COLOR);

    LogList = gtk_tree_view_new_with_model(GTK_TREE_MODEL(logListModel));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(LogList), FALSE);
    gtk_container_add(GTK_CONTAINER(ScrollWindowLogList), LogList);
    gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(LogList), FALSE);
    gtk_widget_set_size_request(LogList, 0, 0);

    renderer = gtk_cell_renderer_text_new();
    column = gtk_tree_view_column_new();
    gtk_tree_view_column_pack_start(column, renderer, FALSE);
    gtk_tree_view_column_set_attributes(column, renderer,
                                        "text",           LOG_TEXT,
                                        "background-gdk", LOG_ROW_BACKGROUND,
                                        "foreground-gdk", LOG_ROW_FOREGROUND,
                                        NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(LogList), column);
    gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
    gtk_tree_view_set_reorderable(GTK_TREE_VIEW(LogList), FALSE);
    gtk_tree_selection_set_mode(gtk_tree_view_get_selection(GTK_TREE_VIEW(LogList)),GTK_SELECTION_MULTIPLE);

    // Create Popup Menu on browser album list
    PopupMenu = gtk_ui_manager_get_widget(UIManager, "/LogPopup");
    g_signal_connect_swapped(G_OBJECT(LogList),"button_press_event",
                             G_CALLBACK(Log_Popup_Menu_Handler), G_OBJECT(PopupMenu));

    // Load pending messages in the Log list
    Log_Print_Tmp_List();

    //gtk_widget_show_all(ScrollWindowLogList);
    gtk_widget_show_all(Frame);

    //return ScrollWindowLogList;
    return Frame;
}


/*
 * Log_Popup_Menu_Handler : displays the corresponding menu
 */
gboolean Log_Popup_Menu_Handler (GtkMenu *menu, GdkEventButton *event)
{
    if (event && (event->type==GDK_BUTTON_PRESS) && (event->button == 3))
    {
        gtk_menu_popup(menu,NULL,NULL,NULL,NULL,event->button,event->time);
        return TRUE;
    }
    return FALSE;
}

/*
 * Remove all lines in the log list
 */
void Log_Clean_Log_List (void)
{
    if (logListModel)
        gtk_list_store_clear(logListModel);
}



/*
 * Function to use anywhere in the application to send a message to the LogList
 */
void Log_Print (gchar const *format, ...)
{
    va_list args;
    gchar *string;

    GtkTreeIter iter;
    static gboolean first_time = TRUE;
    static gchar *file_path = NULL;
    FILE *file = NULL;


    va_start (args, format);
    string = g_strdup_vprintf (format, args);
    va_end (args);

    // If the log window is displayed then messages are displayed, else
    // the messages are stored in a temporary list.
    if (LogList && logListModel)
    {
        gtk_list_store_append(logListModel, &iter);
        gtk_list_store_set(logListModel, &iter,
                           LOG_TEXT, string,
                           LOG_ROW_BACKGROUND, NULL,
                           LOG_ROW_FOREGROUND, NULL,
                           -1);
    }else
    {
        LogPrintTmpList = g_list_append(LogPrintTmpList,string);
        //g_print("%s",string);
    }


    // Store also the messages in the log file.
    if (!file_path)
        file_path = g_strconcat(HOME_VARIABLE,
                                HOME_VARIABLE[strlen(HOME_VARIABLE)-1]!=G_DIR_SEPARATOR ? G_DIR_SEPARATOR_S : "",
                                LOG_FILE,NULL);

    // The first time, the whole file is delete. Else, text is appended
    if (first_time)
        file = fopen(file_path,"w+");
    else
        file = fopen(file_path,"a+");
    //g_free(file_path);

    if (file)
    {
        gchar *data = g_strdup_printf("%s\n",string);
        fwrite(data,strlen(data),1,file);
        g_free(data);

        first_time = FALSE;
        fclose(file);
    }

}

/*
 * Display pending messages in the LogList
 */
void Log_Print_Tmp_List (void)
{
    GtkTreeIter iter;

    LogPrintTmpList = g_list_first(LogPrintTmpList);
    while (LogPrintTmpList)
    {
        
        if (LogList && logListModel)
        {
            gtk_list_store_append(logListModel, &iter);
            gtk_list_store_set(logListModel, &iter,
                               LOG_TEXT, (gchar *)LogPrintTmpList->data,
                               LOG_ROW_BACKGROUND, NULL,
                               LOG_ROW_FOREGROUND, NULL,
                               -1);
        }
        
        if (!LogPrintTmpList->next) break;
        LogPrintTmpList = LogPrintTmpList->next;
    }

    if (LogPrintTmpList)
    {
        g_list_free(LogPrintTmpList);
        LogPrintTmpList = NULL;
    }

}

