/*
 * Gnome Attacks
 *
 * Copyright (c) 2003 by Andy Balaam
 *
 * Released under the GNU General Public License
 * with ABSOLUTELY NO WARRANTY.
 *
 * See the file COPYING for more information.
 *
 */

#include "game.h"
#include "callbacks.h"
#include "graphics.h"
#include "hiscore.h"


/**
 * Set up a random seed and set the levels to the first ones.
 */
void game_setup() {
  
  g_random_set_seed( time( NULL ) );
  
  level = 1;
  graphics_level = 1;
  redraw_pausing = FALSE;
  
}

gint _bullet_tic( gpointer data )
{
    
    /* If the bullet exists, move it */
    if( bullet_exists )
    {
        
        bullet_y++;
        
        /* If the bullet has hit a building */
        if( bullet_y >= BLOCKS_HEIGHT - building_heights[bullet_x] )
        {
            
            int old_height;
            int reduce_amount;
            
            old_height = building_heights[bullet_x];
            reduce_amount = 1 + ( g_random_double() * MAX_HEIGHT_REDUCTION );
            
            if( building_heights[bullet_x] > 0 )
            {
              
              building_heights[bullet_x] -= reduce_amount;
              current_score += 10 * reduce_amount;
              _update_score();
              
              if( building_heights[bullet_x] < 0 ) {
                building_heights[bullet_x] = 0;
              }
              
              /* If we've finished the level, go fast */
              if( _all_buildings_gone() ) {
                
                gtk_timeout_remove( bullet_timeout );
                bullet_timeout = gtk_timeout_add( BULLET_TIC_INTERVAL_FAST,
                  _bullet_tic, (gpointer)NULL );
                
              }
              
            }
            
            graphics_redraw_building( bullet_x, old_height );
            graphics_redraw_ship();

            bullet_exists = FALSE;
            
        }
        else
        {
            graphics_redraw_bullet();
        }
        
    }
    
    no_bullet_tics++;
    
    /* Deal with the ship tic if it's time, or return otherwise */
    if( no_bullet_tics>BULLET_TICS_PER_SHIP_TIC )
    {
        no_bullet_tics = 0;
        return _ship_tic();
    }
    else
    {
        return TRUE;
    }
    
}

gint _titles_tic( gpointer data )
{
    
  switch(status) {
    
    case STATUS_TITLE_SCREEN:
      status = STATUS_INSTRUCTIONS_SCREEN;
      break;
      
    case STATUS_INSTRUCTIONS_SCREEN:
      status = STATUS_HISCORE_SCREEN;
      break;
      
    case STATUS_HISCORE_SCREEN:
      status = STATUS_TITLE_SCREEN;
      break;
    
  }

  graphics_redraw_whole_screen();
  
  return TRUE;
    
}

gint _ship_tic( )
{
    
    ship_x++;
    
    /* Check for going off the edge of the screen */
    if( ship_x >= BLOCKS_WIDTH )
    {
        ship_x = 0;
        ship_y++;
        
        if( ship_y >= BLOCKS_HEIGHT )
        {
            _level_complete();
            return FALSE;
        }
        
    }
    
    graphics_redraw_ship();
    
    /* Check for collision with a building */
    if( ship_y >= BLOCKS_HEIGHT - building_heights[ship_x] )
    {
        
        /* TODO death of the ship */
        
        _level_failed();
        return FALSE;
        
    }
    
    return TRUE;
    
}

void game_pause_event()
{
  gtk_timeout_remove( bullet_timeout );
  
  status = STATUS_PAUSED;
  
  gtk_widget_set_sensitive( GTK_WIDGET( pause_game1 ), FALSE );
  gtk_widget_set_sensitive( GTK_WIDGET( restart_game1 ), TRUE );
  
}

void game_restart_event()
{
  bullet_timeout = gtk_timeout_add( BULLET_TIC_INTERVAL,
      _bullet_tic,
      (gpointer)NULL );
   
  status = STATUS_IN_GAME;
  
  gtk_widget_set_sensitive( GTK_WIDGET( pause_game1 ), TRUE );
  gtk_widget_set_sensitive( GTK_WIDGET( restart_game1 ), FALSE );
  
}

void game_end_event()
{
  gtk_widget_set_sensitive( GTK_WIDGET( pause_game1 ), FALSE );
  gtk_widget_set_sensitive( GTK_WIDGET( levels1 ), TRUE );
  gtk_timeout_remove( bullet_timeout );
  game_restart_titles();
}

/*--------------------------------------------------------------------------
    Function Bodies - game setup/finish
  --------------------------------------------------------------------------*/

void _level_setup()
{

  GString* heights_file_mine;
  GString* file_name_mine;
  
  gchar buffer[50];
  gint i;
  
  file_name_mine = g_string_new( NULL );
  g_string_printf( file_name_mine, "level_%d_heights.txt", graphics_level );
  
  heights_file_mine = levelset_get_file_name( file_name_mine->str );
  
  /* If we didn't find a file of building heights, use random ones. */
  if( heights_file_mine == NULL ) {
    
    for( i=0; i<BLOCKS_WIDTH; i++ )
    {
        
        building_heights[i] = level + g_random_double() * 10;
        
    }
    
  } else {  /* If we did find one, load it in. */
  
    FILE* heights_fileptr;
  
    if( ( heights_fileptr 
      = fopen( heights_file_mine->str, "r" ) ) == NULL )
    {
      fprintf( stderr, "gnomeattacks: error reading file '%s'\n",
        file_name_mine );
      exit( 1 );
    }
    
    for( i=0; i<BLOCKS_WIDTH; i++ )
    {
        
        fgets( buffer, 50, heights_fileptr );
        building_heights[i] = atoi( buffer );
        
    }
  
    fclose( heights_fileptr );
    g_string_free( heights_file_mine, TRUE );
    
  }
  
  g_string_free( file_name_mine, TRUE );
  
  ship_x = 0;
  ship_y = 0;
  
  _update_score();
  
  bullet_exists = FALSE;
  
  _set_status_message( "Level %d", level );

  graphics_setup_level();
  
}

void _update_score()
{
  
  _set_status_message( "Score: %d", current_score );

}

void _set_status_message( gchar* format, ... )
{
  
  va_list varptr;
  gchar msg[255];
  
  varptr = NULL;
  
  va_start( varptr, format );
  
  vsnprintf( msg, 255, format, varptr );
  
  gnome_appbar_set_status( appbar1, msg );
  
}

void _level_complete()
{
    
    gtk_timeout_remove( bullet_timeout );
    
    status = STATUS_LEVEL_COMPLETE;
    
    gtk_widget_set_sensitive( GTK_WIDGET( pause_game1 ), FALSE );
    
    _set_status_message( "Level %d complete! Score=%d. Press any key.",
      level, current_score );
    
    graphics_redraw_whole_screen();
    
}

void _level_failed()
{
    
    gtk_timeout_remove( bullet_timeout );
    
    status = STATUS_DIED;
    
    if( current_score > high_scores[9] )
    {
     
      _set_status_message( "You died on level %d! Score=%d.", level,
        current_score );
      
    }
    else
    {
      
      _set_status_message( "You died on level %d! Score=%d. Press any key.",
        level, current_score );
      
    }
    
    gtk_widget_set_sensitive( GTK_WIDGET( pause_game1 ), FALSE );
    
    graphics_redraw_whole_screen();
    
    if( current_score > high_scores[9] )
    {
     
      hiscore_add_score();
      
      gtk_widget_set_sensitive( GTK_WIDGET( levels1 ), TRUE );
      
      _restart_titles_on_screen( STATUS_HISCORE_SCREEN );
      
    }
    
}

gboolean
game_key_press()
{
  
  /* Different behaviour depending on what state we are in. */
  switch(status)
  {
    case STATUS_IN_GAME:
      game_inlevel_key_press();
      break;
      
    case STATUS_DIED:
      game_died_key_press();
      break;
      
    case STATUS_LEVEL_COMPLETE:
      game_level_complete_key_press();
      break;
      
    case STATUS_TITLE_SCREEN:
      game_titles_key_press();
      break;
      
    case STATUS_HISCORE_SCREEN:
      game_titles_key_press();
      break;
      
    case STATUS_INSTRUCTIONS_SCREEN:
      game_titles_key_press();
      break;
    
  }
  return FALSE;
  
}

gboolean
game_inlevel_key_press()
{

  /* If the bullet exists we do nothing */
  if( !bullet_exists )
  {
        
    bullet_exists = TRUE;
    bullet_x = ship_x;
    bullet_y = ship_y;
        
  }
  
  return FALSE;
  
}

gboolean
game_died_key_press()
{
  gtk_widget_set_sensitive( GTK_WIDGET( levels1 ), TRUE );
  game_restart_titles();
  
}

gboolean
game_level_complete_key_press()
{
  level++;
  graphics_level++;
  game_restart();
}

gboolean
game_titles_key_press()
{
  
  gtk_timeout_remove( titles_timeout );
  
  gtk_widget_set_sensitive( GTK_WIDGET( new1 ), FALSE );
  gtk_widget_set_sensitive( GTK_WIDGET( restart_game1 ), FALSE );
  gtk_widget_set_sensitive( GTK_WIDGET( end_game1 ), TRUE );
  gtk_widget_set_sensitive( GTK_WIDGET( levels1 ), FALSE );
  
  current_score = 0;
  level = 1;
  graphics_level = 1;
  game_restart();
  
}

void game_restart()
{
  
  status = STATUS_IN_GAME;
  
  _level_setup();
  
  gtk_widget_set_sensitive( GTK_WIDGET( pause_game1 ), TRUE );
  
  graphics_redraw_whole_screen();
  
  /* Set up the timer for game tics */
  bullet_timeout = gtk_timeout_add( BULLET_TIC_INTERVAL,
      _bullet_tic,
      (gpointer)NULL );
  
}

void game_restart_titles()
{
  
  _restart_titles_on_screen( STATUS_TITLE_SCREEN );
  
}

void _restart_titles_on_screen( int the_status )
{
  
  status = the_status;
  
  gtk_widget_set_sensitive( GTK_WIDGET( new1 ), TRUE );
  gtk_widget_set_sensitive( GTK_WIDGET( restart_game1 ), FALSE );
  gtk_widget_set_sensitive( GTK_WIDGET( pause_game1 ), FALSE );
  gtk_widget_set_sensitive( GTK_WIDGET( end_game1 ), FALSE );
  
  _set_status_message( "" );
  
  graphics_redraw_whole_screen();
  
  /* Set up the timer for game tics */
  titles_timeout = gtk_timeout_add( TITLE_TIC_INTERVAL,
      _titles_tic,
      (gpointer)NULL );
  
}

gboolean _all_buildings_gone()
{
  
  int i;
  
  for(i=0;i<BLOCKS_WIDTH;i++) {
    
    if(building_heights[i]>0) {
      return FALSE;
    }
    
  }
  
  return TRUE;
  
}

/**
 * Clean up after ourselves: remove any timeouts that are set.
 */
void game_quit()
{
  
  if( status == STATUS_TITLE_SCREEN || status == STATUS_HISCORE_SCREEN
      || status == STATUS_INSTRUCTIONS_SCREEN ) {
        
    gtk_timeout_remove( titles_timeout );
    
  } else if( status == STATUS_IN_GAME ) {
    
    gtk_timeout_remove( bullet_timeout );
    
  }
  
}


