/*
 *  Copyright (C) 2005 Kouji TAKAO <kouji@netlab.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <glib/gi18n.h>
#include <libgnomeui/gnome-druid.h>

#include "gpass/file04.h"
#include "application.h"
#include "intergradation.h"

#define GPASS_DAT_PATH "%s/.gpass/gpass.dat"

static GPassViewClass *parent_class = NULL;

static void
gnome_intergradation_instance_init(GTypeInstance *instance, gpointer g_class)
{
    GPassGnomeIntergradation *self = GPASS_GNOME_INTERGRADATION(instance);
    
    self->gpass_dat_path =
        g_strdup_printf(GPASS_DAT_PATH, g_get_home_dir());
}

static void
gnome_intergradation_instance_finalize(GObject *object)
{
    GPassGnomeIntergradation *self = GPASS_GNOME_INTERGRADATION(object);
    
    g_free(self->gpass_dat_path);
    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
gnome_intergradation_class_init(gpointer g_class, gpointer g_class_data)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(g_class);
    
    parent_class = g_type_class_peek_parent(g_class);
    
    gobject_class->finalize = gnome_intergradation_instance_finalize;
}

GType
gpass_gnome_intergradation_get_type(void)
{
    static GType type = 0;
    
    if (type == 0) {
        static const GTypeInfo info = {
            sizeof(GPassGnomeIntergradationClass),
            NULL,
            NULL,
            gnome_intergradation_class_init,
            NULL,
            NULL,
            sizeof(GPassGnomeIntergradation),
            0,
            gnome_intergradation_instance_init,
        };
        
        type = g_type_register_static(GPASS_TYPE_VIEW,
                                      "GPassGnomeIntergradation", &info, 0);
    }
    return type;
}

/***********************************************************
 *
 * Signal handlers
 *
 ***********************************************************/
gboolean
gpass_gnome_intergradation_on_cancel(GnomeDruidPage *page, GtkWidget *widget,
                                     gpointer user_data)
{
    GPassGnomeIntergradation *self;
    
    gpass_view_self_from_widget(widget, (gpointer **) &self);
    GPASS_VIEW(self)->result = GPASS_VIEW_RESULT_FAILED;
    gpass_view_shutdown_main_loop(GPASS_VIEW(self));
    return FALSE;
}

static gboolean
intergrade_entries(GPassGnomeIntergradation *self)
{
    GtkWidget *password_entry =
        glade_xml_get_widget(GPASS_VIEW(self)->xml, "password");
    const gchar *password = gtk_entry_get_text(GTK_ENTRY(password_entry));
    GPassApplication *app = GPASS_APPLICATION(GPASS_VIEW(self)->model);
    GPassFile04 *file;
    GPassEntry *entries;
    GError *error;
    
    error = gpass_file04_open(&file, self->gpass_dat_path, password);
    if (error != NULL) {
        g_error_free(error);
        gpass_view_show_error_message(GPASS_VIEW(self),
                                      _("Incorrect password!"),
                                      password_entry);
        return FALSE;
    }
    error = gpass_file04_read(file, app->entry_factory, &entries);
    if (error != NULL) {
        gchar *msg = g_strdup_printf(_("failed reading passwords: %s"),
                                     error->message);
        
        gpass_view_show_error_message(GPASS_VIEW(self), msg, password_entry);
        g_free(msg);
        g_error_free(error);
        gpass_file04_close(file);
        return FALSE;
    }
    gpass_file04_close(file);
    
    error = gpass_file_create(app->file_path, password);
    if (error != NULL) {
        gchar *msg = g_strdup_printf(_("failed creating: %s"),
                                     error->message);
        
        gpass_view_show_error_message(GPASS_VIEW(self), msg, password_entry);
        g_free(msg);
        g_error_free(error);
        g_object_unref(entries);
        return FALSE;
    }
    error = gpass_file_open(&app->file, app->file_path, password);
    if (error != NULL) {
        gchar *msg = g_strdup_printf(_("failed opening: %s"),
                                     error->message);
        
        gpass_view_show_error_message(GPASS_VIEW(self), msg, password_entry);
        g_free(msg);
        g_error_free(error);
        g_object_unref(entries);
        return FALSE;
    }
    error = gpass_file_write(app->file, entries);
    if (error != NULL) {
        gchar *msg = g_strdup_printf(_("failed writing: %s"),
                                     error->message);
        
        gpass_view_show_error_message(GPASS_VIEW(self), msg, password_entry);
        g_free(msg);
        g_error_free(error);
        gpass_file_close(app->file);
        g_object_unref(entries);
        return FALSE;
    }
    g_object_unref(entries);
    return TRUE;
}

gboolean
gpass_gnome_intergradation_on_intergradation_page_next(GnomeDruidPage *page,
                                                       GtkWidget *widget,
                                                       gpointer user_data)
{
    GPassGnomeIntergradation *self;
    
    gpass_view_self_from_widget(widget, (gpointer **) &self);
    return !intergrade_entries(self);
}

void
gpass_gnome_intergradation_on_finish(GnomeDruidPage *page, GtkWidget *widget,
                                     gpointer user_data)
{
    GPassGnomeIntergradation *self;
    
    gpass_view_self_from_widget(widget, (gpointer **) &self);
    GPASS_VIEW(self)->result = GPASS_VIEW_RESULT_SUCCEED;
    gpass_view_shutdown_main_loop(GPASS_VIEW(self));
}
