#ifndef __NR_FILTER_H__
#define __NR_FILTER_H__

/*
 * SVG filters rendering
 *
 * Author:
 *   Niko Kiirala <niko@kiirala.com>
 *
 * Copyright (C) 2006 Niko Kiirala
 *
 * Released under GNU GPL, read the file 'COPYING' for more information
 */

//#include "display/nr-arena-item.h"
#include "display/nr-filter-primitive.h"
#include "display/nr-filter-types.h"
#include "libnr/nr-pixblock.h"
#include "libnr/nr-matrix.h"
#include "libnr/nr-rect.h"
#include "svg/svg-length.h"
#include "sp-filter-units.h"
#include "gc-managed.h"

struct NRArenaItem;

namespace NR {

class Filter : public Inkscape::GC::Managed<> {
public:
    int render(NRArenaItem const *item, NRPixBlock *pb);

    /**
     * Creates a new filter primitive under this filter object.
     * New primitive is placed so that it will be executed after all filter
     * primitives defined beforehand for this filter object.
     * Should this filter not have enough space for a new primitive, the filter
     * is enlarged to accomodate the new filter element. It may be enlarged by
     * more that one element.
     * Returns a pointer to the filter primitive created.
     * Returns NULL, if type is not valid filter primitive type or filter
     * primitive of such type cannot be created.
     */
    FilterPrimitive *add_primitive(FilterPrimitiveType type);
    /**
     * Removes all filter primitives from this filter.
     * All pointers to filter primitives inside this filter should be
     * considered invalid after calling this function.
     */
    void clear_primitives();
    /**
     * Replaces filter primitive pointed by 'target' with a new filter
     * primitive of type 'type'
     * If 'target' does not correspond to any primitive inside this filter OR
     * 'type' is not a valid filter primitive type OR
     * filter primitive of such type cannot be created,
     * this function returns NULL and doesn't change the internal state of this
     * filter.
     * Otherwise, a new filter primitive is created. Any pointers to filter
     * primitive 'target' should be considered invalid. A pointer to the
     * newly created primitive is returned.
     */
    FilterPrimitive *replace_primitive(FilterPrimitive *primitive,
                                       FilterPrimitiveType type);

    /**
     * Sets the slot number 'slot' to be used as result from this filter.
     * If output is not set, the output from last filter primitive is used as
     * output from the filter.
     * It is an error to specify a pre-defined slot as 'slot'. Such call does
     * not have any effect to the state of filter or its primitives.
     */
    void set_output(int slot);

    void set_x(SVGLength &lenght);
    void set_y(SVGLength &length);
    void set_width(SVGLength &length);
    void set_height(SVGLength &length);

    /**
     * Sets the filter effects region.
     * Passing an unset length (length._set == false) as any of the parameters
     * results in that parameter not being changed.
     * Filter will not hold any references to the passed SVGLength object after
     * function returns.
     * If any of these parameters does not get set, the default value, as
     * defined in SVG standard, for that parameter is used instead.
     */
    void set_region(SVGLength &x, SVGLength &y,
                    SVGLength &width, SVGLength &height);

    /**
     * Resets the filter effects region to its default value as defined
     * in SVG standard.
     */
    void reset_region();

    /**
     * Sets the width of intermediate images in pixels. If not set, suitable
     * resolution is determined automatically. If x_pixels is less than zero,
     * calling this function results in no changes to filter state.
     */
    void set_resolution(double x_pixels);

    /**
     * Sets the width and height of intermediate images in pixels. If not set,
     * suitable resolution is determined automatically. If either parameter is
     * less than zero, calling this function results in no changes to filter
     * state.
     */
    void set_resolution(double x_pixels, double y_pixels);

    /**
     * Resets the filter resolution to its default value, i.e. automatically
     * determined.
     */
    void reset_resolution();

    /**
     * Set the filterUnits-property. If not set, the default value of 
     * objectBoundingBox is used. If the parameter value is not a
     * valid enumeration value from SPFilterUnits, no changes to filter state
     * are made.
     */
    void set_filter_units(SPFilterUnits unit);

    /**
     * Set the primitiveUnits-properterty. If not set, the default value of
     * userSpaceOnUseis used. If the parameter value is not a valid
     * enumeration value from SPFilterUnits, no changes to filter state
     * are made.
     */
    void set_primitive_units(SPFilterUnits unit);

    /** 
     * Returns the amount of pixels the rendering area should be enlarged
     * to prevent visual artefacts when filter needs to read pixels that
     * are outside its output area (e.g. gaussian blur)
     */
    int get_enlarge(Matrix const &m);
    /**
     * Given an object bounding box, this function enlarges it so that
     * it contains the filter effect area.
     */
    void bbox_enlarge(NRRectL &bbox);

    /** Creates a new filter with space for one filter element */
    Filter();
    /** 
     * Creates a new filter with space for n filter elements. If number of
     * filter elements is known beforehand, it's better to use this
     * constructor.
     */
    Filter(int n);
    /** Destroys the filter and all its primitives */
    ~Filter();

private:
    int _primitive_count;
    int _primitive_table_size;

    /** Amount of image slots used, when this filter was rendered last time */
    int _slot_count;

    /** Image slot, from which filter output should be read.
     * Negative values mean 'not set' */
    int _output_slot;

    SVGLength _region_x;
    SVGLength _region_y;
    SVGLength _region_width;
    SVGLength _region_height;

    /* x- and y-resolutions for filter rendering.
     * Negative values mean 'not set'.
     * If _y_pixels is set, _x_pixels should be set, too. */
    double _x_pixels;
    double _y_pixels;

    SPFilterUnits _filter_units;
    SPFilterUnits _primitive_units;

    FilterPrimitive ** _primitive;

    void _create_constructor_table();
    void _enlarge_primitive_table();
    void _common_init();
};


} /* namespace NR */




#endif /* __NR_FILTER_H__ */
/*
  Local Variables:
  mode:c++
  c-file-style:"stroustrup"
  c-file-offsets:((innamespace . 0)(inline-open . 0)(case-label . +))
  indent-tabs-mode:nil
  fill-column:99
  End:
*/
// vim: filetype=cpp:expandtab:shiftwidth=4:tabstop=8:softtabstop=4:encoding=utf-8:textwidth=99 :
