/*
 * dvbevents.cpp
 *
 * Copyright (C) 2003-2007 Christophe Thommeret <hftom@free.fr>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <stdio.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>

#include <qdatetime.h>
#include <qfile.h>

#include <klocale.h>
#include <kstandarddirs.h>

#include "dvbevents.h"



EventSid::EventSid( int s )
{
	sid = s;
	events.setAutoDelete( true );
}

EventSid::~EventSid()
{
	QMutexLocker locker( &mutex );
	events.clear();
}

void EventSid::remove( EventDesc *d )
{
	QMutexLocker locker( &mutex );
	events.remove( d );
}

EventDesc *EventSid::getEventDesc( int n )
{
	QMutexLocker locker( &mutex );
	return events.at(n);
}



EventTsid::EventTsid( int t )
{
	tsid = t;
	sidList.setAutoDelete( true );
}

EventTsid::~EventTsid()
{
	QMutexLocker locker( &mutex );
	sidList.clear();
}

EventSid *EventTsid::getNEventSid( int n )
{
	QMutexLocker locker( &mutex );
	return sidList.at( n );
}

EventSid *EventTsid::getEventSid( int sid )
{
	int i;

	QMutexLocker locker( &mutex );
	for ( i=0; i<(int)sidList.count(); i++ ) {
		if ( sidList.at(i)->getSid()==sid )
			return sidList.at(i);
	}
	EventSid *es = new EventSid( sid );
	sidList.append( es );
	return es;
}

EventDesc *EventTsid::getEventDesc( int sid, int n )
{
	int i;
	EventSid *es=0;

	mutex.lock();
	for ( i=0; i<(int)sidList.count(); i++ ) {
		if ( sidList.at(i)->getSid()==sid ) {
			es = sidList.at(i);
			break;
		}
	}
	mutex.unlock();
	if ( !es )
		return 0;
	return es->getEventDesc( n );
}



EventSource::EventSource( QString src )
{
	source = src;
	tsidList.setAutoDelete( true );
}

EventSource::~EventSource()
{
	QMutexLocker locker( &mutex );
	tsidList.clear();
}

EventTsid *EventSource::getNEventTsid( int n )
{
	QMutexLocker locker( &mutex );
	return tsidList.at( n );
}

EventSid *EventSource::getEventSid( int tsid, int sid )
{
	int i;
	EventTsid *et=0;

	mutex.lock();
	for ( i=0; i<(int)tsidList.count(); i++ ) {
		if ( tsidList.at(i)->getTsid()==tsid ) {
			et = tsidList.at(i);
			break;
		}
	}
	if ( !et ) {
		et = new EventTsid( tsid );
		tsidList.append( et );
	}
	mutex.unlock();
	return et->getEventSid( sid );
}

EventDesc *EventSource::getEventDesc( int tsid, int sid, int n )
{
	int i;
	EventTsid *et=0;

	mutex.lock();
	for ( i=0; i<(int)tsidList.count(); i++ ) {
		if ( tsidList.at(i)->getTsid()==tsid ) {
			et = tsidList.at(i);
			break;
		}
	}
	mutex.unlock();
	if ( !et )
		return 0;
	return et->getEventDesc( sid, n );
}




EventTable::EventTable()
{
	srcList.setAutoDelete( true );
	connect ( &cleanTimer, SIGNAL(timeout()), this, SLOT(setClean()) );
	cleanTimer.start( 10000 );
	epgLoaded = false;
}

EventTable::~EventTable()
{
	QMutexLocker locker( &mutex );
	srcList.clear();
}

void EventTable::saveEpg()
{
	EventSource *esrc;
	EventTsid *et;
	EventSid *es;
	QPtrList<EventDesc> *esEvents;
	EventDesc *desc;
	int i, k, j, m, n;
	QCString c;
	int count=0;
	unsigned char sync=0xff;
	QTime t1=QTime::currentTime();

	QFile f( locateLocal("appdata", "dvbepg.data" ) );
	if ( f.open(IO_WriteOnly|IO_Truncate) ) {
		QDataStream dt( &f );
		for( i=0; i<getNSource(); i++ ) {
			if ( !(esrc=getNEventSource( i )) )
				continue;
			for ( m=0; m<esrc->getNTsid(); m++ ) {
				if ( !(et=esrc->getNEventTsid( m )) )
					continue;
				for ( n=0; n<et->getNSid(); n++ ) {
					if ( !(es=et->getNEventSid( n )) )
						continue;
					esEvents = es->getEvents();
					es->lock();
					for ( j=0; j<(int)esEvents->count(); j++ ) {
						if ( !(desc=esEvents->at( j )) )
							continue;
						dt << sync; // sync byte
						c = desc->source.utf8();
						dt << c.data();
						dt << desc->tid;
						dt << desc->sid;
						dt << desc->tsid;
						dt << desc->nid;
						dt << desc->sn;
						dt << desc->lsn;
						dt << desc->eid;
						dt << desc->running;
						dt << desc->startDateTime.toTime_t();
						dt << (uint)((desc->duration.hour()*3600)+(desc->duration.minute()*60)+desc->duration.second());
						dt << desc->shortEvents.count();
						for ( k=0; k<(int)desc->shortEvents.count(); k++ ) {
							c = desc->shortEvents.at(k)->name.utf8();
							dt << c.data();
							c = desc->shortEvents.at(k)->text.utf8();
							dt << c.data();
						}
						dt << (uint)desc->extEvents.count();
						for ( k=0; k<(int)desc->extEvents.count(); k++ ) {
							c = desc->extEvents.at(k)->utf8();
							dt << c.data();
						}
						c = desc->title.utf8();
						dt << c.data();
						c = desc->subtitle.utf8();
						dt << c.data();
						++count;
					}
					es->unlock();
				}
			}
		}
		f.close();
		fprintf( stderr, "Saved epg data : %d events (%d msecs)\n", count, t1.msecsTo(QTime::currentTime()) );
	}
}

void EventTable::loadEpg()
{
	EventDesc *desc;
	ShortEvent *sev;
	EventSid *slist;
	uint count, tmp, i;
#define EPGBUFSIZE 500
	char buf[EPGBUFSIZE];
	int num=0;
	unsigned char sync;
	QDateTime cur=QDateTime::currentDateTime();
	QTime t1=QTime::currentTime();

	if ( epgLoaded )
		return;

	epgLoaded = true;
	QFile f( locateLocal("appdata", "dvbepg.data" ) );
	if ( f.open(IO_ReadOnly) ) {
		QDataStream dt( &f );
		while (!dt.eof() ) {
			dt >> sync;
			if ( sync!=0xff ) {
				f.close();
				fprintf( stderr, "Sync error while loading epg data : %d events loaded\n", num );
				return;
			}
			desc = new EventDesc();
			dt >> tmp;
			if ( !validString( f, desc, tmp, EPGBUFSIZE, num ) ) return;
			dt.readRawBytes( buf, tmp );
			desc->source = QString::fromUtf8( buf );
			dt >> desc->tid;
			dt >> desc->sid;
			dt >> desc->tsid;
			dt >> desc->nid;
			dt >> desc->sn;
			dt >> desc->lsn;
			dt >> desc->eid;
			dt >> desc->running;
			dt >> tmp;
			desc->startDateTime.setTime_t( tmp );
			dt >> tmp;
			desc->duration = QTime().addSecs( tmp );
			dt >> count;
			for ( i=0; i<count; i++ ) {
				sev = new ShortEvent();
				dt >> tmp;
				if ( !validString( f, desc, tmp, EPGBUFSIZE, num ) ) return;
				dt.readRawBytes( buf, tmp );
				sev->name = QString::fromUtf8( buf );
				dt >> tmp;
				if ( !validString( f, desc, tmp, EPGBUFSIZE, num ) ) return;
				dt.readRawBytes( buf, tmp );
				sev->text = QString::fromUtf8( buf );
				desc->shortEvents.append( sev );
			}
			dt >> count;
			for ( i=0; i<count; i++ ) {
				dt >> tmp;
				if ( !validString( f, desc, tmp, EPGBUFSIZE, num ) ) return;
				dt.readRawBytes( buf, tmp );
				desc->extEvents.append( new QString( QString::fromUtf8( buf ) ) );
			}
			dt >> tmp;
			if ( !validString( f, desc, tmp, EPGBUFSIZE, num ) ) return;
			dt.readRawBytes( buf, tmp );
			desc->title = QString::fromUtf8( buf );
			dt >> tmp;
			if ( !validString( f, desc, tmp, EPGBUFSIZE, num ) ) return;
			dt.readRawBytes( buf, tmp );
			desc->subtitle = QString::fromUtf8( buf );
			if ( desc->startDateTime.addSecs( (desc->duration.hour()*3600)+(desc->duration.minute()*60)+desc->duration.second() )<cur )
				delete desc;
			else {
				slist = getEventSource( desc->source )->getEventSid( desc->tsid, desc->sid );
				slist->lock();
				slist->getEvents()->append( desc );
				slist->unlock();
				++num;
			}
		}
		f.close();
		fprintf( stderr, "Loaded epg data : %d events (%d msecs)\n", num, t1.msecsTo(QTime::currentTime()) );
	}
}

bool EventTable::validString( QFile &f, EventDesc *d, int len, int buflen, int nev )
{
	if ( len<1 || len>buflen ) {
		f.close();
		fprintf( stderr, "Error while loading epg data : %d events loaded\n", nev );
		delete d;
		return false;
	}
	return true;
}

void EventTable::setClean()
{
	start();
}

void EventTable::doClean( bool b )
{
	if ( b ) {
		if ( cleanTimer.isActive() )
			return;
		cleanTimer.start( 10000 );
	}
	else {
		cleanTimer.stop();
		wait();
	}
}

void EventTable::run()
{
	int k, m, n, sec;
	EventSource *esrc;
	EventTsid *et;
	EventSid *es;
	EventDesc *desc;
	QDateTime dt, cur;

	setpriority(PRIO_PROCESS, 0, 19);

	cur = QDateTime::currentDateTime();
	for( k=0; k<getNSource(); k++ ) {
		if ( !(esrc=getNEventSource( k )) )
			continue;
		for ( m=0; m<esrc->getNTsid(); m++ ) {
			if ( !(et=esrc->getNEventTsid( m )) )
				continue;
			for ( n=0; n<et->getNSid(); n++ ) {
				if ( !(es=et->getNEventSid( n )) )
					continue;
				if ( !(desc=es->getEventDesc( 0 )) )
					continue;
				dt = desc->startDateTime;
				sec = desc->duration.hour()*3600+desc->duration.minute()*60+desc->duration.second();
				if ( dt.addSecs( sec )<cur )
					es->remove( desc );
			}
		}
	}
}

EventSource *EventTable::getNEventSource( int n )
{
	QMutexLocker locker( &mutex );
	return srcList.at( n );
}

EventSource *EventTable::getEventSource( QString src )
{
	int i;

	QMutexLocker locker( &mutex );
	for ( i=0; i<(int)srcList.count(); i++ ) {
		if ( srcList.at(i)->getSource()==src )
			return srcList.at(i);
	}
	EventSource *es = new EventSource( src );
	srcList.append( es );
	return es;
}

EventDesc *EventTable::getEventDesc( QString src, int tsid, int sid, int n )
{
	int i;
	EventSource *es=0;

	mutex.lock();
	for ( i=0; i<(int)srcList.count(); i++ ) {
		if ( srcList.at(i)->getSource()==src ) {
			es = srcList.at(i);
			break;
		}
	}
	mutex.unlock();
	if ( !es )
		return 0;
	return es->getEventDesc( tsid, sid, n );
}



DVBevents::DVBevents( bool *ok, int anum, int tnum, const QString &charset, EventTable *table ) : DVBsection( ok, anum, tnum, charset )
{
	events = table;
	pf[0].fd = fdDemux;
	pf[0].events = POLLIN;
}



DVBevents::~DVBevents()
{
	isRunning = false;
	if ( !wait(2000) ) {
		terminate();
		wait();
	}
}



bool DVBevents::shortEventDesc( unsigned char *buf, EventDesc *desc )
{
	QString name, text;
	int len, len2;
	ShortEvent *ev;

	if ( !safeLen( buf+6 ) )
		return false;
	len = getBits(buf,40,8);
	if ( !safeLen( buf+6+len ) )
		return false;
	name = getText( buf+6, len );
	if ( !safeLen( buf+6+len+1 ) )
		return false;
	len2 = getBits(buf+6+len,0,8);
	if ( !safeLen( buf+7+len+len2 ) )
		return false;
	text = getText( buf+7+len, len2);
	if ( desc->title.isEmpty() ) {
		desc->title=name;
		desc->subtitle=text;
		return true;
	}
	desc->shortEvents.append( new ShortEvent() );
	ev = desc->shortEvents.getLast();
	ev->name = name;
	ev->text = text;
	return true;
}



bool DVBevents::extEventDesc( unsigned char *buf, EventDesc *desc )
{
	int loop, len1, len2;
	unsigned char *b = buf;
	QString s;

	if ( !safeLen( b+7 ) )
		return false;
	loop = getBits(b+6,0,8);
	b +=7;

	while ( loop>0 ) {
		if ( !safeLen( b+1 ) )
			return false;
		len1 = getBits(b,0,8);
		if ( !safeLen( b+1+len1 ) )
			return false;
		s = getText(b+1,len1);
		if ( !safeLen( b+1+len1+1 ) )
			return false;
		len2 = getBits(b+1+len1,0,8);
		if ( !safeLen( buf+2+len1+len2 ) )
			return false;
		if ( !s.isEmpty() )
			s = s+" : ";
		s = s+getText(b+2+len1,len2);
		desc->extEvents.append( new QString( s ) );
		b +=(2+len1+len2);
		loop -=(2+len1+len2);
	}
	if ( !safeLen( b+1 ) )
		return false;
	len1 = getBits(b,0,8);
	if ( !safeLen( b+1+len1 ) )
		return false;
	s = getText(b+1,len1);
	desc->extEvents.append( new QString( s ) );
	return true;
}



bool DVBevents::tableEIT( unsigned char* buffer )
{
	unsigned char* buf = buffer;
	unsigned int length, loop, sid, tid, eid, tsid, sn, lsn, nid;
	int i, sec;
	EventDesc *desc=0, *itdesc=0;
	EventSid *slist;
	QPtrList<EventDesc> *currentEvents;
	bool nodesc, parse;
	QDateTime start, cur, dt;
	unsigned int cdt = QDateTime::currentDateTime().toTime_t();

	tid = getBits(buf,0,8);
	length = getBits(buf,12,12);
	sid = getBits(buf,24,16);
	sn = getBits(buf,48,8);
	lsn = getBits(buf,56,8);
	tsid = getBits(buf,64,16);
	nid = getBits(buf,80,16);
	length -=11;
	buf +=14;

	slist = currentSrc->getEventSid( tsid, sid );
	slist->lock();
	currentEvents = slist->getEvents();
	QPtrListIterator<EventDesc> it( *currentEvents );

	while ( length>4 ) {
		nodesc=parse=false;
		if ( !safeLen( buf+2 ) )
			goto stop;
		eid = getBits(buf,0,16);
		if ( !safeLen( buf+2+5 ) )
			goto stop;
		start = getDateTime( buf+2 );
		nodesc=parse=true;

		it.toFirst();
		while ( (desc=it.current())!=0 ) {
			if ( desc->sid==sid ) {
				if ( desc->startDateTime==start || desc->eid==eid ) {
					if ( desc->tid==0x4e && tid!=0x4e ) {
						parse = false;
						nodesc = false;
						break;
					}
					else {
						nodesc = false;
						if ( (cdt-desc->loop)<300 ) { // only reparse events every 300 seconds
							parse = false;
						}
						else {
							desc->extEvents.clear();
							desc->shortEvents.clear();
							desc->title=desc->subtitle="";
						}
						break;
					}
				}
			}
			++it;
		}

		if ( nodesc )
			desc = new EventDesc();
		if ( parse ) {
			if ( !safeLen( buf+10 ) )
				goto stop;
			desc->duration = getTime( buf+7 );
			if ( !safeLen( buf+11 ) )
				goto stop;
			desc->running = getBits(buf,80,3);
			desc->sid = sid;
			desc->tid = tid;
			desc->tsid = tsid;
			desc->nid = nid;
			desc->lsn = lsn;
			desc->sn = sn;
			desc->eid = eid;
			desc->loop = cdt;
		}

		if ( desc->sn != sn ) {
			slist->unlock();
			return false;
		}
		if ( !safeLen( buf+12 ) )
			goto stop;
		loop = getBits(buf,84,12);
		buf +=12;
		length -=(12+loop);
		while ( loop>0 ) {
			if ( parse ) {
				if ( !safeLen( buf+1 ) )
					goto stop;
				switch ( getBits(buf,0,8) ) {
					case 0x4D :
						if ( !shortEventDesc( buf, desc ) )
							goto stop;
						break;
					case 0x4E :
						if ( !extEventDesc( buf, desc ) )
							goto stop;
						break;
					default :
						break;
				}
			}
			if ( !safeLen( buf+2 ) )
				goto stop;
			loop -=( getBits(buf,8,8)+2 );
			buf +=( getBits(buf,8,8)+2 );
		}
//out:
		if ( parse ) {
			if ( !nodesc ) {
				if ( start==desc->startDateTime )
					goto ifend;
				currentEvents->take( currentEvents->find( desc ) );
			}
			desc->startDateTime = start;
			for ( i=0; i<(int)currentEvents->count(); i++ ) {
				itdesc = currentEvents->at(i);
				if ( desc->startDateTime<itdesc->startDateTime ) {
					currentEvents->insert( i, desc );
					break;
				}
				itdesc = 0;
			}
			if ( !itdesc )
				currentEvents->append( desc );
		}
ifend:
		if ( parse )
			++(desc->sn);
		if ( nodesc ) {
			cur = QDateTime::currentDateTime();
			dt = desc->startDateTime;
			sec = desc->duration.hour()*3600+desc->duration.minute()*60+desc->duration.second();
			if ( dt.addSecs( sec )<cur || desc->title.length()<3 ) {
				currentEvents->remove( desc );
			}
			else
				desc->source = currentSrc->getSource();
		}

	}
	slist->unlock();
	return true;
stop:
	slist->unlock();
	fprintf( stderr, "Stop parsing EIT (%d:%d)\n", adapter, tuner );
	if ( nodesc )
		delete desc;
	return false;
}




bool DVBevents::safeLen( unsigned char* buf )
{
	if ( buf<(secbuf+readSize) )
		return true;
	fprintf( stderr, "EIT (%d:%d) : buffer overflow! Rejected\n", adapter, tuner );
	return false;
}



bool DVBevents::go( QString src, bool all )
{
	int tid;

	if ( isRunning )
		return true;

	if ( all )
		tid = 0;
	else
		tid = 0x4e;

	currentSrc = events->getEventSource( src );

	if ( !setFilter( 0x12, tid, 1000 ) )
		return false;
	isRunning = true;
	start();
	fprintf(stderr,"dvbEvents %d:%d started\n", adapter, tuner);
	return true;
}



void DVBevents::stop()
{
	if ( !isRunning )
		return;

	isRunning = false;
	if ( !wait(2000) ) {
		terminate();
		wait();
		fprintf(stderr,"dvbEvents %d:%d terminated\n", adapter, tuner);
	}
	else
		fprintf(stderr,"dvbEvents %d:%d ended\n", adapter, tuner);
	stopFilter();
}



void DVBevents::run()
{
	int n=0, tid;
	int skip=0;

	setpriority(PRIO_PROCESS, 0, 19); // eit parsing is cpu eater on some astra multiplex.
	while ( isRunning ) {
		if ( !isRunning )
			break;

		if ( poll(pf,1,1000)>0 ){
			if ( pf[0].revents & POLLIN ){
				n = read( fdDemux, secbuf, 4096 );
				skip = 0;
			}
			else
				skip++;
		}
		else
			skip++;

		if (skip)
			continue;
		if ( n<16 )
			continue;
		else
			readSize = n;

		if ( !isRunning )
			break;

		tid = getBits(secbuf,0,8);
		if ( tid>0x4D && tid<0x70 ) {
			tableEIT( secbuf );
		}
	}
}

#include "dvbevents.moc"
